// Check methods that write bit fields
#include "MPU9150.h"
#include "mbed.h"

MPU9150::MPU9150() : i2c_(I2C_SDA, I2C_SCL) {
    address_ = MPU9150_ADDRESS_DEFAULT;
    
    accelRange_ = 0;
    gyroRange_ = 0;
}

MPU9150::MPU9150(const bool AD0) : i2c_(I2C_SDA, I2C_SCL) {
    if(AD0) {
        address_ = MPU9150_ADDRESS_AD0_HIGH;
    } else {
        address_ = MPU9150_ADDRESS_AD0_LOW;
    }
    
    accelRange_ = 0;
    gyroRange_ = 0;
}

MPU9150::MPU9150(const PinName sda, const PinName scl, const bool AD0) : i2c_(sda, scl) {
    if(AD0) {
        address_ = MPU9150_ADDRESS_AD0_HIGH;
    } else {
        address_ = MPU9150_ADDRESS_AD0_LOW;
    }
    
    accelRange_ = 0;
    gyroRange_ = 0;
}

void MPU9150::initialize() {
    i2c_.setFrequency(100000);  // Set frequency to 400 kHz
    setClockSource(MPU9150_PWR_MGMT_CLOCK_GYRO_X);
    setGyroFullScaleRange(MPU9150_GYRO_FS_250);
    setAccelFullScaleRange(MPU9150_ACCEL_FS_2);
    setSleepEnabled(false);
}

bool MPU9150::test() {
    return getDeviceID() == 0x34;
}

uint8_t MPU9150::getSampleRateDivider() {
    uint8_t buf = 0xFF;
    i2c_.readByte(address_, MPU9150_RA_SMPRT_DIV, &buf);
    return buf;
}
void MPU9150::setSampleRateDivider(const uint8_t divider) {
    i2c_.writeByte(address_, MPU9150_RA_SMPRT_DIV, divider);
}

uint8_t MPU9150::getExternalFrameSync() {
    uint8_t buf = 0xFF;
    i2c_.readBits(address_, MPU9150_RA_CONFIG, 5, 3, &buf);
    return buf;
}
void MPU9150::setExternalFrameSync(const uint8_t sync) {
    i2c_.writeBits(address_, MPU9150_RA_CONFIG, 5, 3, sync);
}
uint8_t MPU9150::getDLPFBandwidth() {
    uint8_t buf = 0xFF;
    i2c_.readBits(address_, MPU9150_RA_CONFIG, 2, 3, &buf);
    return buf;
}
void MPU9150::setDLPFBandwidth(const uint8_t bandwidth) {
    i2c_.writeBits(address_, MPU9150_RA_CONFIG, 2, 3, bandwidth);
}

uint8_t MPU9150::getGyroFullScaleRange() {
    uint8_t buf = 0xFF;
    i2c_.readBits(address_, MPU9150_RA_GYRO_CONFIG, 4, 2, &buf);
    return buf;
}
void MPU9150::setGyroFullScaleRange(const uint8_t range) {
    i2c_.writeBits(address_, MPU9150_RA_GYRO_CONFIG, 4, 2, range);
}

uint8_t MPU9150::getAccelFullScaleRange() {
    uint8_t buf = 0xFF;
    i2c_.readBits(address_, MPU9150_RA_ACCEL_CONFIG, 4, 2, &buf);
    return buf;
}
void MPU9150::setAccelFullScaleRange(const uint8_t range) {
    i2c_.writeBits(address_, MPU9150_RA_ACCEL_CONFIG, 4, 2, range);
}

int16_t MPU9150::getAccelX() {
    uint16_t buf = 0xFFFF;
    i2c_.readWord(address_, MPU9150_RA_ACCEL_XOUT_H, &buf);
    return (int16_t)buf;
}
int16_t MPU9150::getAccelY() {
    uint16_t buf = 0xFFFF;
    i2c_.readWord(address_, MPU9150_RA_ACCEL_YOUT_H, &buf);
    return (int16_t)buf;
}
int16_t MPU9150::getAccelZ() {
    uint16_t buf = 0xFFFF;
    i2c_.readWord(address_, MPU9150_RA_ACCEL_ZOUT_H, &buf);
    return (int16_t)buf;
}

int16_t MPU9150::getTemp() {
    uint16_t buf = 0xFFFF;
    i2c_.readWord(address_, MPU9150_RA_TEMP_OUT_H, &buf);
    return (int16_t)buf;
}

int16_t MPU9150::getGyroX() {
    uint16_t buf = 0xFFFF;
    i2c_.readWord(address_, MPU9150_RA_GYRO_XOUT_H, &buf);
    return (int16_t)buf;
}
int16_t MPU9150::getGyroY() {
    uint16_t buf = 0xFFFF;
    i2c_.readWord(address_, MPU9150_RA_GYRO_YOUT_H, &buf);
    return (int16_t)buf;
}
int16_t MPU9150::getGyroZ() {
    uint16_t buf = 0xFFFF;
    i2c_.readWord(address_, MPU9150_RA_GYRO_ZOUT_H, &buf);
    return (int16_t)buf;
}

void MPU9150::resetGyroPath() {
    i2c_.writeBit(address_, MPU9150_RA_SIGNAL_PATH_RESET, 2, 0x01);
}
void MPU9150::resetAccelPath() {
    i2c_.writeBit(address_, MPU9150_RA_SIGNAL_PATH_RESET, 1, 0x01);
}
void MPU9150::resetTempPath() {
    i2c_.writeBit(address_, MPU9150_RA_SIGNAL_PATH_RESET, 0, 0x01);
}

bool MPU9150::getFifoEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_USER_CTRL, 6, &buf);
    return (bool)buf;
}
void MPU9150::setFifoEnabled(const bool fifo) {
    i2c_.writeBit(address_, MPU9150_RA_USER_CTRL, 6, fifo);
}
bool MPU9150::getI2CMasterEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_USER_CTRL, 5, &buf);
    return (bool)buf;
}
void MPU9150::setI2CMasterEnabled(const bool master) {
    i2c_.writeBit(address_, MPU9150_RA_USER_CTRL, 5, master);
}
void MPU9150::resetFifo() {
    i2c_.writeBit(address_, MPU9150_RA_USER_CTRL, 2, 0x01);
}
void MPU9150::resetI2CMaster() {
    i2c_.writeBit(address_, MPU9150_RA_USER_CTRL, 1, 0x01);
}
void MPU9150::resetSensors() {
    i2c_.writeBit(address_, MPU9150_RA_USER_CTRL, 0, 0x01);
}

void MPU9150::reset() {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_1, 7, 0x01);
}
bool MPU9150::getSleepEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_1, 6, &buf);
    return (bool)buf;
}
void MPU9150::setSleepEnabled(const bool sleep) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_1, 6, sleep);
}
bool MPU9150::getCycleEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_1, 5, &buf);
    return (bool)buf;
}
void MPU9150::setCycleEnabled(const bool cycle) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_1, 5, cycle);
}
bool MPU9150::getTempEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_1, 3, &buf);
    return (bool)buf;
}
void MPU9150::setTempEnabled(const bool temp) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_1, 3, temp);
}
uint8_t MPU9150::getClockSource() {
    uint8_t buf = 0xFF;
    i2c_.readBits(address_, MPU9150_RA_PWR_MGMT_1, 2, 3, &buf);
    return buf;
}
void MPU9150::setClockSource(const uint8_t source) {
    i2c_.writeBits(address_, MPU9150_RA_PWR_MGMT_1, 2, 3, source);
}

uint8_t MPU9150::getCycleFrequency() {
    uint8_t buf = 0xFF;
    i2c_.readBits(address_, MPU9150_RA_PWR_MGMT_2, 7, 2, &buf);
    return buf;
}
void MPU9150::setCycleFrequency(const uint8_t frequency) {
    i2c_.writeBits(address_, MPU9150_RA_PWR_MGMT_2, 7, 2, frequency);
}
bool MPU9150::getStandbyAccelXEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_2, 5, &buf);
    return (bool)buf;
}
void MPU9150::setStandbyAccelXEnabled(const bool enabled) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_2, 5, enabled);
}
bool MPU9150::getStandbyAccelYEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_2, 4, &buf);
    return (bool)buf;
}
void MPU9150::setStandbyAccelYEnabled(const bool enabled) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_2, 4, enabled);
}
bool MPU9150::getStandbyAccelZEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_2, 3, &buf);
    return (bool)buf;
}
void MPU9150::setStandbyAccelZEnabled(const bool enabled) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_2, 3, enabled);
}
bool MPU9150::getStandbyGyroXEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_2, 2, &buf);
    return (bool)buf;
}
void MPU9150::setStandbyGyroXEnabled(const bool enabled) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_2, 2, enabled);
}
bool MPU9150::getStandbyGyroYEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_2, 1, &buf);
    return (bool)buf;
}
void MPU9150::setStandbyGyroYEnabled(const bool enabled) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_2, 1, enabled);
}
bool MPU9150::getStandbyGyroZEnabled() {
    uint8_t buf = 0xFF;
    i2c_.readBit(address_, MPU9150_RA_PWR_MGMT_2, 0, &buf);
    return (bool)buf;
}
void MPU9150::setStandbyGyroZEnabled(const bool enabled) {
    i2c_.writeBit(address_, MPU9150_RA_PWR_MGMT_2, 0, enabled);
}

uint8_t MPU9150::getDeviceID() {
    uint8_t buf = 0xFF;
    i2c_.readBits(address_, MPU9150_RA_WHO_AM_I, 6, 6, &buf);
    return buf;
}
