;* Complex 16 bit Radix 4 FFT and Inverse FFT for Cortex-M3    version 1.0
;--------------------------------------------------------------------------
;(c) 2008 Ivan Mellen   
; adapted for ARM7 (mbed) and ARM assembler syntax by Igor Skochinsky
;--------------------------------------------------------------------------
;Free Personal, Non-Commercial Use License. 
;The Software is licensed to you for your personal, NON-COMMERCIAL USE.
;If you have questions about this license or would like a different license
;please email : imellen(at)embeddedsignals(dot)com

; - Radix 4 FFT - supported sizes: N=4,16,64,256,1024,4096
; - N>4096 possible with custom coefficients  
; - 16 bit complex arithmetic, 1Q15 coefficients
; - input data remains unmodified
; - decimation-in-time with auto scale after each stage - no overflow 
; - GCC version (Code Sourcery G++ Lite 2007q3-53), requires C preprocessor
; - hand optimized THUMB2 assembly for Cortex-M3 (e.g. STM32)
; - code optimized with 64 bit flash prefetch and branch speculation in mind
; - single function for multiple FFT sizes
; - functions are both "C" and assembly callable

;Modifications in version 2.0:
; - Constant size FFT (faster execution, but N is constant)
; - purely real input (almost 2x faster)
; - radix 2 stage (for N= 8, 32, 128, 512, 2048, 8192...)
; - RAM based coefficients with size optimized generator (for high flash latency)
; - speed optimized windowing function(Bartlett,Blackman, Hamming, Hanning, Kaiser)


;STM32 FFT benchmarks in CPU cycles based on real hardware measurements:

; N - FFT size
; L - Flash latency
; F,R - coefficients in Flash or RAM

;    N    L=0 F/R    L=1 F   L=1 r*   L=2 F*     L=2 r*   ARM7TDMI  ARM9E  dsPIC
;   64     3575      3797     3636     4588      4007       -        2701   3739
;  256    19425     20685    19743    25144     21685  38461-43920  13740  19055
; 1024    98541    105113   100074   128070    109634       -       68534    N/A

;Notes:ARM9E - DSP enhanced arm9 core, based on STR910 @96 MHz, RAM coefficients
;      dsPIC - dsPIC30x / dsPIC33x - results based on Microchip's DSP library 
;      ARM7TDMI - 3rd party, web based benchmark results
;      *STM32 results for latency L=2 or  - source compiled with LATENCY2 defined

;IFFT benchmarks (calculated and verified by measurement):
;  add 6+N/4 to FFT benchmark {22,70,262} for N={64,256,1024}

;Code size:
;FFT only:  480 bytes
;FFT+IFFT:  682 bytes

;Coefficient size (Flash or RAM):
;   N: 16  64  256  1024  4096
;size: 48 240 1008  4080 16368  bytes

;------------------------------------------------------------------------------
;Usage example: add this file to your project. In C code:

;//declare functions
;void fftR4(short *y, short *x, int N);
;void ifftR4(short *y, short *x, int N);

;// prepare test input data
;short x[512]; // input data 16 bit, 4 byte aligned  x0r,x0i,x1r,x1i,....
;short y[512]; // output data 16 bit,4 byte aligned  y0r,y0i,y1r,y1i,....
;short z[512]; // same format...

;for (i=0;i<512;i++) x[i]=0;
;for (i=0;i<512;i=i+8)
;  { x[i+0]=16384; x[i+2]=16384;    x[i+4]=-16384;  x[i+6]=-16384;}
;// x = [ 16384,16384,-16384,-16384,16384,...]  1/4 Fsampling

;//call functions
;fftR4(y, x, 256);   // y is in frequency domain y[128]=
;ifftR4(z, y, 256);  // z should be x/N + noise introduced by 16 bit truncating 
 

;// expected results:
;//y[128]is 8191; y[129] is -8190   y[384]is 8191; y[385]is 8191 rest 0 + noise
;//z[2n] is 64 64 -64 -64 ..  z[2n+1] is 0    all +- 1 (noise)
;------------------------------------------------------------------------------
;*/

; // This file contains two functions :

; // void fftR4(short *y, short *x, int N);   // radix 4 FFT
; // void ifftR4(short *y, short *x, int N);  // radix 4 inverse FFT
         
;        .syntax unified
;//    .thumb
    EXPORT fftR4  
    EXPORT ifftR4   

    AREA FFT, CODE, READONLY

;//#define LATENCY2  //comment this line if flash latency lower than 2

y      RN    R0    ;/ short *y -output complex array  
x      RN    R1    ;/ short *x -input complex array  
N      RN    R2    ;/ int N - number of samples 4,16,64,256,1024,4096

c      RN    R0    ;/ short *c - coefficient array
Bl     RN    R2    ;/ the number of blocks
R      RN    R3    ;/ the number of samples in each block
x0r    RN    R4
x0i    RN    R5
x1r    RN    R6
x1i    RN    R7
x2r    RN    R8
x2i    RN    R9
x3r    RN    R10
x3i    RN    R11
y3r    RN    R11
y3i    RN    R10
tmp0   RN    R12   ;/ temporary0
tmp1   RN    R14   ;/ temporary1

;       // complex load, x=[a], a+=offset in register data from RAM (0 wait states)

      MACRO    
      LOADC   $dr, $di, $a, $offset   ;//5 cycles
        ldrsh   $di, [$a, #2]
        ldrsh   $dr, [$a]      ;//ldrsh   $x._r, [$a], $offset
        adds     $a, $offset
      MEND

;        // cofficient complex load, c=[a], a=a+ 4  ;Wk from FLASH (1-4 wait states)
      macro
      LOADCF   $xr, $xi, $a  ;              //4 cycles
        ldr   $xr, [$a], #4    ;    //read 2 shorts as 32 bits
        asr   $xi, $xr, #16    ;    //im extract
    IF :DEF:TARGET_LPC1768
       sxth  $xr, $xr        ;    //re extract
    ELSE        
        lsl $xr, #16
        asr $xr, #16
    ENDIF        
      mend
   
;        // coefficient complex load, c=[a], a=a+ 4   ; Wk  from RAM (0 wait states)
; //   .macro  LOADCF  xr,xi, a   //4 (or 3?) cycles
;  //        ldrsh   \xi, [$a, #2]
;  //      ldrsh   \xr, [$a],#4  //ldrsh not pipelined in STM32, macro not used
;    .endm
  
;  // complex load, xc=[a], a=a-register offset
       macro   
       LOADCMi   $xr, $xi, $a, $offset   ;//5 cycles
        ldrsh   $xi, [$a, #2]
        ldrsh   $xr, [$a]
        subs     $a, $offset       ;//ldrsh   \xr, [$a], \offset
       mend

;        // complex store, [a]=x,c a=a+ immediate offset 
      macro   
      STRC   $xr, $xi, $a, $offset   ;//2 cycles
        strh    $xi, [$a, #2]
        strh    $xr, [$a], $offset
      mend        

;       // complex store, [a]=x, a+=offset in register
      macro  
      STRCR   $xr, $xi, $a, $offset   ;//3 cycles
        strh    $xi, [$a, #2]
        strh    $xr, [$a]
        adds     $a, $offset
      mend 


;        // Multiply Complex Conjugate a=(xr+i*xi)*(cr-i*ci)
;        //  x = xr + i*xi 
;        //  c = cr + i*ci  ;6 cycles
      macro   
      MULCC1 $ar, $ai, $xr, $xi, $cr, $ci
        mul  tmp0, $xi, $cr           ; // tmp0=xi*cr
        mul  tmp1, $xi, $ci            ;// tmp1=xi*ci        
    IF :DEF:TARGET_LPC1768
        mls  $ai, $xr, $ci, tmp0   ;    // ai=tmp0-xr*ci  =  xi*cr - xr*ci
    //ELSE
        mul  $ai, $xr, $ci         ;    // ai=xr*ci
        sub  $ai, tmp0, $ai        ;    // ai=tmp0-ai  =  xi*cr - xr*ci
    ENDIF
        mla  $ar, $xr, $cr, tmp1       ;// ar=tmp1+xr*cr  =  xr*cr + xi*ci
      mend

        
;        // complex Fast Fourier Transform - Four point; scale with shift s
       macro    
       BFFT4  $s
       
        add     x2r, x2r, x3r   ;// (x2,x3) = (x2+x3, x2-x3)
        add     x2i, x2i, x3i
        sub     x3r, x2r, x3r, lsl#1
        sub     x3i, x2i, x3i, lsl#1
       
        mov     x0r, x0r, asr#2   ;// (x0,x1) = (x0+(x1>>s), x0-(x1>>s))/4
        mov     x0i, x0i, asr#2
        add     x0r, x0r, x1r, asr#(2+$s)
        add     x0i, x0i, x1i, asr#(2+$s)
        sub     x1r, x0r, x1r, asr#(1+$s)
        sub     x1i, x0i, x1i, asr#(1+$s)
       
        add     x0r, x0r, x2r, asr#(2+$s)   ;// (x0,x2) = (x0+(x2>>s)/4, x0-(x2>>s)/4)
        add     x0i, x0i, x2i, asr#(2+$s) 
        sub     x2r, x0r, x2r, asr#(1+$s)
        sub     x2i, x0i, x2i, asr#(1+$s)
        
        add     x1r, x1r, x3i, asr#(2+$s)  ;// (x1,y3)=(x1-i*(x3>>s)/4, x1+i*(x3>>s)/4)
        sub     x1i, x1i, x3r, asr#(2+$s)
        sub     y3r, x1r, x3i, asr#(1+$s)
        add     y3i, x1i, x3r, asr#(1+$s)
       mend
        
; we can't use RBIT on ARM7 thus this macro
; it can probably be optimized for common cases (x < 64, 256, 1024)...
;unsigned int RBIT(unsigned int x)
;{
;    x = (((x >> 1) & 0x55555555) | ((x & 0x55555555) << 1)); 
;    x = (((x >> 2) & 0x33333333) | ((x & 0x33333333) << 2)); 
;    x = (((x >> 4) & 0x0f0f0f0f) | ((x & 0x0f0f0f0f) << 4)); 
;    x = (((x >> 8) & 0x00ff00ff) | ((x & 0x00ff00ff) << 8)); 
;    return ((x >> 16) | (x << 16)); 
;}
     MACRO 
        MYRBIT  $rd, $rs
     IF :DEF:TARGET_LPC1768
       RBIT $rd, $rs
     ELSE
        PUSH {R0}
        PUSH {R1-R3}
        IF R0 != $rs
          MOV R0, $rs
        ENDIF
        ADR     R3, TableRBIT_V1
        LDR     R1, [R3], #4
        AND     R2, R1, R0,LSR#1
        AND     R0, R0, R1
        LDR     R1, [R3], #4
        ORR     R0, R2, R0,LSL#1
        AND     R2, R1, R0,LSR#2
        AND     R0, R0, R1
        LDR     R1, [R3], #4
        ORR     R0, R2, R0,LSL#2
        AND     R2, R1, R0,LSR#4
        AND     R0, R0, R1
        ORR     R0, R2, R0,LSL#4
        LDR     R1, [R3], #4
        AND     R1, R1, R0,LSR#8
        BIC     R0, R0, #0xFF00
        ORR     R0, R1, R0,LSL#8
        MOV     R0, R0,ROR#16
        POP     {R1-R3}
        IF R0 != $rd
          MOV $rd, R0
        ENDIF
        POP     {R0}
        ENDIF   
        MEND

TableRBIT_V1
        DCD  0x55555555,0x33333333,0xFF0F0F0F,0xFFFF00FF
 
;//----------------------------------------------------------------------------
;// inverse FFT 
;// void ifftR4(short *y, short *x, int N) 
;// custom first stage reorders input data: x(n)=x(N-n) n=0:N-1  x(N)=x(0)
;// extra cost to fft: 6+N/4 cycles
;        .thumb_func
;        align 8  //speed optimization in STM32
;        nop.n //alignment optimization

ifftR4 
        push {r4-r11, lr}
        mov      tmp0, #0  ;// bit reversed counter
        movs     tmp1, N    ; //; first tmp1=N  
        RBIT   R, N
        lsl      R, #3

        adds    tmp1,  x, tmp1, lsl#2 ;// tmp1=&x[tmp1==N] 
        ldrsh   x0i, [x, #2] ;  // replaces  C_LDRmi x[N] by _LDRmi x[0] 
        ldrsh   x0r, [x]
        subs  tmp1, N    ; //tmp1 still needs to be decremented for 2nd C_LDRmi
        b L2 ; // continue with second load

ifirstStage
        ;// first stage load and bit reverse
        adds    tmp1,  x, tmp1, lsl#2 ;// tmp1=&x[tmp1] 
        LOADCMi   x0r,x0i, tmp1, N
L2     
        LOADCMi   x2r,x2i, tmp1, N
        LOADCMi   x1r,x1i, tmp1, N
        LOADCMi   x3r,x3i, tmp1, N
         BFFT4  0
        STRC   x0r,x0i, y, #4
        STRC   x1r,x1i, y, #4
        STRC   x2r,x2i, y, #4
        STRC   y3r,y3i, y, #4
           
       adds tmp0,R
       RBIT tmp1, tmp0
       sub tmp1, N, tmp1  ;//tmp1=N-tmp1    
       bne     ifirstStage ;// loop if count non zero  
       b firstStageFinished
        ;// rest same as normal fft
 
 ;//----------------------------------------------------------------------------
; // void fftR4(short *y, short *x, int N)
;      .thumb_func
fftR4
        push {r4-r11, lr}
        mov     tmp0, #0 ;           // bit reversed counter
        mov     tmp1, #0
        RBIT  R, N
        lsl     R,#3

firstStage
       ; // first stage load and bit reverse
        adds    tmp1,  x, tmp1, lsl#2; // tmp1=&x[tmp1] and clear carry
        LOADC   x0r, x0i, tmp1, N
        LOADC   x2r,x2i, tmp1, N
        LOADC   x1r,x1i, tmp1, N
        LOADC   x3r,x3i, tmp1, N
        BFFT4  0
        STRC   x0r,x0i, y, #4
        STRC   x1r,x1i, y, #4
        STRC   x2r,x2i, y, #4
        STRC   y3r,y3i, y, #4
           
       adds tmp0,R
       RBIT tmp1, tmp0
       bne     firstStage ;// loop if count non zero
   
firstStageFinished        ;// finished the first stage
        sub     x, y, N, lsl #2  ; // x = working buffer
        mov     R, #16
        lsrs Bl,N, #4  
        popeq  {r4-r11, pc}     ;// for N==4 return from function
        adr     c, coef_table    ;//change if table in RAM 
                
nextStage
        ;// Bl = the number of blocks
        ;// R = the number of samples in each block
 
;#ifdef LATENCY2      ; // narrow/wide versions to optimize flash pre-fetch
;        stm   sp!, {x, Bl}   
;        add     tmp0, R, R, lsl#1
;        add.n     x, x, tmp0    
;#else
        push      {x, Bl}  
        add       tmp0, R, R, lsl#1
        add       x, x, tmp0   
;#endif
        sub     Bl, Bl, #1<<16
nextBlock
        add     Bl, Bl, R, lsl#(16-2)

nextButterfly
        ;// Bl=((number butterflies left-1)<<16) + (number of blocks left)
        LOADCMi   x0r,x0i, x, R
        //LOADCF   x3r,x3i, c
        //MULCC1   x3r,x3i, x0r,x0i, x3r,x3i
        LOADCMi   x0r,x0i, x, R
        //LOADCF   x2r,x2i, c
        //MULCC1   x2r,x2i, x0r,x0i, x2r,x2i
        LOADCMi   x0r,x0i, x, R
        //LOADCF   x1r,x1i, c
        //MULCC1 x1r,x1i, x0r,x0i, x1r,x1i
        LOADC   x0r,x0i, x, #0
        BFFT4  15             ; // coefficients are 1Q15
        STRCR  x0r,x0i, x, R
        STRCR  x1r,x1i, x, R
        STRCR  x2r,x2i, x, R
        STRC   y3r,y3i, x, #4
        subs    Bl, Bl, #1<<16
        bge     nextButterfly
        add     tmp0, R, R, lsl#1
        add     x, x, tmp0
        sub     Bl, Bl, #1
        movs    tmp1, Bl, lsl#16
        subne   c, c, tmp0
        bne     nextBlock
        
        pop {r1-r2}   ;//LDM sp!, {x, Bl}
        mov     R, R, lsl#2     ;// block size *=4
        lsrs Bl,Bl, #2            ;//# of blocks /=4
        bne     nextStage
        pop    {r4-r11, pc}     ;//return
      
        align 4 ;//32 bit access acceleration

;// Note: unused portion of coef_table can be commented to reduce size
coef_table
        ;// FFT twiddle table of triplets E(3t), E(t), E(2t)
        ;// Where E(t)=cos(t)+i*sin(t) at 1Q15
        ;// N=16 t=2*PI*k/N for k=0,1,2,..,N/4-1
        DCW 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000
        DCW 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82
        DCW 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff
        DCW 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82
        ;// N=64 t=2*PI*k/N for k=0,1,2,..,N/4-1
        DCW 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000
        DCW 0x7a7d,0x2528, 0x7f62,0x0c8c, 0x7d8a,0x18f9
        DCW 0x6a6e,0x471d, 0x7d8a,0x18f9, 0x7642,0x30fc
        DCW 0x5134,0x62f2, 0x7a7d,0x2528, 0x6a6e,0x471d
        DCW 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82
        DCW 0x0c8c,0x7f62, 0x70e3,0x3c57, 0x471d,0x6a6e
        DCW 0xe707,0x7d8a, 0x6a6e,0x471d, 0x30fc,0x7642
        DCW 0xc3a9,0x70e3, 0x62f2,0x5134, 0x18f9,0x7d8a
        DCW 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff
        DCW 0x8f1d,0x3c57, 0x5134,0x62f2, 0xe707,0x7d8a
        DCW 0x8276,0x18f9, 0x471d,0x6a6e, 0xcf04,0x7642
        DCW 0x809e,0xf374, 0x3c57,0x70e3, 0xb8e3,0x6a6e
        DCW 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82
        DCW 0x9d0e,0xaecc, 0x2528,0x7a7d, 0x9592,0x471d
        DCW 0xb8e3,0x9592, 0x18f9,0x7d8a, 0x89be,0x30fc
        DCW 0xdad8,0x8583, 0x0c8c,0x7f62, 0x8276,0x18f9
        ;// N=256 t=2*PI*k/N for k=0,1,2,..,N/4-1
        DCW 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000
        DCW 0x7fa7,0x096b, 0x7ff6,0x0324, 0x7fd9,0x0648
        DCW 0x7e9d,0x12c8, 0x7fd9,0x0648, 0x7f62,0x0c8c
        DCW 0x7ce4,0x1c0c, 0x7fa7,0x096b, 0x7e9d,0x12c8
        DCW 0x7a7d,0x2528, 0x7f62,0x0c8c, 0x7d8a,0x18f9
        DCW 0x776c,0x2e11, 0x7f0a,0x0fab, 0x7c2a,0x1f1a
        DCW 0x73b6,0x36ba, 0x7e9d,0x12c8, 0x7a7d,0x2528
        DCW 0x6f5f,0x3f17, 0x7e1e,0x15e2, 0x7885,0x2b1f
        DCW 0x6a6e,0x471d, 0x7d8a,0x18f9, 0x7642,0x30fc
        DCW 0x64e9,0x4ec0, 0x7ce4,0x1c0c, 0x73b6,0x36ba
        DCW 0x5ed7,0x55f6, 0x7c2a,0x1f1a, 0x70e3,0x3c57
        DCW 0x5843,0x5cb4, 0x7b5d,0x2224, 0x6dca,0x41ce
        DCW 0x5134,0x62f2, 0x7a7d,0x2528, 0x6a6e,0x471d
        DCW 0x49b4,0x68a7, 0x798a,0x2827, 0x66d0,0x4c40
        DCW 0x41ce,0x6dca, 0x7885,0x2b1f, 0x62f2,0x5134
        DCW 0x398d,0x7255, 0x776c,0x2e11, 0x5ed7,0x55f6
        DCW 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82
        DCW 0x2827,0x798a, 0x7505,0x33df, 0x55f6,0x5ed7
        DCW 0x1f1a,0x7c2a, 0x73b6,0x36ba, 0x5134,0x62f2
        DCW 0x15e2,0x7e1e, 0x7255,0x398d, 0x4c40,0x66d0
        DCW 0x0c8c,0x7f62, 0x70e3,0x3c57, 0x471d,0x6a6e
        DCW 0x0324,0x7ff6, 0x6f5f,0x3f17, 0x41ce,0x6dca
        DCW 0xf9b8,0x7fd9, 0x6dca,0x41ce, 0x3c57,0x70e3
        DCW 0xf055,0x7f0a, 0x6c24,0x447b, 0x36ba,0x73b6
        DCW 0xe707,0x7d8a, 0x6a6e,0x471d, 0x30fc,0x7642
        DCW 0xdddc,0x7b5d, 0x68a7,0x49b4, 0x2b1f,0x7885
        DCW 0xd4e1,0x7885, 0x66d0,0x4c40, 0x2528,0x7a7d
        DCW 0xcc21,0x7505, 0x64e9,0x4ec0, 0x1f1a,0x7c2a
        DCW 0xc3a9,0x70e3, 0x62f2,0x5134, 0x18f9,0x7d8a
        DCW 0xbb85,0x6c24, 0x60ec,0x539b, 0x12c8,0x7e9d
        DCW 0xb3c0,0x66d0, 0x5ed7,0x55f6, 0x0c8c,0x7f62
        DCW 0xac65,0x60ec, 0x5cb4,0x5843, 0x0648,0x7fd9
        DCW 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff
        DCW 0x9f14,0x539b, 0x5843,0x5cb4, 0xf9b8,0x7fd9
        DCW 0x9930,0x4c40, 0x55f6,0x5ed7, 0xf374,0x7f62
        DCW 0x93dc,0x447b, 0x539b,0x60ec, 0xed38,0x7e9d
        DCW 0x8f1d,0x3c57, 0x5134,0x62f2, 0xe707,0x7d8a
        DCW 0x8afb,0x33df, 0x4ec0,0x64e9, 0xe0e6,0x7c2a
        DCW 0x877b,0x2b1f, 0x4c40,0x66d0, 0xdad8,0x7a7d
        DCW 0x84a3,0x2224, 0x49b4,0x68a7, 0xd4e1,0x7885
        DCW 0x8276,0x18f9, 0x471d,0x6a6e, 0xcf04,0x7642
        DCW 0x80f6,0x0fab, 0x447b,0x6c24, 0xc946,0x73b6
        DCW 0x8027,0x0648, 0x41ce,0x6dca, 0xc3a9,0x70e3
        DCW 0x800a,0xfcdc, 0x3f17,0x6f5f, 0xbe32,0x6dca
        DCW 0x809e,0xf374, 0x3c57,0x70e3, 0xb8e3,0x6a6e
        DCW 0x81e2,0xea1e, 0x398d,0x7255, 0xb3c0,0x66d0
        DCW 0x83d6,0xe0e6, 0x36ba,0x73b6, 0xaecc,0x62f2
        DCW 0x8676,0xd7d9, 0x33df,0x7505, 0xaa0a,0x5ed7
        DCW 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82
        DCW 0x8dab,0xc673, 0x2e11,0x776c, 0xa129,0x55f6
        DCW 0x9236,0xbe32, 0x2b1f,0x7885, 0x9d0e,0x5134
        DCW 0x9759,0xb64c, 0x2827,0x798a, 0x9930,0x4c40
        DCW 0x9d0e,0xaecc, 0x2528,0x7a7d, 0x9592,0x471d
        DCW 0xa34c,0xa7bd, 0x2224,0x7b5d, 0x9236,0x41ce
        DCW 0xaa0a,0xa129, 0x1f1a,0x7c2a, 0x8f1d,0x3c57
        DCW 0xb140,0x9b17, 0x1c0c,0x7ce4, 0x8c4a,0x36ba
        DCW 0xb8e3,0x9592, 0x18f9,0x7d8a, 0x89be,0x30fc
        DCW 0xc0e9,0x90a1, 0x15e2,0x7e1e, 0x877b,0x2b1f
        DCW 0xc946,0x8c4a, 0x12c8,0x7e9d, 0x8583,0x2528
        DCW 0xd1ef,0x8894, 0x0fab,0x7f0a, 0x83d6,0x1f1a
        DCW 0xdad8,0x8583, 0x0c8c,0x7f62, 0x8276,0x18f9
        DCW 0xe3f4,0x831c, 0x096b,0x7fa7, 0x8163,0x12c8
        DCW 0xed38,0x8163, 0x0648,0x7fd9, 0x809e,0x0c8c
        DCW 0xf695,0x8059, 0x0324,0x7ff6, 0x8027,0x0648
        ;// N=1024 t=2*PI*k/N for k=0,1,2,..,N/4-1
        DCW 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000
        DCW 0x7ffa,0x025b, 0x7fff,0x00c9, 0x7ffe,0x0192
        DCW 0x7fea,0x04b6, 0x7ffe,0x0192, 0x7ff6,0x0324
        DCW 0x7fce,0x0711, 0x7ffa,0x025b, 0x7fea,0x04b6
        DCW 0x7fa7,0x096b, 0x7ff6,0x0324, 0x7fd9,0x0648
        DCW 0x7f75,0x0bc4, 0x7ff1,0x03ed, 0x7fc2,0x07d9
        DCW 0x7f38,0x0e1c, 0x7fea,0x04b6, 0x7fa7,0x096b
        DCW 0x7ef0,0x1073, 0x7fe2,0x057f, 0x7f87,0x0afb
        DCW 0x7e9d,0x12c8, 0x7fd9,0x0648, 0x7f62,0x0c8c
        DCW 0x7e3f,0x151c, 0x7fce,0x0711, 0x7f38,0x0e1c
        DCW 0x7dd6,0x176e, 0x7fc2,0x07d9, 0x7f0a,0x0fab
        DCW 0x7d63,0x19be, 0x7fb5,0x08a2, 0x7ed6,0x113a
        DCW 0x7ce4,0x1c0c, 0x7fa7,0x096b, 0x7e9d,0x12c8
        DCW 0x7c5a,0x1e57, 0x7f98,0x0a33, 0x7e60,0x1455
        DCW 0x7bc6,0x209f, 0x7f87,0x0afb, 0x7e1e,0x15e2
        DCW 0x7b27,0x22e5, 0x7f75,0x0bc4, 0x7dd6,0x176e
        DCW 0x7a7d,0x2528, 0x7f62,0x0c8c, 0x7d8a,0x18f9
        DCW 0x79c9,0x2768, 0x7f4e,0x0d54, 0x7d3a,0x1a83
        DCW 0x790a,0x29a4, 0x7f38,0x0e1c, 0x7ce4,0x1c0c
        DCW 0x7840,0x2bdc, 0x7f22,0x0ee4, 0x7c89,0x1d93
        DCW 0x776c,0x2e11, 0x7f0a,0x0fab, 0x7c2a,0x1f1a
        DCW 0x768e,0x3042, 0x7ef0,0x1073, 0x7bc6,0x209f
        DCW 0x75a6,0x326e, 0x7ed6,0x113a, 0x7b5d,0x2224
        DCW 0x74b3,0x3497, 0x7eba,0x1201, 0x7aef,0x23a7
        DCW 0x73b6,0x36ba, 0x7e9d,0x12c8, 0x7a7d,0x2528
        DCW 0x72af,0x38d9, 0x7e7f,0x138f, 0x7a06,0x26a8
        DCW 0x719e,0x3af3, 0x7e60,0x1455, 0x798a,0x2827
        DCW 0x7083,0x3d08, 0x7e3f,0x151c, 0x790a,0x29a4
        DCW 0x6f5f,0x3f17, 0x7e1e,0x15e2, 0x7885,0x2b1f
        DCW 0x6e31,0x4121, 0x7dfb,0x16a8, 0x77fb,0x2c99
        DCW 0x6cf9,0x4326, 0x7dd6,0x176e, 0x776c,0x2e11
        DCW 0x6bb8,0x4524, 0x7db1,0x1833, 0x76d9,0x2f87
        DCW 0x6a6e,0x471d, 0x7d8a,0x18f9, 0x7642,0x30fc
        DCW 0x691a,0x490f, 0x7d63,0x19be, 0x75a6,0x326e
        DCW 0x67bd,0x4afb, 0x7d3a,0x1a83, 0x7505,0x33df
        DCW 0x6657,0x4ce1, 0x7d0f,0x1b47, 0x7460,0x354e
        DCW 0x64e9,0x4ec0, 0x7ce4,0x1c0c, 0x73b6,0x36ba
        DCW 0x6371,0x5098, 0x7cb7,0x1cd0, 0x7308,0x3825
        DCW 0x61f1,0x5269, 0x7c89,0x1d93, 0x7255,0x398d
        DCW 0x6068,0x5433, 0x7c5a,0x1e57, 0x719e,0x3af3
        DCW 0x5ed7,0x55f6, 0x7c2a,0x1f1a, 0x70e3,0x3c57
        DCW 0x5d3e,0x57b1, 0x7bf9,0x1fdd, 0x7023,0x3db8
        DCW 0x5b9d,0x5964, 0x7bc6,0x209f, 0x6f5f,0x3f17
        DCW 0x59f4,0x5b10, 0x7b92,0x2162, 0x6e97,0x4074
        DCW 0x5843,0x5cb4, 0x7b5d,0x2224, 0x6dca,0x41ce
        DCW 0x568a,0x5e50, 0x7b27,0x22e5, 0x6cf9,0x4326
        DCW 0x54ca,0x5fe4, 0x7aef,0x23a7, 0x6c24,0x447b
        DCW 0x5303,0x616f, 0x7ab7,0x2467, 0x6b4b,0x45cd
        DCW 0x5134,0x62f2, 0x7a7d,0x2528, 0x6a6e,0x471d
        DCW 0x4f5e,0x646c, 0x7a42,0x25e8, 0x698c,0x486a
        DCW 0x4d81,0x65de, 0x7a06,0x26a8, 0x68a7,0x49b4
        DCW 0x4b9e,0x6747, 0x79c9,0x2768, 0x67bd,0x4afb
        DCW 0x49b4,0x68a7, 0x798a,0x2827, 0x66d0,0x4c40
        DCW 0x47c4,0x69fd, 0x794a,0x28e5, 0x65de,0x4d81
        DCW 0x45cd,0x6b4b, 0x790a,0x29a4, 0x64e9,0x4ec0
        DCW 0x43d1,0x6c8f, 0x78c8,0x2a62, 0x63ef,0x4ffb
        DCW 0x41ce,0x6dca, 0x7885,0x2b1f, 0x62f2,0x5134
        DCW 0x3fc6,0x6efb, 0x7840,0x2bdc, 0x61f1,0x5269
        DCW 0x3db8,0x7023, 0x77fb,0x2c99, 0x60ec,0x539b
        DCW 0x3ba5,0x7141, 0x77b4,0x2d55, 0x5fe4,0x54ca
        DCW 0x398d,0x7255, 0x776c,0x2e11, 0x5ed7,0x55f6
        DCW 0x3770,0x735f, 0x7723,0x2ecc, 0x5dc8,0x571e
        DCW 0x354e,0x7460, 0x76d9,0x2f87, 0x5cb4,0x5843
        DCW 0x3327,0x7556, 0x768e,0x3042, 0x5b9d,0x5964
        DCW 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82
        DCW 0x2ecc,0x7723, 0x75f4,0x31b5, 0x5964,0x5b9d
        DCW 0x2c99,0x77fb, 0x75a6,0x326e, 0x5843,0x5cb4
        DCW 0x2a62,0x78c8, 0x7556,0x3327, 0x571e,0x5dc8
        DCW 0x2827,0x798a, 0x7505,0x33df, 0x55f6,0x5ed7
        DCW 0x25e8,0x7a42, 0x74b3,0x3497, 0x54ca,0x5fe4
        DCW 0x23a7,0x7aef, 0x7460,0x354e, 0x539b,0x60ec
        DCW 0x2162,0x7b92, 0x740b,0x3604, 0x5269,0x61f1
        DCW 0x1f1a,0x7c2a, 0x73b6,0x36ba, 0x5134,0x62f2
        DCW 0x1cd0,0x7cb7, 0x735f,0x3770, 0x4ffb,0x63ef
        DCW 0x1a83,0x7d3a, 0x7308,0x3825, 0x4ec0,0x64e9
        DCW 0x1833,0x7db1, 0x72af,0x38d9, 0x4d81,0x65de
        DCW 0x15e2,0x7e1e, 0x7255,0x398d, 0x4c40,0x66d0
        DCW 0x138f,0x7e7f, 0x71fa,0x3a40, 0x4afb,0x67bd
        DCW 0x113a,0x7ed6, 0x719e,0x3af3, 0x49b4,0x68a7
        DCW 0x0ee4,0x7f22, 0x7141,0x3ba5, 0x486a,0x698c
        DCW 0x0c8c,0x7f62, 0x70e3,0x3c57, 0x471d,0x6a6e
        DCW 0x0a33,0x7f98, 0x7083,0x3d08, 0x45cd,0x6b4b
        DCW 0x07d9,0x7fc2, 0x7023,0x3db8, 0x447b,0x6c24
        DCW 0x057f,0x7fe2, 0x6fc2,0x3e68, 0x4326,0x6cf9
        DCW 0x0324,0x7ff6, 0x6f5f,0x3f17, 0x41ce,0x6dca
        DCW 0x00c9,0x7fff, 0x6efb,0x3fc6, 0x4074,0x6e97
        DCW 0xfe6e,0x7ffe, 0x6e97,0x4074, 0x3f17,0x6f5f
        DCW 0xfc13,0x7ff1, 0x6e31,0x4121, 0x3db8,0x7023
        DCW 0xf9b8,0x7fd9, 0x6dca,0x41ce, 0x3c57,0x70e3
        DCW 0xf75e,0x7fb5, 0x6d62,0x427a, 0x3af3,0x719e
        DCW 0xf505,0x7f87, 0x6cf9,0x4326, 0x398d,0x7255
        DCW 0xf2ac,0x7f4e, 0x6c8f,0x43d1, 0x3825,0x7308
        DCW 0xf055,0x7f0a, 0x6c24,0x447b, 0x36ba,0x73b6
        DCW 0xedff,0x7eba, 0x6bb8,0x4524, 0x354e,0x7460
        DCW 0xebab,0x7e60, 0x6b4b,0x45cd, 0x33df,0x7505
        DCW 0xe958,0x7dfb, 0x6add,0x4675, 0x326e,0x75a6
        DCW 0xe707,0x7d8a, 0x6a6e,0x471d, 0x30fc,0x7642
        DCW 0xe4b9,0x7d0f, 0x69fd,0x47c4, 0x2f87,0x76d9
        DCW 0xe26d,0x7c89, 0x698c,0x486a, 0x2e11,0x776c
        DCW 0xe023,0x7bf9, 0x691a,0x490f, 0x2c99,0x77fb
        DCW 0xdddc,0x7b5d, 0x68a7,0x49b4, 0x2b1f,0x7885
        DCW 0xdb99,0x7ab7, 0x6832,0x4a58, 0x29a4,0x790a
        DCW 0xd958,0x7a06, 0x67bd,0x4afb, 0x2827,0x798a
        DCW 0xd71b,0x794a, 0x6747,0x4b9e, 0x26a8,0x7a06
        DCW 0xd4e1,0x7885, 0x66d0,0x4c40, 0x2528,0x7a7d
        DCW 0xd2ab,0x77b4, 0x6657,0x4ce1, 0x23a7,0x7aef
        DCW 0xd079,0x76d9, 0x65de,0x4d81, 0x2224,0x7b5d
        DCW 0xce4b,0x75f4, 0x6564,0x4e21, 0x209f,0x7bc6
        DCW 0xcc21,0x7505, 0x64e9,0x4ec0, 0x1f1a,0x7c2a
        DCW 0xc9fc,0x740b, 0x646c,0x4f5e, 0x1d93,0x7c89
        DCW 0xc7db,0x7308, 0x63ef,0x4ffb, 0x1c0c,0x7ce4
        DCW 0xc5c0,0x71fa, 0x6371,0x5098, 0x1a83,0x7d3a
        DCW 0xc3a9,0x70e3, 0x62f2,0x5134, 0x18f9,0x7d8a
        DCW 0xc198,0x6fc2, 0x6272,0x51cf, 0x176e,0x7dd6
        DCW 0xbf8c,0x6e97, 0x61f1,0x5269, 0x15e2,0x7e1e
        DCW 0xbd86,0x6d62, 0x616f,0x5303, 0x1455,0x7e60
        DCW 0xbb85,0x6c24, 0x60ec,0x539b, 0x12c8,0x7e9d
        DCW 0xb98b,0x6add, 0x6068,0x5433, 0x113a,0x7ed6
        DCW 0xb796,0x698c, 0x5fe4,0x54ca, 0x0fab,0x7f0a
        DCW 0xb5a8,0x6832, 0x5f5e,0x5560, 0x0e1c,0x7f38
        DCW 0xb3c0,0x66d0, 0x5ed7,0x55f6, 0x0c8c,0x7f62
        DCW 0xb1df,0x6564, 0x5e50,0x568a, 0x0afb,0x7f87
        DCW 0xb005,0x63ef, 0x5dc8,0x571e, 0x096b,0x7fa7
        DCW 0xae31,0x6272, 0x5d3e,0x57b1, 0x07d9,0x7fc2
        DCW 0xac65,0x60ec, 0x5cb4,0x5843, 0x0648,0x7fd9
        DCW 0xaaa0,0x5f5e, 0x5c29,0x58d4, 0x04b6,0x7fea
        DCW 0xa8e2,0x5dc8, 0x5b9d,0x5964, 0x0324,0x7ff6
        DCW 0xa72c,0x5c29, 0x5b10,0x59f4, 0x0192,0x7ffe
        DCW 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff
        DCW 0xa3d7,0x58d4, 0x59f4,0x5b10, 0xfe6e,0x7ffe
        DCW 0xa238,0x571e, 0x5964,0x5b9d, 0xfcdc,0x7ff6
        DCW 0xa0a2,0x5560, 0x58d4,0x5c29, 0xfb4a,0x7fea
        DCW 0x9f14,0x539b, 0x5843,0x5cb4, 0xf9b8,0x7fd9
        DCW 0x9d8e,0x51cf, 0x57b1,0x5d3e, 0xf827,0x7fc2
        DCW 0x9c11,0x4ffb, 0x571e,0x5dc8, 0xf695,0x7fa7
        DCW 0x9a9c,0x4e21, 0x568a,0x5e50, 0xf505,0x7f87
        DCW 0x9930,0x4c40, 0x55f6,0x5ed7, 0xf374,0x7f62
        DCW 0x97ce,0x4a58, 0x5560,0x5f5e, 0xf1e4,0x7f38
        DCW 0x9674,0x486a, 0x54ca,0x5fe4, 0xf055,0x7f0a
        DCW 0x9523,0x4675, 0x5433,0x6068, 0xeec6,0x7ed6
        DCW 0x93dc,0x447b, 0x539b,0x60ec, 0xed38,0x7e9d
        DCW 0x929e,0x427a, 0x5303,0x616f, 0xebab,0x7e60
        DCW 0x9169,0x4074, 0x5269,0x61f1, 0xea1e,0x7e1e
        DCW 0x903e,0x3e68, 0x51cf,0x6272, 0xe892,0x7dd6
        DCW 0x8f1d,0x3c57, 0x5134,0x62f2, 0xe707,0x7d8a
        DCW 0x8e06,0x3a40, 0x5098,0x6371, 0xe57d,0x7d3a
        DCW 0x8cf8,0x3825, 0x4ffb,0x63ef, 0xe3f4,0x7ce4
        DCW 0x8bf5,0x3604, 0x4f5e,0x646c, 0xe26d,0x7c89
        DCW 0x8afb,0x33df, 0x4ec0,0x64e9, 0xe0e6,0x7c2a
        DCW 0x8a0c,0x31b5, 0x4e21,0x6564, 0xdf61,0x7bc6
        DCW 0x8927,0x2f87, 0x4d81,0x65de, 0xdddc,0x7b5d
        DCW 0x884c,0x2d55, 0x4ce1,0x6657, 0xdc59,0x7aef
        DCW 0x877b,0x2b1f, 0x4c40,0x66d0, 0xdad8,0x7a7d
        DCW 0x86b6,0x28e5, 0x4b9e,0x6747, 0xd958,0x7a06
        DCW 0x85fa,0x26a8, 0x4afb,0x67bd, 0xd7d9,0x798a
        DCW 0x8549,0x2467, 0x4a58,0x6832, 0xd65c,0x790a
        DCW 0x84a3,0x2224, 0x49b4,0x68a7, 0xd4e1,0x7885
        DCW 0x8407,0x1fdd, 0x490f,0x691a, 0xd367,0x77fb
        DCW 0x8377,0x1d93, 0x486a,0x698c, 0xd1ef,0x776c
        DCW 0x82f1,0x1b47, 0x47c4,0x69fd, 0xd079,0x76d9
        DCW 0x8276,0x18f9, 0x471d,0x6a6e, 0xcf04,0x7642
        DCW 0x8205,0x16a8, 0x4675,0x6add, 0xcd92,0x75a6
        DCW 0x81a0,0x1455, 0x45cd,0x6b4b, 0xcc21,0x7505
        DCW 0x8146,0x1201, 0x4524,0x6bb8, 0xcab2,0x7460
        DCW 0x80f6,0x0fab, 0x447b,0x6c24, 0xc946,0x73b6
        DCW 0x80b2,0x0d54, 0x43d1,0x6c8f, 0xc7db,0x7308
        DCW 0x8079,0x0afb, 0x4326,0x6cf9, 0xc673,0x7255
        DCW 0x804b,0x08a2, 0x427a,0x6d62, 0xc50d,0x719e
        DCW 0x8027,0x0648, 0x41ce,0x6dca, 0xc3a9,0x70e3
        DCW 0x800f,0x03ed, 0x4121,0x6e31, 0xc248,0x7023
        DCW 0x8002,0x0192, 0x4074,0x6e97, 0xc0e9,0x6f5f
        DCW 0x8001,0xff37, 0x3fc6,0x6efb, 0xbf8c,0x6e97
        DCW 0x800a,0xfcdc, 0x3f17,0x6f5f, 0xbe32,0x6dca
        DCW 0x801e,0xfa81, 0x3e68,0x6fc2, 0xbcda,0x6cf9
        DCW 0x803e,0xf827, 0x3db8,0x7023, 0xbb85,0x6c24
        DCW 0x8068,0xf5cd, 0x3d08,0x7083, 0xba33,0x6b4b
        DCW 0x809e,0xf374, 0x3c57,0x70e3, 0xb8e3,0x6a6e
        DCW 0x80de,0xf11c, 0x3ba5,0x7141, 0xb796,0x698c
        DCW 0x812a,0xeec6, 0x3af3,0x719e, 0xb64c,0x68a7
        DCW 0x8181,0xec71, 0x3a40,0x71fa, 0xb505,0x67bd
        DCW 0x81e2,0xea1e, 0x398d,0x7255, 0xb3c0,0x66d0
        DCW 0x824f,0xe7cd, 0x38d9,0x72af, 0xb27f,0x65de
        DCW 0x82c6,0xe57d, 0x3825,0x7308, 0xb140,0x64e9
        DCW 0x8349,0xe330, 0x3770,0x735f, 0xb005,0x63ef
        DCW 0x83d6,0xe0e6, 0x36ba,0x73b6, 0xaecc,0x62f2
        DCW 0x846e,0xde9e, 0x3604,0x740b, 0xad97,0x61f1
        DCW 0x8511,0xdc59, 0x354e,0x7460, 0xac65,0x60ec
        DCW 0x85be,0xda18, 0x3497,0x74b3, 0xab36,0x5fe4
        DCW 0x8676,0xd7d9, 0x33df,0x7505, 0xaa0a,0x5ed7
        DCW 0x8738,0xd59e, 0x3327,0x7556, 0xa8e2,0x5dc8
        DCW 0x8805,0xd367, 0x326e,0x75a6, 0xa7bd,0x5cb4
        DCW 0x88dd,0xd134, 0x31b5,0x75f4, 0xa69c,0x5b9d
        DCW 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82
        DCW 0x8aaa,0xccd9, 0x3042,0x768e, 0xa463,0x5964
        DCW 0x8ba0,0xcab2, 0x2f87,0x76d9, 0xa34c,0x5843
        DCW 0x8ca1,0xc890, 0x2ecc,0x7723, 0xa238,0x571e
        DCW 0x8dab,0xc673, 0x2e11,0x776c, 0xa129,0x55f6
        DCW 0x8ebf,0xc45b, 0x2d55,0x77b4, 0xa01c,0x54ca
        DCW 0x8fdd,0xc248, 0x2c99,0x77fb, 0x9f14,0x539b
        DCW 0x9105,0xc03a, 0x2bdc,0x7840, 0x9e0f,0x5269
        DCW 0x9236,0xbe32, 0x2b1f,0x7885, 0x9d0e,0x5134
        DCW 0x9371,0xbc2f, 0x2a62,0x78c8, 0x9c11,0x4ffb
        DCW 0x94b5,0xba33, 0x29a4,0x790a, 0x9b17,0x4ec0
        DCW 0x9603,0xb83c, 0x28e5,0x794a, 0x9a22,0x4d81
        DCW 0x9759,0xb64c, 0x2827,0x798a, 0x9930,0x4c40
        DCW 0x98b9,0xb462, 0x2768,0x79c9, 0x9843,0x4afb
        DCW 0x9a22,0xb27f, 0x26a8,0x7a06, 0x9759,0x49b4
        DCW 0x9b94,0xb0a2, 0x25e8,0x7a42, 0x9674,0x486a
        DCW 0x9d0e,0xaecc, 0x2528,0x7a7d, 0x9592,0x471d
        DCW 0x9e91,0xacfd, 0x2467,0x7ab7, 0x94b5,0x45cd
        DCW 0xa01c,0xab36, 0x23a7,0x7aef, 0x93dc,0x447b
        DCW 0xa1b0,0xa976, 0x22e5,0x7b27, 0x9307,0x4326
        DCW 0xa34c,0xa7bd, 0x2224,0x7b5d, 0x9236,0x41ce
        DCW 0xa4f0,0xa60c, 0x2162,0x7b92, 0x9169,0x4074
        DCW 0xa69c,0xa463, 0x209f,0x7bc6, 0x90a1,0x3f17
        DCW 0xa84f,0xa2c2, 0x1fdd,0x7bf9, 0x8fdd,0x3db8
        DCW 0xaa0a,0xa129, 0x1f1a,0x7c2a, 0x8f1d,0x3c57
        DCW 0xabcd,0x9f98, 0x1e57,0x7c5a, 0x8e62,0x3af3
        DCW 0xad97,0x9e0f, 0x1d93,0x7c89, 0x8dab,0x398d
        DCW 0xaf68,0x9c8f, 0x1cd0,0x7cb7, 0x8cf8,0x3825
        DCW 0xb140,0x9b17, 0x1c0c,0x7ce4, 0x8c4a,0x36ba
        DCW 0xb31f,0x99a9, 0x1b47,0x7d0f, 0x8ba0,0x354e
        DCW 0xb505,0x9843, 0x1a83,0x7d3a, 0x8afb,0x33df
        DCW 0xb6f1,0x96e6, 0x19be,0x7d63, 0x8a5a,0x326e
        DCW 0xb8e3,0x9592, 0x18f9,0x7d8a, 0x89be,0x30fc
        DCW 0xbadc,0x9448, 0x1833,0x7db1, 0x8927,0x2f87
        DCW 0xbcda,0x9307, 0x176e,0x7dd6, 0x8894,0x2e11
        DCW 0xbedf,0x91cf, 0x16a8,0x7dfb, 0x8805,0x2c99
        DCW 0xc0e9,0x90a1, 0x15e2,0x7e1e, 0x877b,0x2b1f
        DCW 0xc2f8,0x8f7d, 0x151c,0x7e3f, 0x86f6,0x29a4
        DCW 0xc50d,0x8e62, 0x1455,0x7e60, 0x8676,0x2827
        DCW 0xc727,0x8d51, 0x138f,0x7e7f, 0x85fa,0x26a8
        DCW 0xc946,0x8c4a, 0x12c8,0x7e9d, 0x8583,0x2528
        DCW 0xcb69,0x8b4d, 0x1201,0x7eba, 0x8511,0x23a7
        DCW 0xcd92,0x8a5a, 0x113a,0x7ed6, 0x84a3,0x2224
        DCW 0xcfbe,0x8972, 0x1073,0x7ef0, 0x843a,0x209f
        DCW 0xd1ef,0x8894, 0x0fab,0x7f0a, 0x83d6,0x1f1a
        DCW 0xd424,0x87c0, 0x0ee4,0x7f22, 0x8377,0x1d93
        DCW 0xd65c,0x86f6, 0x0e1c,0x7f38, 0x831c,0x1c0c
        DCW 0xd898,0x8637, 0x0d54,0x7f4e, 0x82c6,0x1a83
        DCW 0xdad8,0x8583, 0x0c8c,0x7f62, 0x8276,0x18f9
        DCW 0xdd1b,0x84d9, 0x0bc4,0x7f75, 0x822a,0x176e
        DCW 0xdf61,0x843a, 0x0afb,0x7f87, 0x81e2,0x15e2
        DCW 0xe1a9,0x83a6, 0x0a33,0x7f98, 0x81a0,0x1455
        DCW 0xe3f4,0x831c, 0x096b,0x7fa7, 0x8163,0x12c8
        DCW 0xe642,0x829d, 0x08a2,0x7fb5, 0x812a,0x113a
        DCW 0xe892,0x822a, 0x07d9,0x7fc2, 0x80f6,0x0fab
        DCW 0xeae4,0x81c1, 0x0711,0x7fce, 0x80c8,0x0e1c
        DCW 0xed38,0x8163, 0x0648,0x7fd9, 0x809e,0x0c8c
        DCW 0xef8d,0x8110, 0x057f,0x7fe2, 0x8079,0x0afb
        DCW 0xf1e4,0x80c8, 0x04b6,0x7fea, 0x8059,0x096b
        DCW 0xf43c,0x808b, 0x03ed,0x7ff1, 0x803e,0x07d9
        DCW 0xf695,0x8059, 0x0324,0x7ff6, 0x8027,0x0648
        DCW 0xf8ef,0x8032, 0x025b,0x7ffa, 0x8016,0x04b6
        DCW 0xfb4a,0x8016, 0x0192,0x7ffe, 0x800a,0x0324
        DCW 0xfda5,0x8006, 0x00c9,0x7fff, 0x8002,0x0192
        ;// N=4096 t=2*PI*k/N for k=0,1,2,..,N/4-1
        DCW 0x7fff,0x0000, 0x7fff,0x0000, 0x7fff,0x0000
        DCW 0x7fff,0x0097, 0x7fff,0x0032, 0x7fff,0x0065
        DCW 0x7fff,0x012e, 0x7fff,0x0065, 0x7fff,0x00c9
        DCW 0x7ffd,0x01c4, 0x7fff,0x0097, 0x7fff,0x012e
        DCW 0x7ffa,0x025b, 0x7fff,0x00c9, 0x7ffe,0x0192
        DCW 0x7ff7,0x02f2, 0x7fff,0x00fb, 0x7ffc,0x01f7
        DCW 0x7ff4,0x0389, 0x7fff,0x012e, 0x7ffa,0x025b
        DCW 0x7fef,0x041f, 0x7ffe,0x0160, 0x7ff8,0x02c0
        DCW 0x7fea,0x04b6, 0x7ffe,0x0192, 0x7ff6,0x0324
        DCW 0x7fe4,0x054d, 0x7ffd,0x01c4, 0x7ff4,0x0389
        DCW 0x7fdd,0x05e3, 0x7ffc,0x01f7, 0x7ff1,0x03ed
        DCW 0x7fd6,0x067a, 0x7ffb,0x0229, 0x7fed,0x0452
        DCW 0x7fce,0x0711, 0x7ffa,0x025b, 0x7fea,0x04b6
        DCW 0x7fc5,0x07a7, 0x7ff9,0x028d, 0x7fe6,0x051b
        DCW 0x7fbc,0x083e, 0x7ff8,0x02c0, 0x7fe2,0x057f
        DCW 0x7fb2,0x08d4, 0x7ff7,0x02f2, 0x7fdd,0x05e3
        DCW 0x7fa7,0x096b, 0x7ff6,0x0324, 0x7fd9,0x0648
        DCW 0x7f9c,0x0a01, 0x7ff5,0x0356, 0x7fd3,0x06ac
        DCW 0x7f90,0x0a97, 0x7ff4,0x0389, 0x7fce,0x0711
        DCW 0x7f83,0x0b2d, 0x7ff2,0x03bb, 0x7fc8,0x0775
        DCW 0x7f75,0x0bc4, 0x7ff1,0x03ed, 0x7fc2,0x07d9
        DCW 0x7f67,0x0c5a, 0x7fef,0x041f, 0x7fbc,0x083e
        DCW 0x7f58,0x0cf0, 0x7fed,0x0452, 0x7fb5,0x08a2
        DCW 0x7f49,0x0d86, 0x7fec,0x0484, 0x7fae,0x0906
        DCW 0x7f38,0x0e1c, 0x7fea,0x04b6, 0x7fa7,0x096b
        DCW 0x7f27,0x0eb2, 0x7fe8,0x04e8, 0x7fa0,0x09cf
        DCW 0x7f16,0x0f47, 0x7fe6,0x051b, 0x7f98,0x0a33
        DCW 0x7f03,0x0fdd, 0x7fe4,0x054d, 0x7f90,0x0a97
        DCW 0x7ef0,0x1073, 0x7fe2,0x057f, 0x7f87,0x0afb
        DCW 0x7edd,0x1108, 0x7fe0,0x05b1, 0x7f7e,0x0b60
        DCW 0x7ec8,0x119e, 0x7fdd,0x05e3, 0x7f75,0x0bc4
        DCW 0x7eb3,0x1233, 0x7fdb,0x0616, 0x7f6c,0x0c28
        DCW 0x7e9d,0x12c8, 0x7fd9,0x0648, 0x7f62,0x0c8c
        DCW 0x7e87,0x135d, 0x7fd6,0x067a, 0x7f58,0x0cf0
        DCW 0x7e70,0x13f2, 0x7fd3,0x06ac, 0x7f4e,0x0d54
        DCW 0x7e58,0x1487, 0x7fd1,0x06de, 0x7f43,0x0db8
        DCW 0x7e3f,0x151c, 0x7fce,0x0711, 0x7f38,0x0e1c
        DCW 0x7e26,0x15b1, 0x7fcb,0x0743, 0x7f2d,0x0e80
        DCW 0x7e0c,0x1645, 0x7fc8,0x0775, 0x7f22,0x0ee4
        DCW 0x7df2,0x16da, 0x7fc5,0x07a7, 0x7f16,0x0f47
        DCW 0x7dd6,0x176e, 0x7fc2,0x07d9, 0x7f0a,0x0fab
        DCW 0x7dba,0x1802, 0x7fbf,0x080c, 0x7efd,0x100f
        DCW 0x7d9e,0x1896, 0x7fbc,0x083e, 0x7ef0,0x1073
        DCW 0x7d81,0x192a, 0x7fb9,0x0870, 0x7ee3,0x10d6
        DCW 0x7d63,0x19be, 0x7fb5,0x08a2, 0x7ed6,0x113a
        DCW 0x7d44,0x1a51, 0x7fb2,0x08d4, 0x7ec8,0x119e
        DCW 0x7d25,0x1ae5, 0x7fae,0x0906, 0x7eba,0x1201
        DCW 0x7d05,0x1b78, 0x7fab,0x0938, 0x7eac,0x1265
        DCW 0x7ce4,0x1c0c, 0x7fa7,0x096b, 0x7e9d,0x12c8
        DCW 0x7cc2,0x1c9f, 0x7fa3,0x099d, 0x7e8e,0x132b
        DCW 0x7ca0,0x1d31, 0x7fa0,0x09cf, 0x7e7f,0x138f
        DCW 0x7c7e,0x1dc4, 0x7f9c,0x0a01, 0x7e70,0x13f2
        DCW 0x7c5a,0x1e57, 0x7f98,0x0a33, 0x7e60,0x1455
        DCW 0x7c36,0x1ee9, 0x7f94,0x0a65, 0x7e50,0x14b9
        DCW 0x7c11,0x1f7b, 0x7f90,0x0a97, 0x7e3f,0x151c
        DCW 0x7bec,0x200e, 0x7f8b,0x0ac9, 0x7e2f,0x157f
        DCW 0x7bc6,0x209f, 0x7f87,0x0afb, 0x7e1e,0x15e2
        DCW 0x7b9f,0x2131, 0x7f83,0x0b2d, 0x7e0c,0x1645
        DCW 0x7b78,0x21c3, 0x7f7e,0x0b60, 0x7dfb,0x16a8
        DCW 0x7b50,0x2254, 0x7f7a,0x0b92, 0x7de9,0x170b
        DCW 0x7b27,0x22e5, 0x7f75,0x0bc4, 0x7dd6,0x176e
        DCW 0x7afd,0x2376, 0x7f71,0x0bf6, 0x7dc4,0x17d1
        DCW 0x7ad3,0x2407, 0x7f6c,0x0c28, 0x7db1,0x1833
        DCW 0x7aa8,0x2498, 0x7f67,0x0c5a, 0x7d9e,0x1896
        DCW 0x7a7d,0x2528, 0x7f62,0x0c8c, 0x7d8a,0x18f9
        DCW 0x7a51,0x25b8, 0x7f5d,0x0cbe, 0x7d77,0x195b
        DCW 0x7a24,0x2648, 0x7f58,0x0cf0, 0x7d63,0x19be
        DCW 0x79f7,0x26d8, 0x7f53,0x0d22, 0x7d4e,0x1a20
        DCW 0x79c9,0x2768, 0x7f4e,0x0d54, 0x7d3a,0x1a83
        DCW 0x799a,0x27f7, 0x7f49,0x0d86, 0x7d25,0x1ae5
        DCW 0x796a,0x2886, 0x7f43,0x0db8, 0x7d0f,0x1b47
        DCW 0x793a,0x2915, 0x7f3e,0x0dea, 0x7cfa,0x1ba9
        DCW 0x790a,0x29a4, 0x7f38,0x0e1c, 0x7ce4,0x1c0c
        DCW 0x78d8,0x2a32, 0x7f33,0x0e4e, 0x7cce,0x1c6e
        DCW 0x78a6,0x2ac1, 0x7f2d,0x0e80, 0x7cb7,0x1cd0
        DCW 0x7874,0x2b4f, 0x7f27,0x0eb2, 0x7ca0,0x1d31
        DCW 0x7840,0x2bdc, 0x7f22,0x0ee4, 0x7c89,0x1d93
        DCW 0x780c,0x2c6a, 0x7f1c,0x0f15, 0x7c72,0x1df5
        DCW 0x77d8,0x2cf7, 0x7f16,0x0f47, 0x7c5a,0x1e57
        DCW 0x77a2,0x2d84, 0x7f10,0x0f79, 0x7c42,0x1eb8
        DCW 0x776c,0x2e11, 0x7f0a,0x0fab, 0x7c2a,0x1f1a
        DCW 0x7736,0x2e9e, 0x7f03,0x0fdd, 0x7c11,0x1f7b
        DCW 0x76fe,0x2f2a, 0x7efd,0x100f, 0x7bf9,0x1fdd
        DCW 0x76c7,0x2fb6, 0x7ef7,0x1041, 0x7bdf,0x203e
        DCW 0x768e,0x3042, 0x7ef0,0x1073, 0x7bc6,0x209f
        DCW 0x7655,0x30cd, 0x7eea,0x10a4, 0x7bac,0x2101
        DCW 0x761b,0x3159, 0x7ee3,0x10d6, 0x7b92,0x2162
        DCW 0x75e1,0x31e4, 0x7edd,0x1108, 0x7b78,0x21c3
        DCW 0x75a6,0x326e, 0x7ed6,0x113a, 0x7b5d,0x2224
        DCW 0x756a,0x32f9, 0x7ecf,0x116c, 0x7b42,0x2284
        DCW 0x752d,0x3383, 0x7ec8,0x119e, 0x7b27,0x22e5
        DCW 0x74f0,0x340d, 0x7ec1,0x11cf, 0x7b0b,0x2346
        DCW 0x74b3,0x3497, 0x7eba,0x1201, 0x7aef,0x23a7
        DCW 0x7475,0x3520, 0x7eb3,0x1233, 0x7ad3,0x2407
        DCW 0x7436,0x35a9, 0x7eac,0x1265, 0x7ab7,0x2467
        DCW 0x73f6,0x3632, 0x7ea5,0x1296, 0x7a9a,0x24c8
        DCW 0x73b6,0x36ba, 0x7e9d,0x12c8, 0x7a7d,0x2528
        DCW 0x7375,0x3742, 0x7e96,0x12fa, 0x7a60,0x2588
        DCW 0x7334,0x37ca, 0x7e8e,0x132b, 0x7a42,0x25e8
        DCW 0x72f2,0x3852, 0x7e87,0x135d, 0x7a24,0x2648
        DCW 0x72af,0x38d9, 0x7e7f,0x138f, 0x7a06,0x26a8
        DCW 0x726c,0x3960, 0x7e78,0x13c1, 0x79e7,0x2708
        DCW 0x7228,0x39e7, 0x7e70,0x13f2, 0x79c9,0x2768
        DCW 0x71e3,0x3a6d, 0x7e68,0x1424, 0x79aa,0x27c7
        DCW 0x719e,0x3af3, 0x7e60,0x1455, 0x798a,0x2827
        DCW 0x7158,0x3b79, 0x7e58,0x1487, 0x796a,0x2886
        DCW 0x7112,0x3bfe, 0x7e50,0x14b9, 0x794a,0x28e5
        DCW 0x70cb,0x3c83, 0x7e48,0x14ea, 0x792a,0x2945
        DCW 0x7083,0x3d08, 0x7e3f,0x151c, 0x790a,0x29a4
        DCW 0x703b,0x3d8c, 0x7e37,0x154d, 0x78e9,0x2a03
        DCW 0x6ff2,0x3e10, 0x7e2f,0x157f, 0x78c8,0x2a62
        DCW 0x6fa9,0x3e94, 0x7e26,0x15b1, 0x78a6,0x2ac1
        DCW 0x6f5f,0x3f17, 0x7e1e,0x15e2, 0x7885,0x2b1f
        DCW 0x6f14,0x3f9a, 0x7e15,0x1614, 0x7863,0x2b7e
        DCW 0x6ec9,0x401d, 0x7e0c,0x1645, 0x7840,0x2bdc
        DCW 0x6e7d,0x409f, 0x7e03,0x1677, 0x781e,0x2c3b
        DCW 0x6e31,0x4121, 0x7dfb,0x16a8, 0x77fb,0x2c99
        DCW 0x6de4,0x41a3, 0x7df2,0x16da, 0x77d8,0x2cf7
        DCW 0x6d96,0x4224, 0x7de9,0x170b, 0x77b4,0x2d55
        DCW 0x6d48,0x42a5, 0x7de0,0x173c, 0x7790,0x2db3
        DCW 0x6cf9,0x4326, 0x7dd6,0x176e, 0x776c,0x2e11
        DCW 0x6caa,0x43a6, 0x7dcd,0x179f, 0x7748,0x2e6f
        DCW 0x6c5a,0x4426, 0x7dc4,0x17d1, 0x7723,0x2ecc
        DCW 0x6c09,0x44a5, 0x7dba,0x1802, 0x76fe,0x2f2a
        DCW 0x6bb8,0x4524, 0x7db1,0x1833, 0x76d9,0x2f87
        DCW 0x6b66,0x45a3, 0x7da7,0x1865, 0x76b4,0x2fe5
        DCW 0x6b14,0x4621, 0x7d9e,0x1896, 0x768e,0x3042
        DCW 0x6ac1,0x469f, 0x7d94,0x18c7, 0x7668,0x309f
        DCW 0x6a6e,0x471d, 0x7d8a,0x18f9, 0x7642,0x30fc
        DCW 0x6a1a,0x479a, 0x7d81,0x192a, 0x761b,0x3159
        DCW 0x69c5,0x4817, 0x7d77,0x195b, 0x75f4,0x31b5
        DCW 0x6970,0x4893, 0x7d6d,0x198d, 0x75cd,0x3212
        DCW 0x691a,0x490f, 0x7d63,0x19be, 0x75a6,0x326e
        DCW 0x68c4,0x498b, 0x7d58,0x19ef, 0x757e,0x32cb
        DCW 0x686d,0x4a06, 0x7d4e,0x1a20, 0x7556,0x3327
        DCW 0x6815,0x4a81, 0x7d44,0x1a51, 0x752d,0x3383
        DCW 0x67bd,0x4afb, 0x7d3a,0x1a83, 0x7505,0x33df
        DCW 0x6764,0x4b75, 0x7d2f,0x1ab4, 0x74dc,0x343b
        DCW 0x670b,0x4bef, 0x7d25,0x1ae5, 0x74b3,0x3497
        DCW 0x66b2,0x4c68, 0x7d1a,0x1b16, 0x7489,0x34f2
        DCW 0x6657,0x4ce1, 0x7d0f,0x1b47, 0x7460,0x354e
        DCW 0x65fc,0x4d59, 0x7d05,0x1b78, 0x7436,0x35a9
        DCW 0x65a1,0x4dd1, 0x7cfa,0x1ba9, 0x740b,0x3604
        DCW 0x6545,0x4e49, 0x7cef,0x1bda, 0x73e1,0x365f
        DCW 0x64e9,0x4ec0, 0x7ce4,0x1c0c, 0x73b6,0x36ba
        DCW 0x648b,0x4f37, 0x7cd9,0x1c3d, 0x738b,0x3715
        DCW 0x642e,0x4fad, 0x7cce,0x1c6e, 0x735f,0x3770
        DCW 0x63d0,0x5023, 0x7cc2,0x1c9f, 0x7334,0x37ca
        DCW 0x6371,0x5098, 0x7cb7,0x1cd0, 0x7308,0x3825
        DCW 0x6312,0x510d, 0x7cac,0x1d01, 0x72dc,0x387f
        DCW 0x62b2,0x5181, 0x7ca0,0x1d31, 0x72af,0x38d9
        DCW 0x6252,0x51f5, 0x7c95,0x1d62, 0x7282,0x3933
        DCW 0x61f1,0x5269, 0x7c89,0x1d93, 0x7255,0x398d
        DCW 0x6190,0x52dc, 0x7c7e,0x1dc4, 0x7228,0x39e7
        DCW 0x612e,0x534f, 0x7c72,0x1df5, 0x71fa,0x3a40
        DCW 0x60cb,0x53c1, 0x7c66,0x1e26, 0x71cc,0x3a9a
        DCW 0x6068,0x5433, 0x7c5a,0x1e57, 0x719e,0x3af3
        DCW 0x6005,0x54a4, 0x7c4e,0x1e88, 0x7170,0x3b4c
        DCW 0x5fa1,0x5515, 0x7c42,0x1eb8, 0x7141,0x3ba5
        DCW 0x5f3c,0x5586, 0x7c36,0x1ee9, 0x7112,0x3bfe
        DCW 0x5ed7,0x55f6, 0x7c2a,0x1f1a, 0x70e3,0x3c57
        DCW 0x5e72,0x5665, 0x7c1e,0x1f4b, 0x70b3,0x3caf
        DCW 0x5e0c,0x56d4, 0x7c11,0x1f7b, 0x7083,0x3d08
        DCW 0x5da5,0x5743, 0x7c05,0x1fac, 0x7053,0x3d60
        DCW 0x5d3e,0x57b1, 0x7bf9,0x1fdd, 0x7023,0x3db8
        DCW 0x5cd7,0x581e, 0x7bec,0x200e, 0x6ff2,0x3e10
        DCW 0x5c6f,0x588c, 0x7bdf,0x203e, 0x6fc2,0x3e68
        DCW 0x5c06,0x58f8, 0x7bd3,0x206f, 0x6f90,0x3ec0
        DCW 0x5b9d,0x5964, 0x7bc6,0x209f, 0x6f5f,0x3f17
        DCW 0x5b34,0x59d0, 0x7bb9,0x20d0, 0x6f2d,0x3f6f
        DCW 0x5ac9,0x5a3b, 0x7bac,0x2101, 0x6efb,0x3fc6
        DCW 0x5a5f,0x5aa6, 0x7b9f,0x2131, 0x6ec9,0x401d

        DCW 0x59f4,0x5b10, 0x7b92,0x2162, 0x6e97,0x4074
        DCW 0x5988,0x5b7a, 0x7b85,0x2192, 0x6e64,0x40cb
        DCW 0x591c,0x5be3, 0x7b78,0x21c3, 0x6e31,0x4121
        DCW 0x58b0,0x5c4c, 0x7b6a,0x21f3, 0x6dfe,0x4178
        DCW 0x5843,0x5cb4, 0x7b5d,0x2224, 0x6dca,0x41ce
        DCW 0x57d5,0x5d1c, 0x7b50,0x2254, 0x6d96,0x4224
        DCW 0x5767,0x5d83, 0x7b42,0x2284, 0x6d62,0x427a
        DCW 0x56f9,0x5dea, 0x7b34,0x22b5, 0x6d2e,0x42d0
        DCW 0x568a,0x5e50, 0x7b27,0x22e5, 0x6cf9,0x4326
        DCW 0x561b,0x5eb6, 0x7b19,0x2316, 0x6cc4,0x437b
        DCW 0x55ab,0x5f1b, 0x7b0b,0x2346, 0x6c8f,0x43d1
        DCW 0x553b,0x5f80, 0x7afd,0x2376, 0x6c5a,0x4426
        DCW 0x54ca,0x5fe4, 0x7aef,0x23a7, 0x6c24,0x447b
        DCW 0x5459,0x6047, 0x7ae1,0x23d7, 0x6bee,0x44d0
        DCW 0x53e7,0x60aa, 0x7ad3,0x2407, 0x6bb8,0x4524
        DCW 0x5375,0x610d, 0x7ac5,0x2437, 0x6b82,0x4579
        DCW 0x5303,0x616f, 0x7ab7,0x2467, 0x6b4b,0x45cd
        DCW 0x5290,0x61d1, 0x7aa8,0x2498, 0x6b14,0x4621
        DCW 0x521c,0x6232, 0x7a9a,0x24c8, 0x6add,0x4675
        DCW 0x51a8,0x6292, 0x7a8c,0x24f8, 0x6aa5,0x46c9
        DCW 0x5134,0x62f2, 0x7a7d,0x2528, 0x6a6e,0x471d
        DCW 0x50bf,0x6351, 0x7a6e,0x2558, 0x6a36,0x4770
        DCW 0x504a,0x63b0, 0x7a60,0x2588, 0x69fd,0x47c4
        DCW 0x4fd4,0x640f, 0x7a51,0x25b8, 0x69c5,0x4817
        DCW 0x4f5e,0x646c, 0x7a42,0x25e8, 0x698c,0x486a
        DCW 0x4ee8,0x64ca, 0x7a33,0x2618, 0x6953,0x48bd
        DCW 0x4e71,0x6526, 0x7a24,0x2648, 0x691a,0x490f
        DCW 0x4df9,0x6582, 0x7a15,0x2678, 0x68e0,0x4962
        DCW 0x4d81,0x65de, 0x7a06,0x26a8, 0x68a7,0x49b4
        DCW 0x4d09,0x6639, 0x79f7,0x26d8, 0x686d,0x4a06
        DCW 0x4c91,0x6693, 0x79e7,0x2708, 0x6832,0x4a58
        DCW 0x4c17,0x66ed, 0x79d8,0x2738, 0x67f8,0x4aaa
        DCW 0x4b9e,0x6747, 0x79c9,0x2768, 0x67bd,0x4afb
        DCW 0x4b24,0x67a0, 0x79b9,0x2797, 0x6782,0x4b4d
        DCW 0x4aaa,0x67f8, 0x79aa,0x27c7, 0x6747,0x4b9e
        DCW 0x4a2f,0x6850, 0x799a,0x27f7, 0x670b,0x4bef
        DCW 0x49b4,0x68a7, 0x798a,0x2827, 0x66d0,0x4c40
        DCW 0x4939,0x68fd, 0x797a,0x2856, 0x6693,0x4c91
        DCW 0x48bd,0x6953, 0x796a,0x2886, 0x6657,0x4ce1
        DCW 0x4840,0x69a9, 0x795b,0x28b6, 0x661b,0x4d31
        DCW 0x47c4,0x69fd, 0x794a,0x28e5, 0x65de,0x4d81
        DCW 0x4747,0x6a52, 0x793a,0x2915, 0x65a1,0x4dd1
        DCW 0x46c9,0x6aa5, 0x792a,0x2945, 0x6564,0x4e21
        DCW 0x464b,0x6af8, 0x791a,0x2974, 0x6526,0x4e71
        DCW 0x45cd,0x6b4b, 0x790a,0x29a4, 0x64e9,0x4ec0
        DCW 0x454f,0x6b9d, 0x78f9,0x29d3, 0x64ab,0x4f0f
        DCW 0x44d0,0x6bee, 0x78e9,0x2a03, 0x646c,0x4f5e
        DCW 0x4450,0x6c3f, 0x78d8,0x2a32, 0x642e,0x4fad
        DCW 0x43d1,0x6c8f, 0x78c8,0x2a62, 0x63ef,0x4ffb
        DCW 0x4351,0x6cdf, 0x78b7,0x2a91, 0x63b0,0x504a
        DCW 0x42d0,0x6d2e, 0x78a6,0x2ac1, 0x6371,0x5098
        DCW 0x424f,0x6d7c, 0x7895,0x2af0, 0x6332,0x50e6
        DCW 0x41ce,0x6dca, 0x7885,0x2b1f, 0x62f2,0x5134
        DCW 0x414d,0x6e17, 0x7874,0x2b4f, 0x62b2,0x5181
        DCW 0x40cb,0x6e64, 0x7863,0x2b7e, 0x6272,0x51cf
        DCW 0x4048,0x6eb0, 0x7851,0x2bad, 0x6232,0x521c
        DCW 0x3fc6,0x6efb, 0x7840,0x2bdc, 0x61f1,0x5269
        DCW 0x3f43,0x6f46, 0x782f,0x2c0c, 0x61b0,0x52b6
        DCW 0x3ec0,0x6f90, 0x781e,0x2c3b, 0x616f,0x5303
        DCW 0x3e3c,0x6fda, 0x780c,0x2c6a, 0x612e,0x534f
        DCW 0x3db8,0x7023, 0x77fb,0x2c99, 0x60ec,0x539b
        DCW 0x3d34,0x706b, 0x77e9,0x2cc8, 0x60aa,0x53e7
        DCW 0x3caf,0x70b3, 0x77d8,0x2cf7, 0x6068,0x5433
        DCW 0x3c2a,0x70fa, 0x77c6,0x2d26, 0x6026,0x547f
        DCW 0x3ba5,0x7141, 0x77b4,0x2d55, 0x5fe4,0x54ca
        DCW 0x3b20,0x7187, 0x77a2,0x2d84, 0x5fa1,0x5515
        DCW 0x3a9a,0x71cc, 0x7790,0x2db3, 0x5f5e,0x5560
        DCW 0x3a13,0x7211, 0x777e,0x2de2, 0x5f1b,0x55ab
        DCW 0x398d,0x7255, 0x776c,0x2e11, 0x5ed7,0x55f6
        DCW 0x3906,0x7299, 0x775a,0x2e40, 0x5e94,0x5640
        DCW 0x387f,0x72dc, 0x7748,0x2e6f, 0x5e50,0x568a
        DCW 0x37f7,0x731e, 0x7736,0x2e9e, 0x5e0c,0x56d4
        DCW 0x3770,0x735f, 0x7723,0x2ecc, 0x5dc8,0x571e
        DCW 0x36e8,0x73a0, 0x7711,0x2efb, 0x5d83,0x5767
        DCW 0x365f,0x73e1, 0x76fe,0x2f2a, 0x5d3e,0x57b1
        DCW 0x35d7,0x7421, 0x76ec,0x2f59, 0x5cf9,0x57fa
        DCW 0x354e,0x7460, 0x76d9,0x2f87, 0x5cb4,0x5843
        DCW 0x34c4,0x749e, 0x76c7,0x2fb6, 0x5c6f,0x588c
        DCW 0x343b,0x74dc, 0x76b4,0x2fe5, 0x5c29,0x58d4
        DCW 0x33b1,0x7519, 0x76a1,0x3013, 0x5be3,0x591c
        DCW 0x3327,0x7556, 0x768e,0x3042, 0x5b9d,0x5964
        DCW 0x329d,0x7592, 0x767b,0x3070, 0x5b57,0x59ac
        DCW 0x3212,0x75cd, 0x7668,0x309f, 0x5b10,0x59f4
        DCW 0x3187,0x7608, 0x7655,0x30cd, 0x5ac9,0x5a3b
        DCW 0x30fc,0x7642, 0x7642,0x30fc, 0x5a82,0x5a82
        DCW 0x3070,0x767b, 0x762e,0x312a, 0x5a3b,0x5ac9
        DCW 0x2fe5,0x76b4, 0x761b,0x3159, 0x59f4,0x5b10
        DCW 0x2f59,0x76ec, 0x7608,0x3187, 0x59ac,0x5b57
        DCW 0x2ecc,0x7723, 0x75f4,0x31b5, 0x5964,0x5b9d
        DCW 0x2e40,0x775a, 0x75e1,0x31e4, 0x591c,0x5be3
        DCW 0x2db3,0x7790, 0x75cd,0x3212, 0x58d4,0x5c29
        DCW 0x2d26,0x77c6, 0x75b9,0x3240, 0x588c,0x5c6f
        DCW 0x2c99,0x77fb, 0x75a6,0x326e, 0x5843,0x5cb4
        DCW 0x2c0c,0x782f, 0x7592,0x329d, 0x57fa,0x5cf9
        DCW 0x2b7e,0x7863, 0x757e,0x32cb, 0x57b1,0x5d3e
        DCW 0x2af0,0x7895, 0x756a,0x32f9, 0x5767,0x5d83
        DCW 0x2a62,0x78c8, 0x7556,0x3327, 0x571e,0x5dc8
        DCW 0x29d3,0x78f9, 0x7542,0x3355, 0x56d4,0x5e0c
        DCW 0x2945,0x792a, 0x752d,0x3383, 0x568a,0x5e50
        DCW 0x28b6,0x795b, 0x7519,0x33b1, 0x5640,0x5e94
        DCW 0x2827,0x798a, 0x7505,0x33df, 0x55f6,0x5ed7
        DCW 0x2797,0x79b9, 0x74f0,0x340d, 0x55ab,0x5f1b
        DCW 0x2708,0x79e7, 0x74dc,0x343b, 0x5560,0x5f5e
        DCW 0x2678,0x7a15, 0x74c7,0x3469, 0x5515,0x5fa1
        DCW 0x25e8,0x7a42, 0x74b3,0x3497, 0x54ca,0x5fe4
        DCW 0x2558,0x7a6e, 0x749e,0x34c4, 0x547f,0x6026
        DCW 0x24c8,0x7a9a, 0x7489,0x34f2, 0x5433,0x6068
        DCW 0x2437,0x7ac5, 0x7475,0x3520, 0x53e7,0x60aa
        DCW 0x23a7,0x7aef, 0x7460,0x354e, 0x539b,0x60ec
        DCW 0x2316,0x7b19, 0x744b,0x357b, 0x534f,0x612e
        DCW 0x2284,0x7b42, 0x7436,0x35a9, 0x5303,0x616f
        DCW 0x21f3,0x7b6a, 0x7421,0x35d7, 0x52b6,0x61b0
        DCW 0x2162,0x7b92, 0x740b,0x3604, 0x5269,0x61f1
        DCW 0x20d0,0x7bb9, 0x73f6,0x3632, 0x521c,0x6232
        DCW 0x203e,0x7bdf, 0x73e1,0x365f, 0x51cf,0x6272
        DCW 0x1fac,0x7c05, 0x73cb,0x368d, 0x5181,0x62b2
        DCW 0x1f1a,0x7c2a, 0x73b6,0x36ba, 0x5134,0x62f2
        DCW 0x1e88,0x7c4e, 0x73a0,0x36e8, 0x50e6,0x6332
        DCW 0x1df5,0x7c72, 0x738b,0x3715, 0x5098,0x6371
        DCW 0x1d62,0x7c95, 0x7375,0x3742, 0x504a,0x63b0
        DCW 0x1cd0,0x7cb7, 0x735f,0x3770, 0x4ffb,0x63ef
        DCW 0x1c3d,0x7cd9, 0x734a,0x379d, 0x4fad,0x642e
        DCW 0x1ba9,0x7cfa, 0x7334,0x37ca, 0x4f5e,0x646c
        DCW 0x1b16,0x7d1a, 0x731e,0x37f7, 0x4f0f,0x64ab
        DCW 0x1a83,0x7d3a, 0x7308,0x3825, 0x4ec0,0x64e9
        DCW 0x19ef,0x7d58, 0x72f2,0x3852, 0x4e71,0x6526
        DCW 0x195b,0x7d77, 0x72dc,0x387f, 0x4e21,0x6564
        DCW 0x18c7,0x7d94, 0x72c5,0x38ac, 0x4dd1,0x65a1
        DCW 0x1833,0x7db1, 0x72af,0x38d9, 0x4d81,0x65de
        DCW 0x179f,0x7dcd, 0x7299,0x3906, 0x4d31,0x661b
        DCW 0x170b,0x7de9, 0x7282,0x3933, 0x4ce1,0x6657
        DCW 0x1677,0x7e03, 0x726c,0x3960, 0x4c91,0x6693
        DCW 0x15e2,0x7e1e, 0x7255,0x398d, 0x4c40,0x66d0
        DCW 0x154d,0x7e37, 0x723f,0x39ba, 0x4bef,0x670b
        DCW 0x14b9,0x7e50, 0x7228,0x39e7, 0x4b9e,0x6747
        DCW 0x1424,0x7e68, 0x7211,0x3a13, 0x4b4d,0x6782
        DCW 0x138f,0x7e7f, 0x71fa,0x3a40, 0x4afb,0x67bd
        DCW 0x12fa,0x7e96, 0x71e3,0x3a6d, 0x4aaa,0x67f8
        DCW 0x1265,0x7eac, 0x71cc,0x3a9a, 0x4a58,0x6832
        DCW 0x11cf,0x7ec1, 0x71b5,0x3ac6, 0x4a06,0x686d
        DCW 0x113a,0x7ed6, 0x719e,0x3af3, 0x49b4,0x68a7
        DCW 0x10a4,0x7eea, 0x7187,0x3b20, 0x4962,0x68e0
        DCW 0x100f,0x7efd, 0x7170,0x3b4c, 0x490f,0x691a
        DCW 0x0f79,0x7f10, 0x7158,0x3b79, 0x48bd,0x6953
        DCW 0x0ee4,0x7f22, 0x7141,0x3ba5, 0x486a,0x698c
        DCW 0x0e4e,0x7f33, 0x712a,0x3bd2, 0x4817,0x69c5
        DCW 0x0db8,0x7f43, 0x7112,0x3bfe, 0x47c4,0x69fd
        DCW 0x0d22,0x7f53, 0x70fa,0x3c2a, 0x4770,0x6a36
        DCW 0x0c8c,0x7f62, 0x70e3,0x3c57, 0x471d,0x6a6e
        DCW 0x0bf6,0x7f71, 0x70cb,0x3c83, 0x46c9,0x6aa5
        DCW 0x0b60,0x7f7e, 0x70b3,0x3caf, 0x4675,0x6add
        DCW 0x0ac9,0x7f8b, 0x709b,0x3cdc, 0x4621,0x6b14
        DCW 0x0a33,0x7f98, 0x7083,0x3d08, 0x45cd,0x6b4b
        DCW 0x099d,0x7fa3, 0x706b,0x3d34, 0x4579,0x6b82
        DCW 0x0906,0x7fae, 0x7053,0x3d60, 0x4524,0x6bb8
        DCW 0x0870,0x7fb9, 0x703b,0x3d8c, 0x44d0,0x6bee
        DCW 0x07d9,0x7fc2, 0x7023,0x3db8, 0x447b,0x6c24
        DCW 0x0743,0x7fcb, 0x700b,0x3de4, 0x4426,0x6c5a
        DCW 0x06ac,0x7fd3, 0x6ff2,0x3e10, 0x43d1,0x6c8f
        DCW 0x0616,0x7fdb, 0x6fda,0x3e3c, 0x437b,0x6cc4
        DCW 0x057f,0x7fe2, 0x6fc2,0x3e68, 0x4326,0x6cf9
        DCW 0x04e8,0x7fe8, 0x6fa9,0x3e94, 0x42d0,0x6d2e
        DCW 0x0452,0x7fed, 0x6f90,0x3ec0, 0x427a,0x6d62
        DCW 0x03bb,0x7ff2, 0x6f78,0x3eec, 0x4224,0x6d96
        DCW 0x0324,0x7ff6, 0x6f5f,0x3f17, 0x41ce,0x6dca
        DCW 0x028d,0x7ff9, 0x6f46,0x3f43, 0x4178,0x6dfe
        DCW 0x01f7,0x7ffc, 0x6f2d,0x3f6f, 0x4121,0x6e31
        DCW 0x0160,0x7ffe, 0x6f14,0x3f9a, 0x40cb,0x6e64
        DCW 0x00c9,0x7fff, 0x6efb,0x3fc6, 0x4074,0x6e97
        DCW 0x0032,0x7fff, 0x6ee2,0x3ff1, 0x401d,0x6ec9
        DCW 0xff9b,0x7fff, 0x6ec9,0x401d, 0x3fc6,0x6efb
        DCW 0xff05,0x7fff, 0x6eb0,0x4048, 0x3f6f,0x6f2d
        DCW 0xfe6e,0x7ffe, 0x6e97,0x4074, 0x3f17,0x6f5f
        DCW 0xfdd7,0x7ffb, 0x6e7d,0x409f, 0x3ec0,0x6f90
        DCW 0xfd40,0x7ff8, 0x6e64,0x40cb, 0x3e68,0x6fc2
        DCW 0xfcaa,0x7ff5, 0x6e4a,0x40f6, 0x3e10,0x6ff2
        DCW 0xfc13,0x7ff1, 0x6e31,0x4121, 0x3db8,0x7023
        DCW 0xfb7c,0x7fec, 0x6e17,0x414d, 0x3d60,0x7053
        DCW 0xfae5,0x7fe6, 0x6dfe,0x4178, 0x3d08,0x7083
        DCW 0xfa4f,0x7fe0, 0x6de4,0x41a3, 0x3caf,0x70b3
        DCW 0xf9b8,0x7fd9, 0x6dca,0x41ce, 0x3c57,0x70e3
        DCW 0xf922,0x7fd1, 0x6db0,0x41f9, 0x3bfe,0x7112
        DCW 0xf88b,0x7fc8, 0x6d96,0x4224, 0x3ba5,0x7141
        DCW 0xf7f4,0x7fbf, 0x6d7c,0x424f, 0x3b4c,0x7170
        DCW 0xf75e,0x7fb5, 0x6d62,0x427a, 0x3af3,0x719e
        DCW 0xf6c8,0x7fab, 0x6d48,0x42a5, 0x3a9a,0x71cc
        DCW 0xf631,0x7fa0, 0x6d2e,0x42d0, 0x3a40,0x71fa
        DCW 0xf59b,0x7f94, 0x6d14,0x42fb, 0x39e7,0x7228
        DCW 0xf505,0x7f87, 0x6cf9,0x4326, 0x398d,0x7255
        DCW 0xf46e,0x7f7a, 0x6cdf,0x4351, 0x3933,0x7282
        DCW 0xf3d8,0x7f6c, 0x6cc4,0x437b, 0x38d9,0x72af
        DCW 0xf342,0x7f5d, 0x6caa,0x43a6, 0x387f,0x72dc
        DCW 0xf2ac,0x7f4e, 0x6c8f,0x43d1, 0x3825,0x7308
        DCW 0xf216,0x7f3e, 0x6c75,0x43fb, 0x37ca,0x7334
        DCW 0xf180,0x7f2d, 0x6c5a,0x4426, 0x3770,0x735f
        DCW 0xf0eb,0x7f1c, 0x6c3f,0x4450, 0x3715,0x738b
        DCW 0xf055,0x7f0a, 0x6c24,0x447b, 0x36ba,0x73b6
        DCW 0xefbf,0x7ef7, 0x6c09,0x44a5, 0x365f,0x73e1
        DCW 0xef2a,0x7ee3, 0x6bee,0x44d0, 0x3604,0x740b
        DCW 0xee94,0x7ecf, 0x6bd3,0x44fa, 0x35a9,0x7436
        DCW 0xedff,0x7eba, 0x6bb8,0x4524, 0x354e,0x7460
        DCW 0xed6a,0x7ea5, 0x6b9d,0x454f, 0x34f2,0x7489
        DCW 0xecd5,0x7e8e, 0x6b82,0x4579, 0x3497,0x74b3
        DCW 0xec3f,0x7e78, 0x6b66,0x45a3, 0x343b,0x74dc
        DCW 0xebab,0x7e60, 0x6b4b,0x45cd, 0x33df,0x7505
        DCW 0xeb16,0x7e48, 0x6b30,0x45f7, 0x3383,0x752d
        DCW 0xea81,0x7e2f, 0x6b14,0x4621, 0x3327,0x7556
        DCW 0xe9ec,0x7e15, 0x6af8,0x464b, 0x32cb,0x757e
        DCW 0xe958,0x7dfb, 0x6add,0x4675, 0x326e,0x75a6
        DCW 0xe8c4,0x7de0, 0x6ac1,0x469f, 0x3212,0x75cd
        DCW 0xe82f,0x7dc4, 0x6aa5,0x46c9, 0x31b5,0x75f4
        DCW 0xe79b,0x7da7, 0x6a89,0x46f3, 0x3159,0x761b
        DCW 0xe707,0x7d8a, 0x6a6e,0x471d, 0x30fc,0x7642
        DCW 0xe673,0x7d6d, 0x6a52,0x4747, 0x309f,0x7668
        DCW 0xe5e0,0x7d4e, 0x6a36,0x4770, 0x3042,0x768e
        DCW 0xe54c,0x7d2f, 0x6a1a,0x479a, 0x2fe5,0x76b4
        DCW 0xe4b9,0x7d0f, 0x69fd,0x47c4, 0x2f87,0x76d9
        DCW 0xe426,0x7cef, 0x69e1,0x47ed, 0x2f2a,0x76fe
        DCW 0xe392,0x7cce, 0x69c5,0x4817, 0x2ecc,0x7723
        DCW 0xe2ff,0x7cac, 0x69a9,0x4840, 0x2e6f,0x7748
        DCW 0xe26d,0x7c89, 0x698c,0x486a, 0x2e11,0x776c
        DCW 0xe1da,0x7c66, 0x6970,0x4893, 0x2db3,0x7790
        DCW 0xe148,0x7c42, 0x6953,0x48bd, 0x2d55,0x77b4
        DCW 0xe0b5,0x7c1e, 0x6937,0x48e6, 0x2cf7,0x77d8
        DCW 0xe023,0x7bf9, 0x691a,0x490f, 0x2c99,0x77fb
        DCW 0xdf91,0x7bd3, 0x68fd,0x4939, 0x2c3b,0x781e
        DCW 0xdeff,0x7bac, 0x68e0,0x4962, 0x2bdc,0x7840
        DCW 0xde6e,0x7b85, 0x68c4,0x498b, 0x2b7e,0x7863
        DCW 0xdddc,0x7b5d, 0x68a7,0x49b4, 0x2b1f,0x7885
        DCW 0xdd4b,0x7b34, 0x688a,0x49dd, 0x2ac1,0x78a6
        DCW 0xdcba,0x7b0b, 0x686d,0x4a06, 0x2a62,0x78c8
        DCW 0xdc29,0x7ae1, 0x6850,0x4a2f, 0x2a03,0x78e9
        DCW 0xdb99,0x7ab7, 0x6832,0x4a58, 0x29a4,0x790a
        DCW 0xdb08,0x7a8c, 0x6815,0x4a81, 0x2945,0x792a
        DCW 0xda78,0x7a60, 0x67f8,0x4aaa, 0x28e5,0x794a
        DCW 0xd9e8,0x7a33, 0x67da,0x4ad3, 0x2886,0x796a
        DCW 0xd958,0x7a06, 0x67bd,0x4afb, 0x2827,0x798a
        DCW 0xd8c8,0x79d8, 0x67a0,0x4b24, 0x27c7,0x79aa
        DCW 0xd839,0x79aa, 0x6782,0x4b4d, 0x2768,0x79c9
        DCW 0xd7aa,0x797a, 0x6764,0x4b75, 0x2708,0x79e7
        DCW 0xd71b,0x794a, 0x6747,0x4b9e, 0x26a8,0x7a06
        DCW 0xd68c,0x791a, 0x6729,0x4bc7, 0x2648,0x7a24
        DCW 0xd5fd,0x78e9, 0x670b,0x4bef, 0x25e8,0x7a42
        DCW 0xd56f,0x78b7, 0x66ed,0x4c17, 0x2588,0x7a60
        DCW 0xd4e1,0x7885, 0x66d0,0x4c40, 0x2528,0x7a7d
        DCW 0xd453,0x7851, 0x66b2,0x4c68, 0x24c8,0x7a9a
        DCW 0xd3c5,0x781e, 0x6693,0x4c91, 0x2467,0x7ab7
        DCW 0xd338,0x77e9, 0x6675,0x4cb9, 0x2407,0x7ad3
        DCW 0xd2ab,0x77b4, 0x6657,0x4ce1, 0x23a7,0x7aef
        DCW 0xd21e,0x777e, 0x6639,0x4d09, 0x2346,0x7b0b
        DCW 0xd191,0x7748, 0x661b,0x4d31, 0x22e5,0x7b27
        DCW 0xd105,0x7711, 0x65fc,0x4d59, 0x2284,0x7b42
        DCW 0xd079,0x76d9, 0x65de,0x4d81, 0x2224,0x7b5d
        DCW 0xcfed,0x76a1, 0x65c0,0x4da9, 0x21c3,0x7b78
        DCW 0xcf61,0x7668, 0x65a1,0x4dd1, 0x2162,0x7b92
        DCW 0xced6,0x762e, 0x6582,0x4df9, 0x2101,0x7bac
        DCW 0xce4b,0x75f4, 0x6564,0x4e21, 0x209f,0x7bc6
        DCW 0xcdc0,0x75b9, 0x6545,0x4e49, 0x203e,0x7bdf
        DCW 0xcd35,0x757e, 0x6526,0x4e71, 0x1fdd,0x7bf9
        DCW 0xccab,0x7542, 0x6507,0x4e98, 0x1f7b,0x7c11
        DCW 0xcc21,0x7505, 0x64e9,0x4ec0, 0x1f1a,0x7c2a
        DCW 0xcb97,0x74c7, 0x64ca,0x4ee8, 0x1eb8,0x7c42
        DCW 0xcb0e,0x7489, 0x64ab,0x4f0f, 0x1e57,0x7c5a
        DCW 0xca85,0x744b, 0x648b,0x4f37, 0x1df5,0x7c72
        DCW 0xc9fc,0x740b, 0x646c,0x4f5e, 0x1d93,0x7c89
        DCW 0xc973,0x73cb, 0x644d,0x4f85, 0x1d31,0x7ca0
        DCW 0xc8eb,0x738b, 0x642e,0x4fad, 0x1cd0,0x7cb7
        DCW 0xc863,0x734a, 0x640f,0x4fd4, 0x1c6e,0x7cce
        DCW 0xc7db,0x7308, 0x63ef,0x4ffb, 0x1c0c,0x7ce4
        DCW 0xc754,0x72c5, 0x63d0,0x5023, 0x1ba9,0x7cfa
        DCW 0xc6cd,0x7282, 0x63b0,0x504a, 0x1b47,0x7d0f
        DCW 0xc646,0x723f, 0x6391,0x5071, 0x1ae5,0x7d25
        DCW 0xc5c0,0x71fa, 0x6371,0x5098, 0x1a83,0x7d3a
        DCW 0xc53a,0x71b5, 0x6351,0x50bf, 0x1a20,0x7d4e
        DCW 0xc4b4,0x7170, 0x6332,0x50e6, 0x19be,0x7d63
        DCW 0xc42e,0x712a, 0x6312,0x510d, 0x195b,0x7d77
        DCW 0xc3a9,0x70e3, 0x62f2,0x5134, 0x18f9,0x7d8a
        DCW 0xc324,0x709b, 0x62d2,0x515b, 0x1896,0x7d9e
        DCW 0xc2a0,0x7053, 0x62b2,0x5181, 0x1833,0x7db1
        DCW 0xc21c,0x700b, 0x6292,0x51a8, 0x17d1,0x7dc4
        DCW 0xc198,0x6fc2, 0x6272,0x51cf, 0x176e,0x7dd6
        DCW 0xc114,0x6f78, 0x6252,0x51f5, 0x170b,0x7de9
        DCW 0xc091,0x6f2d, 0x6232,0x521c, 0x16a8,0x7dfb
        DCW 0xc00f,0x6ee2, 0x6211,0x5243, 0x1645,0x7e0c
        DCW 0xbf8c,0x6e97, 0x61f1,0x5269, 0x15e2,0x7e1e
        DCW 0xbf0a,0x6e4a, 0x61d1,0x5290, 0x157f,0x7e2f
        DCW 0xbe88,0x6dfe, 0x61b0,0x52b6, 0x151c,0x7e3f
        DCW 0xbe07,0x6db0, 0x6190,0x52dc, 0x14b9,0x7e50
        DCW 0xbd86,0x6d62, 0x616f,0x5303, 0x1455,0x7e60
        DCW 0xbd05,0x6d14, 0x614e,0x5329, 0x13f2,0x7e70
        DCW 0xbc85,0x6cc4, 0x612e,0x534f, 0x138f,0x7e7f
        DCW 0xbc05,0x6c75, 0x610d,0x5375, 0x132b,0x7e8e
        DCW 0xbb85,0x6c24, 0x60ec,0x539b, 0x12c8,0x7e9d
        DCW 0xbb06,0x6bd3, 0x60cb,0x53c1, 0x1265,0x7eac
        DCW 0xba87,0x6b82, 0x60aa,0x53e7, 0x1201,0x7eba
        DCW 0xba09,0x6b30, 0x6089,0x540d, 0x119e,0x7ec8
        DCW 0xb98b,0x6add, 0x6068,0x5433, 0x113a,0x7ed6
        DCW 0xb90d,0x6a89, 0x6047,0x5459, 0x10d6,0x7ee3
        DCW 0xb890,0x6a36, 0x6026,0x547f, 0x1073,0x7ef0
        DCW 0xb813,0x69e1, 0x6005,0x54a4, 0x100f,0x7efd
        DCW 0xb796,0x698c, 0x5fe4,0x54ca, 0x0fab,0x7f0a
        DCW 0xb71a,0x6937, 0x5fc2,0x54f0, 0x0f47,0x7f16
        DCW 0xb69e,0x68e0, 0x5fa1,0x5515, 0x0ee4,0x7f22
        DCW 0xb623,0x688a, 0x5f80,0x553b, 0x0e80,0x7f2d
        DCW 0xb5a8,0x6832, 0x5f5e,0x5560, 0x0e1c,0x7f38
        DCW 0xb52d,0x67da, 0x5f3c,0x5586, 0x0db8,0x7f43
        DCW 0xb4b3,0x6782, 0x5f1b,0x55ab, 0x0d54,0x7f4e
        DCW 0xb439,0x6729, 0x5ef9,0x55d0, 0x0cf0,0x7f58
        DCW 0xb3c0,0x66d0, 0x5ed7,0x55f6, 0x0c8c,0x7f62
        DCW 0xb347,0x6675, 0x5eb6,0x561b, 0x0c28,0x7f6c
        DCW 0xb2cf,0x661b, 0x5e94,0x5640, 0x0bc4,0x7f75
        DCW 0xb257,0x65c0, 0x5e72,0x5665, 0x0b60,0x7f7e
        DCW 0xb1df,0x6564, 0x5e50,0x568a, 0x0afb,0x7f87
        DCW 0xb168,0x6507, 0x5e2e,0x56af, 0x0a97,0x7f90
        DCW 0xb0f1,0x64ab, 0x5e0c,0x56d4, 0x0a33,0x7f98
        DCW 0xb07b,0x644d, 0x5dea,0x56f9, 0x09cf,0x7fa0
        DCW 0xb005,0x63ef, 0x5dc8,0x571e, 0x096b,0x7fa7
        DCW 0xaf8f,0x6391, 0x5da5,0x5743, 0x0906,0x7fae
        DCW 0xaf1a,0x6332, 0x5d83,0x5767, 0x08a2,0x7fb5
        DCW 0xaea5,0x62d2, 0x5d61,0x578c, 0x083e,0x7fbc
        DCW 0xae31,0x6272, 0x5d3e,0x57b1, 0x07d9,0x7fc2
        DCW 0xadbd,0x6211, 0x5d1c,0x57d5, 0x0775,0x7fc8
        DCW 0xad4a,0x61b0, 0x5cf9,0x57fa, 0x0711,0x7fce
        DCW 0xacd7,0x614e, 0x5cd7,0x581e, 0x06ac,0x7fd3
        DCW 0xac65,0x60ec, 0x5cb4,0x5843, 0x0648,0x7fd9
        DCW 0xabf3,0x6089, 0x5c91,0x5867, 0x05e3,0x7fdd
        DCW 0xab81,0x6026, 0x5c6f,0x588c, 0x057f,0x7fe2
        DCW 0xab10,0x5fc2, 0x5c4c,0x58b0, 0x051b,0x7fe6
        DCW 0xaaa0,0x5f5e, 0x5c29,0x58d4, 0x04b6,0x7fea
        DCW 0xaa30,0x5ef9, 0x5c06,0x58f8, 0x0452,0x7fed
        DCW 0xa9c0,0x5e94, 0x5be3,0x591c, 0x03ed,0x7ff1
        DCW 0xa951,0x5e2e, 0x5bc0,0x5940, 0x0389,0x7ff4
        DCW 0xa8e2,0x5dc8, 0x5b9d,0x5964, 0x0324,0x7ff6
        DCW 0xa874,0x5d61, 0x5b7a,0x5988, 0x02c0,0x7ff8
        DCW 0xa806,0x5cf9, 0x5b57,0x59ac, 0x025b,0x7ffa
        DCW 0xa799,0x5c91, 0x5b34,0x59d0, 0x01f7,0x7ffc
        DCW 0xa72c,0x5c29, 0x5b10,0x59f4, 0x0192,0x7ffe
        DCW 0xa6c0,0x5bc0, 0x5aed,0x5a18, 0x012e,0x7fff
        DCW 0xa654,0x5b57, 0x5ac9,0x5a3b, 0x00c9,0x7fff
        DCW 0xa5e8,0x5aed, 0x5aa6,0x5a5f, 0x0065,0x7fff
        DCW 0xa57e,0x5a82, 0x5a82,0x5a82, 0x0000,0x7fff
        DCW 0xa513,0x5a18, 0x5a5f,0x5aa6, 0xff9b,0x7fff
        DCW 0xa4a9,0x59ac, 0x5a3b,0x5ac9, 0xff37,0x7fff
        DCW 0xa440,0x5940, 0x5a18,0x5aed, 0xfed2,0x7fff
        DCW 0xa3d7,0x58d4, 0x59f4,0x5b10, 0xfe6e,0x7ffe
        DCW 0xa36f,0x5867, 0x59d0,0x5b34, 0xfe09,0x7ffc
        DCW 0xa307,0x57fa, 0x59ac,0x5b57, 0xfda5,0x7ffa
        DCW 0xa29f,0x578c, 0x5988,0x5b7a, 0xfd40,0x7ff8
        DCW 0xa238,0x571e, 0x5964,0x5b9d, 0xfcdc,0x7ff6
        DCW 0xa1d2,0x56af, 0x5940,0x5bc0, 0xfc77,0x7ff4
        DCW 0xa16c,0x5640, 0x591c,0x5be3, 0xfc13,0x7ff1
        DCW 0xa107,0x55d0, 0x58f8,0x5c06, 0xfbae,0x7fed
        DCW 0xa0a2,0x5560, 0x58d4,0x5c29, 0xfb4a,0x7fea
        DCW 0xa03e,0x54f0, 0x58b0,0x5c4c, 0xfae5,0x7fe6
        DCW 0x9fda,0x547f, 0x588c,0x5c6f, 0xfa81,0x7fe2
        DCW 0x9f77,0x540d, 0x5867,0x5c91, 0xfa1d,0x7fdd
        DCW 0x9f14,0x539b, 0x5843,0x5cb4, 0xf9b8,0x7fd9
        DCW 0x9eb2,0x5329, 0x581e,0x5cd7, 0xf954,0x7fd3
        DCW 0x9e50,0x52b6, 0x57fa,0x5cf9, 0xf8ef,0x7fce
        DCW 0x9def,0x5243, 0x57d5,0x5d1c, 0xf88b,0x7fc8
        DCW 0x9d8e,0x51cf, 0x57b1,0x5d3e, 0xf827,0x7fc2
        DCW 0x9d2e,0x515b, 0x578c,0x5d61, 0xf7c2,0x7fbc
        DCW 0x9cce,0x50e6, 0x5767,0x5d83, 0xf75e,0x7fb5
        DCW 0x9c6f,0x5071, 0x5743,0x5da5, 0xf6fa,0x7fae
        DCW 0x9c11,0x4ffb, 0x571e,0x5dc8, 0xf695,0x7fa7
        DCW 0x9bb3,0x4f85, 0x56f9,0x5dea, 0xf631,0x7fa0
        DCW 0x9b55,0x4f0f, 0x56d4,0x5e0c, 0xf5cd,0x7f98
        DCW 0x9af9,0x4e98, 0x56af,0x5e2e, 0xf569,0x7f90
        DCW 0x9a9c,0x4e21, 0x568a,0x5e50, 0xf505,0x7f87
        DCW 0x9a40,0x4da9, 0x5665,0x5e72, 0xf4a0,0x7f7e
        DCW 0x99e5,0x4d31, 0x5640,0x5e94, 0xf43c,0x7f75
        DCW 0x998b,0x4cb9, 0x561b,0x5eb6, 0xf3d8,0x7f6c
        DCW 0x9930,0x4c40, 0x55f6,0x5ed7, 0xf374,0x7f62
        DCW 0x98d7,0x4bc7, 0x55d0,0x5ef9, 0xf310,0x7f58
        DCW 0x987e,0x4b4d, 0x55ab,0x5f1b, 0xf2ac,0x7f4e
        DCW 0x9826,0x4ad3, 0x5586,0x5f3c, 0xf248,0x7f43
        DCW 0x97ce,0x4a58, 0x5560,0x5f5e, 0xf1e4,0x7f38
        DCW 0x9776,0x49dd, 0x553b,0x5f80, 0xf180,0x7f2d
        DCW 0x9720,0x4962, 0x5515,0x5fa1, 0xf11c,0x7f22
        DCW 0x96c9,0x48e6, 0x54f0,0x5fc2, 0xf0b9,0x7f16
        DCW 0x9674,0x486a, 0x54ca,0x5fe4, 0xf055,0x7f0a
        DCW 0x961f,0x47ed, 0x54a4,0x6005, 0xeff1,0x7efd
        DCW 0x95ca,0x4770, 0x547f,0x6026, 0xef8d,0x7ef0
        DCW 0x9577,0x46f3, 0x5459,0x6047, 0xef2a,0x7ee3
        DCW 0x9523,0x4675, 0x5433,0x6068, 0xeec6,0x7ed6
        DCW 0x94d0,0x45f7, 0x540d,0x6089, 0xee62,0x7ec8
        DCW 0x947e,0x4579, 0x53e7,0x60aa, 0xedff,0x7eba
        DCW 0x942d,0x44fa, 0x53c1,0x60cb, 0xed9b,0x7eac
        DCW 0x93dc,0x447b, 0x539b,0x60ec, 0xed38,0x7e9d
        DCW 0x938b,0x43fb, 0x5375,0x610d, 0xecd5,0x7e8e
        DCW 0x933c,0x437b, 0x534f,0x612e, 0xec71,0x7e7f
        DCW 0x92ec,0x42fb, 0x5329,0x614e, 0xec0e,0x7e70
        DCW 0x929e,0x427a, 0x5303,0x616f, 0xebab,0x7e60
        DCW 0x9250,0x41f9, 0x52dc,0x6190, 0xeb47,0x7e50
        DCW 0x9202,0x4178, 0x52b6,0x61b0, 0xeae4,0x7e3f
        DCW 0x91b6,0x40f6, 0x5290,0x61d1, 0xea81,0x7e2f
        DCW 0x9169,0x4074, 0x5269,0x61f1, 0xea1e,0x7e1e
        DCW 0x911e,0x3ff1, 0x5243,0x6211, 0xe9bb,0x7e0c
        DCW 0x90d3,0x3f6f, 0x521c,0x6232, 0xe958,0x7dfb
        DCW 0x9088,0x3eec, 0x51f5,0x6252, 0xe8f5,0x7de9
        DCW 0x903e,0x3e68, 0x51cf,0x6272, 0xe892,0x7dd6
        DCW 0x8ff5,0x3de4, 0x51a8,0x6292, 0xe82f,0x7dc4
        DCW 0x8fad,0x3d60, 0x5181,0x62b2, 0xe7cd,0x7db1
        DCW 0x8f65,0x3cdc, 0x515b,0x62d2, 0xe76a,0x7d9e
        DCW 0x8f1d,0x3c57, 0x5134,0x62f2, 0xe707,0x7d8a
        DCW 0x8ed6,0x3bd2, 0x510d,0x6312, 0xe6a5,0x7d77
        DCW 0x8e90,0x3b4c, 0x50e6,0x6332, 0xe642,0x7d63
        DCW 0x8e4b,0x3ac6, 0x50bf,0x6351, 0xe5e0,0x7d4e
        DCW 0x8e06,0x3a40, 0x5098,0x6371, 0xe57d,0x7d3a
        DCW 0x8dc1,0x39ba, 0x5071,0x6391, 0xe51b,0x7d25
        DCW 0x8d7e,0x3933, 0x504a,0x63b0, 0xe4b9,0x7d0f
        DCW 0x8d3b,0x38ac, 0x5023,0x63d0, 0xe457,0x7cfa
        DCW 0x8cf8,0x3825, 0x4ffb,0x63ef, 0xe3f4,0x7ce4
        DCW 0x8cb6,0x379d, 0x4fd4,0x640f, 0xe392,0x7cce
        DCW 0x8c75,0x3715, 0x4fad,0x642e, 0xe330,0x7cb7
        DCW 0x8c35,0x368d, 0x4f85,0x644d, 0xe2cf,0x7ca0
        DCW 0x8bf5,0x3604, 0x4f5e,0x646c, 0xe26d,0x7c89
        DCW 0x8bb5,0x357b, 0x4f37,0x648b, 0xe20b,0x7c72
        DCW 0x8b77,0x34f2, 0x4f0f,0x64ab, 0xe1a9,0x7c5a
        DCW 0x8b39,0x3469, 0x4ee8,0x64ca, 0xe148,0x7c42
        DCW 0x8afb,0x33df, 0x4ec0,0x64e9, 0xe0e6,0x7c2a
        DCW 0x8abe,0x3355, 0x4e98,0x6507, 0xe085,0x7c11
        DCW 0x8a82,0x32cb, 0x4e71,0x6526, 0xe023,0x7bf9
        DCW 0x8a47,0x3240, 0x4e49,0x6545, 0xdfc2,0x7bdf
        DCW 0x8a0c,0x31b5, 0x4e21,0x6564, 0xdf61,0x7bc6
        DCW 0x89d2,0x312a, 0x4df9,0x6582, 0xdeff,0x7bac
        DCW 0x8998,0x309f, 0x4dd1,0x65a1, 0xde9e,0x7b92
        DCW 0x895f,0x3013, 0x4da9,0x65c0, 0xde3d,0x7b78
        DCW 0x8927,0x2f87, 0x4d81,0x65de, 0xdddc,0x7b5d
        DCW 0x88ef,0x2efb, 0x4d59,0x65fc, 0xdd7c,0x7b42
        DCW 0x88b8,0x2e6f, 0x4d31,0x661b, 0xdd1b,0x7b27
        DCW 0x8882,0x2de2, 0x4d09,0x6639, 0xdcba,0x7b0b
        DCW 0x884c,0x2d55, 0x4ce1,0x6657, 0xdc59,0x7aef
        DCW 0x8817,0x2cc8, 0x4cb9,0x6675, 0xdbf9,0x7ad3
        DCW 0x87e2,0x2c3b, 0x4c91,0x6693, 0xdb99,0x7ab7
        DCW 0x87af,0x2bad, 0x4c68,0x66b2, 0xdb38,0x7a9a
        DCW 0x877b,0x2b1f, 0x4c40,0x66d0, 0xdad8,0x7a7d
        DCW 0x8749,0x2a91, 0x4c17,0x66ed, 0xda78,0x7a60
        DCW 0x8717,0x2a03, 0x4bef,0x670b, 0xda18,0x7a42
        DCW 0x86e6,0x2974, 0x4bc7,0x6729, 0xd9b8,0x7a24
        DCW 0x86b6,0x28e5, 0x4b9e,0x6747, 0xd958,0x7a06
        DCW 0x8686,0x2856, 0x4b75,0x6764, 0xd8f8,0x79e7
        DCW 0x8656,0x27c7, 0x4b4d,0x6782, 0xd898,0x79c9
        DCW 0x8628,0x2738, 0x4b24,0x67a0, 0xd839,0x79aa
        DCW 0x85fa,0x26a8, 0x4afb,0x67bd, 0xd7d9,0x798a
        DCW 0x85cd,0x2618, 0x4ad3,0x67da, 0xd77a,0x796a
        DCW 0x85a0,0x2588, 0x4aaa,0x67f8, 0xd71b,0x794a
        DCW 0x8574,0x24f8, 0x4a81,0x6815, 0xd6bb,0x792a
        DCW 0x8549,0x2467, 0x4a58,0x6832, 0xd65c,0x790a
        DCW 0x851f,0x23d7, 0x4a2f,0x6850, 0xd5fd,0x78e9
        DCW 0x84f5,0x2346, 0x4a06,0x686d, 0xd59e,0x78c8
        DCW 0x84cc,0x22b5, 0x49dd,0x688a, 0xd53f,0x78a6
        DCW 0x84a3,0x2224, 0x49b4,0x68a7, 0xd4e1,0x7885
        DCW 0x847b,0x2192, 0x498b,0x68c4, 0xd482,0x7863
        DCW 0x8454,0x2101, 0x4962,0x68e0, 0xd424,0x7840
        DCW 0x842d,0x206f, 0x4939,0x68fd, 0xd3c5,0x781e
        DCW 0x8407,0x1fdd, 0x490f,0x691a, 0xd367,0x77fb
        DCW 0x83e2,0x1f4b, 0x48e6,0x6937, 0xd309,0x77d8
        DCW 0x83be,0x1eb8, 0x48bd,0x6953, 0xd2ab,0x77b4
        DCW 0x839a,0x1e26, 0x4893,0x6970, 0xd24d,0x7790
        DCW 0x8377,0x1d93, 0x486a,0x698c, 0xd1ef,0x776c
        DCW 0x8354,0x1d01, 0x4840,0x69a9, 0xd191,0x7748
        DCW 0x8332,0x1c6e, 0x4817,0x69c5, 0xd134,0x7723
        DCW 0x8311,0x1bda, 0x47ed,0x69e1, 0xd0d6,0x76fe
        DCW 0x82f1,0x1b47, 0x47c4,0x69fd, 0xd079,0x76d9
        DCW 0x82d1,0x1ab4, 0x479a,0x6a1a, 0xd01b,0x76b4
        DCW 0x82b2,0x1a20, 0x4770,0x6a36, 0xcfbe,0x768e
        DCW 0x8293,0x198d, 0x4747,0x6a52, 0xcf61,0x7668
        DCW 0x8276,0x18f9, 0x471d,0x6a6e, 0xcf04,0x7642
        DCW 0x8259,0x1865, 0x46f3,0x6a89, 0xcea7,0x761b
        DCW 0x823c,0x17d1, 0x46c9,0x6aa5, 0xce4b,0x75f4
        DCW 0x8220,0x173c, 0x469f,0x6ac1, 0xcdee,0x75cd
        DCW 0x8205,0x16a8, 0x4675,0x6add, 0xcd92,0x75a6
        DCW 0x81eb,0x1614, 0x464b,0x6af8, 0xcd35,0x757e
        DCW 0x81d1,0x157f, 0x4621,0x6b14, 0xccd9,0x7556
        DCW 0x81b8,0x14ea, 0x45f7,0x6b30, 0xcc7d,0x752d
        DCW 0x81a0,0x1455, 0x45cd,0x6b4b, 0xcc21,0x7505
        DCW 0x8188,0x13c1, 0x45a3,0x6b66, 0xcbc5,0x74dc
        DCW 0x8172,0x132b, 0x4579,0x6b82, 0xcb69,0x74b3
        DCW 0x815b,0x1296, 0x454f,0x6b9d, 0xcb0e,0x7489
        DCW 0x8146,0x1201, 0x4524,0x6bb8, 0xcab2,0x7460
        DCW 0x8131,0x116c, 0x44fa,0x6bd3, 0xca57,0x7436
        DCW 0x811d,0x10d6, 0x44d0,0x6bee, 0xc9fc,0x740b
        DCW 0x8109,0x1041, 0x44a5,0x6c09, 0xc9a1,0x73e1
        DCW 0x80f6,0x0fab, 0x447b,0x6c24, 0xc946,0x73b6
        DCW 0x80e4,0x0f15, 0x4450,0x6c3f, 0xc8eb,0x738b
        DCW 0x80d3,0x0e80, 0x4426,0x6c5a, 0xc890,0x735f
        DCW 0x80c2,0x0dea, 0x43fb,0x6c75, 0xc836,0x7334
        DCW 0x80b2,0x0d54, 0x43d1,0x6c8f, 0xc7db,0x7308
        DCW 0x80a3,0x0cbe, 0x43a6,0x6caa, 0xc781,0x72dc
        DCW 0x8094,0x0c28, 0x437b,0x6cc4, 0xc727,0x72af
        DCW 0x8086,0x0b92, 0x4351,0x6cdf, 0xc6cd,0x7282
        DCW 0x8079,0x0afb, 0x4326,0x6cf9, 0xc673,0x7255
        DCW 0x806c,0x0a65, 0x42fb,0x6d14, 0xc619,0x7228
        DCW 0x8060,0x09cf, 0x42d0,0x6d2e, 0xc5c0,0x71fa
        DCW 0x8055,0x0938, 0x42a5,0x6d48, 0xc566,0x71cc
        DCW 0x804b,0x08a2, 0x427a,0x6d62, 0xc50d,0x719e
        DCW 0x8041,0x080c, 0x424f,0x6d7c, 0xc4b4,0x7170
        DCW 0x8038,0x0775, 0x4224,0x6d96, 0xc45b,0x7141
        DCW 0x802f,0x06de, 0x41f9,0x6db0, 0xc402,0x7112
        DCW 0x8027,0x0648, 0x41ce,0x6dca, 0xc3a9,0x70e3
        DCW 0x8020,0x05b1, 0x41a3,0x6de4, 0xc351,0x70b3
        DCW 0x801a,0x051b, 0x4178,0x6dfe, 0xc2f8,0x7083
        DCW 0x8014,0x0484, 0x414d,0x6e17, 0xc2a0,0x7053
        DCW 0x800f,0x03ed, 0x4121,0x6e31, 0xc248,0x7023
        DCW 0x800b,0x0356, 0x40f6,0x6e4a, 0xc1f0,0x6ff2
        DCW 0x8008,0x02c0, 0x40cb,0x6e64, 0xc198,0x6fc2
        DCW 0x8005,0x0229, 0x409f,0x6e7d, 0xc140,0x6f90
        DCW 0x8002,0x0192, 0x4074,0x6e97, 0xc0e9,0x6f5f
        DCW 0x8001,0x00fb, 0x4048,0x6eb0, 0xc091,0x6f2d
        DCW 0x8000,0x0065, 0x401d,0x6ec9, 0xc03a,0x6efb
        DCW 0x8000,0xffce, 0x3ff1,0x6ee2, 0xbfe3,0x6ec9
        DCW 0x8001,0xff37, 0x3fc6,0x6efb, 0xbf8c,0x6e97
        DCW 0x8002,0xfea0, 0x3f9a,0x6f14, 0xbf35,0x6e64
        DCW 0x8004,0xfe09, 0x3f6f,0x6f2d, 0xbedf,0x6e31
        DCW 0x8007,0xfd73, 0x3f43,0x6f46, 0xbe88,0x6dfe
        DCW 0x800a,0xfcdc, 0x3f17,0x6f5f, 0xbe32,0x6dca
        DCW 0x800e,0xfc45, 0x3eec,0x6f78, 0xbddc,0x6d96
        DCW 0x8013,0xfbae, 0x3ec0,0x6f90, 0xbd86,0x6d62
        DCW 0x8018,0xfb18, 0x3e94,0x6fa9, 0xbd30,0x6d2e
        DCW 0x801e,0xfa81, 0x3e68,0x6fc2, 0xbcda,0x6cf9
        DCW 0x8025,0xf9ea, 0x3e3c,0x6fda, 0xbc85,0x6cc4
        DCW 0x802d,0xf954, 0x3e10,0x6ff2, 0xbc2f,0x6c8f
        DCW 0x8035,0xf8bd, 0x3de4,0x700b, 0xbbda,0x6c5a
        DCW 0x803e,0xf827, 0x3db8,0x7023, 0xbb85,0x6c24
        DCW 0x8047,0xf790, 0x3d8c,0x703b, 0xbb30,0x6bee
        DCW 0x8052,0xf6fa, 0x3d60,0x7053, 0xbadc,0x6bb8
        DCW 0x805d,0xf663, 0x3d34,0x706b, 0xba87,0x6b82
        DCW 0x8068,0xf5cd, 0x3d08,0x7083, 0xba33,0x6b4b
        DCW 0x8075,0xf537, 0x3cdc,0x709b, 0xb9df,0x6b14
        DCW 0x8082,0xf4a0, 0x3caf,0x70b3, 0xb98b,0x6add
        DCW 0x808f,0xf40a, 0x3c83,0x70cb, 0xb937,0x6aa5
        DCW 0x809e,0xf374, 0x3c57,0x70e3, 0xb8e3,0x6a6e
        DCW 0x80ad,0xf2de, 0x3c2a,0x70fa, 0xb890,0x6a36
        DCW 0x80bd,0xf248, 0x3bfe,0x7112, 0xb83c,0x69fd
        DCW 0x80cd,0xf1b2, 0x3bd2,0x712a, 0xb7e9,0x69c5
        DCW 0x80de,0xf11c, 0x3ba5,0x7141, 0xb796,0x698c
        DCW 0x80f0,0xf087, 0x3b79,0x7158, 0xb743,0x6953
        DCW 0x8103,0xeff1, 0x3b4c,0x7170, 0xb6f1,0x691a
        DCW 0x8116,0xef5c, 0x3b20,0x7187, 0xb69e,0x68e0
        DCW 0x812a,0xeec6, 0x3af3,0x719e, 0xb64c,0x68a7
        DCW 0x813f,0xee31, 0x3ac6,0x71b5, 0xb5fa,0x686d
        DCW 0x8154,0xed9b, 0x3a9a,0x71cc, 0xb5a8,0x6832
        DCW 0x816a,0xed06, 0x3a6d,0x71e3, 0xb556,0x67f8
        DCW 0x8181,0xec71, 0x3a40,0x71fa, 0xb505,0x67bd
        DCW 0x8198,0xebdc, 0x3a13,0x7211, 0xb4b3,0x6782
        DCW 0x81b0,0xeb47, 0x39e7,0x7228, 0xb462,0x6747
        DCW 0x81c9,0xeab3, 0x39ba,0x723f, 0xb411,0x670b
        DCW 0x81e2,0xea1e, 0x398d,0x7255, 0xb3c0,0x66d0
        DCW 0x81fd,0xe989, 0x3960,0x726c, 0xb36f,0x6693
        DCW 0x8217,0xe8f5, 0x3933,0x7282, 0xb31f,0x6657
        DCW 0x8233,0xe861, 0x3906,0x7299, 0xb2cf,0x661b
        DCW 0x824f,0xe7cd, 0x38d9,0x72af, 0xb27f,0x65de
        DCW 0x826c,0xe739, 0x38ac,0x72c5, 0xb22f,0x65a1
        DCW 0x8289,0xe6a5, 0x387f,0x72dc, 0xb1df,0x6564
        DCW 0x82a8,0xe611, 0x3852,0x72f2, 0xb18f,0x6526
        DCW 0x82c6,0xe57d, 0x3825,0x7308, 0xb140,0x64e9
        DCW 0x82e6,0xe4ea, 0x37f7,0x731e, 0xb0f1,0x64ab
        DCW 0x8306,0xe457, 0x37ca,0x7334, 0xb0a2,0x646c
        DCW 0x8327,0xe3c3, 0x379d,0x734a, 0xb053,0x642e
        DCW 0x8349,0xe330, 0x3770,0x735f, 0xb005,0x63ef
        DCW 0x836b,0xe29e, 0x3742,0x7375, 0xafb6,0x63b0
        DCW 0x838e,0xe20b, 0x3715,0x738b, 0xaf68,0x6371
        DCW 0x83b2,0xe178, 0x36e8,0x73a0, 0xaf1a,0x6332
        DCW 0x83d6,0xe0e6, 0x36ba,0x73b6, 0xaecc,0x62f2
        DCW 0x83fb,0xe054, 0x368d,0x73cb, 0xae7f,0x62b2
        DCW 0x8421,0xdfc2, 0x365f,0x73e1, 0xae31,0x6272
        DCW 0x8447,0xdf30, 0x3632,0x73f6, 0xade4,0x6232
        DCW 0x846e,0xde9e, 0x3604,0x740b, 0xad97,0x61f1
        DCW 0x8496,0xde0d, 0x35d7,0x7421, 0xad4a,0x61b0
        DCW 0x84be,0xdd7c, 0x35a9,0x7436, 0xacfd,0x616f
        DCW 0x84e7,0xdcea, 0x357b,0x744b, 0xacb1,0x612e
        DCW 0x8511,0xdc59, 0x354e,0x7460, 0xac65,0x60ec
        DCW 0x853b,0xdbc9, 0x3520,0x7475, 0xac19,0x60aa
        DCW 0x8566,0xdb38, 0x34f2,0x7489, 0xabcd,0x6068
        DCW 0x8592,0xdaa8, 0x34c4,0x749e, 0xab81,0x6026
        DCW 0x85be,0xda18, 0x3497,0x74b3, 0xab36,0x5fe4
        DCW 0x85eb,0xd988, 0x3469,0x74c7, 0xaaeb,0x5fa1
        DCW 0x8619,0xd8f8, 0x343b,0x74dc, 0xaaa0,0x5f5e
        DCW 0x8647,0xd869, 0x340d,0x74f0, 0xaa55,0x5f1b
        DCW 0x8676,0xd7d9, 0x33df,0x7505, 0xaa0a,0x5ed7
        DCW 0x86a5,0xd74a, 0x33b1,0x7519, 0xa9c0,0x5e94
        DCW 0x86d6,0xd6bb, 0x3383,0x752d, 0xa976,0x5e50
        DCW 0x8707,0xd62d, 0x3355,0x7542, 0xa92c,0x5e0c
        DCW 0x8738,0xd59e, 0x3327,0x7556, 0xa8e2,0x5dc8
        DCW 0x876b,0xd510, 0x32f9,0x756a, 0xa899,0x5d83
        DCW 0x879d,0xd482, 0x32cb,0x757e, 0xa84f,0x5d3e
        DCW 0x87d1,0xd3f4, 0x329d,0x7592, 0xa806,0x5cf9
        DCW 0x8805,0xd367, 0x326e,0x75a6, 0xa7bd,0x5cb4
        DCW 0x883a,0xd2da, 0x3240,0x75b9, 0xa774,0x5c6f
        DCW 0x8870,0xd24d, 0x3212,0x75cd, 0xa72c,0x5c29
        DCW 0x88a6,0xd1c0, 0x31e4,0x75e1, 0xa6e4,0x5be3
        DCW 0x88dd,0xd134, 0x31b5,0x75f4, 0xa69c,0x5b9d
        DCW 0x8914,0xd0a7, 0x3187,0x7608, 0xa654,0x5b57
        DCW 0x894c,0xd01b, 0x3159,0x761b, 0xa60c,0x5b10
        DCW 0x8985,0xcf90, 0x312a,0x762e, 0xa5c5,0x5ac9
        DCW 0x89be,0xcf04, 0x30fc,0x7642, 0xa57e,0x5a82
        DCW 0x89f8,0xce79, 0x30cd,0x7655, 0xa537,0x5a3b
        DCW 0x8a33,0xcdee, 0x309f,0x7668, 0xa4f0,0x59f4
        DCW 0x8a6e,0xcd63, 0x3070,0x767b, 0xa4a9,0x59ac
        DCW 0x8aaa,0xccd9, 0x3042,0x768e, 0xa463,0x5964
        DCW 0x8ae7,0xcc4f, 0x3013,0x76a1, 0xa41d,0x591c
        DCW 0x8b24,0xcbc5, 0x2fe5,0x76b4, 0xa3d7,0x58d4
        DCW 0x8b62,0xcb3c, 0x2fb6,0x76c7, 0xa391,0x588c
        DCW 0x8ba0,0xcab2, 0x2f87,0x76d9, 0xa34c,0x5843
        DCW 0x8bdf,0xca29, 0x2f59,0x76ec, 0xa307,0x57fa
        DCW 0x8c1f,0xc9a1, 0x2f2a,0x76fe, 0xa2c2,0x57b1
        DCW 0x8c60,0xc918, 0x2efb,0x7711, 0xa27d,0x5767
        DCW 0x8ca1,0xc890, 0x2ecc,0x7723, 0xa238,0x571e
        DCW 0x8ce2,0xc809, 0x2e9e,0x7736, 0xa1f4,0x56d4
        DCW 0x8d24,0xc781, 0x2e6f,0x7748, 0xa1b0,0x568a
        DCW 0x8d67,0xc6fa, 0x2e40,0x775a, 0xa16c,0x5640
        DCW 0x8dab,0xc673, 0x2e11,0x776c, 0xa129,0x55f6
        DCW 0x8def,0xc5ed, 0x2de2,0x777e, 0xa0e5,0x55ab
        DCW 0x8e34,0xc566, 0x2db3,0x7790, 0xa0a2,0x5560
        DCW 0x8e79,0xc4e0, 0x2d84,0x77a2, 0xa05f,0x5515
        DCW 0x8ebf,0xc45b, 0x2d55,0x77b4, 0xa01c,0x54ca
        DCW 0x8f06,0xc3d6, 0x2d26,0x77c6, 0x9fda,0x547f
        DCW 0x8f4d,0xc351, 0x2cf7,0x77d8, 0x9f98,0x5433
        DCW 0x8f95,0xc2cc, 0x2cc8,0x77e9, 0x9f56,0x53e7
        DCW 0x8fdd,0xc248, 0x2c99,0x77fb, 0x9f14,0x539b
        DCW 0x9026,0xc1c4, 0x2c6a,0x780c, 0x9ed2,0x534f
        DCW 0x9070,0xc140, 0x2c3b,0x781e, 0x9e91,0x5303
        DCW 0x90ba,0xc0bd, 0x2c0c,0x782f, 0x9e50,0x52b6
        DCW 0x9105,0xc03a, 0x2bdc,0x7840, 0x9e0f,0x5269
        DCW 0x9150,0xbfb8, 0x2bad,0x7851, 0x9dce,0x521c
        DCW 0x919c,0xbf35, 0x2b7e,0x7863, 0x9d8e,0x51cf
        DCW 0x91e9,0xbeb3, 0x2b4f,0x7874, 0x9d4e,0x5181
        DCW 0x9236,0xbe32, 0x2b1f,0x7885, 0x9d0e,0x5134
        DCW 0x9284,0xbdb1, 0x2af0,0x7895, 0x9cce,0x50e6
        DCW 0x92d2,0xbd30, 0x2ac1,0x78a6, 0x9c8f,0x5098
        DCW 0x9321,0xbcaf, 0x2a91,0x78b7, 0x9c50,0x504a
        DCW 0x9371,0xbc2f, 0x2a62,0x78c8, 0x9c11,0x4ffb
        DCW 0x93c1,0xbbb0, 0x2a32,0x78d8, 0x9bd2,0x4fad
        DCW 0x9412,0xbb30, 0x2a03,0x78e9, 0x9b94,0x4f5e
        DCW 0x9463,0xbab1, 0x29d3,0x78f9, 0x9b55,0x4f0f
        DCW 0x94b5,0xba33, 0x29a4,0x790a, 0x9b17,0x4ec0
        DCW 0x9508,0xb9b5, 0x2974,0x791a, 0x9ada,0x4e71
        DCW 0x955b,0xb937, 0x2945,0x792a, 0x9a9c,0x4e21
        DCW 0x95ae,0xb8b9, 0x2915,0x793a, 0x9a5f,0x4dd1
        DCW 0x9603,0xb83c, 0x28e5,0x794a, 0x9a22,0x4d81
        DCW 0x9657,0xb7c0, 0x28b6,0x795b, 0x99e5,0x4d31
        DCW 0x96ad,0xb743, 0x2886,0x796a, 0x99a9,0x4ce1
        DCW 0x9703,0xb6c7, 0x2856,0x797a, 0x996d,0x4c91
        DCW 0x9759,0xb64c, 0x2827,0x798a, 0x9930,0x4c40
        DCW 0x97b0,0xb5d1, 0x27f7,0x799a, 0x98f5,0x4bef
        DCW 0x9808,0xb556, 0x27c7,0x79aa, 0x98b9,0x4b9e
        DCW 0x9860,0xb4dc, 0x2797,0x79b9, 0x987e,0x4b4d
        DCW 0x98b9,0xb462, 0x2768,0x79c9, 0x9843,0x4afb
        DCW 0x9913,0xb3e9, 0x2738,0x79d8, 0x9808,0x4aaa
        DCW 0x996d,0xb36f, 0x2708,0x79e7, 0x97ce,0x4a58
        DCW 0x99c7,0xb2f7, 0x26d8,0x79f7, 0x9793,0x4a06
        DCW 0x9a22,0xb27f, 0x26a8,0x7a06, 0x9759,0x49b4
        DCW 0x9a7e,0xb207, 0x2678,0x7a15, 0x9720,0x4962
        DCW 0x9ada,0xb18f, 0x2648,0x7a24, 0x96e6,0x490f
        DCW 0x9b36,0xb118, 0x2618,0x7a33, 0x96ad,0x48bd
        DCW 0x9b94,0xb0a2, 0x25e8,0x7a42, 0x9674,0x486a
        DCW 0x9bf1,0xb02c, 0x25b8,0x7a51, 0x963b,0x4817
        DCW 0x9c50,0xafb6, 0x2588,0x7a60, 0x9603,0x47c4
        DCW 0x9caf,0xaf41, 0x2558,0x7a6e, 0x95ca,0x4770
        DCW 0x9d0e,0xaecc, 0x2528,0x7a7d, 0x9592,0x471d
        DCW 0x9d6e,0xae58, 0x24f8,0x7a8c, 0x955b,0x46c9
        DCW 0x9dce,0xade4, 0x24c8,0x7a9a, 0x9523,0x4675
        DCW 0x9e2f,0xad70, 0x2498,0x7aa8, 0x94ec,0x4621
        DCW 0x9e91,0xacfd, 0x2467,0x7ab7, 0x94b5,0x45cd
        DCW 0x9ef3,0xac8b, 0x2437,0x7ac5, 0x947e,0x4579
        DCW 0x9f56,0xac19, 0x2407,0x7ad3, 0x9448,0x4524
        DCW 0x9fb9,0xaba7, 0x23d7,0x7ae1, 0x9412,0x44d0
        DCW 0xa01c,0xab36, 0x23a7,0x7aef, 0x93dc,0x447b
        DCW 0xa080,0xaac5, 0x2376,0x7afd, 0x93a6,0x4426
        DCW 0xa0e5,0xaa55, 0x2346,0x7b0b, 0x9371,0x43d1
        DCW 0xa14a,0xa9e5, 0x2316,0x7b19, 0x933c,0x437b
        DCW 0xa1b0,0xa976, 0x22e5,0x7b27, 0x9307,0x4326
        DCW 0xa216,0xa907, 0x22b5,0x7b34, 0x92d2,0x42d0
        DCW 0xa27d,0xa899, 0x2284,0x7b42, 0x929e,0x427a
        DCW 0xa2e4,0xa82b, 0x2254,0x7b50, 0x926a,0x4224
        DCW 0xa34c,0xa7bd, 0x2224,0x7b5d, 0x9236,0x41ce
        DCW 0xa3b4,0xa750, 0x21f3,0x7b6a, 0x9202,0x4178
        DCW 0xa41d,0xa6e4, 0x21c3,0x7b78, 0x91cf,0x4121
        DCW 0xa486,0xa678, 0x2192,0x7b85, 0x919c,0x40cb
        DCW 0xa4f0,0xa60c, 0x2162,0x7b92, 0x9169,0x4074
        DCW 0xa55a,0xa5a1, 0x2131,0x7b9f, 0x9137,0x401d
        DCW 0xa5c5,0xa537, 0x2101,0x7bac, 0x9105,0x3fc6
        DCW 0xa630,0xa4cc, 0x20d0,0x7bb9, 0x90d3,0x3f6f
        DCW 0xa69c,0xa463, 0x209f,0x7bc6, 0x90a1,0x3f17
        DCW 0xa708,0xa3fa, 0x206f,0x7bd3, 0x9070,0x3ec0
        DCW 0xa774,0xa391, 0x203e,0x7bdf, 0x903e,0x3e68
        DCW 0xa7e2,0xa329, 0x200e,0x7bec, 0x900e,0x3e10
        DCW 0xa84f,0xa2c2, 0x1fdd,0x7bf9, 0x8fdd,0x3db8
        DCW 0xa8bd,0xa25b, 0x1fac,0x7c05, 0x8fad,0x3d60
        DCW 0xa92c,0xa1f4, 0x1f7b,0x7c11, 0x8f7d,0x3d08
        DCW 0xa99b,0xa18e, 0x1f4b,0x7c1e, 0x8f4d,0x3caf
        DCW 0xaa0a,0xa129, 0x1f1a,0x7c2a, 0x8f1d,0x3c57
        DCW 0xaa7a,0xa0c4, 0x1ee9,0x7c36, 0x8eee,0x3bfe
        DCW 0xaaeb,0xa05f, 0x1eb8,0x7c42, 0x8ebf,0x3ba5
        DCW 0xab5c,0x9ffb, 0x1e88,0x7c4e, 0x8e90,0x3b4c
        DCW 0xabcd,0x9f98, 0x1e57,0x7c5a, 0x8e62,0x3af3
        DCW 0xac3f,0x9f35, 0x1e26,0x7c66, 0x8e34,0x3a9a
        DCW 0xacb1,0x9ed2, 0x1df5,0x7c72, 0x8e06,0x3a40
        DCW 0xad24,0x9e70, 0x1dc4,0x7c7e, 0x8dd8,0x39e7
        DCW 0xad97,0x9e0f, 0x1d93,0x7c89, 0x8dab,0x398d
        DCW 0xae0b,0x9dae, 0x1d62,0x7c95, 0x8d7e,0x3933
        DCW 0xae7f,0x9d4e, 0x1d31,0x7ca0, 0x8d51,0x38d9
        DCW 0xaef3,0x9cee, 0x1d01,0x7cac, 0x8d24,0x387f
        DCW 0xaf68,0x9c8f, 0x1cd0,0x7cb7, 0x8cf8,0x3825
        DCW 0xafdd,0x9c30, 0x1c9f,0x7cc2, 0x8ccc,0x37ca
        DCW 0xb053,0x9bd2, 0x1c6e,0x7cce, 0x8ca1,0x3770
        DCW 0xb0c9,0x9b75, 0x1c3d,0x7cd9, 0x8c75,0x3715
        DCW 0xb140,0x9b17, 0x1c0c,0x7ce4, 0x8c4a,0x36ba
        DCW 0xb1b7,0x9abb, 0x1bda,0x7cef, 0x8c1f,0x365f
        DCW 0xb22f,0x9a5f, 0x1ba9,0x7cfa, 0x8bf5,0x3604
        DCW 0xb2a7,0x9a04, 0x1b78,0x7d05, 0x8bca,0x35a9
        DCW 0xb31f,0x99a9, 0x1b47,0x7d0f, 0x8ba0,0x354e
        DCW 0xb398,0x994e, 0x1b16,0x7d1a, 0x8b77,0x34f2
        DCW 0xb411,0x98f5, 0x1ae5,0x7d25, 0x8b4d,0x3497
        DCW 0xb48b,0x989c, 0x1ab4,0x7d2f, 0x8b24,0x343b
        DCW 0xb505,0x9843, 0x1a83,0x7d3a, 0x8afb,0x33df
        DCW 0xb57f,0x97eb, 0x1a51,0x7d44, 0x8ad3,0x3383
        DCW 0xb5fa,0x9793, 0x1a20,0x7d4e, 0x8aaa,0x3327
        DCW 0xb675,0x973c, 0x19ef,0x7d58, 0x8a82,0x32cb
        DCW 0xb6f1,0x96e6, 0x19be,0x7d63, 0x8a5a,0x326e
        DCW 0xb76d,0x9690, 0x198d,0x7d6d, 0x8a33,0x3212
        DCW 0xb7e9,0x963b, 0x195b,0x7d77, 0x8a0c,0x31b5
        DCW 0xb866,0x95e6, 0x192a,0x7d81, 0x89e5,0x3159
        DCW 0xb8e3,0x9592, 0x18f9,0x7d8a, 0x89be,0x30fc
        DCW 0xb961,0x953f, 0x18c7,0x7d94, 0x8998,0x309f
        DCW 0xb9df,0x94ec, 0x1896,0x7d9e, 0x8972,0x3042
        DCW 0xba5d,0x949a, 0x1865,0x7da7, 0x894c,0x2fe5
        DCW 0xbadc,0x9448, 0x1833,0x7db1, 0x8927,0x2f87
        DCW 0xbb5b,0x93f7, 0x1802,0x7dba, 0x8902,0x2f2a
        DCW 0xbbda,0x93a6, 0x17d1,0x7dc4, 0x88dd,0x2ecc
        DCW 0xbc5a,0x9356, 0x179f,0x7dcd, 0x88b8,0x2e6f
        DCW 0xbcda,0x9307, 0x176e,0x7dd6, 0x8894,0x2e11
        DCW 0xbd5b,0x92b8, 0x173c,0x7de0, 0x8870,0x2db3
        DCW 0xbddc,0x926a, 0x170b,0x7de9, 0x884c,0x2d55
        DCW 0xbe5d,0x921c, 0x16da,0x7df2, 0x8828,0x2cf7
        DCW 0xbedf,0x91cf, 0x16a8,0x7dfb, 0x8805,0x2c99
        DCW 0xbf61,0x9183, 0x1677,0x7e03, 0x87e2,0x2c3b
        DCW 0xbfe3,0x9137, 0x1645,0x7e0c, 0x87c0,0x2bdc
        DCW 0xc066,0x90ec, 0x1614,0x7e15, 0x879d,0x2b7e
        DCW 0xc0e9,0x90a1, 0x15e2,0x7e1e, 0x877b,0x2b1f
        DCW 0xc16c,0x9057, 0x15b1,0x7e26, 0x875a,0x2ac1
        DCW 0xc1f0,0x900e, 0x157f,0x7e2f, 0x8738,0x2a62
        DCW 0xc274,0x8fc5, 0x154d,0x7e37, 0x8717,0x2a03
        DCW 0xc2f8,0x8f7d, 0x151c,0x7e3f, 0x86f6,0x29a4
        DCW 0xc37d,0x8f35, 0x14ea,0x7e48, 0x86d6,0x2945
        DCW 0xc402,0x8eee, 0x14b9,0x7e50, 0x86b6,0x28e5
        DCW 0xc487,0x8ea8, 0x1487,0x7e58, 0x8696,0x2886
        DCW 0xc50d,0x8e62, 0x1455,0x7e60, 0x8676,0x2827
        DCW 0xc593,0x8e1d, 0x1424,0x7e68, 0x8656,0x27c7
        DCW 0xc619,0x8dd8, 0x13f2,0x7e70, 0x8637,0x2768
        DCW 0xc6a0,0x8d94, 0x13c1,0x7e78, 0x8619,0x2708
        DCW 0xc727,0x8d51, 0x138f,0x7e7f, 0x85fa,0x26a8
        DCW 0xc7ae,0x8d0e, 0x135d,0x7e87, 0x85dc,0x2648
        DCW 0xc836,0x8ccc, 0x132b,0x7e8e, 0x85be,0x25e8
        DCW 0xc8be,0x8c8b, 0x12fa,0x7e96, 0x85a0,0x2588
        DCW 0xc946,0x8c4a, 0x12c8,0x7e9d, 0x8583,0x2528
        DCW 0xc9ce,0x8c0a, 0x1296,0x7ea5, 0x8566,0x24c8
        DCW 0xca57,0x8bca, 0x1265,0x7eac, 0x8549,0x2467
        DCW 0xcae0,0x8b8b, 0x1233,0x7eb3, 0x852d,0x2407
        DCW 0xcb69,0x8b4d, 0x1201,0x7eba, 0x8511,0x23a7
        DCW 0xcbf3,0x8b10, 0x11cf,0x7ec1, 0x84f5,0x2346
        DCW 0xcc7d,0x8ad3, 0x119e,0x7ec8, 0x84d9,0x22e5
        DCW 0xcd07,0x8a96, 0x116c,0x7ecf, 0x84be,0x2284
        DCW 0xcd92,0x8a5a, 0x113a,0x7ed6, 0x84a3,0x2224
        DCW 0xce1c,0x8a1f, 0x1108,0x7edd, 0x8488,0x21c3
        DCW 0xcea7,0x89e5, 0x10d6,0x7ee3, 0x846e,0x2162
        DCW 0xcf33,0x89ab, 0x10a4,0x7eea, 0x8454,0x2101
        DCW 0xcfbe,0x8972, 0x1073,0x7ef0, 0x843a,0x209f
        DCW 0xd04a,0x8939, 0x1041,0x7ef7, 0x8421,0x203e
        DCW 0xd0d6,0x8902, 0x100f,0x7efd, 0x8407,0x1fdd
        DCW 0xd162,0x88ca, 0x0fdd,0x7f03, 0x83ef,0x1f7b
        DCW 0xd1ef,0x8894, 0x0fab,0x7f0a, 0x83d6,0x1f1a
        DCW 0xd27c,0x885e, 0x0f79,0x7f10, 0x83be,0x1eb8
        DCW 0xd309,0x8828, 0x0f47,0x7f16, 0x83a6,0x1e57
        DCW 0xd396,0x87f4, 0x0f15,0x7f1c, 0x838e,0x1df5
        DCW 0xd424,0x87c0, 0x0ee4,0x7f22, 0x8377,0x1d93
        DCW 0xd4b1,0x878c, 0x0eb2,0x7f27, 0x8360,0x1d31
        DCW 0xd53f,0x875a, 0x0e80,0x7f2d, 0x8349,0x1cd0
        DCW 0xd5ce,0x8728, 0x0e4e,0x7f33, 0x8332,0x1c6e
        DCW 0xd65c,0x86f6, 0x0e1c,0x7f38, 0x831c,0x1c0c
        DCW 0xd6eb,0x86c6, 0x0dea,0x7f3e, 0x8306,0x1ba9
        DCW 0xd77a,0x8696, 0x0db8,0x7f43, 0x82f1,0x1b47
        DCW 0xd809,0x8666, 0x0d86,0x7f49, 0x82db,0x1ae5
        DCW 0xd898,0x8637, 0x0d54,0x7f4e, 0x82c6,0x1a83
        DCW 0xd928,0x8609, 0x0d22,0x7f53, 0x82b2,0x1a20
        DCW 0xd9b8,0x85dc, 0x0cf0,0x7f58, 0x829d,0x19be
        DCW 0xda48,0x85af, 0x0cbe,0x7f5d, 0x8289,0x195b
        DCW 0xdad8,0x8583, 0x0c8c,0x7f62, 0x8276,0x18f9
        DCW 0xdb68,0x8558, 0x0c5a,0x7f67, 0x8262,0x1896
        DCW 0xdbf9,0x852d, 0x0c28,0x7f6c, 0x824f,0x1833
        DCW 0xdc8a,0x8503, 0x0bf6,0x7f71, 0x823c,0x17d1
        DCW 0xdd1b,0x84d9, 0x0bc4,0x7f75, 0x822a,0x176e
        DCW 0xddac,0x84b0, 0x0b92,0x7f7a, 0x8217,0x170b
        DCW 0xde3d,0x8488, 0x0b60,0x7f7e, 0x8205,0x16a8
        DCW 0xdecf,0x8461, 0x0b2d,0x7f83, 0x81f4,0x1645
        DCW 0xdf61,0x843a, 0x0afb,0x7f87, 0x81e2,0x15e2
        DCW 0xdff2,0x8414, 0x0ac9,0x7f8b, 0x81d1,0x157f
        DCW 0xe085,0x83ef, 0x0a97,0x7f90, 0x81c1,0x151c
        DCW 0xe117,0x83ca, 0x0a65,0x7f94, 0x81b0,0x14b9
        DCW 0xe1a9,0x83a6, 0x0a33,0x7f98, 0x81a0,0x1455
        DCW 0xe23c,0x8382, 0x0a01,0x7f9c, 0x8190,0x13f2
        DCW 0xe2cf,0x8360, 0x09cf,0x7fa0, 0x8181,0x138f
        DCW 0xe361,0x833e, 0x099d,0x7fa3, 0x8172,0x132b
        DCW 0xe3f4,0x831c, 0x096b,0x7fa7, 0x8163,0x12c8
        DCW 0xe488,0x82fb, 0x0938,0x7fab, 0x8154,0x1265
        DCW 0xe51b,0x82db, 0x0906,0x7fae, 0x8146,0x1201
        DCW 0xe5af,0x82bc, 0x08d4,0x7fb2, 0x8138,0x119e
        DCW 0xe642,0x829d, 0x08a2,0x7fb5, 0x812a,0x113a
        DCW 0xe6d6,0x827f, 0x0870,0x7fb9, 0x811d,0x10d6
        DCW 0xe76a,0x8262, 0x083e,0x7fbc, 0x8110,0x1073
        DCW 0xe7fe,0x8246, 0x080c,0x7fbf, 0x8103,0x100f
        DCW 0xe892,0x822a, 0x07d9,0x7fc2, 0x80f6,0x0fab
        DCW 0xe926,0x820e, 0x07a7,0x7fc5, 0x80ea,0x0f47
        DCW 0xe9bb,0x81f4, 0x0775,0x7fc8, 0x80de,0x0ee4
        DCW 0xea4f,0x81da, 0x0743,0x7fcb, 0x80d3,0x0e80
        DCW 0xeae4,0x81c1, 0x0711,0x7fce, 0x80c8,0x0e1c
        DCW 0xeb79,0x81a8, 0x06de,0x7fd1, 0x80bd,0x0db8
        DCW 0xec0e,0x8190, 0x06ac,0x7fd3, 0x80b2,0x0d54
        DCW 0xeca3,0x8179, 0x067a,0x7fd6, 0x80a8,0x0cf0
        DCW 0xed38,0x8163, 0x0648,0x7fd9, 0x809e,0x0c8c
        DCW 0xedcd,0x814d, 0x0616,0x7fdb, 0x8094,0x0c28
        DCW 0xee62,0x8138, 0x05e3,0x7fdd, 0x808b,0x0bc4
        DCW 0xeef8,0x8123, 0x05b1,0x7fe0, 0x8082,0x0b60
        DCW 0xef8d,0x8110, 0x057f,0x7fe2, 0x8079,0x0afb
        DCW 0xf023,0x80fd, 0x054d,0x7fe4, 0x8070,0x0a97
        DCW 0xf0b9,0x80ea, 0x051b,0x7fe6, 0x8068,0x0a33
        DCW 0xf14e,0x80d9, 0x04e8,0x7fe8, 0x8060,0x09cf
        DCW 0xf1e4,0x80c8, 0x04b6,0x7fea, 0x8059,0x096b
        DCW 0xf27a,0x80b7, 0x0484,0x7fec, 0x8052,0x0906
        DCW 0xf310,0x80a8, 0x0452,0x7fed, 0x804b,0x08a2
        DCW 0xf3a6,0x8099, 0x041f,0x7fef, 0x8044,0x083e
        DCW 0xf43c,0x808b, 0x03ed,0x7ff1, 0x803e,0x07d9
        DCW 0xf4d3,0x807d, 0x03bb,0x7ff2, 0x8038,0x0775
        DCW 0xf569,0x8070, 0x0389,0x7ff4, 0x8032,0x0711
        DCW 0xf5ff,0x8064, 0x0356,0x7ff5, 0x802d,0x06ac
        DCW 0xf695,0x8059, 0x0324,0x7ff6, 0x8027,0x0648
        DCW 0xf72c,0x804e, 0x02f2,0x7ff7, 0x8023,0x05e3
        DCW 0xf7c2,0x8044, 0x02c0,0x7ff8, 0x801e,0x057f
        DCW 0xf859,0x803b, 0x028d,0x7ff9, 0x801a,0x051b
        DCW 0xf8ef,0x8032, 0x025b,0x7ffa, 0x8016,0x04b6
        DCW 0xf986,0x802a, 0x0229,0x7ffb, 0x8013,0x0452
        DCW 0xfa1d,0x8023, 0x01f7,0x7ffc, 0x800f,0x03ed
        DCW 0xfab3,0x801c, 0x01c4,0x7ffd, 0x800c,0x0389
        DCW 0xfb4a,0x8016, 0x0192,0x7ffe, 0x800a,0x0324
        DCW 0xfbe1,0x8011, 0x0160,0x7ffe, 0x8008,0x02c0
        DCW 0xfc77,0x800c, 0x012e,0x7fff, 0x8006,0x025b
        DCW 0xfd0e,0x8009, 0x00fb,0x7fff, 0x8004,0x01f7
        DCW 0xfda5,0x8006, 0x00c9,0x7fff, 0x8002,0x0192
        DCW 0xfe3c,0x8003, 0x0097,0x7fff, 0x8001,0x012e
        DCW 0xfed2,0x8001, 0x0065,0x7fff, 0x8001,0x00c9
        DCW 0xff69,0x8000, 0x0032,0x7fff, 0x8000,0x0065
        end
