/**
 ******************************************************************************
 * @file    main.cpp
 * @date    February 07, 2020
 * @brief   mbed test application Esmacat Shield working together with 
 *          STMicroelectronics X-NUCLEO-6180XA1
 *          The Expansion Board for range measurement and als measurement.
 *The original code is by STMicroelectronics which is modified to integrate EASE.
 *The Copyright of STMicroelectronics is retained below.
 ******************************************************************************
 
  Copyright (c) 2020 https://www.esmacat.com/

  Licensed under the Apache License, Version 2.0 (the "License");
  you may not use this file except in compliance with the License.
  You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

  Unless required by applicable law or agreed to in writing, software
  distributed under the License is distributed on an "AS IS" BASIS,
  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  See the License for the specific language governing permissions and
  limitations under the License.

  EsmacatShield.h - Library for using EtherCAT Arduino Shield by Esmacat(EASE).
  Created by Esmacat, 01/22/2020
  
*******************************************************************************
* @file EsmacatShield.h
*******************************************************************************
*/

/*
   This VL6180X Expansion board test application performs a range measurement
   and an als measurement in interrupt mode on the onboard embedded top sensor. 
   The board red slider selects on the flight the measurement type as ALS or
   RANGE; the measured data is diplayed on the on bord 4digits display.

   User Blue button allows to stop current measurement and the entire program
   releasing all the resources.
   Reset button is used to restart the program.

   Polling operating modes don`t require callback function that handles IRQ 
   callbacks. IRQ functions are used only for measures that require interrupts.

   Notes:
   + get_measurement() is asynchronous! It returns NOT_READY if the measurement
     value is not ready to be read from the corresponding register. So you need
     to wait for the result to be ready.\
*/


/* Includes ------------------------------------------------------------------*/

#include "mbed.h"
#include "XNucleo6180XA1.h"
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <EsmacatShield.h>    //Esmacat : Include Esmacat Shield Library



/* Definitions ---------------------------------------------------------------*/

#define VL6180X_I2C_SDA   D14 
#define VL6180X_I2C_SCL   D15 

#define RANGE   0
#define ALS     1

#define DELAY 2000  // 2Sec


/* Types ---------------------------------------------------------------------*/


/* Esmacat communication------------------------------------------------------*/
DigitalOut selectPin(D10); // D10 is used to drive chip enable low 
SPI spi(D11, D12, D13); // mosi, miso, sclk 
EsmacatShield slave(spi, selectPin); // EsmacatShield Slave object creation
int sendToEsmacat; 
bool led_on; 
/* Esmacat communication------------------------------------------------------*/

/* Operating mode */
operating_mode_t operating_mode, prev_operating_mode;
enum op_mode_int_poll_t {
    PollMeasure,
    IntMeasure
};


/* Variables -----------------------------------------------------------------*/

/* Expansion board */
static XNucleo6180XA1 *board = NULL;

/* Measure data */
measure_data_t data_sensor_top;

/* Flags that handle interrupt request */
bool int_sensor_top = false, int_stop_measure = false;  


/* Functions -----------------------------------------------------------------*/

/* ISR callback function of the sensor_top */
void sensor_top_irq(void)
{
    int_sensor_top = true;
    board->sensor_top->disable_interrupt_measure_detection_irq();
}

/* ISR callback function of the user blue button to stop program */
void stop_measure_irq(void)
{
    int_stop_measure = true;
}

/* On board 4 digit local display refresh */
void display_refresh(operating_mode_t op_mode)
{
    char str[5];

    if (op_mode==range_continuous_interrupt || op_mode==range_continuous_polling) {
        if (data_sensor_top.range_mm!=0xFFFFFFFF) {
            sprintf(str,"%d",data_sensor_top.range_mm);
            /*Copy Sensor data to be sent to EASE*/
            sendToEsmacat = data_sensor_top.range_mm; 
        } else {
            /*Copy Sensor data to be sent to EASE*/
            sendToEsmacat = 0; // Esmacat
            sprintf(str,"%s","----");
        }
    } else if (op_mode==als_continuous_interrupt || op_mode==als_continuous_polling) {
        if (data_sensor_top.lux!=0xFFFFFFFF) {
            sprintf(str,"%d",data_sensor_top.lux);
            /*Copy Sensor data to be sent to EASE*/
            sendToEsmacat = data_sensor_top.lux; // Esmacat
        } else {
            /*If not valid data then send 0 to EASE*/
            sendToEsmacat = 0; // Esmacat
            sprintf(str,"%s","----");
        }
    }
  
    /*Toggle the LED on EASE board*/
    led_on = !led_on; 
    printf("sendToEsmacat value %d \n", sendToEsmacat); // Esmacat
    slave.write_reg_value(0,sendToEsmacat, led_on); //  Write data to EASE
          
    board->display->display_string(str, strlen(str));
}

/* On board red slider position check */
operating_mode_t check_slider(enum op_mode_int_poll_t op_mode)
{
    operating_mode_t ret;
    int measure = board->rd_switch();

    switch (op_mode) {
        case PollMeasure:
            if (measure==RANGE) {
                ret = range_continuous_polling;
            } else if (measure==ALS) {
                ret = als_continuous_polling;
            }
            break;
        case IntMeasure:
            if (measure==RANGE) {
                ret = range_continuous_interrupt;
            } else if (measure==ALS) {
                ret = als_continuous_interrupt;
            }
            break;
    }
    return ret;      
}

/* Print on USB Serial the started operating_mode_t */
void print_start_message(operating_mode_t op_mode)
{
    if (op_mode==range_continuous_interrupt) {
        printf("\nStarted range continuous interrupt measure\n\r");
    } else if (prev_operating_mode==als_continuous_interrupt) {
        printf("\nStarted als continuous interrupt measure\n\r");
    }
}

/* Print on USB Serial the stopped operating_mode_t */
void print_stop_message(operating_mode_t op_mode)
{
    if (op_mode==range_continuous_interrupt) {
        printf("Stopped range continuous interrupt measure\n\r");
    } else if (prev_operating_mode==als_continuous_interrupt) {
        printf("Stopped als continuous interrupt measure\n\r");
    }
}

/* Print on board 4 Digit display the indicated message <= 4 char */
void display_msg(const char * msg)
{
    Timer timer;
    char str[5];

    timer.start();
    for (int i=0; i<DELAY; i=timer.read_ms())
    {
        sprintf(str,"%s",msg);
        board->display->display_string(str, strlen(str));
    }
    timer.stop();
}

/* Handle continuous ALS or Range measurement. */
void int_continous_als_or_range_measure (DevI2C *device_i2c) {
   
    int status;

    /* Creates the 6180XA1 expansion board singleton obj */
#ifdef TARGET_STM32F429
    board = XNucleo6180XA1::instance(device_i2c, A5, A2, D7, D2);//Pratima D13 -> D7
#else
    board = XNucleo6180XA1::instance(device_i2c, A3, A2, D7, D2);//Pratima D13 -> D7
#endif
    display_msg("INT");
    
    /* Init the 6180XA1 expansion board with default values */
    status = board->init_board();
    if (status) {
        printf("Failed to init board!\n\r");
    }

    /* Check the red slider position for ALS/Range measure */
    operating_mode=check_slider(IntMeasure);

    /* Start the measure on sensor top */
    status = board->sensor_top->start_measurement(operating_mode, sensor_top_irq, NULL, NULL);
    if (!status) {
        prev_operating_mode=operating_mode;
        print_start_message(operating_mode);
        while (true) {
            if (int_sensor_top) { /* 6180 isr was triggered */
                int_sensor_top = false;
                status = board->sensor_top->handle_irq(operating_mode, &data_sensor_top); /* handle the isr and read the meaure */
                display_refresh(operating_mode);
            }
            if (int_stop_measure) { /* Blue Button isr was triggered */
                status = board->sensor_top->stop_measurement(prev_operating_mode); /* stop the measure and exit */
                if (!status) {
                    print_stop_message(prev_operating_mode);
                }
                int_stop_measure = false;
                printf("\nProgram stopped!\n\n\r");
                break;
            }
            operating_mode = check_slider(IntMeasure); /* check if red slider was moved */
            if (operating_mode!=prev_operating_mode) {
                display_refresh(prev_operating_mode);
                status = board->sensor_top->stop_measurement(prev_operating_mode); /* stop the running measure */
                if (!status) {
                    print_stop_message(prev_operating_mode);
                }
                prev_operating_mode = operating_mode;
                status = board->sensor_top->start_measurement(operating_mode, sensor_top_irq, NULL, NULL); /* start the new measure */
                if (!status) {
                    print_start_message(operating_mode);
                }
            } else {
                display_refresh(operating_mode);
            }
            
        }
    }
    display_msg("BYE");

}


/*=================================== Main ==================================
 Move the VL6180X Expansion board red slider to switch between ALS or Range
 measures.
 Press the blue user button to stop the measurements in progress.   
=============================================================================*/
int main()
{   
/* Esmacat communication*/
   
    slave.setup_spi();           //Setup SPI for EASE



#if USER_BUTTON==PC_13  // Cross compiling for Nucleo-F401
    InterruptIn stop_button (USER_BUTTON);
    stop_button.rise (&stop_measure_irq);  
#endif   
    DevI2C *device_i2c = new DevI2C(VL6180X_I2C_SDA, VL6180X_I2C_SCL);     

    /* Start continous measures Interrupt based */
    int_continous_als_or_range_measure (device_i2c);
}


