#include "mbed.h"
#include <CoOS.h>

Serial pc(USBTX, USBRX);

/*---------------------------------- Define  ---------------------------------*/
#define STK_SIZE        128                /*!< Stack size.                    */
#define INIT_TASK_PRIO    11                  /*!< Priority of "init_task" task.    */
#define A_TASK_PRIO        10                  /*!< Priority of "A" task.            */
#define B_TASK_PRIO        10                  /*!< Priority of "B" task.            */
#define CLK_TASK_PRIO    10                  /*!< Priority of "clock" task.        */

/*---------------------------------- Variable Define -------------------------*/
OS_STK    init_task_stk[STK_SIZE];        /*!< The stack of "init_task" task.    */
OS_STK    a_task_stk[STK_SIZE];            /*!< The stack of "a" task.            */
OS_STK    b_task_stk[STK_SIZE];            /*!< The stack of "b" task.            */
OS_STK    clk_task_stk[STK_SIZE];            /*!< The stack of "clcok" task.        */

OS_MutexID    mut_LED;                    /*!< Mutex id related to LED.        */                        

OS_FlagID    a_flg;                                 
OS_FlagID    b_flg;
U8            errInfo[32];                /*!< Save error code.                */


DigitalOut myled(LED1);

/**
 *******************************************************************************
 * @brief        Switch on LED      
 * @param[in]    None      
 * @param[out]  None     
 * @retval        None 
 * @par Description
 * @details           This function use to Switch on specify led in LCD.
 *******************************************************************************
 */
void LED_On  () 
{
    CoEnterMutexSection(mut_LED);
    myled =1;
    CoLeaveMutexSection(mut_LED);
}

/**
 *******************************************************************************
 * @brief        Switch off LED      
 * @param[in]    None       
 * @param[out]  None     
 * @retval        None 
 * @par Description
 * @details           This function use to Switch off specify led in LCD.
 *******************************************************************************
 */
void LED_Off () {
      
    CoEnterMutexSection(mut_LED);
    myled = 0;
    CoLeaveMutexSection(mut_LED);
}


/**
 *******************************************************************************
 * @brief        Function 'signal_func' called from multiple tasks     
 * @param[in]    None      
 * @param[out]  None     
 * @retval        None 
 * @par Description
 * @details           This function use to Set flag for multiple tasks.
 *******************************************************************************
 */
void signal_func (const char* fromString)  
{

    pc.printf(fromString);
    CoSetFlag(b_flg);
    CoTickDelay(50);
    CoSetFlag(b_flg);
    CoTickDelay(50);
    CoSetFlag(a_flg);
    CoTickDelay(50);
}



/**
 *******************************************************************************
 * @brief        Task 1 'phaseA': Phase A output     
 * @param[in]     pdata    A pointer to parameter passed to task.     
 * @param[out]    None   
 * @retval        None              
 *******************************************************************************
 */
void taskA (void *pdata) {
   for (;;) {
        CoWaitForSingleFlag(a_flg,0);
        LED_On();
        signal_func("\r\nSignal from Task A");                     /*!< call common signal function */
        LED_Off();
      }
}


/**
 *******************************************************************************
 * @brief        Task 2 'phaseB': Phase B output     
 * @param[in]     pdata    A pointer to parameter passed to task.     
 * @param[out]    None   
 * @retval        None              
 *******************************************************************************
 */
void taskB (void *pdata) {
      for (;;) {
    CoWaitForSingleFlag(a_flg,0);
    LED_On ();
    signal_func("\r\nSignal from Task B");                          /*!< call common signal function */
    LED_Off();
  }
}



/**
 *******************************************************************************
 * @brief        Task 5 'clock': Signal Clock     
 * @param[in]     pdata    A pointer to parameter passed to task.     
 * @param[out]    None   
 * @retval        None              
 *******************************************************************************
 */
void clock (void *pdata) {
       for (;;) {
      CoWaitForSingleFlag(b_flg,0);
    LED_On ();
      CoTickDelay(10);
    LED_Off();
 }
}


/**
 *******************************************************************************
 * @brief        "init_task" task code     
 * @param[in]     pdata    A pointer to parameter passed to task.     
 * @param[out]    None   
 * @retval        None         
 * @par Description
 * @details        This task use to create other tasks, all mutexs and flags,and so on.     
 *******************************************************************************
 */
void init_task (void *pdata)
{
    pdata = pdata;
 
    mut_LED = CoCreateMutex();
    a_flg    = CoCreateFlag(TRUE,0);    
       b_flg   = CoCreateFlag(TRUE,0);

    CoCreateTask(taskA,0,A_TASK_PRIO,&a_task_stk[STK_SIZE-1],STK_SIZE);
    CoCreateTask(taskB,0,B_TASK_PRIO,&b_task_stk[STK_SIZE-1],STK_SIZE); 
    CoCreateTask(clock,0,CLK_TASK_PRIO,&clk_task_stk[STK_SIZE-1],STK_SIZE);
    CoSetFlag(a_flg);
    CoExitTask();                                /*!< Delete "init_task" task. */             
}


/**
 *******************************************************************************
 * @brief        main function       
 * @param[in]     None     
 * @param[out]     None  
 * @retval        None     
 *******************************************************************************
 */
int main()
{
      //SystemInit();                           /*!< Initialize the MCU clocks  */

    pc.printf ("\r\n");
    pc.printf ("\r\nCooCox RTOS Demo\n");
    CoInitOS();                              /*!< Initial CooCox RTOS.       */    
    pc.printf ("\r\n CoInitOS done.");
    CoCreateTask( init_task,0,INIT_TASK_PRIO,&init_task_stk[STK_SIZE-1],STK_SIZE);
    pc.printf ("\r\n CoCreateTask  init_task done.");
    CoStartOS();
    pc.printf ("\r\n CoStartOS done.");
    
    while (1);
}