#include "DataTypes.h"
#include "GFX.h"
#include "SmartSwitch.h"
#include "FixedMath.h"

//Linking Functions to Physical Screen
//***********************************************************************************

void GFX_InitPhysicalScreen()
{
     InitSmartSwitch();
}

void GFX_DumpRenderContextToPhysicalScreen(RenderContext *Image)
{
   int x,y;
   
   SmartSwitch_SS = 0;
   SmartSwitchWriteByte(SMART_SWITCH_CMD_DISPLAY_DATA);
   
   for(y=0;y<PHYSICAL_DISPLAY_YRES;y++)
   {
       for(x=0;x<PHYSICAL_DISPLAY_X_WIDTH_IN_BYTES;x++)
       {
          SmartSwitchWriteByte(Image->RenderPlane.BitPlaneSpace[(y*PHYSICAL_DISPLAY_X_WIDTH_IN_BYTES) + (PHYSICAL_DISPLAY_X_WIDTH_IN_BYTES - x - 1)]);
       }
   }
   SmartSwitch_SS = 1;
}

void GFX_PowerUpScreen()
{
    PowerUpSmartSwitch();
}

void GFX_PowerDownScreen()
{
    PowerDownSmartSwitch();
}

//Device Independent Functions
//***********************************************************************************

//Reserve Space for the backbuffer

RenderContext BackBuffer;
BYTE BackBufferRenderPlaneSpace[PHYSICAL_DISPLAY_PLANE_BUFFER_SIZE];


//FontData

#define FONT5x7_FONT_WIDTH 5
#define FONT5x7_FONT_HEIGHT 8
#define FONT5x7_FONT_ELEMENTS 128
#define FONT5x7_FONT_COLUMN_SIZE_IN_BYTE  1

BYTE FontTable_Font5x7 [640] = {
0x00 ,0x08 ,0x0C ,0xFA ,0x81 ,0xFA ,0x0C ,0x08 ,0x00 ,0x00 ,0x00 ,0x10 ,0x30 ,0x5F ,0x81 ,0x5F ,
0x30 ,0x10 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0xBE ,0x00 ,0x00 ,0x00 ,0x00 ,0x06 ,0x00 ,0x06 ,0x00 ,0x00 ,0x28 ,
0xFE ,0x28 ,0xFE ,0x28 ,0x48 ,0xFE ,0x54 ,0xFE ,0x24 ,0x06 ,0xE6 ,0x10 ,0xCE ,0xC0 ,0x60 ,0x92 ,
0x94 ,0x78 ,0x10 ,0x06 ,0x00 ,0x00 ,0x00 ,0x00 ,0x7C ,0x82 ,0x00 ,0x00 ,0x00 ,0x82 ,0x7C ,0x00 ,
0x00 ,0x00 ,0x54 ,0x38 ,0xFE ,0x38 ,0x54 ,0x10 ,0x10 ,0x7C ,0x10 ,0x10 ,0x80 ,0x60 ,0x00 ,0x00 ,
0x00 ,0x10 ,0x10 ,0x10 ,0x10 ,0x10 ,0x80 ,0x00 ,0x00 ,0x00 ,0x00 ,0xC0 ,0x30 ,0x0C ,0x02 ,0x00 ,
0x7C ,0xA2 ,0x92 ,0x8A ,0x7C ,0x88 ,0x84 ,0xFE ,0x80 ,0x80 ,0x84 ,0xC2 ,0xA2 ,0x92 ,0x8C ,0x44 ,
0x92 ,0x92 ,0x92 ,0x6C ,0x10 ,0x18 ,0x14 ,0xFE ,0x10 ,0x4E ,0x8A ,0x8A ,0x8A ,0x72 ,0x7C ,0x92 ,
0x92 ,0x92 ,0x64 ,0x02 ,0xC2 ,0x22 ,0x12 ,0x0E ,0x6C ,0x92 ,0x92 ,0x92 ,0x6C ,0x0C ,0x92 ,0x92 ,
0x92 ,0x7C ,0x48 ,0x00 ,0x00 ,0x00 ,0x00 ,0x80 ,0x68 ,0x00 ,0x00 ,0x00 ,0x10 ,0x28 ,0x44 ,0x82 ,
0x00 ,0x28 ,0x28 ,0x28 ,0x28 ,0x00 ,0x82 ,0x44 ,0x28 ,0x10 ,0x00 ,0x04 ,0x02 ,0xA2 ,0x12 ,0x0C ,
0x3C ,0x42 ,0x9A ,0xA2 ,0x1C ,0xF8 ,0x14 ,0x12 ,0x14 ,0xF8 ,0xFE ,0x92 ,0x92 ,0x92 ,0x6C ,0x7C ,
0x82 ,0x82 ,0x82 ,0x44 ,0xFE ,0x82 ,0x82 ,0x44 ,0x38 ,0xFE ,0x92 ,0x92 ,0x82 ,0x82 ,0xFE ,0x12 ,
0x12 ,0x02 ,0x02 ,0x7C ,0x92 ,0x92 ,0x92 ,0x74 ,0xFE ,0x10 ,0x10 ,0x10 ,0xFE ,0x82 ,0x82 ,0xFE ,
0x82 ,0x82 ,0x40 ,0x80 ,0x80 ,0x80 ,0x7E ,0xFE ,0x10 ,0x28 ,0x44 ,0x82 ,0xFE ,0x80 ,0x80 ,0x80 ,
0x00 ,0xFE ,0x04 ,0x08 ,0x04 ,0xFE ,0xFE ,0x04 ,0x18 ,0x20 ,0xFE ,0x7C ,0x82 ,0x82 ,0x82 ,0x7C ,
0xFE ,0x12 ,0x12 ,0x12 ,0x0C ,0x7C ,0x82 ,0xA2 ,0xC2 ,0xFC ,0xFE ,0x12 ,0x32 ,0x52 ,0x8C ,0x4C ,
0x92 ,0x92 ,0x92 ,0x64 ,0x02 ,0x02 ,0xFE ,0x02 ,0x02 ,0x7E ,0x80 ,0x80 ,0x80 ,0x7E ,0x3E ,0x40 ,
0x80 ,0x40 ,0x3E ,0xFE ,0x40 ,0x20 ,0x40 ,0xFE ,0xC6 ,0x28 ,0x10 ,0x28 ,0xC6 ,0x02 ,0x04 ,0xF8 ,
0x04 ,0x02 ,0xC2 ,0xA2 ,0x92 ,0x8A ,0x86 ,0xFE ,0x82 ,0x82 ,0x00 ,0x00 ,0x02 ,0x0C ,0x30 ,0xC0 ,
0x00 ,0x82 ,0x82 ,0xFE ,0x00 ,0x00 ,0x04 ,0x02 ,0x04 ,0x00 ,0x00 ,0x80 ,0x80 ,0x80 ,0x80 ,0x80 ,
0x06 ,0x08 ,0x00 ,0x00 ,0x00 ,0x70 ,0x88 ,0x88 ,0x70 ,0x80 ,0xFC ,0x90 ,0x90 ,0x60 ,0x00 ,0x70 ,
0x88 ,0x88 ,0x88 ,0x00 ,0x60 ,0x90 ,0x90 ,0x7C ,0x80 ,0x70 ,0xA8 ,0xA8 ,0x90 ,0x00 ,0x10 ,0xF8 ,
0x14 ,0x04 ,0x00 ,0x98 ,0xA4 ,0xA4 ,0x78 ,0x00 ,0xFC ,0x20 ,0x10 ,0xE0 ,0x00 ,0xE8 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x40 ,0x80 ,0x80 ,0x74 ,0x00 ,0xFC ,0x20 ,0x50 ,0x88 ,0x00 ,0xFC ,0x00 ,0x00 ,0x00 ,
0x00 ,0xF0 ,0x08 ,0x30 ,0x08 ,0xF0 ,0xF8 ,0x08 ,0x08 ,0xF0 ,0x00 ,0x70 ,0x88 ,0x88 ,0x70 ,0x00 ,
0xF8 ,0x24 ,0x24 ,0x18 ,0x00 ,0x18 ,0x24 ,0x24 ,0xF8 ,0x00 ,0xF0 ,0x08 ,0x08 ,0x10 ,0x00 ,0x90 ,
0xA8 ,0xA8 ,0x48 ,0x00 ,0x08 ,0x7C ,0x88 ,0x00 ,0x00 ,0x78 ,0x80 ,0x80 ,0x78 ,0x00 ,0x38 ,0x40 ,
0x80 ,0x40 ,0x38 ,0x78 ,0x80 ,0x40 ,0x80 ,0x78 ,0x88 ,0x50 ,0x20 ,0x50 ,0x88 ,0x08 ,0x10 ,0xE0 ,
0x10 ,0x08 ,0xC8 ,0xA8 ,0x98 ,0x00 ,0x00 ,0x10 ,0x6C ,0x82 ,0x00 ,0x00 ,0xFE ,0x00 ,0x00 ,0x00 ,
0x00 ,0x82 ,0x6C ,0x10 ,0x00 ,0x00 ,0x08 ,0x04 ,0x08 ,0x10 ,0x08 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00  };

BYTE CharacterWidthTable_Font5x7   [128] = {
0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,
0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,
0x05 ,0x01 ,0x03 ,0x05 ,0x05 ,0x05 ,0x05 ,0x01 ,0x02 ,0x02 ,0x05 ,0x05 ,0x02 ,0x05 ,0x01 ,0x04 ,
0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x01 ,0x01 ,0x04 ,0x04 ,0x04 ,0x05 ,
0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x04 ,0x05 ,0x05 ,0x05 ,
0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x05 ,0x03 ,0x04 ,0x05 ,0x03 ,0x05 ,
0x02 ,0x05 ,0x04 ,0x04 ,0x05 ,0x04 ,0x04 ,0x04 ,0x04 ,0x01 ,0x04 ,0x04 ,0x01 ,0x05 ,0x04 ,0x05 ,
0x05 ,0x05 ,0x04 ,0x04 ,0x03 ,0x04 ,0x05 ,0x05 ,0x05 ,0x05 ,0x03 ,0x03 ,0x01 ,0x05 ,0x05 ,0x05  };

#define FONT3x5_FONT_WIDTH 3
#define FONT3x5_FONT_HEIGHT 5
#define FONT3x5_ELEMENTS 128
#define FONT3x5_FONT_COLUMN_SIZE_IN_BYTE  1


BYTE FontTable_Font3x5 [384] = {
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,
0x00 ,0x00 ,0x00 ,0x17 ,0x00 ,0x00 ,0x03 ,0x00 ,0x03 ,0x0E ,0x1F ,0x0E ,0x14 ,0x1F ,0x0A ,0x00 ,
0x00 ,0x00 ,0x00 ,0x00 ,0x00 ,0x03 ,0x00 ,0x00 ,0x0E ,0x11 ,0x00 ,0x11 ,0x0E ,0x00 ,0x15 ,0x0E ,
0x15 ,0x04 ,0x0E ,0x04 ,0x10 ,0x08 ,0x00 ,0x04 ,0x04 ,0x04 ,0x10 ,0x00 ,0x00 ,0x18 ,0x04 ,0x03 ,
0x0E ,0x11 ,0x0E ,0x12 ,0x1F ,0x10 ,0x19 ,0x15 ,0x16 ,0x11 ,0x15 ,0x0A ,0x07 ,0x04 ,0x1F ,0x17 ,
0x15 ,0x09 ,0x1E ,0x15 ,0x18 ,0x01 ,0x1D ,0x03 ,0x1B ,0x15 ,0x1B ,0x06 ,0x15 ,0x0E ,0x0A ,0x00 ,
0x00 ,0x10 ,0x0A ,0x00 ,0x04 ,0x0A ,0x11 ,0x00 ,0x00 ,0x00 ,0x11 ,0x0A ,0x04 ,0x01 ,0x15 ,0x02 ,
0x09 ,0x15 ,0x0E ,0x1E ,0x05 ,0x1E ,0x1F ,0x15 ,0x0A ,0x0E ,0x11 ,0x0A ,0x1F ,0x11 ,0x0E ,0x1F ,
0x15 ,0x11 ,0x1F ,0x05 ,0x05 ,0x1E ,0x15 ,0x1D ,0x1F ,0x04 ,0x1F ,0x11 ,0x1F ,0x11 ,0x08 ,0x10 ,
0x0F ,0x1F ,0x06 ,0x19 ,0x1F ,0x10 ,0x10 ,0x1F ,0x02 ,0x1F ,0x1F ,0x06 ,0x1F ,0x1F ,0x11 ,0x1F ,
0x1F ,0x05 ,0x07 ,0x1F ,0x19 ,0x1F ,0x1F ,0x0D ,0x16 ,0x16 ,0x15 ,0x1D ,0x01 ,0x1F ,0x01 ,0x1F ,
0x10 ,0x1F ,0x0F ,0x10 ,0x0F ,0x1F ,0x08 ,0x1F ,0x1B ,0x04 ,0x1B ,0x01 ,0x1E ,0x01 ,0x19 ,0x15 ,
0x13 ,0x1F ,0x11 ,0x00 ,0x03 ,0x0C ,0x10 ,0x11 ,0x1F ,0x00 ,0x02 ,0x01 ,0x02 ,0x10 ,0x10 ,0x10 ,
0x01 ,0x02 ,0x00 ,0x08 ,0x14 ,0x1C ,0x1F ,0x14 ,0x08 ,0x0C ,0x12 ,0x12 ,0x08 ,0x14 ,0x1F ,0x0C ,
0x16 ,0x16 ,0x14 ,0x0E ,0x05 ,0x06 ,0x15 ,0x0F ,0x1F ,0x04 ,0x18 ,0x1D ,0x00 ,0x00 ,0x10 ,0x0D ,
0x00 ,0x1F ,0x0C ,0x12 ,0x1F ,0x00 ,0x00 ,0x1C ,0x08 ,0x1C ,0x1C ,0x02 ,0x1C ,0x0C ,0x12 ,0x0C ,
0x1E ,0x05 ,0x02 ,0x02 ,0x05 ,0x1E ,0x1C ,0x02 ,0x04 ,0x14 ,0x1A ,0x00 ,0x04 ,0x1E ,0x04 ,0x1E ,
0x10 ,0x1E ,0x0E ,0x10 ,0x0E ,0x1C ,0x08 ,0x1C ,0x12 ,0x0C ,0x12 ,0x12 ,0x0C ,0x02 ,0x12 ,0x1A ,
0x16 ,0x04 ,0x0E ,0x11 ,0x1F ,0x00 ,0x00 ,0x11 ,0x0E ,0x04 ,0x02 ,0x02 ,0x04 ,0x00 ,0x00 ,0x00  };


BYTE CharacterWidthTable_Font3x5 [128] = {
0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,
0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,
0x03 ,0x01 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x01 ,0x02 ,0x03 ,0x03 ,0x03 ,0x02 ,0x03 ,0x01 ,0x03 ,
0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x01 ,0x02 ,0x03 ,0x03 ,0x03 ,0x03 ,
0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,
0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x02 ,0x03 ,0x02 ,0x03 ,0x03 ,
0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x01 ,0x03 ,0x03 ,0x01 ,0x03 ,0x03 ,0x03 ,
0x03 ,0x03 ,0x03 ,0x02 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x03 ,0x01 ,0x03 ,0x03 ,0x03  };

GFXFont Font5x7;
GFXFont Font3x5;

void GFX_Init()
{
    
    //Staticially Allocate and setup the backbuffer space;
    
    BackBuffer.RenderPlane.BitPlaneSpace = &BackBufferRenderPlaneSpace[0];
   
    BackBuffer.SizeX = BACK_BUFFER_SIZE_X;
    BackBuffer.SizeY = BACK_BUFFER_SIZE_Y;

    BackBuffer.RenderPlane.SizeX = BACK_BUFFER_SIZE_X;
    BackBuffer.RenderPlane.SizeY = BACK_BUFFER_SIZE_Y;

    
    GFX_FullDisplayBufferClear(&BackBuffer);

    //Initialize the stock fonts

    Font5x7.CharacterWidthTable = (BYTE *)CharacterWidthTable_Font5x7;
    Font5x7.FontBuffer = (BYTE *)FontTable_Font5x7;
    Font5x7.FontHeight = FONT5x7_FONT_HEIGHT;
    Font5x7.FontWidth = FONT5x7_FONT_WIDTH;
    Font5x7.BytesPerColumn = FONT5x7_FONT_COLUMN_SIZE_IN_BYTE;

    Font3x5.CharacterWidthTable = (BYTE *)CharacterWidthTable_Font3x5;
    Font3x5.FontBuffer = (BYTE *)FontTable_Font3x5;
    Font3x5.FontHeight = FONT3x5_FONT_HEIGHT;
    Font3x5.FontWidth = FONT3x5_FONT_WIDTH;
    Font3x5.BytesPerColumn = FONT3x5_FONT_COLUMN_SIZE_IN_BYTE;
    
    GFX_InitPhysicalScreen();
    
}



void GFX_FullDisplayBufferClear(RenderContext *Image)
{
    BitPlane_Clear(&Image->RenderPlane);
}


void GFX_PutPixel(RenderContext *Image, SIGNED_WORD x, SIGNED_WORD y)
{
    if((x<Image->SizeX) && (y<Image->SizeY) && (x>=0) && (y>=0))
    {
         BitPlane_Put(&Image->RenderPlane,x,y,TRUE);
    }
}



#ifndef _INLINE_GFX_GET_PIXEL
BYTE GFX_GetPixel(RenderContext *Image, SIGNED_WORD x, SIGNED_WORD y)
{
    BYTE PixelColor = 0;
    
    if((x<Image->SizeX) && (y<Image->SizeY) && (x>=0) && (y>=0))
    {
        
        if(BitPlane_Get(&Image->RedBitPlane,x,y))
            PixelColor |= BICOLOR_RED;
            
        if(BitPlane_Get(&Image->GreenBitPlane,x,y))
            PixelColor |= BICOLOR_GREEN;
    }
    
    return PixelColor;
}
#endif


void GFX_DrawHline(RenderContext *Image, SIGNED_WORD XStart, SIGNED_WORD XStop, SIGNED_WORD Y)
{
    SIGNED_WORD LineStart;
    SIGNED_WORD LineStop;
    WORD i;
    
    if((Y<Image->SizeY)  && (Y>=0))
    {
        if(XStart>XStop)
        {
            LineStart = XStop;    
            LineStop = XStart;
        }    
        else
        {
            LineStart = XStart;    
            LineStop = XStop;
        }
        
        if(LineStart<0)
        {
            LineStart = 0;
        }
    
        if(LineStop>Image->SizeX)
        {
            LineStop =     Image->SizeX-1;
        }
    
        if(LineStart == LineStop)
        {
            GFX_PutPixel(Image,LineStart,Y);    
        }
        else
        {
            for(i=LineStart; i<=LineStop ; i++)
            {
                GFX_PutPixel(Image,i,Y);    
            }
        }    
    }    
    
}



 void GFX_DrawVline(RenderContext *Image, SIGNED_WORD YStart, SIGNED_WORD YStop, SIGNED_WORD X)
{
    SIGNED_WORD LineStart;
    SIGNED_WORD LineStop;
    SIGNED_WORD i;

    if((X<Image->SizeX) && (X>=0))
    {
    
        if(YStart>YStop)
        {
            LineStart = YStop;    
            LineStop = YStart;
        }    
        else
        {
            LineStart = YStart;    
            LineStop = YStop;
        }
        
        if(LineStart<0)
        {
            LineStart = 0;
        }
    

        if(LineStop>Image->SizeY)
        {
            LineStop =     Image->SizeY-1;
        }
    
        for(i=LineStart; i<=LineStop ; i++)
        {
            GFX_PutPixel(Image,X,i);    
        }    
    }    
}


void GFX_DrawLine(RenderContext * Image, SIGNED_WORD X1,SIGNED_WORD Y1, SIGNED_WORD X2,SIGNED_WORD Y2)
{
    //A simple Implementation of Bresenham's line Algorithm
    SIGNED_WORD StartX,StopX,StartY,StopY;
    SIGNED_WORD dX,dY;
    SIGNED_WORD Y_Numerator;
    SIGNED_WORD X_Numerator;
    SIGNED_WORD Y;
    SIGNED_WORD X;
    SIGNED_WORD i;
    BOOL YDir = 0;
    //First Make sure that it is left to right
    //If not them flop them
    if(X2>X1)
    {
        StartX = X1;
        StopX = X2;    
        StartY = Y1;
        StopY = Y2;    
    }    
    else
    {
        StartX = X2;
        StopX = X1;    
        StartY = Y2;
        StopY = Y1;    
    }
    GFX_PutPixel(Image, StopX,StopY);
    if(StopY>=StartY)
    {
        dY = StopY - StartY;
        YDir = 0;
    }
    else
    {
        dY = StartY - StopY;
        YDir = 1;
    }
    dX = StopX - StartX;
    //Now, if the slope is less greater than one,  we need to swap all X/Y operations
    if(dY<=dX)
    {
        //Slope is less than one, proceed at normal and step along the x axis
        Y=StartY;   //start the whole part of the Y value at the starting pixeel.
        X=StartX;
        //We need to start the numerator of the fraction half way through the fraction so evertyhing rounds at
        //fraction midpoint
        Y_Numerator = dX>>1;   //The fraction demonimator is assumeed to be dX
                                // out fixed point Y value is  Y + (Y_Numerator / dX)
                                //Every time we step the X coordinate by one, we need to step
                                //out Y coordinate by dY/dX.  We do this by just adding dY to our
                                //numerator.  When the numerator gets bigger than the
                                //denomiator, the increment the whole part by one and decrement the numerator
                                //by the denominator
        for(i=0;i<dX;i++)
        {
            GFX_PutPixel(Image,X,Y);
            X++;
            //Now do all the fractional stuff
            Y_Numerator += dY;
            if(Y_Numerator >= dX)
            {    
                Y_Numerator-=dX;
                if(StopY > StartY)
                {
                    Y++;
                }
                else
                {
                    Y--;    
                }
            }
        }
    }
    else
    {
        //Same as before by step along the y axis.    
        Y=StartY;   
        X=StartX;
        X_Numerator = dY>>1;   
        for(i=0;i<dY;i++)
        {
            GFX_PutPixel(Image,X,Y);
            //Now do all the fractional stuff
            if(YDir)
            {
                Y--;
            }
            else
            {
                Y++;
            }
            X_Numerator += dX;
            if(X_Numerator >= dY)
            {    
                X_Numerator-=dY;
                if(StopX > StartX)
                {
                    X++;
                }
                else
                {
                    X--;    
                }
            }
        }
    }
}





void GFX_DrawBox(RenderContext *Image, GFXDisplayBox *Box)
{
    GFX_DrawHline(Image,Box->P1.X,Box->P2.X,Box->P1.Y);
    GFX_DrawHline(Image,Box->P1.X,Box->P2.X,Box->P2.Y);
    GFX_DrawVline(Image,Box->P1.Y,Box->P2.Y,Box->P1.X);
    GFX_DrawVline(Image,Box->P1.Y,Box->P2.Y,Box->P2.X);
}



SIGNED_WORD GFX_DrawCharacter(RenderContext * Image,BYTE Character,SIGNED_WORD StartX, SIGNED_WORD StartY, GFXFont * MyFont)
{
    BYTE i,j,Mask;
    WORD CharStartIndex,ColumnStartIndex,ByteOffset;

    CharStartIndex = (Character * (MyFont->BytesPerColumn) * (MyFont->FontWidth));

    for(j=0;j<MyFont->CharacterWidthTable[Character];j++)
        {
            //Draw the current slice
            ColumnStartIndex = j* (MyFont->BytesPerColumn);

            for(i=0;i<MyFont->FontHeight;i++)
            {
                ByteOffset = i>>3;
                Mask = 0x01 << (i&0x07);
            
                if( (MyFont->FontBuffer[CharStartIndex + ColumnStartIndex + ByteOffset]) & Mask)
                {
                    GFX_PutPixel(Image, StartX, StartY + i);
                }        
            }
          StartX++;
        }
    return StartX;
}


SIGNED_WORD GFX_GetStringWidth(CHAR * String,GFXFont * MyFont)
{
    BYTE Ptr = 0;
    BYTE NextChar;
    SIGNED_WORD StringSize = 0;
    
    NextChar = String[Ptr];
    Ptr++;
    
    while((NextChar!=0) && (Ptr <GFX_MAX_STRING_LEN))
    {
        StringSize += MyFont->CharacterWidthTable[NextChar] + 1;
        NextChar = String[Ptr];
        Ptr++;
    }
    
    return StringSize;
}

void GFX_DrawCenteredString(RenderContext * Image,CHAR * String,SIGNED_WORD StartX, SIGNED_WORD StartY, GFXFont * MyFont)
{
    StartX -= (GFX_GetStringWidth(String,MyFont)>>1);
    GFX_DrawString(Image,String,StartX,StartY,MyFont);
}

void GFX_DrawString(RenderContext * Image,CHAR * String,SIGNED_WORD StartX, SIGNED_WORD StartY, GFXFont * MyFont)
{

BYTE Ptr = 0;
BYTE NextChar;

NextChar = String[Ptr];

    while((NextChar!=0) && (Ptr <GFX_MAX_STRING_LEN))
    {
        StartX = GFX_DrawCharacter(Image,NextChar,StartX,StartY,MyFont);
        Ptr++;
        NextChar = String[Ptr];
        StartX++;
    }

}


CHAR GFXStringBuf[64];

void  GFX_printf(RenderContext * Image,SIGNED_WORD StartX, SIGNED_WORD StartY, GFXFont * MyFont, const char *FormatString,...)
{
     va_list argptr; 
     va_start(argptr,FormatString); 
     vsprintf((CHAR *)GFXStringBuf,FormatString,argptr);
     va_end(argptr);   
     
     GFX_DrawString(Image,GFXStringBuf,StartX,StartY,MyFont);
}



void GFX_DrawListPrimitive(RenderContext * Image,GFXListPrimitive *LP)
{
    BYTE Points;
    BYTE i;
    
    if(LP->NumPoints >1)
    {    

        switch(LP->DrawMode)
        {

        default:
        case GFX_LIST_PRIMITIVE_CLOSED:
             for(i=1;i<LP->NumPoints;i++)
                    {
                        GFX_DrawLine(Image,(SIGNED_WORD)(LP->PointList[i-1].X + LP->Center.X),(SIGNED_WORD)(LP->PointList[i-1].Y + LP->Center.Y),
                                          (SIGNED_WORD)(LP->PointList[i].X + LP->Center.X),(SIGNED_WORD)(LP->PointList[i].Y + LP->Center.Y));
                    }
                        GFX_DrawLine(Image,(SIGNED_WORD)(LP->PointList[LP->NumPoints-1].X + LP->Center.X),(SIGNED_WORD)(LP->PointList[LP->NumPoints-1].Y + LP->Center.Y),
                                          (SIGNED_WORD)(LP->PointList[0].X + LP->Center.X),(SIGNED_WORD)(LP->PointList[0].Y + LP->Center.Y));
            break;

        case GFX_LIST_PRIMITIVE_CONNECTED:
            for(i=1;i<LP->NumPoints;i++)
                    {
                        GFX_DrawLine(Image,(SIGNED_WORD)(LP->PointList[i-1].X + LP->Center.X),(SIGNED_WORD)(LP->PointList[i-1].Y + LP->Center.Y),
                                          (SIGNED_WORD)(LP->PointList[i].X + LP->Center.X),(SIGNED_WORD)(LP->PointList[i].Y + LP->Center.Y));
                    }
            break;

        case GFX_LIST_PRIMITIVE_DISCONNECTED:
                    Points = LP->NumPoints>>1;
                    for(i=0;i<Points;i++)
                    {
                        GFX_DrawLine(Image,(SIGNED_WORD)(LP->PointList[i*2].X + LP->Center.X),(SIGNED_WORD)(LP->PointList[i*2].Y + LP->Center.Y),
                                          (SIGNED_WORD)(LP->PointList[i*2+1].X + LP->Center.X),(SIGNED_WORD)(LP->PointList[i*2+1].Y + LP->Center.Y));
                    }
            break;
            
        case GFX_LIST_PRIMITIVE_CLOSED_YFLIPPED:
             for(i=1;i<LP->NumPoints;i++)
                    {
                        GFX_DrawLine(Image,(SIGNED_WORD)(LP->PointList[i-1].X + LP->Center.X),(SIGNED_WORD)(-LP->PointList[i-1].Y + LP->Center.Y),
                                          (SIGNED_WORD)(LP->PointList[i].X + LP->Center.X),(SIGNED_WORD)(-LP->PointList[i].Y + LP->Center.Y));
                    }
                        GFX_DrawLine(Image,(SIGNED_WORD)(LP->PointList[LP->NumPoints-1].X + LP->Center.X),(SIGNED_WORD)(-LP->PointList[LP->NumPoints-1].Y + LP->Center.Y),
                                          (SIGNED_WORD)(LP->PointList[0].X + LP->Center.X),(SIGNED_WORD)(-LP->PointList[0].Y + LP->Center.Y));
            break;

        case GFX_LIST_PRIMITIVE_CONNECTED_YFLIPPED:
            for(i=1;i<LP->NumPoints;i++)
                    {
                        GFX_DrawLine(Image,(SIGNED_WORD)(LP->PointList[i-1].X + LP->Center.X),(SIGNED_WORD)(-LP->PointList[i-1].Y + LP->Center.Y),
                                          (SIGNED_WORD)(LP->PointList[i].X + LP->Center.X),(SIGNED_WORD)(-LP->PointList[i].Y + LP->Center.Y));
                    }
            break;

        case GFX_LIST_PRIMITIVE_DISCONNECTED_YFLIPPED:
                    Points = LP->NumPoints>>1;
                    for(i=0;i<Points;i++)
                    {
                        GFX_DrawLine(Image,(SIGNED_WORD)(LP->PointList[i*2].X + LP->Center.X),(SIGNED_WORD)(-LP->PointList[i*2].Y + LP->Center.Y),
                                          (SIGNED_WORD)(LP->PointList[i*2+1].X + LP->Center.X),(SIGNED_WORD)(-LP->PointList[i*2+1].Y + LP->Center.Y));
                    }
            break;

        }
    }
}




void GFX_DrawScaledListPrimitive(RenderContext * Image,GFXListPrimitive *LP , FIXED_7_8 Scale)
{
    BYTE Points;
    BYTE i;
    GFXRelativePoint TransformedPoint[2] = {0};

    if(LP->NumPoints >1)
    {    
        switch(LP->DrawMode)
        {

        default:
        case GFX_LIST_PRIMITIVE_CLOSED:
             for(i=1;i<LP->NumPoints;i++)
                    {

                    TransformedPoint[0].X = FMul_15_0to7_8(LP->PointList[i-1].X,Scale)     + LP->Center.X;
                    TransformedPoint[0].Y = FMul_15_0to7_8(LP->PointList[i-1].Y,Scale)   + LP->Center.Y;
                    TransformedPoint[1].X = FMul_15_0to7_8(LP->PointList[i].X,Scale)     + LP->Center.X;
                    TransformedPoint[1].Y = FMul_15_0to7_8(LP->PointList[i].Y,Scale)     + LP->Center.Y;

                    GFX_DrawLine(Image,TransformedPoint[0].X,TransformedPoint[0].Y,
                                      TransformedPoint[1].X,TransformedPoint[1].Y);

                     }

                     TransformedPoint[0].X = FMul_15_0to7_8(LP->PointList[LP->NumPoints-1].X,Scale)   + LP->Center.X;
                     TransformedPoint[0].Y = FMul_15_0to7_8(LP->PointList[LP->NumPoints-1].Y,Scale)   + LP->Center.Y;
                     TransformedPoint[1].X = FMul_15_0to7_8(LP->PointList[0].X,Scale)                 + LP->Center.X;
                     TransformedPoint[1].Y = FMul_15_0to7_8(LP->PointList[0].Y,Scale)                 + LP->Center.Y;

                     GFX_DrawLine(Image,TransformedPoint[0].X,TransformedPoint[0].Y,
                                      TransformedPoint[1].X,TransformedPoint[1].Y);

            break;

        case GFX_LIST_PRIMITIVE_CONNECTED:
            for(i=1;i<LP->NumPoints;i++)
                    {
                        TransformedPoint[0].X = FMul_15_0to7_8(LP->PointList[i-1].X,Scale)     + LP->Center.X;
                        TransformedPoint[0].Y = FMul_15_0to7_8(LP->PointList[i-1].Y,Scale)   + LP->Center.Y;
                        TransformedPoint[1].X = FMul_15_0to7_8(LP->PointList[i].X,Scale)     + LP->Center.X;
                        TransformedPoint[1].Y = FMul_15_0to7_8(LP->PointList[i].Y,Scale)     + LP->Center.Y;
                        
                        GFX_DrawLine(Image,TransformedPoint[0].X,TransformedPoint[0].Y,
                                      TransformedPoint[1].X,TransformedPoint[1].Y);
                    }
            break;

        case GFX_LIST_PRIMITIVE_DISCONNECTED:
                    Points = LP->NumPoints>>1;
                    
                    for(i=0;i<Points;i++)
                    {
                        TransformedPoint[0].X = FMul_15_0to7_8(LP->PointList[i*2].X,Scale)        + LP->Center.X;
                        TransformedPoint[0].Y = FMul_15_0to7_8(LP->PointList[i*2].Y,Scale)        + LP->Center.Y;
                        TransformedPoint[1].X = FMul_15_0to7_8(LP->PointList[(i*2)+1].X,Scale)   + LP->Center.X;
                        TransformedPoint[1].Y = FMul_15_0to7_8(LP->PointList[(i*2)+1].Y,Scale)   + LP->Center.Y;

                        GFX_DrawLine(Image,TransformedPoint[0].X,TransformedPoint[0].Y,
                                      TransformedPoint[1].X,TransformedPoint[1].Y);
                    }
            break;

        }

    }

}

void GFX_DrawRotatedListPrimitive(RenderContext * Image,GFXListPrimitive *LP , BYTE Angle, BYTE Color)
{
    BYTE Points;
    BYTE i;

    GFXRelativePoint TransformedPoint[2] = {0};

    if(LP->NumPoints >1)
    {
        //If we are drawing non connected points,  there must be an even number of them.
        switch(LP->DrawMode)
        {
            case GFX_LIST_PRIMITIVE_DISCONNECTED:
    
                Points = LP->NumPoints>>1;
                
                for(i=0;i<Points;i++)
                {
    
                    TransformedPoint[0].X = FMul_15_0to1_14((LP->PointList[i*2].X), Fixed_1_14_SineTable[(Angle + 64)&0xff]) - 
                                            FMul_15_0to1_14(LP->PointList[i*2].Y, Fixed_1_14_SineTable[Angle]);
    
                    TransformedPoint[0].Y = FMul_15_0to1_14(LP->PointList[i*2].X, Fixed_1_14_SineTable[Angle ]) + 
                                            FMul_15_0to1_14(LP->PointList[i*2].Y, Fixed_1_14_SineTable[(Angle + 64)&0xff]);
    
                    TransformedPoint[1].X = FMul_15_0to1_14(LP->PointList[(i*2)+1].X, Fixed_1_14_SineTable[(Angle + 64)&0xff]) - 
                                            FMul_15_0to1_14(LP->PointList[(i*2)+1].Y, Fixed_1_14_SineTable[Angle]);
    
                    TransformedPoint[1].Y = FMul_15_0to1_14(LP->PointList[(i*2)+1].X, Fixed_1_14_SineTable[Angle]) + 
                                            FMul_15_0to1_14(LP->PointList[(i*2)+1].Y, Fixed_1_14_SineTable[(Angle + 64)&0xff]);
    
    
                    TransformedPoint[0].X += LP->Center.X;
                    TransformedPoint[0].Y += LP->Center.Y;
                    TransformedPoint[1].X += LP->Center.X;
                    TransformedPoint[1].Y += LP->Center.Y;
    
    
                    GFX_DrawLine(Image,(TransformedPoint[0].X + LP->Center.X),(TransformedPoint[0].Y  + LP->Center.Y),
                                      (TransformedPoint[1].X + LP->Center.X),(TransformedPoint[1].Y + LP->Center.Y));
                }
            break;
        
            case GFX_LIST_PRIMITIVE_CONNECTED:
            
                for(i=1;i<LP->NumPoints;i++)
                {
    
                    TransformedPoint[0].X = (FMul_15_0to1_14(LP->PointList[i-1].X, Fixed_1_14_SineTable[(Angle + 64)&0xff]) - 
                                            FMul_15_0to1_14(LP->PointList[i-1].Y, Fixed_1_14_SineTable[Angle])  );
    
                    TransformedPoint[0].Y = ( FMul_15_0to1_14(LP->PointList[i-1].X, Fixed_1_14_SineTable[Angle]) + 
                                            FMul_15_0to1_14(LP->PointList[i-1].Y, Fixed_1_14_SineTable[(Angle + 64)&0xff] ) );
    
                    TransformedPoint[1].X = (FMul_15_0to1_14(LP->PointList[i].X, Fixed_1_14_SineTable[(Angle + 64)&0xff]) - 
                                             FMul_15_0to1_14(LP->PointList[i].Y, Fixed_1_14_SineTable[Angle]) );
    
                    TransformedPoint[1].Y = (FMul_15_0to1_14(LP->PointList[i].X, Fixed_1_14_SineTable[Angle]) + 
                                            FMul_15_0to1_14(LP->PointList[i].Y, Fixed_1_14_SineTable[(Angle + 64)&0xff]));
                    
                    TransformedPoint[0].X +=  LP->Center.X;
                    TransformedPoint[0].Y +=  LP->Center.Y;
                    TransformedPoint[1].X +=  LP->Center.X;
                    TransformedPoint[1].Y +=  LP->Center.Y;
    
                    GFX_DrawLine(Image,(TransformedPoint[0].X),(TransformedPoint[0].Y),
                                      (TransformedPoint[1].X),(TransformedPoint[1].Y));
    
                }
            break;
            default:
            case GFX_LIST_PRIMITIVE_CLOSED:
                
                    for(i=1;i<LP->NumPoints;i++)
                    {
        
                        TransformedPoint[0].X = (FMul_15_0to1_14(LP->PointList[i-1].X, Fixed_1_14_SineTable[(Angle + 64)&0xff]) - 
                                                FMul_15_0to1_14(LP->PointList[i-1].Y, Fixed_1_14_SineTable[Angle])  );
        
                        TransformedPoint[0].Y = ( FMul_15_0to1_14(LP->PointList[i-1].X, Fixed_1_14_SineTable[Angle]) + 
                                                FMul_15_0to1_14(LP->PointList[i-1].Y, Fixed_1_14_SineTable[(Angle + 64)&0xff] ) );
        
                        TransformedPoint[1].X = (FMul_15_0to1_14(LP->PointList[i].X, Fixed_1_14_SineTable[(Angle + 64)&0xff]) - 
                                                 FMul_15_0to1_14(LP->PointList[i].Y, Fixed_1_14_SineTable[Angle]) );
        
                        TransformedPoint[1].Y = (FMul_15_0to1_14(LP->PointList[i].X, Fixed_1_14_SineTable[Angle]) + 
                                                FMul_15_0to1_14(LP->PointList[i].Y, Fixed_1_14_SineTable[(Angle + 64)&0xff]));
                        
                        TransformedPoint[0].X +=  LP->Center.X;
                        TransformedPoint[0].Y +=  LP->Center.Y;
                        TransformedPoint[1].X +=  LP->Center.X;
                        TransformedPoint[1].Y +=  LP->Center.Y;
        
                        GFX_DrawLine(Image,(TransformedPoint[0].X),(TransformedPoint[0].Y),
                                          (TransformedPoint[1].X),(TransformedPoint[1].Y));
                    }
                    
                        TransformedPoint[0].X = (FMul_15_0to1_14(LP->PointList[LP->NumPoints-1].X, Fixed_1_14_SineTable[(Angle + 64)&0xff]) - 
                                                FMul_15_0to1_14(LP->PointList[LP->NumPoints-1].Y, Fixed_1_14_SineTable[Angle])  );
        
                        TransformedPoint[0].Y = ( FMul_15_0to1_14(LP->PointList[LP->NumPoints-1].X, Fixed_1_14_SineTable[Angle]) + 
                                                FMul_15_0to1_14(LP->PointList[LP->NumPoints-1].Y, Fixed_1_14_SineTable[(Angle + 64)&0xff] ) );
        
                        TransformedPoint[1].X = (FMul_15_0to1_14(LP->PointList[0].X, Fixed_1_14_SineTable[(Angle + 64)&0xff]) - 
                                                 FMul_15_0to1_14(LP->PointList[0].Y, Fixed_1_14_SineTable[Angle]) );
        
                        TransformedPoint[1].Y = (FMul_15_0to1_14(LP->PointList[0].X, Fixed_1_14_SineTable[Angle]) + 
                                                FMul_15_0to1_14(LP->PointList[0].Y, Fixed_1_14_SineTable[(Angle + 64)&0xff]));
                        
                        TransformedPoint[0].X +=  LP->Center.X;
                        TransformedPoint[0].Y +=  LP->Center.Y;
                        TransformedPoint[1].X +=  LP->Center.X;
                        TransformedPoint[1].Y +=  LP->Center.Y;
        
                        GFX_DrawLine(Image,(TransformedPoint[0].X),(TransformedPoint[0].Y),
                                          (TransformedPoint[1].X),(TransformedPoint[1].Y));
                break;
        }
    }
}
/*
void GFX_DrawScaledRotatedListPrimitive(BiColorRenderContext * Image,GFXListPrimitive *LP , BYTE Angle,GFXFixed_7_8 Scale, BYTE Color)
{
    BYTE Points;
    BYTE i;
    
    GFXRelativePoint TransformedPoint[2];

    if(LP->NumPoints >1)
    {
        //If we are drawing non connected points,  there must be an even number of them.
        if(LP->Connected == FALSE)
        {
            Points = LP->NumPoints>>1;
            
            for(i=0;i<Points;i++)
            {

                TransformedPoint[0].X = FMul_15_0to0_15(LP->PointList[i*2].X, Fixed_0_15_SineTable[(Angle + 64)&0xff]) - 
                                        FMul_15_0to0_15(LP->PointList[i*2].Y, Fixed_0_15_SineTable[Angle]);

                TransformedPoint[0].Y = FMul_15_0to0_15(LP->PointList[i*2].X, Fixed_0_15_SineTable[Angle ]) + 
                                        FMul_15_0to0_15(LP->PointList[i*2].Y, Fixed_0_15_SineTable[(Angle + 64)&0xff]);

                TransformedPoint[1].X = FMul_15_0to0_15(LP->PointList[(i*2)+1].X, Fixed_0_15_SineTable[(Angle + 64)&0xff]) - 
                                        FMul_15_0to0_15(LP->PointList[(i*2)+1].Y, Fixed_0_15_SineTable[Angle]);

                TransformedPoint[1].Y = FMul_15_0to0_15(LP->PointList[(i*2)+1].X, Fixed_0_15_SineTable[Angle ]) + 
                                        FMul_15_0to0_15(LP->PointList[(i*2)+1].Y, Fixed_0_15_SineTable[(Angle + 64)&0xff]);


                TransformedPoint[0].X = FMul_15_0to7_8(TransformedPoint[0].X,Scale) + LP->Center.X;
                TransformedPoint[0].Y = FMul_15_0to7_8(TransformedPoint[0].Y,Scale)  + LP->Center.Y;
                TransformedPoint[1].X = FMul_15_0to7_8(TransformedPoint[1].X,Scale) + LP->Center.X;
                TransformedPoint[1].Y = FMul_15_0to7_8(TransformedPoint[1].Y,Scale)  + LP->Center.Y;

                GFX_DrawLine(Image,(TransformedPoint[0].X),(TransformedPoint[0].Y),
                                  (TransformedPoint[1].X),(TransformedPoint[1].Y),Color);
            
            }
        
        }
        else
        {
            for(i=1;i<LP->NumPoints;i++)
            {

                TransformedPoint[0].X = (     FMul_15_0to0_15(LP->PointList[i-1].X, Fixed_0_15_SineTable[(Angle + 64)&0xff]) - 
                                        FMul_15_0to0_15(LP->PointList[i-1].Y, Fixed_0_15_SineTable[Angle])  ) ;

                TransformedPoint[0].Y = ( FMul_15_0to0_15(LP->PointList[i-1].X, Fixed_0_15_SineTable[Angle ]) + 
                                        FMul_15_0to0_15(LP->PointList[i-1].Y, Fixed_0_15_SineTable[(Angle + 64)&0xff] ) );

                TransformedPoint[1].X = (FMul_15_0to0_15(LP->PointList[i].X, Fixed_0_15_SineTable[(Angle + 64)&0xff]) - 
                                         FMul_15_0to0_15(LP->PointList[i].Y, Fixed_0_15_SineTable[Angle]) ) ;

                TransformedPoint[1].Y = (FMul_15_0to0_15(LP->PointList[i].X, Fixed_0_15_SineTable[Angle ]) + 
                                        FMul_15_0to0_15(LP->PointList[i].Y, Fixed_0_15_SineTable[(Angle + 64)&0xff]) ) ;


                TransformedPoint[0].X = FMul_15_0to7_8(TransformedPoint[0].X,Scale)  +    LP->Center.X;
                TransformedPoint[0].Y = FMul_15_0to7_8(TransformedPoint[0].Y,Scale)+ LP->Center.Y;
                TransformedPoint[1].X = FMul_15_0to7_8(TransformedPoint[1].X,Scale)  +    LP->Center.X;
                TransformedPoint[1].Y = FMul_15_0to7_8(TransformedPoint[1].Y,Scale)+ LP->Center.Y;

                GFX_DrawLine(Image,(TransformedPoint[0].X),(TransformedPoint[0].Y),
                                  (TransformedPoint[1].X),(TransformedPoint[1].Y),Color);

            }
        
        }
    }
}

*/


