
#include "Terminal.h"
#include "Types.h"
#include "Queue.h"
#include <string.h>
#include <stdint.h>


//*****************************************************************
//Terminal Configuration
//*****************************************************************

#define MAX_TERMINAL_LINE_CHARS 64
#define MAX_TERMINAL_CMD_CHARS  32

ByteQueue TERMINAL_OUTPUT_QUEUE;
ByteQueue TERMINAL_INPUT_QUEUE;

#define TERMINAL_QUEUE_SIZE     512

uint8_t TERMINAL_OUTPUT_QUEUE_Storage[TERMINAL_QUEUE_SIZE];
uint8_t TERMINAL_INPUT_QUEUE_Storage[TERMINAL_QUEUE_SIZE];



//Terminal Prototypes Here!
void TerminalCmd_Help(char *arg);
void TerminalCmd_ssr(char *arg);

//Populate this array with the callback functions and their terminal command string
TerminalCallbackRecord MyTerminalCallbackRecords[] ={   {"help",TerminalCmd_Help,"Lists available commands"},
                                                        {"ssr",TerminalCmd_ssr,"Enables /disables an ssr"}
                                                 
                                                        };


#define NUM_TERMINAL_COMMANDS  (sizeof(MyTerminalCallbackRecords)/sizeof(TerminalCallbackRecord))
                                                        
                                                        
//Terminal Callback Implementation Here!                                              

void TerminalCmd_Help(char *arg)
{
    uint8_t i;

    TERMINAL_PRINTF("\r\n\r\nCommand List:\r\n");
    TERMINAL_PRINTF("----------------------\r\n");

    for(i=0;i<NUM_TERMINAL_COMMANDS;i++)
    {
         TERMINAL_PRINTF("%s  ---->  %s\r\n",MyTerminalCallbackRecords[i].CommandString,MyTerminalCallbackRecords[i].HelpString);    
    }

    TERMINAL_PRINTF("\r\n\r\n");
}

void TerminalCmd_ssr(char *arg)
{
    uint32_t Items;
    uint32_t RelayIndex;
    uint32_t RelayState;
    Items = sscanf(arg,"%d %d",&RelayIndex,&RelayState);   
    
    if(Items!=2)
    {
        TERMINAL_PRINTF("\r\nssr requires 2 arguments:\r\n");
        TERMINAL_PRINTF("1st:   Relay Index    Valid values are 0 (high current) and 1 (low current)\r\n");
        TERMINAL_PRINTF("2nd:   Relay State     0 == off  1 == on   anything else == off\r\n");
    }
    else
    {
        if(RelayIndex >1)
            RelayIndex = 1;
            
        if(RelayState > 1)
            RelayState = 0;
            
         TERMINAL_PRINTF("Setting ssr %d to state of %d",RelayIndex,RelayState);   
    
        if(RelayIndex == 0)
        {
            if(RelayState == 0 )
            {
                  DISABLE_HIGH_CURRENT_RELAY ;
            }
            else
            {
                 ENABLE_HIGH_CURRENT_RELAY;
            }
        }
        else
        {
             if(RelayState == 0 )
            {
                   DISABLE_LOW_CURRENT_RELAY;
            }
            else
            {
                   ENABLE_LOW_CURRENT_RELAY;
            }
        }
        
    
    } 
}




//*****************************************************************
//Plumbing.....
//*****************************************************************



char TerminalLineBuf[MAX_TERMINAL_LINE_CHARS];
uint8_t TerminalPos;
char TerminalCmdBuf[MAX_TERMINAL_CMD_CHARS+1];
char TerminalArgs[MAX_TERMINAL_LINE_CHARS-MAX_TERMINAL_CMD_CHARS];
uint8_t NextCharIn;
uint8_t CmdFound;
 
void TerminalBootMsg()
{
     // TERMINAL_PRINTF("\r\n\r\nMonkey Do!\r\n");
}

void InitTerminal()
{
    TerminalPos = 0;
    CmdFound = 0;
    
    InitByteQueue(&TERMINAL_OUTPUT_QUEUE,TERMINAL_QUEUE_SIZE,&TERMINAL_OUTPUT_QUEUE_Storage[0]); 
    InitByteQueue(&TERMINAL_INPUT_QUEUE,TERMINAL_QUEUE_SIZE,&TERMINAL_INPUT_QUEUE_Storage[0]); 
    
    
  //  TerminalBootMsg();
}

  

void ProcessTerminal()
{
     uint8_t i,j;
     uint8_t ArgsFound;
        
    if(TERMINAL_READABLE)
    {
       NextCharIn = TERMINAL_GETC;
       
        switch(NextCharIn)
        {
            case '\r':
             
             TerminalLineBuf[TerminalPos++] = 0x0;
             TERMINAL_PUTC(NextCharIn);
           
             if(TerminalPos > 1)
             {
                 //find the command
                 i=0;
                 while(TerminalLineBuf[i]>0x20 &&  TerminalLineBuf[i]<0x7f)
                 {
                      TerminalCmdBuf[i] = TerminalLineBuf[i];
                      i++;
    
                    if(i==MAX_TERMINAL_CMD_CHARS)
                        {
                         break;
                        }
                 }
                    
                TerminalCmdBuf[i] = 0;
                TerminalCmdBuf[i+1] = 0;
                
                
                ArgsFound = TRUE;
                memset(TerminalArgs,0x00,sizeof(TerminalArgs));
                //scan for num terminator or next non whitespace
                while(TerminalLineBuf[i]<=0x20 && (i<MAX_TERMINAL_LINE_CHARS))
                {
                    if(TerminalLineBuf[i] == 0x00)
                    {
                    
                        //if we find a NULL terminator before a non whitespace character they flag for no arguments
                        ArgsFound = FALSE;
                        break;
                    }   
                    i++; 
                }
                
                if(ArgsFound == TRUE)
                {
                    strcpy(TerminalArgs,&TerminalLineBuf[i]);
                    
                    //trim trailing whitespace
                    i = sizeof(TerminalArgs)-1;
                    
                    while((TerminalArgs[i]<0x21) && (i>0))
                    {
                        TerminalArgs[i]= 0x00;
                        i--;
                    }       
                }
                
                CmdFound = FALSE;
                for(j=0;j<NUM_TERMINAL_COMMANDS;j++)
                {           
                    if(strcmp(TerminalCmdBuf,MyTerminalCallbackRecords[j].CommandString) == 0)
                    {
                        TERMINAL_PRINTF("\r\n");
                        if(MyTerminalCallbackRecords[j].Callback != NULL)
                            MyTerminalCallbackRecords[j].Callback(TerminalArgs);
                    
                        CmdFound = TRUE;
                        break;
                    }             
                }        
                if(CmdFound == FALSE)
                {
                  TERMINAL_PRINTF("\r\n%s command not recognized.\r\n\r\n",TerminalCmdBuf);
                 
                }
              }    
             TERMINAL_PRINTF("\r\n>");
             TerminalPos = 0;
            
            break;
            
            case '\b':
                if(TerminalPos > 0)
                {
                    TerminalPos--;    
                    TERMINAL_PUTC(NextCharIn);
                }
            break;
            
            default:
                
                if(TerminalPos == 0 && NextCharIn == 0x020)
                {
                     //Do nothing if space bar is pressed at beginning of line    
                }
                   else if(NextCharIn >= 0x20 && NextCharIn<0x7F)
                {
                    
                    if(TerminalPos < MAX_TERMINAL_LINE_CHARS-1)
                    {
                         TerminalLineBuf[TerminalPos++] = NextCharIn;
                        TERMINAL_PUTC(NextCharIn);
                    }
                }
            
            break;
        
        }
    }
 
}


