/*

--------------------------------------------
|                                          |
|                     ....                 |
|                    7OO$?I78.             |
|                   .?8++++7+II?D.         |
|                   .?O++=I++II+?=         |
|                   .IO++?7==I??$.         |
|                   .8++=$===?+I$          |
|                   ?+++===+===+           |
|                   ???=+I++==+?           |
|                 .??++====+==++           |
|                 ?+++==========~          |
|                $+++==========+=          |
|              =?+===+==+I========         |
|           ..++======~~~~========?        |
|         .$?I??+=~~===~~~===~===++.       |
|       .+==.+=~~~=~==~~~~==~~=~==+?       |
|      ?===I+====~~=~~~=~~=====~~~=?.      |
|     .=~~~+==~==..~~~~~~=    ~~~~=7=      |
|     +=~~?+~~=.  ==~~~~=.     ~~~~=?.     |
|     =~~~=~~~   ?===~~+.       ~~~~+      |
|     +~~:+~~=    =~~==.        =~~+.      |
|     ~:~ =~~=    =~~~=         ~===       |
|         I=~~   ,=~~=            ,.       |
|          ~~.   ,====                     |
|                 ====                     |
|                 =~~.                     |
|                                          |
|------------------------------------------|                                                          
|              Internet Of Thing           |
|                  Eli Hughes              |
| Freescale / Hack-a-day Make-It-Challenge |
|              FTF 2014 - Dallas, Tx       |
|------------------------------------------|

*/
#include "Terminal.h"
#include "Types.h"
#include "Queue.h"
#include <string.h>
#include <stdint.h>
#include "System.h"

//*****************************************************************
//Terminal Configuration
//*****************************************************************

#define MAX_TERMINAL_LINE_CHARS 64
#define MAX_TERMINAL_CMD_CHARS  32

ByteQueue TERMINAL_OUTPUT_QUEUE;
ByteQueue TERMINAL_INPUT_QUEUE;

#define TERMINAL_QUEUE_SIZE     2048

uint8_t TERMINAL_OUTPUT_QUEUE_Storage[TERMINAL_QUEUE_SIZE];
uint8_t TERMINAL_INPUT_QUEUE_Storage[TERMINAL_QUEUE_SIZE];

typedef void (*TerminalCallback)(char *);


typedef struct 
{
    const char *CommandString;
    TerminalCallback Callback;
    const char *HelpString;
        
} TerminalCallbackRecord;

//Callback function prototypes
void TerminalCmd_Help(char *arg);
void TerminalCmd_Go(char *arg);
void TerminalCmd_SpinCW(char *arg);
void TerminalCmd_SpinCCW(char *arg);
void TerminalCmd_Stop(char *arg);

void TerminalCmd_Reboot(char *arg);
void TerminalCmd_Back(char *arg);

//Populate this array with the callback functions and their terminal command string
TerminalCallbackRecord MyTerminalCallbackRecords[] ={   {"help",TerminalCmd_Help,"Lists available commands"},
                                                        {"thing_go",TerminalCmd_Go," Tells the thing to go forward"},
                                                        {"thing_cw",TerminalCmd_SpinCW," Tells the thing to spin clock-wise"},
                                                        {"thing_ccw",TerminalCmd_SpinCCW," Tells the thing to spin counter clockwise"},
                                                        {"thing_stop",TerminalCmd_Stop,"Tells the thing to stop"},
                                                        {"thing_back",TerminalCmd_Back,"Tells the thing to go"},
                                                        };

//*****************************************************************
//Plumbing.....
//*****************************************************************

#define NUM_TERMINAL_COMMANDS  (sizeof(MyTerminalCallbackRecords)/sizeof(TerminalCallbackRecord))

char TerminalLineBuf[MAX_TERMINAL_LINE_CHARS];
uint8_t TerminalPos;
char TerminalCmdBuf[MAX_TERMINAL_CMD_CHARS+1];
char TerminalArgs[MAX_TERMINAL_LINE_CHARS-MAX_TERMINAL_CMD_CHARS];
uint8_t NextCharIn;
uint8_t CmdFound;
 
void TerminalBootMsg()
{
TERMINAL_PRINTF("-------------------------------------------- \r\n");
TERMINAL_PRINTF("|                                          |\r\n");
TERMINAL_PRINTF("|                     ....                 | \r\n");
TERMINAL_PRINTF("|                    7OO$?I78.             | \r\n");
TERMINAL_PRINTF("|                   .?8++++7+II?D.         | \r\n");
TERMINAL_PRINTF("|                   .?O++=I++II+?=         | \r\n");
TERMINAL_PRINTF("|                   .IO++?7==I??$.         | \r\n");
TERMINAL_PRINTF("|                   .8++=$===?+I$          | \r\n");
TERMINAL_PRINTF("|                   ?+++===+===+           | \r\n");
TERMINAL_PRINTF("|                   ???=+I++==+?           | \r\n");
TERMINAL_PRINTF("|                 .??++====+==++           | \r\n");
TERMINAL_PRINTF("|                 ?+++==========~          | \r\n");
TERMINAL_PRINTF("|                $+++==========+=          | \r\n");
TERMINAL_PRINTF("|              =?+===+==+I========         | \r\n");
TERMINAL_PRINTF("|           ..++======~~~~========?        | \r\n");
TERMINAL_PRINTF("|         .$?I??+=~~===~~~===~===++.       | \r\n");
TERMINAL_PRINTF("|       .+==.+=~~~=~==~~~~==~~=~==+?       | \r\n");
TERMINAL_PRINTF("|      ?===I+====~~=~~~=~~=====~~~=?.      | \r\n");
TERMINAL_PRINTF("|     .=~~~+==~==..~~~~~~=    ~~~~=7=      | \r\n");
TERMINAL_PRINTF("|     +=~~?+~~=.  ==~~~~=.     ~~~~=?.     | \r\n");
TERMINAL_PRINTF("|     =~~~=~~~   ?===~~+.       ~~~~+      | \r\n");
TERMINAL_PRINTF("|     +~~:+~~=    =~~==.        =~~+.      | \r\n");
TERMINAL_PRINTF("|     ~:~ =~~=    =~~~=         ~===       | \r\n");
TERMINAL_PRINTF("|         I=~~   ,=~~=            ,.       | \r\n");
TERMINAL_PRINTF("|          ~~.   ,====                     | \r\n");
TERMINAL_PRINTF("|                 ====                     | \r\n");
TERMINAL_PRINTF("|                 =~~.                     | \r\n");
TERMINAL_PRINTF("|                                          |\r\n");
TERMINAL_PRINTF("|------------------------------------------|\r\n");                                                                
TERMINAL_PRINTF("|              Internet Of Thing           |\r\n");
TERMINAL_PRINTF("|                  Eli Hughes              |\r\n");
TERMINAL_PRINTF("| Freescale / Hack-a-day Make-It-Challenge |\r\n"); 
TERMINAL_PRINTF("|              FTF 2014 - Dallas, Tx       |\r\n");
TERMINAL_PRINTF("|------------------------------------------|\r\n\r\n>"); 
}

void TFC_InitTerminal()
{
    TerminalPos = 0;
    CmdFound = 0;
    
    InitByteQueue(&TERMINAL_OUTPUT_QUEUE,TERMINAL_QUEUE_SIZE,&TERMINAL_OUTPUT_QUEUE_Storage[0]); 
    InitByteQueue(&TERMINAL_INPUT_QUEUE,TERMINAL_QUEUE_SIZE,&TERMINAL_INPUT_QUEUE_Storage[0]); 
    
    
    TerminalBootMsg();
}

void TerminalCmd_Help(char *arg)
{
    uint8_t i;

    TERMINAL_PRINTF("\r\n\r\nCommand List:\r\n");
    TERMINAL_PRINTF("----------------------\r\n");

    for(i=0;i<NUM_TERMINAL_COMMANDS;i++)
    {
         TERMINAL_PRINTF("%s  ---->  %s\r\n",MyTerminalCallbackRecords[i].CommandString,MyTerminalCallbackRecords[i].HelpString);    
    }

    TERMINAL_PRINTF("\r\n\r\n");
}


void TerminalCmd_Go(char *arg)
{
    SPIN_CCW = 0;
    SPIN_CW = 0;
    GO_FORWARD = 1;
    GO_REVERSE = 0;
    
}

void TerminalCmd_Back(char *arg)
{
    SPIN_CCW = 0;
    SPIN_CW = 0;
    GO_FORWARD = 0;
    GO_REVERSE = 1;
    
}
    
void TerminalCmd_SpinCW(char *arg)
{
    SPIN_CCW = 0;
    SPIN_CW = 1;
    GO_FORWARD = 0;
    GO_REVERSE = 0;

}
void TerminalCmd_SpinCCW(char *arg)
{
    SPIN_CCW = 1;
    SPIN_CW = 0;
    GO_FORWARD = 0;
    GO_REVERSE = 0;
}

void TerminalCmd_Stop(char *arg)
{

    SPIN_CCW = 0;
    SPIN_CW = 0;
    GO_FORWARD = 0;
    GO_REVERSE = 0;

}
    

void TerminalCmd_Reboot(char *arg)
{
      TerminalBootMsg();
}


void TFC_ProcessTerminal()
{
     uint8_t i,j;
     uint8_t ArgsFound;
        
    if(TERMINAL_READABLE)
    {
       NextCharIn = TERMINAL_GETC;
       
        switch(NextCharIn)
        {
            case '\r':
             
             TerminalLineBuf[TerminalPos++] = 0x0;
             TERMINAL_PUTC(NextCharIn);
           
             if(TerminalPos > 1)
             {
                 //find the command
                 i=0;
                 while(TerminalLineBuf[i]>0x20 &&  TerminalLineBuf[i]<0x7f)
                 {
                      TerminalCmdBuf[i] = TerminalLineBuf[i];
                      i++;
    
                    if(i==MAX_TERMINAL_CMD_CHARS)
                        {
                         break;
                        }
                 }
                    
                TerminalCmdBuf[i] = 0;
                TerminalCmdBuf[i+1] = 0;
                
                
                ArgsFound = TRUE;
                memset(TerminalArgs,0x00,sizeof(TerminalArgs));
                //scan for num terminator or next non whitespace
                while(TerminalLineBuf[i]<=0x20 && (i<MAX_TERMINAL_LINE_CHARS))
                {
                    if(TerminalLineBuf[i] == 0x00)
                    {
                    
                        //if we find a NULL terminator before a non whitespace character they flag for no arguments
                        ArgsFound = FALSE;
                        break;
                    }   
                    i++; 
                }
                
                if(ArgsFound == TRUE)
                {
                    strcpy(TerminalArgs,&TerminalLineBuf[i]);
                    
                    //trim trailing whitespace
                    i = sizeof(TerminalArgs)-1;
                    
                    while((TerminalArgs[i]<0x21) && (i>0))
                    {
                        TerminalArgs[i]= 0x00;
                        i--;
                    }       
                }
                
                CmdFound = FALSE;
                for(j=0;j<NUM_TERMINAL_COMMANDS;j++)
                {           
                    if(strcmp(TerminalCmdBuf,MyTerminalCallbackRecords[j].CommandString) == 0)
                    {
                        TERMINAL_PRINTF("\r\n");
                        if(MyTerminalCallbackRecords[j].Callback != NULL)
                            MyTerminalCallbackRecords[j].Callback(TerminalArgs);
                    
                        CmdFound = TRUE;
                        break;
                    }             
                }        
                if(CmdFound == FALSE)
                {
                  TERMINAL_PRINTF("\r\n%s command not recognized.\r\n\r\n",TerminalCmdBuf);
                  TerminalCmd_Help("no arg");
                  
                }
              }    
             TERMINAL_PRINTF("\r\n>");
             TerminalPos = 0;
            
            break;
            
            case '\b':
                if(TerminalPos > 0)
                {
                    TerminalPos--;    
                    TERMINAL_PUTC(NextCharIn);
                }
            break;
            
            default:
                
                if(TerminalPos == 0 && NextCharIn == 0x020)
                {
                     //Do nothing if space bar is pressed at beginning of line    
                }
                   else if(NextCharIn >= 0x20 && NextCharIn<0x7F)
                {
                    
                    if(TerminalPos < MAX_TERMINAL_LINE_CHARS-1)
                    {
                         TerminalLineBuf[TerminalPos++] = NextCharIn;
                        TERMINAL_PUTC(NextCharIn);
                    }
                }
            
            break;
        
        }
    }
 
}

