#include "Terminal.h"
#include "DataTypes.h"
#include "mbed.h"
#include "MODSERIAL.h"
#include "diskio.h"
#include "USBHost.h"

//*****************************************************************
//Terminal Configuration
//*****************************************************************

#define MAX_TERMINAL_LINE_CHARS 64
#define MAX_TERMINAL_CMD_CHARS  32
#define TERMINAL_QUEUE_SIZE     512
#define NUM_TERMINAL_CMDS       3


MODSERIAL pc(USBTX, USBRX,TERMINAL_QUEUE_SIZE,TERMINAL_QUEUE_SIZE); // tx, rx

Timer DataSpeedTimer;

typedef void (*TerminalCallback)(char *);

//These are the terminal command names that map to the callbacks
char *TerminalCommands[NUM_TERMINAL_CMDS] = {"help","reboot","dprobe"};

void TerminalCmd_Help(char *arg);
void TerminalCmd_Reboot(char *arg);
void TerminalCmd_dprobe(char *arg);


//Probes a disk
BOOL dprobe(BYTE Disk);
void SpeedTest(BYTE Disk,DWORD SECTORS_TO_READ_WRITE);



//Populate this array with the callback functions
TerminalCallback TerminalCallbacks[NUM_TERMINAL_CMDS] ={
                                                            TerminalCmd_Help,
                                                            TerminalCmd_Reboot,
                                                            TerminalCmd_dprobe
                                                        };

//*****************************************************************
//Plumbing.....
//*****************************************************************


BYTE TerminalInQueueStore[TERMINAL_QUEUE_SIZE];
BYTE TerminalOutQueueStore[TERMINAL_QUEUE_SIZE];
CHAR TerminalLineBuf[MAX_TERMINAL_LINE_CHARS];
BYTE TerminalPos;
CHAR TerminalCmdBuf[MAX_TERMINAL_CMD_CHARS+1];
CHAR TerminalArgs[MAX_TERMINAL_LINE_CHARS-MAX_TERMINAL_CMD_CHARS];
BYTE NextCharIn;
BOOL CmdFound;
 
void TerminalBootMsg()
{

TERMINAL_PRINTF("\r\n\r\n"); 
TERMINAL_PRINTF("***********************************\r\n"); 
TERMINAL_PRINTF("                       /|\r\n"); 
TERMINAL_PRINTF("     ____       __   -- |\r\n");     
TERMINAL_PRINTF("    (___ \\      \\ \\    _|_\r\n");    
TERMINAL_PRINTF("      __) )______\\ \\      \r\n");      
TERMINAL_PRINTF("     / __/(  __  )> \\     \r\n");     
TERMINAL_PRINTF("    | |___ | || |/ ^ \\    \r\n");    
TERMINAL_PRINTF("    |_____)|_||_/_/ \\_\\   \r\n"); 
TERMINAL_PRINTF("                           \r\n");
TERMINAL_PRINTF("Active Pickguard            \r\n");
TERMINAL_PRINTF("Copyright (C) <2011>  Eli Hughes\r\n");
TERMINAL_PRINTF("Wavenumber LLC\r\n"); 
TERMINAL_PRINTF("***********************************\r\n\r\n>"); 

}

void InitTerminal()
{
    TerminalBootMsg();
}

void TerminalCmd_Help(char *arg)
{
    BYTE i;

      TERMINAL_PRINTF("\r\n\r\bCommandList:\r\n");
      TERMINAL_PRINTF("----------------------\r\n");

    for(i=0;i<NUM_TERMINAL_CMDS;i++)
    {
         TERMINAL_PRINTF("%s\r\n",TerminalCommands[i]);    
    }

}

void TerminalCmd_Reboot(char *arg)
{
      TerminalBootMsg();
}

void TerminalCmd_dprobe(char *arg)
{
    if(strcmp(arg,"CF") == 0)
    {
            if(dprobe(COMPACT_FLASH)==TRUE)
                SpeedTest(COMPACT_FLASH,4096);
    }
    else if (strcmp(arg,"USB") == 0)
    {
            if(dprobe(USB)==TRUE)
                SpeedTest(USB,256);
    }
    else if (strcmp(arg,"RAM") == 0)
    {

            if(dprobe(RAM)==TRUE)
                SpeedTest(RAM,8192);
    }
    else
    {
        TERMINAL_PRINTF("\r\nUsage:  dprobe DISK\r\n\r\nDISK can be CF, USB or RAM\r\n");
    }
}


void ProcessTerminal()
{
     BYTE i,j;
     BOOL ArgsFound;
        
    if(TERMINAL_READABLE)
    {
       NextCharIn = TERMINAL_GETC;
       
        switch(NextCharIn)
        {
            case '\r':
             
             TerminalLineBuf[TerminalPos++] = 0x0;
             TERMINAL_PUTC(NextCharIn);
           
             if(TerminalPos > 1)
             {
                 //find the command
                 i=0;
                 while(TerminalLineBuf[i]>0x20 &&  TerminalLineBuf[i]<0x7f)
                 {
                      TerminalCmdBuf[i] = TerminalLineBuf[i];
                      i++;
    
                    if(i==MAX_TERMINAL_CMD_CHARS)
                        {
                         break;
                        }
                 }
                    
                TerminalCmdBuf[i] = 0;
                TerminalCmdBuf[i+1] = 0;
                
                
                ArgsFound = TRUE;
                memset(TerminalArgs,0x00,sizeof(TerminalArgs));
                //scan for num terminator or next non whitespace
                while(TerminalLineBuf[i]<=0x20 && (i<MAX_TERMINAL_LINE_CHARS))
                {
                    if(TerminalLineBuf[i] == 0x00)
                    {
                    
                        //if we find a NULL terminator before a non whitespace character they flag for no arguments
                        ArgsFound = FALSE;
                        break;
                    }   
                    i++; 
                }
                
                if(ArgsFound == TRUE)
                {
                    strcpy(TerminalArgs,&TerminalLineBuf[i]);
                    
                    //trim trailing whitespace
                    i = sizeof(TerminalArgs)-1;
                    
                    while((TerminalArgs[i]<0x21) && (i>0))
                    {
                        TerminalArgs[i]= 0x00;
                        i--;
                    }       
                }
                
                CmdFound = FALSE;
                for(j=0;j<NUM_TERMINAL_CMDS;j++)
                {           
                    if(strcmp(TerminalCmdBuf,TerminalCommands[j]) == 0)
                    {
                        TERMINAL_PRINTF("\r\n");
                        if(TerminalCallbacks[j] != NULL)
                            TerminalCallbacks[j](TerminalArgs);
                    
                        CmdFound = TRUE;
                        break;
                    }             
                }        
                if(CmdFound == FALSE)
                {
                  TERMINAL_PRINTF("\r\n%s command not recognized.\r\n",TerminalCmdBuf);
                }
              }    
             TERMINAL_PRINTF("\r\n>");
             TerminalPos = 0;
            
            break;
            
            case '\b':
                if(TerminalPos > 0)
                {
                    TerminalPos--;    
                    TERMINAL_PUTC(NextCharIn);
                }
            break;
            
            default:
                
                if(TerminalPos == 0 && NextCharIn == 0x020)
                {
                     //Do nothing if space bar is pressed at beginning of line    
                }
                   else if(NextCharIn >= 0x20 && NextCharIn<0x7F)
                {
                    
                    if(TerminalPos < MAX_TERMINAL_LINE_CHARS-1)
                    {
                         TerminalLineBuf[TerminalPos++] = NextCharIn;
                        TERMINAL_PUTC(NextCharIn);
                    }
                }
            
            break;
        
        }
    }
 
}



BYTE DataBuf[512];
BYTE DataBufCheck[512];

void SpeedTest(BYTE Disk, DWORD SECTORS_TO_READ_WRITE)
{

        float DataWriteSpeed = 0;
        float DataReadSpeed = 0;
        float DataReadWriteSpeed = 0;
        int i,j,k;   
        
          
        TERMINAL_PRINTF("\r\nData write test started...\r\n",DataReadSpeed);
        DataSpeedTimer.reset();
        DataSpeedTimer.start();    
        
        for(i=0;i<SECTORS_TO_READ_WRITE;i++)
        {
           USBLoop();
            if(disk_write (Disk,&DataBuf[0],i,1) == RES_ERROR)
               {
                   TERMINAL_PRINTF("Error writing test block %d, stopping.\r\n", i);
                   return; 
               }
        }    
        DataSpeedTimer.stop();
        
        DataWriteSpeed = (((SECTORS_TO_READ_WRITE * 512.0)/1024.0))/(DataSpeedTimer.read());
        
        TERMINAL_PRINTF("Data write speed %.1f kb/s\r\n\r\n",DataWriteSpeed);
        TERMINAL_PRINTF("Data read test started...\r\n",DataReadSpeed);
        
        DataSpeedTimer.reset();
        DataSpeedTimer.start();    
        for(i=0;i<SECTORS_TO_READ_WRITE;i++)
        {
           USBLoop();
            if(disk_read ( Disk,&DataBuf[0],i,1) == RES_ERROR)
               {
               
                   TERMINAL_PRINTF("Error reading test block %d, stopping.\r\n", i);
                   return;
               }
        }
        DataSpeedTimer.stop();
        
        DataReadSpeed = (((SECTORS_TO_READ_WRITE * 512.0)/1024.0))/(DataSpeedTimer.read());
        TERMINAL_PRINTF("Data read speed %.1f kb/s\r\r\n\n",DataReadSpeed);
        TERMINAL_PRINTF("Data read/write test started...\r\n",DataReadSpeed);
        
        DataSpeedTimer.reset();
        DataSpeedTimer.start();    
        
        for(j=0;j<SECTORS_TO_READ_WRITE;j++)
        {
           
           USBLoop();
           for(i=0;i<512;i++)
           {
               DataBuf[i] = rand();
               DataBufCheck[i] = 0;
           }
          
              if(disk_write (Disk,&DataBuf[0],j,1) == RES_ERROR)
               {
                   TERMINAL_PRINTF("Error writing test block %d, stopping.\r\n", i);
                   return;
               }
           
            if(disk_read (Disk,&DataBufCheck[0],j,1) == RES_ERROR)
               {
                   TERMINAL_PRINTF("Error reading test block %d, stopping.\r\n", i);
                   return;
               }
        
               
           for(i=0;i<512;i++)
           {
               if(DataBuf[i]!=DataBufCheck[i])
               {
                     TERMINAL_PRINTF("Readback failure on test block %d byte %d\r\n",j,i);
        
                       for(k=0;k<512;k++)
                       {
                           TERMINAL_PRINTF("k:%d     Wrote: %d  Read: %d\r\n",k,DataBuf[k],DataBufCheck[k]);
                            break;
                       }
                     return;
               }
           }
}
DataSpeedTimer.stop();

DataReadWriteSpeed = (((SECTORS_TO_READ_WRITE * 512.0)/1024.0))/(DataSpeedTimer.read());
TERMINAL_PRINTF("Data read/write speed %.1f kb/s\r\n",DataReadWriteSpeed);
}

BOOL dprobe(BYTE Disk)
{
    DWORD NumSectors;
    CHAR ModelName[41] = {0};
    CHAR SerialNumber[21] = {0};
    CHAR FirmwareVerion[9] = {0};
    BOOL RetVal; 
        
    if(disk_status(Disk)&STA_NODISK)
    {
          TERMINAL_PRINTF("\r\nDisk not attached");
          RetVal = FALSE;
    }
    else
    {
              TERMINAL_PRINTF("\r\n\r\nDisk Detected\r\n\r\n\n");
              if(disk_initialize(Disk)&STA_NOINIT)
              {
                TERMINAL_PRINTF("Disk Failed Initialization\r\n");
                RetVal = FALSE;
              }
              else
              {
                  TERMINAL_PRINTF("Disk Initialization Success!\r\n");
                  
                  
                  TERMINAL_PRINTF("Probing attached Disk\r\n");
                  TERMINAL_PRINTF("-----------------------\r\n");
                  
                  disk_ioctl (Disk,       
                              GET_SECTOR_COUNT,     
                              &NumSectors       
                              ); 
                  disk_ioctl (Disk,       
                              ATA_GET_MODEL,       
                              ModelName       
                              ); 
                  disk_ioctl (Disk,    
                              ATA_GET_REV,      
                              FirmwareVerion      
                              ); 
                  disk_ioctl (Disk,     
                             ATA_GET_SN,      
                              SerialNumber       
                              ); 
                
                
                 TERMINAL_PRINTF("Model Name: %s\r\n", ModelName);
                 TERMINAL_PRINTF("Firmware Version: %s\r\n",FirmwareVerion);
                 TERMINAL_PRINTF("Serial Number: %s\r\n",SerialNumber);
                 TERMINAL_PRINTF("Number of Sectors: %d\r\n",NumSectors);  
                 RetVal = TRUE;
               }
    }
 
    return RetVal;
}