/* ----------------------------------------------------------------------    
* Copyright (C) 2010-2014 ARM Limited. All rights reserved.    
*    
* $Date:        12. March 2014  
* $Revision: 	V1.4.3  
*    
* Project: 	    CMSIS DSP Library    
* Title:	    arm_common_tables.c    
*    
* Description:	This file has common tables like fft twiddle factors, Bitreverse, reciprocal etc which are used across different functions    
*    
* Target Processor: Cortex-M4/Cortex-M3/Cortex-M0
*  
* Redistribution and use in source and binary forms, with or without 
* modification, are permitted provided that the following conditions
* are met:
*   - Redistributions of source code must retain the above copyright
*     notice, this list of conditions and the following disclaimer.
*   - Redistributions in binary form must reproduce the above copyright
*     notice, this list of conditions and the following disclaimer in
*     the documentation and/or other materials provided with the 
*     distribution.
*   - Neither the name of ARM LIMITED nor the names of its contributors
*     may be used to endorse or promote products derived from this
*     software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
* "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
* LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
* FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE 
* COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
* INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
* LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
* ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.   
* -------------------------------------------------------------------- */


#include "arm_math.h"
#include "arm_common_tables.h"

/**    
 * @ingroup groupTransforms    
 */

/**    
 * @addtogroup CFFT_CIFFT Complex FFT Tables  
 * @{    
 */

/**    
* \par    
* Pseudo code for Generation of Bit reversal Table is    
* \par    
* <pre>for(l=1;l <= N/4;l++)    
* {    
*   for(i=0;i<logN2;i++)    
*   {     
*     a[i]=l&(1<<i);    
*   }    
*   for(j=0; j<logN2; j++)    
*   {    
*     if (a[j]!=0)    
*     y[l]+=(1<<((logN2-1)-j));    
*   }    
*   y[l] = y[l] >> 1;    
*  } </pre>    
* \par    
* where N = 4096	logN2 = 12   
* \par    
* N is the maximum FFT Size supported    
*/

/*    
* @brief  Table for bit reversal process    
*/
const uint16_t armBitRevTable[1024] = {
   0x400, 0x200, 0x600, 0x100, 0x500, 0x300, 0x700, 0x80, 0x480, 0x280, 
   0x680, 0x180, 0x580, 0x380, 0x780, 0x40, 0x440, 0x240, 0x640, 0x140, 
   0x540, 0x340, 0x740, 0xc0, 0x4c0, 0x2c0, 0x6c0, 0x1c0, 0x5c0, 0x3c0, 
   0x7c0, 0x20, 0x420, 0x220, 0x620, 0x120, 0x520, 0x320, 0x720, 0xa0, 
   0x4a0, 0x2a0, 0x6a0, 0x1a0, 0x5a0, 0x3a0, 0x7a0, 0x60, 0x460, 0x260, 
   0x660, 0x160, 0x560, 0x360, 0x760, 0xe0, 0x4e0, 0x2e0, 0x6e0, 0x1e0, 
   0x5e0, 0x3e0, 0x7e0, 0x10, 0x410, 0x210, 0x610, 0x110, 0x510, 0x310, 
   0x710, 0x90, 0x490, 0x290, 0x690, 0x190, 0x590, 0x390, 0x790, 0x50, 
   0x450, 0x250, 0x650, 0x150, 0x550, 0x350, 0x750, 0xd0, 0x4d0, 0x2d0, 
   0x6d0, 0x1d0, 0x5d0, 0x3d0, 0x7d0, 0x30, 0x430, 0x230, 0x630, 0x130, 
   0x530, 0x330, 0x730, 0xb0, 0x4b0, 0x2b0, 0x6b0, 0x1b0, 0x5b0, 0x3b0, 
   0x7b0, 0x70, 0x470, 0x270, 0x670, 0x170, 0x570, 0x370, 0x770, 0xf0, 
   0x4f0, 0x2f0, 0x6f0, 0x1f0, 0x5f0, 0x3f0, 0x7f0, 0x8, 0x408, 0x208, 
   0x608, 0x108, 0x508, 0x308, 0x708, 0x88, 0x488, 0x288, 0x688, 0x188, 
   0x588, 0x388, 0x788, 0x48, 0x448, 0x248, 0x648, 0x148, 0x548, 0x348, 
   0x748, 0xc8, 0x4c8, 0x2c8, 0x6c8, 0x1c8, 0x5c8, 0x3c8, 0x7c8, 0x28, 
   0x428, 0x228, 0x628, 0x128, 0x528, 0x328, 0x728, 0xa8, 0x4a8, 0x2a8, 
   0x6a8, 0x1a8, 0x5a8, 0x3a8, 0x7a8, 0x68, 0x468, 0x268, 0x668, 0x168, 
   0x568, 0x368, 0x768, 0xe8, 0x4e8, 0x2e8, 0x6e8, 0x1e8, 0x5e8, 0x3e8, 
   0x7e8, 0x18, 0x418, 0x218, 0x618, 0x118, 0x518, 0x318, 0x718, 0x98, 
   0x498, 0x298, 0x698, 0x198, 0x598, 0x398, 0x798, 0x58, 0x458, 0x258, 
   0x658, 0x158, 0x558, 0x358, 0x758, 0xd8, 0x4d8, 0x2d8, 0x6d8, 0x1d8, 
   0x5d8, 0x3d8, 0x7d8, 0x38, 0x438, 0x238, 0x638, 0x138, 0x538, 0x338, 
   0x738, 0xb8, 0x4b8, 0x2b8, 0x6b8, 0x1b8, 0x5b8, 0x3b8, 0x7b8, 0x78, 
   0x478, 0x278, 0x678, 0x178, 0x578, 0x378, 0x778, 0xf8, 0x4f8, 0x2f8, 
   0x6f8, 0x1f8, 0x5f8, 0x3f8, 0x7f8, 0x4, 0x404, 0x204, 0x604, 0x104, 
   0x504, 0x304, 0x704, 0x84, 0x484, 0x284, 0x684, 0x184, 0x584, 0x384, 
   0x784, 0x44, 0x444, 0x244, 0x644, 0x144, 0x544, 0x344, 0x744, 0xc4, 
   0x4c4, 0x2c4, 0x6c4, 0x1c4, 0x5c4, 0x3c4, 0x7c4, 0x24, 0x424, 0x224, 
   0x624, 0x124, 0x524, 0x324, 0x724, 0xa4, 0x4a4, 0x2a4, 0x6a4, 0x1a4, 
   0x5a4, 0x3a4, 0x7a4, 0x64, 0x464, 0x264, 0x664, 0x164, 0x564, 0x364, 
   0x764, 0xe4, 0x4e4, 0x2e4, 0x6e4, 0x1e4, 0x5e4, 0x3e4, 0x7e4, 0x14, 
   0x414, 0x214, 0x614, 0x114, 0x514, 0x314, 0x714, 0x94, 0x494, 0x294, 
   0x694, 0x194, 0x594, 0x394, 0x794, 0x54, 0x454, 0x254, 0x654, 0x154, 
   0x554, 0x354, 0x754, 0xd4, 0x4d4, 0x2d4, 0x6d4, 0x1d4, 0x5d4, 0x3d4, 
   0x7d4, 0x34, 0x434, 0x234, 0x634, 0x134, 0x534, 0x334, 0x734, 0xb4, 
   0x4b4, 0x2b4, 0x6b4, 0x1b4, 0x5b4, 0x3b4, 0x7b4, 0x74, 0x474, 0x274, 
   0x674, 0x174, 0x574, 0x374, 0x774, 0xf4, 0x4f4, 0x2f4, 0x6f4, 0x1f4, 
   0x5f4, 0x3f4, 0x7f4, 0xc, 0x40c, 0x20c, 0x60c, 0x10c, 0x50c, 0x30c, 
   0x70c, 0x8c, 0x48c, 0x28c, 0x68c, 0x18c, 0x58c, 0x38c, 0x78c, 0x4c, 
   0x44c, 0x24c, 0x64c, 0x14c, 0x54c, 0x34c, 0x74c, 0xcc, 0x4cc, 0x2cc, 
   0x6cc, 0x1cc, 0x5cc, 0x3cc, 0x7cc, 0x2c, 0x42c, 0x22c, 0x62c, 0x12c, 
   0x52c, 0x32c, 0x72c, 0xac, 0x4ac, 0x2ac, 0x6ac, 0x1ac, 0x5ac, 0x3ac, 
   0x7ac, 0x6c, 0x46c, 0x26c, 0x66c, 0x16c, 0x56c, 0x36c, 0x76c, 0xec, 
   0x4ec, 0x2ec, 0x6ec, 0x1ec, 0x5ec, 0x3ec, 0x7ec, 0x1c, 0x41c, 0x21c, 
   0x61c, 0x11c, 0x51c, 0x31c, 0x71c, 0x9c, 0x49c, 0x29c, 0x69c, 0x19c, 
   0x59c, 0x39c, 0x79c, 0x5c, 0x45c, 0x25c, 0x65c, 0x15c, 0x55c, 0x35c, 
   0x75c, 0xdc, 0x4dc, 0x2dc, 0x6dc, 0x1dc, 0x5dc, 0x3dc, 0x7dc, 0x3c, 
   0x43c, 0x23c, 0x63c, 0x13c, 0x53c, 0x33c, 0x73c, 0xbc, 0x4bc, 0x2bc, 
   0x6bc, 0x1bc, 0x5bc, 0x3bc, 0x7bc, 0x7c, 0x47c, 0x27c, 0x67c, 0x17c, 
   0x57c, 0x37c, 0x77c, 0xfc, 0x4fc, 0x2fc, 0x6fc, 0x1fc, 0x5fc, 0x3fc, 
   0x7fc, 0x2, 0x402, 0x202, 0x602, 0x102, 0x502, 0x302, 0x702, 0x82, 
   0x482, 0x282, 0x682, 0x182, 0x582, 0x382, 0x782, 0x42, 0x442, 0x242, 
   0x642, 0x142, 0x542, 0x342, 0x742, 0xc2, 0x4c2, 0x2c2, 0x6c2, 0x1c2, 
   0x5c2, 0x3c2, 0x7c2, 0x22, 0x422, 0x222, 0x622, 0x122, 0x522, 0x322, 
   0x722, 0xa2, 0x4a2, 0x2a2, 0x6a2, 0x1a2, 0x5a2, 0x3a2, 0x7a2, 0x62, 
   0x462, 0x262, 0x662, 0x162, 0x562, 0x362, 0x762, 0xe2, 0x4e2, 0x2e2, 
   0x6e2, 0x1e2, 0x5e2, 0x3e2, 0x7e2, 0x12, 0x412, 0x212, 0x612, 0x112, 
   0x512, 0x312, 0x712, 0x92, 0x492, 0x292, 0x692, 0x192, 0x592, 0x392, 
   0x792, 0x52, 0x452, 0x252, 0x652, 0x152, 0x552, 0x352, 0x752, 0xd2, 
   0x4d2, 0x2d2, 0x6d2, 0x1d2, 0x5d2, 0x3d2, 0x7d2, 0x32, 0x432, 0x232, 
   0x632, 0x132, 0x532, 0x332, 0x732, 0xb2, 0x4b2, 0x2b2, 0x6b2, 0x1b2, 
   0x5b2, 0x3b2, 0x7b2, 0x72, 0x472, 0x272, 0x672, 0x172, 0x572, 0x372, 
   0x772, 0xf2, 0x4f2, 0x2f2, 0x6f2, 0x1f2, 0x5f2, 0x3f2, 0x7f2, 0xa, 
   0x40a, 0x20a, 0x60a, 0x10a, 0x50a, 0x30a, 0x70a, 0x8a, 0x48a, 0x28a, 
   0x68a, 0x18a, 0x58a, 0x38a, 0x78a, 0x4a, 0x44a, 0x24a, 0x64a, 0x14a, 
   0x54a, 0x34a, 0x74a, 0xca, 0x4ca, 0x2ca, 0x6ca, 0x1ca, 0x5ca, 0x3ca, 
   0x7ca, 0x2a, 0x42a, 0x22a, 0x62a, 0x12a, 0x52a, 0x32a, 0x72a, 0xaa, 
   0x4aa, 0x2aa, 0x6aa, 0x1aa, 0x5aa, 0x3aa, 0x7aa, 0x6a, 0x46a, 0x26a, 
   0x66a, 0x16a, 0x56a, 0x36a, 0x76a, 0xea, 0x4ea, 0x2ea, 0x6ea, 0x1ea, 
   0x5ea, 0x3ea, 0x7ea, 0x1a, 0x41a, 0x21a, 0x61a, 0x11a, 0x51a, 0x31a, 
   0x71a, 0x9a, 0x49a, 0x29a, 0x69a, 0x19a, 0x59a, 0x39a, 0x79a, 0x5a, 
   0x45a, 0x25a, 0x65a, 0x15a, 0x55a, 0x35a, 0x75a, 0xda, 0x4da, 0x2da, 
   0x6da, 0x1da, 0x5da, 0x3da, 0x7da, 0x3a, 0x43a, 0x23a, 0x63a, 0x13a, 
   0x53a, 0x33a, 0x73a, 0xba, 0x4ba, 0x2ba, 0x6ba, 0x1ba, 0x5ba, 0x3ba, 
   0x7ba, 0x7a, 0x47a, 0x27a, 0x67a, 0x17a, 0x57a, 0x37a, 0x77a, 0xfa, 
   0x4fa, 0x2fa, 0x6fa, 0x1fa, 0x5fa, 0x3fa, 0x7fa, 0x6, 0x406, 0x206, 
   0x606, 0x106, 0x506, 0x306, 0x706, 0x86, 0x486, 0x286, 0x686, 0x186, 
   0x586, 0x386, 0x786, 0x46, 0x446, 0x246, 0x646, 0x146, 0x546, 0x346, 
   0x746, 0xc6, 0x4c6, 0x2c6, 0x6c6, 0x1c6, 0x5c6, 0x3c6, 0x7c6, 0x26, 
   0x426, 0x226, 0x626, 0x126, 0x526, 0x326, 0x726, 0xa6, 0x4a6, 0x2a6, 
   0x6a6, 0x1a6, 0x5a6, 0x3a6, 0x7a6, 0x66, 0x466, 0x266, 0x666, 0x166, 
   0x566, 0x366, 0x766, 0xe6, 0x4e6, 0x2e6, 0x6e6, 0x1e6, 0x5e6, 0x3e6, 
   0x7e6, 0x16, 0x416, 0x216, 0x616, 0x116, 0x516, 0x316, 0x716, 0x96, 
   0x496, 0x296, 0x696, 0x196, 0x596, 0x396, 0x796, 0x56, 0x456, 0x256, 
   0x656, 0x156, 0x556, 0x356, 0x756, 0xd6, 0x4d6, 0x2d6, 0x6d6, 0x1d6, 
   0x5d6, 0x3d6, 0x7d6, 0x36, 0x436, 0x236, 0x636, 0x136, 0x536, 0x336, 
   0x736, 0xb6, 0x4b6, 0x2b6, 0x6b6, 0x1b6, 0x5b6, 0x3b6, 0x7b6, 0x76, 
   0x476, 0x276, 0x676, 0x176, 0x576, 0x376, 0x776, 0xf6, 0x4f6, 0x2f6, 
   0x6f6, 0x1f6, 0x5f6, 0x3f6, 0x7f6, 0xe, 0x40e, 0x20e, 0x60e, 0x10e, 
   0x50e, 0x30e, 0x70e, 0x8e, 0x48e, 0x28e, 0x68e, 0x18e, 0x58e, 0x38e, 
   0x78e, 0x4e, 0x44e, 0x24e, 0x64e, 0x14e, 0x54e, 0x34e, 0x74e, 0xce, 
   0x4ce, 0x2ce, 0x6ce, 0x1ce, 0x5ce, 0x3ce, 0x7ce, 0x2e, 0x42e, 0x22e, 
   0x62e, 0x12e, 0x52e, 0x32e, 0x72e, 0xae, 0x4ae, 0x2ae, 0x6ae, 0x1ae, 
   0x5ae, 0x3ae, 0x7ae, 0x6e, 0x46e, 0x26e, 0x66e, 0x16e, 0x56e, 0x36e, 
   0x76e, 0xee, 0x4ee, 0x2ee, 0x6ee, 0x1ee, 0x5ee, 0x3ee, 0x7ee, 0x1e, 
   0x41e, 0x21e, 0x61e, 0x11e, 0x51e, 0x31e, 0x71e, 0x9e, 0x49e, 0x29e, 
   0x69e, 0x19e, 0x59e, 0x39e, 0x79e, 0x5e, 0x45e, 0x25e, 0x65e, 0x15e, 
   0x55e, 0x35e, 0x75e, 0xde, 0x4de, 0x2de, 0x6de, 0x1de, 0x5de, 0x3de, 
   0x7de, 0x3e, 0x43e, 0x23e, 0x63e, 0x13e, 0x53e, 0x33e, 0x73e, 0xbe, 
   0x4be, 0x2be, 0x6be, 0x1be, 0x5be, 0x3be, 0x7be, 0x7e, 0x47e, 0x27e, 
   0x67e, 0x17e, 0x57e, 0x37e, 0x77e, 0xfe, 0x4fe, 0x2fe, 0x6fe, 0x1fe, 
   0x5fe, 0x3fe, 0x7fe, 0x1 
};


/*    
* @brief  Floating-point Twiddle factors Table Generation    
*/

/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/
const float32_t twiddleCoef_16[32] = {
    1.000000000f,  0.000000000f,
    0.923879533f,  0.382683432f,
    0.707106781f,  0.707106781f,
    0.382683432f,  0.923879533f,
    0.000000000f,  1.000000000f,
   -0.382683432f,  0.923879533f,
   -0.707106781f,  0.707106781f,
   -0.923879533f,  0.382683432f,
   -1.000000000f,  0.000000000f,
   -0.923879533f, -0.382683432f,
   -0.707106781f, -0.707106781f,
   -0.382683432f, -0.923879533f,
   -0.000000000f, -1.000000000f,
    0.382683432f, -0.923879533f,
    0.707106781f, -0.707106781f,
    0.923879533f, -0.382683432f
};

/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/
const float32_t twiddleCoef_32[64] = {
    1.000000000f,  0.000000000f,
    0.980785280f,  0.195090322f,
    0.923879533f,  0.382683432f,
    0.831469612f,  0.555570233f,
    0.707106781f,  0.707106781f,
    0.555570233f,  0.831469612f,
    0.382683432f,  0.923879533f,
    0.195090322f,  0.980785280f,
    0.000000000f,  1.000000000f,
   -0.195090322f,  0.980785280f,
   -0.382683432f,  0.923879533f,
   -0.555570233f,  0.831469612f,
   -0.707106781f,  0.707106781f,
   -0.831469612f,  0.555570233f,
   -0.923879533f,  0.382683432f,
   -0.980785280f,  0.195090322f,
   -1.000000000f,  0.000000000f,
   -0.980785280f, -0.195090322f,
   -0.923879533f, -0.382683432f,
   -0.831469612f, -0.555570233f,
   -0.707106781f, -0.707106781f,
   -0.555570233f, -0.831469612f,
   -0.382683432f, -0.923879533f,
   -0.195090322f, -0.980785280f,
   -0.000000000f, -1.000000000f,
    0.195090322f, -0.980785280f,
    0.382683432f, -0.923879533f,
    0.555570233f, -0.831469612f,
    0.707106781f, -0.707106781f,
    0.831469612f, -0.555570233f,
    0.923879533f, -0.382683432f,
    0.980785280f, -0.195090322f
};

/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/
const float32_t twiddleCoef_64[128] = {
    1.000000000f,  0.000000000f,
    0.995184727f,  0.098017140f,
    0.980785280f,  0.195090322f,
    0.956940336f,  0.290284677f,
    0.923879533f,  0.382683432f,
    0.881921264f,  0.471396737f,
    0.831469612f,  0.555570233f,
    0.773010453f,  0.634393284f,
    0.707106781f,  0.707106781f,
    0.634393284f,  0.773010453f,
    0.555570233f,  0.831469612f,
    0.471396737f,  0.881921264f,
    0.382683432f,  0.923879533f,
    0.290284677f,  0.956940336f,
    0.195090322f,  0.980785280f,
    0.098017140f,  0.995184727f,
    0.000000000f,  1.000000000f,
   -0.098017140f,  0.995184727f,
   -0.195090322f,  0.980785280f,
   -0.290284677f,  0.956940336f,
   -0.382683432f,  0.923879533f,
   -0.471396737f,  0.881921264f,
   -0.555570233f,  0.831469612f,
   -0.634393284f,  0.773010453f,
   -0.707106781f,  0.707106781f,
   -0.773010453f,  0.634393284f,
   -0.831469612f,  0.555570233f,
   -0.881921264f,  0.471396737f,
   -0.923879533f,  0.382683432f,
   -0.956940336f,  0.290284677f,
   -0.980785280f,  0.195090322f,
   -0.995184727f,  0.098017140f,
   -1.000000000f,  0.000000000f,
   -0.995184727f, -0.098017140f,
   -0.980785280f, -0.195090322f,
   -0.956940336f, -0.290284677f,
   -0.923879533f, -0.382683432f,
   -0.881921264f, -0.471396737f,
   -0.831469612f, -0.555570233f,
   -0.773010453f, -0.634393284f,
   -0.707106781f, -0.707106781f,
   -0.634393284f, -0.773010453f,
   -0.555570233f, -0.831469612f,
   -0.471396737f, -0.881921264f,
   -0.382683432f, -0.923879533f,
   -0.290284677f, -0.956940336f,
   -0.195090322f, -0.980785280f,
   -0.098017140f, -0.995184727f,
   -0.000000000f, -1.000000000f,
    0.098017140f, -0.995184727f,
    0.195090322f, -0.980785280f,
    0.290284677f, -0.956940336f,
    0.382683432f, -0.923879533f,
    0.471396737f, -0.881921264f,
    0.555570233f, -0.831469612f,
    0.634393284f, -0.773010453f,
    0.707106781f, -0.707106781f,
    0.773010453f, -0.634393284f,
    0.831469612f, -0.555570233f,
    0.881921264f, -0.471396737f,
    0.923879533f, -0.382683432f,
    0.956940336f, -0.290284677f,
    0.980785280f, -0.195090322f,
    0.995184727f, -0.098017140f
};

/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/

const float32_t twiddleCoef_128[256] = {
1.000000000f	,	0.000000000f	,
0.998795456f	,	0.049067674f	,
0.995184727f	,	0.098017140f	,
0.989176510f	,	0.146730474f	,
0.980785280f	,	0.195090322f	,
0.970031253f	,	0.242980180f	,
0.956940336f	,	0.290284677f	,
0.941544065f	,	0.336889853f	,
0.923879533f	,	0.382683432f	,
0.903989293f	,	0.427555093f	,
0.881921264f	,	0.471396737f	,
0.857728610f	,	0.514102744f	,
0.831469612f	,	0.555570233f	,
0.803207531f	,	0.595699304f	,
0.773010453f	,	0.634393284f	,
0.740951125f	,	0.671558955f	,
0.707106781f	,	0.707106781f	,
0.671558955f	,	0.740951125f	,
0.634393284f	,	0.773010453f	,
0.595699304f	,	0.803207531f	,
0.555570233f	,	0.831469612f	,
0.514102744f	,	0.857728610f	,
0.471396737f	,	0.881921264f	,
0.427555093f	,	0.903989293f	,
0.382683432f	,	0.923879533f	,
0.336889853f	,	0.941544065f	,
0.290284677f	,	0.956940336f	,
0.242980180f	,	0.970031253f	,
0.195090322f	,	0.980785280f	,
0.146730474f	,	0.989176510f	,
0.098017140f	,	0.995184727f	,
0.049067674f	,	0.998795456f	,
0.000000000f	,	1.000000000f	,
-0.049067674f	,	0.998795456f	,
-0.098017140f	,	0.995184727f	,
-0.146730474f	,	0.989176510f	,
-0.195090322f	,	0.980785280f	,
-0.242980180f	,	0.970031253f	,
-0.290284677f	,	0.956940336f	,
-0.336889853f	,	0.941544065f	,
-0.382683432f	,	0.923879533f	,
-0.427555093f	,	0.903989293f	,
-0.471396737f	,	0.881921264f	,
-0.514102744f	,	0.857728610f	,
-0.555570233f	,	0.831469612f	,
-0.595699304f	,	0.803207531f	,
-0.634393284f	,	0.773010453f	,
-0.671558955f	,	0.740951125f	,
-0.707106781f	,	0.707106781f	,
-0.740951125f	,	0.671558955f	,
-0.773010453f	,	0.634393284f	,
-0.803207531f	,	0.595699304f	,
-0.831469612f	,	0.555570233f	,
-0.857728610f	,	0.514102744f	,
-0.881921264f	,	0.471396737f	,
-0.903989293f	,	0.427555093f	,
-0.923879533f	,	0.382683432f	,
-0.941544065f	,	0.336889853f	,
-0.956940336f	,	0.290284677f	,
-0.970031253f	,	0.242980180f	,
-0.980785280f	,	0.195090322f	,
-0.989176510f	,	0.146730474f	,
-0.995184727f	,	0.098017140f	,
-0.998795456f	,	0.049067674f	,
-1.000000000f	,	0.000000000f	,
-0.998795456f	,	-0.049067674f	,
-0.995184727f	,	-0.098017140f	,
-0.989176510f	,	-0.146730474f	,
-0.980785280f	,	-0.195090322f	,
-0.970031253f	,	-0.242980180f	,
-0.956940336f	,	-0.290284677f	,
-0.941544065f	,	-0.336889853f	,
-0.923879533f	,	-0.382683432f	,
-0.903989293f	,	-0.427555093f	,
-0.881921264f	,	-0.471396737f	,
-0.857728610f	,	-0.514102744f	,
-0.831469612f	,	-0.555570233f	,
-0.803207531f	,	-0.595699304f	,
-0.773010453f	,	-0.634393284f	,
-0.740951125f	,	-0.671558955f	,
-0.707106781f	,	-0.707106781f	,
-0.671558955f	,	-0.740951125f	,
-0.634393284f	,	-0.773010453f	,
-0.595699304f	,	-0.803207531f	,
-0.555570233f	,	-0.831469612f	,
-0.514102744f	,	-0.857728610f	,
-0.471396737f	,	-0.881921264f	,
-0.427555093f	,	-0.903989293f	,
-0.382683432f	,	-0.923879533f	,
-0.336889853f	,	-0.941544065f	,
-0.290284677f	,	-0.956940336f	,
-0.242980180f	,	-0.970031253f	,
-0.195090322f	,	-0.980785280f	,
-0.146730474f	,	-0.989176510f	,
-0.098017140f	,	-0.995184727f	,
-0.049067674f	,	-0.998795456f	,
-0.000000000f	,	-1.000000000f	,
0.049067674f	,	-0.998795456f	,
0.098017140f	,	-0.995184727f	,
0.146730474f	,	-0.989176510f	,
0.195090322f	,	-0.980785280f	,
0.242980180f	,	-0.970031253f	,
0.290284677f	,	-0.956940336f	,
0.336889853f	,	-0.941544065f	,
0.382683432f	,	-0.923879533f	,
0.427555093f	,	-0.903989293f	,
0.471396737f	,	-0.881921264f	,
0.514102744f	,	-0.857728610f	,
0.555570233f	,	-0.831469612f	,
0.595699304f	,	-0.803207531f	,
0.634393284f	,	-0.773010453f	,
0.671558955f	,	-0.740951125f	,
0.707106781f	,	-0.707106781f	,
0.740951125f	,	-0.671558955f	,
0.773010453f	,	-0.634393284f	,
0.803207531f	,	-0.595699304f	,
0.831469612f	,	-0.555570233f	,
0.857728610f	,	-0.514102744f	,
0.881921264f	,	-0.471396737f	,
0.903989293f	,	-0.427555093f	,
0.923879533f	,	-0.382683432f	,
0.941544065f	,	-0.336889853f	,
0.956940336f	,	-0.290284677f	,
0.970031253f	,	-0.242980180f	,
0.980785280f	,	-0.195090322f	,
0.989176510f	,	-0.146730474f	,
0.995184727f	,	-0.098017140f	,
0.998795456f	,	-0.049067674f
};

/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/
const float32_t twiddleCoef_256[512] = {
    1.000000000f,  0.000000000f,
    0.999698819f,  0.024541229f,
    0.998795456f,  0.049067674f,
    0.997290457f,  0.073564564f,
    0.995184727f,  0.098017140f,
    0.992479535f,  0.122410675f,
    0.989176510f,  0.146730474f,
    0.985277642f,  0.170961889f,
    0.980785280f,  0.195090322f,
    0.975702130f,  0.219101240f,
    0.970031253f,  0.242980180f,
    0.963776066f,  0.266712757f,
    0.956940336f,  0.290284677f,
    0.949528181f,  0.313681740f,
    0.941544065f,  0.336889853f,
    0.932992799f,  0.359895037f,
    0.923879533f,  0.382683432f,
    0.914209756f,  0.405241314f,
    0.903989293f,  0.427555093f,
    0.893224301f,  0.449611330f,
    0.881921264f,  0.471396737f,
    0.870086991f,  0.492898192f,
    0.857728610f,  0.514102744f,
    0.844853565f,  0.534997620f,
    0.831469612f,  0.555570233f,
    0.817584813f,  0.575808191f,
    0.803207531f,  0.595699304f,
    0.788346428f,  0.615231591f,
    0.773010453f,  0.634393284f,
    0.757208847f,  0.653172843f,
    0.740951125f,  0.671558955f,
    0.724247083f,  0.689540545f,
    0.707106781f,  0.707106781f,
    0.689540545f,  0.724247083f,
    0.671558955f,  0.740951125f,
    0.653172843f,  0.757208847f,
    0.634393284f,  0.773010453f,
    0.615231591f,  0.788346428f,
    0.595699304f,  0.803207531f,
    0.575808191f,  0.817584813f,
    0.555570233f,  0.831469612f,
    0.534997620f,  0.844853565f,
    0.514102744f,  0.857728610f,
    0.492898192f,  0.870086991f,
    0.471396737f,  0.881921264f,
    0.449611330f,  0.893224301f,
    0.427555093f,  0.903989293f,
    0.405241314f,  0.914209756f,
    0.382683432f,  0.923879533f,
    0.359895037f,  0.932992799f,
    0.336889853f,  0.941544065f,
    0.313681740f,  0.949528181f,
    0.290284677f,  0.956940336f,
    0.266712757f,  0.963776066f,
    0.242980180f,  0.970031253f,
    0.219101240f,  0.975702130f,
    0.195090322f,  0.980785280f,
    0.170961889f,  0.985277642f,
    0.146730474f,  0.989176510f,
    0.122410675f,  0.992479535f,
    0.098017140f,  0.995184727f,
    0.073564564f,  0.997290457f,
    0.049067674f,  0.998795456f,
    0.024541229f,  0.999698819f,
    0.000000000f,  1.000000000f,
   -0.024541229f,  0.999698819f,
   -0.049067674f,  0.998795456f,
   -0.073564564f,  0.997290457f,
   -0.098017140f,  0.995184727f,
   -0.122410675f,  0.992479535f,
   -0.146730474f,  0.989176510f,
   -0.170961889f,  0.985277642f,
   -0.195090322f,  0.980785280f,
   -0.219101240f,  0.975702130f,
   -0.242980180f,  0.970031253f,
   -0.266712757f,  0.963776066f,
   -0.290284677f,  0.956940336f,
   -0.313681740f,  0.949528181f,
   -0.336889853f,  0.941544065f,
   -0.359895037f,  0.932992799f,
   -0.382683432f,  0.923879533f,
   -0.405241314f,  0.914209756f,
   -0.427555093f,  0.903989293f,
   -0.449611330f,  0.893224301f,
   -0.471396737f,  0.881921264f,
   -0.492898192f,  0.870086991f,
   -0.514102744f,  0.857728610f,
   -0.534997620f,  0.844853565f,
   -0.555570233f,  0.831469612f,
   -0.575808191f,  0.817584813f,
   -0.595699304f,  0.803207531f,
   -0.615231591f,  0.788346428f,
   -0.634393284f,  0.773010453f,
   -0.653172843f,  0.757208847f,
   -0.671558955f,  0.740951125f,
   -0.689540545f,  0.724247083f,
   -0.707106781f,  0.707106781f,
   -0.724247083f,  0.689540545f,
   -0.740951125f,  0.671558955f,
   -0.757208847f,  0.653172843f,
   -0.773010453f,  0.634393284f,
   -0.788346428f,  0.615231591f,
   -0.803207531f,  0.595699304f,
   -0.817584813f,  0.575808191f,
   -0.831469612f,  0.555570233f,
   -0.844853565f,  0.534997620f,
   -0.857728610f,  0.514102744f,
   -0.870086991f,  0.492898192f,
   -0.881921264f,  0.471396737f,
   -0.893224301f,  0.449611330f,
   -0.903989293f,  0.427555093f,
   -0.914209756f,  0.405241314f,
   -0.923879533f,  0.382683432f,
   -0.932992799f,  0.359895037f,
   -0.941544065f,  0.336889853f,
   -0.949528181f,  0.313681740f,
   -0.956940336f,  0.290284677f,
   -0.963776066f,  0.266712757f,
   -0.970031253f,  0.242980180f,
   -0.975702130f,  0.219101240f,
   -0.980785280f,  0.195090322f,
   -0.985277642f,  0.170961889f,
   -0.989176510f,  0.146730474f,
   -0.992479535f,  0.122410675f,
   -0.995184727f,  0.098017140f,
   -0.997290457f,  0.073564564f,
   -0.998795456f,  0.049067674f,
   -0.999698819f,  0.024541229f,
   -1.000000000f,  0.000000000f,
   -0.999698819f, -0.024541229f,
   -0.998795456f, -0.049067674f,
   -0.997290457f, -0.073564564f,
   -0.995184727f, -0.098017140f,
   -0.992479535f, -0.122410675f,
   -0.989176510f, -0.146730474f,
   -0.985277642f, -0.170961889f,
   -0.980785280f, -0.195090322f,
   -0.975702130f, -0.219101240f,
   -0.970031253f, -0.242980180f,
   -0.963776066f, -0.266712757f,
   -0.956940336f, -0.290284677f,
   -0.949528181f, -0.313681740f,
   -0.941544065f, -0.336889853f,
   -0.932992799f, -0.359895037f,
   -0.923879533f, -0.382683432f,
   -0.914209756f, -0.405241314f,
   -0.903989293f, -0.427555093f,
   -0.893224301f, -0.449611330f,
   -0.881921264f, -0.471396737f,
   -0.870086991f, -0.492898192f,
   -0.857728610f, -0.514102744f,
   -0.844853565f, -0.534997620f,
   -0.831469612f, -0.555570233f,
   -0.817584813f, -0.575808191f,
   -0.803207531f, -0.595699304f,
   -0.788346428f, -0.615231591f,
   -0.773010453f, -0.634393284f,
   -0.757208847f, -0.653172843f,
   -0.740951125f, -0.671558955f,
   -0.724247083f, -0.689540545f,
   -0.707106781f, -0.707106781f,
   -0.689540545f, -0.724247083f,
   -0.671558955f, -0.740951125f,
   -0.653172843f, -0.757208847f,
   -0.634393284f, -0.773010453f,
   -0.615231591f, -0.788346428f,
   -0.595699304f, -0.803207531f,
   -0.575808191f, -0.817584813f,
   -0.555570233f, -0.831469612f,
   -0.534997620f, -0.844853565f,
   -0.514102744f, -0.857728610f,
   -0.492898192f, -0.870086991f,
   -0.471396737f, -0.881921264f,
   -0.449611330f, -0.893224301f,
   -0.427555093f, -0.903989293f,
   -0.405241314f, -0.914209756f,
   -0.382683432f, -0.923879533f,
   -0.359895037f, -0.932992799f,
   -0.336889853f, -0.941544065f,
   -0.313681740f, -0.949528181f,
   -0.290284677f, -0.956940336f,
   -0.266712757f, -0.963776066f,
   -0.242980180f, -0.970031253f,
   -0.219101240f, -0.975702130f,
   -0.195090322f, -0.980785280f,
   -0.170961889f, -0.985277642f,
   -0.146730474f, -0.989176510f,
   -0.122410675f, -0.992479535f,
   -0.098017140f, -0.995184727f,
   -0.073564564f, -0.997290457f,
   -0.049067674f, -0.998795456f,
   -0.024541229f, -0.999698819f,
   -0.000000000f, -1.000000000f,
    0.024541229f, -0.999698819f,
    0.049067674f, -0.998795456f,
    0.073564564f, -0.997290457f,
    0.098017140f, -0.995184727f,
    0.122410675f, -0.992479535f,
    0.146730474f, -0.989176510f,
    0.170961889f, -0.985277642f,
    0.195090322f, -0.980785280f,
    0.219101240f, -0.975702130f,
    0.242980180f, -0.970031253f,
    0.266712757f, -0.963776066f,
    0.290284677f, -0.956940336f,
    0.313681740f, -0.949528181f,
    0.336889853f, -0.941544065f,
    0.359895037f, -0.932992799f,
    0.382683432f, -0.923879533f,
    0.405241314f, -0.914209756f,
    0.427555093f, -0.903989293f,
    0.449611330f, -0.893224301f,
    0.471396737f, -0.881921264f,
    0.492898192f, -0.870086991f,
    0.514102744f, -0.857728610f,
    0.534997620f, -0.844853565f,
    0.555570233f, -0.831469612f,
    0.575808191f, -0.817584813f,
    0.595699304f, -0.803207531f,
    0.615231591f, -0.788346428f,
    0.634393284f, -0.773010453f,
    0.653172843f, -0.757208847f,
    0.671558955f, -0.740951125f,
    0.689540545f, -0.724247083f,
    0.707106781f, -0.707106781f,
    0.724247083f, -0.689540545f,
    0.740951125f, -0.671558955f,
    0.757208847f, -0.653172843f,
    0.773010453f, -0.634393284f,
    0.788346428f, -0.615231591f,
    0.803207531f, -0.595699304f,
    0.817584813f, -0.575808191f,
    0.831469612f, -0.555570233f,
    0.844853565f, -0.534997620f,
    0.857728610f, -0.514102744f,
    0.870086991f, -0.492898192f,
    0.881921264f, -0.471396737f,
    0.893224301f, -0.449611330f,
    0.903989293f, -0.427555093f,
    0.914209756f, -0.405241314f,
    0.923879533f, -0.382683432f,
    0.932992799f, -0.359895037f,
    0.941544065f, -0.336889853f,
    0.949528181f, -0.313681740f,
    0.956940336f, -0.290284677f,
    0.963776066f, -0.266712757f,
    0.970031253f, -0.242980180f,
    0.975702130f, -0.219101240f,
    0.980785280f, -0.195090322f,
    0.985277642f, -0.170961889f,
    0.989176510f, -0.146730474f,
    0.992479535f, -0.122410675f,
    0.995184727f, -0.098017140f,
    0.997290457f, -0.073564564f,
    0.998795456f, -0.049067674f,
    0.999698819f, -0.024541229f
};

/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/
const float32_t twiddleCoef_512[1024] = {
    1.000000000f,  0.000000000f,
    0.999924702f,  0.012271538f,
    0.999698819f,  0.024541229f,
    0.999322385f,  0.036807223f,
    0.998795456f,  0.049067674f,
    0.998118113f,  0.061320736f,
    0.997290457f,  0.073564564f,
    0.996312612f,  0.085797312f,
    0.995184727f,  0.098017140f,
    0.993906970f,  0.110222207f,
    0.992479535f,  0.122410675f,
    0.990902635f,  0.134580709f,
    0.989176510f,  0.146730474f,
    0.987301418f,  0.158858143f,
    0.985277642f,  0.170961889f,
    0.983105487f,  0.183039888f,
    0.980785280f,  0.195090322f,
    0.978317371f,  0.207111376f,
    0.975702130f,  0.219101240f,
    0.972939952f,  0.231058108f,
    0.970031253f,  0.242980180f,
    0.966976471f,  0.254865660f,
    0.963776066f,  0.266712757f,
    0.960430519f,  0.278519689f,
    0.956940336f,  0.290284677f,
    0.953306040f,  0.302005949f,
    0.949528181f,  0.313681740f,
    0.945607325f,  0.325310292f,
    0.941544065f,  0.336889853f,
    0.937339012f,  0.348418680f,
    0.932992799f,  0.359895037f,
    0.928506080f,  0.371317194f,
    0.923879533f,  0.382683432f,
    0.919113852f,  0.393992040f,
    0.914209756f,  0.405241314f,
    0.909167983f,  0.416429560f,
    0.903989293f,  0.427555093f,
    0.898674466f,  0.438616239f,
    0.893224301f,  0.449611330f,
    0.887639620f,  0.460538711f,
    0.881921264f,  0.471396737f,
    0.876070094f,  0.482183772f,
    0.870086991f,  0.492898192f,
    0.863972856f,  0.503538384f,
    0.857728610f,  0.514102744f,
    0.851355193f,  0.524589683f,
    0.844853565f,  0.534997620f,
    0.838224706f,  0.545324988f,
    0.831469612f,  0.555570233f,
    0.824589303f,  0.565731811f,
    0.817584813f,  0.575808191f,
    0.810457198f,  0.585797857f,
    0.803207531f,  0.595699304f,
    0.795836905f,  0.605511041f,
    0.788346428f,  0.615231591f,
    0.780737229f,  0.624859488f,
    0.773010453f,  0.634393284f,
    0.765167266f,  0.643831543f,
    0.757208847f,  0.653172843f,
    0.749136395f,  0.662415778f,
    0.740951125f,  0.671558955f,
    0.732654272f,  0.680600998f,
    0.724247083f,  0.689540545f,
    0.715730825f,  0.698376249f,
    0.707106781f,  0.707106781f,
    0.698376249f,  0.715730825f,
    0.689540545f,  0.724247083f,
    0.680600998f,  0.732654272f,
    0.671558955f,  0.740951125f,
    0.662415778f,  0.749136395f,
    0.653172843f,  0.757208847f,
    0.643831543f,  0.765167266f,
    0.634393284f,  0.773010453f,
    0.624859488f,  0.780737229f,
    0.615231591f,  0.788346428f,
    0.605511041f,  0.795836905f,
    0.595699304f,  0.803207531f,
    0.585797857f,  0.810457198f,
    0.575808191f,  0.817584813f,
    0.565731811f,  0.824589303f,
    0.555570233f,  0.831469612f,
    0.545324988f,  0.838224706f,
    0.534997620f,  0.844853565f,
    0.524589683f,  0.851355193f,
    0.514102744f,  0.857728610f,
    0.503538384f,  0.863972856f,
    0.492898192f,  0.870086991f,
    0.482183772f,  0.876070094f,
    0.471396737f,  0.881921264f,
    0.460538711f,  0.887639620f,
    0.449611330f,  0.893224301f,
    0.438616239f,  0.898674466f,
    0.427555093f,  0.903989293f,
    0.416429560f,  0.909167983f,
    0.405241314f,  0.914209756f,
    0.393992040f,  0.919113852f,
    0.382683432f,  0.923879533f,
    0.371317194f,  0.928506080f,
    0.359895037f,  0.932992799f,
    0.348418680f,  0.937339012f,
    0.336889853f,  0.941544065f,
    0.325310292f,  0.945607325f,
    0.313681740f,  0.949528181f,
    0.302005949f,  0.953306040f,
    0.290284677f,  0.956940336f,
    0.278519689f,  0.960430519f,
    0.266712757f,  0.963776066f,
    0.254865660f,  0.966976471f,
    0.242980180f,  0.970031253f,
    0.231058108f,  0.972939952f,
    0.219101240f,  0.975702130f,
    0.207111376f,  0.978317371f,
    0.195090322f,  0.980785280f,
    0.183039888f,  0.983105487f,
    0.170961889f,  0.985277642f,
    0.158858143f,  0.987301418f,
    0.146730474f,  0.989176510f,
    0.134580709f,  0.990902635f,
    0.122410675f,  0.992479535f,
    0.110222207f,  0.993906970f,
    0.098017140f,  0.995184727f,
    0.085797312f,  0.996312612f,
    0.073564564f,  0.997290457f,
    0.061320736f,  0.998118113f,
    0.049067674f,  0.998795456f,
    0.036807223f,  0.999322385f,
    0.024541229f,  0.999698819f,
    0.012271538f,  0.999924702f,
    0.000000000f,  1.000000000f,
   -0.012271538f,  0.999924702f,
   -0.024541229f,  0.999698819f,
   -0.036807223f,  0.999322385f,
   -0.049067674f,  0.998795456f,
   -0.061320736f,  0.998118113f,
   -0.073564564f,  0.997290457f,
   -0.085797312f,  0.996312612f,
   -0.098017140f,  0.995184727f,
   -0.110222207f,  0.993906970f,
   -0.122410675f,  0.992479535f,
   -0.134580709f,  0.990902635f,
   -0.146730474f,  0.989176510f,
   -0.158858143f,  0.987301418f,
   -0.170961889f,  0.985277642f,
   -0.183039888f,  0.983105487f,
   -0.195090322f,  0.980785280f,
   -0.207111376f,  0.978317371f,
   -0.219101240f,  0.975702130f,
   -0.231058108f,  0.972939952f,
   -0.242980180f,  0.970031253f,
   -0.254865660f,  0.966976471f,
   -0.266712757f,  0.963776066f,
   -0.278519689f,  0.960430519f,
   -0.290284677f,  0.956940336f,
   -0.302005949f,  0.953306040f,
   -0.313681740f,  0.949528181f,
   -0.325310292f,  0.945607325f,
   -0.336889853f,  0.941544065f,
   -0.348418680f,  0.937339012f,
   -0.359895037f,  0.932992799f,
   -0.371317194f,  0.928506080f,
   -0.382683432f,  0.923879533f,
   -0.393992040f,  0.919113852f,
   -0.405241314f,  0.914209756f,
   -0.416429560f,  0.909167983f,
   -0.427555093f,  0.903989293f,
   -0.438616239f,  0.898674466f,
   -0.449611330f,  0.893224301f,
   -0.460538711f,  0.887639620f,
   -0.471396737f,  0.881921264f,
   -0.482183772f,  0.876070094f,
   -0.492898192f,  0.870086991f,
   -0.503538384f,  0.863972856f,
   -0.514102744f,  0.857728610f,
   -0.524589683f,  0.851355193f,
   -0.534997620f,  0.844853565f,
   -0.545324988f,  0.838224706f,
   -0.555570233f,  0.831469612f,
   -0.565731811f,  0.824589303f,
   -0.575808191f,  0.817584813f,
   -0.585797857f,  0.810457198f,
   -0.595699304f,  0.803207531f,
   -0.605511041f,  0.795836905f,
   -0.615231591f,  0.788346428f,
   -0.624859488f,  0.780737229f,
   -0.634393284f,  0.773010453f,
   -0.643831543f,  0.765167266f,
   -0.653172843f,  0.757208847f,
   -0.662415778f,  0.749136395f,
   -0.671558955f,  0.740951125f,
   -0.680600998f,  0.732654272f,
   -0.689540545f,  0.724247083f,
   -0.698376249f,  0.715730825f,
   -0.707106781f,  0.707106781f,
   -0.715730825f,  0.698376249f,
   -0.724247083f,  0.689540545f,
   -0.732654272f,  0.680600998f,
   -0.740951125f,  0.671558955f,
   -0.749136395f,  0.662415778f,
   -0.757208847f,  0.653172843f,
   -0.765167266f,  0.643831543f,
   -0.773010453f,  0.634393284f,
   -0.780737229f,  0.624859488f,
   -0.788346428f,  0.615231591f,
   -0.795836905f,  0.605511041f,
   -0.803207531f,  0.595699304f,
   -0.810457198f,  0.585797857f,
   -0.817584813f,  0.575808191f,
   -0.824589303f,  0.565731811f,
   -0.831469612f,  0.555570233f,
   -0.838224706f,  0.545324988f,
   -0.844853565f,  0.534997620f,
   -0.851355193f,  0.524589683f,
   -0.857728610f,  0.514102744f,
   -0.863972856f,  0.503538384f,
   -0.870086991f,  0.492898192f,
   -0.876070094f,  0.482183772f,
   -0.881921264f,  0.471396737f,
   -0.887639620f,  0.460538711f,
   -0.893224301f,  0.449611330f,
   -0.898674466f,  0.438616239f,
   -0.903989293f,  0.427555093f,
   -0.909167983f,  0.416429560f,
   -0.914209756f,  0.405241314f,
   -0.919113852f,  0.393992040f,
   -0.923879533f,  0.382683432f,
   -0.928506080f,  0.371317194f,
   -0.932992799f,  0.359895037f,
   -0.937339012f,  0.348418680f,
   -0.941544065f,  0.336889853f,
   -0.945607325f,  0.325310292f,
   -0.949528181f,  0.313681740f,
   -0.953306040f,  0.302005949f,
   -0.956940336f,  0.290284677f,
   -0.960430519f,  0.278519689f,
   -0.963776066f,  0.266712757f,
   -0.966976471f,  0.254865660f,
   -0.970031253f,  0.242980180f,
   -0.972939952f,  0.231058108f,
   -0.975702130f,  0.219101240f,
   -0.978317371f,  0.207111376f,
   -0.980785280f,  0.195090322f,
   -0.983105487f,  0.183039888f,
   -0.985277642f,  0.170961889f,
   -0.987301418f,  0.158858143f,
   -0.989176510f,  0.146730474f,
   -0.990902635f,  0.134580709f,
   -0.992479535f,  0.122410675f,
   -0.993906970f,  0.110222207f,
   -0.995184727f,  0.098017140f,
   -0.996312612f,  0.085797312f,
   -0.997290457f,  0.073564564f,
   -0.998118113f,  0.061320736f,
   -0.998795456f,  0.049067674f,
   -0.999322385f,  0.036807223f,
   -0.999698819f,  0.024541229f,
   -0.999924702f,  0.012271538f,
   -1.000000000f,  0.000000000f,
   -0.999924702f, -0.012271538f,
   -0.999698819f, -0.024541229f,
   -0.999322385f, -0.036807223f,
   -0.998795456f, -0.049067674f,
   -0.998118113f, -0.061320736f,
   -0.997290457f, -0.073564564f,
   -0.996312612f, -0.085797312f,
   -0.995184727f, -0.098017140f,
   -0.993906970f, -0.110222207f,
   -0.992479535f, -0.122410675f,
   -0.990902635f, -0.134580709f,
   -0.989176510f, -0.146730474f,
   -0.987301418f, -0.158858143f,
   -0.985277642f, -0.170961889f,
   -0.983105487f, -0.183039888f,
   -0.980785280f, -0.195090322f,
   -0.978317371f, -0.207111376f,
   -0.975702130f, -0.219101240f,
   -0.972939952f, -0.231058108f,
   -0.970031253f, -0.242980180f,
   -0.966976471f, -0.254865660f,
   -0.963776066f, -0.266712757f,
   -0.960430519f, -0.278519689f,
   -0.956940336f, -0.290284677f,
   -0.953306040f, -0.302005949f,
   -0.949528181f, -0.313681740f,
   -0.945607325f, -0.325310292f,
   -0.941544065f, -0.336889853f,
   -0.937339012f, -0.348418680f,
   -0.932992799f, -0.359895037f,
   -0.928506080f, -0.371317194f,
   -0.923879533f, -0.382683432f,
   -0.919113852f, -0.393992040f,
   -0.914209756f, -0.405241314f,
   -0.909167983f, -0.416429560f,
   -0.903989293f, -0.427555093f,
   -0.898674466f, -0.438616239f,
   -0.893224301f, -0.449611330f,
   -0.887639620f, -0.460538711f,
   -0.881921264f, -0.471396737f,
   -0.876070094f, -0.482183772f,
   -0.870086991f, -0.492898192f,
   -0.863972856f, -0.503538384f,
   -0.857728610f, -0.514102744f,
   -0.851355193f, -0.524589683f,
   -0.844853565f, -0.534997620f,
   -0.838224706f, -0.545324988f,
   -0.831469612f, -0.555570233f,
   -0.824589303f, -0.565731811f,
   -0.817584813f, -0.575808191f,
   -0.810457198f, -0.585797857f,
   -0.803207531f, -0.595699304f,
   -0.795836905f, -0.605511041f,
   -0.788346428f, -0.615231591f,
   -0.780737229f, -0.624859488f,
   -0.773010453f, -0.634393284f,
   -0.765167266f, -0.643831543f,
   -0.757208847f, -0.653172843f,
   -0.749136395f, -0.662415778f,
   -0.740951125f, -0.671558955f,
   -0.732654272f, -0.680600998f,
   -0.724247083f, -0.689540545f,
   -0.715730825f, -0.698376249f,
   -0.707106781f, -0.707106781f,
   -0.698376249f, -0.715730825f,
   -0.689540545f, -0.724247083f,
   -0.680600998f, -0.732654272f,
   -0.671558955f, -0.740951125f,
   -0.662415778f, -0.749136395f,
   -0.653172843f, -0.757208847f,
   -0.643831543f, -0.765167266f,
   -0.634393284f, -0.773010453f,
   -0.624859488f, -0.780737229f,
   -0.615231591f, -0.788346428f,
   -0.605511041f, -0.795836905f,
   -0.595699304f, -0.803207531f,
   -0.585797857f, -0.810457198f,
   -0.575808191f, -0.817584813f,
   -0.565731811f, -0.824589303f,
   -0.555570233f, -0.831469612f,
   -0.545324988f, -0.838224706f,
   -0.534997620f, -0.844853565f,
   -0.524589683f, -0.851355193f,
   -0.514102744f, -0.857728610f,
   -0.503538384f, -0.863972856f,
   -0.492898192f, -0.870086991f,
   -0.482183772f, -0.876070094f,
   -0.471396737f, -0.881921264f,
   -0.460538711f, -0.887639620f,
   -0.449611330f, -0.893224301f,
   -0.438616239f, -0.898674466f,
   -0.427555093f, -0.903989293f,
   -0.416429560f, -0.909167983f,
   -0.405241314f, -0.914209756f,
   -0.393992040f, -0.919113852f,
   -0.382683432f, -0.923879533f,
   -0.371317194f, -0.928506080f,
   -0.359895037f, -0.932992799f,
   -0.348418680f, -0.937339012f,
   -0.336889853f, -0.941544065f,
   -0.325310292f, -0.945607325f,
   -0.313681740f, -0.949528181f,
   -0.302005949f, -0.953306040f,
   -0.290284677f, -0.956940336f,
   -0.278519689f, -0.960430519f,
   -0.266712757f, -0.963776066f,
   -0.254865660f, -0.966976471f,
   -0.242980180f, -0.970031253f,
   -0.231058108f, -0.972939952f,
   -0.219101240f, -0.975702130f,
   -0.207111376f, -0.978317371f,
   -0.195090322f, -0.980785280f,
   -0.183039888f, -0.983105487f,
   -0.170961889f, -0.985277642f,
   -0.158858143f, -0.987301418f,
   -0.146730474f, -0.989176510f,
   -0.134580709f, -0.990902635f,
   -0.122410675f, -0.992479535f,
   -0.110222207f, -0.993906970f,
   -0.098017140f, -0.995184727f,
   -0.085797312f, -0.996312612f,
   -0.073564564f, -0.997290457f,
   -0.061320736f, -0.998118113f,
   -0.049067674f, -0.998795456f,
   -0.036807223f, -0.999322385f,
   -0.024541229f, -0.999698819f,
   -0.012271538f, -0.999924702f,
   -0.000000000f, -1.000000000f,
    0.012271538f, -0.999924702f,
    0.024541229f, -0.999698819f,
    0.036807223f, -0.999322385f,
    0.049067674f, -0.998795456f,
    0.061320736f, -0.998118113f,
    0.073564564f, -0.997290457f,
    0.085797312f, -0.996312612f,
    0.098017140f, -0.995184727f,
    0.110222207f, -0.993906970f,
    0.122410675f, -0.992479535f,
    0.134580709f, -0.990902635f,
    0.146730474f, -0.989176510f,
    0.158858143f, -0.987301418f,
    0.170961889f, -0.985277642f,
    0.183039888f, -0.983105487f,
    0.195090322f, -0.980785280f,
    0.207111376f, -0.978317371f,
    0.219101240f, -0.975702130f,
    0.231058108f, -0.972939952f,
    0.242980180f, -0.970031253f,
    0.254865660f, -0.966976471f,
    0.266712757f, -0.963776066f,
    0.278519689f, -0.960430519f,
    0.290284677f, -0.956940336f,
    0.302005949f, -0.953306040f,
    0.313681740f, -0.949528181f,
    0.325310292f, -0.945607325f,
    0.336889853f, -0.941544065f,
    0.348418680f, -0.937339012f,
    0.359895037f, -0.932992799f,
    0.371317194f, -0.928506080f,
    0.382683432f, -0.923879533f,
    0.393992040f, -0.919113852f,
    0.405241314f, -0.914209756f,
    0.416429560f, -0.909167983f,
    0.427555093f, -0.903989293f,
    0.438616239f, -0.898674466f,
    0.449611330f, -0.893224301f,
    0.460538711f, -0.887639620f,
    0.471396737f, -0.881921264f,
    0.482183772f, -0.876070094f,
    0.492898192f, -0.870086991f,
    0.503538384f, -0.863972856f,
    0.514102744f, -0.857728610f,
    0.524589683f, -0.851355193f,
    0.534997620f, -0.844853565f,
    0.545324988f, -0.838224706f,
    0.555570233f, -0.831469612f,
    0.565731811f, -0.824589303f,
    0.575808191f, -0.817584813f,
    0.585797857f, -0.810457198f,
    0.595699304f, -0.803207531f,
    0.605511041f, -0.795836905f,
    0.615231591f, -0.788346428f,
    0.624859488f, -0.780737229f,
    0.634393284f, -0.773010453f,
    0.643831543f, -0.765167266f,
    0.653172843f, -0.757208847f,
    0.662415778f, -0.749136395f,
    0.671558955f, -0.740951125f,
    0.680600998f, -0.732654272f,
    0.689540545f, -0.724247083f,
    0.698376249f, -0.715730825f,
    0.707106781f, -0.707106781f,
    0.715730825f, -0.698376249f,
    0.724247083f, -0.689540545f,
    0.732654272f, -0.680600998f,
    0.740951125f, -0.671558955f,
    0.749136395f, -0.662415778f,
    0.757208847f, -0.653172843f,
    0.765167266f, -0.643831543f,
    0.773010453f, -0.634393284f,
    0.780737229f, -0.624859488f,
    0.788346428f, -0.615231591f,
    0.795836905f, -0.605511041f,
    0.803207531f, -0.595699304f,
    0.810457198f, -0.585797857f,
    0.817584813f, -0.575808191f,
    0.824589303f, -0.565731811f,
    0.831469612f, -0.555570233f,
    0.838224706f, -0.545324988f,
    0.844853565f, -0.534997620f,
    0.851355193f, -0.524589683f,
    0.857728610f, -0.514102744f,
    0.863972856f, -0.503538384f,
    0.870086991f, -0.492898192f,
    0.876070094f, -0.482183772f,
    0.881921264f, -0.471396737f,
    0.887639620f, -0.460538711f,
    0.893224301f, -0.449611330f,
    0.898674466f, -0.438616239f,
    0.903989293f, -0.427555093f,
    0.909167983f, -0.416429560f,
    0.914209756f, -0.405241314f,
    0.919113852f, -0.393992040f,
    0.923879533f, -0.382683432f,
    0.928506080f, -0.371317194f,
    0.932992799f, -0.359895037f,
    0.937339012f, -0.348418680f,
    0.941544065f, -0.336889853f,
    0.945607325f, -0.325310292f,
    0.949528181f, -0.313681740f,
    0.953306040f, -0.302005949f,
    0.956940336f, -0.290284677f,
    0.960430519f, -0.278519689f,
    0.963776066f, -0.266712757f,
    0.966976471f, -0.254865660f,
    0.970031253f, -0.242980180f,
    0.972939952f, -0.231058108f,
    0.975702130f, -0.219101240f,
    0.978317371f, -0.207111376f,
    0.980785280f, -0.195090322f,
    0.983105487f, -0.183039888f,
    0.985277642f, -0.170961889f,
    0.987301418f, -0.158858143f,
    0.989176510f, -0.146730474f,
    0.990902635f, -0.134580709f,
    0.992479535f, -0.122410675f,
    0.993906970f, -0.110222207f,
    0.995184727f, -0.098017140f,
    0.996312612f, -0.085797312f,
    0.997290457f, -0.073564564f,
    0.998118113f, -0.061320736f,
    0.998795456f, -0.049067674f,
    0.999322385f, -0.036807223f,
    0.999698819f, -0.024541229f,
    0.999924702f, -0.012271538f
};
/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/
const float32_t twiddleCoef_1024[2048] = {
1.000000000f	,	0.000000000f	,
0.999981175f	,	0.006135885f	,
0.999924702f	,	0.012271538f	,
0.999830582f	,	0.018406730f	,
0.999698819f	,	0.024541229f	,
0.999529418f	,	0.030674803f	,
0.999322385f	,	0.036807223f	,
0.999077728f	,	0.042938257f	,
0.998795456f	,	0.049067674f	,
0.998475581f	,	0.055195244f	,
0.998118113f	,	0.061320736f	,
0.997723067f	,	0.067443920f	,
0.997290457f	,	0.073564564f	,
0.996820299f	,	0.079682438f	,
0.996312612f	,	0.085797312f	,
0.995767414f	,	0.091908956f	,
0.995184727f	,	0.098017140f	,
0.994564571f	,	0.104121634f	,
0.993906970f	,	0.110222207f	,
0.993211949f	,	0.116318631f	,
0.992479535f	,	0.122410675f	,
0.991709754f	,	0.128498111f	,
0.990902635f	,	0.134580709f	,
0.990058210f	,	0.140658239f	,
0.989176510f	,	0.146730474f	,
0.988257568f	,	0.152797185f	,
0.987301418f	,	0.158858143f	,
0.986308097f	,	0.164913120f	,
0.985277642f	,	0.170961889f	,
0.984210092f	,	0.177004220f	,
0.983105487f	,	0.183039888f	,
0.981963869f	,	0.189068664f	,
0.980785280f	,	0.195090322f	,
0.979569766f	,	0.201104635f	,
0.978317371f	,	0.207111376f	,
0.977028143f	,	0.213110320f	,
0.975702130f	,	0.219101240f	,
0.974339383f	,	0.225083911f	,
0.972939952f	,	0.231058108f	,
0.971503891f	,	0.237023606f	,
0.970031253f	,	0.242980180f	,
0.968522094f	,	0.248927606f	,
0.966976471f	,	0.254865660f	,
0.965394442f	,	0.260794118f	,
0.963776066f	,	0.266712757f	,
0.962121404f	,	0.272621355f	,
0.960430519f	,	0.278519689f	,
0.958703475f	,	0.284407537f	,
0.956940336f	,	0.290284677f	,
0.955141168f	,	0.296150888f	,
0.953306040f	,	0.302005949f	,
0.951435021f	,	0.307849640f	,
0.949528181f	,	0.313681740f	,
0.947585591f	,	0.319502031f	,
0.945607325f	,	0.325310292f	,
0.943593458f	,	0.331106306f	,
0.941544065f	,	0.336889853f	,
0.939459224f	,	0.342660717f	,
0.937339012f	,	0.348418680f	,
0.935183510f	,	0.354163525f	,
0.932992799f	,	0.359895037f	,
0.930766961f	,	0.365612998f	,
0.928506080f	,	0.371317194f	,
0.926210242f	,	0.377007410f	,
0.923879533f	,	0.382683432f	,
0.921514039f	,	0.388345047f	,
0.919113852f	,	0.393992040f	,
0.916679060f	,	0.399624200f	,
0.914209756f	,	0.405241314f	,
0.911706032f	,	0.410843171f	,
0.909167983f	,	0.416429560f	,
0.906595705f	,	0.422000271f	,
0.903989293f	,	0.427555093f	,
0.901348847f	,	0.433093819f	,
0.898674466f	,	0.438616239f	,
0.895966250f	,	0.444122145f	,
0.893224301f	,	0.449611330f	,
0.890448723f	,	0.455083587f	,
0.887639620f	,	0.460538711f	,
0.884797098f	,	0.465976496f	,
0.881921264f	,	0.471396737f	,
0.879012226f	,	0.476799230f	,
0.876070094f	,	0.482183772f	,
0.873094978f	,	0.487550160f	,
0.870086991f	,	0.492898192f	,
0.867046246f	,	0.498227667f	,
0.863972856f	,	0.503538384f	,
0.860866939f	,	0.508830143f	,
0.857728610f	,	0.514102744f	,
0.854557988f	,	0.519355990f	,
0.851355193f	,	0.524589683f	,
0.848120345f	,	0.529803625f	,
0.844853565f	,	0.534997620f	,
0.841554977f	,	0.540171473f	,
0.838224706f	,	0.545324988f	,
0.834862875f	,	0.550457973f	,
0.831469612f	,	0.555570233f	,
0.828045045f	,	0.560661576f	,
0.824589303f	,	0.565731811f	,
0.821102515f	,	0.570780746f	,
0.817584813f	,	0.575808191f	,
0.814036330f	,	0.580813958f	,
0.810457198f	,	0.585797857f	,
0.806847554f	,	0.590759702f	,
0.803207531f	,	0.595699304f	,
0.799537269f	,	0.600616479f	,
0.795836905f	,	0.605511041f	,
0.792106577f	,	0.610382806f	,
0.788346428f	,	0.615231591f	,
0.784556597f	,	0.620057212f	,
0.780737229f	,	0.624859488f	,
0.776888466f	,	0.629638239f	,
0.773010453f	,	0.634393284f	,
0.769103338f	,	0.639124445f	,
0.765167266f	,	0.643831543f	,
0.761202385f	,	0.648514401f	,
0.757208847f	,	0.653172843f	,
0.753186799f	,	0.657806693f	,
0.749136395f	,	0.662415778f	,
0.745057785f	,	0.666999922f	,
0.740951125f	,	0.671558955f	,
0.736816569f	,	0.676092704f	,
0.732654272f	,	0.680600998f	,
0.728464390f	,	0.685083668f	,
0.724247083f	,	0.689540545f	,
0.720002508f	,	0.693971461f	,
0.715730825f	,	0.698376249f	,
0.711432196f	,	0.702754744f	,
0.707106781f	,	0.707106781f	,
0.702754744f	,	0.711432196f	,
0.698376249f	,	0.715730825f	,
0.693971461f	,	0.720002508f	,
0.689540545f	,	0.724247083f	,
0.685083668f	,	0.728464390f	,
0.680600998f	,	0.732654272f	,
0.676092704f	,	0.736816569f	,
0.671558955f	,	0.740951125f	,
0.666999922f	,	0.745057785f	,
0.662415778f	,	0.749136395f	,
0.657806693f	,	0.753186799f	,
0.653172843f	,	0.757208847f	,
0.648514401f	,	0.761202385f	,
0.643831543f	,	0.765167266f	,
0.639124445f	,	0.769103338f	,
0.634393284f	,	0.773010453f	,
0.629638239f	,	0.776888466f	,
0.624859488f	,	0.780737229f	,
0.620057212f	,	0.784556597f	,
0.615231591f	,	0.788346428f	,
0.610382806f	,	0.792106577f	,
0.605511041f	,	0.795836905f	,
0.600616479f	,	0.799537269f	,
0.595699304f	,	0.803207531f	,
0.590759702f	,	0.806847554f	,
0.585797857f	,	0.810457198f	,
0.580813958f	,	0.814036330f	,
0.575808191f	,	0.817584813f	,
0.570780746f	,	0.821102515f	,
0.565731811f	,	0.824589303f	,
0.560661576f	,	0.828045045f	,
0.555570233f	,	0.831469612f	,
0.550457973f	,	0.834862875f	,
0.545324988f	,	0.838224706f	,
0.540171473f	,	0.841554977f	,
0.534997620f	,	0.844853565f	,
0.529803625f	,	0.848120345f	,
0.524589683f	,	0.851355193f	,
0.519355990f	,	0.854557988f	,
0.514102744f	,	0.857728610f	,
0.508830143f	,	0.860866939f	,
0.503538384f	,	0.863972856f	,
0.498227667f	,	0.867046246f	,
0.492898192f	,	0.870086991f	,
0.487550160f	,	0.873094978f	,
0.482183772f	,	0.876070094f	,
0.476799230f	,	0.879012226f	,
0.471396737f	,	0.881921264f	,
0.465976496f	,	0.884797098f	,
0.460538711f	,	0.887639620f	,
0.455083587f	,	0.890448723f	,
0.449611330f	,	0.893224301f	,
0.444122145f	,	0.895966250f	,
0.438616239f	,	0.898674466f	,
0.433093819f	,	0.901348847f	,
0.427555093f	,	0.903989293f	,
0.422000271f	,	0.906595705f	,
0.416429560f	,	0.909167983f	,
0.410843171f	,	0.911706032f	,
0.405241314f	,	0.914209756f	,
0.399624200f	,	0.916679060f	,
0.393992040f	,	0.919113852f	,
0.388345047f	,	0.921514039f	,
0.382683432f	,	0.923879533f	,
0.377007410f	,	0.926210242f	,
0.371317194f	,	0.928506080f	,
0.365612998f	,	0.930766961f	,
0.359895037f	,	0.932992799f	,
0.354163525f	,	0.935183510f	,
0.348418680f	,	0.937339012f	,
0.342660717f	,	0.939459224f	,
0.336889853f	,	0.941544065f	,
0.331106306f	,	0.943593458f	,
0.325310292f	,	0.945607325f	,
0.319502031f	,	0.947585591f	,
0.313681740f	,	0.949528181f	,
0.307849640f	,	0.951435021f	,
0.302005949f	,	0.953306040f	,
0.296150888f	,	0.955141168f	,
0.290284677f	,	0.956940336f	,
0.284407537f	,	0.958703475f	,
0.278519689f	,	0.960430519f	,
0.272621355f	,	0.962121404f	,
0.266712757f	,	0.963776066f	,
0.260794118f	,	0.965394442f	,
0.254865660f	,	0.966976471f	,
0.248927606f	,	0.968522094f	,
0.242980180f	,	0.970031253f	,
0.237023606f	,	0.971503891f	,
0.231058108f	,	0.972939952f	,
0.225083911f	,	0.974339383f	,
0.219101240f	,	0.975702130f	,
0.213110320f	,	0.977028143f	,
0.207111376f	,	0.978317371f	,
0.201104635f	,	0.979569766f	,
0.195090322f	,	0.980785280f	,
0.189068664f	,	0.981963869f	,
0.183039888f	,	0.983105487f	,
0.177004220f	,	0.984210092f	,
0.170961889f	,	0.985277642f	,
0.164913120f	,	0.986308097f	,
0.158858143f	,	0.987301418f	,
0.152797185f	,	0.988257568f	,
0.146730474f	,	0.989176510f	,
0.140658239f	,	0.990058210f	,
0.134580709f	,	0.990902635f	,
0.128498111f	,	0.991709754f	,
0.122410675f	,	0.992479535f	,
0.116318631f	,	0.993211949f	,
0.110222207f	,	0.993906970f	,
0.104121634f	,	0.994564571f	,
0.098017140f	,	0.995184727f	,
0.091908956f	,	0.995767414f	,
0.085797312f	,	0.996312612f	,
0.079682438f	,	0.996820299f	,
0.073564564f	,	0.997290457f	,
0.067443920f	,	0.997723067f	,
0.061320736f	,	0.998118113f	,
0.055195244f	,	0.998475581f	,
0.049067674f	,	0.998795456f	,
0.042938257f	,	0.999077728f	,
0.036807223f	,	0.999322385f	,
0.030674803f	,	0.999529418f	,
0.024541229f	,	0.999698819f	,
0.018406730f	,	0.999830582f	,
0.012271538f	,	0.999924702f	,
0.006135885f	,	0.999981175f	,
0.000000000f	,	1.000000000f	,
-0.006135885f	,	0.999981175f	,
-0.012271538f	,	0.999924702f	,
-0.018406730f	,	0.999830582f	,
-0.024541229f	,	0.999698819f	,
-0.030674803f	,	0.999529418f	,
-0.036807223f	,	0.999322385f	,
-0.042938257f	,	0.999077728f	,
-0.049067674f	,	0.998795456f	,
-0.055195244f	,	0.998475581f	,
-0.061320736f	,	0.998118113f	,
-0.067443920f	,	0.997723067f	,
-0.073564564f	,	0.997290457f	,
-0.079682438f	,	0.996820299f	,
-0.085797312f	,	0.996312612f	,
-0.091908956f	,	0.995767414f	,
-0.098017140f	,	0.995184727f	,
-0.104121634f	,	0.994564571f	,
-0.110222207f	,	0.993906970f	,
-0.116318631f	,	0.993211949f	,
-0.122410675f	,	0.992479535f	,
-0.128498111f	,	0.991709754f	,
-0.134580709f	,	0.990902635f	,
-0.140658239f	,	0.990058210f	,
-0.146730474f	,	0.989176510f	,
-0.152797185f	,	0.988257568f	,
-0.158858143f	,	0.987301418f	,
-0.164913120f	,	0.986308097f	,
-0.170961889f	,	0.985277642f	,
-0.177004220f	,	0.984210092f	,
-0.183039888f	,	0.983105487f	,
-0.189068664f	,	0.981963869f	,
-0.195090322f	,	0.980785280f	,
-0.201104635f	,	0.979569766f	,
-0.207111376f	,	0.978317371f	,
-0.213110320f	,	0.977028143f	,
-0.219101240f	,	0.975702130f	,
-0.225083911f	,	0.974339383f	,
-0.231058108f	,	0.972939952f	,
-0.237023606f	,	0.971503891f	,
-0.242980180f	,	0.970031253f	,
-0.248927606f	,	0.968522094f	,
-0.254865660f	,	0.966976471f	,
-0.260794118f	,	0.965394442f	,
-0.266712757f	,	0.963776066f	,
-0.272621355f	,	0.962121404f	,
-0.278519689f	,	0.960430519f	,
-0.284407537f	,	0.958703475f	,
-0.290284677f	,	0.956940336f	,
-0.296150888f	,	0.955141168f	,
-0.302005949f	,	0.953306040f	,
-0.307849640f	,	0.951435021f	,
-0.313681740f	,	0.949528181f	,
-0.319502031f	,	0.947585591f	,
-0.325310292f	,	0.945607325f	,
-0.331106306f	,	0.943593458f	,
-0.336889853f	,	0.941544065f	,
-0.342660717f	,	0.939459224f	,
-0.348418680f	,	0.937339012f	,
-0.354163525f	,	0.935183510f	,
-0.359895037f	,	0.932992799f	,
-0.365612998f	,	0.930766961f	,
-0.371317194f	,	0.928506080f	,
-0.377007410f	,	0.926210242f	,
-0.382683432f	,	0.923879533f	,
-0.388345047f	,	0.921514039f	,
-0.393992040f	,	0.919113852f	,
-0.399624200f	,	0.916679060f	,
-0.405241314f	,	0.914209756f	,
-0.410843171f	,	0.911706032f	,
-0.416429560f	,	0.909167983f	,
-0.422000271f	,	0.906595705f	,
-0.427555093f	,	0.903989293f	,
-0.433093819f	,	0.901348847f	,
-0.438616239f	,	0.898674466f	,
-0.444122145f	,	0.895966250f	,
-0.449611330f	,	0.893224301f	,
-0.455083587f	,	0.890448723f	,
-0.460538711f	,	0.887639620f	,
-0.465976496f	,	0.884797098f	,
-0.471396737f	,	0.881921264f	,
-0.476799230f	,	0.879012226f	,
-0.482183772f	,	0.876070094f	,
-0.487550160f	,	0.873094978f	,
-0.492898192f	,	0.870086991f	,
-0.498227667f	,	0.867046246f	,
-0.503538384f	,	0.863972856f	,
-0.508830143f	,	0.860866939f	,
-0.514102744f	,	0.857728610f	,
-0.519355990f	,	0.854557988f	,
-0.524589683f	,	0.851355193f	,
-0.529803625f	,	0.848120345f	,
-0.534997620f	,	0.844853565f	,
-0.540171473f	,	0.841554977f	,
-0.545324988f	,	0.838224706f	,
-0.550457973f	,	0.834862875f	,
-0.555570233f	,	0.831469612f	,
-0.560661576f	,	0.828045045f	,
-0.565731811f	,	0.824589303f	,
-0.570780746f	,	0.821102515f	,
-0.575808191f	,	0.817584813f	,
-0.580813958f	,	0.814036330f	,
-0.585797857f	,	0.810457198f	,
-0.590759702f	,	0.806847554f	,
-0.595699304f	,	0.803207531f	,
-0.600616479f	,	0.799537269f	,
-0.605511041f	,	0.795836905f	,
-0.610382806f	,	0.792106577f	,
-0.615231591f	,	0.788346428f	,
-0.620057212f	,	0.784556597f	,
-0.624859488f	,	0.780737229f	,
-0.629638239f	,	0.776888466f	,
-0.634393284f	,	0.773010453f	,
-0.639124445f	,	0.769103338f	,
-0.643831543f	,	0.765167266f	,
-0.648514401f	,	0.761202385f	,
-0.653172843f	,	0.757208847f	,
-0.657806693f	,	0.753186799f	,
-0.662415778f	,	0.749136395f	,
-0.666999922f	,	0.745057785f	,
-0.671558955f	,	0.740951125f	,
-0.676092704f	,	0.736816569f	,
-0.680600998f	,	0.732654272f	,
-0.685083668f	,	0.728464390f	,
-0.689540545f	,	0.724247083f	,
-0.693971461f	,	0.720002508f	,
-0.698376249f	,	0.715730825f	,
-0.702754744f	,	0.711432196f	,
-0.707106781f	,	0.707106781f	,
-0.711432196f	,	0.702754744f	,
-0.715730825f	,	0.698376249f	,
-0.720002508f	,	0.693971461f	,
-0.724247083f	,	0.689540545f	,
-0.728464390f	,	0.685083668f	,
-0.732654272f	,	0.680600998f	,
-0.736816569f	,	0.676092704f	,
-0.740951125f	,	0.671558955f	,
-0.745057785f	,	0.666999922f	,
-0.749136395f	,	0.662415778f	,
-0.753186799f	,	0.657806693f	,
-0.757208847f	,	0.653172843f	,
-0.761202385f	,	0.648514401f	,
-0.765167266f	,	0.643831543f	,
-0.769103338f	,	0.639124445f	,
-0.773010453f	,	0.634393284f	,
-0.776888466f	,	0.629638239f	,
-0.780737229f	,	0.624859488f	,
-0.784556597f	,	0.620057212f	,
-0.788346428f	,	0.615231591f	,
-0.792106577f	,	0.610382806f	,
-0.795836905f	,	0.605511041f	,
-0.799537269f	,	0.600616479f	,
-0.803207531f	,	0.595699304f	,
-0.806847554f	,	0.590759702f	,
-0.810457198f	,	0.585797857f	,
-0.814036330f	,	0.580813958f	,
-0.817584813f	,	0.575808191f	,
-0.821102515f	,	0.570780746f	,
-0.824589303f	,	0.565731811f	,
-0.828045045f	,	0.560661576f	,
-0.831469612f	,	0.555570233f	,
-0.834862875f	,	0.550457973f	,
-0.838224706f	,	0.545324988f	,
-0.841554977f	,	0.540171473f	,
-0.844853565f	,	0.534997620f	,
-0.848120345f	,	0.529803625f	,
-0.851355193f	,	0.524589683f	,
-0.854557988f	,	0.519355990f	,
-0.857728610f	,	0.514102744f	,
-0.860866939f	,	0.508830143f	,
-0.863972856f	,	0.503538384f	,
-0.867046246f	,	0.498227667f	,
-0.870086991f	,	0.492898192f	,
-0.873094978f	,	0.487550160f	,
-0.876070094f	,	0.482183772f	,
-0.879012226f	,	0.476799230f	,
-0.881921264f	,	0.471396737f	,
-0.884797098f	,	0.465976496f	,
-0.887639620f	,	0.460538711f	,
-0.890448723f	,	0.455083587f	,
-0.893224301f	,	0.449611330f	,
-0.895966250f	,	0.444122145f	,
-0.898674466f	,	0.438616239f	,
-0.901348847f	,	0.433093819f	,
-0.903989293f	,	0.427555093f	,
-0.906595705f	,	0.422000271f	,
-0.909167983f	,	0.416429560f	,
-0.911706032f	,	0.410843171f	,
-0.914209756f	,	0.405241314f	,
-0.916679060f	,	0.399624200f	,
-0.919113852f	,	0.393992040f	,
-0.921514039f	,	0.388345047f	,
-0.923879533f	,	0.382683432f	,
-0.926210242f	,	0.377007410f	,
-0.928506080f	,	0.371317194f	,
-0.930766961f	,	0.365612998f	,
-0.932992799f	,	0.359895037f	,
-0.935183510f	,	0.354163525f	,
-0.937339012f	,	0.348418680f	,
-0.939459224f	,	0.342660717f	,
-0.941544065f	,	0.336889853f	,
-0.943593458f	,	0.331106306f	,
-0.945607325f	,	0.325310292f	,
-0.947585591f	,	0.319502031f	,
-0.949528181f	,	0.313681740f	,
-0.951435021f	,	0.307849640f	,
-0.953306040f	,	0.302005949f	,
-0.955141168f	,	0.296150888f	,
-0.956940336f	,	0.290284677f	,
-0.958703475f	,	0.284407537f	,
-0.960430519f	,	0.278519689f	,
-0.962121404f	,	0.272621355f	,
-0.963776066f	,	0.266712757f	,
-0.965394442f	,	0.260794118f	,
-0.966976471f	,	0.254865660f	,
-0.968522094f	,	0.248927606f	,
-0.970031253f	,	0.242980180f	,
-0.971503891f	,	0.237023606f	,
-0.972939952f	,	0.231058108f	,
-0.974339383f	,	0.225083911f	,
-0.975702130f	,	0.219101240f	,
-0.977028143f	,	0.213110320f	,
-0.978317371f	,	0.207111376f	,
-0.979569766f	,	0.201104635f	,
-0.980785280f	,	0.195090322f	,
-0.981963869f	,	0.189068664f	,
-0.983105487f	,	0.183039888f	,
-0.984210092f	,	0.177004220f	,
-0.985277642f	,	0.170961889f	,
-0.986308097f	,	0.164913120f	,
-0.987301418f	,	0.158858143f	,
-0.988257568f	,	0.152797185f	,
-0.989176510f	,	0.146730474f	,
-0.990058210f	,	0.140658239f	,
-0.990902635f	,	0.134580709f	,
-0.991709754f	,	0.128498111f	,
-0.992479535f	,	0.122410675f	,
-0.993211949f	,	0.116318631f	,
-0.993906970f	,	0.110222207f	,
-0.994564571f	,	0.104121634f	,
-0.995184727f	,	0.098017140f	,
-0.995767414f	,	0.091908956f	,
-0.996312612f	,	0.085797312f	,
-0.996820299f	,	0.079682438f	,
-0.997290457f	,	0.073564564f	,
-0.997723067f	,	0.067443920f	,
-0.998118113f	,	0.061320736f	,
-0.998475581f	,	0.055195244f	,
-0.998795456f	,	0.049067674f	,
-0.999077728f	,	0.042938257f	,
-0.999322385f	,	0.036807223f	,
-0.999529418f	,	0.030674803f	,
-0.999698819f	,	0.024541229f	,
-0.999830582f	,	0.018406730f	,
-0.999924702f	,	0.012271538f	,
-0.999981175f	,	0.006135885f	,
-1.000000000f	,	0.000000000f	,
-0.999981175f	,	-0.006135885f	,
-0.999924702f	,	-0.012271538f	,
-0.999830582f	,	-0.018406730f	,
-0.999698819f	,	-0.024541229f	,
-0.999529418f	,	-0.030674803f	,
-0.999322385f	,	-0.036807223f	,
-0.999077728f	,	-0.042938257f	,
-0.998795456f	,	-0.049067674f	,
-0.998475581f	,	-0.055195244f	,
-0.998118113f	,	-0.061320736f	,
-0.997723067f	,	-0.067443920f	,
-0.997290457f	,	-0.073564564f	,
-0.996820299f	,	-0.079682438f	,
-0.996312612f	,	-0.085797312f	,
-0.995767414f	,	-0.091908956f	,
-0.995184727f	,	-0.098017140f	,
-0.994564571f	,	-0.104121634f	,
-0.993906970f	,	-0.110222207f	,
-0.993211949f	,	-0.116318631f	,
-0.992479535f	,	-0.122410675f	,
-0.991709754f	,	-0.128498111f	,
-0.990902635f	,	-0.134580709f	,
-0.990058210f	,	-0.140658239f	,
-0.989176510f	,	-0.146730474f	,
-0.988257568f	,	-0.152797185f	,
-0.987301418f	,	-0.158858143f	,
-0.986308097f	,	-0.164913120f	,
-0.985277642f	,	-0.170961889f	,
-0.984210092f	,	-0.177004220f	,
-0.983105487f	,	-0.183039888f	,
-0.981963869f	,	-0.189068664f	,
-0.980785280f	,	-0.195090322f	,
-0.979569766f	,	-0.201104635f	,
-0.978317371f	,	-0.207111376f	,
-0.977028143f	,	-0.213110320f	,
-0.975702130f	,	-0.219101240f	,
-0.974339383f	,	-0.225083911f	,
-0.972939952f	,	-0.231058108f	,
-0.971503891f	,	-0.237023606f	,
-0.970031253f	,	-0.242980180f	,
-0.968522094f	,	-0.248927606f	,
-0.966976471f	,	-0.254865660f	,
-0.965394442f	,	-0.260794118f	,
-0.963776066f	,	-0.266712757f	,
-0.962121404f	,	-0.272621355f	,
-0.960430519f	,	-0.278519689f	,
-0.958703475f	,	-0.284407537f	,
-0.956940336f	,	-0.290284677f	,
-0.955141168f	,	-0.296150888f	,
-0.953306040f	,	-0.302005949f	,
-0.951435021f	,	-0.307849640f	,
-0.949528181f	,	-0.313681740f	,
-0.947585591f	,	-0.319502031f	,
-0.945607325f	,	-0.325310292f	,
-0.943593458f	,	-0.331106306f	,
-0.941544065f	,	-0.336889853f	,
-0.939459224f	,	-0.342660717f	,
-0.937339012f	,	-0.348418680f	,
-0.935183510f	,	-0.354163525f	,
-0.932992799f	,	-0.359895037f	,
-0.930766961f	,	-0.365612998f	,
-0.928506080f	,	-0.371317194f	,
-0.926210242f	,	-0.377007410f	,
-0.923879533f	,	-0.382683432f	,
-0.921514039f	,	-0.388345047f	,
-0.919113852f	,	-0.393992040f	,
-0.916679060f	,	-0.399624200f	,
-0.914209756f	,	-0.405241314f	,
-0.911706032f	,	-0.410843171f	,
-0.909167983f	,	-0.416429560f	,
-0.906595705f	,	-0.422000271f	,
-0.903989293f	,	-0.427555093f	,
-0.901348847f	,	-0.433093819f	,
-0.898674466f	,	-0.438616239f	,
-0.895966250f	,	-0.444122145f	,
-0.893224301f	,	-0.449611330f	,
-0.890448723f	,	-0.455083587f	,
-0.887639620f	,	-0.460538711f	,
-0.884797098f	,	-0.465976496f	,
-0.881921264f	,	-0.471396737f	,
-0.879012226f	,	-0.476799230f	,
-0.876070094f	,	-0.482183772f	,
-0.873094978f	,	-0.487550160f	,
-0.870086991f	,	-0.492898192f	,
-0.867046246f	,	-0.498227667f	,
-0.863972856f	,	-0.503538384f	,
-0.860866939f	,	-0.508830143f	,
-0.857728610f	,	-0.514102744f	,
-0.854557988f	,	-0.519355990f	,
-0.851355193f	,	-0.524589683f	,
-0.848120345f	,	-0.529803625f	,
-0.844853565f	,	-0.534997620f	,
-0.841554977f	,	-0.540171473f	,
-0.838224706f	,	-0.545324988f	,
-0.834862875f	,	-0.550457973f	,
-0.831469612f	,	-0.555570233f	,
-0.828045045f	,	-0.560661576f	,
-0.824589303f	,	-0.565731811f	,
-0.821102515f	,	-0.570780746f	,
-0.817584813f	,	-0.575808191f	,
-0.814036330f	,	-0.580813958f	,
-0.810457198f	,	-0.585797857f	,
-0.806847554f	,	-0.590759702f	,
-0.803207531f	,	-0.595699304f	,
-0.799537269f	,	-0.600616479f	,
-0.795836905f	,	-0.605511041f	,
-0.792106577f	,	-0.610382806f	,
-0.788346428f	,	-0.615231591f	,
-0.784556597f	,	-0.620057212f	,
-0.780737229f	,	-0.624859488f	,
-0.776888466f	,	-0.629638239f	,
-0.773010453f	,	-0.634393284f	,
-0.769103338f	,	-0.639124445f	,
-0.765167266f	,	-0.643831543f	,
-0.761202385f	,	-0.648514401f	,
-0.757208847f	,	-0.653172843f	,
-0.753186799f	,	-0.657806693f	,
-0.749136395f	,	-0.662415778f	,
-0.745057785f	,	-0.666999922f	,
-0.740951125f	,	-0.671558955f	,
-0.736816569f	,	-0.676092704f	,
-0.732654272f	,	-0.680600998f	,
-0.728464390f	,	-0.685083668f	,
-0.724247083f	,	-0.689540545f	,
-0.720002508f	,	-0.693971461f	,
-0.715730825f	,	-0.698376249f	,
-0.711432196f	,	-0.702754744f	,
-0.707106781f	,	-0.707106781f	,
-0.702754744f	,	-0.711432196f	,
-0.698376249f	,	-0.715730825f	,
-0.693971461f	,	-0.720002508f	,
-0.689540545f	,	-0.724247083f	,
-0.685083668f	,	-0.728464390f	,
-0.680600998f	,	-0.732654272f	,
-0.676092704f	,	-0.736816569f	,
-0.671558955f	,	-0.740951125f	,
-0.666999922f	,	-0.745057785f	,
-0.662415778f	,	-0.749136395f	,
-0.657806693f	,	-0.753186799f	,
-0.653172843f	,	-0.757208847f	,
-0.648514401f	,	-0.761202385f	,
-0.643831543f	,	-0.765167266f	,
-0.639124445f	,	-0.769103338f	,
-0.634393284f	,	-0.773010453f	,
-0.629638239f	,	-0.776888466f	,
-0.624859488f	,	-0.780737229f	,
-0.620057212f	,	-0.784556597f	,
-0.615231591f	,	-0.788346428f	,
-0.610382806f	,	-0.792106577f	,
-0.605511041f	,	-0.795836905f	,
-0.600616479f	,	-0.799537269f	,
-0.595699304f	,	-0.803207531f	,
-0.590759702f	,	-0.806847554f	,
-0.585797857f	,	-0.810457198f	,
-0.580813958f	,	-0.814036330f	,
-0.575808191f	,	-0.817584813f	,
-0.570780746f	,	-0.821102515f	,
-0.565731811f	,	-0.824589303f	,
-0.560661576f	,	-0.828045045f	,
-0.555570233f	,	-0.831469612f	,
-0.550457973f	,	-0.834862875f	,
-0.545324988f	,	-0.838224706f	,
-0.540171473f	,	-0.841554977f	,
-0.534997620f	,	-0.844853565f	,
-0.529803625f	,	-0.848120345f	,
-0.524589683f	,	-0.851355193f	,
-0.519355990f	,	-0.854557988f	,
-0.514102744f	,	-0.857728610f	,
-0.508830143f	,	-0.860866939f	,
-0.503538384f	,	-0.863972856f	,
-0.498227667f	,	-0.867046246f	,
-0.492898192f	,	-0.870086991f	,
-0.487550160f	,	-0.873094978f	,
-0.482183772f	,	-0.876070094f	,
-0.476799230f	,	-0.879012226f	,
-0.471396737f	,	-0.881921264f	,
-0.465976496f	,	-0.884797098f	,
-0.460538711f	,	-0.887639620f	,
-0.455083587f	,	-0.890448723f	,
-0.449611330f	,	-0.893224301f	,
-0.444122145f	,	-0.895966250f	,
-0.438616239f	,	-0.898674466f	,
-0.433093819f	,	-0.901348847f	,
-0.427555093f	,	-0.903989293f	,
-0.422000271f	,	-0.906595705f	,
-0.416429560f	,	-0.909167983f	,
-0.410843171f	,	-0.911706032f	,
-0.405241314f	,	-0.914209756f	,
-0.399624200f	,	-0.916679060f	,
-0.393992040f	,	-0.919113852f	,
-0.388345047f	,	-0.921514039f	,
-0.382683432f	,	-0.923879533f	,
-0.377007410f	,	-0.926210242f	,
-0.371317194f	,	-0.928506080f	,
-0.365612998f	,	-0.930766961f	,
-0.359895037f	,	-0.932992799f	,
-0.354163525f	,	-0.935183510f	,
-0.348418680f	,	-0.937339012f	,
-0.342660717f	,	-0.939459224f	,
-0.336889853f	,	-0.941544065f	,
-0.331106306f	,	-0.943593458f	,
-0.325310292f	,	-0.945607325f	,
-0.319502031f	,	-0.947585591f	,
-0.313681740f	,	-0.949528181f	,
-0.307849640f	,	-0.951435021f	,
-0.302005949f	,	-0.953306040f	,
-0.296150888f	,	-0.955141168f	,
-0.290284677f	,	-0.956940336f	,
-0.284407537f	,	-0.958703475f	,
-0.278519689f	,	-0.960430519f	,
-0.272621355f	,	-0.962121404f	,
-0.266712757f	,	-0.963776066f	,
-0.260794118f	,	-0.965394442f	,
-0.254865660f	,	-0.966976471f	,
-0.248927606f	,	-0.968522094f	,
-0.242980180f	,	-0.970031253f	,
-0.237023606f	,	-0.971503891f	,
-0.231058108f	,	-0.972939952f	,
-0.225083911f	,	-0.974339383f	,
-0.219101240f	,	-0.975702130f	,
-0.213110320f	,	-0.977028143f	,
-0.207111376f	,	-0.978317371f	,
-0.201104635f	,	-0.979569766f	,
-0.195090322f	,	-0.980785280f	,
-0.189068664f	,	-0.981963869f	,
-0.183039888f	,	-0.983105487f	,
-0.177004220f	,	-0.984210092f	,
-0.170961889f	,	-0.985277642f	,
-0.164913120f	,	-0.986308097f	,
-0.158858143f	,	-0.987301418f	,
-0.152797185f	,	-0.988257568f	,
-0.146730474f	,	-0.989176510f	,
-0.140658239f	,	-0.990058210f	,
-0.134580709f	,	-0.990902635f	,
-0.128498111f	,	-0.991709754f	,
-0.122410675f	,	-0.992479535f	,
-0.116318631f	,	-0.993211949f	,
-0.110222207f	,	-0.993906970f	,
-0.104121634f	,	-0.994564571f	,
-0.098017140f	,	-0.995184727f	,
-0.091908956f	,	-0.995767414f	,
-0.085797312f	,	-0.996312612f	,
-0.079682438f	,	-0.996820299f	,
-0.073564564f	,	-0.997290457f	,
-0.067443920f	,	-0.997723067f	,
-0.061320736f	,	-0.998118113f	,
-0.055195244f	,	-0.998475581f	,
-0.049067674f	,	-0.998795456f	,
-0.042938257f	,	-0.999077728f	,
-0.036807223f	,	-0.999322385f	,
-0.030674803f	,	-0.999529418f	,
-0.024541229f	,	-0.999698819f	,
-0.018406730f	,	-0.999830582f	,
-0.012271538f	,	-0.999924702f	,
-0.006135885f	,	-0.999981175f	,
-0.000000000f	,	-1.000000000f	,
0.006135885f	,	-0.999981175f	,
0.012271538f	,	-0.999924702f	,
0.018406730f	,	-0.999830582f	,
0.024541229f	,	-0.999698819f	,
0.030674803f	,	-0.999529418f	,
0.036807223f	,	-0.999322385f	,
0.042938257f	,	-0.999077728f	,
0.049067674f	,	-0.998795456f	,
0.055195244f	,	-0.998475581f	,
0.061320736f	,	-0.998118113f	,
0.067443920f	,	-0.997723067f	,
0.073564564f	,	-0.997290457f	,
0.079682438f	,	-0.996820299f	,
0.085797312f	,	-0.996312612f	,
0.091908956f	,	-0.995767414f	,
0.098017140f	,	-0.995184727f	,
0.104121634f	,	-0.994564571f	,
0.110222207f	,	-0.993906970f	,
0.116318631f	,	-0.993211949f	,
0.122410675f	,	-0.992479535f	,
0.128498111f	,	-0.991709754f	,
0.134580709f	,	-0.990902635f	,
0.140658239f	,	-0.990058210f	,
0.146730474f	,	-0.989176510f	,
0.152797185f	,	-0.988257568f	,
0.158858143f	,	-0.987301418f	,
0.164913120f	,	-0.986308097f	,
0.170961889f	,	-0.985277642f	,
0.177004220f	,	-0.984210092f	,
0.183039888f	,	-0.983105487f	,
0.189068664f	,	-0.981963869f	,
0.195090322f	,	-0.980785280f	,
0.201104635f	,	-0.979569766f	,
0.207111376f	,	-0.978317371f	,
0.213110320f	,	-0.977028143f	,
0.219101240f	,	-0.975702130f	,
0.225083911f	,	-0.974339383f	,
0.231058108f	,	-0.972939952f	,
0.237023606f	,	-0.971503891f	,
0.242980180f	,	-0.970031253f	,
0.248927606f	,	-0.968522094f	,
0.254865660f	,	-0.966976471f	,
0.260794118f	,	-0.965394442f	,
0.266712757f	,	-0.963776066f	,
0.272621355f	,	-0.962121404f	,
0.278519689f	,	-0.960430519f	,
0.284407537f	,	-0.958703475f	,
0.290284677f	,	-0.956940336f	,
0.296150888f	,	-0.955141168f	,
0.302005949f	,	-0.953306040f	,
0.307849640f	,	-0.951435021f	,
0.313681740f	,	-0.949528181f	,
0.319502031f	,	-0.947585591f	,
0.325310292f	,	-0.945607325f	,
0.331106306f	,	-0.943593458f	,
0.336889853f	,	-0.941544065f	,
0.342660717f	,	-0.939459224f	,
0.348418680f	,	-0.937339012f	,
0.354163525f	,	-0.935183510f	,
0.359895037f	,	-0.932992799f	,
0.365612998f	,	-0.930766961f	,
0.371317194f	,	-0.928506080f	,
0.377007410f	,	-0.926210242f	,
0.382683432f	,	-0.923879533f	,
0.388345047f	,	-0.921514039f	,
0.393992040f	,	-0.919113852f	,
0.399624200f	,	-0.916679060f	,
0.405241314f	,	-0.914209756f	,
0.410843171f	,	-0.911706032f	,
0.416429560f	,	-0.909167983f	,
0.422000271f	,	-0.906595705f	,
0.427555093f	,	-0.903989293f	,
0.433093819f	,	-0.901348847f	,
0.438616239f	,	-0.898674466f	,
0.444122145f	,	-0.895966250f	,
0.449611330f	,	-0.893224301f	,
0.455083587f	,	-0.890448723f	,
0.460538711f	,	-0.887639620f	,
0.465976496f	,	-0.884797098f	,
0.471396737f	,	-0.881921264f	,
0.476799230f	,	-0.879012226f	,
0.482183772f	,	-0.876070094f	,
0.487550160f	,	-0.873094978f	,
0.492898192f	,	-0.870086991f	,
0.498227667f	,	-0.867046246f	,
0.503538384f	,	-0.863972856f	,
0.508830143f	,	-0.860866939f	,
0.514102744f	,	-0.857728610f	,
0.519355990f	,	-0.854557988f	,
0.524589683f	,	-0.851355193f	,
0.529803625f	,	-0.848120345f	,
0.534997620f	,	-0.844853565f	,
0.540171473f	,	-0.841554977f	,
0.545324988f	,	-0.838224706f	,
0.550457973f	,	-0.834862875f	,
0.555570233f	,	-0.831469612f	,
0.560661576f	,	-0.828045045f	,
0.565731811f	,	-0.824589303f	,
0.570780746f	,	-0.821102515f	,
0.575808191f	,	-0.817584813f	,
0.580813958f	,	-0.814036330f	,
0.585797857f	,	-0.810457198f	,
0.590759702f	,	-0.806847554f	,
0.595699304f	,	-0.803207531f	,
0.600616479f	,	-0.799537269f	,
0.605511041f	,	-0.795836905f	,
0.610382806f	,	-0.792106577f	,
0.615231591f	,	-0.788346428f	,
0.620057212f	,	-0.784556597f	,
0.624859488f	,	-0.780737229f	,
0.629638239f	,	-0.776888466f	,
0.634393284f	,	-0.773010453f	,
0.639124445f	,	-0.769103338f	,
0.643831543f	,	-0.765167266f	,
0.648514401f	,	-0.761202385f	,
0.653172843f	,	-0.757208847f	,
0.657806693f	,	-0.753186799f	,
0.662415778f	,	-0.749136395f	,
0.666999922f	,	-0.745057785f	,
0.671558955f	,	-0.740951125f	,
0.676092704f	,	-0.736816569f	,
0.680600998f	,	-0.732654272f	,
0.685083668f	,	-0.728464390f	,
0.689540545f	,	-0.724247083f	,
0.693971461f	,	-0.720002508f	,
0.698376249f	,	-0.715730825f	,
0.702754744f	,	-0.711432196f	,
0.707106781f	,	-0.707106781f	,
0.711432196f	,	-0.702754744f	,
0.715730825f	,	-0.698376249f	,
0.720002508f	,	-0.693971461f	,
0.724247083f	,	-0.689540545f	,
0.728464390f	,	-0.685083668f	,
0.732654272f	,	-0.680600998f	,
0.736816569f	,	-0.676092704f	,
0.740951125f	,	-0.671558955f	,
0.745057785f	,	-0.666999922f	,
0.749136395f	,	-0.662415778f	,
0.753186799f	,	-0.657806693f	,
0.757208847f	,	-0.653172843f	,
0.761202385f	,	-0.648514401f	,
0.765167266f	,	-0.643831543f	,
0.769103338f	,	-0.639124445f	,
0.773010453f	,	-0.634393284f	,
0.776888466f	,	-0.629638239f	,
0.780737229f	,	-0.624859488f	,
0.784556597f	,	-0.620057212f	,
0.788346428f	,	-0.615231591f	,
0.792106577f	,	-0.610382806f	,
0.795836905f	,	-0.605511041f	,
0.799537269f	,	-0.600616479f	,
0.803207531f	,	-0.595699304f	,
0.806847554f	,	-0.590759702f	,
0.810457198f	,	-0.585797857f	,
0.814036330f	,	-0.580813958f	,
0.817584813f	,	-0.575808191f	,
0.821102515f	,	-0.570780746f	,
0.824589303f	,	-0.565731811f	,
0.828045045f	,	-0.560661576f	,
0.831469612f	,	-0.555570233f	,
0.834862875f	,	-0.550457973f	,
0.838224706f	,	-0.545324988f	,
0.841554977f	,	-0.540171473f	,
0.844853565f	,	-0.534997620f	,
0.848120345f	,	-0.529803625f	,
0.851355193f	,	-0.524589683f	,
0.854557988f	,	-0.519355990f	,
0.857728610f	,	-0.514102744f	,
0.860866939f	,	-0.508830143f	,
0.863972856f	,	-0.503538384f	,
0.867046246f	,	-0.498227667f	,
0.870086991f	,	-0.492898192f	,
0.873094978f	,	-0.487550160f	,
0.876070094f	,	-0.482183772f	,
0.879012226f	,	-0.476799230f	,
0.881921264f	,	-0.471396737f	,
0.884797098f	,	-0.465976496f	,
0.887639620f	,	-0.460538711f	,
0.890448723f	,	-0.455083587f	,
0.893224301f	,	-0.449611330f	,
0.895966250f	,	-0.444122145f	,
0.898674466f	,	-0.438616239f	,
0.901348847f	,	-0.433093819f	,
0.903989293f	,	-0.427555093f	,
0.906595705f	,	-0.422000271f	,
0.909167983f	,	-0.416429560f	,
0.911706032f	,	-0.410843171f	,
0.914209756f	,	-0.405241314f	,
0.916679060f	,	-0.399624200f	,
0.919113852f	,	-0.393992040f	,
0.921514039f	,	-0.388345047f	,
0.923879533f	,	-0.382683432f	,
0.926210242f	,	-0.377007410f	,
0.928506080f	,	-0.371317194f	,
0.930766961f	,	-0.365612998f	,
0.932992799f	,	-0.359895037f	,
0.935183510f	,	-0.354163525f	,
0.937339012f	,	-0.348418680f	,
0.939459224f	,	-0.342660717f	,
0.941544065f	,	-0.336889853f	,
0.943593458f	,	-0.331106306f	,
0.945607325f	,	-0.325310292f	,
0.947585591f	,	-0.319502031f	,
0.949528181f	,	-0.313681740f	,
0.951435021f	,	-0.307849640f	,
0.953306040f	,	-0.302005949f	,
0.955141168f	,	-0.296150888f	,
0.956940336f	,	-0.290284677f	,
0.958703475f	,	-0.284407537f	,
0.960430519f	,	-0.278519689f	,
0.962121404f	,	-0.272621355f	,
0.963776066f	,	-0.266712757f	,
0.965394442f	,	-0.260794118f	,
0.966976471f	,	-0.254865660f	,
0.968522094f	,	-0.248927606f	,
0.970031253f	,	-0.242980180f	,
0.971503891f	,	-0.237023606f	,
0.972939952f	,	-0.231058108f	,
0.974339383f	,	-0.225083911f	,
0.975702130f	,	-0.219101240f	,
0.977028143f	,	-0.213110320f	,
0.978317371f	,	-0.207111376f	,
0.979569766f	,	-0.201104635f	,
0.980785280f	,	-0.195090322f	,
0.981963869f	,	-0.189068664f	,
0.983105487f	,	-0.183039888f	,
0.984210092f	,	-0.177004220f	,
0.985277642f	,	-0.170961889f	,
0.986308097f	,	-0.164913120f	,
0.987301418f	,	-0.158858143f	,
0.988257568f	,	-0.152797185f	,
0.989176510f	,	-0.146730474f	,
0.990058210f	,	-0.140658239f	,
0.990902635f	,	-0.134580709f	,
0.991709754f	,	-0.128498111f	,
0.992479535f	,	-0.122410675f	,
0.993211949f	,	-0.116318631f	,
0.993906970f	,	-0.110222207f	,
0.994564571f	,	-0.104121634f	,
0.995184727f	,	-0.098017140f	,
0.995767414f	,	-0.091908956f	,
0.996312612f	,	-0.085797312f	,
0.996820299f	,	-0.079682438f	,
0.997290457f	,	-0.073564564f	,
0.997723067f	,	-0.067443920f	,
0.998118113f	,	-0.061320736f	,
0.998475581f	,	-0.055195244f	,
0.998795456f	,	-0.049067674f	,
0.999077728f	,	-0.042938257f	,
0.999322385f	,	-0.036807223f	,
0.999529418f	,	-0.030674803f	,
0.999698819f	,	-0.024541229f	,
0.999830582f	,	-0.018406730f	,
0.999924702f	,	-0.012271538f	,
0.999981175f	,	-0.006135885f
};

/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/
const float32_t twiddleCoef_2048[4096] = {
    1.000000000f,  0.000000000f,
    0.999995294f,  0.003067957f,
    0.999981175f,  0.006135885f,
    0.999957645f,  0.009203755f,
    0.999924702f,  0.012271538f,
    0.999882347f,  0.015339206f,
    0.999830582f,  0.018406730f,
    0.999769405f,  0.021474080f,
    0.999698819f,  0.024541229f,
    0.999618822f,  0.027608146f,
    0.999529418f,  0.030674803f,
    0.999430605f,  0.033741172f,
    0.999322385f,  0.036807223f,
    0.999204759f,  0.039872928f,
    0.999077728f,  0.042938257f,
    0.998941293f,  0.046003182f,
    0.998795456f,  0.049067674f,
    0.998640218f,  0.052131705f,
    0.998475581f,  0.055195244f,
    0.998301545f,  0.058258265f,
    0.998118113f,  0.061320736f,
    0.997925286f,  0.064382631f,
    0.997723067f,  0.067443920f,
    0.997511456f,  0.070504573f,
    0.997290457f,  0.073564564f,
    0.997060070f,  0.076623861f,
    0.996820299f,  0.079682438f,
    0.996571146f,  0.082740265f,
    0.996312612f,  0.085797312f,
    0.996044701f,  0.088853553f,
    0.995767414f,  0.091908956f,
    0.995480755f,  0.094963495f,
    0.995184727f,  0.098017140f,
    0.994879331f,  0.101069863f,
    0.994564571f,  0.104121634f,
    0.994240449f,  0.107172425f,
    0.993906970f,  0.110222207f,
    0.993564136f,  0.113270952f,
    0.993211949f,  0.116318631f,
    0.992850414f,  0.119365215f,
    0.992479535f,  0.122410675f,
    0.992099313f,  0.125454983f,
    0.991709754f,  0.128498111f,
    0.991310860f,  0.131540029f,
    0.990902635f,  0.134580709f,
    0.990485084f,  0.137620122f,
    0.990058210f,  0.140658239f,
    0.989622017f,  0.143695033f,
    0.989176510f,  0.146730474f,
    0.988721692f,  0.149764535f,
    0.988257568f,  0.152797185f,
    0.987784142f,  0.155828398f,
    0.987301418f,  0.158858143f,
    0.986809402f,  0.161886394f,
    0.986308097f,  0.164913120f,
    0.985797509f,  0.167938295f,
    0.985277642f,  0.170961889f,
    0.984748502f,  0.173983873f,
    0.984210092f,  0.177004220f,
    0.983662419f,  0.180022901f,
    0.983105487f,  0.183039888f,
    0.982539302f,  0.186055152f,
    0.981963869f,  0.189068664f,
    0.981379193f,  0.192080397f,
    0.980785280f,  0.195090322f,
    0.980182136f,  0.198098411f,
    0.979569766f,  0.201104635f,
    0.978948175f,  0.204108966f,
    0.978317371f,  0.207111376f,
    0.977677358f,  0.210111837f,
    0.977028143f,  0.213110320f,
    0.976369731f,  0.216106797f,
    0.975702130f,  0.219101240f,
    0.975025345f,  0.222093621f,
    0.974339383f,  0.225083911f,
    0.973644250f,  0.228072083f,
    0.972939952f,  0.231058108f,
    0.972226497f,  0.234041959f,
    0.971503891f,  0.237023606f,
    0.970772141f,  0.240003022f,
    0.970031253f,  0.242980180f,
    0.969281235f,  0.245955050f,
    0.968522094f,  0.248927606f,
    0.967753837f,  0.251897818f,
    0.966976471f,  0.254865660f,
    0.966190003f,  0.257831102f,
    0.965394442f,  0.260794118f,
    0.964589793f,  0.263754679f,
    0.963776066f,  0.266712757f,
    0.962953267f,  0.269668326f,
    0.962121404f,  0.272621355f,
    0.961280486f,  0.275571819f,
    0.960430519f,  0.278519689f,
    0.959571513f,  0.281464938f,
    0.958703475f,  0.284407537f,
    0.957826413f,  0.287347460f,
    0.956940336f,  0.290284677f,
    0.956045251f,  0.293219163f,
    0.955141168f,  0.296150888f,
    0.954228095f,  0.299079826f,
    0.953306040f,  0.302005949f,
    0.952375013f,  0.304929230f,
    0.951435021f,  0.307849640f,
    0.950486074f,  0.310767153f,
    0.949528181f,  0.313681740f,
    0.948561350f,  0.316593376f,
    0.947585591f,  0.319502031f,
    0.946600913f,  0.322407679f,
    0.945607325f,  0.325310292f,
    0.944604837f,  0.328209844f,
    0.943593458f,  0.331106306f,
    0.942573198f,  0.333999651f,
    0.941544065f,  0.336889853f,
    0.940506071f,  0.339776884f,
    0.939459224f,  0.342660717f,
    0.938403534f,  0.345541325f,
    0.937339012f,  0.348418680f,
    0.936265667f,  0.351292756f,
    0.935183510f,  0.354163525f,
    0.934092550f,  0.357030961f,
    0.932992799f,  0.359895037f,
    0.931884266f,  0.362755724f,
    0.930766961f,  0.365612998f,
    0.929640896f,  0.368466830f,
    0.928506080f,  0.371317194f,
    0.927362526f,  0.374164063f,
    0.926210242f,  0.377007410f,
    0.925049241f,  0.379847209f,
    0.923879533f,  0.382683432f,
    0.922701128f,  0.385516054f,
    0.921514039f,  0.388345047f,
    0.920318277f,  0.391170384f,
    0.919113852f,  0.393992040f,
    0.917900776f,  0.396809987f,
    0.916679060f,  0.399624200f,
    0.915448716f,  0.402434651f,
    0.914209756f,  0.405241314f,
    0.912962190f,  0.408044163f,
    0.911706032f,  0.410843171f,
    0.910441292f,  0.413638312f,
    0.909167983f,  0.416429560f,
    0.907886116f,  0.419216888f,
    0.906595705f,  0.422000271f,
    0.905296759f,  0.424779681f,
    0.903989293f,  0.427555093f,
    0.902673318f,  0.430326481f,
    0.901348847f,  0.433093819f,
    0.900015892f,  0.435857080f,
    0.898674466f,  0.438616239f,
    0.897324581f,  0.441371269f,
    0.895966250f,  0.444122145f,
    0.894599486f,  0.446868840f,
    0.893224301f,  0.449611330f,
    0.891840709f,  0.452349587f,
    0.890448723f,  0.455083587f,
    0.889048356f,  0.457813304f,
    0.887639620f,  0.460538711f,
    0.886222530f,  0.463259784f,
    0.884797098f,  0.465976496f,
    0.883363339f,  0.468688822f,
    0.881921264f,  0.471396737f,
    0.880470889f,  0.474100215f,
    0.879012226f,  0.476799230f,
    0.877545290f,  0.479493758f,
    0.876070094f,  0.482183772f,
    0.874586652f,  0.484869248f,
    0.873094978f,  0.487550160f,
    0.871595087f,  0.490226483f,
    0.870086991f,  0.492898192f,
    0.868570706f,  0.495565262f,
    0.867046246f,  0.498227667f,
    0.865513624f,  0.500885383f,
    0.863972856f,  0.503538384f,
    0.862423956f,  0.506186645f,
    0.860866939f,  0.508830143f,
    0.859301818f,  0.511468850f,
    0.857728610f,  0.514102744f,
    0.856147328f,  0.516731799f,
    0.854557988f,  0.519355990f,
    0.852960605f,  0.521975293f,
    0.851355193f,  0.524589683f,
    0.849741768f,  0.527199135f,
    0.848120345f,  0.529803625f,
    0.846490939f,  0.532403128f,
    0.844853565f,  0.534997620f,
    0.843208240f,  0.537587076f,
    0.841554977f,  0.540171473f,
    0.839893794f,  0.542750785f,
    0.838224706f,  0.545324988f,
    0.836547727f,  0.547894059f,
    0.834862875f,  0.550457973f,
    0.833170165f,  0.553016706f,
    0.831469612f,  0.555570233f,
    0.829761234f,  0.558118531f,
    0.828045045f,  0.560661576f,
    0.826321063f,  0.563199344f,
    0.824589303f,  0.565731811f,
    0.822849781f,  0.568258953f,
    0.821102515f,  0.570780746f,
    0.819347520f,  0.573297167f,
    0.817584813f,  0.575808191f,
    0.815814411f,  0.578313796f,
    0.814036330f,  0.580813958f,
    0.812250587f,  0.583308653f,
    0.810457198f,  0.585797857f,
    0.808656182f,  0.588281548f,
    0.806847554f,  0.590759702f,
    0.805031331f,  0.593232295f,
    0.803207531f,  0.595699304f,
    0.801376172f,  0.598160707f,
    0.799537269f,  0.600616479f,
    0.797690841f,  0.603066599f,
    0.795836905f,  0.605511041f,
    0.793975478f,  0.607949785f,
    0.792106577f,  0.610382806f,
    0.790230221f,  0.612810082f,
    0.788346428f,  0.615231591f,
    0.786455214f,  0.617647308f,
    0.784556597f,  0.620057212f,
    0.782650596f,  0.622461279f,
    0.780737229f,  0.624859488f,
    0.778816512f,  0.627251815f,
    0.776888466f,  0.629638239f,
    0.774953107f,  0.632018736f,
    0.773010453f,  0.634393284f,
    0.771060524f,  0.636761861f,
    0.769103338f,  0.639124445f,
    0.767138912f,  0.641481013f,
    0.765167266f,  0.643831543f,
    0.763188417f,  0.646176013f,
    0.761202385f,  0.648514401f,
    0.759209189f,  0.650846685f,
    0.757208847f,  0.653172843f,
    0.755201377f,  0.655492853f,
    0.753186799f,  0.657806693f,
    0.751165132f,  0.660114342f,
    0.749136395f,  0.662415778f,
    0.747100606f,  0.664710978f,
    0.745057785f,  0.666999922f,
    0.743007952f,  0.669282588f,
    0.740951125f,  0.671558955f,
    0.738887324f,  0.673829000f,
    0.736816569f,  0.676092704f,
    0.734738878f,  0.678350043f,
    0.732654272f,  0.680600998f,
    0.730562769f,  0.682845546f,
    0.728464390f,  0.685083668f,
    0.726359155f,  0.687315341f,
    0.724247083f,  0.689540545f,
    0.722128194f,  0.691759258f,
    0.720002508f,  0.693971461f,
    0.717870045f,  0.696177131f,
    0.715730825f,  0.698376249f,
    0.713584869f,  0.700568794f,
    0.711432196f,  0.702754744f,
    0.709272826f,  0.704934080f,
    0.707106781f,  0.707106781f,
    0.704934080f,  0.709272826f,
    0.702754744f,  0.711432196f,
    0.700568794f,  0.713584869f,
    0.698376249f,  0.715730825f,
    0.696177131f,  0.717870045f,
    0.693971461f,  0.720002508f,
    0.691759258f,  0.722128194f,
    0.689540545f,  0.724247083f,
    0.687315341f,  0.726359155f,
    0.685083668f,  0.728464390f,
    0.682845546f,  0.730562769f,
    0.680600998f,  0.732654272f,
    0.678350043f,  0.734738878f,
    0.676092704f,  0.736816569f,
    0.673829000f,  0.738887324f,
    0.671558955f,  0.740951125f,
    0.669282588f,  0.743007952f,
    0.666999922f,  0.745057785f,
    0.664710978f,  0.747100606f,
    0.662415778f,  0.749136395f,
    0.660114342f,  0.751165132f,
    0.657806693f,  0.753186799f,
    0.655492853f,  0.755201377f,
    0.653172843f,  0.757208847f,
    0.650846685f,  0.759209189f,
    0.648514401f,  0.761202385f,
    0.646176013f,  0.763188417f,
    0.643831543f,  0.765167266f,
    0.641481013f,  0.767138912f,
    0.639124445f,  0.769103338f,
    0.636761861f,  0.771060524f,
    0.634393284f,  0.773010453f,
    0.632018736f,  0.774953107f,
    0.629638239f,  0.776888466f,
    0.627251815f,  0.778816512f,
    0.624859488f,  0.780737229f,
    0.622461279f,  0.782650596f,
    0.620057212f,  0.784556597f,
    0.617647308f,  0.786455214f,
    0.615231591f,  0.788346428f,
    0.612810082f,  0.790230221f,
    0.610382806f,  0.792106577f,
    0.607949785f,  0.793975478f,
    0.605511041f,  0.795836905f,
    0.603066599f,  0.797690841f,
    0.600616479f,  0.799537269f,
    0.598160707f,  0.801376172f,
    0.595699304f,  0.803207531f,
    0.593232295f,  0.805031331f,
    0.590759702f,  0.806847554f,
    0.588281548f,  0.808656182f,
    0.585797857f,  0.810457198f,
    0.583308653f,  0.812250587f,
    0.580813958f,  0.814036330f,
    0.578313796f,  0.815814411f,
    0.575808191f,  0.817584813f,
    0.573297167f,  0.819347520f,
    0.570780746f,  0.821102515f,
    0.568258953f,  0.822849781f,
    0.565731811f,  0.824589303f,
    0.563199344f,  0.826321063f,
    0.560661576f,  0.828045045f,
    0.558118531f,  0.829761234f,
    0.555570233f,  0.831469612f,
    0.553016706f,  0.833170165f,
    0.550457973f,  0.834862875f,
    0.547894059f,  0.836547727f,
    0.545324988f,  0.838224706f,
    0.542750785f,  0.839893794f,
    0.540171473f,  0.841554977f,
    0.537587076f,  0.843208240f,
    0.534997620f,  0.844853565f,
    0.532403128f,  0.846490939f,
    0.529803625f,  0.848120345f,
    0.527199135f,  0.849741768f,
    0.524589683f,  0.851355193f,
    0.521975293f,  0.852960605f,
    0.519355990f,  0.854557988f,
    0.516731799f,  0.856147328f,
    0.514102744f,  0.857728610f,
    0.511468850f,  0.859301818f,
    0.508830143f,  0.860866939f,
    0.506186645f,  0.862423956f,
    0.503538384f,  0.863972856f,
    0.500885383f,  0.865513624f,
    0.498227667f,  0.867046246f,
    0.495565262f,  0.868570706f,
    0.492898192f,  0.870086991f,
    0.490226483f,  0.871595087f,
    0.487550160f,  0.873094978f,
    0.484869248f,  0.874586652f,
    0.482183772f,  0.876070094f,
    0.479493758f,  0.877545290f,
    0.476799230f,  0.879012226f,
    0.474100215f,  0.880470889f,
    0.471396737f,  0.881921264f,
    0.468688822f,  0.883363339f,
    0.465976496f,  0.884797098f,
    0.463259784f,  0.886222530f,
    0.460538711f,  0.887639620f,
    0.457813304f,  0.889048356f,
    0.455083587f,  0.890448723f,
    0.452349587f,  0.891840709f,
    0.449611330f,  0.893224301f,
    0.446868840f,  0.894599486f,
    0.444122145f,  0.895966250f,
    0.441371269f,  0.897324581f,
    0.438616239f,  0.898674466f,
    0.435857080f,  0.900015892f,
    0.433093819f,  0.901348847f,
    0.430326481f,  0.902673318f,
    0.427555093f,  0.903989293f,
    0.424779681f,  0.905296759f,
    0.422000271f,  0.906595705f,
    0.419216888f,  0.907886116f,
    0.416429560f,  0.909167983f,
    0.413638312f,  0.910441292f,
    0.410843171f,  0.911706032f,
    0.408044163f,  0.912962190f,
    0.405241314f,  0.914209756f,
    0.402434651f,  0.915448716f,
    0.399624200f,  0.916679060f,
    0.396809987f,  0.917900776f,
    0.393992040f,  0.919113852f,
    0.391170384f,  0.920318277f,
    0.388345047f,  0.921514039f,
    0.385516054f,  0.922701128f,
    0.382683432f,  0.923879533f,
    0.379847209f,  0.925049241f,
    0.377007410f,  0.926210242f,
    0.374164063f,  0.927362526f,
    0.371317194f,  0.928506080f,
    0.368466830f,  0.929640896f,
    0.365612998f,  0.930766961f,
    0.362755724f,  0.931884266f,
    0.359895037f,  0.932992799f,
    0.357030961f,  0.934092550f,
    0.354163525f,  0.935183510f,
    0.351292756f,  0.936265667f,
    0.348418680f,  0.937339012f,
    0.345541325f,  0.938403534f,
    0.342660717f,  0.939459224f,
    0.339776884f,  0.940506071f,
    0.336889853f,  0.941544065f,
    0.333999651f,  0.942573198f,
    0.331106306f,  0.943593458f,
    0.328209844f,  0.944604837f,
    0.325310292f,  0.945607325f,
    0.322407679f,  0.946600913f,
    0.319502031f,  0.947585591f,
    0.316593376f,  0.948561350f,
    0.313681740f,  0.949528181f,
    0.310767153f,  0.950486074f,
    0.307849640f,  0.951435021f,
    0.304929230f,  0.952375013f,
    0.302005949f,  0.953306040f,
    0.299079826f,  0.954228095f,
    0.296150888f,  0.955141168f,
    0.293219163f,  0.956045251f,
    0.290284677f,  0.956940336f,
    0.287347460f,  0.957826413f,
    0.284407537f,  0.958703475f,
    0.281464938f,  0.959571513f,
    0.278519689f,  0.960430519f,
    0.275571819f,  0.961280486f,
    0.272621355f,  0.962121404f,
    0.269668326f,  0.962953267f,
    0.266712757f,  0.963776066f,
    0.263754679f,  0.964589793f,
    0.260794118f,  0.965394442f,
    0.257831102f,  0.966190003f,
    0.254865660f,  0.966976471f,
    0.251897818f,  0.967753837f,
    0.248927606f,  0.968522094f,
    0.245955050f,  0.969281235f,
    0.242980180f,  0.970031253f,
    0.240003022f,  0.970772141f,
    0.237023606f,  0.971503891f,
    0.234041959f,  0.972226497f,
    0.231058108f,  0.972939952f,
    0.228072083f,  0.973644250f,
    0.225083911f,  0.974339383f,
    0.222093621f,  0.975025345f,
    0.219101240f,  0.975702130f,
    0.216106797f,  0.976369731f,
    0.213110320f,  0.977028143f,
    0.210111837f,  0.977677358f,
    0.207111376f,  0.978317371f,
    0.204108966f,  0.978948175f,
    0.201104635f,  0.979569766f,
    0.198098411f,  0.980182136f,
    0.195090322f,  0.980785280f,
    0.192080397f,  0.981379193f,
    0.189068664f,  0.981963869f,
    0.186055152f,  0.982539302f,
    0.183039888f,  0.983105487f,
    0.180022901f,  0.983662419f,
    0.177004220f,  0.984210092f,
    0.173983873f,  0.984748502f,
    0.170961889f,  0.985277642f,
    0.167938295f,  0.985797509f,
    0.164913120f,  0.986308097f,
    0.161886394f,  0.986809402f,
    0.158858143f,  0.987301418f,
    0.155828398f,  0.987784142f,
    0.152797185f,  0.988257568f,
    0.149764535f,  0.988721692f,
    0.146730474f,  0.989176510f,
    0.143695033f,  0.989622017f,
    0.140658239f,  0.990058210f,
    0.137620122f,  0.990485084f,
    0.134580709f,  0.990902635f,
    0.131540029f,  0.991310860f,
    0.128498111f,  0.991709754f,
    0.125454983f,  0.992099313f,
    0.122410675f,  0.992479535f,
    0.119365215f,  0.992850414f,
    0.116318631f,  0.993211949f,
    0.113270952f,  0.993564136f,
    0.110222207f,  0.993906970f,
    0.107172425f,  0.994240449f,
    0.104121634f,  0.994564571f,
    0.101069863f,  0.994879331f,
    0.098017140f,  0.995184727f,
    0.094963495f,  0.995480755f,
    0.091908956f,  0.995767414f,
    0.088853553f,  0.996044701f,
    0.085797312f,  0.996312612f,
    0.082740265f,  0.996571146f,
    0.079682438f,  0.996820299f,
    0.076623861f,  0.997060070f,
    0.073564564f,  0.997290457f,
    0.070504573f,  0.997511456f,
    0.067443920f,  0.997723067f,
    0.064382631f,  0.997925286f,
    0.061320736f,  0.998118113f,
    0.058258265f,  0.998301545f,
    0.055195244f,  0.998475581f,
    0.052131705f,  0.998640218f,
    0.049067674f,  0.998795456f,
    0.046003182f,  0.998941293f,
    0.042938257f,  0.999077728f,
    0.039872928f,  0.999204759f,
    0.036807223f,  0.999322385f,
    0.033741172f,  0.999430605f,
    0.030674803f,  0.999529418f,
    0.027608146f,  0.999618822f,
    0.024541229f,  0.999698819f,
    0.021474080f,  0.999769405f,
    0.018406730f,  0.999830582f,
    0.015339206f,  0.999882347f,
    0.012271538f,  0.999924702f,
    0.009203755f,  0.999957645f,
    0.006135885f,  0.999981175f,
    0.003067957f,  0.999995294f,
    0.000000000f,  1.000000000f,
   -0.003067957f,  0.999995294f,
   -0.006135885f,  0.999981175f,
   -0.009203755f,  0.999957645f,
   -0.012271538f,  0.999924702f,
   -0.015339206f,  0.999882347f,
   -0.018406730f,  0.999830582f,
   -0.021474080f,  0.999769405f,
   -0.024541229f,  0.999698819f,
   -0.027608146f,  0.999618822f,
   -0.030674803f,  0.999529418f,
   -0.033741172f,  0.999430605f,
   -0.036807223f,  0.999322385f,
   -0.039872928f,  0.999204759f,
   -0.042938257f,  0.999077728f,
   -0.046003182f,  0.998941293f,
   -0.049067674f,  0.998795456f,
   -0.052131705f,  0.998640218f,
   -0.055195244f,  0.998475581f,
   -0.058258265f,  0.998301545f,
   -0.061320736f,  0.998118113f,
   -0.064382631f,  0.997925286f,
   -0.067443920f,  0.997723067f,
   -0.070504573f,  0.997511456f,
   -0.073564564f,  0.997290457f,
   -0.076623861f,  0.997060070f,
   -0.079682438f,  0.996820299f,
   -0.082740265f,  0.996571146f,
   -0.085797312f,  0.996312612f,
   -0.088853553f,  0.996044701f,
   -0.091908956f,  0.995767414f,
   -0.094963495f,  0.995480755f,
   -0.098017140f,  0.995184727f,
   -0.101069863f,  0.994879331f,
   -0.104121634f,  0.994564571f,
   -0.107172425f,  0.994240449f,
   -0.110222207f,  0.993906970f,
   -0.113270952f,  0.993564136f,
   -0.116318631f,  0.993211949f,
   -0.119365215f,  0.992850414f,
   -0.122410675f,  0.992479535f,
   -0.125454983f,  0.992099313f,
   -0.128498111f,  0.991709754f,
   -0.131540029f,  0.991310860f,
   -0.134580709f,  0.990902635f,
   -0.137620122f,  0.990485084f,
   -0.140658239f,  0.990058210f,
   -0.143695033f,  0.989622017f,
   -0.146730474f,  0.989176510f,
   -0.149764535f,  0.988721692f,
   -0.152797185f,  0.988257568f,
   -0.155828398f,  0.987784142f,
   -0.158858143f,  0.987301418f,
   -0.161886394f,  0.986809402f,
   -0.164913120f,  0.986308097f,
   -0.167938295f,  0.985797509f,
   -0.170961889f,  0.985277642f,
   -0.173983873f,  0.984748502f,
   -0.177004220f,  0.984210092f,
   -0.180022901f,  0.983662419f,
   -0.183039888f,  0.983105487f,
   -0.186055152f,  0.982539302f,
   -0.189068664f,  0.981963869f,
   -0.192080397f,  0.981379193f,
   -0.195090322f,  0.980785280f,
   -0.198098411f,  0.980182136f,
   -0.201104635f,  0.979569766f,
   -0.204108966f,  0.978948175f,
   -0.207111376f,  0.978317371f,
   -0.210111837f,  0.977677358f,
   -0.213110320f,  0.977028143f,
   -0.216106797f,  0.976369731f,
   -0.219101240f,  0.975702130f,
   -0.222093621f,  0.975025345f,
   -0.225083911f,  0.974339383f,
   -0.228072083f,  0.973644250f,
   -0.231058108f,  0.972939952f,
   -0.234041959f,  0.972226497f,
   -0.237023606f,  0.971503891f,
   -0.240003022f,  0.970772141f,
   -0.242980180f,  0.970031253f,
   -0.245955050f,  0.969281235f,
   -0.248927606f,  0.968522094f,
   -0.251897818f,  0.967753837f,
   -0.254865660f,  0.966976471f,
   -0.257831102f,  0.966190003f,
   -0.260794118f,  0.965394442f,
   -0.263754679f,  0.964589793f,
   -0.266712757f,  0.963776066f,
   -0.269668326f,  0.962953267f,
   -0.272621355f,  0.962121404f,
   -0.275571819f,  0.961280486f,
   -0.278519689f,  0.960430519f,
   -0.281464938f,  0.959571513f,
   -0.284407537f,  0.958703475f,
   -0.287347460f,  0.957826413f,
   -0.290284677f,  0.956940336f,
   -0.293219163f,  0.956045251f,
   -0.296150888f,  0.955141168f,
   -0.299079826f,  0.954228095f,
   -0.302005949f,  0.953306040f,
   -0.304929230f,  0.952375013f,
   -0.307849640f,  0.951435021f,
   -0.310767153f,  0.950486074f,
   -0.313681740f,  0.949528181f,
   -0.316593376f,  0.948561350f,
   -0.319502031f,  0.947585591f,
   -0.322407679f,  0.946600913f,
   -0.325310292f,  0.945607325f,
   -0.328209844f,  0.944604837f,
   -0.331106306f,  0.943593458f,
   -0.333999651f,  0.942573198f,
   -0.336889853f,  0.941544065f,
   -0.339776884f,  0.940506071f,
   -0.342660717f,  0.939459224f,
   -0.345541325f,  0.938403534f,
   -0.348418680f,  0.937339012f,
   -0.351292756f,  0.936265667f,
   -0.354163525f,  0.935183510f,
   -0.357030961f,  0.934092550f,
   -0.359895037f,  0.932992799f,
   -0.362755724f,  0.931884266f,
   -0.365612998f,  0.930766961f,
   -0.368466830f,  0.929640896f,
   -0.371317194f,  0.928506080f,
   -0.374164063f,  0.927362526f,
   -0.377007410f,  0.926210242f,
   -0.379847209f,  0.925049241f,
   -0.382683432f,  0.923879533f,
   -0.385516054f,  0.922701128f,
   -0.388345047f,  0.921514039f,
   -0.391170384f,  0.920318277f,
   -0.393992040f,  0.919113852f,
   -0.396809987f,  0.917900776f,
   -0.399624200f,  0.916679060f,
   -0.402434651f,  0.915448716f,
   -0.405241314f,  0.914209756f,
   -0.408044163f,  0.912962190f,
   -0.410843171f,  0.911706032f,
   -0.413638312f,  0.910441292f,
   -0.416429560f,  0.909167983f,
   -0.419216888f,  0.907886116f,
   -0.422000271f,  0.906595705f,
   -0.424779681f,  0.905296759f,
   -0.427555093f,  0.903989293f,
   -0.430326481f,  0.902673318f,
   -0.433093819f,  0.901348847f,
   -0.435857080f,  0.900015892f,
   -0.438616239f,  0.898674466f,
   -0.441371269f,  0.897324581f,
   -0.444122145f,  0.895966250f,
   -0.446868840f,  0.894599486f,
   -0.449611330f,  0.893224301f,
   -0.452349587f,  0.891840709f,
   -0.455083587f,  0.890448723f,
   -0.457813304f,  0.889048356f,
   -0.460538711f,  0.887639620f,
   -0.463259784f,  0.886222530f,
   -0.465976496f,  0.884797098f,
   -0.468688822f,  0.883363339f,
   -0.471396737f,  0.881921264f,
   -0.474100215f,  0.880470889f,
   -0.476799230f,  0.879012226f,
   -0.479493758f,  0.877545290f,
   -0.482183772f,  0.876070094f,
   -0.484869248f,  0.874586652f,
   -0.487550160f,  0.873094978f,
   -0.490226483f,  0.871595087f,
   -0.492898192f,  0.870086991f,
   -0.495565262f,  0.868570706f,
   -0.498227667f,  0.867046246f,
   -0.500885383f,  0.865513624f,
   -0.503538384f,  0.863972856f,
   -0.506186645f,  0.862423956f,
   -0.508830143f,  0.860866939f,
   -0.511468850f,  0.859301818f,
   -0.514102744f,  0.857728610f,
   -0.516731799f,  0.856147328f,
   -0.519355990f,  0.854557988f,
   -0.521975293f,  0.852960605f,
   -0.524589683f,  0.851355193f,
   -0.527199135f,  0.849741768f,
   -0.529803625f,  0.848120345f,
   -0.532403128f,  0.846490939f,
   -0.534997620f,  0.844853565f,
   -0.537587076f,  0.843208240f,
   -0.540171473f,  0.841554977f,
   -0.542750785f,  0.839893794f,
   -0.545324988f,  0.838224706f,
   -0.547894059f,  0.836547727f,
   -0.550457973f,  0.834862875f,
   -0.553016706f,  0.833170165f,
   -0.555570233f,  0.831469612f,
   -0.558118531f,  0.829761234f,
   -0.560661576f,  0.828045045f,
   -0.563199344f,  0.826321063f,
   -0.565731811f,  0.824589303f,
   -0.568258953f,  0.822849781f,
   -0.570780746f,  0.821102515f,
   -0.573297167f,  0.819347520f,
   -0.575808191f,  0.817584813f,
   -0.578313796f,  0.815814411f,
   -0.580813958f,  0.814036330f,
   -0.583308653f,  0.812250587f,
   -0.585797857f,  0.810457198f,
   -0.588281548f,  0.808656182f,
   -0.590759702f,  0.806847554f,
   -0.593232295f,  0.805031331f,
   -0.595699304f,  0.803207531f,
   -0.598160707f,  0.801376172f,
   -0.600616479f,  0.799537269f,
   -0.603066599f,  0.797690841f,
   -0.605511041f,  0.795836905f,
   -0.607949785f,  0.793975478f,
   -0.610382806f,  0.792106577f,
   -0.612810082f,  0.790230221f,
   -0.615231591f,  0.788346428f,
   -0.617647308f,  0.786455214f,
   -0.620057212f,  0.784556597f,
   -0.622461279f,  0.782650596f,
   -0.624859488f,  0.780737229f,
   -0.627251815f,  0.778816512f,
   -0.629638239f,  0.776888466f,
   -0.632018736f,  0.774953107f,
   -0.634393284f,  0.773010453f,
   -0.636761861f,  0.771060524f,
   -0.639124445f,  0.769103338f,
   -0.641481013f,  0.767138912f,
   -0.643831543f,  0.765167266f,
   -0.646176013f,  0.763188417f,
   -0.648514401f,  0.761202385f,
   -0.650846685f,  0.759209189f,
   -0.653172843f,  0.757208847f,
   -0.655492853f,  0.755201377f,
   -0.657806693f,  0.753186799f,
   -0.660114342f,  0.751165132f,
   -0.662415778f,  0.749136395f,
   -0.664710978f,  0.747100606f,
   -0.666999922f,  0.745057785f,
   -0.669282588f,  0.743007952f,
   -0.671558955f,  0.740951125f,
   -0.673829000f,  0.738887324f,
   -0.676092704f,  0.736816569f,
   -0.678350043f,  0.734738878f,
   -0.680600998f,  0.732654272f,
   -0.682845546f,  0.730562769f,
   -0.685083668f,  0.728464390f,
   -0.687315341f,  0.726359155f,
   -0.689540545f,  0.724247083f,
   -0.691759258f,  0.722128194f,
   -0.693971461f,  0.720002508f,
   -0.696177131f,  0.717870045f,
   -0.698376249f,  0.715730825f,
   -0.700568794f,  0.713584869f,
   -0.702754744f,  0.711432196f,
   -0.704934080f,  0.709272826f,
   -0.707106781f,  0.707106781f,
   -0.709272826f,  0.704934080f,
   -0.711432196f,  0.702754744f,
   -0.713584869f,  0.700568794f,
   -0.715730825f,  0.698376249f,
   -0.717870045f,  0.696177131f,
   -0.720002508f,  0.693971461f,
   -0.722128194f,  0.691759258f,
   -0.724247083f,  0.689540545f,
   -0.726359155f,  0.687315341f,
   -0.728464390f,  0.685083668f,
   -0.730562769f,  0.682845546f,
   -0.732654272f,  0.680600998f,
   -0.734738878f,  0.678350043f,
   -0.736816569f,  0.676092704f,
   -0.738887324f,  0.673829000f,
   -0.740951125f,  0.671558955f,
   -0.743007952f,  0.669282588f,
   -0.745057785f,  0.666999922f,
   -0.747100606f,  0.664710978f,
   -0.749136395f,  0.662415778f,
   -0.751165132f,  0.660114342f,
   -0.753186799f,  0.657806693f,
   -0.755201377f,  0.655492853f,
   -0.757208847f,  0.653172843f,
   -0.759209189f,  0.650846685f,
   -0.761202385f,  0.648514401f,
   -0.763188417f,  0.646176013f,
   -0.765167266f,  0.643831543f,
   -0.767138912f,  0.641481013f,
   -0.769103338f,  0.639124445f,
   -0.771060524f,  0.636761861f,
   -0.773010453f,  0.634393284f,
   -0.774953107f,  0.632018736f,
   -0.776888466f,  0.629638239f,
   -0.778816512f,  0.627251815f,
   -0.780737229f,  0.624859488f,
   -0.782650596f,  0.622461279f,
   -0.784556597f,  0.620057212f,
   -0.786455214f,  0.617647308f,
   -0.788346428f,  0.615231591f,
   -0.790230221f,  0.612810082f,
   -0.792106577f,  0.610382806f,
   -0.793975478f,  0.607949785f,
   -0.795836905f,  0.605511041f,
   -0.797690841f,  0.603066599f,
   -0.799537269f,  0.600616479f,
   -0.801376172f,  0.598160707f,
   -0.803207531f,  0.595699304f,
   -0.805031331f,  0.593232295f,
   -0.806847554f,  0.590759702f,
   -0.808656182f,  0.588281548f,
   -0.810457198f,  0.585797857f,
   -0.812250587f,  0.583308653f,
   -0.814036330f,  0.580813958f,
   -0.815814411f,  0.578313796f,
   -0.817584813f,  0.575808191f,
   -0.819347520f,  0.573297167f,
   -0.821102515f,  0.570780746f,
   -0.822849781f,  0.568258953f,
   -0.824589303f,  0.565731811f,
   -0.826321063f,  0.563199344f,
   -0.828045045f,  0.560661576f,
   -0.829761234f,  0.558118531f,
   -0.831469612f,  0.555570233f,
   -0.833170165f,  0.553016706f,
   -0.834862875f,  0.550457973f,
   -0.836547727f,  0.547894059f,
   -0.838224706f,  0.545324988f,
   -0.839893794f,  0.542750785f,
   -0.841554977f,  0.540171473f,
   -0.843208240f,  0.537587076f,
   -0.844853565f,  0.534997620f,
   -0.846490939f,  0.532403128f,
   -0.848120345f,  0.529803625f,
   -0.849741768f,  0.527199135f,
   -0.851355193f,  0.524589683f,
   -0.852960605f,  0.521975293f,
   -0.854557988f,  0.519355990f,
   -0.856147328f,  0.516731799f,
   -0.857728610f,  0.514102744f,
   -0.859301818f,  0.511468850f,
   -0.860866939f,  0.508830143f,
   -0.862423956f,  0.506186645f,
   -0.863972856f,  0.503538384f,
   -0.865513624f,  0.500885383f,
   -0.867046246f,  0.498227667f,
   -0.868570706f,  0.495565262f,
   -0.870086991f,  0.492898192f,
   -0.871595087f,  0.490226483f,
   -0.873094978f,  0.487550160f,
   -0.874586652f,  0.484869248f,
   -0.876070094f,  0.482183772f,
   -0.877545290f,  0.479493758f,
   -0.879012226f,  0.476799230f,
   -0.880470889f,  0.474100215f,
   -0.881921264f,  0.471396737f,
   -0.883363339f,  0.468688822f,
   -0.884797098f,  0.465976496f,
   -0.886222530f,  0.463259784f,
   -0.887639620f,  0.460538711f,
   -0.889048356f,  0.457813304f,
   -0.890448723f,  0.455083587f,
   -0.891840709f,  0.452349587f,
   -0.893224301f,  0.449611330f,
   -0.894599486f,  0.446868840f,
   -0.895966250f,  0.444122145f,
   -0.897324581f,  0.441371269f,
   -0.898674466f,  0.438616239f,
   -0.900015892f,  0.435857080f,
   -0.901348847f,  0.433093819f,
   -0.902673318f,  0.430326481f,
   -0.903989293f,  0.427555093f,
   -0.905296759f,  0.424779681f,
   -0.906595705f,  0.422000271f,
   -0.907886116f,  0.419216888f,
   -0.909167983f,  0.416429560f,
   -0.910441292f,  0.413638312f,
   -0.911706032f,  0.410843171f,
   -0.912962190f,  0.408044163f,
   -0.914209756f,  0.405241314f,
   -0.915448716f,  0.402434651f,
   -0.916679060f,  0.399624200f,
   -0.917900776f,  0.396809987f,
   -0.919113852f,  0.393992040f,
   -0.920318277f,  0.391170384f,
   -0.921514039f,  0.388345047f,
   -0.922701128f,  0.385516054f,
   -0.923879533f,  0.382683432f,
   -0.925049241f,  0.379847209f,
   -0.926210242f,  0.377007410f,
   -0.927362526f,  0.374164063f,
   -0.928506080f,  0.371317194f,
   -0.929640896f,  0.368466830f,
   -0.930766961f,  0.365612998f,
   -0.931884266f,  0.362755724f,
   -0.932992799f,  0.359895037f,
   -0.934092550f,  0.357030961f,
   -0.935183510f,  0.354163525f,
   -0.936265667f,  0.351292756f,
   -0.937339012f,  0.348418680f,
   -0.938403534f,  0.345541325f,
   -0.939459224f,  0.342660717f,
   -0.940506071f,  0.339776884f,
   -0.941544065f,  0.336889853f,
   -0.942573198f,  0.333999651f,
   -0.943593458f,  0.331106306f,
   -0.944604837f,  0.328209844f,
   -0.945607325f,  0.325310292f,
   -0.946600913f,  0.322407679f,
   -0.947585591f,  0.319502031f,
   -0.948561350f,  0.316593376f,
   -0.949528181f,  0.313681740f,
   -0.950486074f,  0.310767153f,
   -0.951435021f,  0.307849640f,
   -0.952375013f,  0.304929230f,
   -0.953306040f,  0.302005949f,
   -0.954228095f,  0.299079826f,
   -0.955141168f,  0.296150888f,
   -0.956045251f,  0.293219163f,
   -0.956940336f,  0.290284677f,
   -0.957826413f,  0.287347460f,
   -0.958703475f,  0.284407537f,
   -0.959571513f,  0.281464938f,
   -0.960430519f,  0.278519689f,
   -0.961280486f,  0.275571819f,
   -0.962121404f,  0.272621355f,
   -0.962953267f,  0.269668326f,
   -0.963776066f,  0.266712757f,
   -0.964589793f,  0.263754679f,
   -0.965394442f,  0.260794118f,
   -0.966190003f,  0.257831102f,
   -0.966976471f,  0.254865660f,
   -0.967753837f,  0.251897818f,
   -0.968522094f,  0.248927606f,
   -0.969281235f,  0.245955050f,
   -0.970031253f,  0.242980180f,
   -0.970772141f,  0.240003022f,
   -0.971503891f,  0.237023606f,
   -0.972226497f,  0.234041959f,
   -0.972939952f,  0.231058108f,
   -0.973644250f,  0.228072083f,
   -0.974339383f,  0.225083911f,
   -0.975025345f,  0.222093621f,
   -0.975702130f,  0.219101240f,
   -0.976369731f,  0.216106797f,
   -0.977028143f,  0.213110320f,
   -0.977677358f,  0.210111837f,
   -0.978317371f,  0.207111376f,
   -0.978948175f,  0.204108966f,
   -0.979569766f,  0.201104635f,
   -0.980182136f,  0.198098411f,
   -0.980785280f,  0.195090322f,
   -0.981379193f,  0.192080397f,
   -0.981963869f,  0.189068664f,
   -0.982539302f,  0.186055152f,
   -0.983105487f,  0.183039888f,
   -0.983662419f,  0.180022901f,
   -0.984210092f,  0.177004220f,
   -0.984748502f,  0.173983873f,
   -0.985277642f,  0.170961889f,
   -0.985797509f,  0.167938295f,
   -0.986308097f,  0.164913120f,
   -0.986809402f,  0.161886394f,
   -0.987301418f,  0.158858143f,
   -0.987784142f,  0.155828398f,
   -0.988257568f,  0.152797185f,
   -0.988721692f,  0.149764535f,
   -0.989176510f,  0.146730474f,
   -0.989622017f,  0.143695033f,
   -0.990058210f,  0.140658239f,
   -0.990485084f,  0.137620122f,
   -0.990902635f,  0.134580709f,
   -0.991310860f,  0.131540029f,
   -0.991709754f,  0.128498111f,
   -0.992099313f,  0.125454983f,
   -0.992479535f,  0.122410675f,
   -0.992850414f,  0.119365215f,
   -0.993211949f,  0.116318631f,
   -0.993564136f,  0.113270952f,
   -0.993906970f,  0.110222207f,
   -0.994240449f,  0.107172425f,
   -0.994564571f,  0.104121634f,
   -0.994879331f,  0.101069863f,
   -0.995184727f,  0.098017140f,
   -0.995480755f,  0.094963495f,
   -0.995767414f,  0.091908956f,
   -0.996044701f,  0.088853553f,
   -0.996312612f,  0.085797312f,
   -0.996571146f,  0.082740265f,
   -0.996820299f,  0.079682438f,
   -0.997060070f,  0.076623861f,
   -0.997290457f,  0.073564564f,
   -0.997511456f,  0.070504573f,
   -0.997723067f,  0.067443920f,
   -0.997925286f,  0.064382631f,
   -0.998118113f,  0.061320736f,
   -0.998301545f,  0.058258265f,
   -0.998475581f,  0.055195244f,
   -0.998640218f,  0.052131705f,
   -0.998795456f,  0.049067674f,
   -0.998941293f,  0.046003182f,
   -0.999077728f,  0.042938257f,
   -0.999204759f,  0.039872928f,
   -0.999322385f,  0.036807223f,
   -0.999430605f,  0.033741172f,
   -0.999529418f,  0.030674803f,
   -0.999618822f,  0.027608146f,
   -0.999698819f,  0.024541229f,
   -0.999769405f,  0.021474080f,
   -0.999830582f,  0.018406730f,
   -0.999882347f,  0.015339206f,
   -0.999924702f,  0.012271538f,
   -0.999957645f,  0.009203755f,
   -0.999981175f,  0.006135885f,
   -0.999995294f,  0.003067957f,
   -1.000000000f,  0.000000000f,
   -0.999995294f, -0.003067957f,
   -0.999981175f, -0.006135885f,
   -0.999957645f, -0.009203755f,
   -0.999924702f, -0.012271538f,
   -0.999882347f, -0.015339206f,
   -0.999830582f, -0.018406730f,
   -0.999769405f, -0.021474080f,
   -0.999698819f, -0.024541229f,
   -0.999618822f, -0.027608146f,
   -0.999529418f, -0.030674803f,
   -0.999430605f, -0.033741172f,
   -0.999322385f, -0.036807223f,
   -0.999204759f, -0.039872928f,
   -0.999077728f, -0.042938257f,
   -0.998941293f, -0.046003182f,
   -0.998795456f, -0.049067674f,
   -0.998640218f, -0.052131705f,
   -0.998475581f, -0.055195244f,
   -0.998301545f, -0.058258265f,
   -0.998118113f, -0.061320736f,
   -0.997925286f, -0.064382631f,
   -0.997723067f, -0.067443920f,
   -0.997511456f, -0.070504573f,
   -0.997290457f, -0.073564564f,
   -0.997060070f, -0.076623861f,
   -0.996820299f, -0.079682438f,
   -0.996571146f, -0.082740265f,
   -0.996312612f, -0.085797312f,
   -0.996044701f, -0.088853553f,
   -0.995767414f, -0.091908956f,
   -0.995480755f, -0.094963495f,
   -0.995184727f, -0.098017140f,
   -0.994879331f, -0.101069863f,
   -0.994564571f, -0.104121634f,
   -0.994240449f, -0.107172425f,
   -0.993906970f, -0.110222207f,
   -0.993564136f, -0.113270952f,
   -0.993211949f, -0.116318631f,
   -0.992850414f, -0.119365215f,
   -0.992479535f, -0.122410675f,
   -0.992099313f, -0.125454983f,
   -0.991709754f, -0.128498111f,
   -0.991310860f, -0.131540029f,
   -0.990902635f, -0.134580709f,
   -0.990485084f, -0.137620122f,
   -0.990058210f, -0.140658239f,
   -0.989622017f, -0.143695033f,
   -0.989176510f, -0.146730474f,
   -0.988721692f, -0.149764535f,
   -0.988257568f, -0.152797185f,
   -0.987784142f, -0.155828398f,
   -0.987301418f, -0.158858143f,
   -0.986809402f, -0.161886394f,
   -0.986308097f, -0.164913120f,
   -0.985797509f, -0.167938295f,
   -0.985277642f, -0.170961889f,
   -0.984748502f, -0.173983873f,
   -0.984210092f, -0.177004220f,
   -0.983662419f, -0.180022901f,
   -0.983105487f, -0.183039888f,
   -0.982539302f, -0.186055152f,
   -0.981963869f, -0.189068664f,
   -0.981379193f, -0.192080397f,
   -0.980785280f, -0.195090322f,
   -0.980182136f, -0.198098411f,
   -0.979569766f, -0.201104635f,
   -0.978948175f, -0.204108966f,
   -0.978317371f, -0.207111376f,
   -0.977677358f, -0.210111837f,
   -0.977028143f, -0.213110320f,
   -0.976369731f, -0.216106797f,
   -0.975702130f, -0.219101240f,
   -0.975025345f, -0.222093621f,
   -0.974339383f, -0.225083911f,
   -0.973644250f, -0.228072083f,
   -0.972939952f, -0.231058108f,
   -0.972226497f, -0.234041959f,
   -0.971503891f, -0.237023606f,
   -0.970772141f, -0.240003022f,
   -0.970031253f, -0.242980180f,
   -0.969281235f, -0.245955050f,
   -0.968522094f, -0.248927606f,
   -0.967753837f, -0.251897818f,
   -0.966976471f, -0.254865660f,
   -0.966190003f, -0.257831102f,
   -0.965394442f, -0.260794118f,
   -0.964589793f, -0.263754679f,
   -0.963776066f, -0.266712757f,
   -0.962953267f, -0.269668326f,
   -0.962121404f, -0.272621355f,
   -0.961280486f, -0.275571819f,
   -0.960430519f, -0.278519689f,
   -0.959571513f, -0.281464938f,
   -0.958703475f, -0.284407537f,
   -0.957826413f, -0.287347460f,
   -0.956940336f, -0.290284677f,
   -0.956045251f, -0.293219163f,
   -0.955141168f, -0.296150888f,
   -0.954228095f, -0.299079826f,
   -0.953306040f, -0.302005949f,
   -0.952375013f, -0.304929230f,
   -0.951435021f, -0.307849640f,
   -0.950486074f, -0.310767153f,
   -0.949528181f, -0.313681740f,
   -0.948561350f, -0.316593376f,
   -0.947585591f, -0.319502031f,
   -0.946600913f, -0.322407679f,
   -0.945607325f, -0.325310292f,
   -0.944604837f, -0.328209844f,
   -0.943593458f, -0.331106306f,
   -0.942573198f, -0.333999651f,
   -0.941544065f, -0.336889853f,
   -0.940506071f, -0.339776884f,
   -0.939459224f, -0.342660717f,
   -0.938403534f, -0.345541325f,
   -0.937339012f, -0.348418680f,
   -0.936265667f, -0.351292756f,
   -0.935183510f, -0.354163525f,
   -0.934092550f, -0.357030961f,
   -0.932992799f, -0.359895037f,
   -0.931884266f, -0.362755724f,
   -0.930766961f, -0.365612998f,
   -0.929640896f, -0.368466830f,
   -0.928506080f, -0.371317194f,
   -0.927362526f, -0.374164063f,
   -0.926210242f, -0.377007410f,
   -0.925049241f, -0.379847209f,
   -0.923879533f, -0.382683432f,
   -0.922701128f, -0.385516054f,
   -0.921514039f, -0.388345047f,
   -0.920318277f, -0.391170384f,
   -0.919113852f, -0.393992040f,
   -0.917900776f, -0.396809987f,
   -0.916679060f, -0.399624200f,
   -0.915448716f, -0.402434651f,
   -0.914209756f, -0.405241314f,
   -0.912962190f, -0.408044163f,
   -0.911706032f, -0.410843171f,
   -0.910441292f, -0.413638312f,
   -0.909167983f, -0.416429560f,
   -0.907886116f, -0.419216888f,
   -0.906595705f, -0.422000271f,
   -0.905296759f, -0.424779681f,
   -0.903989293f, -0.427555093f,
   -0.902673318f, -0.430326481f,
   -0.901348847f, -0.433093819f,
   -0.900015892f, -0.435857080f,
   -0.898674466f, -0.438616239f,
   -0.897324581f, -0.441371269f,
   -0.895966250f, -0.444122145f,
   -0.894599486f, -0.446868840f,
   -0.893224301f, -0.449611330f,
   -0.891840709f, -0.452349587f,
   -0.890448723f, -0.455083587f,
   -0.889048356f, -0.457813304f,
   -0.887639620f, -0.460538711f,
   -0.886222530f, -0.463259784f,
   -0.884797098f, -0.465976496f,
   -0.883363339f, -0.468688822f,
   -0.881921264f, -0.471396737f,
   -0.880470889f, -0.474100215f,
   -0.879012226f, -0.476799230f,
   -0.877545290f, -0.479493758f,
   -0.876070094f, -0.482183772f,
   -0.874586652f, -0.484869248f,
   -0.873094978f, -0.487550160f,
   -0.871595087f, -0.490226483f,
   -0.870086991f, -0.492898192f,
   -0.868570706f, -0.495565262f,
   -0.867046246f, -0.498227667f,
   -0.865513624f, -0.500885383f,
   -0.863972856f, -0.503538384f,
   -0.862423956f, -0.506186645f,
   -0.860866939f, -0.508830143f,
   -0.859301818f, -0.511468850f,
   -0.857728610f, -0.514102744f,
   -0.856147328f, -0.516731799f,
   -0.854557988f, -0.519355990f,
   -0.852960605f, -0.521975293f,
   -0.851355193f, -0.524589683f,
   -0.849741768f, -0.527199135f,
   -0.848120345f, -0.529803625f,
   -0.846490939f, -0.532403128f,
   -0.844853565f, -0.534997620f,
   -0.843208240f, -0.537587076f,
   -0.841554977f, -0.540171473f,
   -0.839893794f, -0.542750785f,
   -0.838224706f, -0.545324988f,
   -0.836547727f, -0.547894059f,
   -0.834862875f, -0.550457973f,
   -0.833170165f, -0.553016706f,
   -0.831469612f, -0.555570233f,
   -0.829761234f, -0.558118531f,
   -0.828045045f, -0.560661576f,
   -0.826321063f, -0.563199344f,
   -0.824589303f, -0.565731811f,
   -0.822849781f, -0.568258953f,
   -0.821102515f, -0.570780746f,
   -0.819347520f, -0.573297167f,
   -0.817584813f, -0.575808191f,
   -0.815814411f, -0.578313796f,
   -0.814036330f, -0.580813958f,
   -0.812250587f, -0.583308653f,
   -0.810457198f, -0.585797857f,
   -0.808656182f, -0.588281548f,
   -0.806847554f, -0.590759702f,
   -0.805031331f, -0.593232295f,
   -0.803207531f, -0.595699304f,
   -0.801376172f, -0.598160707f,
   -0.799537269f, -0.600616479f,
   -0.797690841f, -0.603066599f,
   -0.795836905f, -0.605511041f,
   -0.793975478f, -0.607949785f,
   -0.792106577f, -0.610382806f,
   -0.790230221f, -0.612810082f,
   -0.788346428f, -0.615231591f,
   -0.786455214f, -0.617647308f,
   -0.784556597f, -0.620057212f,
   -0.782650596f, -0.622461279f,
   -0.780737229f, -0.624859488f,
   -0.778816512f, -0.627251815f,
   -0.776888466f, -0.629638239f,
   -0.774953107f, -0.632018736f,
   -0.773010453f, -0.634393284f,
   -0.771060524f, -0.636761861f,
   -0.769103338f, -0.639124445f,
   -0.767138912f, -0.641481013f,
   -0.765167266f, -0.643831543f,
   -0.763188417f, -0.646176013f,
   -0.761202385f, -0.648514401f,
   -0.759209189f, -0.650846685f,
   -0.757208847f, -0.653172843f,
   -0.755201377f, -0.655492853f,
   -0.753186799f, -0.657806693f,
   -0.751165132f, -0.660114342f,
   -0.749136395f, -0.662415778f,
   -0.747100606f, -0.664710978f,
   -0.745057785f, -0.666999922f,
   -0.743007952f, -0.669282588f,
   -0.740951125f, -0.671558955f,
   -0.738887324f, -0.673829000f,
   -0.736816569f, -0.676092704f,
   -0.734738878f, -0.678350043f,
   -0.732654272f, -0.680600998f,
   -0.730562769f, -0.682845546f,
   -0.728464390f, -0.685083668f,
   -0.726359155f, -0.687315341f,
   -0.724247083f, -0.689540545f,
   -0.722128194f, -0.691759258f,
   -0.720002508f, -0.693971461f,
   -0.717870045f, -0.696177131f,
   -0.715730825f, -0.698376249f,
   -0.713584869f, -0.700568794f,
   -0.711432196f, -0.702754744f,
   -0.709272826f, -0.704934080f,
   -0.707106781f, -0.707106781f,
   -0.704934080f, -0.709272826f,
   -0.702754744f, -0.711432196f,
   -0.700568794f, -0.713584869f,
   -0.698376249f, -0.715730825f,
   -0.696177131f, -0.717870045f,
   -0.693971461f, -0.720002508f,
   -0.691759258f, -0.722128194f,
   -0.689540545f, -0.724247083f,
   -0.687315341f, -0.726359155f,
   -0.685083668f, -0.728464390f,
   -0.682845546f, -0.730562769f,
   -0.680600998f, -0.732654272f,
   -0.678350043f, -0.734738878f,
   -0.676092704f, -0.736816569f,
   -0.673829000f, -0.738887324f,
   -0.671558955f, -0.740951125f,
   -0.669282588f, -0.743007952f,
   -0.666999922f, -0.745057785f,
   -0.664710978f, -0.747100606f,
   -0.662415778f, -0.749136395f,
   -0.660114342f, -0.751165132f,
   -0.657806693f, -0.753186799f,
   -0.655492853f, -0.755201377f,
   -0.653172843f, -0.757208847f,
   -0.650846685f, -0.759209189f,
   -0.648514401f, -0.761202385f,
   -0.646176013f, -0.763188417f,
   -0.643831543f, -0.765167266f,
   -0.641481013f, -0.767138912f,
   -0.639124445f, -0.769103338f,
   -0.636761861f, -0.771060524f,
   -0.634393284f, -0.773010453f,
   -0.632018736f, -0.774953107f,
   -0.629638239f, -0.776888466f,
   -0.627251815f, -0.778816512f,
   -0.624859488f, -0.780737229f,
   -0.622461279f, -0.782650596f,
   -0.620057212f, -0.784556597f,
   -0.617647308f, -0.786455214f,
   -0.615231591f, -0.788346428f,
   -0.612810082f, -0.790230221f,
   -0.610382806f, -0.792106577f,
   -0.607949785f, -0.793975478f,
   -0.605511041f, -0.795836905f,
   -0.603066599f, -0.797690841f,
   -0.600616479f, -0.799537269f,
   -0.598160707f, -0.801376172f,
   -0.595699304f, -0.803207531f,
   -0.593232295f, -0.805031331f,
   -0.590759702f, -0.806847554f,
   -0.588281548f, -0.808656182f,
   -0.585797857f, -0.810457198f,
   -0.583308653f, -0.812250587f,
   -0.580813958f, -0.814036330f,
   -0.578313796f, -0.815814411f,
   -0.575808191f, -0.817584813f,
   -0.573297167f, -0.819347520f,
   -0.570780746f, -0.821102515f,
   -0.568258953f, -0.822849781f,
   -0.565731811f, -0.824589303f,
   -0.563199344f, -0.826321063f,
   -0.560661576f, -0.828045045f,
   -0.558118531f, -0.829761234f,
   -0.555570233f, -0.831469612f,
   -0.553016706f, -0.833170165f,
   -0.550457973f, -0.834862875f,
   -0.547894059f, -0.836547727f,
   -0.545324988f, -0.838224706f,
   -0.542750785f, -0.839893794f,
   -0.540171473f, -0.841554977f,
   -0.537587076f, -0.843208240f,
   -0.534997620f, -0.844853565f,
   -0.532403128f, -0.846490939f,
   -0.529803625f, -0.848120345f,
   -0.527199135f, -0.849741768f,
   -0.524589683f, -0.851355193f,
   -0.521975293f, -0.852960605f,
   -0.519355990f, -0.854557988f,
   -0.516731799f, -0.856147328f,
   -0.514102744f, -0.857728610f,
   -0.511468850f, -0.859301818f,
   -0.508830143f, -0.860866939f,
   -0.506186645f, -0.862423956f,
   -0.503538384f, -0.863972856f,
   -0.500885383f, -0.865513624f,
   -0.498227667f, -0.867046246f,
   -0.495565262f, -0.868570706f,
   -0.492898192f, -0.870086991f,
   -0.490226483f, -0.871595087f,
   -0.487550160f, -0.873094978f,
   -0.484869248f, -0.874586652f,
   -0.482183772f, -0.876070094f,
   -0.479493758f, -0.877545290f,
   -0.476799230f, -0.879012226f,
   -0.474100215f, -0.880470889f,
   -0.471396737f, -0.881921264f,
   -0.468688822f, -0.883363339f,
   -0.465976496f, -0.884797098f,
   -0.463259784f, -0.886222530f,
   -0.460538711f, -0.887639620f,
   -0.457813304f, -0.889048356f,
   -0.455083587f, -0.890448723f,
   -0.452349587f, -0.891840709f,
   -0.449611330f, -0.893224301f,
   -0.446868840f, -0.894599486f,
   -0.444122145f, -0.895966250f,
   -0.441371269f, -0.897324581f,
   -0.438616239f, -0.898674466f,
   -0.435857080f, -0.900015892f,
   -0.433093819f, -0.901348847f,
   -0.430326481f, -0.902673318f,
   -0.427555093f, -0.903989293f,
   -0.424779681f, -0.905296759f,
   -0.422000271f, -0.906595705f,
   -0.419216888f, -0.907886116f,
   -0.416429560f, -0.909167983f,
   -0.413638312f, -0.910441292f,
   -0.410843171f, -0.911706032f,
   -0.408044163f, -0.912962190f,
   -0.405241314f, -0.914209756f,
   -0.402434651f, -0.915448716f,
   -0.399624200f, -0.916679060f,
   -0.396809987f, -0.917900776f,
   -0.393992040f, -0.919113852f,
   -0.391170384f, -0.920318277f,
   -0.388345047f, -0.921514039f,
   -0.385516054f, -0.922701128f,
   -0.382683432f, -0.923879533f,
   -0.379847209f, -0.925049241f,
   -0.377007410f, -0.926210242f,
   -0.374164063f, -0.927362526f,
   -0.371317194f, -0.928506080f,
   -0.368466830f, -0.929640896f,
   -0.365612998f, -0.930766961f,
   -0.362755724f, -0.931884266f,
   -0.359895037f, -0.932992799f,
   -0.357030961f, -0.934092550f,
   -0.354163525f, -0.935183510f,
   -0.351292756f, -0.936265667f,
   -0.348418680f, -0.937339012f,
   -0.345541325f, -0.938403534f,
   -0.342660717f, -0.939459224f,
   -0.339776884f, -0.940506071f,
   -0.336889853f, -0.941544065f,
   -0.333999651f, -0.942573198f,
   -0.331106306f, -0.943593458f,
   -0.328209844f, -0.944604837f,
   -0.325310292f, -0.945607325f,
   -0.322407679f, -0.946600913f,
   -0.319502031f, -0.947585591f,
   -0.316593376f, -0.948561350f,
   -0.313681740f, -0.949528181f,
   -0.310767153f, -0.950486074f,
   -0.307849640f, -0.951435021f,
   -0.304929230f, -0.952375013f,
   -0.302005949f, -0.953306040f,
   -0.299079826f, -0.954228095f,
   -0.296150888f, -0.955141168f,
   -0.293219163f, -0.956045251f,
   -0.290284677f, -0.956940336f,
   -0.287347460f, -0.957826413f,
   -0.284407537f, -0.958703475f,
   -0.281464938f, -0.959571513f,
   -0.278519689f, -0.960430519f,
   -0.275571819f, -0.961280486f,
   -0.272621355f, -0.962121404f,
   -0.269668326f, -0.962953267f,
   -0.266712757f, -0.963776066f,
   -0.263754679f, -0.964589793f,
   -0.260794118f, -0.965394442f,
   -0.257831102f, -0.966190003f,
   -0.254865660f, -0.966976471f,
   -0.251897818f, -0.967753837f,
   -0.248927606f, -0.968522094f,
   -0.245955050f, -0.969281235f,
   -0.242980180f, -0.970031253f,
   -0.240003022f, -0.970772141f,
   -0.237023606f, -0.971503891f,
   -0.234041959f, -0.972226497f,
   -0.231058108f, -0.972939952f,
   -0.228072083f, -0.973644250f,
   -0.225083911f, -0.974339383f,
   -0.222093621f, -0.975025345f,
   -0.219101240f, -0.975702130f,
   -0.216106797f, -0.976369731f,
   -0.213110320f, -0.977028143f,
   -0.210111837f, -0.977677358f,
   -0.207111376f, -0.978317371f,
   -0.204108966f, -0.978948175f,
   -0.201104635f, -0.979569766f,
   -0.198098411f, -0.980182136f,
   -0.195090322f, -0.980785280f,
   -0.192080397f, -0.981379193f,
   -0.189068664f, -0.981963869f,
   -0.186055152f, -0.982539302f,
   -0.183039888f, -0.983105487f,
   -0.180022901f, -0.983662419f,
   -0.177004220f, -0.984210092f,
   -0.173983873f, -0.984748502f,
   -0.170961889f, -0.985277642f,
   -0.167938295f, -0.985797509f,
   -0.164913120f, -0.986308097f,
   -0.161886394f, -0.986809402f,
   -0.158858143f, -0.987301418f,
   -0.155828398f, -0.987784142f,
   -0.152797185f, -0.988257568f,
   -0.149764535f, -0.988721692f,
   -0.146730474f, -0.989176510f,
   -0.143695033f, -0.989622017f,
   -0.140658239f, -0.990058210f,
   -0.137620122f, -0.990485084f,
   -0.134580709f, -0.990902635f,
   -0.131540029f, -0.991310860f,
   -0.128498111f, -0.991709754f,
   -0.125454983f, -0.992099313f,
   -0.122410675f, -0.992479535f,
   -0.119365215f, -0.992850414f,
   -0.116318631f, -0.993211949f,
   -0.113270952f, -0.993564136f,
   -0.110222207f, -0.993906970f,
   -0.107172425f, -0.994240449f,
   -0.104121634f, -0.994564571f,
   -0.101069863f, -0.994879331f,
   -0.098017140f, -0.995184727f,
   -0.094963495f, -0.995480755f,
   -0.091908956f, -0.995767414f,
   -0.088853553f, -0.996044701f,
   -0.085797312f, -0.996312612f,
   -0.082740265f, -0.996571146f,
   -0.079682438f, -0.996820299f,
   -0.076623861f, -0.997060070f,
   -0.073564564f, -0.997290457f,
   -0.070504573f, -0.997511456f,
   -0.067443920f, -0.997723067f,
   -0.064382631f, -0.997925286f,
   -0.061320736f, -0.998118113f,
   -0.058258265f, -0.998301545f,
   -0.055195244f, -0.998475581f,
   -0.052131705f, -0.998640218f,
   -0.049067674f, -0.998795456f,
   -0.046003182f, -0.998941293f,
   -0.042938257f, -0.999077728f,
   -0.039872928f, -0.999204759f,
   -0.036807223f, -0.999322385f,
   -0.033741172f, -0.999430605f,
   -0.030674803f, -0.999529418f,
   -0.027608146f, -0.999618822f,
   -0.024541229f, -0.999698819f,
   -0.021474080f, -0.999769405f,
   -0.018406730f, -0.999830582f,
   -0.015339206f, -0.999882347f,
   -0.012271538f, -0.999924702f,
   -0.009203755f, -0.999957645f,
   -0.006135885f, -0.999981175f,
   -0.003067957f, -0.999995294f,
   -0.000000000f, -1.000000000f,
    0.003067957f, -0.999995294f,
    0.006135885f, -0.999981175f,
    0.009203755f, -0.999957645f,
    0.012271538f, -0.999924702f,
    0.015339206f, -0.999882347f,
    0.018406730f, -0.999830582f,
    0.021474080f, -0.999769405f,
    0.024541229f, -0.999698819f,
    0.027608146f, -0.999618822f,
    0.030674803f, -0.999529418f,
    0.033741172f, -0.999430605f,
    0.036807223f, -0.999322385f,
    0.039872928f, -0.999204759f,
    0.042938257f, -0.999077728f,
    0.046003182f, -0.998941293f,
    0.049067674f, -0.998795456f,
    0.052131705f, -0.998640218f,
    0.055195244f, -0.998475581f,
    0.058258265f, -0.998301545f,
    0.061320736f, -0.998118113f,
    0.064382631f, -0.997925286f,
    0.067443920f, -0.997723067f,
    0.070504573f, -0.997511456f,
    0.073564564f, -0.997290457f,
    0.076623861f, -0.997060070f,
    0.079682438f, -0.996820299f,
    0.082740265f, -0.996571146f,
    0.085797312f, -0.996312612f,
    0.088853553f, -0.996044701f,
    0.091908956f, -0.995767414f,
    0.094963495f, -0.995480755f,
    0.098017140f, -0.995184727f,
    0.101069863f, -0.994879331f,
    0.104121634f, -0.994564571f,
    0.107172425f, -0.994240449f,
    0.110222207f, -0.993906970f,
    0.113270952f, -0.993564136f,
    0.116318631f, -0.993211949f,
    0.119365215f, -0.992850414f,
    0.122410675f, -0.992479535f,
    0.125454983f, -0.992099313f,
    0.128498111f, -0.991709754f,
    0.131540029f, -0.991310860f,
    0.134580709f, -0.990902635f,
    0.137620122f, -0.990485084f,
    0.140658239f, -0.990058210f,
    0.143695033f, -0.989622017f,
    0.146730474f, -0.989176510f,
    0.149764535f, -0.988721692f,
    0.152797185f, -0.988257568f,
    0.155828398f, -0.987784142f,
    0.158858143f, -0.987301418f,
    0.161886394f, -0.986809402f,
    0.164913120f, -0.986308097f,
    0.167938295f, -0.985797509f,
    0.170961889f, -0.985277642f,
    0.173983873f, -0.984748502f,
    0.177004220f, -0.984210092f,
    0.180022901f, -0.983662419f,
    0.183039888f, -0.983105487f,
    0.186055152f, -0.982539302f,
    0.189068664f, -0.981963869f,
    0.192080397f, -0.981379193f,
    0.195090322f, -0.980785280f,
    0.198098411f, -0.980182136f,
    0.201104635f, -0.979569766f,
    0.204108966f, -0.978948175f,
    0.207111376f, -0.978317371f,
    0.210111837f, -0.977677358f,
    0.213110320f, -0.977028143f,
    0.216106797f, -0.976369731f,
    0.219101240f, -0.975702130f,
    0.222093621f, -0.975025345f,
    0.225083911f, -0.974339383f,
    0.228072083f, -0.973644250f,
    0.231058108f, -0.972939952f,
    0.234041959f, -0.972226497f,
    0.237023606f, -0.971503891f,
    0.240003022f, -0.970772141f,
    0.242980180f, -0.970031253f,
    0.245955050f, -0.969281235f,
    0.248927606f, -0.968522094f,
    0.251897818f, -0.967753837f,
    0.254865660f, -0.966976471f,
    0.257831102f, -0.966190003f,
    0.260794118f, -0.965394442f,
    0.263754679f, -0.964589793f,
    0.266712757f, -0.963776066f,
    0.269668326f, -0.962953267f,
    0.272621355f, -0.962121404f,
    0.275571819f, -0.961280486f,
    0.278519689f, -0.960430519f,
    0.281464938f, -0.959571513f,
    0.284407537f, -0.958703475f,
    0.287347460f, -0.957826413f,
    0.290284677f, -0.956940336f,
    0.293219163f, -0.956045251f,
    0.296150888f, -0.955141168f,
    0.299079826f, -0.954228095f,
    0.302005949f, -0.953306040f,
    0.304929230f, -0.952375013f,
    0.307849640f, -0.951435021f,
    0.310767153f, -0.950486074f,
    0.313681740f, -0.949528181f,
    0.316593376f, -0.948561350f,
    0.319502031f, -0.947585591f,
    0.322407679f, -0.946600913f,
    0.325310292f, -0.945607325f,
    0.328209844f, -0.944604837f,
    0.331106306f, -0.943593458f,
    0.333999651f, -0.942573198f,
    0.336889853f, -0.941544065f,
    0.339776884f, -0.940506071f,
    0.342660717f, -0.939459224f,
    0.345541325f, -0.938403534f,
    0.348418680f, -0.937339012f,
    0.351292756f, -0.936265667f,
    0.354163525f, -0.935183510f,
    0.357030961f, -0.934092550f,
    0.359895037f, -0.932992799f,
    0.362755724f, -0.931884266f,
    0.365612998f, -0.930766961f,
    0.368466830f, -0.929640896f,
    0.371317194f, -0.928506080f,
    0.374164063f, -0.927362526f,
    0.377007410f, -0.926210242f,
    0.379847209f, -0.925049241f,
    0.382683432f, -0.923879533f,
    0.385516054f, -0.922701128f,
    0.388345047f, -0.921514039f,
    0.391170384f, -0.920318277f,
    0.393992040f, -0.919113852f,
    0.396809987f, -0.917900776f,
    0.399624200f, -0.916679060f,
    0.402434651f, -0.915448716f,
    0.405241314f, -0.914209756f,
    0.408044163f, -0.912962190f,
    0.410843171f, -0.911706032f,
    0.413638312f, -0.910441292f,
    0.416429560f, -0.909167983f,
    0.419216888f, -0.907886116f,
    0.422000271f, -0.906595705f,
    0.424779681f, -0.905296759f,
    0.427555093f, -0.903989293f,
    0.430326481f, -0.902673318f,
    0.433093819f, -0.901348847f,
    0.435857080f, -0.900015892f,
    0.438616239f, -0.898674466f,
    0.441371269f, -0.897324581f,
    0.444122145f, -0.895966250f,
    0.446868840f, -0.894599486f,
    0.449611330f, -0.893224301f,
    0.452349587f, -0.891840709f,
    0.455083587f, -0.890448723f,
    0.457813304f, -0.889048356f,
    0.460538711f, -0.887639620f,
    0.463259784f, -0.886222530f,
    0.465976496f, -0.884797098f,
    0.468688822f, -0.883363339f,
    0.471396737f, -0.881921264f,
    0.474100215f, -0.880470889f,
    0.476799230f, -0.879012226f,
    0.479493758f, -0.877545290f,
    0.482183772f, -0.876070094f,
    0.484869248f, -0.874586652f,
    0.487550160f, -0.873094978f,
    0.490226483f, -0.871595087f,
    0.492898192f, -0.870086991f,
    0.495565262f, -0.868570706f,
    0.498227667f, -0.867046246f,
    0.500885383f, -0.865513624f,
    0.503538384f, -0.863972856f,
    0.506186645f, -0.862423956f,
    0.508830143f, -0.860866939f,
    0.511468850f, -0.859301818f,
    0.514102744f, -0.857728610f,
    0.516731799f, -0.856147328f,
    0.519355990f, -0.854557988f,
    0.521975293f, -0.852960605f,
    0.524589683f, -0.851355193f,
    0.527199135f, -0.849741768f,
    0.529803625f, -0.848120345f,
    0.532403128f, -0.846490939f,
    0.534997620f, -0.844853565f,
    0.537587076f, -0.843208240f,
    0.540171473f, -0.841554977f,
    0.542750785f, -0.839893794f,
    0.545324988f, -0.838224706f,
    0.547894059f, -0.836547727f,
    0.550457973f, -0.834862875f,
    0.553016706f, -0.833170165f,
    0.555570233f, -0.831469612f,
    0.558118531f, -0.829761234f,
    0.560661576f, -0.828045045f,
    0.563199344f, -0.826321063f,
    0.565731811f, -0.824589303f,
    0.568258953f, -0.822849781f,
    0.570780746f, -0.821102515f,
    0.573297167f, -0.819347520f,
    0.575808191f, -0.817584813f,
    0.578313796f, -0.815814411f,
    0.580813958f, -0.814036330f,
    0.583308653f, -0.812250587f,
    0.585797857f, -0.810457198f,
    0.588281548f, -0.808656182f,
    0.590759702f, -0.806847554f,
    0.593232295f, -0.805031331f,
    0.595699304f, -0.803207531f,
    0.598160707f, -0.801376172f,
    0.600616479f, -0.799537269f,
    0.603066599f, -0.797690841f,
    0.605511041f, -0.795836905f,
    0.607949785f, -0.793975478f,
    0.610382806f, -0.792106577f,
    0.612810082f, -0.790230221f,
    0.615231591f, -0.788346428f,
    0.617647308f, -0.786455214f,
    0.620057212f, -0.784556597f,
    0.622461279f, -0.782650596f,
    0.624859488f, -0.780737229f,
    0.627251815f, -0.778816512f,
    0.629638239f, -0.776888466f,
    0.632018736f, -0.774953107f,
    0.634393284f, -0.773010453f,
    0.636761861f, -0.771060524f,
    0.639124445f, -0.769103338f,
    0.641481013f, -0.767138912f,
    0.643831543f, -0.765167266f,
    0.646176013f, -0.763188417f,
    0.648514401f, -0.761202385f,
    0.650846685f, -0.759209189f,
    0.653172843f, -0.757208847f,
    0.655492853f, -0.755201377f,
    0.657806693f, -0.753186799f,
    0.660114342f, -0.751165132f,
    0.662415778f, -0.749136395f,
    0.664710978f, -0.747100606f,
    0.666999922f, -0.745057785f,
    0.669282588f, -0.743007952f,
    0.671558955f, -0.740951125f,
    0.673829000f, -0.738887324f,
    0.676092704f, -0.736816569f,
    0.678350043f, -0.734738878f,
    0.680600998f, -0.732654272f,
    0.682845546f, -0.730562769f,
    0.685083668f, -0.728464390f,
    0.687315341f, -0.726359155f,
    0.689540545f, -0.724247083f,
    0.691759258f, -0.722128194f,
    0.693971461f, -0.720002508f,
    0.696177131f, -0.717870045f,
    0.698376249f, -0.715730825f,
    0.700568794f, -0.713584869f,
    0.702754744f, -0.711432196f,
    0.704934080f, -0.709272826f,
    0.707106781f, -0.707106781f,
    0.709272826f, -0.704934080f,
    0.711432196f, -0.702754744f,
    0.713584869f, -0.700568794f,
    0.715730825f, -0.698376249f,
    0.717870045f, -0.696177131f,
    0.720002508f, -0.693971461f,
    0.722128194f, -0.691759258f,
    0.724247083f, -0.689540545f,
    0.726359155f, -0.687315341f,
    0.728464390f, -0.685083668f,
    0.730562769f, -0.682845546f,
    0.732654272f, -0.680600998f,
    0.734738878f, -0.678350043f,
    0.736816569f, -0.676092704f,
    0.738887324f, -0.673829000f,
    0.740951125f, -0.671558955f,
    0.743007952f, -0.669282588f,
    0.745057785f, -0.666999922f,
    0.747100606f, -0.664710978f,
    0.749136395f, -0.662415778f,
    0.751165132f, -0.660114342f,
    0.753186799f, -0.657806693f,
    0.755201377f, -0.655492853f,
    0.757208847f, -0.653172843f,
    0.759209189f, -0.650846685f,
    0.761202385f, -0.648514401f,
    0.763188417f, -0.646176013f,
    0.765167266f, -0.643831543f,
    0.767138912f, -0.641481013f,
    0.769103338f, -0.639124445f,
    0.771060524f, -0.636761861f,
    0.773010453f, -0.634393284f,
    0.774953107f, -0.632018736f,
    0.776888466f, -0.629638239f,
    0.778816512f, -0.627251815f,
    0.780737229f, -0.624859488f,
    0.782650596f, -0.622461279f,
    0.784556597f, -0.620057212f,
    0.786455214f, -0.617647308f,
    0.788346428f, -0.615231591f,
    0.790230221f, -0.612810082f,
    0.792106577f, -0.610382806f,
    0.793975478f, -0.607949785f,
    0.795836905f, -0.605511041f,
    0.797690841f, -0.603066599f,
    0.799537269f, -0.600616479f,
    0.801376172f, -0.598160707f,
    0.803207531f, -0.595699304f,
    0.805031331f, -0.593232295f,
    0.806847554f, -0.590759702f,
    0.808656182f, -0.588281548f,
    0.810457198f, -0.585797857f,
    0.812250587f, -0.583308653f,
    0.814036330f, -0.580813958f,
    0.815814411f, -0.578313796f,
    0.817584813f, -0.575808191f,
    0.819347520f, -0.573297167f,
    0.821102515f, -0.570780746f,
    0.822849781f, -0.568258953f,
    0.824589303f, -0.565731811f,
    0.826321063f, -0.563199344f,
    0.828045045f, -0.560661576f,
    0.829761234f, -0.558118531f,
    0.831469612f, -0.555570233f,
    0.833170165f, -0.553016706f,
    0.834862875f, -0.550457973f,
    0.836547727f, -0.547894059f,
    0.838224706f, -0.545324988f,
    0.839893794f, -0.542750785f,
    0.841554977f, -0.540171473f,
    0.843208240f, -0.537587076f,
    0.844853565f, -0.534997620f,
    0.846490939f, -0.532403128f,
    0.848120345f, -0.529803625f,
    0.849741768f, -0.527199135f,
    0.851355193f, -0.524589683f,
    0.852960605f, -0.521975293f,
    0.854557988f, -0.519355990f,
    0.856147328f, -0.516731799f,
    0.857728610f, -0.514102744f,
    0.859301818f, -0.511468850f,
    0.860866939f, -0.508830143f,
    0.862423956f, -0.506186645f,
    0.863972856f, -0.503538384f,
    0.865513624f, -0.500885383f,
    0.867046246f, -0.498227667f,
    0.868570706f, -0.495565262f,
    0.870086991f, -0.492898192f,
    0.871595087f, -0.490226483f,
    0.873094978f, -0.487550160f,
    0.874586652f, -0.484869248f,
    0.876070094f, -0.482183772f,
    0.877545290f, -0.479493758f,
    0.879012226f, -0.476799230f,
    0.880470889f, -0.474100215f,
    0.881921264f, -0.471396737f,
    0.883363339f, -0.468688822f,
    0.884797098f, -0.465976496f,
    0.886222530f, -0.463259784f,
    0.887639620f, -0.460538711f,
    0.889048356f, -0.457813304f,
    0.890448723f, -0.455083587f,
    0.891840709f, -0.452349587f,
    0.893224301f, -0.449611330f,
    0.894599486f, -0.446868840f,
    0.895966250f, -0.444122145f,
    0.897324581f, -0.441371269f,
    0.898674466f, -0.438616239f,
    0.900015892f, -0.435857080f,
    0.901348847f, -0.433093819f,
    0.902673318f, -0.430326481f,
    0.903989293f, -0.427555093f,
    0.905296759f, -0.424779681f,
    0.906595705f, -0.422000271f,
    0.907886116f, -0.419216888f,
    0.909167983f, -0.416429560f,
    0.910441292f, -0.413638312f,
    0.911706032f, -0.410843171f,
    0.912962190f, -0.408044163f,
    0.914209756f, -0.405241314f,
    0.915448716f, -0.402434651f,
    0.916679060f, -0.399624200f,
    0.917900776f, -0.396809987f,
    0.919113852f, -0.393992040f,
    0.920318277f, -0.391170384f,
    0.921514039f, -0.388345047f,
    0.922701128f, -0.385516054f,
    0.923879533f, -0.382683432f,
    0.925049241f, -0.379847209f,
    0.926210242f, -0.377007410f,
    0.927362526f, -0.374164063f,
    0.928506080f, -0.371317194f,
    0.929640896f, -0.368466830f,
    0.930766961f, -0.365612998f,
    0.931884266f, -0.362755724f,
    0.932992799f, -0.359895037f,
    0.934092550f, -0.357030961f,
    0.935183510f, -0.354163525f,
    0.936265667f, -0.351292756f,
    0.937339012f, -0.348418680f,
    0.938403534f, -0.345541325f,
    0.939459224f, -0.342660717f,
    0.940506071f, -0.339776884f,
    0.941544065f, -0.336889853f,
    0.942573198f, -0.333999651f,
    0.943593458f, -0.331106306f,
    0.944604837f, -0.328209844f,
    0.945607325f, -0.325310292f,
    0.946600913f, -0.322407679f,
    0.947585591f, -0.319502031f,
    0.948561350f, -0.316593376f,
    0.949528181f, -0.313681740f,
    0.950486074f, -0.310767153f,
    0.951435021f, -0.307849640f,
    0.952375013f, -0.304929230f,
    0.953306040f, -0.302005949f,
    0.954228095f, -0.299079826f,
    0.955141168f, -0.296150888f,
    0.956045251f, -0.293219163f,
    0.956940336f, -0.290284677f,
    0.957826413f, -0.287347460f,
    0.958703475f, -0.284407537f,
    0.959571513f, -0.281464938f,
    0.960430519f, -0.278519689f,
    0.961280486f, -0.275571819f,
    0.962121404f, -0.272621355f,
    0.962953267f, -0.269668326f,
    0.963776066f, -0.266712757f,
    0.964589793f, -0.263754679f,
    0.965394442f, -0.260794118f,
    0.966190003f, -0.257831102f,
    0.966976471f, -0.254865660f,
    0.967753837f, -0.251897818f,
    0.968522094f, -0.248927606f,
    0.969281235f, -0.245955050f,
    0.970031253f, -0.242980180f,
    0.970772141f, -0.240003022f,
    0.971503891f, -0.237023606f,
    0.972226497f, -0.234041959f,
    0.972939952f, -0.231058108f,
    0.973644250f, -0.228072083f,
    0.974339383f, -0.225083911f,
    0.975025345f, -0.222093621f,
    0.975702130f, -0.219101240f,
    0.976369731f, -0.216106797f,
    0.977028143f, -0.213110320f,
    0.977677358f, -0.210111837f,
    0.978317371f, -0.207111376f,
    0.978948175f, -0.204108966f,
    0.979569766f, -0.201104635f,
    0.980182136f, -0.198098411f,
    0.980785280f, -0.195090322f,
    0.981379193f, -0.192080397f,
    0.981963869f, -0.189068664f,
    0.982539302f, -0.186055152f,
    0.983105487f, -0.183039888f,
    0.983662419f, -0.180022901f,
    0.984210092f, -0.177004220f,
    0.984748502f, -0.173983873f,
    0.985277642f, -0.170961889f,
    0.985797509f, -0.167938295f,
    0.986308097f, -0.164913120f,
    0.986809402f, -0.161886394f,
    0.987301418f, -0.158858143f,
    0.987784142f, -0.155828398f,
    0.988257568f, -0.152797185f,
    0.988721692f, -0.149764535f,
    0.989176510f, -0.146730474f,
    0.989622017f, -0.143695033f,
    0.990058210f, -0.140658239f,
    0.990485084f, -0.137620122f,
    0.990902635f, -0.134580709f,
    0.991310860f, -0.131540029f,
    0.991709754f, -0.128498111f,
    0.992099313f, -0.125454983f,
    0.992479535f, -0.122410675f,
    0.992850414f, -0.119365215f,
    0.993211949f, -0.116318631f,
    0.993564136f, -0.113270952f,
    0.993906970f, -0.110222207f,
    0.994240449f, -0.107172425f,
    0.994564571f, -0.104121634f,
    0.994879331f, -0.101069863f,
    0.995184727f, -0.098017140f,
    0.995480755f, -0.094963495f,
    0.995767414f, -0.091908956f,
    0.996044701f, -0.088853553f,
    0.996312612f, -0.085797312f,
    0.996571146f, -0.082740265f,
    0.996820299f, -0.079682438f,
    0.997060070f, -0.076623861f,
    0.997290457f, -0.073564564f,
    0.997511456f, -0.070504573f,
    0.997723067f, -0.067443920f,
    0.997925286f, -0.064382631f,
    0.998118113f, -0.061320736f,
    0.998301545f, -0.058258265f,
    0.998475581f, -0.055195244f,
    0.998640218f, -0.052131705f,
    0.998795456f, -0.049067674f,
    0.998941293f, -0.046003182f,
    0.999077728f, -0.042938257f,
    0.999204759f, -0.039872928f,
    0.999322385f, -0.036807223f,
    0.999430605f, -0.033741172f,
    0.999529418f, -0.030674803f,
    0.999618822f, -0.027608146f,
    0.999698819f, -0.024541229f,
    0.999769405f, -0.021474080f,
    0.999830582f, -0.018406730f,
    0.999882347f, -0.015339206f,
    0.999924702f, -0.012271538f,
    0.999957645f, -0.009203755f,
    0.999981175f, -0.006135885f,
    0.999995294f, -0.003067957f
};

/**    
* \par    
* Example code for Floating-point Twiddle factors Generation:    
* \par    
* <pre>for(i = 0; i< N/; i++)    
* {    
*	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are in interleaved fashion    
*     
*/
const float32_t twiddleCoef_4096[8192] = {
    1.000000000f,  0.000000000f,
    0.999998823f,  0.001533980f,
    0.999995294f,  0.003067957f,
    0.999989411f,  0.004601926f,
    0.999981175f,  0.006135885f,
    0.999970586f,  0.007669829f,
    0.999957645f,  0.009203755f,
    0.999942350f,  0.010737659f,
    0.999924702f,  0.012271538f,
    0.999904701f,  0.013805389f,
    0.999882347f,  0.015339206f,
    0.999857641f,  0.016872988f,
    0.999830582f,  0.018406730f,
    0.999801170f,  0.019940429f,
    0.999769405f,  0.021474080f,
    0.999735288f,  0.023007681f,
    0.999698819f,  0.024541229f,
    0.999659997f,  0.026074718f,
    0.999618822f,  0.027608146f,
    0.999575296f,  0.029141509f,
    0.999529418f,  0.030674803f,
    0.999481187f,  0.032208025f,
    0.999430605f,  0.033741172f,
    0.999377670f,  0.035274239f,
    0.999322385f,  0.036807223f,
    0.999264747f,  0.038340120f,
    0.999204759f,  0.039872928f,
    0.999142419f,  0.041405641f,
    0.999077728f,  0.042938257f,
    0.999010686f,  0.044470772f,
    0.998941293f,  0.046003182f,
    0.998869550f,  0.047535484f,
    0.998795456f,  0.049067674f,
    0.998719012f,  0.050599749f,
    0.998640218f,  0.052131705f,
    0.998559074f,  0.053663538f,
    0.998475581f,  0.055195244f,
    0.998389737f,  0.056726821f,
    0.998301545f,  0.058258265f,
    0.998211003f,  0.059789571f,
    0.998118113f,  0.061320736f,
    0.998022874f,  0.062851758f,
    0.997925286f,  0.064382631f,
    0.997825350f,  0.065913353f,
    0.997723067f,  0.067443920f,
    0.997618435f,  0.068974328f,
    0.997511456f,  0.070504573f,
    0.997402130f,  0.072034653f,
    0.997290457f,  0.073564564f,
    0.997176437f,  0.075094301f,
    0.997060070f,  0.076623861f,
    0.996941358f,  0.078153242f,
    0.996820299f,  0.079682438f,
    0.996696895f,  0.081211447f,
    0.996571146f,  0.082740265f,
    0.996443051f,  0.084268888f,
    0.996312612f,  0.085797312f,
    0.996179829f,  0.087325535f,
    0.996044701f,  0.088853553f,
    0.995907229f,  0.090381361f,
    0.995767414f,  0.091908956f,
    0.995625256f,  0.093436336f,
    0.995480755f,  0.094963495f,
    0.995333912f,  0.096490431f,
    0.995184727f,  0.098017140f,
    0.995033199f,  0.099543619f,
    0.994879331f,  0.101069863f,
    0.994723121f,  0.102595869f,
    0.994564571f,  0.104121634f,
    0.994403680f,  0.105647154f,
    0.994240449f,  0.107172425f,
    0.994074879f,  0.108697444f,
    0.993906970f,  0.110222207f,
    0.993736722f,  0.111746711f,
    0.993564136f,  0.113270952f,
    0.993389211f,  0.114794927f,
    0.993211949f,  0.116318631f,
    0.993032350f,  0.117842062f,
    0.992850414f,  0.119365215f,
    0.992666142f,  0.120888087f,
    0.992479535f,  0.122410675f,
    0.992290591f,  0.123932975f,
    0.992099313f,  0.125454983f,
    0.991905700f,  0.126976696f,
    0.991709754f,  0.128498111f,
    0.991511473f,  0.130019223f,
    0.991310860f,  0.131540029f,
    0.991107914f,  0.133060525f,
    0.990902635f,  0.134580709f,
    0.990695025f,  0.136100575f,
    0.990485084f,  0.137620122f,
    0.990272812f,  0.139139344f,
    0.990058210f,  0.140658239f,
    0.989841278f,  0.142176804f,
    0.989622017f,  0.143695033f,
    0.989400428f,  0.145212925f,
    0.989176510f,  0.146730474f,
    0.988950265f,  0.148247679f,
    0.988721692f,  0.149764535f,
    0.988490793f,  0.151281038f,
    0.988257568f,  0.152797185f,
    0.988022017f,  0.154312973f,
    0.987784142f,  0.155828398f,
    0.987543942f,  0.157343456f,
    0.987301418f,  0.158858143f,
    0.987056571f,  0.160372457f,
    0.986809402f,  0.161886394f,
    0.986559910f,  0.163399949f,
    0.986308097f,  0.164913120f,
    0.986053963f,  0.166425904f,
    0.985797509f,  0.167938295f,
    0.985538735f,  0.169450291f,
    0.985277642f,  0.170961889f,
    0.985014231f,  0.172473084f,
    0.984748502f,  0.173983873f,
    0.984480455f,  0.175494253f,
    0.984210092f,  0.177004220f,
    0.983937413f,  0.178513771f,
    0.983662419f,  0.180022901f,
    0.983385110f,  0.181531608f,
    0.983105487f,  0.183039888f,
    0.982823551f,  0.184547737f,
    0.982539302f,  0.186055152f,
    0.982252741f,  0.187562129f,
    0.981963869f,  0.189068664f,
    0.981672686f,  0.190574755f,
    0.981379193f,  0.192080397f,
    0.981083391f,  0.193585587f,
    0.980785280f,  0.195090322f,
    0.980484862f,  0.196594598f,
    0.980182136f,  0.198098411f,
    0.979877104f,  0.199601758f,
    0.979569766f,  0.201104635f,
    0.979260123f,  0.202607039f,
    0.978948175f,  0.204108966f,
    0.978633924f,  0.205610413f,
    0.978317371f,  0.207111376f,
    0.977998515f,  0.208611852f,
    0.977677358f,  0.210111837f,
    0.977353900f,  0.211611327f,
    0.977028143f,  0.213110320f,
    0.976700086f,  0.214608811f,
    0.976369731f,  0.216106797f,
    0.976037079f,  0.217604275f,
    0.975702130f,  0.219101240f,
    0.975364885f,  0.220597690f,
    0.975025345f,  0.222093621f,
    0.974683511f,  0.223589029f,
    0.974339383f,  0.225083911f,
    0.973992962f,  0.226578264f,
    0.973644250f,  0.228072083f,
    0.973293246f,  0.229565366f,
    0.972939952f,  0.231058108f,
    0.972584369f,  0.232550307f,
    0.972226497f,  0.234041959f,
    0.971866337f,  0.235533059f,
    0.971503891f,  0.237023606f,
    0.971139158f,  0.238513595f,
    0.970772141f,  0.240003022f,
    0.970402839f,  0.241491885f,
    0.970031253f,  0.242980180f,
    0.969657385f,  0.244467903f,
    0.969281235f,  0.245955050f,
    0.968902805f,  0.247441619f,
    0.968522094f,  0.248927606f,
    0.968139105f,  0.250413007f,
    0.967753837f,  0.251897818f,
    0.967366292f,  0.253382037f,
    0.966976471f,  0.254865660f,
    0.966584374f,  0.256348682f,
    0.966190003f,  0.257831102f,
    0.965793359f,  0.259312915f,
    0.965394442f,  0.260794118f,
    0.964993253f,  0.262274707f,
    0.964589793f,  0.263754679f,
    0.964184064f,  0.265234030f,
    0.963776066f,  0.266712757f,
    0.963365800f,  0.268190857f,
    0.962953267f,  0.269668326f,
    0.962538468f,  0.271145160f,
    0.962121404f,  0.272621355f,
    0.961702077f,  0.274096910f,
    0.961280486f,  0.275571819f,
    0.960856633f,  0.277046080f,
    0.960430519f,  0.278519689f,
    0.960002146f,  0.279992643f,
    0.959571513f,  0.281464938f,
    0.959138622f,  0.282936570f,
    0.958703475f,  0.284407537f,
    0.958266071f,  0.285877835f,
    0.957826413f,  0.287347460f,
    0.957384501f,  0.288816408f,
    0.956940336f,  0.290284677f,
    0.956493919f,  0.291752263f,
    0.956045251f,  0.293219163f,
    0.955594334f,  0.294685372f,
    0.955141168f,  0.296150888f,
    0.954685755f,  0.297615707f,
    0.954228095f,  0.299079826f,
    0.953768190f,  0.300543241f,
    0.953306040f,  0.302005949f,
    0.952841648f,  0.303467947f,
    0.952375013f,  0.304929230f,
    0.951906137f,  0.306389795f,
    0.951435021f,  0.307849640f,
    0.950961666f,  0.309308760f,
    0.950486074f,  0.310767153f,
    0.950008245f,  0.312224814f,
    0.949528181f,  0.313681740f,
    0.949045882f,  0.315137929f,
    0.948561350f,  0.316593376f,
    0.948074586f,  0.318048077f,
    0.947585591f,  0.319502031f,
    0.947094366f,  0.320955232f,
    0.946600913f,  0.322407679f,
    0.946105232f,  0.323859367f,
    0.945607325f,  0.325310292f,
    0.945107193f,  0.326760452f,
    0.944604837f,  0.328209844f,
    0.944100258f,  0.329658463f,
    0.943593458f,  0.331106306f,
    0.943084437f,  0.332553370f,
    0.942573198f,  0.333999651f,
    0.942059740f,  0.335445147f,
    0.941544065f,  0.336889853f,
    0.941026175f,  0.338333767f,
    0.940506071f,  0.339776884f,
    0.939983753f,  0.341219202f,
    0.939459224f,  0.342660717f,
    0.938932484f,  0.344101426f,
    0.938403534f,  0.345541325f,
    0.937872376f,  0.346980411f,
    0.937339012f,  0.348418680f,
    0.936803442f,  0.349856130f,
    0.936265667f,  0.351292756f,
    0.935725689f,  0.352728556f,
    0.935183510f,  0.354163525f,
    0.934639130f,  0.355597662f,
    0.934092550f,  0.357030961f,
    0.933543773f,  0.358463421f,
    0.932992799f,  0.359895037f,
    0.932439629f,  0.361325806f,
    0.931884266f,  0.362755724f,
    0.931326709f,  0.364184790f,
    0.930766961f,  0.365612998f,
    0.930205023f,  0.367040346f,
    0.929640896f,  0.368466830f,
    0.929074581f,  0.369892447f,
    0.928506080f,  0.371317194f,
    0.927935395f,  0.372741067f,
    0.927362526f,  0.374164063f,
    0.926787474f,  0.375586178f,
    0.926210242f,  0.377007410f,
    0.925630831f,  0.378427755f,
    0.925049241f,  0.379847209f,
    0.924465474f,  0.381265769f,
    0.923879533f,  0.382683432f,
    0.923291417f,  0.384100195f,
    0.922701128f,  0.385516054f,
    0.922108669f,  0.386931006f,
    0.921514039f,  0.388345047f,
    0.920917242f,  0.389758174f,
    0.920318277f,  0.391170384f,
    0.919717146f,  0.392581674f,
    0.919113852f,  0.393992040f,
    0.918508394f,  0.395401479f,
    0.917900776f,  0.396809987f,
    0.917290997f,  0.398217562f,
    0.916679060f,  0.399624200f,
    0.916064966f,  0.401029897f,
    0.915448716f,  0.402434651f,
    0.914830312f,  0.403838458f,
    0.914209756f,  0.405241314f,
    0.913587048f,  0.406643217f,
    0.912962190f,  0.408044163f,
    0.912335185f,  0.409444149f,
    0.911706032f,  0.410843171f,
    0.911074734f,  0.412241227f,
    0.910441292f,  0.413638312f,
    0.909805708f,  0.415034424f,
    0.909167983f,  0.416429560f,
    0.908528119f,  0.417823716f,
    0.907886116f,  0.419216888f,
    0.907241978f,  0.420609074f,
    0.906595705f,  0.422000271f,
    0.905947298f,  0.423390474f,
    0.905296759f,  0.424779681f,
    0.904644091f,  0.426167889f,
    0.903989293f,  0.427555093f,
    0.903332368f,  0.428941292f,
    0.902673318f,  0.430326481f,
    0.902012144f,  0.431710658f,
    0.901348847f,  0.433093819f,
    0.900683429f,  0.434475961f,
    0.900015892f,  0.435857080f,
    0.899346237f,  0.437237174f,
    0.898674466f,  0.438616239f,
    0.898000580f,  0.439994271f,
    0.897324581f,  0.441371269f,
    0.896646470f,  0.442747228f,
    0.895966250f,  0.444122145f,
    0.895283921f,  0.445496017f,
    0.894599486f,  0.446868840f,
    0.893912945f,  0.448240612f,
    0.893224301f,  0.449611330f,
    0.892533555f,  0.450980989f,
    0.891840709f,  0.452349587f,
    0.891145765f,  0.453717121f,
    0.890448723f,  0.455083587f,
    0.889749586f,  0.456448982f,
    0.889048356f,  0.457813304f,
    0.888345033f,  0.459176548f,
    0.887639620f,  0.460538711f,
    0.886932119f,  0.461899791f,
    0.886222530f,  0.463259784f,
    0.885510856f,  0.464618686f,
    0.884797098f,  0.465976496f,
    0.884081259f,  0.467333209f,
    0.883363339f,  0.468688822f,
    0.882643340f,  0.470043332f,
    0.881921264f,  0.471396737f,
    0.881197113f,  0.472749032f,
    0.880470889f,  0.474100215f,
    0.879742593f,  0.475450282f,
    0.879012226f,  0.476799230f,
    0.878279792f,  0.478147056f,
    0.877545290f,  0.479493758f,
    0.876808724f,  0.480839331f,
    0.876070094f,  0.482183772f,
    0.875329403f,  0.483527079f,
    0.874586652f,  0.484869248f,
    0.873841843f,  0.486210276f,
    0.873094978f,  0.487550160f,
    0.872346059f,  0.488888897f,
    0.871595087f,  0.490226483f,
    0.870842063f,  0.491562916f,
    0.870086991f,  0.492898192f,
    0.869329871f,  0.494232309f,
    0.868570706f,  0.495565262f,
    0.867809497f,  0.496897049f,
    0.867046246f,  0.498227667f,
    0.866280954f,  0.499557113f,
    0.865513624f,  0.500885383f,
    0.864744258f,  0.502212474f,
    0.863972856f,  0.503538384f,
    0.863199422f,  0.504863109f,
    0.862423956f,  0.506186645f,
    0.861646461f,  0.507508991f,
    0.860866939f,  0.508830143f,
    0.860085390f,  0.510150097f,
    0.859301818f,  0.511468850f,
    0.858516224f,  0.512786401f,
    0.857728610f,  0.514102744f,
    0.856938977f,  0.515417878f,
    0.856147328f,  0.516731799f,
    0.855353665f,  0.518044504f,
    0.854557988f,  0.519355990f,
    0.853760301f,  0.520666254f,
    0.852960605f,  0.521975293f,
    0.852158902f,  0.523283103f,
    0.851355193f,  0.524589683f,
    0.850549481f,  0.525895027f,
    0.849741768f,  0.527199135f,
    0.848932055f,  0.528502002f,
    0.848120345f,  0.529803625f,
    0.847306639f,  0.531104001f,
    0.846490939f,  0.532403128f,
    0.845673247f,  0.533701002f,
    0.844853565f,  0.534997620f,
    0.844031895f,  0.536292979f,
    0.843208240f,  0.537587076f,
    0.842382600f,  0.538879909f,
    0.841554977f,  0.540171473f,
    0.840725375f,  0.541461766f,
    0.839893794f,  0.542750785f,
    0.839060237f,  0.544038527f,
    0.838224706f,  0.545324988f,
    0.837387202f,  0.546610167f,
    0.836547727f,  0.547894059f,
    0.835706284f,  0.549176662f,
    0.834862875f,  0.550457973f,
    0.834017501f,  0.551737988f,
    0.833170165f,  0.553016706f,
    0.832320868f,  0.554294121f,
    0.831469612f,  0.555570233f,
    0.830616400f,  0.556845037f,
    0.829761234f,  0.558118531f,
    0.828904115f,  0.559390712f,
    0.828045045f,  0.560661576f,
    0.827184027f,  0.561931121f,
    0.826321063f,  0.563199344f,
    0.825456154f,  0.564466242f,
    0.824589303f,  0.565731811f,
    0.823720511f,  0.566996049f,
    0.822849781f,  0.568258953f,
    0.821977115f,  0.569520519f,
    0.821102515f,  0.570780746f,
    0.820225983f,  0.572039629f,
    0.819347520f,  0.573297167f,
    0.818467130f,  0.574553355f,
    0.817584813f,  0.575808191f,
    0.816700573f,  0.577061673f,
    0.815814411f,  0.578313796f,
    0.814926329f,  0.579564559f,
    0.814036330f,  0.580813958f,
    0.813144415f,  0.582061990f,
    0.812250587f,  0.583308653f,
    0.811354847f,  0.584553943f,
    0.810457198f,  0.585797857f,
    0.809557642f,  0.587040394f,
    0.808656182f,  0.588281548f,
    0.807752818f,  0.589521319f,
    0.806847554f,  0.590759702f,
    0.805940391f,  0.591996695f,
    0.805031331f,  0.593232295f,
    0.804120377f,  0.594466499f,
    0.803207531f,  0.595699304f,
    0.802292796f,  0.596930708f,
    0.801376172f,  0.598160707f,
    0.800457662f,  0.599389298f,
    0.799537269f,  0.600616479f,
    0.798614995f,  0.601842247f,
    0.797690841f,  0.603066599f,
    0.796764810f,  0.604289531f,
    0.795836905f,  0.605511041f,
    0.794907126f,  0.606731127f,
    0.793975478f,  0.607949785f,
    0.793041960f,  0.609167012f,
    0.792106577f,  0.610382806f,
    0.791169330f,  0.611597164f,
    0.790230221f,  0.612810082f,
    0.789289253f,  0.614021559f,
    0.788346428f,  0.615231591f,
    0.787401747f,  0.616440175f,
    0.786455214f,  0.617647308f,
    0.785506830f,  0.618852988f,
    0.784556597f,  0.620057212f,
    0.783604519f,  0.621259977f,
    0.782650596f,  0.622461279f,
    0.781694832f,  0.623661118f,
    0.780737229f,  0.624859488f,
    0.779777788f,  0.626056388f,
    0.778816512f,  0.627251815f,
    0.777853404f,  0.628445767f,
    0.776888466f,  0.629638239f,
    0.775921699f,  0.630829230f,
    0.774953107f,  0.632018736f,
    0.773982691f,  0.633206755f,
    0.773010453f,  0.634393284f,
    0.772036397f,  0.635578320f,
    0.771060524f,  0.636761861f,
    0.770082837f,  0.637943904f,
    0.769103338f,  0.639124445f,
    0.768122029f,  0.640303482f,
    0.767138912f,  0.641481013f,
    0.766153990f,  0.642657034f,
    0.765167266f,  0.643831543f,
    0.764178741f,  0.645004537f,
    0.763188417f,  0.646176013f,
    0.762196298f,  0.647345969f,
    0.761202385f,  0.648514401f,
    0.760206682f,  0.649681307f,
    0.759209189f,  0.650846685f,
    0.758209910f,  0.652010531f,
    0.757208847f,  0.653172843f,
    0.756206001f,  0.654333618f,
    0.755201377f,  0.655492853f,
    0.754194975f,  0.656650546f,
    0.753186799f,  0.657806693f,
    0.752176850f,  0.658961293f,
    0.751165132f,  0.660114342f,
    0.750151646f,  0.661265838f,
    0.749136395f,  0.662415778f,
    0.748119380f,  0.663564159f,
    0.747100606f,  0.664710978f,
    0.746080074f,  0.665856234f,
    0.745057785f,  0.666999922f,
    0.744033744f,  0.668142041f,
    0.743007952f,  0.669282588f,
    0.741980412f,  0.670421560f,
    0.740951125f,  0.671558955f,
    0.739920095f,  0.672694769f,
    0.738887324f,  0.673829000f,
    0.737852815f,  0.674961646f,
    0.736816569f,  0.676092704f,
    0.735778589f,  0.677222170f,
    0.734738878f,  0.678350043f,
    0.733697438f,  0.679476320f,
    0.732654272f,  0.680600998f,
    0.731609381f,  0.681724074f,
    0.730562769f,  0.682845546f,
    0.729514438f,  0.683965412f,
    0.728464390f,  0.685083668f,
    0.727412629f,  0.686200312f,
    0.726359155f,  0.687315341f,
    0.725303972f,  0.688428753f,
    0.724247083f,  0.689540545f,
    0.723188489f,  0.690650714f,
    0.722128194f,  0.691759258f,
    0.721066199f,  0.692866175f,
    0.720002508f,  0.693971461f,
    0.718937122f,  0.695075114f,
    0.717870045f,  0.696177131f,
    0.716801279f,  0.697277511f,
    0.715730825f,  0.698376249f,
    0.714658688f,  0.699473345f,
    0.713584869f,  0.700568794f,
    0.712509371f,  0.701662595f,
    0.711432196f,  0.702754744f,
    0.710353347f,  0.703845241f,
    0.709272826f,  0.704934080f,
    0.708190637f,  0.706021261f,
    0.707106781f,  0.707106781f,
    0.706021261f,  0.708190637f,
    0.704934080f,  0.709272826f,
    0.703845241f,  0.710353347f,
    0.702754744f,  0.711432196f,
    0.701662595f,  0.712509371f,
    0.700568794f,  0.713584869f,
    0.699473345f,  0.714658688f,
    0.698376249f,  0.715730825f,
    0.697277511f,  0.716801279f,
    0.696177131f,  0.717870045f,
    0.695075114f,  0.718937122f,
    0.693971461f,  0.720002508f,
    0.692866175f,  0.721066199f,
    0.691759258f,  0.722128194f,
    0.690650714f,  0.723188489f,
    0.689540545f,  0.724247083f,
    0.688428753f,  0.725303972f,
    0.687315341f,  0.726359155f,
    0.686200312f,  0.727412629f,
    0.685083668f,  0.728464390f,
    0.683965412f,  0.729514438f,
    0.682845546f,  0.730562769f,
    0.681724074f,  0.731609381f,
    0.680600998f,  0.732654272f,
    0.679476320f,  0.733697438f,
    0.678350043f,  0.734738878f,
    0.677222170f,  0.735778589f,
    0.676092704f,  0.736816569f,
    0.674961646f,  0.737852815f,
    0.673829000f,  0.738887324f,
    0.672694769f,  0.739920095f,
    0.671558955f,  0.740951125f,
    0.670421560f,  0.741980412f,
    0.669282588f,  0.743007952f,
    0.668142041f,  0.744033744f,
    0.666999922f,  0.745057785f,
    0.665856234f,  0.746080074f,
    0.664710978f,  0.747100606f,
    0.663564159f,  0.748119380f,
    0.662415778f,  0.749136395f,
    0.661265838f,  0.750151646f,
    0.660114342f,  0.751165132f,
    0.658961293f,  0.752176850f,
    0.657806693f,  0.753186799f,
    0.656650546f,  0.754194975f,
    0.655492853f,  0.755201377f,
    0.654333618f,  0.756206001f,
    0.653172843f,  0.757208847f,
    0.652010531f,  0.758209910f,
    0.650846685f,  0.759209189f,
    0.649681307f,  0.760206682f,
    0.648514401f,  0.761202385f,
    0.647345969f,  0.762196298f,
    0.646176013f,  0.763188417f,
    0.645004537f,  0.764178741f,
    0.643831543f,  0.765167266f,
    0.642657034f,  0.766153990f,
    0.641481013f,  0.767138912f,
    0.640303482f,  0.768122029f,
    0.639124445f,  0.769103338f,
    0.637943904f,  0.770082837f,
    0.636761861f,  0.771060524f,
    0.635578320f,  0.772036397f,
    0.634393284f,  0.773010453f,
    0.633206755f,  0.773982691f,
    0.632018736f,  0.774953107f,
    0.630829230f,  0.775921699f,
    0.629638239f,  0.776888466f,
    0.628445767f,  0.777853404f,
    0.627251815f,  0.778816512f,
    0.626056388f,  0.779777788f,
    0.624859488f,  0.780737229f,
    0.623661118f,  0.781694832f,
    0.622461279f,  0.782650596f,
    0.621259977f,  0.783604519f,
    0.620057212f,  0.784556597f,
    0.618852988f,  0.785506830f,
    0.617647308f,  0.786455214f,
    0.616440175f,  0.787401747f,
    0.615231591f,  0.788346428f,
    0.614021559f,  0.789289253f,
    0.612810082f,  0.790230221f,
    0.611597164f,  0.791169330f,
    0.610382806f,  0.792106577f,
    0.609167012f,  0.793041960f,
    0.607949785f,  0.793975478f,
    0.606731127f,  0.794907126f,
    0.605511041f,  0.795836905f,
    0.604289531f,  0.796764810f,
    0.603066599f,  0.797690841f,
    0.601842247f,  0.798614995f,
    0.600616479f,  0.799537269f,
    0.599389298f,  0.800457662f,
    0.598160707f,  0.801376172f,
    0.596930708f,  0.802292796f,
    0.595699304f,  0.803207531f,
    0.594466499f,  0.804120377f,
    0.593232295f,  0.805031331f,
    0.591996695f,  0.805940391f,
    0.590759702f,  0.806847554f,
    0.589521319f,  0.807752818f,
    0.588281548f,  0.808656182f,
    0.587040394f,  0.809557642f,
    0.585797857f,  0.810457198f,
    0.584553943f,  0.811354847f,
    0.583308653f,  0.812250587f,
    0.582061990f,  0.813144415f,
    0.580813958f,  0.814036330f,
    0.579564559f,  0.814926329f,
    0.578313796f,  0.815814411f,
    0.577061673f,  0.816700573f,
    0.575808191f,  0.817584813f,
    0.574553355f,  0.818467130f,
    0.573297167f,  0.819347520f,
    0.572039629f,  0.820225983f,
    0.570780746f,  0.821102515f,
    0.569520519f,  0.821977115f,
    0.568258953f,  0.822849781f,
    0.566996049f,  0.823720511f,
    0.565731811f,  0.824589303f,
    0.564466242f,  0.825456154f,
    0.563199344f,  0.826321063f,
    0.561931121f,  0.827184027f,
    0.560661576f,  0.828045045f,
    0.559390712f,  0.828904115f,
    0.558118531f,  0.829761234f,
    0.556845037f,  0.830616400f,
    0.555570233f,  0.831469612f,
    0.554294121f,  0.832320868f,
    0.553016706f,  0.833170165f,
    0.551737988f,  0.834017501f,
    0.550457973f,  0.834862875f,
    0.549176662f,  0.835706284f,
    0.547894059f,  0.836547727f,
    0.546610167f,  0.837387202f,
    0.545324988f,  0.838224706f,
    0.544038527f,  0.839060237f,
    0.542750785f,  0.839893794f,
    0.541461766f,  0.840725375f,
    0.540171473f,  0.841554977f,
    0.538879909f,  0.842382600f,
    0.537587076f,  0.843208240f,
    0.536292979f,  0.844031895f,
    0.534997620f,  0.844853565f,
    0.533701002f,  0.845673247f,
    0.532403128f,  0.846490939f,
    0.531104001f,  0.847306639f,
    0.529803625f,  0.848120345f,
    0.528502002f,  0.848932055f,
    0.527199135f,  0.849741768f,
    0.525895027f,  0.850549481f,
    0.524589683f,  0.851355193f,
    0.523283103f,  0.852158902f,
    0.521975293f,  0.852960605f,
    0.520666254f,  0.853760301f,
    0.519355990f,  0.854557988f,
    0.518044504f,  0.855353665f,
    0.516731799f,  0.856147328f,
    0.515417878f,  0.856938977f,
    0.514102744f,  0.857728610f,
    0.512786401f,  0.858516224f,
    0.511468850f,  0.859301818f,
    0.510150097f,  0.860085390f,
    0.508830143f,  0.860866939f,
    0.507508991f,  0.861646461f,
    0.506186645f,  0.862423956f,
    0.504863109f,  0.863199422f,
    0.503538384f,  0.863972856f,
    0.502212474f,  0.864744258f,
    0.500885383f,  0.865513624f,
    0.499557113f,  0.866280954f,
    0.498227667f,  0.867046246f,
    0.496897049f,  0.867809497f,
    0.495565262f,  0.868570706f,
    0.494232309f,  0.869329871f,
    0.492898192f,  0.870086991f,
    0.491562916f,  0.870842063f,
    0.490226483f,  0.871595087f,
    0.488888897f,  0.872346059f,
    0.487550160f,  0.873094978f,
    0.486210276f,  0.873841843f,
    0.484869248f,  0.874586652f,
    0.483527079f,  0.875329403f,
    0.482183772f,  0.876070094f,
    0.480839331f,  0.876808724f,
    0.479493758f,  0.877545290f,
    0.478147056f,  0.878279792f,
    0.476799230f,  0.879012226f,
    0.475450282f,  0.879742593f,
    0.474100215f,  0.880470889f,
    0.472749032f,  0.881197113f,
    0.471396737f,  0.881921264f,
    0.470043332f,  0.882643340f,
    0.468688822f,  0.883363339f,
    0.467333209f,  0.884081259f,
    0.465976496f,  0.884797098f,
    0.464618686f,  0.885510856f,
    0.463259784f,  0.886222530f,
    0.461899791f,  0.886932119f,
    0.460538711f,  0.887639620f,
    0.459176548f,  0.888345033f,
    0.457813304f,  0.889048356f,
    0.456448982f,  0.889749586f,
    0.455083587f,  0.890448723f,
    0.453717121f,  0.891145765f,
    0.452349587f,  0.891840709f,
    0.450980989f,  0.892533555f,
    0.449611330f,  0.893224301f,
    0.448240612f,  0.893912945f,
    0.446868840f,  0.894599486f,
    0.445496017f,  0.895283921f,
    0.444122145f,  0.895966250f,
    0.442747228f,  0.896646470f,
    0.441371269f,  0.897324581f,
    0.439994271f,  0.898000580f,
    0.438616239f,  0.898674466f,
    0.437237174f,  0.899346237f,
    0.435857080f,  0.900015892f,
    0.434475961f,  0.900683429f,
    0.433093819f,  0.901348847f,
    0.431710658f,  0.902012144f,
    0.430326481f,  0.902673318f,
    0.428941292f,  0.903332368f,
    0.427555093f,  0.903989293f,
    0.426167889f,  0.904644091f,
    0.424779681f,  0.905296759f,
    0.423390474f,  0.905947298f,
    0.422000271f,  0.906595705f,
    0.420609074f,  0.907241978f,
    0.419216888f,  0.907886116f,
    0.417823716f,  0.908528119f,
    0.416429560f,  0.909167983f,
    0.415034424f,  0.909805708f,
    0.413638312f,  0.910441292f,
    0.412241227f,  0.911074734f,
    0.410843171f,  0.911706032f,
    0.409444149f,  0.912335185f,
    0.408044163f,  0.912962190f,
    0.406643217f,  0.913587048f,
    0.405241314f,  0.914209756f,
    0.403838458f,  0.914830312f,
    0.402434651f,  0.915448716f,
    0.401029897f,  0.916064966f,
    0.399624200f,  0.916679060f,
    0.398217562f,  0.917290997f,
    0.396809987f,  0.917900776f,
    0.395401479f,  0.918508394f,
    0.393992040f,  0.919113852f,
    0.392581674f,  0.919717146f,
    0.391170384f,  0.920318277f,
    0.389758174f,  0.920917242f,
    0.388345047f,  0.921514039f,
    0.386931006f,  0.922108669f,
    0.385516054f,  0.922701128f,
    0.384100195f,  0.923291417f,
    0.382683432f,  0.923879533f,
    0.381265769f,  0.924465474f,
    0.379847209f,  0.925049241f,
    0.378427755f,  0.925630831f,
    0.377007410f,  0.926210242f,
    0.375586178f,  0.926787474f,
    0.374164063f,  0.927362526f,
    0.372741067f,  0.927935395f,
    0.371317194f,  0.928506080f,
    0.369892447f,  0.929074581f,
    0.368466830f,  0.929640896f,
    0.367040346f,  0.930205023f,
    0.365612998f,  0.930766961f,
    0.364184790f,  0.931326709f,
    0.362755724f,  0.931884266f,
    0.361325806f,  0.932439629f,
    0.359895037f,  0.932992799f,
    0.358463421f,  0.933543773f,
    0.357030961f,  0.934092550f,
    0.355597662f,  0.934639130f,
    0.354163525f,  0.935183510f,
    0.352728556f,  0.935725689f,
    0.351292756f,  0.936265667f,
    0.349856130f,  0.936803442f,
    0.348418680f,  0.937339012f,
    0.346980411f,  0.937872376f,
    0.345541325f,  0.938403534f,
    0.344101426f,  0.938932484f,
    0.342660717f,  0.939459224f,
    0.341219202f,  0.939983753f,
    0.339776884f,  0.940506071f,
    0.338333767f,  0.941026175f,
    0.336889853f,  0.941544065f,
    0.335445147f,  0.942059740f,
    0.333999651f,  0.942573198f,
    0.332553370f,  0.943084437f,
    0.331106306f,  0.943593458f,
    0.329658463f,  0.944100258f,
    0.328209844f,  0.944604837f,
    0.326760452f,  0.945107193f,
    0.325310292f,  0.945607325f,
    0.323859367f,  0.946105232f,
    0.322407679f,  0.946600913f,
    0.320955232f,  0.947094366f,
    0.319502031f,  0.947585591f,
    0.318048077f,  0.948074586f,
    0.316593376f,  0.948561350f,
    0.315137929f,  0.949045882f,
    0.313681740f,  0.949528181f,
    0.312224814f,  0.950008245f,
    0.310767153f,  0.950486074f,
    0.309308760f,  0.950961666f,
    0.307849640f,  0.951435021f,
    0.306389795f,  0.951906137f,
    0.304929230f,  0.952375013f,
    0.303467947f,  0.952841648f,
    0.302005949f,  0.953306040f,
    0.300543241f,  0.953768190f,
    0.299079826f,  0.954228095f,
    0.297615707f,  0.954685755f,
    0.296150888f,  0.955141168f,
    0.294685372f,  0.955594334f,
    0.293219163f,  0.956045251f,
    0.291752263f,  0.956493919f,
    0.290284677f,  0.956940336f,
    0.288816408f,  0.957384501f,
    0.287347460f,  0.957826413f,
    0.285877835f,  0.958266071f,
    0.284407537f,  0.958703475f,
    0.282936570f,  0.959138622f,
    0.281464938f,  0.959571513f,
    0.279992643f,  0.960002146f,
    0.278519689f,  0.960430519f,
    0.277046080f,  0.960856633f,
    0.275571819f,  0.961280486f,
    0.274096910f,  0.961702077f,
    0.272621355f,  0.962121404f,
    0.271145160f,  0.962538468f,
    0.269668326f,  0.962953267f,
    0.268190857f,  0.963365800f,
    0.266712757f,  0.963776066f,
    0.265234030f,  0.964184064f,
    0.263754679f,  0.964589793f,
    0.262274707f,  0.964993253f,
    0.260794118f,  0.965394442f,
    0.259312915f,  0.965793359f,
    0.257831102f,  0.966190003f,
    0.256348682f,  0.966584374f,
    0.254865660f,  0.966976471f,
    0.253382037f,  0.967366292f,
    0.251897818f,  0.967753837f,
    0.250413007f,  0.968139105f,
    0.248927606f,  0.968522094f,
    0.247441619f,  0.968902805f,
    0.245955050f,  0.969281235f,
    0.244467903f,  0.969657385f,
    0.242980180f,  0.970031253f,
    0.241491885f,  0.970402839f,
    0.240003022f,  0.970772141f,
    0.238513595f,  0.971139158f,
    0.237023606f,  0.971503891f,
    0.235533059f,  0.971866337f,
    0.234041959f,  0.972226497f,
    0.232550307f,  0.972584369f,
    0.231058108f,  0.972939952f,
    0.229565366f,  0.973293246f,
    0.228072083f,  0.973644250f,
    0.226578264f,  0.973992962f,
    0.225083911f,  0.974339383f,
    0.223589029f,  0.974683511f,
    0.222093621f,  0.975025345f,
    0.220597690f,  0.975364885f,
    0.219101240f,  0.975702130f,
    0.217604275f,  0.976037079f,
    0.216106797f,  0.976369731f,
    0.214608811f,  0.976700086f,
    0.213110320f,  0.977028143f,
    0.211611327f,  0.977353900f,
    0.210111837f,  0.977677358f,
    0.208611852f,  0.977998515f,
    0.207111376f,  0.978317371f,
    0.205610413f,  0.978633924f,
    0.204108966f,  0.978948175f,
    0.202607039f,  0.979260123f,
    0.201104635f,  0.979569766f,
    0.199601758f,  0.979877104f,
    0.198098411f,  0.980182136f,
    0.196594598f,  0.980484862f,
    0.195090322f,  0.980785280f,
    0.193585587f,  0.981083391f,
    0.192080397f,  0.981379193f,
    0.190574755f,  0.981672686f,
    0.189068664f,  0.981963869f,
    0.187562129f,  0.982252741f,
    0.186055152f,  0.982539302f,
    0.184547737f,  0.982823551f,
    0.183039888f,  0.983105487f,
    0.181531608f,  0.983385110f,
    0.180022901f,  0.983662419f,
    0.178513771f,  0.983937413f,
    0.177004220f,  0.984210092f,
    0.175494253f,  0.984480455f,
    0.173983873f,  0.984748502f,
    0.172473084f,  0.985014231f,
    0.170961889f,  0.985277642f,
    0.169450291f,  0.985538735f,
    0.167938295f,  0.985797509f,
    0.166425904f,  0.986053963f,
    0.164913120f,  0.986308097f,
    0.163399949f,  0.986559910f,
    0.161886394f,  0.986809402f,
    0.160372457f,  0.987056571f,
    0.158858143f,  0.987301418f,
    0.157343456f,  0.987543942f,
    0.155828398f,  0.987784142f,
    0.154312973f,  0.988022017f,
    0.152797185f,  0.988257568f,
    0.151281038f,  0.988490793f,
    0.149764535f,  0.988721692f,
    0.148247679f,  0.988950265f,
    0.146730474f,  0.989176510f,
    0.145212925f,  0.989400428f,
    0.143695033f,  0.989622017f,
    0.142176804f,  0.989841278f,
    0.140658239f,  0.990058210f,
    0.139139344f,  0.990272812f,
    0.137620122f,  0.990485084f,
    0.136100575f,  0.990695025f,
    0.134580709f,  0.990902635f,
    0.133060525f,  0.991107914f,
    0.131540029f,  0.991310860f,
    0.130019223f,  0.991511473f,
    0.128498111f,  0.991709754f,
    0.126976696f,  0.991905700f,
    0.125454983f,  0.992099313f,
    0.123932975f,  0.992290591f,
    0.122410675f,  0.992479535f,
    0.120888087f,  0.992666142f,
    0.119365215f,  0.992850414f,
    0.117842062f,  0.993032350f,
    0.116318631f,  0.993211949f,
    0.114794927f,  0.993389211f,
    0.113270952f,  0.993564136f,
    0.111746711f,  0.993736722f,
    0.110222207f,  0.993906970f,
    0.108697444f,  0.994074879f,
    0.107172425f,  0.994240449f,
    0.105647154f,  0.994403680f,
    0.104121634f,  0.994564571f,
    0.102595869f,  0.994723121f,
    0.101069863f,  0.994879331f,
    0.099543619f,  0.995033199f,
    0.098017140f,  0.995184727f,
    0.096490431f,  0.995333912f,
    0.094963495f,  0.995480755f,
    0.093436336f,  0.995625256f,
    0.091908956f,  0.995767414f,
    0.090381361f,  0.995907229f,
    0.088853553f,  0.996044701f,
    0.087325535f,  0.996179829f,
    0.085797312f,  0.996312612f,
    0.084268888f,  0.996443051f,
    0.082740265f,  0.996571146f,
    0.081211447f,  0.996696895f,
    0.079682438f,  0.996820299f,
    0.078153242f,  0.996941358f,
    0.076623861f,  0.997060070f,
    0.075094301f,  0.997176437f,
    0.073564564f,  0.997290457f,
    0.072034653f,  0.997402130f,
    0.070504573f,  0.997511456f,
    0.068974328f,  0.997618435f,
    0.067443920f,  0.997723067f,
    0.065913353f,  0.997825350f,
    0.064382631f,  0.997925286f,
    0.062851758f,  0.998022874f,
    0.061320736f,  0.998118113f,
    0.059789571f,  0.998211003f,
    0.058258265f,  0.998301545f,
    0.056726821f,  0.998389737f,
    0.055195244f,  0.998475581f,
    0.053663538f,  0.998559074f,
    0.052131705f,  0.998640218f,
    0.050599749f,  0.998719012f,
    0.049067674f,  0.998795456f,
    0.047535484f,  0.998869550f,
    0.046003182f,  0.998941293f,
    0.044470772f,  0.999010686f,
    0.042938257f,  0.999077728f,
    0.041405641f,  0.999142419f,
    0.039872928f,  0.999204759f,
    0.038340120f,  0.999264747f,
    0.036807223f,  0.999322385f,
    0.035274239f,  0.999377670f,
    0.033741172f,  0.999430605f,
    0.032208025f,  0.999481187f,
    0.030674803f,  0.999529418f,
    0.029141509f,  0.999575296f,
    0.027608146f,  0.999618822f,
    0.026074718f,  0.999659997f,
    0.024541229f,  0.999698819f,
    0.023007681f,  0.999735288f,
    0.021474080f,  0.999769405f,
    0.019940429f,  0.999801170f,
    0.018406730f,  0.999830582f,
    0.016872988f,  0.999857641f,
    0.015339206f,  0.999882347f,
    0.013805389f,  0.999904701f,
    0.012271538f,  0.999924702f,
    0.010737659f,  0.999942350f,
    0.009203755f,  0.999957645f,
    0.007669829f,  0.999970586f,
    0.006135885f,  0.999981175f,
    0.004601926f,  0.999989411f,
    0.003067957f,  0.999995294f,
    0.001533980f,  0.999998823f,
    0.000000000f,  1.000000000f,
   -0.001533980f,  0.999998823f,
   -0.003067957f,  0.999995294f,
   -0.004601926f,  0.999989411f,
   -0.006135885f,  0.999981175f,
   -0.007669829f,  0.999970586f,
   -0.009203755f,  0.999957645f,
   -0.010737659f,  0.999942350f,
   -0.012271538f,  0.999924702f,
   -0.013805389f,  0.999904701f,
   -0.015339206f,  0.999882347f,
   -0.016872988f,  0.999857641f,
   -0.018406730f,  0.999830582f,
   -0.019940429f,  0.999801170f,
   -0.021474080f,  0.999769405f,
   -0.023007681f,  0.999735288f,
   -0.024541229f,  0.999698819f,
   -0.026074718f,  0.999659997f,
   -0.027608146f,  0.999618822f,
   -0.029141509f,  0.999575296f,
   -0.030674803f,  0.999529418f,
   -0.032208025f,  0.999481187f,
   -0.033741172f,  0.999430605f,
   -0.035274239f,  0.999377670f,
   -0.036807223f,  0.999322385f,
   -0.038340120f,  0.999264747f,
   -0.039872928f,  0.999204759f,
   -0.041405641f,  0.999142419f,
   -0.042938257f,  0.999077728f,
   -0.044470772f,  0.999010686f,
   -0.046003182f,  0.998941293f,
   -0.047535484f,  0.998869550f,
   -0.049067674f,  0.998795456f,
   -0.050599749f,  0.998719012f,
   -0.052131705f,  0.998640218f,
   -0.053663538f,  0.998559074f,
   -0.055195244f,  0.998475581f,
   -0.056726821f,  0.998389737f,
   -0.058258265f,  0.998301545f,
   -0.059789571f,  0.998211003f,
   -0.061320736f,  0.998118113f,
   -0.062851758f,  0.998022874f,
   -0.064382631f,  0.997925286f,
   -0.065913353f,  0.997825350f,
   -0.067443920f,  0.997723067f,
   -0.068974328f,  0.997618435f,
   -0.070504573f,  0.997511456f,
   -0.072034653f,  0.997402130f,
   -0.073564564f,  0.997290457f,
   -0.075094301f,  0.997176437f,
   -0.076623861f,  0.997060070f,
   -0.078153242f,  0.996941358f,
   -0.079682438f,  0.996820299f,
   -0.081211447f,  0.996696895f,
   -0.082740265f,  0.996571146f,
   -0.084268888f,  0.996443051f,
   -0.085797312f,  0.996312612f,
   -0.087325535f,  0.996179829f,
   -0.088853553f,  0.996044701f,
   -0.090381361f,  0.995907229f,
   -0.091908956f,  0.995767414f,
   -0.093436336f,  0.995625256f,
   -0.094963495f,  0.995480755f,
   -0.096490431f,  0.995333912f,
   -0.098017140f,  0.995184727f,
   -0.099543619f,  0.995033199f,
   -0.101069863f,  0.994879331f,
   -0.102595869f,  0.994723121f,
   -0.104121634f,  0.994564571f,
   -0.105647154f,  0.994403680f,
   -0.107172425f,  0.994240449f,
   -0.108697444f,  0.994074879f,
   -0.110222207f,  0.993906970f,
   -0.111746711f,  0.993736722f,
   -0.113270952f,  0.993564136f,
   -0.114794927f,  0.993389211f,
   -0.116318631f,  0.993211949f,
   -0.117842062f,  0.993032350f,
   -0.119365215f,  0.992850414f,
   -0.120888087f,  0.992666142f,
   -0.122410675f,  0.992479535f,
   -0.123932975f,  0.992290591f,
   -0.125454983f,  0.992099313f,
   -0.126976696f,  0.991905700f,
   -0.128498111f,  0.991709754f,
   -0.130019223f,  0.991511473f,
   -0.131540029f,  0.991310860f,
   -0.133060525f,  0.991107914f,
   -0.134580709f,  0.990902635f,
   -0.136100575f,  0.990695025f,
   -0.137620122f,  0.990485084f,
   -0.139139344f,  0.990272812f,
   -0.140658239f,  0.990058210f,
   -0.142176804f,  0.989841278f,
   -0.143695033f,  0.989622017f,
   -0.145212925f,  0.989400428f,
   -0.146730474f,  0.989176510f,
   -0.148247679f,  0.988950265f,
   -0.149764535f,  0.988721692f,
   -0.151281038f,  0.988490793f,
   -0.152797185f,  0.988257568f,
   -0.154312973f,  0.988022017f,
   -0.155828398f,  0.987784142f,
   -0.157343456f,  0.987543942f,
   -0.158858143f,  0.987301418f,
   -0.160372457f,  0.987056571f,
   -0.161886394f,  0.986809402f,
   -0.163399949f,  0.986559910f,
   -0.164913120f,  0.986308097f,
   -0.166425904f,  0.986053963f,
   -0.167938295f,  0.985797509f,
   -0.169450291f,  0.985538735f,
   -0.170961889f,  0.985277642f,
   -0.172473084f,  0.985014231f,
   -0.173983873f,  0.984748502f,
   -0.175494253f,  0.984480455f,
   -0.177004220f,  0.984210092f,
   -0.178513771f,  0.983937413f,
   -0.180022901f,  0.983662419f,
   -0.181531608f,  0.983385110f,
   -0.183039888f,  0.983105487f,
   -0.184547737f,  0.982823551f,
   -0.186055152f,  0.982539302f,
   -0.187562129f,  0.982252741f,
   -0.189068664f,  0.981963869f,
   -0.190574755f,  0.981672686f,
   -0.192080397f,  0.981379193f,
   -0.193585587f,  0.981083391f,
   -0.195090322f,  0.980785280f,
   -0.196594598f,  0.980484862f,
   -0.198098411f,  0.980182136f,
   -0.199601758f,  0.979877104f,
   -0.201104635f,  0.979569766f,
   -0.202607039f,  0.979260123f,
   -0.204108966f,  0.978948175f,
   -0.205610413f,  0.978633924f,
   -0.207111376f,  0.978317371f,
   -0.208611852f,  0.977998515f,
   -0.210111837f,  0.977677358f,
   -0.211611327f,  0.977353900f,
   -0.213110320f,  0.977028143f,
   -0.214608811f,  0.976700086f,
   -0.216106797f,  0.976369731f,
   -0.217604275f,  0.976037079f,
   -0.219101240f,  0.975702130f,
   -0.220597690f,  0.975364885f,
   -0.222093621f,  0.975025345f,
   -0.223589029f,  0.974683511f,
   -0.225083911f,  0.974339383f,
   -0.226578264f,  0.973992962f,
   -0.228072083f,  0.973644250f,
   -0.229565366f,  0.973293246f,
   -0.231058108f,  0.972939952f,
   -0.232550307f,  0.972584369f,
   -0.234041959f,  0.972226497f,
   -0.235533059f,  0.971866337f,
   -0.237023606f,  0.971503891f,
   -0.238513595f,  0.971139158f,
   -0.240003022f,  0.970772141f,
   -0.241491885f,  0.970402839f,
   -0.242980180f,  0.970031253f,
   -0.244467903f,  0.969657385f,
   -0.245955050f,  0.969281235f,
   -0.247441619f,  0.968902805f,
   -0.248927606f,  0.968522094f,
   -0.250413007f,  0.968139105f,
   -0.251897818f,  0.967753837f,
   -0.253382037f,  0.967366292f,
   -0.254865660f,  0.966976471f,
   -0.256348682f,  0.966584374f,
   -0.257831102f,  0.966190003f,
   -0.259312915f,  0.965793359f,
   -0.260794118f,  0.965394442f,
   -0.262274707f,  0.964993253f,
   -0.263754679f,  0.964589793f,
   -0.265234030f,  0.964184064f,
   -0.266712757f,  0.963776066f,
   -0.268190857f,  0.963365800f,
   -0.269668326f,  0.962953267f,
   -0.271145160f,  0.962538468f,
   -0.272621355f,  0.962121404f,
   -0.274096910f,  0.961702077f,
   -0.275571819f,  0.961280486f,
   -0.277046080f,  0.960856633f,
   -0.278519689f,  0.960430519f,
   -0.279992643f,  0.960002146f,
   -0.281464938f,  0.959571513f,
   -0.282936570f,  0.959138622f,
   -0.284407537f,  0.958703475f,
   -0.285877835f,  0.958266071f,
   -0.287347460f,  0.957826413f,
   -0.288816408f,  0.957384501f,
   -0.290284677f,  0.956940336f,
   -0.291752263f,  0.956493919f,
   -0.293219163f,  0.956045251f,
   -0.294685372f,  0.955594334f,
   -0.296150888f,  0.955141168f,
   -0.297615707f,  0.954685755f,
   -0.299079826f,  0.954228095f,
   -0.300543241f,  0.953768190f,
   -0.302005949f,  0.953306040f,
   -0.303467947f,  0.952841648f,
   -0.304929230f,  0.952375013f,
   -0.306389795f,  0.951906137f,
   -0.307849640f,  0.951435021f,
   -0.309308760f,  0.950961666f,
   -0.310767153f,  0.950486074f,
   -0.312224814f,  0.950008245f,
   -0.313681740f,  0.949528181f,
   -0.315137929f,  0.949045882f,
   -0.316593376f,  0.948561350f,
   -0.318048077f,  0.948074586f,
   -0.319502031f,  0.947585591f,
   -0.320955232f,  0.947094366f,
   -0.322407679f,  0.946600913f,
   -0.323859367f,  0.946105232f,
   -0.325310292f,  0.945607325f,
   -0.326760452f,  0.945107193f,
   -0.328209844f,  0.944604837f,
   -0.329658463f,  0.944100258f,
   -0.331106306f,  0.943593458f,
   -0.332553370f,  0.943084437f,
   -0.333999651f,  0.942573198f,
   -0.335445147f,  0.942059740f,
   -0.336889853f,  0.941544065f,
   -0.338333767f,  0.941026175f,
   -0.339776884f,  0.940506071f,
   -0.341219202f,  0.939983753f,
   -0.342660717f,  0.939459224f,
   -0.344101426f,  0.938932484f,
   -0.345541325f,  0.938403534f,
   -0.346980411f,  0.937872376f,
   -0.348418680f,  0.937339012f,
   -0.349856130f,  0.936803442f,
   -0.351292756f,  0.936265667f,
   -0.352728556f,  0.935725689f,
   -0.354163525f,  0.935183510f,
   -0.355597662f,  0.934639130f,
   -0.357030961f,  0.934092550f,
   -0.358463421f,  0.933543773f,
   -0.359895037f,  0.932992799f,
   -0.361325806f,  0.932439629f,
   -0.362755724f,  0.931884266f,
   -0.364184790f,  0.931326709f,
   -0.365612998f,  0.930766961f,
   -0.367040346f,  0.930205023f,
   -0.368466830f,  0.929640896f,
   -0.369892447f,  0.929074581f,
   -0.371317194f,  0.928506080f,
   -0.372741067f,  0.927935395f,
   -0.374164063f,  0.927362526f,
   -0.375586178f,  0.926787474f,
   -0.377007410f,  0.926210242f,
   -0.378427755f,  0.925630831f,
   -0.379847209f,  0.925049241f,
   -0.381265769f,  0.924465474f,
   -0.382683432f,  0.923879533f,
   -0.384100195f,  0.923291417f,
   -0.385516054f,  0.922701128f,
   -0.386931006f,  0.922108669f,
   -0.388345047f,  0.921514039f,
   -0.389758174f,  0.920917242f,
   -0.391170384f,  0.920318277f,
   -0.392581674f,  0.919717146f,
   -0.393992040f,  0.919113852f,
   -0.395401479f,  0.918508394f,
   -0.396809987f,  0.917900776f,
   -0.398217562f,  0.917290997f,
   -0.399624200f,  0.916679060f,
   -0.401029897f,  0.916064966f,
   -0.402434651f,  0.915448716f,
   -0.403838458f,  0.914830312f,
   -0.405241314f,  0.914209756f,
   -0.406643217f,  0.913587048f,
   -0.408044163f,  0.912962190f,
   -0.409444149f,  0.912335185f,
   -0.410843171f,  0.911706032f,
   -0.412241227f,  0.911074734f,
   -0.413638312f,  0.910441292f,
   -0.415034424f,  0.909805708f,
   -0.416429560f,  0.909167983f,
   -0.417823716f,  0.908528119f,
   -0.419216888f,  0.907886116f,
   -0.420609074f,  0.907241978f,
   -0.422000271f,  0.906595705f,
   -0.423390474f,  0.905947298f,
   -0.424779681f,  0.905296759f,
   -0.426167889f,  0.904644091f,
   -0.427555093f,  0.903989293f,
   -0.428941292f,  0.903332368f,
   -0.430326481f,  0.902673318f,
   -0.431710658f,  0.902012144f,
   -0.433093819f,  0.901348847f,
   -0.434475961f,  0.900683429f,
   -0.435857080f,  0.900015892f,
   -0.437237174f,  0.899346237f,
   -0.438616239f,  0.898674466f,
   -0.439994271f,  0.898000580f,
   -0.441371269f,  0.897324581f,
   -0.442747228f,  0.896646470f,
   -0.444122145f,  0.895966250f,
   -0.445496017f,  0.895283921f,
   -0.446868840f,  0.894599486f,
   -0.448240612f,  0.893912945f,
   -0.449611330f,  0.893224301f,
   -0.450980989f,  0.892533555f,
   -0.452349587f,  0.891840709f,
   -0.453717121f,  0.891145765f,
   -0.455083587f,  0.890448723f,
   -0.456448982f,  0.889749586f,
   -0.457813304f,  0.889048356f,
   -0.459176548f,  0.888345033f,
   -0.460538711f,  0.887639620f,
   -0.461899791f,  0.886932119f,
   -0.463259784f,  0.886222530f,
   -0.464618686f,  0.885510856f,
   -0.465976496f,  0.884797098f,
   -0.467333209f,  0.884081259f,
   -0.468688822f,  0.883363339f,
   -0.470043332f,  0.882643340f,
   -0.471396737f,  0.881921264f,
   -0.472749032f,  0.881197113f,
   -0.474100215f,  0.880470889f,
   -0.475450282f,  0.879742593f,
   -0.476799230f,  0.879012226f,
   -0.478147056f,  0.878279792f,
   -0.479493758f,  0.877545290f,
   -0.480839331f,  0.876808724f,
   -0.482183772f,  0.876070094f,
   -0.483527079f,  0.875329403f,
   -0.484869248f,  0.874586652f,
   -0.486210276f,  0.873841843f,
   -0.487550160f,  0.873094978f,
   -0.488888897f,  0.872346059f,
   -0.490226483f,  0.871595087f,
   -0.491562916f,  0.870842063f,
   -0.492898192f,  0.870086991f,
   -0.494232309f,  0.869329871f,
   -0.495565262f,  0.868570706f,
   -0.496897049f,  0.867809497f,
   -0.498227667f,  0.867046246f,
   -0.499557113f,  0.866280954f,
   -0.500885383f,  0.865513624f,
   -0.502212474f,  0.864744258f,
   -0.503538384f,  0.863972856f,
   -0.504863109f,  0.863199422f,
   -0.506186645f,  0.862423956f,
   -0.507508991f,  0.861646461f,
   -0.508830143f,  0.860866939f,
   -0.510150097f,  0.860085390f,
   -0.511468850f,  0.859301818f,
   -0.512786401f,  0.858516224f,
   -0.514102744f,  0.857728610f,
   -0.515417878f,  0.856938977f,
   -0.516731799f,  0.856147328f,
   -0.518044504f,  0.855353665f,
   -0.519355990f,  0.854557988f,
   -0.520666254f,  0.853760301f,
   -0.521975293f,  0.852960605f,
   -0.523283103f,  0.852158902f,
   -0.524589683f,  0.851355193f,
   -0.525895027f,  0.850549481f,
   -0.527199135f,  0.849741768f,
   -0.528502002f,  0.848932055f,
   -0.529803625f,  0.848120345f,
   -0.531104001f,  0.847306639f,
   -0.532403128f,  0.846490939f,
   -0.533701002f,  0.845673247f,
   -0.534997620f,  0.844853565f,
   -0.536292979f,  0.844031895f,
   -0.537587076f,  0.843208240f,
   -0.538879909f,  0.842382600f,
   -0.540171473f,  0.841554977f,
   -0.541461766f,  0.840725375f,
   -0.542750785f,  0.839893794f,
   -0.544038527f,  0.839060237f,
   -0.545324988f,  0.838224706f,
   -0.546610167f,  0.837387202f,
   -0.547894059f,  0.836547727f,
   -0.549176662f,  0.835706284f,
   -0.550457973f,  0.834862875f,
   -0.551737988f,  0.834017501f,
   -0.553016706f,  0.833170165f,
   -0.554294121f,  0.832320868f,
   -0.555570233f,  0.831469612f,
   -0.556845037f,  0.830616400f,
   -0.558118531f,  0.829761234f,
   -0.559390712f,  0.828904115f,
   -0.560661576f,  0.828045045f,
   -0.561931121f,  0.827184027f,
   -0.563199344f,  0.826321063f,
   -0.564466242f,  0.825456154f,
   -0.565731811f,  0.824589303f,
   -0.566996049f,  0.823720511f,
   -0.568258953f,  0.822849781f,
   -0.569520519f,  0.821977115f,
   -0.570780746f,  0.821102515f,
   -0.572039629f,  0.820225983f,
   -0.573297167f,  0.819347520f,
   -0.574553355f,  0.818467130f,
   -0.575808191f,  0.817584813f,
   -0.577061673f,  0.816700573f,
   -0.578313796f,  0.815814411f,
   -0.579564559f,  0.814926329f,
   -0.580813958f,  0.814036330f,
   -0.582061990f,  0.813144415f,
   -0.583308653f,  0.812250587f,
   -0.584553943f,  0.811354847f,
   -0.585797857f,  0.810457198f,
   -0.587040394f,  0.809557642f,
   -0.588281548f,  0.808656182f,
   -0.589521319f,  0.807752818f,
   -0.590759702f,  0.806847554f,
   -0.591996695f,  0.805940391f,
   -0.593232295f,  0.805031331f,
   -0.594466499f,  0.804120377f,
   -0.595699304f,  0.803207531f,
   -0.596930708f,  0.802292796f,
   -0.598160707f,  0.801376172f,
   -0.599389298f,  0.800457662f,
   -0.600616479f,  0.799537269f,
   -0.601842247f,  0.798614995f,
   -0.603066599f,  0.797690841f,
   -0.604289531f,  0.796764810f,
   -0.605511041f,  0.795836905f,
   -0.606731127f,  0.794907126f,
   -0.607949785f,  0.793975478f,
   -0.609167012f,  0.793041960f,
   -0.610382806f,  0.792106577f,
   -0.611597164f,  0.791169330f,
   -0.612810082f,  0.790230221f,
   -0.614021559f,  0.789289253f,
   -0.615231591f,  0.788346428f,
   -0.616440175f,  0.787401747f,
   -0.617647308f,  0.786455214f,
   -0.618852988f,  0.785506830f,
   -0.620057212f,  0.784556597f,
   -0.621259977f,  0.783604519f,
   -0.622461279f,  0.782650596f,
   -0.623661118f,  0.781694832f,
   -0.624859488f,  0.780737229f,
   -0.626056388f,  0.779777788f,
   -0.627251815f,  0.778816512f,
   -0.628445767f,  0.777853404f,
   -0.629638239f,  0.776888466f,
   -0.630829230f,  0.775921699f,
   -0.632018736f,  0.774953107f,
   -0.633206755f,  0.773982691f,
   -0.634393284f,  0.773010453f,
   -0.635578320f,  0.772036397f,
   -0.636761861f,  0.771060524f,
   -0.637943904f,  0.770082837f,
   -0.639124445f,  0.769103338f,
   -0.640303482f,  0.768122029f,
   -0.641481013f,  0.767138912f,
   -0.642657034f,  0.766153990f,
   -0.643831543f,  0.765167266f,
   -0.645004537f,  0.764178741f,
   -0.646176013f,  0.763188417f,
   -0.647345969f,  0.762196298f,
   -0.648514401f,  0.761202385f,
   -0.649681307f,  0.760206682f,
   -0.650846685f,  0.759209189f,
   -0.652010531f,  0.758209910f,
   -0.653172843f,  0.757208847f,
   -0.654333618f,  0.756206001f,
   -0.655492853f,  0.755201377f,
   -0.656650546f,  0.754194975f,
   -0.657806693f,  0.753186799f,
   -0.658961293f,  0.752176850f,
   -0.660114342f,  0.751165132f,
   -0.661265838f,  0.750151646f,
   -0.662415778f,  0.749136395f,
   -0.663564159f,  0.748119380f,
   -0.664710978f,  0.747100606f,
   -0.665856234f,  0.746080074f,
   -0.666999922f,  0.745057785f,
   -0.668142041f,  0.744033744f,
   -0.669282588f,  0.743007952f,
   -0.670421560f,  0.741980412f,
   -0.671558955f,  0.740951125f,
   -0.672694769f,  0.739920095f,
   -0.673829000f,  0.738887324f,
   -0.674961646f,  0.737852815f,
   -0.676092704f,  0.736816569f,
   -0.677222170f,  0.735778589f,
   -0.678350043f,  0.734738878f,
   -0.679476320f,  0.733697438f,
   -0.680600998f,  0.732654272f,
   -0.681724074f,  0.731609381f,
   -0.682845546f,  0.730562769f,
   -0.683965412f,  0.729514438f,
   -0.685083668f,  0.728464390f,
   -0.686200312f,  0.727412629f,
   -0.687315341f,  0.726359155f,
   -0.688428753f,  0.725303972f,
   -0.689540545f,  0.724247083f,
   -0.690650714f,  0.723188489f,
   -0.691759258f,  0.722128194f,
   -0.692866175f,  0.721066199f,
   -0.693971461f,  0.720002508f,
   -0.695075114f,  0.718937122f,
   -0.696177131f,  0.717870045f,
   -0.697277511f,  0.716801279f,
   -0.698376249f,  0.715730825f,
   -0.699473345f,  0.714658688f,
   -0.700568794f,  0.713584869f,
   -0.701662595f,  0.712509371f,
   -0.702754744f,  0.711432196f,
   -0.703845241f,  0.710353347f,
   -0.704934080f,  0.709272826f,
   -0.706021261f,  0.708190637f,
   -0.707106781f,  0.707106781f,
   -0.708190637f,  0.706021261f,
   -0.709272826f,  0.704934080f,
   -0.710353347f,  0.703845241f,
   -0.711432196f,  0.702754744f,
   -0.712509371f,  0.701662595f,
   -0.713584869f,  0.700568794f,
   -0.714658688f,  0.699473345f,
   -0.715730825f,  0.698376249f,
   -0.716801279f,  0.697277511f,
   -0.717870045f,  0.696177131f,
   -0.718937122f,  0.695075114f,
   -0.720002508f,  0.693971461f,
   -0.721066199f,  0.692866175f,
   -0.722128194f,  0.691759258f,
   -0.723188489f,  0.690650714f,
   -0.724247083f,  0.689540545f,
   -0.725303972f,  0.688428753f,
   -0.726359155f,  0.687315341f,
   -0.727412629f,  0.686200312f,
   -0.728464390f,  0.685083668f,
   -0.729514438f,  0.683965412f,
   -0.730562769f,  0.682845546f,
   -0.731609381f,  0.681724074f,
   -0.732654272f,  0.680600998f,
   -0.733697438f,  0.679476320f,
   -0.734738878f,  0.678350043f,
   -0.735778589f,  0.677222170f,
   -0.736816569f,  0.676092704f,
   -0.737852815f,  0.674961646f,
   -0.738887324f,  0.673829000f,
   -0.739920095f,  0.672694769f,
   -0.740951125f,  0.671558955f,
   -0.741980412f,  0.670421560f,
   -0.743007952f,  0.669282588f,
   -0.744033744f,  0.668142041f,
   -0.745057785f,  0.666999922f,
   -0.746080074f,  0.665856234f,
   -0.747100606f,  0.664710978f,
   -0.748119380f,  0.663564159f,
   -0.749136395f,  0.662415778f,
   -0.750151646f,  0.661265838f,
   -0.751165132f,  0.660114342f,
   -0.752176850f,  0.658961293f,
   -0.753186799f,  0.657806693f,
   -0.754194975f,  0.656650546f,
   -0.755201377f,  0.655492853f,
   -0.756206001f,  0.654333618f,
   -0.757208847f,  0.653172843f,
   -0.758209910f,  0.652010531f,
   -0.759209189f,  0.650846685f,
   -0.760206682f,  0.649681307f,
   -0.761202385f,  0.648514401f,
   -0.762196298f,  0.647345969f,
   -0.763188417f,  0.646176013f,
   -0.764178741f,  0.645004537f,
   -0.765167266f,  0.643831543f,
   -0.766153990f,  0.642657034f,
   -0.767138912f,  0.641481013f,
   -0.768122029f,  0.640303482f,
   -0.769103338f,  0.639124445f,
   -0.770082837f,  0.637943904f,
   -0.771060524f,  0.636761861f,
   -0.772036397f,  0.635578320f,
   -0.773010453f,  0.634393284f,
   -0.773982691f,  0.633206755f,
   -0.774953107f,  0.632018736f,
   -0.775921699f,  0.630829230f,
   -0.776888466f,  0.629638239f,
   -0.777853404f,  0.628445767f,
   -0.778816512f,  0.627251815f,
   -0.779777788f,  0.626056388f,
   -0.780737229f,  0.624859488f,
   -0.781694832f,  0.623661118f,
   -0.782650596f,  0.622461279f,
   -0.783604519f,  0.621259977f,
   -0.784556597f,  0.620057212f,
   -0.785506830f,  0.618852988f,
   -0.786455214f,  0.617647308f,
   -0.787401747f,  0.616440175f,
   -0.788346428f,  0.615231591f,
   -0.789289253f,  0.614021559f,
   -0.790230221f,  0.612810082f,
   -0.791169330f,  0.611597164f,
   -0.792106577f,  0.610382806f,
   -0.793041960f,  0.609167012f,
   -0.793975478f,  0.607949785f,
   -0.794907126f,  0.606731127f,
   -0.795836905f,  0.605511041f,
   -0.796764810f,  0.604289531f,
   -0.797690841f,  0.603066599f,
   -0.798614995f,  0.601842247f,
   -0.799537269f,  0.600616479f,
   -0.800457662f,  0.599389298f,
   -0.801376172f,  0.598160707f,
   -0.802292796f,  0.596930708f,
   -0.803207531f,  0.595699304f,
   -0.804120377f,  0.594466499f,
   -0.805031331f,  0.593232295f,
   -0.805940391f,  0.591996695f,
   -0.806847554f,  0.590759702f,
   -0.807752818f,  0.589521319f,
   -0.808656182f,  0.588281548f,
   -0.809557642f,  0.587040394f,
   -0.810457198f,  0.585797857f,
   -0.811354847f,  0.584553943f,
   -0.812250587f,  0.583308653f,
   -0.813144415f,  0.582061990f,
   -0.814036330f,  0.580813958f,
   -0.814926329f,  0.579564559f,
   -0.815814411f,  0.578313796f,
   -0.816700573f,  0.577061673f,
   -0.817584813f,  0.575808191f,
   -0.818467130f,  0.574553355f,
   -0.819347520f,  0.573297167f,
   -0.820225983f,  0.572039629f,
   -0.821102515f,  0.570780746f,
   -0.821977115f,  0.569520519f,
   -0.822849781f,  0.568258953f,
   -0.823720511f,  0.566996049f,
   -0.824589303f,  0.565731811f,
   -0.825456154f,  0.564466242f,
   -0.826321063f,  0.563199344f,
   -0.827184027f,  0.561931121f,
   -0.828045045f,  0.560661576f,
   -0.828904115f,  0.559390712f,
   -0.829761234f,  0.558118531f,
   -0.830616400f,  0.556845037f,
   -0.831469612f,  0.555570233f,
   -0.832320868f,  0.554294121f,
   -0.833170165f,  0.553016706f,
   -0.834017501f,  0.551737988f,
   -0.834862875f,  0.550457973f,
   -0.835706284f,  0.549176662f,
   -0.836547727f,  0.547894059f,
   -0.837387202f,  0.546610167f,
   -0.838224706f,  0.545324988f,
   -0.839060237f,  0.544038527f,
   -0.839893794f,  0.542750785f,
   -0.840725375f,  0.541461766f,
   -0.841554977f,  0.540171473f,
   -0.842382600f,  0.538879909f,
   -0.843208240f,  0.537587076f,
   -0.844031895f,  0.536292979f,
   -0.844853565f,  0.534997620f,
   -0.845673247f,  0.533701002f,
   -0.846490939f,  0.532403128f,
   -0.847306639f,  0.531104001f,
   -0.848120345f,  0.529803625f,
   -0.848932055f,  0.528502002f,
   -0.849741768f,  0.527199135f,
   -0.850549481f,  0.525895027f,
   -0.851355193f,  0.524589683f,
   -0.852158902f,  0.523283103f,
   -0.852960605f,  0.521975293f,
   -0.853760301f,  0.520666254f,
   -0.854557988f,  0.519355990f,
   -0.855353665f,  0.518044504f,
   -0.856147328f,  0.516731799f,
   -0.856938977f,  0.515417878f,
   -0.857728610f,  0.514102744f,
   -0.858516224f,  0.512786401f,
   -0.859301818f,  0.511468850f,
   -0.860085390f,  0.510150097f,
   -0.860866939f,  0.508830143f,
   -0.861646461f,  0.507508991f,
   -0.862423956f,  0.506186645f,
   -0.863199422f,  0.504863109f,
   -0.863972856f,  0.503538384f,
   -0.864744258f,  0.502212474f,
   -0.865513624f,  0.500885383f,
   -0.866280954f,  0.499557113f,
   -0.867046246f,  0.498227667f,
   -0.867809497f,  0.496897049f,
   -0.868570706f,  0.495565262f,
   -0.869329871f,  0.494232309f,
   -0.870086991f,  0.492898192f,
   -0.870842063f,  0.491562916f,
   -0.871595087f,  0.490226483f,
   -0.872346059f,  0.488888897f,
   -0.873094978f,  0.487550160f,
   -0.873841843f,  0.486210276f,
   -0.874586652f,  0.484869248f,
   -0.875329403f,  0.483527079f,
   -0.876070094f,  0.482183772f,
   -0.876808724f,  0.480839331f,
   -0.877545290f,  0.479493758f,
   -0.878279792f,  0.478147056f,
   -0.879012226f,  0.476799230f,
   -0.879742593f,  0.475450282f,
   -0.880470889f,  0.474100215f,
   -0.881197113f,  0.472749032f,
   -0.881921264f,  0.471396737f,
   -0.882643340f,  0.470043332f,
   -0.883363339f,  0.468688822f,
   -0.884081259f,  0.467333209f,
   -0.884797098f,  0.465976496f,
   -0.885510856f,  0.464618686f,
   -0.886222530f,  0.463259784f,
   -0.886932119f,  0.461899791f,
   -0.887639620f,  0.460538711f,
   -0.888345033f,  0.459176548f,
   -0.889048356f,  0.457813304f,
   -0.889749586f,  0.456448982f,
   -0.890448723f,  0.455083587f,
   -0.891145765f,  0.453717121f,
   -0.891840709f,  0.452349587f,
   -0.892533555f,  0.450980989f,
   -0.893224301f,  0.449611330f,
   -0.893912945f,  0.448240612f,
   -0.894599486f,  0.446868840f,
   -0.895283921f,  0.445496017f,
   -0.895966250f,  0.444122145f,
   -0.896646470f,  0.442747228f,
   -0.897324581f,  0.441371269f,
   -0.898000580f,  0.439994271f,
   -0.898674466f,  0.438616239f,
   -0.899346237f,  0.437237174f,
   -0.900015892f,  0.435857080f,
   -0.900683429f,  0.434475961f,
   -0.901348847f,  0.433093819f,
   -0.902012144f,  0.431710658f,
   -0.902673318f,  0.430326481f,
   -0.903332368f,  0.428941292f,
   -0.903989293f,  0.427555093f,
   -0.904644091f,  0.426167889f,
   -0.905296759f,  0.424779681f,
   -0.905947298f,  0.423390474f,
   -0.906595705f,  0.422000271f,
   -0.907241978f,  0.420609074f,
   -0.907886116f,  0.419216888f,
   -0.908528119f,  0.417823716f,
   -0.909167983f,  0.416429560f,
   -0.909805708f,  0.415034424f,
   -0.910441292f,  0.413638312f,
   -0.911074734f,  0.412241227f,
   -0.911706032f,  0.410843171f,
   -0.912335185f,  0.409444149f,
   -0.912962190f,  0.408044163f,
   -0.913587048f,  0.406643217f,
   -0.914209756f,  0.405241314f,
   -0.914830312f,  0.403838458f,
   -0.915448716f,  0.402434651f,
   -0.916064966f,  0.401029897f,
   -0.916679060f,  0.399624200f,
   -0.917290997f,  0.398217562f,
   -0.917900776f,  0.396809987f,
   -0.918508394f,  0.395401479f,
   -0.919113852f,  0.393992040f,
   -0.919717146f,  0.392581674f,
   -0.920318277f,  0.391170384f,
   -0.920917242f,  0.389758174f,
   -0.921514039f,  0.388345047f,
   -0.922108669f,  0.386931006f,
   -0.922701128f,  0.385516054f,
   -0.923291417f,  0.384100195f,
   -0.923879533f,  0.382683432f,
   -0.924465474f,  0.381265769f,
   -0.925049241f,  0.379847209f,
   -0.925630831f,  0.378427755f,
   -0.926210242f,  0.377007410f,
   -0.926787474f,  0.375586178f,
   -0.927362526f,  0.374164063f,
   -0.927935395f,  0.372741067f,
   -0.928506080f,  0.371317194f,
   -0.929074581f,  0.369892447f,
   -0.929640896f,  0.368466830f,
   -0.930205023f,  0.367040346f,
   -0.930766961f,  0.365612998f,
   -0.931326709f,  0.364184790f,
   -0.931884266f,  0.362755724f,
   -0.932439629f,  0.361325806f,
   -0.932992799f,  0.359895037f,
   -0.933543773f,  0.358463421f,
   -0.934092550f,  0.357030961f,
   -0.934639130f,  0.355597662f,
   -0.935183510f,  0.354163525f,
   -0.935725689f,  0.352728556f,
   -0.936265667f,  0.351292756f,
   -0.936803442f,  0.349856130f,
   -0.937339012f,  0.348418680f,
   -0.937872376f,  0.346980411f,
   -0.938403534f,  0.345541325f,
   -0.938932484f,  0.344101426f,
   -0.939459224f,  0.342660717f,
   -0.939983753f,  0.341219202f,
   -0.940506071f,  0.339776884f,
   -0.941026175f,  0.338333767f,
   -0.941544065f,  0.336889853f,
   -0.942059740f,  0.335445147f,
   -0.942573198f,  0.333999651f,
   -0.943084437f,  0.332553370f,
   -0.943593458f,  0.331106306f,
   -0.944100258f,  0.329658463f,
   -0.944604837f,  0.328209844f,
   -0.945107193f,  0.326760452f,
   -0.945607325f,  0.325310292f,
   -0.946105232f,  0.323859367f,
   -0.946600913f,  0.322407679f,
   -0.947094366f,  0.320955232f,
   -0.947585591f,  0.319502031f,
   -0.948074586f,  0.318048077f,
   -0.948561350f,  0.316593376f,
   -0.949045882f,  0.315137929f,
   -0.949528181f,  0.313681740f,
   -0.950008245f,  0.312224814f,
   -0.950486074f,  0.310767153f,
   -0.950961666f,  0.309308760f,
   -0.951435021f,  0.307849640f,
   -0.951906137f,  0.306389795f,
   -0.952375013f,  0.304929230f,
   -0.952841648f,  0.303467947f,
   -0.953306040f,  0.302005949f,
   -0.953768190f,  0.300543241f,
   -0.954228095f,  0.299079826f,
   -0.954685755f,  0.297615707f,
   -0.955141168f,  0.296150888f,
   -0.955594334f,  0.294685372f,
   -0.956045251f,  0.293219163f,
   -0.956493919f,  0.291752263f,
   -0.956940336f,  0.290284677f,
   -0.957384501f,  0.288816408f,
   -0.957826413f,  0.287347460f,
   -0.958266071f,  0.285877835f,
   -0.958703475f,  0.284407537f,
   -0.959138622f,  0.282936570f,
   -0.959571513f,  0.281464938f,
   -0.960002146f,  0.279992643f,
   -0.960430519f,  0.278519689f,
   -0.960856633f,  0.277046080f,
   -0.961280486f,  0.275571819f,
   -0.961702077f,  0.274096910f,
   -0.962121404f,  0.272621355f,
   -0.962538468f,  0.271145160f,
   -0.962953267f,  0.269668326f,
   -0.963365800f,  0.268190857f,
   -0.963776066f,  0.266712757f,
   -0.964184064f,  0.265234030f,
   -0.964589793f,  0.263754679f,
   -0.964993253f,  0.262274707f,
   -0.965394442f,  0.260794118f,
   -0.965793359f,  0.259312915f,
   -0.966190003f,  0.257831102f,
   -0.966584374f,  0.256348682f,
   -0.966976471f,  0.254865660f,
   -0.967366292f,  0.253382037f,
   -0.967753837f,  0.251897818f,
   -0.968139105f,  0.250413007f,
   -0.968522094f,  0.248927606f,
   -0.968902805f,  0.247441619f,
   -0.969281235f,  0.245955050f,
   -0.969657385f,  0.244467903f,
   -0.970031253f,  0.242980180f,
   -0.970402839f,  0.241491885f,
   -0.970772141f,  0.240003022f,
   -0.971139158f,  0.238513595f,
   -0.971503891f,  0.237023606f,
   -0.971866337f,  0.235533059f,
   -0.972226497f,  0.234041959f,
   -0.972584369f,  0.232550307f,
   -0.972939952f,  0.231058108f,
   -0.973293246f,  0.229565366f,
   -0.973644250f,  0.228072083f,
   -0.973992962f,  0.226578264f,
   -0.974339383f,  0.225083911f,
   -0.974683511f,  0.223589029f,
   -0.975025345f,  0.222093621f,
   -0.975364885f,  0.220597690f,
   -0.975702130f,  0.219101240f,
   -0.976037079f,  0.217604275f,
   -0.976369731f,  0.216106797f,
   -0.976700086f,  0.214608811f,
   -0.977028143f,  0.213110320f,
   -0.977353900f,  0.211611327f,
   -0.977677358f,  0.210111837f,
   -0.977998515f,  0.208611852f,
   -0.978317371f,  0.207111376f,
   -0.978633924f,  0.205610413f,
   -0.978948175f,  0.204108966f,
   -0.979260123f,  0.202607039f,
   -0.979569766f,  0.201104635f,
   -0.979877104f,  0.199601758f,
   -0.980182136f,  0.198098411f,
   -0.980484862f,  0.196594598f,
   -0.980785280f,  0.195090322f,
   -0.981083391f,  0.193585587f,
   -0.981379193f,  0.192080397f,
   -0.981672686f,  0.190574755f,
   -0.981963869f,  0.189068664f,
   -0.982252741f,  0.187562129f,
   -0.982539302f,  0.186055152f,
   -0.982823551f,  0.184547737f,
   -0.983105487f,  0.183039888f,
   -0.983385110f,  0.181531608f,
   -0.983662419f,  0.180022901f,
   -0.983937413f,  0.178513771f,
   -0.984210092f,  0.177004220f,
   -0.984480455f,  0.175494253f,
   -0.984748502f,  0.173983873f,
   -0.985014231f,  0.172473084f,
   -0.985277642f,  0.170961889f,
   -0.985538735f,  0.169450291f,
   -0.985797509f,  0.167938295f,
   -0.986053963f,  0.166425904f,
   -0.986308097f,  0.164913120f,
   -0.986559910f,  0.163399949f,
   -0.986809402f,  0.161886394f,
   -0.987056571f,  0.160372457f,
   -0.987301418f,  0.158858143f,
   -0.987543942f,  0.157343456f,
   -0.987784142f,  0.155828398f,
   -0.988022017f,  0.154312973f,
   -0.988257568f,  0.152797185f,
   -0.988490793f,  0.151281038f,
   -0.988721692f,  0.149764535f,
   -0.988950265f,  0.148247679f,
   -0.989176510f,  0.146730474f,
   -0.989400428f,  0.145212925f,
   -0.989622017f,  0.143695033f,
   -0.989841278f,  0.142176804f,
   -0.990058210f,  0.140658239f,
   -0.990272812f,  0.139139344f,
   -0.990485084f,  0.137620122f,
   -0.990695025f,  0.136100575f,
   -0.990902635f,  0.134580709f,
   -0.991107914f,  0.133060525f,
   -0.991310860f,  0.131540029f,
   -0.991511473f,  0.130019223f,
   -0.991709754f,  0.128498111f,
   -0.991905700f,  0.126976696f,
   -0.992099313f,  0.125454983f,
   -0.992290591f,  0.123932975f,
   -0.992479535f,  0.122410675f,
   -0.992666142f,  0.120888087f,
   -0.992850414f,  0.119365215f,
   -0.993032350f,  0.117842062f,
   -0.993211949f,  0.116318631f,
   -0.993389211f,  0.114794927f,
   -0.993564136f,  0.113270952f,
   -0.993736722f,  0.111746711f,
   -0.993906970f,  0.110222207f,
   -0.994074879f,  0.108697444f,
   -0.994240449f,  0.107172425f,
   -0.994403680f,  0.105647154f,
   -0.994564571f,  0.104121634f,
   -0.994723121f,  0.102595869f,
   -0.994879331f,  0.101069863f,
   -0.995033199f,  0.099543619f,
   -0.995184727f,  0.098017140f,
   -0.995333912f,  0.096490431f,
   -0.995480755f,  0.094963495f,
   -0.995625256f,  0.093436336f,
   -0.995767414f,  0.091908956f,
   -0.995907229f,  0.090381361f,
   -0.996044701f,  0.088853553f,
   -0.996179829f,  0.087325535f,
   -0.996312612f,  0.085797312f,
   -0.996443051f,  0.084268888f,
   -0.996571146f,  0.082740265f,
   -0.996696895f,  0.081211447f,
   -0.996820299f,  0.079682438f,
   -0.996941358f,  0.078153242f,
   -0.997060070f,  0.076623861f,
   -0.997176437f,  0.075094301f,
   -0.997290457f,  0.073564564f,
   -0.997402130f,  0.072034653f,
   -0.997511456f,  0.070504573f,
   -0.997618435f,  0.068974328f,
   -0.997723067f,  0.067443920f,
   -0.997825350f,  0.065913353f,
   -0.997925286f,  0.064382631f,
   -0.998022874f,  0.062851758f,
   -0.998118113f,  0.061320736f,
   -0.998211003f,  0.059789571f,
   -0.998301545f,  0.058258265f,
   -0.998389737f,  0.056726821f,
   -0.998475581f,  0.055195244f,
   -0.998559074f,  0.053663538f,
   -0.998640218f,  0.052131705f,
   -0.998719012f,  0.050599749f,
   -0.998795456f,  0.049067674f,
   -0.998869550f,  0.047535484f,
   -0.998941293f,  0.046003182f,
   -0.999010686f,  0.044470772f,
   -0.999077728f,  0.042938257f,
   -0.999142419f,  0.041405641f,
   -0.999204759f,  0.039872928f,
   -0.999264747f,  0.038340120f,
   -0.999322385f,  0.036807223f,
   -0.999377670f,  0.035274239f,
   -0.999430605f,  0.033741172f,
   -0.999481187f,  0.032208025f,
   -0.999529418f,  0.030674803f,
   -0.999575296f,  0.029141509f,
   -0.999618822f,  0.027608146f,
   -0.999659997f,  0.026074718f,
   -0.999698819f,  0.024541229f,
   -0.999735288f,  0.023007681f,
   -0.999769405f,  0.021474080f,
   -0.999801170f,  0.019940429f,
   -0.999830582f,  0.018406730f,
   -0.999857641f,  0.016872988f,
   -0.999882347f,  0.015339206f,
   -0.999904701f,  0.013805389f,
   -0.999924702f,  0.012271538f,
   -0.999942350f,  0.010737659f,
   -0.999957645f,  0.009203755f,
   -0.999970586f,  0.007669829f,
   -0.999981175f,  0.006135885f,
   -0.999989411f,  0.004601926f,
   -0.999995294f,  0.003067957f,
   -0.999998823f,  0.001533980f,
   -1.000000000f,  0.000000000f,
   -0.999998823f, -0.001533980f,
   -0.999995294f, -0.003067957f,
   -0.999989411f, -0.004601926f,
   -0.999981175f, -0.006135885f,
   -0.999970586f, -0.007669829f,
   -0.999957645f, -0.009203755f,
   -0.999942350f, -0.010737659f,
   -0.999924702f, -0.012271538f,
   -0.999904701f, -0.013805389f,
   -0.999882347f, -0.015339206f,
   -0.999857641f, -0.016872988f,
   -0.999830582f, -0.018406730f,
   -0.999801170f, -0.019940429f,
   -0.999769405f, -0.021474080f,
   -0.999735288f, -0.023007681f,
   -0.999698819f, -0.024541229f,
   -0.999659997f, -0.026074718f,
   -0.999618822f, -0.027608146f,
   -0.999575296f, -0.029141509f,
   -0.999529418f, -0.030674803f,
   -0.999481187f, -0.032208025f,
   -0.999430605f, -0.033741172f,
   -0.999377670f, -0.035274239f,
   -0.999322385f, -0.036807223f,
   -0.999264747f, -0.038340120f,
   -0.999204759f, -0.039872928f,
   -0.999142419f, -0.041405641f,
   -0.999077728f, -0.042938257f,
   -0.999010686f, -0.044470772f,
   -0.998941293f, -0.046003182f,
   -0.998869550f, -0.047535484f,
   -0.998795456f, -0.049067674f,
   -0.998719012f, -0.050599749f,
   -0.998640218f, -0.052131705f,
   -0.998559074f, -0.053663538f,
   -0.998475581f, -0.055195244f,
   -0.998389737f, -0.056726821f,
   -0.998301545f, -0.058258265f,
   -0.998211003f, -0.059789571f,
   -0.998118113f, -0.061320736f,
   -0.998022874f, -0.062851758f,
   -0.997925286f, -0.064382631f,
   -0.997825350f, -0.065913353f,
   -0.997723067f, -0.067443920f,
   -0.997618435f, -0.068974328f,
   -0.997511456f, -0.070504573f,
   -0.997402130f, -0.072034653f,
   -0.997290457f, -0.073564564f,
   -0.997176437f, -0.075094301f,
   -0.997060070f, -0.076623861f,
   -0.996941358f, -0.078153242f,
   -0.996820299f, -0.079682438f,
   -0.996696895f, -0.081211447f,
   -0.996571146f, -0.082740265f,
   -0.996443051f, -0.084268888f,
   -0.996312612f, -0.085797312f,
   -0.996179829f, -0.087325535f,
   -0.996044701f, -0.088853553f,
   -0.995907229f, -0.090381361f,
   -0.995767414f, -0.091908956f,
   -0.995625256f, -0.093436336f,
   -0.995480755f, -0.094963495f,
   -0.995333912f, -0.096490431f,
   -0.995184727f, -0.098017140f,
   -0.995033199f, -0.099543619f,
   -0.994879331f, -0.101069863f,
   -0.994723121f, -0.102595869f,
   -0.994564571f, -0.104121634f,
   -0.994403680f, -0.105647154f,
   -0.994240449f, -0.107172425f,
   -0.994074879f, -0.108697444f,
   -0.993906970f, -0.110222207f,
   -0.993736722f, -0.111746711f,
   -0.993564136f, -0.113270952f,
   -0.993389211f, -0.114794927f,
   -0.993211949f, -0.116318631f,
   -0.993032350f, -0.117842062f,
   -0.992850414f, -0.119365215f,
   -0.992666142f, -0.120888087f,
   -0.992479535f, -0.122410675f,
   -0.992290591f, -0.123932975f,
   -0.992099313f, -0.125454983f,
   -0.991905700f, -0.126976696f,
   -0.991709754f, -0.128498111f,
   -0.991511473f, -0.130019223f,
   -0.991310860f, -0.131540029f,
   -0.991107914f, -0.133060525f,
   -0.990902635f, -0.134580709f,
   -0.990695025f, -0.136100575f,
   -0.990485084f, -0.137620122f,
   -0.990272812f, -0.139139344f,
   -0.990058210f, -0.140658239f,
   -0.989841278f, -0.142176804f,
   -0.989622017f, -0.143695033f,
   -0.989400428f, -0.145212925f,
   -0.989176510f, -0.146730474f,
   -0.988950265f, -0.148247679f,
   -0.988721692f, -0.149764535f,
   -0.988490793f, -0.151281038f,
   -0.988257568f, -0.152797185f,
   -0.988022017f, -0.154312973f,
   -0.987784142f, -0.155828398f,
   -0.987543942f, -0.157343456f,
   -0.987301418f, -0.158858143f,
   -0.987056571f, -0.160372457f,
   -0.986809402f, -0.161886394f,
   -0.986559910f, -0.163399949f,
   -0.986308097f, -0.164913120f,
   -0.986053963f, -0.166425904f,
   -0.985797509f, -0.167938295f,
   -0.985538735f, -0.169450291f,
   -0.985277642f, -0.170961889f,
   -0.985014231f, -0.172473084f,
   -0.984748502f, -0.173983873f,
   -0.984480455f, -0.175494253f,
   -0.984210092f, -0.177004220f,
   -0.983937413f, -0.178513771f,
   -0.983662419f, -0.180022901f,
   -0.983385110f, -0.181531608f,
   -0.983105487f, -0.183039888f,
   -0.982823551f, -0.184547737f,
   -0.982539302f, -0.186055152f,
   -0.982252741f, -0.187562129f,
   -0.981963869f, -0.189068664f,
   -0.981672686f, -0.190574755f,
   -0.981379193f, -0.192080397f,
   -0.981083391f, -0.193585587f,
   -0.980785280f, -0.195090322f,
   -0.980484862f, -0.196594598f,
   -0.980182136f, -0.198098411f,
   -0.979877104f, -0.199601758f,
   -0.979569766f, -0.201104635f,
   -0.979260123f, -0.202607039f,
   -0.978948175f, -0.204108966f,
   -0.978633924f, -0.205610413f,
   -0.978317371f, -0.207111376f,
   -0.977998515f, -0.208611852f,
   -0.977677358f, -0.210111837f,
   -0.977353900f, -0.211611327f,
   -0.977028143f, -0.213110320f,
   -0.976700086f, -0.214608811f,
   -0.976369731f, -0.216106797f,
   -0.976037079f, -0.217604275f,
   -0.975702130f, -0.219101240f,
   -0.975364885f, -0.220597690f,
   -0.975025345f, -0.222093621f,
   -0.974683511f, -0.223589029f,
   -0.974339383f, -0.225083911f,
   -0.973992962f, -0.226578264f,
   -0.973644250f, -0.228072083f,
   -0.973293246f, -0.229565366f,
   -0.972939952f, -0.231058108f,
   -0.972584369f, -0.232550307f,
   -0.972226497f, -0.234041959f,
   -0.971866337f, -0.235533059f,
   -0.971503891f, -0.237023606f,
   -0.971139158f, -0.238513595f,
   -0.970772141f, -0.240003022f,
   -0.970402839f, -0.241491885f,
   -0.970031253f, -0.242980180f,
   -0.969657385f, -0.244467903f,
   -0.969281235f, -0.245955050f,
   -0.968902805f, -0.247441619f,
   -0.968522094f, -0.248927606f,
   -0.968139105f, -0.250413007f,
   -0.967753837f, -0.251897818f,
   -0.967366292f, -0.253382037f,
   -0.966976471f, -0.254865660f,
   -0.966584374f, -0.256348682f,
   -0.966190003f, -0.257831102f,
   -0.965793359f, -0.259312915f,
   -0.965394442f, -0.260794118f,
   -0.964993253f, -0.262274707f,
   -0.964589793f, -0.263754679f,
   -0.964184064f, -0.265234030f,
   -0.963776066f, -0.266712757f,
   -0.963365800f, -0.268190857f,
   -0.962953267f, -0.269668326f,
   -0.962538468f, -0.271145160f,
   -0.962121404f, -0.272621355f,
   -0.961702077f, -0.274096910f,
   -0.961280486f, -0.275571819f,
   -0.960856633f, -0.277046080f,
   -0.960430519f, -0.278519689f,
   -0.960002146f, -0.279992643f,
   -0.959571513f, -0.281464938f,
   -0.959138622f, -0.282936570f,
   -0.958703475f, -0.284407537f,
   -0.958266071f, -0.285877835f,
   -0.957826413f, -0.287347460f,
   -0.957384501f, -0.288816408f,
   -0.956940336f, -0.290284677f,
   -0.956493919f, -0.291752263f,
   -0.956045251f, -0.293219163f,
   -0.955594334f, -0.294685372f,
   -0.955141168f, -0.296150888f,
   -0.954685755f, -0.297615707f,
   -0.954228095f, -0.299079826f,
   -0.953768190f, -0.300543241f,
   -0.953306040f, -0.302005949f,
   -0.952841648f, -0.303467947f,
   -0.952375013f, -0.304929230f,
   -0.951906137f, -0.306389795f,
   -0.951435021f, -0.307849640f,
   -0.950961666f, -0.309308760f,
   -0.950486074f, -0.310767153f,
   -0.950008245f, -0.312224814f,
   -0.949528181f, -0.313681740f,
   -0.949045882f, -0.315137929f,
   -0.948561350f, -0.316593376f,
   -0.948074586f, -0.318048077f,
   -0.947585591f, -0.319502031f,
   -0.947094366f, -0.320955232f,
   -0.946600913f, -0.322407679f,
   -0.946105232f, -0.323859367f,
   -0.945607325f, -0.325310292f,
   -0.945107193f, -0.326760452f,
   -0.944604837f, -0.328209844f,
   -0.944100258f, -0.329658463f,
   -0.943593458f, -0.331106306f,
   -0.943084437f, -0.332553370f,
   -0.942573198f, -0.333999651f,
   -0.942059740f, -0.335445147f,
   -0.941544065f, -0.336889853f,
   -0.941026175f, -0.338333767f,
   -0.940506071f, -0.339776884f,
   -0.939983753f, -0.341219202f,
   -0.939459224f, -0.342660717f,
   -0.938932484f, -0.344101426f,
   -0.938403534f, -0.345541325f,
   -0.937872376f, -0.346980411f,
   -0.937339012f, -0.348418680f,
   -0.936803442f, -0.349856130f,
   -0.936265667f, -0.351292756f,
   -0.935725689f, -0.352728556f,
   -0.935183510f, -0.354163525f,
   -0.934639130f, -0.355597662f,
   -0.934092550f, -0.357030961f,
   -0.933543773f, -0.358463421f,
   -0.932992799f, -0.359895037f,
   -0.932439629f, -0.361325806f,
   -0.931884266f, -0.362755724f,
   -0.931326709f, -0.364184790f,
   -0.930766961f, -0.365612998f,
   -0.930205023f, -0.367040346f,
   -0.929640896f, -0.368466830f,
   -0.929074581f, -0.369892447f,
   -0.928506080f, -0.371317194f,
   -0.927935395f, -0.372741067f,
   -0.927362526f, -0.374164063f,
   -0.926787474f, -0.375586178f,
   -0.926210242f, -0.377007410f,
   -0.925630831f, -0.378427755f,
   -0.925049241f, -0.379847209f,
   -0.924465474f, -0.381265769f,
   -0.923879533f, -0.382683432f,
   -0.923291417f, -0.384100195f,
   -0.922701128f, -0.385516054f,
   -0.922108669f, -0.386931006f,
   -0.921514039f, -0.388345047f,
   -0.920917242f, -0.389758174f,
   -0.920318277f, -0.391170384f,
   -0.919717146f, -0.392581674f,
   -0.919113852f, -0.393992040f,
   -0.918508394f, -0.395401479f,
   -0.917900776f, -0.396809987f,
   -0.917290997f, -0.398217562f,
   -0.916679060f, -0.399624200f,
   -0.916064966f, -0.401029897f,
   -0.915448716f, -0.402434651f,
   -0.914830312f, -0.403838458f,
   -0.914209756f, -0.405241314f,
   -0.913587048f, -0.406643217f,
   -0.912962190f, -0.408044163f,
   -0.912335185f, -0.409444149f,
   -0.911706032f, -0.410843171f,
   -0.911074734f, -0.412241227f,
   -0.910441292f, -0.413638312f,
   -0.909805708f, -0.415034424f,
   -0.909167983f, -0.416429560f,
   -0.908528119f, -0.417823716f,
   -0.907886116f, -0.419216888f,
   -0.907241978f, -0.420609074f,
   -0.906595705f, -0.422000271f,
   -0.905947298f, -0.423390474f,
   -0.905296759f, -0.424779681f,
   -0.904644091f, -0.426167889f,
   -0.903989293f, -0.427555093f,
   -0.903332368f, -0.428941292f,
   -0.902673318f, -0.430326481f,
   -0.902012144f, -0.431710658f,
   -0.901348847f, -0.433093819f,
   -0.900683429f, -0.434475961f,
   -0.900015892f, -0.435857080f,
   -0.899346237f, -0.437237174f,
   -0.898674466f, -0.438616239f,
   -0.898000580f, -0.439994271f,
   -0.897324581f, -0.441371269f,
   -0.896646470f, -0.442747228f,
   -0.895966250f, -0.444122145f,
   -0.895283921f, -0.445496017f,
   -0.894599486f, -0.446868840f,
   -0.893912945f, -0.448240612f,
   -0.893224301f, -0.449611330f,
   -0.892533555f, -0.450980989f,
   -0.891840709f, -0.452349587f,
   -0.891145765f, -0.453717121f,
   -0.890448723f, -0.455083587f,
   -0.889749586f, -0.456448982f,
   -0.889048356f, -0.457813304f,
   -0.888345033f, -0.459176548f,
   -0.887639620f, -0.460538711f,
   -0.886932119f, -0.461899791f,
   -0.886222530f, -0.463259784f,
   -0.885510856f, -0.464618686f,
   -0.884797098f, -0.465976496f,
   -0.884081259f, -0.467333209f,
   -0.883363339f, -0.468688822f,
   -0.882643340f, -0.470043332f,
   -0.881921264f, -0.471396737f,
   -0.881197113f, -0.472749032f,
   -0.880470889f, -0.474100215f,
   -0.879742593f, -0.475450282f,
   -0.879012226f, -0.476799230f,
   -0.878279792f, -0.478147056f,
   -0.877545290f, -0.479493758f,
   -0.876808724f, -0.480839331f,
   -0.876070094f, -0.482183772f,
   -0.875329403f, -0.483527079f,
   -0.874586652f, -0.484869248f,
   -0.873841843f, -0.486210276f,
   -0.873094978f, -0.487550160f,
   -0.872346059f, -0.488888897f,
   -0.871595087f, -0.490226483f,
   -0.870842063f, -0.491562916f,
   -0.870086991f, -0.492898192f,
   -0.869329871f, -0.494232309f,
   -0.868570706f, -0.495565262f,
   -0.867809497f, -0.496897049f,
   -0.867046246f, -0.498227667f,
   -0.866280954f, -0.499557113f,
   -0.865513624f, -0.500885383f,
   -0.864744258f, -0.502212474f,
   -0.863972856f, -0.503538384f,
   -0.863199422f, -0.504863109f,
   -0.862423956f, -0.506186645f,
   -0.861646461f, -0.507508991f,
   -0.860866939f, -0.508830143f,
   -0.860085390f, -0.510150097f,
   -0.859301818f, -0.511468850f,
   -0.858516224f, -0.512786401f,
   -0.857728610f, -0.514102744f,
   -0.856938977f, -0.515417878f,
   -0.856147328f, -0.516731799f,
   -0.855353665f, -0.518044504f,
   -0.854557988f, -0.519355990f,
   -0.853760301f, -0.520666254f,
   -0.852960605f, -0.521975293f,
   -0.852158902f, -0.523283103f,
   -0.851355193f, -0.524589683f,
   -0.850549481f, -0.525895027f,
   -0.849741768f, -0.527199135f,
   -0.848932055f, -0.528502002f,
   -0.848120345f, -0.529803625f,
   -0.847306639f, -0.531104001f,
   -0.846490939f, -0.532403128f,
   -0.845673247f, -0.533701002f,
   -0.844853565f, -0.534997620f,
   -0.844031895f, -0.536292979f,
   -0.843208240f, -0.537587076f,
   -0.842382600f, -0.538879909f,
   -0.841554977f, -0.540171473f,
   -0.840725375f, -0.541461766f,
   -0.839893794f, -0.542750785f,
   -0.839060237f, -0.544038527f,
   -0.838224706f, -0.545324988f,
   -0.837387202f, -0.546610167f,
   -0.836547727f, -0.547894059f,
   -0.835706284f, -0.549176662f,
   -0.834862875f, -0.550457973f,
   -0.834017501f, -0.551737988f,
   -0.833170165f, -0.553016706f,
   -0.832320868f, -0.554294121f,
   -0.831469612f, -0.555570233f,
   -0.830616400f, -0.556845037f,
   -0.829761234f, -0.558118531f,
   -0.828904115f, -0.559390712f,
   -0.828045045f, -0.560661576f,
   -0.827184027f, -0.561931121f,
   -0.826321063f, -0.563199344f,
   -0.825456154f, -0.564466242f,
   -0.824589303f, -0.565731811f,
   -0.823720511f, -0.566996049f,
   -0.822849781f, -0.568258953f,
   -0.821977115f, -0.569520519f,
   -0.821102515f, -0.570780746f,
   -0.820225983f, -0.572039629f,
   -0.819347520f, -0.573297167f,
   -0.818467130f, -0.574553355f,
   -0.817584813f, -0.575808191f,
   -0.816700573f, -0.577061673f,
   -0.815814411f, -0.578313796f,
   -0.814926329f, -0.579564559f,
   -0.814036330f, -0.580813958f,
   -0.813144415f, -0.582061990f,
   -0.812250587f, -0.583308653f,
   -0.811354847f, -0.584553943f,
   -0.810457198f, -0.585797857f,
   -0.809557642f, -0.587040394f,
   -0.808656182f, -0.588281548f,
   -0.807752818f, -0.589521319f,
   -0.806847554f, -0.590759702f,
   -0.805940391f, -0.591996695f,
   -0.805031331f, -0.593232295f,
   -0.804120377f, -0.594466499f,
   -0.803207531f, -0.595699304f,
   -0.802292796f, -0.596930708f,
   -0.801376172f, -0.598160707f,
   -0.800457662f, -0.599389298f,
   -0.799537269f, -0.600616479f,
   -0.798614995f, -0.601842247f,
   -0.797690841f, -0.603066599f,
   -0.796764810f, -0.604289531f,
   -0.795836905f, -0.605511041f,
   -0.794907126f, -0.606731127f,
   -0.793975478f, -0.607949785f,
   -0.793041960f, -0.609167012f,
   -0.792106577f, -0.610382806f,
   -0.791169330f, -0.611597164f,
   -0.790230221f, -0.612810082f,
   -0.789289253f, -0.614021559f,
   -0.788346428f, -0.615231591f,
   -0.787401747f, -0.616440175f,
   -0.786455214f, -0.617647308f,
   -0.785506830f, -0.618852988f,
   -0.784556597f, -0.620057212f,
   -0.783604519f, -0.621259977f,
   -0.782650596f, -0.622461279f,
   -0.781694832f, -0.623661118f,
   -0.780737229f, -0.624859488f,
   -0.779777788f, -0.626056388f,
   -0.778816512f, -0.627251815f,
   -0.777853404f, -0.628445767f,
   -0.776888466f, -0.629638239f,
   -0.775921699f, -0.630829230f,
   -0.774953107f, -0.632018736f,
   -0.773982691f, -0.633206755f,
   -0.773010453f, -0.634393284f,
   -0.772036397f, -0.635578320f,
   -0.771060524f, -0.636761861f,
   -0.770082837f, -0.637943904f,
   -0.769103338f, -0.639124445f,
   -0.768122029f, -0.640303482f,
   -0.767138912f, -0.641481013f,
   -0.766153990f, -0.642657034f,
   -0.765167266f, -0.643831543f,
   -0.764178741f, -0.645004537f,
   -0.763188417f, -0.646176013f,
   -0.762196298f, -0.647345969f,
   -0.761202385f, -0.648514401f,
   -0.760206682f, -0.649681307f,
   -0.759209189f, -0.650846685f,
   -0.758209910f, -0.652010531f,
   -0.757208847f, -0.653172843f,
   -0.756206001f, -0.654333618f,
   -0.755201377f, -0.655492853f,
   -0.754194975f, -0.656650546f,
   -0.753186799f, -0.657806693f,
   -0.752176850f, -0.658961293f,
   -0.751165132f, -0.660114342f,
   -0.750151646f, -0.661265838f,
   -0.749136395f, -0.662415778f,
   -0.748119380f, -0.663564159f,
   -0.747100606f, -0.664710978f,
   -0.746080074f, -0.665856234f,
   -0.745057785f, -0.666999922f,
   -0.744033744f, -0.668142041f,
   -0.743007952f, -0.669282588f,
   -0.741980412f, -0.670421560f,
   -0.740951125f, -0.671558955f,
   -0.739920095f, -0.672694769f,
   -0.738887324f, -0.673829000f,
   -0.737852815f, -0.674961646f,
   -0.736816569f, -0.676092704f,
   -0.735778589f, -0.677222170f,
   -0.734738878f, -0.678350043f,
   -0.733697438f, -0.679476320f,
   -0.732654272f, -0.680600998f,
   -0.731609381f, -0.681724074f,
   -0.730562769f, -0.682845546f,
   -0.729514438f, -0.683965412f,
   -0.728464390f, -0.685083668f,
   -0.727412629f, -0.686200312f,
   -0.726359155f, -0.687315341f,
   -0.725303972f, -0.688428753f,
   -0.724247083f, -0.689540545f,
   -0.723188489f, -0.690650714f,
   -0.722128194f, -0.691759258f,
   -0.721066199f, -0.692866175f,
   -0.720002508f, -0.693971461f,
   -0.718937122f, -0.695075114f,
   -0.717870045f, -0.696177131f,
   -0.716801279f, -0.697277511f,
   -0.715730825f, -0.698376249f,
   -0.714658688f, -0.699473345f,
   -0.713584869f, -0.700568794f,
   -0.712509371f, -0.701662595f,
   -0.711432196f, -0.702754744f,
   -0.710353347f, -0.703845241f,
   -0.709272826f, -0.704934080f,
   -0.708190637f, -0.706021261f,
   -0.707106781f, -0.707106781f,
   -0.706021261f, -0.708190637f,
   -0.704934080f, -0.709272826f,
   -0.703845241f, -0.710353347f,
   -0.702754744f, -0.711432196f,
   -0.701662595f, -0.712509371f,
   -0.700568794f, -0.713584869f,
   -0.699473345f, -0.714658688f,
   -0.698376249f, -0.715730825f,
   -0.697277511f, -0.716801279f,
   -0.696177131f, -0.717870045f,
   -0.695075114f, -0.718937122f,
   -0.693971461f, -0.720002508f,
   -0.692866175f, -0.721066199f,
   -0.691759258f, -0.722128194f,
   -0.690650714f, -0.723188489f,
   -0.689540545f, -0.724247083f,
   -0.688428753f, -0.725303972f,
   -0.687315341f, -0.726359155f,
   -0.686200312f, -0.727412629f,
   -0.685083668f, -0.728464390f,
   -0.683965412f, -0.729514438f,
   -0.682845546f, -0.730562769f,
   -0.681724074f, -0.731609381f,
   -0.680600998f, -0.732654272f,
   -0.679476320f, -0.733697438f,
   -0.678350043f, -0.734738878f,
   -0.677222170f, -0.735778589f,
   -0.676092704f, -0.736816569f,
   -0.674961646f, -0.737852815f,
   -0.673829000f, -0.738887324f,
   -0.672694769f, -0.739920095f,
   -0.671558955f, -0.740951125f,
   -0.670421560f, -0.741980412f,
   -0.669282588f, -0.743007952f,
   -0.668142041f, -0.744033744f,
   -0.666999922f, -0.745057785f,
   -0.665856234f, -0.746080074f,
   -0.664710978f, -0.747100606f,
   -0.663564159f, -0.748119380f,
   -0.662415778f, -0.749136395f,
   -0.661265838f, -0.750151646f,
   -0.660114342f, -0.751165132f,
   -0.658961293f, -0.752176850f,
   -0.657806693f, -0.753186799f,
   -0.656650546f, -0.754194975f,
   -0.655492853f, -0.755201377f,
   -0.654333618f, -0.756206001f,
   -0.653172843f, -0.757208847f,
   -0.652010531f, -0.758209910f,
   -0.650846685f, -0.759209189f,
   -0.649681307f, -0.760206682f,
   -0.648514401f, -0.761202385f,
   -0.647345969f, -0.762196298f,
   -0.646176013f, -0.763188417f,
   -0.645004537f, -0.764178741f,
   -0.643831543f, -0.765167266f,
   -0.642657034f, -0.766153990f,
   -0.641481013f, -0.767138912f,
   -0.640303482f, -0.768122029f,
   -0.639124445f, -0.769103338f,
   -0.637943904f, -0.770082837f,
   -0.636761861f, -0.771060524f,
   -0.635578320f, -0.772036397f,
   -0.634393284f, -0.773010453f,
   -0.633206755f, -0.773982691f,
   -0.632018736f, -0.774953107f,
   -0.630829230f, -0.775921699f,
   -0.629638239f, -0.776888466f,
   -0.628445767f, -0.777853404f,
   -0.627251815f, -0.778816512f,
   -0.626056388f, -0.779777788f,
   -0.624859488f, -0.780737229f,
   -0.623661118f, -0.781694832f,
   -0.622461279f, -0.782650596f,
   -0.621259977f, -0.783604519f,
   -0.620057212f, -0.784556597f,
   -0.618852988f, -0.785506830f,
   -0.617647308f, -0.786455214f,
   -0.616440175f, -0.787401747f,
   -0.615231591f, -0.788346428f,
   -0.614021559f, -0.789289253f,
   -0.612810082f, -0.790230221f,
   -0.611597164f, -0.791169330f,
   -0.610382806f, -0.792106577f,
   -0.609167012f, -0.793041960f,
   -0.607949785f, -0.793975478f,
   -0.606731127f, -0.794907126f,
   -0.605511041f, -0.795836905f,
   -0.604289531f, -0.796764810f,
   -0.603066599f, -0.797690841f,
   -0.601842247f, -0.798614995f,
   -0.600616479f, -0.799537269f,
   -0.599389298f, -0.800457662f,
   -0.598160707f, -0.801376172f,
   -0.596930708f, -0.802292796f,
   -0.595699304f, -0.803207531f,
   -0.594466499f, -0.804120377f,
   -0.593232295f, -0.805031331f,
   -0.591996695f, -0.805940391f,
   -0.590759702f, -0.806847554f,
   -0.589521319f, -0.807752818f,
   -0.588281548f, -0.808656182f,
   -0.587040394f, -0.809557642f,
   -0.585797857f, -0.810457198f,
   -0.584553943f, -0.811354847f,
   -0.583308653f, -0.812250587f,
   -0.582061990f, -0.813144415f,
   -0.580813958f, -0.814036330f,
   -0.579564559f, -0.814926329f,
   -0.578313796f, -0.815814411f,
   -0.577061673f, -0.816700573f,
   -0.575808191f, -0.817584813f,
   -0.574553355f, -0.818467130f,
   -0.573297167f, -0.819347520f,
   -0.572039629f, -0.820225983f,
   -0.570780746f, -0.821102515f,
   -0.569520519f, -0.821977115f,
   -0.568258953f, -0.822849781f,
   -0.566996049f, -0.823720511f,
   -0.565731811f, -0.824589303f,
   -0.564466242f, -0.825456154f,
   -0.563199344f, -0.826321063f,
   -0.561931121f, -0.827184027f,
   -0.560661576f, -0.828045045f,
   -0.559390712f, -0.828904115f,
   -0.558118531f, -0.829761234f,
   -0.556845037f, -0.830616400f,
   -0.555570233f, -0.831469612f,
   -0.554294121f, -0.832320868f,
   -0.553016706f, -0.833170165f,
   -0.551737988f, -0.834017501f,
   -0.550457973f, -0.834862875f,
   -0.549176662f, -0.835706284f,
   -0.547894059f, -0.836547727f,
   -0.546610167f, -0.837387202f,
   -0.545324988f, -0.838224706f,
   -0.544038527f, -0.839060237f,
   -0.542750785f, -0.839893794f,
   -0.541461766f, -0.840725375f,
   -0.540171473f, -0.841554977f,
   -0.538879909f, -0.842382600f,
   -0.537587076f, -0.843208240f,
   -0.536292979f, -0.844031895f,
   -0.534997620f, -0.844853565f,
   -0.533701002f, -0.845673247f,
   -0.532403128f, -0.846490939f,
   -0.531104001f, -0.847306639f,
   -0.529803625f, -0.848120345f,
   -0.528502002f, -0.848932055f,
   -0.527199135f, -0.849741768f,
   -0.525895027f, -0.850549481f,
   -0.524589683f, -0.851355193f,
   -0.523283103f, -0.852158902f,
   -0.521975293f, -0.852960605f,
   -0.520666254f, -0.853760301f,
   -0.519355990f, -0.854557988f,
   -0.518044504f, -0.855353665f,
   -0.516731799f, -0.856147328f,
   -0.515417878f, -0.856938977f,
   -0.514102744f, -0.857728610f,
   -0.512786401f, -0.858516224f,
   -0.511468850f, -0.859301818f,
   -0.510150097f, -0.860085390f,
   -0.508830143f, -0.860866939f,
   -0.507508991f, -0.861646461f,
   -0.506186645f, -0.862423956f,
   -0.504863109f, -0.863199422f,
   -0.503538384f, -0.863972856f,
   -0.502212474f, -0.864744258f,
   -0.500885383f, -0.865513624f,
   -0.499557113f, -0.866280954f,
   -0.498227667f, -0.867046246f,
   -0.496897049f, -0.867809497f,
   -0.495565262f, -0.868570706f,
   -0.494232309f, -0.869329871f,
   -0.492898192f, -0.870086991f,
   -0.491562916f, -0.870842063f,
   -0.490226483f, -0.871595087f,
   -0.488888897f, -0.872346059f,
   -0.487550160f, -0.873094978f,
   -0.486210276f, -0.873841843f,
   -0.484869248f, -0.874586652f,
   -0.483527079f, -0.875329403f,
   -0.482183772f, -0.876070094f,
   -0.480839331f, -0.876808724f,
   -0.479493758f, -0.877545290f,
   -0.478147056f, -0.878279792f,
   -0.476799230f, -0.879012226f,
   -0.475450282f, -0.879742593f,
   -0.474100215f, -0.880470889f,
   -0.472749032f, -0.881197113f,
   -0.471396737f, -0.881921264f,
   -0.470043332f, -0.882643340f,
   -0.468688822f, -0.883363339f,
   -0.467333209f, -0.884081259f,
   -0.465976496f, -0.884797098f,
   -0.464618686f, -0.885510856f,
   -0.463259784f, -0.886222530f,
   -0.461899791f, -0.886932119f,
   -0.460538711f, -0.887639620f,
   -0.459176548f, -0.888345033f,
   -0.457813304f, -0.889048356f,
   -0.456448982f, -0.889749586f,
   -0.455083587f, -0.890448723f,
   -0.453717121f, -0.891145765f,
   -0.452349587f, -0.891840709f,
   -0.450980989f, -0.892533555f,
   -0.449611330f, -0.893224301f,
   -0.448240612f, -0.893912945f,
   -0.446868840f, -0.894599486f,
   -0.445496017f, -0.895283921f,
   -0.444122145f, -0.895966250f,
   -0.442747228f, -0.896646470f,
   -0.441371269f, -0.897324581f,
   -0.439994271f, -0.898000580f,
   -0.438616239f, -0.898674466f,
   -0.437237174f, -0.899346237f,
   -0.435857080f, -0.900015892f,
   -0.434475961f, -0.900683429f,
   -0.433093819f, -0.901348847f,
   -0.431710658f, -0.902012144f,
   -0.430326481f, -0.902673318f,
   -0.428941292f, -0.903332368f,
   -0.427555093f, -0.903989293f,
   -0.426167889f, -0.904644091f,
   -0.424779681f, -0.905296759f,
   -0.423390474f, -0.905947298f,
   -0.422000271f, -0.906595705f,
   -0.420609074f, -0.907241978f,
   -0.419216888f, -0.907886116f,
   -0.417823716f, -0.908528119f,
   -0.416429560f, -0.909167983f,
   -0.415034424f, -0.909805708f,
   -0.413638312f, -0.910441292f,
   -0.412241227f, -0.911074734f,
   -0.410843171f, -0.911706032f,
   -0.409444149f, -0.912335185f,
   -0.408044163f, -0.912962190f,
   -0.406643217f, -0.913587048f,
   -0.405241314f, -0.914209756f,
   -0.403838458f, -0.914830312f,
   -0.402434651f, -0.915448716f,
   -0.401029897f, -0.916064966f,
   -0.399624200f, -0.916679060f,
   -0.398217562f, -0.917290997f,
   -0.396809987f, -0.917900776f,
   -0.395401479f, -0.918508394f,
   -0.393992040f, -0.919113852f,
   -0.392581674f, -0.919717146f,
   -0.391170384f, -0.920318277f,
   -0.389758174f, -0.920917242f,
   -0.388345047f, -0.921514039f,
   -0.386931006f, -0.922108669f,
   -0.385516054f, -0.922701128f,
   -0.384100195f, -0.923291417f,
   -0.382683432f, -0.923879533f,
   -0.381265769f, -0.924465474f,
   -0.379847209f, -0.925049241f,
   -0.378427755f, -0.925630831f,
   -0.377007410f, -0.926210242f,
   -0.375586178f, -0.926787474f,
   -0.374164063f, -0.927362526f,
   -0.372741067f, -0.927935395f,
   -0.371317194f, -0.928506080f,
   -0.369892447f, -0.929074581f,
   -0.368466830f, -0.929640896f,
   -0.367040346f, -0.930205023f,
   -0.365612998f, -0.930766961f,
   -0.364184790f, -0.931326709f,
   -0.362755724f, -0.931884266f,
   -0.361325806f, -0.932439629f,
   -0.359895037f, -0.932992799f,
   -0.358463421f, -0.933543773f,
   -0.357030961f, -0.934092550f,
   -0.355597662f, -0.934639130f,
   -0.354163525f, -0.935183510f,
   -0.352728556f, -0.935725689f,
   -0.351292756f, -0.936265667f,
   -0.349856130f, -0.936803442f,
   -0.348418680f, -0.937339012f,
   -0.346980411f, -0.937872376f,
   -0.345541325f, -0.938403534f,
   -0.344101426f, -0.938932484f,
   -0.342660717f, -0.939459224f,
   -0.341219202f, -0.939983753f,
   -0.339776884f, -0.940506071f,
   -0.338333767f, -0.941026175f,
   -0.336889853f, -0.941544065f,
   -0.335445147f, -0.942059740f,
   -0.333999651f, -0.942573198f,
   -0.332553370f, -0.943084437f,
   -0.331106306f, -0.943593458f,
   -0.329658463f, -0.944100258f,
   -0.328209844f, -0.944604837f,
   -0.326760452f, -0.945107193f,
   -0.325310292f, -0.945607325f,
   -0.323859367f, -0.946105232f,
   -0.322407679f, -0.946600913f,
   -0.320955232f, -0.947094366f,
   -0.319502031f, -0.947585591f,
   -0.318048077f, -0.948074586f,
   -0.316593376f, -0.948561350f,
   -0.315137929f, -0.949045882f,
   -0.313681740f, -0.949528181f,
   -0.312224814f, -0.950008245f,
   -0.310767153f, -0.950486074f,
   -0.309308760f, -0.950961666f,
   -0.307849640f, -0.951435021f,
   -0.306389795f, -0.951906137f,
   -0.304929230f, -0.952375013f,
   -0.303467947f, -0.952841648f,
   -0.302005949f, -0.953306040f,
   -0.300543241f, -0.953768190f,
   -0.299079826f, -0.954228095f,
   -0.297615707f, -0.954685755f,
   -0.296150888f, -0.955141168f,
   -0.294685372f, -0.955594334f,
   -0.293219163f, -0.956045251f,
   -0.291752263f, -0.956493919f,
   -0.290284677f, -0.956940336f,
   -0.288816408f, -0.957384501f,
   -0.287347460f, -0.957826413f,
   -0.285877835f, -0.958266071f,
   -0.284407537f, -0.958703475f,
   -0.282936570f, -0.959138622f,
   -0.281464938f, -0.959571513f,
   -0.279992643f, -0.960002146f,
   -0.278519689f, -0.960430519f,
   -0.277046080f, -0.960856633f,
   -0.275571819f, -0.961280486f,
   -0.274096910f, -0.961702077f,
   -0.272621355f, -0.962121404f,
   -0.271145160f, -0.962538468f,
   -0.269668326f, -0.962953267f,
   -0.268190857f, -0.963365800f,
   -0.266712757f, -0.963776066f,
   -0.265234030f, -0.964184064f,
   -0.263754679f, -0.964589793f,
   -0.262274707f, -0.964993253f,
   -0.260794118f, -0.965394442f,
   -0.259312915f, -0.965793359f,
   -0.257831102f, -0.966190003f,
   -0.256348682f, -0.966584374f,
   -0.254865660f, -0.966976471f,
   -0.253382037f, -0.967366292f,
   -0.251897818f, -0.967753837f,
   -0.250413007f, -0.968139105f,
   -0.248927606f, -0.968522094f,
   -0.247441619f, -0.968902805f,
   -0.245955050f, -0.969281235f,
   -0.244467903f, -0.969657385f,
   -0.242980180f, -0.970031253f,
   -0.241491885f, -0.970402839f,
   -0.240003022f, -0.970772141f,
   -0.238513595f, -0.971139158f,
   -0.237023606f, -0.971503891f,
   -0.235533059f, -0.971866337f,
   -0.234041959f, -0.972226497f,
   -0.232550307f, -0.972584369f,
   -0.231058108f, -0.972939952f,
   -0.229565366f, -0.973293246f,
   -0.228072083f, -0.973644250f,
   -0.226578264f, -0.973992962f,
   -0.225083911f, -0.974339383f,
   -0.223589029f, -0.974683511f,
   -0.222093621f, -0.975025345f,
   -0.220597690f, -0.975364885f,
   -0.219101240f, -0.975702130f,
   -0.217604275f, -0.976037079f,
   -0.216106797f, -0.976369731f,
   -0.214608811f, -0.976700086f,
   -0.213110320f, -0.977028143f,
   -0.211611327f, -0.977353900f,
   -0.210111837f, -0.977677358f,
   -0.208611852f, -0.977998515f,
   -0.207111376f, -0.978317371f,
   -0.205610413f, -0.978633924f,
   -0.204108966f, -0.978948175f,
   -0.202607039f, -0.979260123f,
   -0.201104635f, -0.979569766f,
   -0.199601758f, -0.979877104f,
   -0.198098411f, -0.980182136f,
   -0.196594598f, -0.980484862f,
   -0.195090322f, -0.980785280f,
   -0.193585587f, -0.981083391f,
   -0.192080397f, -0.981379193f,
   -0.190574755f, -0.981672686f,
   -0.189068664f, -0.981963869f,
   -0.187562129f, -0.982252741f,
   -0.186055152f, -0.982539302f,
   -0.184547737f, -0.982823551f,
   -0.183039888f, -0.983105487f,
   -0.181531608f, -0.983385110f,
   -0.180022901f, -0.983662419f,
   -0.178513771f, -0.983937413f,
   -0.177004220f, -0.984210092f,
   -0.175494253f, -0.984480455f,
   -0.173983873f, -0.984748502f,
   -0.172473084f, -0.985014231f,
   -0.170961889f, -0.985277642f,
   -0.169450291f, -0.985538735f,
   -0.167938295f, -0.985797509f,
   -0.166425904f, -0.986053963f,
   -0.164913120f, -0.986308097f,
   -0.163399949f, -0.986559910f,
   -0.161886394f, -0.986809402f,
   -0.160372457f, -0.987056571f,
   -0.158858143f, -0.987301418f,
   -0.157343456f, -0.987543942f,
   -0.155828398f, -0.987784142f,
   -0.154312973f, -0.988022017f,
   -0.152797185f, -0.988257568f,
   -0.151281038f, -0.988490793f,
   -0.149764535f, -0.988721692f,
   -0.148247679f, -0.988950265f,
   -0.146730474f, -0.989176510f,
   -0.145212925f, -0.989400428f,
   -0.143695033f, -0.989622017f,
   -0.142176804f, -0.989841278f,
   -0.140658239f, -0.990058210f,
   -0.139139344f, -0.990272812f,
   -0.137620122f, -0.990485084f,
   -0.136100575f, -0.990695025f,
   -0.134580709f, -0.990902635f,
   -0.133060525f, -0.991107914f,
   -0.131540029f, -0.991310860f,
   -0.130019223f, -0.991511473f,
   -0.128498111f, -0.991709754f,
   -0.126976696f, -0.991905700f,
   -0.125454983f, -0.992099313f,
   -0.123932975f, -0.992290591f,
   -0.122410675f, -0.992479535f,
   -0.120888087f, -0.992666142f,
   -0.119365215f, -0.992850414f,
   -0.117842062f, -0.993032350f,
   -0.116318631f, -0.993211949f,
   -0.114794927f, -0.993389211f,
   -0.113270952f, -0.993564136f,
   -0.111746711f, -0.993736722f,
   -0.110222207f, -0.993906970f,
   -0.108697444f, -0.994074879f,
   -0.107172425f, -0.994240449f,
   -0.105647154f, -0.994403680f,
   -0.104121634f, -0.994564571f,
   -0.102595869f, -0.994723121f,
   -0.101069863f, -0.994879331f,
   -0.099543619f, -0.995033199f,
   -0.098017140f, -0.995184727f,
   -0.096490431f, -0.995333912f,
   -0.094963495f, -0.995480755f,
   -0.093436336f, -0.995625256f,
   -0.091908956f, -0.995767414f,
   -0.090381361f, -0.995907229f,
   -0.088853553f, -0.996044701f,
   -0.087325535f, -0.996179829f,
   -0.085797312f, -0.996312612f,
   -0.084268888f, -0.996443051f,
   -0.082740265f, -0.996571146f,
   -0.081211447f, -0.996696895f,
   -0.079682438f, -0.996820299f,
   -0.078153242f, -0.996941358f,
   -0.076623861f, -0.997060070f,
   -0.075094301f, -0.997176437f,
   -0.073564564f, -0.997290457f,
   -0.072034653f, -0.997402130f,
   -0.070504573f, -0.997511456f,
   -0.068974328f, -0.997618435f,
   -0.067443920f, -0.997723067f,
   -0.065913353f, -0.997825350f,
   -0.064382631f, -0.997925286f,
   -0.062851758f, -0.998022874f,
   -0.061320736f, -0.998118113f,
   -0.059789571f, -0.998211003f,
   -0.058258265f, -0.998301545f,
   -0.056726821f, -0.998389737f,
   -0.055195244f, -0.998475581f,
   -0.053663538f, -0.998559074f,
   -0.052131705f, -0.998640218f,
   -0.050599749f, -0.998719012f,
   -0.049067674f, -0.998795456f,
   -0.047535484f, -0.998869550f,
   -0.046003182f, -0.998941293f,
   -0.044470772f, -0.999010686f,
   -0.042938257f, -0.999077728f,
   -0.041405641f, -0.999142419f,
   -0.039872928f, -0.999204759f,
   -0.038340120f, -0.999264747f,
   -0.036807223f, -0.999322385f,
   -0.035274239f, -0.999377670f,
   -0.033741172f, -0.999430605f,
   -0.032208025f, -0.999481187f,
   -0.030674803f, -0.999529418f,
   -0.029141509f, -0.999575296f,
   -0.027608146f, -0.999618822f,
   -0.026074718f, -0.999659997f,
   -0.024541229f, -0.999698819f,
   -0.023007681f, -0.999735288f,
   -0.021474080f, -0.999769405f,
   -0.019940429f, -0.999801170f,
   -0.018406730f, -0.999830582f,
   -0.016872988f, -0.999857641f,
   -0.015339206f, -0.999882347f,
   -0.013805389f, -0.999904701f,
   -0.012271538f, -0.999924702f,
   -0.010737659f, -0.999942350f,
   -0.009203755f, -0.999957645f,
   -0.007669829f, -0.999970586f,
   -0.006135885f, -0.999981175f,
   -0.004601926f, -0.999989411f,
   -0.003067957f, -0.999995294f,
   -0.001533980f, -0.999998823f,
   -0.000000000f, -1.000000000f,
    0.001533980f, -0.999998823f,
    0.003067957f, -0.999995294f,
    0.004601926f, -0.999989411f,
    0.006135885f, -0.999981175f,
    0.007669829f, -0.999970586f,
    0.009203755f, -0.999957645f,
    0.010737659f, -0.999942350f,
    0.012271538f, -0.999924702f,
    0.013805389f, -0.999904701f,
    0.015339206f, -0.999882347f,
    0.016872988f, -0.999857641f,
    0.018406730f, -0.999830582f,
    0.019940429f, -0.999801170f,
    0.021474080f, -0.999769405f,
    0.023007681f, -0.999735288f,
    0.024541229f, -0.999698819f,
    0.026074718f, -0.999659997f,
    0.027608146f, -0.999618822f,
    0.029141509f, -0.999575296f,
    0.030674803f, -0.999529418f,
    0.032208025f, -0.999481187f,
    0.033741172f, -0.999430605f,
    0.035274239f, -0.999377670f,
    0.036807223f, -0.999322385f,
    0.038340120f, -0.999264747f,
    0.039872928f, -0.999204759f,
    0.041405641f, -0.999142419f,
    0.042938257f, -0.999077728f,
    0.044470772f, -0.999010686f,
    0.046003182f, -0.998941293f,
    0.047535484f, -0.998869550f,
    0.049067674f, -0.998795456f,
    0.050599749f, -0.998719012f,
    0.052131705f, -0.998640218f,
    0.053663538f, -0.998559074f,
    0.055195244f, -0.998475581f,
    0.056726821f, -0.998389737f,
    0.058258265f, -0.998301545f,
    0.059789571f, -0.998211003f,
    0.061320736f, -0.998118113f,
    0.062851758f, -0.998022874f,
    0.064382631f, -0.997925286f,
    0.065913353f, -0.997825350f,
    0.067443920f, -0.997723067f,
    0.068974328f, -0.997618435f,
    0.070504573f, -0.997511456f,
    0.072034653f, -0.997402130f,
    0.073564564f, -0.997290457f,
    0.075094301f, -0.997176437f,
    0.076623861f, -0.997060070f,
    0.078153242f, -0.996941358f,
    0.079682438f, -0.996820299f,
    0.081211447f, -0.996696895f,
    0.082740265f, -0.996571146f,
    0.084268888f, -0.996443051f,
    0.085797312f, -0.996312612f,
    0.087325535f, -0.996179829f,
    0.088853553f, -0.996044701f,
    0.090381361f, -0.995907229f,
    0.091908956f, -0.995767414f,
    0.093436336f, -0.995625256f,
    0.094963495f, -0.995480755f,
    0.096490431f, -0.995333912f,
    0.098017140f, -0.995184727f,
    0.099543619f, -0.995033199f,
    0.101069863f, -0.994879331f,
    0.102595869f, -0.994723121f,
    0.104121634f, -0.994564571f,
    0.105647154f, -0.994403680f,
    0.107172425f, -0.994240449f,
    0.108697444f, -0.994074879f,
    0.110222207f, -0.993906970f,
    0.111746711f, -0.993736722f,
    0.113270952f, -0.993564136f,
    0.114794927f, -0.993389211f,
    0.116318631f, -0.993211949f,
    0.117842062f, -0.993032350f,
    0.119365215f, -0.992850414f,
    0.120888087f, -0.992666142f,
    0.122410675f, -0.992479535f,
    0.123932975f, -0.992290591f,
    0.125454983f, -0.992099313f,
    0.126976696f, -0.991905700f,
    0.128498111f, -0.991709754f,
    0.130019223f, -0.991511473f,
    0.131540029f, -0.991310860f,
    0.133060525f, -0.991107914f,
    0.134580709f, -0.990902635f,
    0.136100575f, -0.990695025f,
    0.137620122f, -0.990485084f,
    0.139139344f, -0.990272812f,
    0.140658239f, -0.990058210f,
    0.142176804f, -0.989841278f,
    0.143695033f, -0.989622017f,
    0.145212925f, -0.989400428f,
    0.146730474f, -0.989176510f,
    0.148247679f, -0.988950265f,
    0.149764535f, -0.988721692f,
    0.151281038f, -0.988490793f,
    0.152797185f, -0.988257568f,
    0.154312973f, -0.988022017f,
    0.155828398f, -0.987784142f,
    0.157343456f, -0.987543942f,
    0.158858143f, -0.987301418f,
    0.160372457f, -0.987056571f,
    0.161886394f, -0.986809402f,
    0.163399949f, -0.986559910f,
    0.164913120f, -0.986308097f,
    0.166425904f, -0.986053963f,
    0.167938295f, -0.985797509f,
    0.169450291f, -0.985538735f,
    0.170961889f, -0.985277642f,
    0.172473084f, -0.985014231f,
    0.173983873f, -0.984748502f,
    0.175494253f, -0.984480455f,
    0.177004220f, -0.984210092f,
    0.178513771f, -0.983937413f,
    0.180022901f, -0.983662419f,
    0.181531608f, -0.983385110f,
    0.183039888f, -0.983105487f,
    0.184547737f, -0.982823551f,
    0.186055152f, -0.982539302f,
    0.187562129f, -0.982252741f,
    0.189068664f, -0.981963869f,
    0.190574755f, -0.981672686f,
    0.192080397f, -0.981379193f,
    0.193585587f, -0.981083391f,
    0.195090322f, -0.980785280f,
    0.196594598f, -0.980484862f,
    0.198098411f, -0.980182136f,
    0.199601758f, -0.979877104f,
    0.201104635f, -0.979569766f,
    0.202607039f, -0.979260123f,
    0.204108966f, -0.978948175f,
    0.205610413f, -0.978633924f,
    0.207111376f, -0.978317371f,
    0.208611852f, -0.977998515f,
    0.210111837f, -0.977677358f,
    0.211611327f, -0.977353900f,
    0.213110320f, -0.977028143f,
    0.214608811f, -0.976700086f,
    0.216106797f, -0.976369731f,
    0.217604275f, -0.976037079f,
    0.219101240f, -0.975702130f,
    0.220597690f, -0.975364885f,
    0.222093621f, -0.975025345f,
    0.223589029f, -0.974683511f,
    0.225083911f, -0.974339383f,
    0.226578264f, -0.973992962f,
    0.228072083f, -0.973644250f,
    0.229565366f, -0.973293246f,
    0.231058108f, -0.972939952f,
    0.232550307f, -0.972584369f,
    0.234041959f, -0.972226497f,
    0.235533059f, -0.971866337f,
    0.237023606f, -0.971503891f,
    0.238513595f, -0.971139158f,
    0.240003022f, -0.970772141f,
    0.241491885f, -0.970402839f,
    0.242980180f, -0.970031253f,
    0.244467903f, -0.969657385f,
    0.245955050f, -0.969281235f,
    0.247441619f, -0.968902805f,
    0.248927606f, -0.968522094f,
    0.250413007f, -0.968139105f,
    0.251897818f, -0.967753837f,
    0.253382037f, -0.967366292f,
    0.254865660f, -0.966976471f,
    0.256348682f, -0.966584374f,
    0.257831102f, -0.966190003f,
    0.259312915f, -0.965793359f,
    0.260794118f, -0.965394442f,
    0.262274707f, -0.964993253f,
    0.263754679f, -0.964589793f,
    0.265234030f, -0.964184064f,
    0.266712757f, -0.963776066f,
    0.268190857f, -0.963365800f,
    0.269668326f, -0.962953267f,
    0.271145160f, -0.962538468f,
    0.272621355f, -0.962121404f,
    0.274096910f, -0.961702077f,
    0.275571819f, -0.961280486f,
    0.277046080f, -0.960856633f,
    0.278519689f, -0.960430519f,
    0.279992643f, -0.960002146f,
    0.281464938f, -0.959571513f,
    0.282936570f, -0.959138622f,
    0.284407537f, -0.958703475f,
    0.285877835f, -0.958266071f,
    0.287347460f, -0.957826413f,
    0.288816408f, -0.957384501f,
    0.290284677f, -0.956940336f,
    0.291752263f, -0.956493919f,
    0.293219163f, -0.956045251f,
    0.294685372f, -0.955594334f,
    0.296150888f, -0.955141168f,
    0.297615707f, -0.954685755f,
    0.299079826f, -0.954228095f,
    0.300543241f, -0.953768190f,
    0.302005949f, -0.953306040f,
    0.303467947f, -0.952841648f,
    0.304929230f, -0.952375013f,
    0.306389795f, -0.951906137f,
    0.307849640f, -0.951435021f,
    0.309308760f, -0.950961666f,
    0.310767153f, -0.950486074f,
    0.312224814f, -0.950008245f,
    0.313681740f, -0.949528181f,
    0.315137929f, -0.949045882f,
    0.316593376f, -0.948561350f,
    0.318048077f, -0.948074586f,
    0.319502031f, -0.947585591f,
    0.320955232f, -0.947094366f,
    0.322407679f, -0.946600913f,
    0.323859367f, -0.946105232f,
    0.325310292f, -0.945607325f,
    0.326760452f, -0.945107193f,
    0.328209844f, -0.944604837f,
    0.329658463f, -0.944100258f,
    0.331106306f, -0.943593458f,
    0.332553370f, -0.943084437f,
    0.333999651f, -0.942573198f,
    0.335445147f, -0.942059740f,
    0.336889853f, -0.941544065f,
    0.338333767f, -0.941026175f,
    0.339776884f, -0.940506071f,
    0.341219202f, -0.939983753f,
    0.342660717f, -0.939459224f,
    0.344101426f, -0.938932484f,
    0.345541325f, -0.938403534f,
    0.346980411f, -0.937872376f,
    0.348418680f, -0.937339012f,
    0.349856130f, -0.936803442f,
    0.351292756f, -0.936265667f,
    0.352728556f, -0.935725689f,
    0.354163525f, -0.935183510f,
    0.355597662f, -0.934639130f,
    0.357030961f, -0.934092550f,
    0.358463421f, -0.933543773f,
    0.359895037f, -0.932992799f,
    0.361325806f, -0.932439629f,
    0.362755724f, -0.931884266f,
    0.364184790f, -0.931326709f,
    0.365612998f, -0.930766961f,
    0.367040346f, -0.930205023f,
    0.368466830f, -0.929640896f,
    0.369892447f, -0.929074581f,
    0.371317194f, -0.928506080f,
    0.372741067f, -0.927935395f,
    0.374164063f, -0.927362526f,
    0.375586178f, -0.926787474f,
    0.377007410f, -0.926210242f,
    0.378427755f, -0.925630831f,
    0.379847209f, -0.925049241f,
    0.381265769f, -0.924465474f,
    0.382683432f, -0.923879533f,
    0.384100195f, -0.923291417f,
    0.385516054f, -0.922701128f,
    0.386931006f, -0.922108669f,
    0.388345047f, -0.921514039f,
    0.389758174f, -0.920917242f,
    0.391170384f, -0.920318277f,
    0.392581674f, -0.919717146f,
    0.393992040f, -0.919113852f,
    0.395401479f, -0.918508394f,
    0.396809987f, -0.917900776f,
    0.398217562f, -0.917290997f,
    0.399624200f, -0.916679060f,
    0.401029897f, -0.916064966f,
    0.402434651f, -0.915448716f,
    0.403838458f, -0.914830312f,
    0.405241314f, -0.914209756f,
    0.406643217f, -0.913587048f,
    0.408044163f, -0.912962190f,
    0.409444149f, -0.912335185f,
    0.410843171f, -0.911706032f,
    0.412241227f, -0.911074734f,
    0.413638312f, -0.910441292f,
    0.415034424f, -0.909805708f,
    0.416429560f, -0.909167983f,
    0.417823716f, -0.908528119f,
    0.419216888f, -0.907886116f,
    0.420609074f, -0.907241978f,
    0.422000271f, -0.906595705f,
    0.423390474f, -0.905947298f,
    0.424779681f, -0.905296759f,
    0.426167889f, -0.904644091f,
    0.427555093f, -0.903989293f,
    0.428941292f, -0.903332368f,
    0.430326481f, -0.902673318f,
    0.431710658f, -0.902012144f,
    0.433093819f, -0.901348847f,
    0.434475961f, -0.900683429f,
    0.435857080f, -0.900015892f,
    0.437237174f, -0.899346237f,
    0.438616239f, -0.898674466f,
    0.439994271f, -0.898000580f,
    0.441371269f, -0.897324581f,
    0.442747228f, -0.896646470f,
    0.444122145f, -0.895966250f,
    0.445496017f, -0.895283921f,
    0.446868840f, -0.894599486f,
    0.448240612f, -0.893912945f,
    0.449611330f, -0.893224301f,
    0.450980989f, -0.892533555f,
    0.452349587f, -0.891840709f,
    0.453717121f, -0.891145765f,
    0.455083587f, -0.890448723f,
    0.456448982f, -0.889749586f,
    0.457813304f, -0.889048356f,
    0.459176548f, -0.888345033f,
    0.460538711f, -0.887639620f,
    0.461899791f, -0.886932119f,
    0.463259784f, -0.886222530f,
    0.464618686f, -0.885510856f,
    0.465976496f, -0.884797098f,
    0.467333209f, -0.884081259f,
    0.468688822f, -0.883363339f,
    0.470043332f, -0.882643340f,
    0.471396737f, -0.881921264f,
    0.472749032f, -0.881197113f,
    0.474100215f, -0.880470889f,
    0.475450282f, -0.879742593f,
    0.476799230f, -0.879012226f,
    0.478147056f, -0.878279792f,
    0.479493758f, -0.877545290f,
    0.480839331f, -0.876808724f,
    0.482183772f, -0.876070094f,
    0.483527079f, -0.875329403f,
    0.484869248f, -0.874586652f,
    0.486210276f, -0.873841843f,
    0.487550160f, -0.873094978f,
    0.488888897f, -0.872346059f,
    0.490226483f, -0.871595087f,
    0.491562916f, -0.870842063f,
    0.492898192f, -0.870086991f,
    0.494232309f, -0.869329871f,
    0.495565262f, -0.868570706f,
    0.496897049f, -0.867809497f,
    0.498227667f, -0.867046246f,
    0.499557113f, -0.866280954f,
    0.500885383f, -0.865513624f,
    0.502212474f, -0.864744258f,
    0.503538384f, -0.863972856f,
    0.504863109f, -0.863199422f,
    0.506186645f, -0.862423956f,
    0.507508991f, -0.861646461f,
    0.508830143f, -0.860866939f,
    0.510150097f, -0.860085390f,
    0.511468850f, -0.859301818f,
    0.512786401f, -0.858516224f,
    0.514102744f, -0.857728610f,
    0.515417878f, -0.856938977f,
    0.516731799f, -0.856147328f,
    0.518044504f, -0.855353665f,
    0.519355990f, -0.854557988f,
    0.520666254f, -0.853760301f,
    0.521975293f, -0.852960605f,
    0.523283103f, -0.852158902f,
    0.524589683f, -0.851355193f,
    0.525895027f, -0.850549481f,
    0.527199135f, -0.849741768f,
    0.528502002f, -0.848932055f,
    0.529803625f, -0.848120345f,
    0.531104001f, -0.847306639f,
    0.532403128f, -0.846490939f,
    0.533701002f, -0.845673247f,
    0.534997620f, -0.844853565f,
    0.536292979f, -0.844031895f,
    0.537587076f, -0.843208240f,
    0.538879909f, -0.842382600f,
    0.540171473f, -0.841554977f,
    0.541461766f, -0.840725375f,
    0.542750785f, -0.839893794f,
    0.544038527f, -0.839060237f,
    0.545324988f, -0.838224706f,
    0.546610167f, -0.837387202f,
    0.547894059f, -0.836547727f,
    0.549176662f, -0.835706284f,
    0.550457973f, -0.834862875f,
    0.551737988f, -0.834017501f,
    0.553016706f, -0.833170165f,
    0.554294121f, -0.832320868f,
    0.555570233f, -0.831469612f,
    0.556845037f, -0.830616400f,
    0.558118531f, -0.829761234f,
    0.559390712f, -0.828904115f,
    0.560661576f, -0.828045045f,
    0.561931121f, -0.827184027f,
    0.563199344f, -0.826321063f,
    0.564466242f, -0.825456154f,
    0.565731811f, -0.824589303f,
    0.566996049f, -0.823720511f,
    0.568258953f, -0.822849781f,
    0.569520519f, -0.821977115f,
    0.570780746f, -0.821102515f,
    0.572039629f, -0.820225983f,
    0.573297167f, -0.819347520f,
    0.574553355f, -0.818467130f,
    0.575808191f, -0.817584813f,
    0.577061673f, -0.816700573f,
    0.578313796f, -0.815814411f,
    0.579564559f, -0.814926329f,
    0.580813958f, -0.814036330f,
    0.582061990f, -0.813144415f,
    0.583308653f, -0.812250587f,
    0.584553943f, -0.811354847f,
    0.585797857f, -0.810457198f,
    0.587040394f, -0.809557642f,
    0.588281548f, -0.808656182f,
    0.589521319f, -0.807752818f,
    0.590759702f, -0.806847554f,
    0.591996695f, -0.805940391f,
    0.593232295f, -0.805031331f,
    0.594466499f, -0.804120377f,
    0.595699304f, -0.803207531f,
    0.596930708f, -0.802292796f,
    0.598160707f, -0.801376172f,
    0.599389298f, -0.800457662f,
    0.600616479f, -0.799537269f,
    0.601842247f, -0.798614995f,
    0.603066599f, -0.797690841f,
    0.604289531f, -0.796764810f,
    0.605511041f, -0.795836905f,
    0.606731127f, -0.794907126f,
    0.607949785f, -0.793975478f,
    0.609167012f, -0.793041960f,
    0.610382806f, -0.792106577f,
    0.611597164f, -0.791169330f,
    0.612810082f, -0.790230221f,
    0.614021559f, -0.789289253f,
    0.615231591f, -0.788346428f,
    0.616440175f, -0.787401747f,
    0.617647308f, -0.786455214f,
    0.618852988f, -0.785506830f,
    0.620057212f, -0.784556597f,
    0.621259977f, -0.783604519f,
    0.622461279f, -0.782650596f,
    0.623661118f, -0.781694832f,
    0.624859488f, -0.780737229f,
    0.626056388f, -0.779777788f,
    0.627251815f, -0.778816512f,
    0.628445767f, -0.777853404f,
    0.629638239f, -0.776888466f,
    0.630829230f, -0.775921699f,
    0.632018736f, -0.774953107f,
    0.633206755f, -0.773982691f,
    0.634393284f, -0.773010453f,
    0.635578320f, -0.772036397f,
    0.636761861f, -0.771060524f,
    0.637943904f, -0.770082837f,
    0.639124445f, -0.769103338f,
    0.640303482f, -0.768122029f,
    0.641481013f, -0.767138912f,
    0.642657034f, -0.766153990f,
    0.643831543f, -0.765167266f,
    0.645004537f, -0.764178741f,
    0.646176013f, -0.763188417f,
    0.647345969f, -0.762196298f,
    0.648514401f, -0.761202385f,
    0.649681307f, -0.760206682f,
    0.650846685f, -0.759209189f,
    0.652010531f, -0.758209910f,
    0.653172843f, -0.757208847f,
    0.654333618f, -0.756206001f,
    0.655492853f, -0.755201377f,
    0.656650546f, -0.754194975f,
    0.657806693f, -0.753186799f,
    0.658961293f, -0.752176850f,
    0.660114342f, -0.751165132f,
    0.661265838f, -0.750151646f,
    0.662415778f, -0.749136395f,
    0.663564159f, -0.748119380f,
    0.664710978f, -0.747100606f,
    0.665856234f, -0.746080074f,
    0.666999922f, -0.745057785f,
    0.668142041f, -0.744033744f,
    0.669282588f, -0.743007952f,
    0.670421560f, -0.741980412f,
    0.671558955f, -0.740951125f,
    0.672694769f, -0.739920095f,
    0.673829000f, -0.738887324f,
    0.674961646f, -0.737852815f,
    0.676092704f, -0.736816569f,
    0.677222170f, -0.735778589f,
    0.678350043f, -0.734738878f,
    0.679476320f, -0.733697438f,
    0.680600998f, -0.732654272f,
    0.681724074f, -0.731609381f,
    0.682845546f, -0.730562769f,
    0.683965412f, -0.729514438f,
    0.685083668f, -0.728464390f,
    0.686200312f, -0.727412629f,
    0.687315341f, -0.726359155f,
    0.688428753f, -0.725303972f,
    0.689540545f, -0.724247083f,
    0.690650714f, -0.723188489f,
    0.691759258f, -0.722128194f,
    0.692866175f, -0.721066199f,
    0.693971461f, -0.720002508f,
    0.695075114f, -0.718937122f,
    0.696177131f, -0.717870045f,
    0.697277511f, -0.716801279f,
    0.698376249f, -0.715730825f,
    0.699473345f, -0.714658688f,
    0.700568794f, -0.713584869f,
    0.701662595f, -0.712509371f,
    0.702754744f, -0.711432196f,
    0.703845241f, -0.710353347f,
    0.704934080f, -0.709272826f,
    0.706021261f, -0.708190637f,
    0.707106781f, -0.707106781f,
    0.708190637f, -0.706021261f,
    0.709272826f, -0.704934080f,
    0.710353347f, -0.703845241f,
    0.711432196f, -0.702754744f,
    0.712509371f, -0.701662595f,
    0.713584869f, -0.700568794f,
    0.714658688f, -0.699473345f,
    0.715730825f, -0.698376249f,
    0.716801279f, -0.697277511f,
    0.717870045f, -0.696177131f,
    0.718937122f, -0.695075114f,
    0.720002508f, -0.693971461f,
    0.721066199f, -0.692866175f,
    0.722128194f, -0.691759258f,
    0.723188489f, -0.690650714f,
    0.724247083f, -0.689540545f,
    0.725303972f, -0.688428753f,
    0.726359155f, -0.687315341f,
    0.727412629f, -0.686200312f,
    0.728464390f, -0.685083668f,
    0.729514438f, -0.683965412f,
    0.730562769f, -0.682845546f,
    0.731609381f, -0.681724074f,
    0.732654272f, -0.680600998f,
    0.733697438f, -0.679476320f,
    0.734738878f, -0.678350043f,
    0.735778589f, -0.677222170f,
    0.736816569f, -0.676092704f,
    0.737852815f, -0.674961646f,
    0.738887324f, -0.673829000f,
    0.739920095f, -0.672694769f,
    0.740951125f, -0.671558955f,
    0.741980412f, -0.670421560f,
    0.743007952f, -0.669282588f,
    0.744033744f, -0.668142041f,
    0.745057785f, -0.666999922f,
    0.746080074f, -0.665856234f,
    0.747100606f, -0.664710978f,
    0.748119380f, -0.663564159f,
    0.749136395f, -0.662415778f,
    0.750151646f, -0.661265838f,
    0.751165132f, -0.660114342f,
    0.752176850f, -0.658961293f,
    0.753186799f, -0.657806693f,
    0.754194975f, -0.656650546f,
    0.755201377f, -0.655492853f,
    0.756206001f, -0.654333618f,
    0.757208847f, -0.653172843f,
    0.758209910f, -0.652010531f,
    0.759209189f, -0.650846685f,
    0.760206682f, -0.649681307f,
    0.761202385f, -0.648514401f,
    0.762196298f, -0.647345969f,
    0.763188417f, -0.646176013f,
    0.764178741f, -0.645004537f,
    0.765167266f, -0.643831543f,
    0.766153990f, -0.642657034f,
    0.767138912f, -0.641481013f,
    0.768122029f, -0.640303482f,
    0.769103338f, -0.639124445f,
    0.770082837f, -0.637943904f,
    0.771060524f, -0.636761861f,
    0.772036397f, -0.635578320f,
    0.773010453f, -0.634393284f,
    0.773982691f, -0.633206755f,
    0.774953107f, -0.632018736f,
    0.775921699f, -0.630829230f,
    0.776888466f, -0.629638239f,
    0.777853404f, -0.628445767f,
    0.778816512f, -0.627251815f,
    0.779777788f, -0.626056388f,
    0.780737229f, -0.624859488f,
    0.781694832f, -0.623661118f,
    0.782650596f, -0.622461279f,
    0.783604519f, -0.621259977f,
    0.784556597f, -0.620057212f,
    0.785506830f, -0.618852988f,
    0.786455214f, -0.617647308f,
    0.787401747f, -0.616440175f,
    0.788346428f, -0.615231591f,
    0.789289253f, -0.614021559f,
    0.790230221f, -0.612810082f,
    0.791169330f, -0.611597164f,
    0.792106577f, -0.610382806f,
    0.793041960f, -0.609167012f,
    0.793975478f, -0.607949785f,
    0.794907126f, -0.606731127f,
    0.795836905f, -0.605511041f,
    0.796764810f, -0.604289531f,
    0.797690841f, -0.603066599f,
    0.798614995f, -0.601842247f,
    0.799537269f, -0.600616479f,
    0.800457662f, -0.599389298f,
    0.801376172f, -0.598160707f,
    0.802292796f, -0.596930708f,
    0.803207531f, -0.595699304f,
    0.804120377f, -0.594466499f,
    0.805031331f, -0.593232295f,
    0.805940391f, -0.591996695f,
    0.806847554f, -0.590759702f,
    0.807752818f, -0.589521319f,
    0.808656182f, -0.588281548f,
    0.809557642f, -0.587040394f,
    0.810457198f, -0.585797857f,
    0.811354847f, -0.584553943f,
    0.812250587f, -0.583308653f,
    0.813144415f, -0.582061990f,
    0.814036330f, -0.580813958f,
    0.814926329f, -0.579564559f,
    0.815814411f, -0.578313796f,
    0.816700573f, -0.577061673f,
    0.817584813f, -0.575808191f,
    0.818467130f, -0.574553355f,
    0.819347520f, -0.573297167f,
    0.820225983f, -0.572039629f,
    0.821102515f, -0.570780746f,
    0.821977115f, -0.569520519f,
    0.822849781f, -0.568258953f,
    0.823720511f, -0.566996049f,
    0.824589303f, -0.565731811f,
    0.825456154f, -0.564466242f,
    0.826321063f, -0.563199344f,
    0.827184027f, -0.561931121f,
    0.828045045f, -0.560661576f,
    0.828904115f, -0.559390712f,
    0.829761234f, -0.558118531f,
    0.830616400f, -0.556845037f,
    0.831469612f, -0.555570233f,
    0.832320868f, -0.554294121f,
    0.833170165f, -0.553016706f,
    0.834017501f, -0.551737988f,
    0.834862875f, -0.550457973f,
    0.835706284f, -0.549176662f,
    0.836547727f, -0.547894059f,
    0.837387202f, -0.546610167f,
    0.838224706f, -0.545324988f,
    0.839060237f, -0.544038527f,
    0.839893794f, -0.542750785f,
    0.840725375f, -0.541461766f,
    0.841554977f, -0.540171473f,
    0.842382600f, -0.538879909f,
    0.843208240f, -0.537587076f,
    0.844031895f, -0.536292979f,
    0.844853565f, -0.534997620f,
    0.845673247f, -0.533701002f,
    0.846490939f, -0.532403128f,
    0.847306639f, -0.531104001f,
    0.848120345f, -0.529803625f,
    0.848932055f, -0.528502002f,
    0.849741768f, -0.527199135f,
    0.850549481f, -0.525895027f,
    0.851355193f, -0.524589683f,
    0.852158902f, -0.523283103f,
    0.852960605f, -0.521975293f,
    0.853760301f, -0.520666254f,
    0.854557988f, -0.519355990f,
    0.855353665f, -0.518044504f,
    0.856147328f, -0.516731799f,
    0.856938977f, -0.515417878f,
    0.857728610f, -0.514102744f,
    0.858516224f, -0.512786401f,
    0.859301818f, -0.511468850f,
    0.860085390f, -0.510150097f,
    0.860866939f, -0.508830143f,
    0.861646461f, -0.507508991f,
    0.862423956f, -0.506186645f,
    0.863199422f, -0.504863109f,
    0.863972856f, -0.503538384f,
    0.864744258f, -0.502212474f,
    0.865513624f, -0.500885383f,
    0.866280954f, -0.499557113f,
    0.867046246f, -0.498227667f,
    0.867809497f, -0.496897049f,
    0.868570706f, -0.495565262f,
    0.869329871f, -0.494232309f,
    0.870086991f, -0.492898192f,
    0.870842063f, -0.491562916f,
    0.871595087f, -0.490226483f,
    0.872346059f, -0.488888897f,
    0.873094978f, -0.487550160f,
    0.873841843f, -0.486210276f,
    0.874586652f, -0.484869248f,
    0.875329403f, -0.483527079f,
    0.876070094f, -0.482183772f,
    0.876808724f, -0.480839331f,
    0.877545290f, -0.479493758f,
    0.878279792f, -0.478147056f,
    0.879012226f, -0.476799230f,
    0.879742593f, -0.475450282f,
    0.880470889f, -0.474100215f,
    0.881197113f, -0.472749032f,
    0.881921264f, -0.471396737f,
    0.882643340f, -0.470043332f,
    0.883363339f, -0.468688822f,
    0.884081259f, -0.467333209f,
    0.884797098f, -0.465976496f,
    0.885510856f, -0.464618686f,
    0.886222530f, -0.463259784f,
    0.886932119f, -0.461899791f,
    0.887639620f, -0.460538711f,
    0.888345033f, -0.459176548f,
    0.889048356f, -0.457813304f,
    0.889749586f, -0.456448982f,
    0.890448723f, -0.455083587f,
    0.891145765f, -0.453717121f,
    0.891840709f, -0.452349587f,
    0.892533555f, -0.450980989f,
    0.893224301f, -0.449611330f,
    0.893912945f, -0.448240612f,
    0.894599486f, -0.446868840f,
    0.895283921f, -0.445496017f,
    0.895966250f, -0.444122145f,
    0.896646470f, -0.442747228f,
    0.897324581f, -0.441371269f,
    0.898000580f, -0.439994271f,
    0.898674466f, -0.438616239f,
    0.899346237f, -0.437237174f,
    0.900015892f, -0.435857080f,
    0.900683429f, -0.434475961f,
    0.901348847f, -0.433093819f,
    0.902012144f, -0.431710658f,
    0.902673318f, -0.430326481f,
    0.903332368f, -0.428941292f,
    0.903989293f, -0.427555093f,
    0.904644091f, -0.426167889f,
    0.905296759f, -0.424779681f,
    0.905947298f, -0.423390474f,
    0.906595705f, -0.422000271f,
    0.907241978f, -0.420609074f,
    0.907886116f, -0.419216888f,
    0.908528119f, -0.417823716f,
    0.909167983f, -0.416429560f,
    0.909805708f, -0.415034424f,
    0.910441292f, -0.413638312f,
    0.911074734f, -0.412241227f,
    0.911706032f, -0.410843171f,
    0.912335185f, -0.409444149f,
    0.912962190f, -0.408044163f,
    0.913587048f, -0.406643217f,
    0.914209756f, -0.405241314f,
    0.914830312f, -0.403838458f,
    0.915448716f, -0.402434651f,
    0.916064966f, -0.401029897f,
    0.916679060f, -0.399624200f,
    0.917290997f, -0.398217562f,
    0.917900776f, -0.396809987f,
    0.918508394f, -0.395401479f,
    0.919113852f, -0.393992040f,
    0.919717146f, -0.392581674f,
    0.920318277f, -0.391170384f,
    0.920917242f, -0.389758174f,
    0.921514039f, -0.388345047f,
    0.922108669f, -0.386931006f,
    0.922701128f, -0.385516054f,
    0.923291417f, -0.384100195f,
    0.923879533f, -0.382683432f,
    0.924465474f, -0.381265769f,
    0.925049241f, -0.379847209f,
    0.925630831f, -0.378427755f,
    0.926210242f, -0.377007410f,
    0.926787474f, -0.375586178f,
    0.927362526f, -0.374164063f,
    0.927935395f, -0.372741067f,
    0.928506080f, -0.371317194f,
    0.929074581f, -0.369892447f,
    0.929640896f, -0.368466830f,
    0.930205023f, -0.367040346f,
    0.930766961f, -0.365612998f,
    0.931326709f, -0.364184790f,
    0.931884266f, -0.362755724f,
    0.932439629f, -0.361325806f,
    0.932992799f, -0.359895037f,
    0.933543773f, -0.358463421f,
    0.934092550f, -0.357030961f,
    0.934639130f, -0.355597662f,
    0.935183510f, -0.354163525f,
    0.935725689f, -0.352728556f,
    0.936265667f, -0.351292756f,
    0.936803442f, -0.349856130f,
    0.937339012f, -0.348418680f,
    0.937872376f, -0.346980411f,
    0.938403534f, -0.345541325f,
    0.938932484f, -0.344101426f,
    0.939459224f, -0.342660717f,
    0.939983753f, -0.341219202f,
    0.940506071f, -0.339776884f,
    0.941026175f, -0.338333767f,
    0.941544065f, -0.336889853f,
    0.942059740f, -0.335445147f,
    0.942573198f, -0.333999651f,
    0.943084437f, -0.332553370f,
    0.943593458f, -0.331106306f,
    0.944100258f, -0.329658463f,
    0.944604837f, -0.328209844f,
    0.945107193f, -0.326760452f,
    0.945607325f, -0.325310292f,
    0.946105232f, -0.323859367f,
    0.946600913f, -0.322407679f,
    0.947094366f, -0.320955232f,
    0.947585591f, -0.319502031f,
    0.948074586f, -0.318048077f,
    0.948561350f, -0.316593376f,
    0.949045882f, -0.315137929f,
    0.949528181f, -0.313681740f,
    0.950008245f, -0.312224814f,
    0.950486074f, -0.310767153f,
    0.950961666f, -0.309308760f,
    0.951435021f, -0.307849640f,
    0.951906137f, -0.306389795f,
    0.952375013f, -0.304929230f,
    0.952841648f, -0.303467947f,
    0.953306040f, -0.302005949f,
    0.953768190f, -0.300543241f,
    0.954228095f, -0.299079826f,
    0.954685755f, -0.297615707f,
    0.955141168f, -0.296150888f,
    0.955594334f, -0.294685372f,
    0.956045251f, -0.293219163f,
    0.956493919f, -0.291752263f,
    0.956940336f, -0.290284677f,
    0.957384501f, -0.288816408f,
    0.957826413f, -0.287347460f,
    0.958266071f, -0.285877835f,
    0.958703475f, -0.284407537f,
    0.959138622f, -0.282936570f,
    0.959571513f, -0.281464938f,
    0.960002146f, -0.279992643f,
    0.960430519f, -0.278519689f,
    0.960856633f, -0.277046080f,
    0.961280486f, -0.275571819f,
    0.961702077f, -0.274096910f,
    0.962121404f, -0.272621355f,
    0.962538468f, -0.271145160f,
    0.962953267f, -0.269668326f,
    0.963365800f, -0.268190857f,
    0.963776066f, -0.266712757f,
    0.964184064f, -0.265234030f,
    0.964589793f, -0.263754679f,
    0.964993253f, -0.262274707f,
    0.965394442f, -0.260794118f,
    0.965793359f, -0.259312915f,
    0.966190003f, -0.257831102f,
    0.966584374f, -0.256348682f,
    0.966976471f, -0.254865660f,
    0.967366292f, -0.253382037f,
    0.967753837f, -0.251897818f,
    0.968139105f, -0.250413007f,
    0.968522094f, -0.248927606f,
    0.968902805f, -0.247441619f,
    0.969281235f, -0.245955050f,
    0.969657385f, -0.244467903f,
    0.970031253f, -0.242980180f,
    0.970402839f, -0.241491885f,
    0.970772141f, -0.240003022f,
    0.971139158f, -0.238513595f,
    0.971503891f, -0.237023606f,
    0.971866337f, -0.235533059f,
    0.972226497f, -0.234041959f,
    0.972584369f, -0.232550307f,
    0.972939952f, -0.231058108f,
    0.973293246f, -0.229565366f,
    0.973644250f, -0.228072083f,
    0.973992962f, -0.226578264f,
    0.974339383f, -0.225083911f,
    0.974683511f, -0.223589029f,
    0.975025345f, -0.222093621f,
    0.975364885f, -0.220597690f,
    0.975702130f, -0.219101240f,
    0.976037079f, -0.217604275f,
    0.976369731f, -0.216106797f,
    0.976700086f, -0.214608811f,
    0.977028143f, -0.213110320f,
    0.977353900f, -0.211611327f,
    0.977677358f, -0.210111837f,
    0.977998515f, -0.208611852f,
    0.978317371f, -0.207111376f,
    0.978633924f, -0.205610413f,
    0.978948175f, -0.204108966f,
    0.979260123f, -0.202607039f,
    0.979569766f, -0.201104635f,
    0.979877104f, -0.199601758f,
    0.980182136f, -0.198098411f,
    0.980484862f, -0.196594598f,
    0.980785280f, -0.195090322f,
    0.981083391f, -0.193585587f,
    0.981379193f, -0.192080397f,
    0.981672686f, -0.190574755f,
    0.981963869f, -0.189068664f,
    0.982252741f, -0.187562129f,
    0.982539302f, -0.186055152f,
    0.982823551f, -0.184547737f,
    0.983105487f, -0.183039888f,
    0.983385110f, -0.181531608f,
    0.983662419f, -0.180022901f,
    0.983937413f, -0.178513771f,
    0.984210092f, -0.177004220f,
    0.984480455f, -0.175494253f,
    0.984748502f, -0.173983873f,
    0.985014231f, -0.172473084f,
    0.985277642f, -0.170961889f,
    0.985538735f, -0.169450291f,
    0.985797509f, -0.167938295f,
    0.986053963f, -0.166425904f,
    0.986308097f, -0.164913120f,
    0.986559910f, -0.163399949f,
    0.986809402f, -0.161886394f,
    0.987056571f, -0.160372457f,
    0.987301418f, -0.158858143f,
    0.987543942f, -0.157343456f,
    0.987784142f, -0.155828398f,
    0.988022017f, -0.154312973f,
    0.988257568f, -0.152797185f,
    0.988490793f, -0.151281038f,
    0.988721692f, -0.149764535f,
    0.988950265f, -0.148247679f,
    0.989176510f, -0.146730474f,
    0.989400428f, -0.145212925f,
    0.989622017f, -0.143695033f,
    0.989841278f, -0.142176804f,
    0.990058210f, -0.140658239f,
    0.990272812f, -0.139139344f,
    0.990485084f, -0.137620122f,
    0.990695025f, -0.136100575f,
    0.990902635f, -0.134580709f,
    0.991107914f, -0.133060525f,
    0.991310860f, -0.131540029f,
    0.991511473f, -0.130019223f,
    0.991709754f, -0.128498111f,
    0.991905700f, -0.126976696f,
    0.992099313f, -0.125454983f,
    0.992290591f, -0.123932975f,
    0.992479535f, -0.122410675f,
    0.992666142f, -0.120888087f,
    0.992850414f, -0.119365215f,
    0.993032350f, -0.117842062f,
    0.993211949f, -0.116318631f,
    0.993389211f, -0.114794927f,
    0.993564136f, -0.113270952f,
    0.993736722f, -0.111746711f,
    0.993906970f, -0.110222207f,
    0.994074879f, -0.108697444f,
    0.994240449f, -0.107172425f,
    0.994403680f, -0.105647154f,
    0.994564571f, -0.104121634f,
    0.994723121f, -0.102595869f,
    0.994879331f, -0.101069863f,
    0.995033199f, -0.099543619f,
    0.995184727f, -0.098017140f,
    0.995333912f, -0.096490431f,
    0.995480755f, -0.094963495f,
    0.995625256f, -0.093436336f,
    0.995767414f, -0.091908956f,
    0.995907229f, -0.090381361f,
    0.996044701f, -0.088853553f,
    0.996179829f, -0.087325535f,
    0.996312612f, -0.085797312f,
    0.996443051f, -0.084268888f,
    0.996571146f, -0.082740265f,
    0.996696895f, -0.081211447f,
    0.996820299f, -0.079682438f,
    0.996941358f, -0.078153242f,
    0.997060070f, -0.076623861f,
    0.997176437f, -0.075094301f,
    0.997290457f, -0.073564564f,
    0.997402130f, -0.072034653f,
    0.997511456f, -0.070504573f,
    0.997618435f, -0.068974328f,
    0.997723067f, -0.067443920f,
    0.997825350f, -0.065913353f,
    0.997925286f, -0.064382631f,
    0.998022874f, -0.062851758f,
    0.998118113f, -0.061320736f,
    0.998211003f, -0.059789571f,
    0.998301545f, -0.058258265f,
    0.998389737f, -0.056726821f,
    0.998475581f, -0.055195244f,
    0.998559074f, -0.053663538f,
    0.998640218f, -0.052131705f,
    0.998719012f, -0.050599749f,
    0.998795456f, -0.049067674f,
    0.998869550f, -0.047535484f,
    0.998941293f, -0.046003182f,
    0.999010686f, -0.044470772f,
    0.999077728f, -0.042938257f,
    0.999142419f, -0.041405641f,
    0.999204759f, -0.039872928f,
    0.999264747f, -0.038340120f,
    0.999322385f, -0.036807223f,
    0.999377670f, -0.035274239f,
    0.999430605f, -0.033741172f,
    0.999481187f, -0.032208025f,
    0.999529418f, -0.030674803f,
    0.999575296f, -0.029141509f,
    0.999618822f, -0.027608146f,
    0.999659997f, -0.026074718f,
    0.999698819f, -0.024541229f,
    0.999735288f, -0.023007681f,
    0.999769405f, -0.021474080f,
    0.999801170f, -0.019940429f,
    0.999830582f, -0.018406730f,
    0.999857641f, -0.016872988f,
    0.999882347f, -0.015339206f,
    0.999904701f, -0.013805389f,
    0.999924702f, -0.012271538f,
    0.999942350f, -0.010737659f,
    0.999957645f, -0.009203755f,
    0.999970586f, -0.007669829f,
    0.999981175f, -0.006135885f,
    0.999989411f, -0.004601926f,
    0.999995294f, -0.003067957f,
    0.999998823f, -0.001533980f
};

/*    
* @brief  Q31 Twiddle factors Table    
*/

/**    
* \par   
* Example code for Q31 Twiddle factors Generation::    
* \par    
* <pre>for(i = 0; i< 3N/4; i++)    
* {    
*    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);    
*    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are interleaved fashion    
* \par    
* Convert Floating point to Q31(Fixed point 1.31):    
*	round(twiddleCoefQ31(i) * pow(2, 31))    
*    
*/

const q31_t twiddleCoefQ31[6144] = {
   0x7fffffff, 0x0, 0x7ffff621, 0x3243f5, 0x7fffd886, 0x6487e3, 
   0x7fffa72c, 0x96cbc1, 0x7fff6216, 0xc90f88, 0x7fff0943, 0xfb5330, 
   0x7ffe9cb2, 0x12d96b1, 0x7ffe1c65, 0x15fda03, 0x7ffd885a, 0x1921d20, 
   0x7ffce093, 0x1c45ffe, 0x7ffc250f, 0x1f6a297, 0x7ffb55ce, 0x228e4e2, 
   0x7ffa72d1, 0x25b26d7, 0x7ff97c18, 0x28d6870, 0x7ff871a2, 0x2bfa9a4, 
   0x7ff75370, 0x2f1ea6c, 0x7ff62182, 0x3242abf, 0x7ff4dbd9, 0x3566a96, 
   0x7ff38274, 0x388a9ea, 0x7ff21553, 0x3bae8b2, 0x7ff09478, 0x3ed26e6, 
   0x7feeffe1, 0x41f6480, 0x7fed5791, 0x451a177, 0x7feb9b85, 0x483ddc3, 
   0x7fe9cbc0, 0x4b6195d, 0x7fe7e841, 0x4e8543e, 0x7fe5f108, 0x51a8e5c, 
   0x7fe3e616, 0x54cc7b1, 0x7fe1c76b, 0x57f0035, 0x7fdf9508, 0x5b137df, 
   0x7fdd4eec, 0x5e36ea9, 0x7fdaf519, 0x615a48b, 0x7fd8878e, 0x647d97c, 
   0x7fd6064c, 0x67a0d76, 0x7fd37153, 0x6ac406f, 0x7fd0c8a3, 0x6de7262, 
   0x7fce0c3e, 0x710a345, 0x7fcb3c23, 0x742d311, 0x7fc85854, 0x77501be, 
   0x7fc560cf, 0x7a72f45, 0x7fc25596, 0x7d95b9e, 0x7fbf36aa, 0x80b86c2, 
   0x7fbc040a, 0x83db0a7, 0x7fb8bdb8, 0x86fd947, 0x7fb563b3, 0x8a2009a, 
   0x7fb1f5fc, 0x8d42699, 0x7fae7495, 0x9064b3a, 0x7faadf7c, 0x9386e78, 
   0x7fa736b4, 0x96a9049, 0x7fa37a3c, 0x99cb0a7, 0x7f9faa15, 0x9cecf89, 
   0x7f9bc640, 0xa00ece8, 0x7f97cebd, 0xa3308bd, 0x7f93c38c, 0xa6522fe, 
   0x7f8fa4b0, 0xa973ba5, 0x7f8b7227, 0xac952aa, 0x7f872bf3, 0xafb6805, 
   0x7f82d214, 0xb2d7baf, 0x7f7e648c, 0xb5f8d9f, 0x7f79e35a, 0xb919dcf, 
   0x7f754e80, 0xbc3ac35, 0x7f70a5fe, 0xbf5b8cb, 0x7f6be9d4, 0xc27c389, 
   0x7f671a05, 0xc59cc68, 0x7f62368f, 0xc8bd35e, 0x7f5d3f75, 0xcbdd865, 
   0x7f5834b7, 0xcefdb76, 0x7f531655, 0xd21dc87, 0x7f4de451, 0xd53db92, 
   0x7f489eaa, 0xd85d88f, 0x7f434563, 0xdb7d376, 0x7f3dd87c, 0xde9cc40, 
   0x7f3857f6, 0xe1bc2e4, 0x7f32c3d1, 0xe4db75b, 0x7f2d1c0e, 0xe7fa99e, 
   0x7f2760af, 0xeb199a4, 0x7f2191b4, 0xee38766, 0x7f1baf1e, 0xf1572dc, 
   0x7f15b8ee, 0xf475bff, 0x7f0faf25, 0xf7942c7, 0x7f0991c4, 0xfab272b, 
   0x7f0360cb, 0xfdd0926, 0x7efd1c3c, 0x100ee8ad, 0x7ef6c418, 0x1040c5bb, 
   0x7ef05860, 0x1072a048, 0x7ee9d914, 0x10a4784b, 0x7ee34636, 0x10d64dbd, 
   0x7edc9fc6, 0x11082096, 0x7ed5e5c6, 0x1139f0cf, 0x7ecf1837, 0x116bbe60, 
   0x7ec8371a, 0x119d8941, 0x7ec14270, 0x11cf516a, 0x7eba3a39, 0x120116d5, 
   0x7eb31e78, 0x1232d979, 0x7eabef2c, 0x1264994e, 0x7ea4ac58, 0x1296564d, 
   0x7e9d55fc, 0x12c8106f, 0x7e95ec1a, 0x12f9c7aa, 0x7e8e6eb2, 0x132b7bf9, 
   0x7e86ddc6, 0x135d2d53, 0x7e7f3957, 0x138edbb1, 0x7e778166, 0x13c0870a, 
   0x7e6fb5f4, 0x13f22f58, 0x7e67d703, 0x1423d492, 0x7e5fe493, 0x145576b1, 
   0x7e57dea7, 0x148715ae, 0x7e4fc53e, 0x14b8b17f, 0x7e47985b, 0x14ea4a1f, 
   0x7e3f57ff, 0x151bdf86, 0x7e37042a, 0x154d71aa, 0x7e2e9cdf, 0x157f0086, 
   0x7e26221f, 0x15b08c12, 0x7e1d93ea, 0x15e21445, 0x7e14f242, 0x16139918, 
   0x7e0c3d29, 0x16451a83, 0x7e0374a0, 0x1676987f, 0x7dfa98a8, 0x16a81305, 
   0x7df1a942, 0x16d98a0c, 0x7de8a670, 0x170afd8d, 0x7ddf9034, 0x173c6d80, 
   0x7dd6668f, 0x176dd9de, 0x7dcd2981, 0x179f429f, 0x7dc3d90d, 0x17d0a7bc, 
   0x7dba7534, 0x1802092c, 0x7db0fdf8, 0x183366e9, 0x7da77359, 0x1864c0ea, 
   0x7d9dd55a, 0x18961728, 0x7d9423fc, 0x18c7699b, 0x7d8a5f40, 0x18f8b83c, 
   0x7d808728, 0x192a0304, 0x7d769bb5, 0x195b49ea, 0x7d6c9ce9, 0x198c8ce7, 
   0x7d628ac6, 0x19bdcbf3, 0x7d58654d, 0x19ef0707, 0x7d4e2c7f, 0x1a203e1b, 
   0x7d43e05e, 0x1a517128, 0x7d3980ec, 0x1a82a026, 0x7d2f0e2b, 0x1ab3cb0d, 
   0x7d24881b, 0x1ae4f1d6, 0x7d19eebf, 0x1b161479, 0x7d0f4218, 0x1b4732ef, 
   0x7d048228, 0x1b784d30, 0x7cf9aef0, 0x1ba96335, 0x7ceec873, 0x1bda74f6, 
   0x7ce3ceb2, 0x1c0b826a, 0x7cd8c1ae, 0x1c3c8b8c, 0x7ccda169, 0x1c6d9053, 
   0x7cc26de5, 0x1c9e90b8, 0x7cb72724, 0x1ccf8cb3, 0x7cabcd28, 0x1d00843d, 
   0x7ca05ff1, 0x1d31774d, 0x7c94df83, 0x1d6265dd, 0x7c894bde, 0x1d934fe5, 
   0x7c7da505, 0x1dc4355e, 0x7c71eaf9, 0x1df5163f, 0x7c661dbc, 0x1e25f282, 
   0x7c5a3d50, 0x1e56ca1e, 0x7c4e49b7, 0x1e879d0d, 0x7c4242f2, 0x1eb86b46, 
   0x7c362904, 0x1ee934c3, 0x7c29fbee, 0x1f19f97b, 0x7c1dbbb3, 0x1f4ab968, 
   0x7c116853, 0x1f7b7481, 0x7c0501d2, 0x1fac2abf, 0x7bf88830, 0x1fdcdc1b, 
   0x7bebfb70, 0x200d888d, 0x7bdf5b94, 0x203e300d, 0x7bd2a89e, 0x206ed295, 
   0x7bc5e290, 0x209f701c, 0x7bb9096b, 0x20d0089c, 0x7bac1d31, 0x21009c0c, 
   0x7b9f1de6, 0x21312a65, 0x7b920b89, 0x2161b3a0, 0x7b84e61f, 0x219237b5, 
   0x7b77ada8, 0x21c2b69c, 0x7b6a6227, 0x21f3304f, 0x7b5d039e, 0x2223a4c5, 
   0x7b4f920e, 0x225413f8, 0x7b420d7a, 0x22847de0, 0x7b3475e5, 0x22b4e274, 
   0x7b26cb4f, 0x22e541af, 0x7b190dbc, 0x23159b88, 0x7b0b3d2c, 0x2345eff8, 
   0x7afd59a4, 0x23763ef7, 0x7aef6323, 0x23a6887f, 0x7ae159ae, 0x23d6cc87, 
   0x7ad33d45, 0x24070b08, 0x7ac50dec, 0x243743fa, 0x7ab6cba4, 0x24677758, 
   0x7aa8766f, 0x2497a517, 0x7a9a0e50, 0x24c7cd33, 0x7a8b9348, 0x24f7efa2, 
   0x7a7d055b, 0x25280c5e, 0x7a6e648a, 0x2558235f, 0x7a5fb0d8, 0x2588349d, 
   0x7a50ea47, 0x25b84012, 0x7a4210d8, 0x25e845b6, 0x7a332490, 0x26184581, 
   0x7a24256f, 0x26483f6c, 0x7a151378, 0x26783370, 0x7a05eead, 0x26a82186, 
   0x79f6b711, 0x26d809a5, 0x79e76ca7, 0x2707ebc7, 0x79d80f6f, 0x2737c7e3, 
   0x79c89f6e, 0x27679df4, 0x79b91ca4, 0x27976df1, 0x79a98715, 0x27c737d3, 
   0x7999dec4, 0x27f6fb92, 0x798a23b1, 0x2826b928, 0x797a55e0, 0x2856708d, 
   0x796a7554, 0x288621b9, 0x795a820e, 0x28b5cca5, 0x794a7c12, 0x28e5714b, 
   0x793a6361, 0x29150fa1, 0x792a37fe, 0x2944a7a2, 0x7919f9ec, 0x29743946, 
   0x7909a92d, 0x29a3c485, 0x78f945c3, 0x29d34958, 0x78e8cfb2, 0x2a02c7b8, 
   0x78d846fb, 0x2a323f9e, 0x78c7aba2, 0x2a61b101, 0x78b6fda8, 0x2a911bdc, 
   0x78a63d11, 0x2ac08026, 0x789569df, 0x2aefddd8, 0x78848414, 0x2b1f34eb, 
   0x78738bb3, 0x2b4e8558, 0x786280bf, 0x2b7dcf17, 0x7851633b, 0x2bad1221, 
   0x78403329, 0x2bdc4e6f, 0x782ef08b, 0x2c0b83fa, 0x781d9b65, 0x2c3ab2b9, 
   0x780c33b8, 0x2c69daa6, 0x77fab989, 0x2c98fbba, 0x77e92cd9, 0x2cc815ee, 
   0x77d78daa, 0x2cf72939, 0x77c5dc01, 0x2d263596, 0x77b417df, 0x2d553afc, 
   0x77a24148, 0x2d843964, 0x7790583e, 0x2db330c7, 0x777e5cc3, 0x2de2211e, 
   0x776c4edb, 0x2e110a62, 0x775a2e89, 0x2e3fec8b, 0x7747fbce, 0x2e6ec792, 
   0x7735b6af, 0x2e9d9b70, 0x77235f2d, 0x2ecc681e, 0x7710f54c, 0x2efb2d95, 
   0x76fe790e, 0x2f29ebcc, 0x76ebea77, 0x2f58a2be, 0x76d94989, 0x2f875262, 
   0x76c69647, 0x2fb5fab2, 0x76b3d0b4, 0x2fe49ba7, 0x76a0f8d2, 0x30133539, 
   0x768e0ea6, 0x3041c761, 0x767b1231, 0x30705217, 0x76680376, 0x309ed556, 
   0x7654e279, 0x30cd5115, 0x7641af3d, 0x30fbc54d, 0x762e69c4, 0x312a31f8, 
   0x761b1211, 0x3158970e, 0x7607a828, 0x3186f487, 0x75f42c0b, 0x31b54a5e, 
   0x75e09dbd, 0x31e39889, 0x75ccfd42, 0x3211df04, 0x75b94a9c, 0x32401dc6, 
   0x75a585cf, 0x326e54c7, 0x7591aedd, 0x329c8402, 0x757dc5ca, 0x32caab6f, 
   0x7569ca99, 0x32f8cb07, 0x7555bd4c, 0x3326e2c3, 0x75419de7, 0x3354f29b, 
   0x752d6c6c, 0x3382fa88, 0x751928e0, 0x33b0fa84, 0x7504d345, 0x33def287, 
   0x74f06b9e, 0x340ce28b, 0x74dbf1ef, 0x343aca87, 0x74c7663a, 0x3468aa76, 
   0x74b2c884, 0x34968250, 0x749e18cd, 0x34c4520d, 0x7489571c, 0x34f219a8, 
   0x74748371, 0x351fd918, 0x745f9dd1, 0x354d9057, 0x744aa63f, 0x357b3f5d, 
   0x74359cbd, 0x35a8e625, 0x74208150, 0x35d684a6, 0x740b53fb, 0x36041ad9, 
   0x73f614c0, 0x3631a8b8, 0x73e0c3a3, 0x365f2e3b, 0x73cb60a8, 0x368cab5c, 
   0x73b5ebd1, 0x36ba2014, 0x73a06522, 0x36e78c5b, 0x738acc9e, 0x3714f02a, 
   0x73752249, 0x37424b7b, 0x735f6626, 0x376f9e46, 0x73499838, 0x379ce885, 
   0x7333b883, 0x37ca2a30, 0x731dc70a, 0x37f76341, 0x7307c3d0, 0x382493b0, 
   0x72f1aed9, 0x3851bb77, 0x72db8828, 0x387eda8e, 0x72c54fc1, 0x38abf0ef, 
   0x72af05a7, 0x38d8fe93, 0x7298a9dd, 0x39060373, 0x72823c67, 0x3932ff87, 
   0x726bbd48, 0x395ff2c9, 0x72552c85, 0x398cdd32, 0x723e8a20, 0x39b9bebc, 
   0x7227d61c, 0x39e6975e, 0x7211107e, 0x3a136712, 0x71fa3949, 0x3a402dd2, 
   0x71e35080, 0x3a6ceb96, 0x71cc5626, 0x3a99a057, 0x71b54a41, 0x3ac64c0f, 
   0x719e2cd2, 0x3af2eeb7, 0x7186fdde, 0x3b1f8848, 0x716fbd68, 0x3b4c18ba, 
   0x71586b74, 0x3b78a007, 0x71410805, 0x3ba51e29, 0x7129931f, 0x3bd19318, 
   0x71120cc5, 0x3bfdfecd, 0x70fa74fc, 0x3c2a6142, 0x70e2cbc6, 0x3c56ba70, 
   0x70cb1128, 0x3c830a50, 0x70b34525, 0x3caf50da, 0x709b67c0, 0x3cdb8e09, 
   0x708378ff, 0x3d07c1d6, 0x706b78e3, 0x3d33ec39, 0x70536771, 0x3d600d2c, 
   0x703b44ad, 0x3d8c24a8, 0x7023109a, 0x3db832a6, 0x700acb3c, 0x3de4371f, 
   0x6ff27497, 0x3e10320d, 0x6fda0cae, 0x3e3c2369, 0x6fc19385, 0x3e680b2c, 
   0x6fa90921, 0x3e93e950, 0x6f906d84, 0x3ebfbdcd, 0x6f77c0b3, 0x3eeb889c, 
   0x6f5f02b2, 0x3f1749b8, 0x6f463383, 0x3f430119, 0x6f2d532c, 0x3f6eaeb8, 
   0x6f1461b0, 0x3f9a5290, 0x6efb5f12, 0x3fc5ec98, 0x6ee24b57, 0x3ff17cca, 
   0x6ec92683, 0x401d0321, 0x6eaff099, 0x40487f94, 0x6e96a99d, 0x4073f21d, 
   0x6e7d5193, 0x409f5ab6, 0x6e63e87f, 0x40cab958, 0x6e4a6e66, 0x40f60dfb, 
   0x6e30e34a, 0x4121589b, 0x6e174730, 0x414c992f, 0x6dfd9a1c, 0x4177cfb1, 
   0x6de3dc11, 0x41a2fc1a, 0x6dca0d14, 0x41ce1e65, 0x6db02d29, 0x41f93689, 
   0x6d963c54, 0x42244481, 0x6d7c3a98, 0x424f4845, 0x6d6227fa, 0x427a41d0, 
   0x6d48047e, 0x42a5311b, 0x6d2dd027, 0x42d0161e, 0x6d138afb, 0x42faf0d4, 
   0x6cf934fc, 0x4325c135, 0x6cdece2f, 0x4350873c, 0x6cc45698, 0x437b42e1, 
   0x6ca9ce3b, 0x43a5f41e, 0x6c8f351c, 0x43d09aed, 0x6c748b3f, 0x43fb3746, 
   0x6c59d0a9, 0x4425c923, 0x6c3f055d, 0x4450507e, 0x6c242960, 0x447acd50, 
   0x6c093cb6, 0x44a53f93, 0x6bee3f62, 0x44cfa740, 0x6bd3316a, 0x44fa0450, 
   0x6bb812d1, 0x452456bd, 0x6b9ce39b, 0x454e9e80, 0x6b81a3cd, 0x4578db93, 
   0x6b66536b, 0x45a30df0, 0x6b4af279, 0x45cd358f, 0x6b2f80fb, 0x45f7526b, 
   0x6b13fef5, 0x4621647d, 0x6af86c6c, 0x464b6bbe, 0x6adcc964, 0x46756828, 
   0x6ac115e2, 0x469f59b4, 0x6aa551e9, 0x46c9405c, 0x6a897d7d, 0x46f31c1a, 
   0x6a6d98a4, 0x471cece7, 0x6a51a361, 0x4746b2bc, 0x6a359db9, 0x47706d93, 
   0x6a1987b0, 0x479a1d67, 0x69fd614a, 0x47c3c22f, 0x69e12a8c, 0x47ed5be6, 
   0x69c4e37a, 0x4816ea86, 0x69a88c19, 0x48406e08, 0x698c246c, 0x4869e665, 
   0x696fac78, 0x48935397, 0x69532442, 0x48bcb599, 0x69368bce, 0x48e60c62, 
   0x6919e320, 0x490f57ee, 0x68fd2a3d, 0x49389836, 0x68e06129, 0x4961cd33, 
   0x68c387e9, 0x498af6df, 0x68a69e81, 0x49b41533, 0x6889a4f6, 0x49dd282a, 
   0x686c9b4b, 0x4a062fbd, 0x684f8186, 0x4a2f2be6, 0x683257ab, 0x4a581c9e, 
   0x68151dbe, 0x4a8101de, 0x67f7d3c5, 0x4aa9dba2, 0x67da79c3, 0x4ad2a9e2, 
   0x67bd0fbd, 0x4afb6c98, 0x679f95b7, 0x4b2423be, 0x67820bb7, 0x4b4ccf4d, 
   0x676471c0, 0x4b756f40, 0x6746c7d8, 0x4b9e0390, 0x67290e02, 0x4bc68c36, 
   0x670b4444, 0x4bef092d, 0x66ed6aa1, 0x4c177a6e, 0x66cf8120, 0x4c3fdff4, 
   0x66b187c3, 0x4c6839b7, 0x66937e91, 0x4c9087b1, 0x6675658c, 0x4cb8c9dd, 
   0x66573cbb, 0x4ce10034, 0x66390422, 0x4d092ab0, 0x661abbc5, 0x4d31494b, 
   0x65fc63a9, 0x4d595bfe, 0x65ddfbd3, 0x4d8162c4, 0x65bf8447, 0x4da95d96, 
   0x65a0fd0b, 0x4dd14c6e, 0x65826622, 0x4df92f46, 0x6563bf92, 0x4e210617, 
   0x6545095f, 0x4e48d0dd, 0x6526438f, 0x4e708f8f, 0x65076e25, 0x4e984229, 
   0x64e88926, 0x4ebfe8a5, 0x64c99498, 0x4ee782fb, 0x64aa907f, 0x4f0f1126, 
   0x648b7ce0, 0x4f369320, 0x646c59bf, 0x4f5e08e3, 0x644d2722, 0x4f857269, 
   0x642de50d, 0x4faccfab, 0x640e9386, 0x4fd420a4, 0x63ef3290, 0x4ffb654d, 
   0x63cfc231, 0x50229da1, 0x63b0426d, 0x5049c999, 0x6390b34a, 0x5070e92f, 
   0x637114cc, 0x5097fc5e, 0x635166f9, 0x50bf031f, 0x6331a9d4, 0x50e5fd6d, 
   0x6311dd64, 0x510ceb40, 0x62f201ac, 0x5133cc94, 0x62d216b3, 0x515aa162, 
   0x62b21c7b, 0x518169a5, 0x6292130c, 0x51a82555, 0x6271fa69, 0x51ced46e, 
   0x6251d298, 0x51f576ea, 0x62319b9d, 0x521c0cc2, 0x6211557e, 0x524295f0, 
   0x61f1003f, 0x5269126e, 0x61d09be5, 0x528f8238, 0x61b02876, 0x52b5e546, 
   0x618fa5f7, 0x52dc3b92, 0x616f146c, 0x53028518, 0x614e73da, 0x5328c1d0, 
   0x612dc447, 0x534ef1b5, 0x610d05b7, 0x537514c2, 0x60ec3830, 0x539b2af0, 
   0x60cb5bb7, 0x53c13439, 0x60aa7050, 0x53e73097, 0x60897601, 0x540d2005, 
   0x60686ccf, 0x5433027d, 0x604754bf, 0x5458d7f9, 0x60262dd6, 0x547ea073, 
   0x6004f819, 0x54a45be6, 0x5fe3b38d, 0x54ca0a4b, 0x5fc26038, 0x54efab9c, 
   0x5fa0fe1f, 0x55153fd4, 0x5f7f8d46, 0x553ac6ee, 0x5f5e0db3, 0x556040e2, 
   0x5f3c7f6b, 0x5585adad, 0x5f1ae274, 0x55ab0d46, 0x5ef936d1, 0x55d05faa, 
   0x5ed77c8a, 0x55f5a4d2, 0x5eb5b3a2, 0x561adcb9, 0x5e93dc1f, 0x56400758, 
   0x5e71f606, 0x566524aa, 0x5e50015d, 0x568a34a9, 0x5e2dfe29, 0x56af3750, 
   0x5e0bec6e, 0x56d42c99, 0x5de9cc33, 0x56f9147e, 0x5dc79d7c, 0x571deefa, 
   0x5da5604f, 0x5742bc06, 0x5d8314b1, 0x57677b9d, 0x5d60baa7, 0x578c2dba, 
   0x5d3e5237, 0x57b0d256, 0x5d1bdb65, 0x57d5696d, 0x5cf95638, 0x57f9f2f8, 
   0x5cd6c2b5, 0x581e6ef1, 0x5cb420e0, 0x5842dd54, 0x5c9170bf, 0x58673e1b, 
   0x5c6eb258, 0x588b9140, 0x5c4be5b0, 0x58afd6bd, 0x5c290acc, 0x58d40e8c, 
   0x5c0621b2, 0x58f838a9, 0x5be32a67, 0x591c550e, 0x5bc024f0, 0x594063b5, 
   0x5b9d1154, 0x59646498, 0x5b79ef96, 0x598857b2, 0x5b56bfbd, 0x59ac3cfd, 
   0x5b3381ce, 0x59d01475, 0x5b1035cf, 0x59f3de12, 0x5aecdbc5, 0x5a1799d1, 
   0x5ac973b5, 0x5a3b47ab, 0x5aa5fda5, 0x5a5ee79a, 0x5a82799a, 0x5a82799a, 
   0x5a5ee79a, 0x5aa5fda5, 0x5a3b47ab, 0x5ac973b5, 0x5a1799d1, 0x5aecdbc5, 
   0x59f3de12, 0x5b1035cf, 0x59d01475, 0x5b3381ce, 0x59ac3cfd, 0x5b56bfbd, 
   0x598857b2, 0x5b79ef96, 0x59646498, 0x5b9d1154, 0x594063b5, 0x5bc024f0, 
   0x591c550e, 0x5be32a67, 0x58f838a9, 0x5c0621b2, 0x58d40e8c, 0x5c290acc, 
   0x58afd6bd, 0x5c4be5b0, 0x588b9140, 0x5c6eb258, 0x58673e1b, 0x5c9170bf, 
   0x5842dd54, 0x5cb420e0, 0x581e6ef1, 0x5cd6c2b5, 0x57f9f2f8, 0x5cf95638, 
   0x57d5696d, 0x5d1bdb65, 0x57b0d256, 0x5d3e5237, 0x578c2dba, 0x5d60baa7, 
   0x57677b9d, 0x5d8314b1, 0x5742bc06, 0x5da5604f, 0x571deefa, 0x5dc79d7c, 
   0x56f9147e, 0x5de9cc33, 0x56d42c99, 0x5e0bec6e, 0x56af3750, 0x5e2dfe29, 
   0x568a34a9, 0x5e50015d, 0x566524aa, 0x5e71f606, 0x56400758, 0x5e93dc1f, 
   0x561adcb9, 0x5eb5b3a2, 0x55f5a4d2, 0x5ed77c8a, 0x55d05faa, 0x5ef936d1, 
   0x55ab0d46, 0x5f1ae274, 0x5585adad, 0x5f3c7f6b, 0x556040e2, 0x5f5e0db3, 
   0x553ac6ee, 0x5f7f8d46, 0x55153fd4, 0x5fa0fe1f, 0x54efab9c, 0x5fc26038, 
   0x54ca0a4b, 0x5fe3b38d, 0x54a45be6, 0x6004f819, 0x547ea073, 0x60262dd6, 
   0x5458d7f9, 0x604754bf, 0x5433027d, 0x60686ccf, 0x540d2005, 0x60897601, 
   0x53e73097, 0x60aa7050, 0x53c13439, 0x60cb5bb7, 0x539b2af0, 0x60ec3830, 
   0x537514c2, 0x610d05b7, 0x534ef1b5, 0x612dc447, 0x5328c1d0, 0x614e73da, 
   0x53028518, 0x616f146c, 0x52dc3b92, 0x618fa5f7, 0x52b5e546, 0x61b02876, 
   0x528f8238, 0x61d09be5, 0x5269126e, 0x61f1003f, 0x524295f0, 0x6211557e, 
   0x521c0cc2, 0x62319b9d, 0x51f576ea, 0x6251d298, 0x51ced46e, 0x6271fa69, 
   0x51a82555, 0x6292130c, 0x518169a5, 0x62b21c7b, 0x515aa162, 0x62d216b3, 
   0x5133cc94, 0x62f201ac, 0x510ceb40, 0x6311dd64, 0x50e5fd6d, 0x6331a9d4, 
   0x50bf031f, 0x635166f9, 0x5097fc5e, 0x637114cc, 0x5070e92f, 0x6390b34a, 
   0x5049c999, 0x63b0426d, 0x50229da1, 0x63cfc231, 0x4ffb654d, 0x63ef3290, 
   0x4fd420a4, 0x640e9386, 0x4faccfab, 0x642de50d, 0x4f857269, 0x644d2722, 
   0x4f5e08e3, 0x646c59bf, 0x4f369320, 0x648b7ce0, 0x4f0f1126, 0x64aa907f, 
   0x4ee782fb, 0x64c99498, 0x4ebfe8a5, 0x64e88926, 0x4e984229, 0x65076e25, 
   0x4e708f8f, 0x6526438f, 0x4e48d0dd, 0x6545095f, 0x4e210617, 0x6563bf92, 
   0x4df92f46, 0x65826622, 0x4dd14c6e, 0x65a0fd0b, 0x4da95d96, 0x65bf8447, 
   0x4d8162c4, 0x65ddfbd3, 0x4d595bfe, 0x65fc63a9, 0x4d31494b, 0x661abbc5, 
   0x4d092ab0, 0x66390422, 0x4ce10034, 0x66573cbb, 0x4cb8c9dd, 0x6675658c, 
   0x4c9087b1, 0x66937e91, 0x4c6839b7, 0x66b187c3, 0x4c3fdff4, 0x66cf8120, 
   0x4c177a6e, 0x66ed6aa1, 0x4bef092d, 0x670b4444, 0x4bc68c36, 0x67290e02, 
   0x4b9e0390, 0x6746c7d8, 0x4b756f40, 0x676471c0, 0x4b4ccf4d, 0x67820bb7, 
   0x4b2423be, 0x679f95b7, 0x4afb6c98, 0x67bd0fbd, 0x4ad2a9e2, 0x67da79c3, 
   0x4aa9dba2, 0x67f7d3c5, 0x4a8101de, 0x68151dbe, 0x4a581c9e, 0x683257ab, 
   0x4a2f2be6, 0x684f8186, 0x4a062fbd, 0x686c9b4b, 0x49dd282a, 0x6889a4f6, 
   0x49b41533, 0x68a69e81, 0x498af6df, 0x68c387e9, 0x4961cd33, 0x68e06129, 
   0x49389836, 0x68fd2a3d, 0x490f57ee, 0x6919e320, 0x48e60c62, 0x69368bce, 
   0x48bcb599, 0x69532442, 0x48935397, 0x696fac78, 0x4869e665, 0x698c246c, 
   0x48406e08, 0x69a88c19, 0x4816ea86, 0x69c4e37a, 0x47ed5be6, 0x69e12a8c, 
   0x47c3c22f, 0x69fd614a, 0x479a1d67, 0x6a1987b0, 0x47706d93, 0x6a359db9, 
   0x4746b2bc, 0x6a51a361, 0x471cece7, 0x6a6d98a4, 0x46f31c1a, 0x6a897d7d, 
   0x46c9405c, 0x6aa551e9, 0x469f59b4, 0x6ac115e2, 0x46756828, 0x6adcc964, 
   0x464b6bbe, 0x6af86c6c, 0x4621647d, 0x6b13fef5, 0x45f7526b, 0x6b2f80fb, 
   0x45cd358f, 0x6b4af279, 0x45a30df0, 0x6b66536b, 0x4578db93, 0x6b81a3cd, 
   0x454e9e80, 0x6b9ce39b, 0x452456bd, 0x6bb812d1, 0x44fa0450, 0x6bd3316a, 
   0x44cfa740, 0x6bee3f62, 0x44a53f93, 0x6c093cb6, 0x447acd50, 0x6c242960, 
   0x4450507e, 0x6c3f055d, 0x4425c923, 0x6c59d0a9, 0x43fb3746, 0x6c748b3f, 
   0x43d09aed, 0x6c8f351c, 0x43a5f41e, 0x6ca9ce3b, 0x437b42e1, 0x6cc45698, 
   0x4350873c, 0x6cdece2f, 0x4325c135, 0x6cf934fc, 0x42faf0d4, 0x6d138afb, 
   0x42d0161e, 0x6d2dd027, 0x42a5311b, 0x6d48047e, 0x427a41d0, 0x6d6227fa, 
   0x424f4845, 0x6d7c3a98, 0x42244481, 0x6d963c54, 0x41f93689, 0x6db02d29, 
   0x41ce1e65, 0x6dca0d14, 0x41a2fc1a, 0x6de3dc11, 0x4177cfb1, 0x6dfd9a1c, 
   0x414c992f, 0x6e174730, 0x4121589b, 0x6e30e34a, 0x40f60dfb, 0x6e4a6e66, 
   0x40cab958, 0x6e63e87f, 0x409f5ab6, 0x6e7d5193, 0x4073f21d, 0x6e96a99d, 
   0x40487f94, 0x6eaff099, 0x401d0321, 0x6ec92683, 0x3ff17cca, 0x6ee24b57, 
   0x3fc5ec98, 0x6efb5f12, 0x3f9a5290, 0x6f1461b0, 0x3f6eaeb8, 0x6f2d532c, 
   0x3f430119, 0x6f463383, 0x3f1749b8, 0x6f5f02b2, 0x3eeb889c, 0x6f77c0b3, 
   0x3ebfbdcd, 0x6f906d84, 0x3e93e950, 0x6fa90921, 0x3e680b2c, 0x6fc19385, 
   0x3e3c2369, 0x6fda0cae, 0x3e10320d, 0x6ff27497, 0x3de4371f, 0x700acb3c, 
   0x3db832a6, 0x7023109a, 0x3d8c24a8, 0x703b44ad, 0x3d600d2c, 0x70536771, 
   0x3d33ec39, 0x706b78e3, 0x3d07c1d6, 0x708378ff, 0x3cdb8e09, 0x709b67c0, 
   0x3caf50da, 0x70b34525, 0x3c830a50, 0x70cb1128, 0x3c56ba70, 0x70e2cbc6, 
   0x3c2a6142, 0x70fa74fc, 0x3bfdfecd, 0x71120cc5, 0x3bd19318, 0x7129931f, 
   0x3ba51e29, 0x71410805, 0x3b78a007, 0x71586b74, 0x3b4c18ba, 0x716fbd68, 
   0x3b1f8848, 0x7186fdde, 0x3af2eeb7, 0x719e2cd2, 0x3ac64c0f, 0x71b54a41, 
   0x3a99a057, 0x71cc5626, 0x3a6ceb96, 0x71e35080, 0x3a402dd2, 0x71fa3949, 
   0x3a136712, 0x7211107e, 0x39e6975e, 0x7227d61c, 0x39b9bebc, 0x723e8a20, 
   0x398cdd32, 0x72552c85, 0x395ff2c9, 0x726bbd48, 0x3932ff87, 0x72823c67, 
   0x39060373, 0x7298a9dd, 0x38d8fe93, 0x72af05a7, 0x38abf0ef, 0x72c54fc1, 
   0x387eda8e, 0x72db8828, 0x3851bb77, 0x72f1aed9, 0x382493b0, 0x7307c3d0, 
   0x37f76341, 0x731dc70a, 0x37ca2a30, 0x7333b883, 0x379ce885, 0x73499838, 
   0x376f9e46, 0x735f6626, 0x37424b7b, 0x73752249, 0x3714f02a, 0x738acc9e, 
   0x36e78c5b, 0x73a06522, 0x36ba2014, 0x73b5ebd1, 0x368cab5c, 0x73cb60a8, 
   0x365f2e3b, 0x73e0c3a3, 0x3631a8b8, 0x73f614c0, 0x36041ad9, 0x740b53fb, 
   0x35d684a6, 0x74208150, 0x35a8e625, 0x74359cbd, 0x357b3f5d, 0x744aa63f, 
   0x354d9057, 0x745f9dd1, 0x351fd918, 0x74748371, 0x34f219a8, 0x7489571c, 
   0x34c4520d, 0x749e18cd, 0x34968250, 0x74b2c884, 0x3468aa76, 0x74c7663a, 
   0x343aca87, 0x74dbf1ef, 0x340ce28b, 0x74f06b9e, 0x33def287, 0x7504d345, 
   0x33b0fa84, 0x751928e0, 0x3382fa88, 0x752d6c6c, 0x3354f29b, 0x75419de7, 
   0x3326e2c3, 0x7555bd4c, 0x32f8cb07, 0x7569ca99, 0x32caab6f, 0x757dc5ca, 
   0x329c8402, 0x7591aedd, 0x326e54c7, 0x75a585cf, 0x32401dc6, 0x75b94a9c, 
   0x3211df04, 0x75ccfd42, 0x31e39889, 0x75e09dbd, 0x31b54a5e, 0x75f42c0b, 
   0x3186f487, 0x7607a828, 0x3158970e, 0x761b1211, 0x312a31f8, 0x762e69c4, 
   0x30fbc54d, 0x7641af3d, 0x30cd5115, 0x7654e279, 0x309ed556, 0x76680376, 
   0x30705217, 0x767b1231, 0x3041c761, 0x768e0ea6, 0x30133539, 0x76a0f8d2, 
   0x2fe49ba7, 0x76b3d0b4, 0x2fb5fab2, 0x76c69647, 0x2f875262, 0x76d94989, 
   0x2f58a2be, 0x76ebea77, 0x2f29ebcc, 0x76fe790e, 0x2efb2d95, 0x7710f54c, 
   0x2ecc681e, 0x77235f2d, 0x2e9d9b70, 0x7735b6af, 0x2e6ec792, 0x7747fbce, 
   0x2e3fec8b, 0x775a2e89, 0x2e110a62, 0x776c4edb, 0x2de2211e, 0x777e5cc3, 
   0x2db330c7, 0x7790583e, 0x2d843964, 0x77a24148, 0x2d553afc, 0x77b417df, 
   0x2d263596, 0x77c5dc01, 0x2cf72939, 0x77d78daa, 0x2cc815ee, 0x77e92cd9, 
   0x2c98fbba, 0x77fab989, 0x2c69daa6, 0x780c33b8, 0x2c3ab2b9, 0x781d9b65, 
   0x2c0b83fa, 0x782ef08b, 0x2bdc4e6f, 0x78403329, 0x2bad1221, 0x7851633b, 
   0x2b7dcf17, 0x786280bf, 0x2b4e8558, 0x78738bb3, 0x2b1f34eb, 0x78848414, 
   0x2aefddd8, 0x789569df, 0x2ac08026, 0x78a63d11, 0x2a911bdc, 0x78b6fda8, 
   0x2a61b101, 0x78c7aba2, 0x2a323f9e, 0x78d846fb, 0x2a02c7b8, 0x78e8cfb2, 
   0x29d34958, 0x78f945c3, 0x29a3c485, 0x7909a92d, 0x29743946, 0x7919f9ec, 
   0x2944a7a2, 0x792a37fe, 0x29150fa1, 0x793a6361, 0x28e5714b, 0x794a7c12, 
   0x28b5cca5, 0x795a820e, 0x288621b9, 0x796a7554, 0x2856708d, 0x797a55e0, 
   0x2826b928, 0x798a23b1, 0x27f6fb92, 0x7999dec4, 0x27c737d3, 0x79a98715, 
   0x27976df1, 0x79b91ca4, 0x27679df4, 0x79c89f6e, 0x2737c7e3, 0x79d80f6f, 
   0x2707ebc7, 0x79e76ca7, 0x26d809a5, 0x79f6b711, 0x26a82186, 0x7a05eead, 
   0x26783370, 0x7a151378, 0x26483f6c, 0x7a24256f, 0x26184581, 0x7a332490, 
   0x25e845b6, 0x7a4210d8, 0x25b84012, 0x7a50ea47, 0x2588349d, 0x7a5fb0d8, 
   0x2558235f, 0x7a6e648a, 0x25280c5e, 0x7a7d055b, 0x24f7efa2, 0x7a8b9348, 
   0x24c7cd33, 0x7a9a0e50, 0x2497a517, 0x7aa8766f, 0x24677758, 0x7ab6cba4, 
   0x243743fa, 0x7ac50dec, 0x24070b08, 0x7ad33d45, 0x23d6cc87, 0x7ae159ae, 
   0x23a6887f, 0x7aef6323, 0x23763ef7, 0x7afd59a4, 0x2345eff8, 0x7b0b3d2c, 
   0x23159b88, 0x7b190dbc, 0x22e541af, 0x7b26cb4f, 0x22b4e274, 0x7b3475e5, 
   0x22847de0, 0x7b420d7a, 0x225413f8, 0x7b4f920e, 0x2223a4c5, 0x7b5d039e, 
   0x21f3304f, 0x7b6a6227, 0x21c2b69c, 0x7b77ada8, 0x219237b5, 0x7b84e61f, 
   0x2161b3a0, 0x7b920b89, 0x21312a65, 0x7b9f1de6, 0x21009c0c, 0x7bac1d31, 
   0x20d0089c, 0x7bb9096b, 0x209f701c, 0x7bc5e290, 0x206ed295, 0x7bd2a89e, 
   0x203e300d, 0x7bdf5b94, 0x200d888d, 0x7bebfb70, 0x1fdcdc1b, 0x7bf88830, 
   0x1fac2abf, 0x7c0501d2, 0x1f7b7481, 0x7c116853, 0x1f4ab968, 0x7c1dbbb3, 
   0x1f19f97b, 0x7c29fbee, 0x1ee934c3, 0x7c362904, 0x1eb86b46, 0x7c4242f2, 
   0x1e879d0d, 0x7c4e49b7, 0x1e56ca1e, 0x7c5a3d50, 0x1e25f282, 0x7c661dbc, 
   0x1df5163f, 0x7c71eaf9, 0x1dc4355e, 0x7c7da505, 0x1d934fe5, 0x7c894bde, 
   0x1d6265dd, 0x7c94df83, 0x1d31774d, 0x7ca05ff1, 0x1d00843d, 0x7cabcd28, 
   0x1ccf8cb3, 0x7cb72724, 0x1c9e90b8, 0x7cc26de5, 0x1c6d9053, 0x7ccda169, 
   0x1c3c8b8c, 0x7cd8c1ae, 0x1c0b826a, 0x7ce3ceb2, 0x1bda74f6, 0x7ceec873, 
   0x1ba96335, 0x7cf9aef0, 0x1b784d30, 0x7d048228, 0x1b4732ef, 0x7d0f4218, 
   0x1b161479, 0x7d19eebf, 0x1ae4f1d6, 0x7d24881b, 0x1ab3cb0d, 0x7d2f0e2b, 
   0x1a82a026, 0x7d3980ec, 0x1a517128, 0x7d43e05e, 0x1a203e1b, 0x7d4e2c7f, 
   0x19ef0707, 0x7d58654d, 0x19bdcbf3, 0x7d628ac6, 0x198c8ce7, 0x7d6c9ce9, 
   0x195b49ea, 0x7d769bb5, 0x192a0304, 0x7d808728, 0x18f8b83c, 0x7d8a5f40, 
   0x18c7699b, 0x7d9423fc, 0x18961728, 0x7d9dd55a, 0x1864c0ea, 0x7da77359, 
   0x183366e9, 0x7db0fdf8, 0x1802092c, 0x7dba7534, 0x17d0a7bc, 0x7dc3d90d, 
   0x179f429f, 0x7dcd2981, 0x176dd9de, 0x7dd6668f, 0x173c6d80, 0x7ddf9034, 
   0x170afd8d, 0x7de8a670, 0x16d98a0c, 0x7df1a942, 0x16a81305, 0x7dfa98a8, 
   0x1676987f, 0x7e0374a0, 0x16451a83, 0x7e0c3d29, 0x16139918, 0x7e14f242, 
   0x15e21445, 0x7e1d93ea, 0x15b08c12, 0x7e26221f, 0x157f0086, 0x7e2e9cdf, 
   0x154d71aa, 0x7e37042a, 0x151bdf86, 0x7e3f57ff, 0x14ea4a1f, 0x7e47985b, 
   0x14b8b17f, 0x7e4fc53e, 0x148715ae, 0x7e57dea7, 0x145576b1, 0x7e5fe493, 
   0x1423d492, 0x7e67d703, 0x13f22f58, 0x7e6fb5f4, 0x13c0870a, 0x7e778166, 
   0x138edbb1, 0x7e7f3957, 0x135d2d53, 0x7e86ddc6, 0x132b7bf9, 0x7e8e6eb2, 
   0x12f9c7aa, 0x7e95ec1a, 0x12c8106f, 0x7e9d55fc, 0x1296564d, 0x7ea4ac58, 
   0x1264994e, 0x7eabef2c, 0x1232d979, 0x7eb31e78, 0x120116d5, 0x7eba3a39, 
   0x11cf516a, 0x7ec14270, 0x119d8941, 0x7ec8371a, 0x116bbe60, 0x7ecf1837, 
   0x1139f0cf, 0x7ed5e5c6, 0x11082096, 0x7edc9fc6, 0x10d64dbd, 0x7ee34636, 
   0x10a4784b, 0x7ee9d914, 0x1072a048, 0x7ef05860, 0x1040c5bb, 0x7ef6c418, 
   0x100ee8ad, 0x7efd1c3c, 0xfdd0926, 0x7f0360cb, 0xfab272b, 0x7f0991c4, 
   0xf7942c7, 0x7f0faf25, 0xf475bff, 0x7f15b8ee, 0xf1572dc, 0x7f1baf1e, 
   0xee38766, 0x7f2191b4, 0xeb199a4, 0x7f2760af, 0xe7fa99e, 0x7f2d1c0e, 
   0xe4db75b, 0x7f32c3d1, 0xe1bc2e4, 0x7f3857f6, 0xde9cc40, 0x7f3dd87c, 
   0xdb7d376, 0x7f434563, 0xd85d88f, 0x7f489eaa, 0xd53db92, 0x7f4de451, 
   0xd21dc87, 0x7f531655, 0xcefdb76, 0x7f5834b7, 0xcbdd865, 0x7f5d3f75, 
   0xc8bd35e, 0x7f62368f, 0xc59cc68, 0x7f671a05, 0xc27c389, 0x7f6be9d4, 
   0xbf5b8cb, 0x7f70a5fe, 0xbc3ac35, 0x7f754e80, 0xb919dcf, 0x7f79e35a, 
   0xb5f8d9f, 0x7f7e648c, 0xb2d7baf, 0x7f82d214, 0xafb6805, 0x7f872bf3, 
   0xac952aa, 0x7f8b7227, 0xa973ba5, 0x7f8fa4b0, 0xa6522fe, 0x7f93c38c, 
   0xa3308bd, 0x7f97cebd, 0xa00ece8, 0x7f9bc640, 0x9cecf89, 0x7f9faa15, 
   0x99cb0a7, 0x7fa37a3c, 0x96a9049, 0x7fa736b4, 0x9386e78, 0x7faadf7c, 
   0x9064b3a, 0x7fae7495, 0x8d42699, 0x7fb1f5fc, 0x8a2009a, 0x7fb563b3, 
   0x86fd947, 0x7fb8bdb8, 0x83db0a7, 0x7fbc040a, 0x80b86c2, 0x7fbf36aa, 
   0x7d95b9e, 0x7fc25596, 0x7a72f45, 0x7fc560cf, 0x77501be, 0x7fc85854, 
   0x742d311, 0x7fcb3c23, 0x710a345, 0x7fce0c3e, 0x6de7262, 0x7fd0c8a3, 
   0x6ac406f, 0x7fd37153, 0x67a0d76, 0x7fd6064c, 0x647d97c, 0x7fd8878e, 
   0x615a48b, 0x7fdaf519, 0x5e36ea9, 0x7fdd4eec, 0x5b137df, 0x7fdf9508, 
   0x57f0035, 0x7fe1c76b, 0x54cc7b1, 0x7fe3e616, 0x51a8e5c, 0x7fe5f108, 
   0x4e8543e, 0x7fe7e841, 0x4b6195d, 0x7fe9cbc0, 0x483ddc3, 0x7feb9b85, 
   0x451a177, 0x7fed5791, 0x41f6480, 0x7feeffe1, 0x3ed26e6, 0x7ff09478, 
   0x3bae8b2, 0x7ff21553, 0x388a9ea, 0x7ff38274, 0x3566a96, 0x7ff4dbd9, 
   0x3242abf, 0x7ff62182, 0x2f1ea6c, 0x7ff75370, 0x2bfa9a4, 0x7ff871a2, 
   0x28d6870, 0x7ff97c18, 0x25b26d7, 0x7ffa72d1, 0x228e4e2, 0x7ffb55ce, 
   0x1f6a297, 0x7ffc250f, 0x1c45ffe, 0x7ffce093, 0x1921d20, 0x7ffd885a, 
   0x15fda03, 0x7ffe1c65, 0x12d96b1, 0x7ffe9cb2, 0xfb5330, 0x7fff0943, 
   0xc90f88, 0x7fff6216, 0x96cbc1, 0x7fffa72c, 0x6487e3, 0x7fffd886, 
   0x3243f5, 0x7ffff621, 0x0, 0x7fffffff, 0xffcdbc0b, 0x7ffff621, 
   0xff9b781d, 0x7fffd886, 0xff69343f, 0x7fffa72c, 0xff36f078, 0x7fff6216, 
   0xff04acd0, 0x7fff0943, 0xfed2694f, 0x7ffe9cb2, 0xfea025fd, 0x7ffe1c65, 
   0xfe6de2e0, 0x7ffd885a, 0xfe3ba002, 0x7ffce093, 0xfe095d69, 0x7ffc250f, 
   0xfdd71b1e, 0x7ffb55ce, 0xfda4d929, 0x7ffa72d1, 0xfd729790, 0x7ff97c18, 
   0xfd40565c, 0x7ff871a2, 0xfd0e1594, 0x7ff75370, 0xfcdbd541, 0x7ff62182, 
   0xfca9956a, 0x7ff4dbd9, 0xfc775616, 0x7ff38274, 0xfc45174e, 0x7ff21553, 
   0xfc12d91a, 0x7ff09478, 0xfbe09b80, 0x7feeffe1, 0xfbae5e89, 0x7fed5791, 
   0xfb7c223d, 0x7feb9b85, 0xfb49e6a3, 0x7fe9cbc0, 0xfb17abc2, 0x7fe7e841, 
   0xfae571a4, 0x7fe5f108, 0xfab3384f, 0x7fe3e616, 0xfa80ffcb, 0x7fe1c76b, 
   0xfa4ec821, 0x7fdf9508, 0xfa1c9157, 0x7fdd4eec, 0xf9ea5b75, 0x7fdaf519, 
   0xf9b82684, 0x7fd8878e, 0xf985f28a, 0x7fd6064c, 0xf953bf91, 0x7fd37153, 
   0xf9218d9e, 0x7fd0c8a3, 0xf8ef5cbb, 0x7fce0c3e, 0xf8bd2cef, 0x7fcb3c23, 
   0xf88afe42, 0x7fc85854, 0xf858d0bb, 0x7fc560cf, 0xf826a462, 0x7fc25596, 
   0xf7f4793e, 0x7fbf36aa, 0xf7c24f59, 0x7fbc040a, 0xf79026b9, 0x7fb8bdb8, 
   0xf75dff66, 0x7fb563b3, 0xf72bd967, 0x7fb1f5fc, 0xf6f9b4c6, 0x7fae7495, 
   0xf6c79188, 0x7faadf7c, 0xf6956fb7, 0x7fa736b4, 0xf6634f59, 0x7fa37a3c, 
   0xf6313077, 0x7f9faa15, 0xf5ff1318, 0x7f9bc640, 0xf5ccf743, 0x7f97cebd, 
   0xf59add02, 0x7f93c38c, 0xf568c45b, 0x7f8fa4b0, 0xf536ad56, 0x7f8b7227, 
   0xf50497fb, 0x7f872bf3, 0xf4d28451, 0x7f82d214, 0xf4a07261, 0x7f7e648c, 
   0xf46e6231, 0x7f79e35a, 0xf43c53cb, 0x7f754e80, 0xf40a4735, 0x7f70a5fe, 
   0xf3d83c77, 0x7f6be9d4, 0xf3a63398, 0x7f671a05, 0xf3742ca2, 0x7f62368f, 
   0xf342279b, 0x7f5d3f75, 0xf310248a, 0x7f5834b7, 0xf2de2379, 0x7f531655, 
   0xf2ac246e, 0x7f4de451, 0xf27a2771, 0x7f489eaa, 0xf2482c8a, 0x7f434563, 
   0xf21633c0, 0x7f3dd87c, 0xf1e43d1c, 0x7f3857f6, 0xf1b248a5, 0x7f32c3d1, 
   0xf1805662, 0x7f2d1c0e, 0xf14e665c, 0x7f2760af, 0xf11c789a, 0x7f2191b4, 
   0xf0ea8d24, 0x7f1baf1e, 0xf0b8a401, 0x7f15b8ee, 0xf086bd39, 0x7f0faf25, 
   0xf054d8d5, 0x7f0991c4, 0xf022f6da, 0x7f0360cb, 0xeff11753, 0x7efd1c3c, 
   0xefbf3a45, 0x7ef6c418, 0xef8d5fb8, 0x7ef05860, 0xef5b87b5, 0x7ee9d914, 
   0xef29b243, 0x7ee34636, 0xeef7df6a, 0x7edc9fc6, 0xeec60f31, 0x7ed5e5c6, 
   0xee9441a0, 0x7ecf1837, 0xee6276bf, 0x7ec8371a, 0xee30ae96, 0x7ec14270, 
   0xedfee92b, 0x7eba3a39, 0xedcd2687, 0x7eb31e78, 0xed9b66b2, 0x7eabef2c, 
   0xed69a9b3, 0x7ea4ac58, 0xed37ef91, 0x7e9d55fc, 0xed063856, 0x7e95ec1a, 
   0xecd48407, 0x7e8e6eb2, 0xeca2d2ad, 0x7e86ddc6, 0xec71244f, 0x7e7f3957, 
   0xec3f78f6, 0x7e778166, 0xec0dd0a8, 0x7e6fb5f4, 0xebdc2b6e, 0x7e67d703, 
   0xebaa894f, 0x7e5fe493, 0xeb78ea52, 0x7e57dea7, 0xeb474e81, 0x7e4fc53e, 
   0xeb15b5e1, 0x7e47985b, 0xeae4207a, 0x7e3f57ff, 0xeab28e56, 0x7e37042a, 
   0xea80ff7a, 0x7e2e9cdf, 0xea4f73ee, 0x7e26221f, 0xea1debbb, 0x7e1d93ea, 
   0xe9ec66e8, 0x7e14f242, 0xe9bae57d, 0x7e0c3d29, 0xe9896781, 0x7e0374a0, 
   0xe957ecfb, 0x7dfa98a8, 0xe92675f4, 0x7df1a942, 0xe8f50273, 0x7de8a670, 
   0xe8c39280, 0x7ddf9034, 0xe8922622, 0x7dd6668f, 0xe860bd61, 0x7dcd2981, 
   0xe82f5844, 0x7dc3d90d, 0xe7fdf6d4, 0x7dba7534, 0xe7cc9917, 0x7db0fdf8, 
   0xe79b3f16, 0x7da77359, 0xe769e8d8, 0x7d9dd55a, 0xe7389665, 0x7d9423fc, 
   0xe70747c4, 0x7d8a5f40, 0xe6d5fcfc, 0x7d808728, 0xe6a4b616, 0x7d769bb5, 
   0xe6737319, 0x7d6c9ce9, 0xe642340d, 0x7d628ac6, 0xe610f8f9, 0x7d58654d, 
   0xe5dfc1e5, 0x7d4e2c7f, 0xe5ae8ed8, 0x7d43e05e, 0xe57d5fda, 0x7d3980ec, 
   0xe54c34f3, 0x7d2f0e2b, 0xe51b0e2a, 0x7d24881b, 0xe4e9eb87, 0x7d19eebf, 
   0xe4b8cd11, 0x7d0f4218, 0xe487b2d0, 0x7d048228, 0xe4569ccb, 0x7cf9aef0, 
   0xe4258b0a, 0x7ceec873, 0xe3f47d96, 0x7ce3ceb2, 0xe3c37474, 0x7cd8c1ae, 
   0xe3926fad, 0x7ccda169, 0xe3616f48, 0x7cc26de5, 0xe330734d, 0x7cb72724, 
   0xe2ff7bc3, 0x7cabcd28, 0xe2ce88b3, 0x7ca05ff1, 0xe29d9a23, 0x7c94df83, 
   0xe26cb01b, 0x7c894bde, 0xe23bcaa2, 0x7c7da505, 0xe20ae9c1, 0x7c71eaf9, 
   0xe1da0d7e, 0x7c661dbc, 0xe1a935e2, 0x7c5a3d50, 0xe17862f3, 0x7c4e49b7, 
   0xe14794ba, 0x7c4242f2, 0xe116cb3d, 0x7c362904, 0xe0e60685, 0x7c29fbee, 
   0xe0b54698, 0x7c1dbbb3, 0xe0848b7f, 0x7c116853, 0xe053d541, 0x7c0501d2, 
   0xe02323e5, 0x7bf88830, 0xdff27773, 0x7bebfb70, 0xdfc1cff3, 0x7bdf5b94, 
   0xdf912d6b, 0x7bd2a89e, 0xdf608fe4, 0x7bc5e290, 0xdf2ff764, 0x7bb9096b, 
   0xdeff63f4, 0x7bac1d31, 0xdeced59b, 0x7b9f1de6, 0xde9e4c60, 0x7b920b89, 
   0xde6dc84b, 0x7b84e61f, 0xde3d4964, 0x7b77ada8, 0xde0ccfb1, 0x7b6a6227, 
   0xdddc5b3b, 0x7b5d039e, 0xddabec08, 0x7b4f920e, 0xdd7b8220, 0x7b420d7a, 
   0xdd4b1d8c, 0x7b3475e5, 0xdd1abe51, 0x7b26cb4f, 0xdcea6478, 0x7b190dbc, 
   0xdcba1008, 0x7b0b3d2c, 0xdc89c109, 0x7afd59a4, 0xdc597781, 0x7aef6323, 
   0xdc293379, 0x7ae159ae, 0xdbf8f4f8, 0x7ad33d45, 0xdbc8bc06, 0x7ac50dec, 
   0xdb9888a8, 0x7ab6cba4, 0xdb685ae9, 0x7aa8766f, 0xdb3832cd, 0x7a9a0e50, 
   0xdb08105e, 0x7a8b9348, 0xdad7f3a2, 0x7a7d055b, 0xdaa7dca1, 0x7a6e648a, 
   0xda77cb63, 0x7a5fb0d8, 0xda47bfee, 0x7a50ea47, 0xda17ba4a, 0x7a4210d8, 
   0xd9e7ba7f, 0x7a332490, 0xd9b7c094, 0x7a24256f, 0xd987cc90, 0x7a151378, 
   0xd957de7a, 0x7a05eead, 0xd927f65b, 0x79f6b711, 0xd8f81439, 0x79e76ca7, 
   0xd8c8381d, 0x79d80f6f, 0xd898620c, 0x79c89f6e, 0xd868920f, 0x79b91ca4, 
   0xd838c82d, 0x79a98715, 0xd809046e, 0x7999dec4, 0xd7d946d8, 0x798a23b1, 
   0xd7a98f73, 0x797a55e0, 0xd779de47, 0x796a7554, 0xd74a335b, 0x795a820e, 
   0xd71a8eb5, 0x794a7c12, 0xd6eaf05f, 0x793a6361, 0xd6bb585e, 0x792a37fe, 
   0xd68bc6ba, 0x7919f9ec, 0xd65c3b7b, 0x7909a92d, 0xd62cb6a8, 0x78f945c3, 
   0xd5fd3848, 0x78e8cfb2, 0xd5cdc062, 0x78d846fb, 0xd59e4eff, 0x78c7aba2, 
   0xd56ee424, 0x78b6fda8, 0xd53f7fda, 0x78a63d11, 0xd5102228, 0x789569df, 
   0xd4e0cb15, 0x78848414, 0xd4b17aa8, 0x78738bb3, 0xd48230e9, 0x786280bf, 
   0xd452eddf, 0x7851633b, 0xd423b191, 0x78403329, 0xd3f47c06, 0x782ef08b, 
   0xd3c54d47, 0x781d9b65, 0xd396255a, 0x780c33b8, 0xd3670446, 0x77fab989, 
   0xd337ea12, 0x77e92cd9, 0xd308d6c7, 0x77d78daa, 0xd2d9ca6a, 0x77c5dc01, 
   0xd2aac504, 0x77b417df, 0xd27bc69c, 0x77a24148, 0xd24ccf39, 0x7790583e, 
   0xd21ddee2, 0x777e5cc3, 0xd1eef59e, 0x776c4edb, 0xd1c01375, 0x775a2e89, 
   0xd191386e, 0x7747fbce, 0xd1626490, 0x7735b6af, 0xd13397e2, 0x77235f2d, 
   0xd104d26b, 0x7710f54c, 0xd0d61434, 0x76fe790e, 0xd0a75d42, 0x76ebea77, 
   0xd078ad9e, 0x76d94989, 0xd04a054e, 0x76c69647, 0xd01b6459, 0x76b3d0b4, 
   0xcfeccac7, 0x76a0f8d2, 0xcfbe389f, 0x768e0ea6, 0xcf8fade9, 0x767b1231, 
   0xcf612aaa, 0x76680376, 0xcf32aeeb, 0x7654e279, 0xcf043ab3, 0x7641af3d, 
   0xced5ce08, 0x762e69c4, 0xcea768f2, 0x761b1211, 0xce790b79, 0x7607a828, 
   0xce4ab5a2, 0x75f42c0b, 0xce1c6777, 0x75e09dbd, 0xcdee20fc, 0x75ccfd42, 
   0xcdbfe23a, 0x75b94a9c, 0xcd91ab39, 0x75a585cf, 0xcd637bfe, 0x7591aedd, 
   0xcd355491, 0x757dc5ca, 0xcd0734f9, 0x7569ca99, 0xccd91d3d, 0x7555bd4c, 
   0xccab0d65, 0x75419de7, 0xcc7d0578, 0x752d6c6c, 0xcc4f057c, 0x751928e0, 
   0xcc210d79, 0x7504d345, 0xcbf31d75, 0x74f06b9e, 0xcbc53579, 0x74dbf1ef, 
   0xcb97558a, 0x74c7663a, 0xcb697db0, 0x74b2c884, 0xcb3badf3, 0x749e18cd, 
   0xcb0de658, 0x7489571c, 0xcae026e8, 0x74748371, 0xcab26fa9, 0x745f9dd1, 
   0xca84c0a3, 0x744aa63f, 0xca5719db, 0x74359cbd, 0xca297b5a, 0x74208150, 
   0xc9fbe527, 0x740b53fb, 0xc9ce5748, 0x73f614c0, 0xc9a0d1c5, 0x73e0c3a3, 
   0xc97354a4, 0x73cb60a8, 0xc945dfec, 0x73b5ebd1, 0xc91873a5, 0x73a06522, 
   0xc8eb0fd6, 0x738acc9e, 0xc8bdb485, 0x73752249, 0xc89061ba, 0x735f6626, 
   0xc863177b, 0x73499838, 0xc835d5d0, 0x7333b883, 0xc8089cbf, 0x731dc70a, 
   0xc7db6c50, 0x7307c3d0, 0xc7ae4489, 0x72f1aed9, 0xc7812572, 0x72db8828, 
   0xc7540f11, 0x72c54fc1, 0xc727016d, 0x72af05a7, 0xc6f9fc8d, 0x7298a9dd, 
   0xc6cd0079, 0x72823c67, 0xc6a00d37, 0x726bbd48, 0xc67322ce, 0x72552c85, 
   0xc6464144, 0x723e8a20, 0xc61968a2, 0x7227d61c, 0xc5ec98ee, 0x7211107e, 
   0xc5bfd22e, 0x71fa3949, 0xc593146a, 0x71e35080, 0xc5665fa9, 0x71cc5626, 
   0xc539b3f1, 0x71b54a41, 0xc50d1149, 0x719e2cd2, 0xc4e077b8, 0x7186fdde, 
   0xc4b3e746, 0x716fbd68, 0xc4875ff9, 0x71586b74, 0xc45ae1d7, 0x71410805, 
   0xc42e6ce8, 0x7129931f, 0xc4020133, 0x71120cc5, 0xc3d59ebe, 0x70fa74fc, 
   0xc3a94590, 0x70e2cbc6, 0xc37cf5b0, 0x70cb1128, 0xc350af26, 0x70b34525, 
   0xc32471f7, 0x709b67c0, 0xc2f83e2a, 0x708378ff, 0xc2cc13c7, 0x706b78e3, 
   0xc29ff2d4, 0x70536771, 0xc273db58, 0x703b44ad, 0xc247cd5a, 0x7023109a, 
   0xc21bc8e1, 0x700acb3c, 0xc1efcdf3, 0x6ff27497, 0xc1c3dc97, 0x6fda0cae, 
   0xc197f4d4, 0x6fc19385, 0xc16c16b0, 0x6fa90921, 0xc1404233, 0x6f906d84, 
   0xc1147764, 0x6f77c0b3, 0xc0e8b648, 0x6f5f02b2, 0xc0bcfee7, 0x6f463383, 
   0xc0915148, 0x6f2d532c, 0xc065ad70, 0x6f1461b0, 0xc03a1368, 0x6efb5f12, 
   0xc00e8336, 0x6ee24b57, 0xbfe2fcdf, 0x6ec92683, 0xbfb7806c, 0x6eaff099, 
   0xbf8c0de3, 0x6e96a99d, 0xbf60a54a, 0x6e7d5193, 0xbf3546a8, 0x6e63e87f, 
   0xbf09f205, 0x6e4a6e66, 0xbedea765, 0x6e30e34a, 0xbeb366d1, 0x6e174730, 
   0xbe88304f, 0x6dfd9a1c, 0xbe5d03e6, 0x6de3dc11, 0xbe31e19b, 0x6dca0d14, 
   0xbe06c977, 0x6db02d29, 0xbddbbb7f, 0x6d963c54, 0xbdb0b7bb, 0x6d7c3a98, 
   0xbd85be30, 0x6d6227fa, 0xbd5acee5, 0x6d48047e, 0xbd2fe9e2, 0x6d2dd027, 
   0xbd050f2c, 0x6d138afb, 0xbcda3ecb, 0x6cf934fc, 0xbcaf78c4, 0x6cdece2f, 
   0xbc84bd1f, 0x6cc45698, 0xbc5a0be2, 0x6ca9ce3b, 0xbc2f6513, 0x6c8f351c, 
   0xbc04c8ba, 0x6c748b3f, 0xbbda36dd, 0x6c59d0a9, 0xbbafaf82, 0x6c3f055d, 
   0xbb8532b0, 0x6c242960, 0xbb5ac06d, 0x6c093cb6, 0xbb3058c0, 0x6bee3f62, 
   0xbb05fbb0, 0x6bd3316a, 0xbadba943, 0x6bb812d1, 0xbab16180, 0x6b9ce39b, 
   0xba87246d, 0x6b81a3cd, 0xba5cf210, 0x6b66536b, 0xba32ca71, 0x6b4af279, 
   0xba08ad95, 0x6b2f80fb, 0xb9de9b83, 0x6b13fef5, 0xb9b49442, 0x6af86c6c, 
   0xb98a97d8, 0x6adcc964, 0xb960a64c, 0x6ac115e2, 0xb936bfa4, 0x6aa551e9, 
   0xb90ce3e6, 0x6a897d7d, 0xb8e31319, 0x6a6d98a4, 0xb8b94d44, 0x6a51a361, 
   0xb88f926d, 0x6a359db9, 0xb865e299, 0x6a1987b0, 0xb83c3dd1, 0x69fd614a, 
   0xb812a41a, 0x69e12a8c, 0xb7e9157a, 0x69c4e37a, 0xb7bf91f8, 0x69a88c19, 
   0xb796199b, 0x698c246c, 0xb76cac69, 0x696fac78, 0xb7434a67, 0x69532442, 
   0xb719f39e, 0x69368bce, 0xb6f0a812, 0x6919e320, 0xb6c767ca, 0x68fd2a3d, 
   0xb69e32cd, 0x68e06129, 0xb6750921, 0x68c387e9, 0xb64beacd, 0x68a69e81, 
   0xb622d7d6, 0x6889a4f6, 0xb5f9d043, 0x686c9b4b, 0xb5d0d41a, 0x684f8186, 
   0xb5a7e362, 0x683257ab, 0xb57efe22, 0x68151dbe, 0xb556245e, 0x67f7d3c5, 
   0xb52d561e, 0x67da79c3, 0xb5049368, 0x67bd0fbd, 0xb4dbdc42, 0x679f95b7, 
   0xb4b330b3, 0x67820bb7, 0xb48a90c0, 0x676471c0, 0xb461fc70, 0x6746c7d8, 
   0xb43973ca, 0x67290e02, 0xb410f6d3, 0x670b4444, 0xb3e88592, 0x66ed6aa1, 
   0xb3c0200c, 0x66cf8120, 0xb397c649, 0x66b187c3, 0xb36f784f, 0x66937e91, 
   0xb3473623, 0x6675658c, 0xb31effcc, 0x66573cbb, 0xb2f6d550, 0x66390422, 
   0xb2ceb6b5, 0x661abbc5, 0xb2a6a402, 0x65fc63a9, 0xb27e9d3c, 0x65ddfbd3, 
   0xb256a26a, 0x65bf8447, 0xb22eb392, 0x65a0fd0b, 0xb206d0ba, 0x65826622, 
   0xb1def9e9, 0x6563bf92, 0xb1b72f23, 0x6545095f, 0xb18f7071, 0x6526438f, 
   0xb167bdd7, 0x65076e25, 0xb140175b, 0x64e88926, 0xb1187d05, 0x64c99498, 
   0xb0f0eeda, 0x64aa907f, 0xb0c96ce0, 0x648b7ce0, 0xb0a1f71d, 0x646c59bf, 
   0xb07a8d97, 0x644d2722, 0xb0533055, 0x642de50d, 0xb02bdf5c, 0x640e9386, 
   0xb0049ab3, 0x63ef3290, 0xafdd625f, 0x63cfc231, 0xafb63667, 0x63b0426d, 
   0xaf8f16d1, 0x6390b34a, 0xaf6803a2, 0x637114cc, 0xaf40fce1, 0x635166f9, 
   0xaf1a0293, 0x6331a9d4, 0xaef314c0, 0x6311dd64, 0xaecc336c, 0x62f201ac, 
   0xaea55e9e, 0x62d216b3, 0xae7e965b, 0x62b21c7b, 0xae57daab, 0x6292130c, 
   0xae312b92, 0x6271fa69, 0xae0a8916, 0x6251d298, 0xade3f33e, 0x62319b9d, 
   0xadbd6a10, 0x6211557e, 0xad96ed92, 0x61f1003f, 0xad707dc8, 0x61d09be5, 
   0xad4a1aba, 0x61b02876, 0xad23c46e, 0x618fa5f7, 0xacfd7ae8, 0x616f146c, 
   0xacd73e30, 0x614e73da, 0xacb10e4b, 0x612dc447, 0xac8aeb3e, 0x610d05b7, 
   0xac64d510, 0x60ec3830, 0xac3ecbc7, 0x60cb5bb7, 0xac18cf69, 0x60aa7050, 
   0xabf2dffb, 0x60897601, 0xabccfd83, 0x60686ccf, 0xaba72807, 0x604754bf, 
   0xab815f8d, 0x60262dd6, 0xab5ba41a, 0x6004f819, 0xab35f5b5, 0x5fe3b38d, 
   0xab105464, 0x5fc26038, 0xaaeac02c, 0x5fa0fe1f, 0xaac53912, 0x5f7f8d46, 
   0xaa9fbf1e, 0x5f5e0db3, 0xaa7a5253, 0x5f3c7f6b, 0xaa54f2ba, 0x5f1ae274, 
   0xaa2fa056, 0x5ef936d1, 0xaa0a5b2e, 0x5ed77c8a, 0xa9e52347, 0x5eb5b3a2, 
   0xa9bff8a8, 0x5e93dc1f, 0xa99adb56, 0x5e71f606, 0xa975cb57, 0x5e50015d, 
   0xa950c8b0, 0x5e2dfe29, 0xa92bd367, 0x5e0bec6e, 0xa906eb82, 0x5de9cc33, 
   0xa8e21106, 0x5dc79d7c, 0xa8bd43fa, 0x5da5604f, 0xa8988463, 0x5d8314b1, 
   0xa873d246, 0x5d60baa7, 0xa84f2daa, 0x5d3e5237, 0xa82a9693, 0x5d1bdb65, 
   0xa8060d08, 0x5cf95638, 0xa7e1910f, 0x5cd6c2b5, 0xa7bd22ac, 0x5cb420e0, 
   0xa798c1e5, 0x5c9170bf, 0xa7746ec0, 0x5c6eb258, 0xa7502943, 0x5c4be5b0, 
   0xa72bf174, 0x5c290acc, 0xa707c757, 0x5c0621b2, 0xa6e3aaf2, 0x5be32a67, 
   0xa6bf9c4b, 0x5bc024f0, 0xa69b9b68, 0x5b9d1154, 0xa677a84e, 0x5b79ef96, 
   0xa653c303, 0x5b56bfbd, 0xa62feb8b, 0x5b3381ce, 0xa60c21ee, 0x5b1035cf, 
   0xa5e8662f, 0x5aecdbc5, 0xa5c4b855, 0x5ac973b5, 0xa5a11866, 0x5aa5fda5, 
   0xa57d8666, 0x5a82799a, 0xa55a025b, 0x5a5ee79a, 0xa5368c4b, 0x5a3b47ab, 
   0xa513243b, 0x5a1799d1, 0xa4efca31, 0x59f3de12, 0xa4cc7e32, 0x59d01475, 
   0xa4a94043, 0x59ac3cfd, 0xa486106a, 0x598857b2, 0xa462eeac, 0x59646498, 
   0xa43fdb10, 0x594063b5, 0xa41cd599, 0x591c550e, 0xa3f9de4e, 0x58f838a9, 
   0xa3d6f534, 0x58d40e8c, 0xa3b41a50, 0x58afd6bd, 0xa3914da8, 0x588b9140, 
   0xa36e8f41, 0x58673e1b, 0xa34bdf20, 0x5842dd54, 0xa3293d4b, 0x581e6ef1, 
   0xa306a9c8, 0x57f9f2f8, 0xa2e4249b, 0x57d5696d, 0xa2c1adc9, 0x57b0d256, 
   0xa29f4559, 0x578c2dba, 0xa27ceb4f, 0x57677b9d, 0xa25a9fb1, 0x5742bc06, 
   0xa2386284, 0x571deefa, 0xa21633cd, 0x56f9147e, 0xa1f41392, 0x56d42c99, 
   0xa1d201d7, 0x56af3750, 0xa1affea3, 0x568a34a9, 0xa18e09fa, 0x566524aa, 
   0xa16c23e1, 0x56400758, 0xa14a4c5e, 0x561adcb9, 0xa1288376, 0x55f5a4d2, 
   0xa106c92f, 0x55d05faa, 0xa0e51d8c, 0x55ab0d46, 0xa0c38095, 0x5585adad, 
   0xa0a1f24d, 0x556040e2, 0xa08072ba, 0x553ac6ee, 0xa05f01e1, 0x55153fd4, 
   0xa03d9fc8, 0x54efab9c, 0xa01c4c73, 0x54ca0a4b, 0x9ffb07e7, 0x54a45be6, 
   0x9fd9d22a, 0x547ea073, 0x9fb8ab41, 0x5458d7f9, 0x9f979331, 0x5433027d, 
   0x9f7689ff, 0x540d2005, 0x9f558fb0, 0x53e73097, 0x9f34a449, 0x53c13439, 
   0x9f13c7d0, 0x539b2af0, 0x9ef2fa49, 0x537514c2, 0x9ed23bb9, 0x534ef1b5, 
   0x9eb18c26, 0x5328c1d0, 0x9e90eb94, 0x53028518, 0x9e705a09, 0x52dc3b92, 
   0x9e4fd78a, 0x52b5e546, 0x9e2f641b, 0x528f8238, 0x9e0effc1, 0x5269126e, 
   0x9deeaa82, 0x524295f0, 0x9dce6463, 0x521c0cc2, 0x9dae2d68, 0x51f576ea, 
   0x9d8e0597, 0x51ced46e, 0x9d6decf4, 0x51a82555, 0x9d4de385, 0x518169a5, 
   0x9d2de94d, 0x515aa162, 0x9d0dfe54, 0x5133cc94, 0x9cee229c, 0x510ceb40, 
   0x9cce562c, 0x50e5fd6d, 0x9cae9907, 0x50bf031f, 0x9c8eeb34, 0x5097fc5e, 
   0x9c6f4cb6, 0x5070e92f, 0x9c4fbd93, 0x5049c999, 0x9c303dcf, 0x50229da1, 
   0x9c10cd70, 0x4ffb654d, 0x9bf16c7a, 0x4fd420a4, 0x9bd21af3, 0x4faccfab, 
   0x9bb2d8de, 0x4f857269, 0x9b93a641, 0x4f5e08e3, 0x9b748320, 0x4f369320, 
   0x9b556f81, 0x4f0f1126, 0x9b366b68, 0x4ee782fb, 0x9b1776da, 0x4ebfe8a5, 
   0x9af891db, 0x4e984229, 0x9ad9bc71, 0x4e708f8f, 0x9abaf6a1, 0x4e48d0dd, 
   0x9a9c406e, 0x4e210617, 0x9a7d99de, 0x4df92f46, 0x9a5f02f5, 0x4dd14c6e, 
   0x9a407bb9, 0x4da95d96, 0x9a22042d, 0x4d8162c4, 0x9a039c57, 0x4d595bfe, 
   0x99e5443b, 0x4d31494b, 0x99c6fbde, 0x4d092ab0, 0x99a8c345, 0x4ce10034, 
   0x998a9a74, 0x4cb8c9dd, 0x996c816f, 0x4c9087b1, 0x994e783d, 0x4c6839b7, 
   0x99307ee0, 0x4c3fdff4, 0x9912955f, 0x4c177a6e, 0x98f4bbbc, 0x4bef092d, 
   0x98d6f1fe, 0x4bc68c36, 0x98b93828, 0x4b9e0390, 0x989b8e40, 0x4b756f40, 
   0x987df449, 0x4b4ccf4d, 0x98606a49, 0x4b2423be, 0x9842f043, 0x4afb6c98, 
   0x9825863d, 0x4ad2a9e2, 0x98082c3b, 0x4aa9dba2, 0x97eae242, 0x4a8101de, 
   0x97cda855, 0x4a581c9e, 0x97b07e7a, 0x4a2f2be6, 0x979364b5, 0x4a062fbd, 
   0x97765b0a, 0x49dd282a, 0x9759617f, 0x49b41533, 0x973c7817, 0x498af6df, 
   0x971f9ed7, 0x4961cd33, 0x9702d5c3, 0x49389836, 0x96e61ce0, 0x490f57ee, 
   0x96c97432, 0x48e60c62, 0x96acdbbe, 0x48bcb599, 0x96905388, 0x48935397, 
   0x9673db94, 0x4869e665, 0x965773e7, 0x48406e08, 0x963b1c86, 0x4816ea86, 
   0x961ed574, 0x47ed5be6, 0x96029eb6, 0x47c3c22f, 0x95e67850, 0x479a1d67, 
   0x95ca6247, 0x47706d93, 0x95ae5c9f, 0x4746b2bc, 0x9592675c, 0x471cece7, 
   0x95768283, 0x46f31c1a, 0x955aae17, 0x46c9405c, 0x953eea1e, 0x469f59b4, 
   0x9523369c, 0x46756828, 0x95079394, 0x464b6bbe, 0x94ec010b, 0x4621647d, 
   0x94d07f05, 0x45f7526b, 0x94b50d87, 0x45cd358f, 0x9499ac95, 0x45a30df0, 
   0x947e5c33, 0x4578db93, 0x94631c65, 0x454e9e80, 0x9447ed2f, 0x452456bd, 
   0x942cce96, 0x44fa0450, 0x9411c09e, 0x44cfa740, 0x93f6c34a, 0x44a53f93, 
   0x93dbd6a0, 0x447acd50, 0x93c0faa3, 0x4450507e, 0x93a62f57, 0x4425c923, 
   0x938b74c1, 0x43fb3746, 0x9370cae4, 0x43d09aed, 0x935631c5, 0x43a5f41e, 
   0x933ba968, 0x437b42e1, 0x932131d1, 0x4350873c, 0x9306cb04, 0x4325c135, 
   0x92ec7505, 0x42faf0d4, 0x92d22fd9, 0x42d0161e, 0x92b7fb82, 0x42a5311b, 
   0x929dd806, 0x427a41d0, 0x9283c568, 0x424f4845, 0x9269c3ac, 0x42244481, 
   0x924fd2d7, 0x41f93689, 0x9235f2ec, 0x41ce1e65, 0x921c23ef, 0x41a2fc1a, 
   0x920265e4, 0x4177cfb1, 0x91e8b8d0, 0x414c992f, 0x91cf1cb6, 0x4121589b, 
   0x91b5919a, 0x40f60dfb, 0x919c1781, 0x40cab958, 0x9182ae6d, 0x409f5ab6, 
   0x91695663, 0x4073f21d, 0x91500f67, 0x40487f94, 0x9136d97d, 0x401d0321, 
   0x911db4a9, 0x3ff17cca, 0x9104a0ee, 0x3fc5ec98, 0x90eb9e50, 0x3f9a5290, 
   0x90d2acd4, 0x3f6eaeb8, 0x90b9cc7d, 0x3f430119, 0x90a0fd4e, 0x3f1749b8, 
   0x90883f4d, 0x3eeb889c, 0x906f927c, 0x3ebfbdcd, 0x9056f6df, 0x3e93e950, 
   0x903e6c7b, 0x3e680b2c, 0x9025f352, 0x3e3c2369, 0x900d8b69, 0x3e10320d, 
   0x8ff534c4, 0x3de4371f, 0x8fdcef66, 0x3db832a6, 0x8fc4bb53, 0x3d8c24a8, 
   0x8fac988f, 0x3d600d2c, 0x8f94871d, 0x3d33ec39, 0x8f7c8701, 0x3d07c1d6, 
   0x8f649840, 0x3cdb8e09, 0x8f4cbadb, 0x3caf50da, 0x8f34eed8, 0x3c830a50, 
   0x8f1d343a, 0x3c56ba70, 0x8f058b04, 0x3c2a6142, 0x8eedf33b, 0x3bfdfecd, 
   0x8ed66ce1, 0x3bd19318, 0x8ebef7fb, 0x3ba51e29, 0x8ea7948c, 0x3b78a007, 
   0x8e904298, 0x3b4c18ba, 0x8e790222, 0x3b1f8848, 0x8e61d32e, 0x3af2eeb7, 
   0x8e4ab5bf, 0x3ac64c0f, 0x8e33a9da, 0x3a99a057, 0x8e1caf80, 0x3a6ceb96, 
   0x8e05c6b7, 0x3a402dd2, 0x8deeef82, 0x3a136712, 0x8dd829e4, 0x39e6975e, 
   0x8dc175e0, 0x39b9bebc, 0x8daad37b, 0x398cdd32, 0x8d9442b8, 0x395ff2c9, 
   0x8d7dc399, 0x3932ff87, 0x8d675623, 0x39060373, 0x8d50fa59, 0x38d8fe93, 
   0x8d3ab03f, 0x38abf0ef, 0x8d2477d8, 0x387eda8e, 0x8d0e5127, 0x3851bb77, 
   0x8cf83c30, 0x382493b0, 0x8ce238f6, 0x37f76341, 0x8ccc477d, 0x37ca2a30, 
   0x8cb667c8, 0x379ce885, 0x8ca099da, 0x376f9e46, 0x8c8addb7, 0x37424b7b, 
   0x8c753362, 0x3714f02a, 0x8c5f9ade, 0x36e78c5b, 0x8c4a142f, 0x36ba2014, 
   0x8c349f58, 0x368cab5c, 0x8c1f3c5d, 0x365f2e3b, 0x8c09eb40, 0x3631a8b8, 
   0x8bf4ac05, 0x36041ad9, 0x8bdf7eb0, 0x35d684a6, 0x8bca6343, 0x35a8e625, 
   0x8bb559c1, 0x357b3f5d, 0x8ba0622f, 0x354d9057, 0x8b8b7c8f, 0x351fd918, 
   0x8b76a8e4, 0x34f219a8, 0x8b61e733, 0x34c4520d, 0x8b4d377c, 0x34968250, 
   0x8b3899c6, 0x3468aa76, 0x8b240e11, 0x343aca87, 0x8b0f9462, 0x340ce28b, 
   0x8afb2cbb, 0x33def287, 0x8ae6d720, 0x33b0fa84, 0x8ad29394, 0x3382fa88, 
   0x8abe6219, 0x3354f29b, 0x8aaa42b4, 0x3326e2c3, 0x8a963567, 0x32f8cb07, 
   0x8a823a36, 0x32caab6f, 0x8a6e5123, 0x329c8402, 0x8a5a7a31, 0x326e54c7, 
   0x8a46b564, 0x32401dc6, 0x8a3302be, 0x3211df04, 0x8a1f6243, 0x31e39889, 
   0x8a0bd3f5, 0x31b54a5e, 0x89f857d8, 0x3186f487, 0x89e4edef, 0x3158970e, 
   0x89d1963c, 0x312a31f8, 0x89be50c3, 0x30fbc54d, 0x89ab1d87, 0x30cd5115, 
   0x8997fc8a, 0x309ed556, 0x8984edcf, 0x30705217, 0x8971f15a, 0x3041c761, 
   0x895f072e, 0x30133539, 0x894c2f4c, 0x2fe49ba7, 0x893969b9, 0x2fb5fab2, 
   0x8926b677, 0x2f875262, 0x89141589, 0x2f58a2be, 0x890186f2, 0x2f29ebcc, 
   0x88ef0ab4, 0x2efb2d95, 0x88dca0d3, 0x2ecc681e, 0x88ca4951, 0x2e9d9b70, 
   0x88b80432, 0x2e6ec792, 0x88a5d177, 0x2e3fec8b, 0x8893b125, 0x2e110a62, 
   0x8881a33d, 0x2de2211e, 0x886fa7c2, 0x2db330c7, 0x885dbeb8, 0x2d843964, 
   0x884be821, 0x2d553afc, 0x883a23ff, 0x2d263596, 0x88287256, 0x2cf72939, 
   0x8816d327, 0x2cc815ee, 0x88054677, 0x2c98fbba, 0x87f3cc48, 0x2c69daa6, 
   0x87e2649b, 0x2c3ab2b9, 0x87d10f75, 0x2c0b83fa, 0x87bfccd7, 0x2bdc4e6f, 
   0x87ae9cc5, 0x2bad1221, 0x879d7f41, 0x2b7dcf17, 0x878c744d, 0x2b4e8558, 
   0x877b7bec, 0x2b1f34eb, 0x876a9621, 0x2aefddd8, 0x8759c2ef, 0x2ac08026, 
   0x87490258, 0x2a911bdc, 0x8738545e, 0x2a61b101, 0x8727b905, 0x2a323f9e, 
   0x8717304e, 0x2a02c7b8, 0x8706ba3d, 0x29d34958, 0x86f656d3, 0x29a3c485, 
   0x86e60614, 0x29743946, 0x86d5c802, 0x2944a7a2, 0x86c59c9f, 0x29150fa1, 
   0x86b583ee, 0x28e5714b, 0x86a57df2, 0x28b5cca5, 0x86958aac, 0x288621b9, 
   0x8685aa20, 0x2856708d, 0x8675dc4f, 0x2826b928, 0x8666213c, 0x27f6fb92, 
   0x865678eb, 0x27c737d3, 0x8646e35c, 0x27976df1, 0x86376092, 0x27679df4, 
   0x8627f091, 0x2737c7e3, 0x86189359, 0x2707ebc7, 0x860948ef, 0x26d809a5, 
   0x85fa1153, 0x26a82186, 0x85eaec88, 0x26783370, 0x85dbda91, 0x26483f6c, 
   0x85ccdb70, 0x26184581, 0x85bdef28, 0x25e845b6, 0x85af15b9, 0x25b84012, 
   0x85a04f28, 0x2588349d, 0x85919b76, 0x2558235f, 0x8582faa5, 0x25280c5e, 
   0x85746cb8, 0x24f7efa2, 0x8565f1b0, 0x24c7cd33, 0x85578991, 0x2497a517, 
   0x8549345c, 0x24677758, 0x853af214, 0x243743fa, 0x852cc2bb, 0x24070b08, 
   0x851ea652, 0x23d6cc87, 0x85109cdd, 0x23a6887f, 0x8502a65c, 0x23763ef7, 
   0x84f4c2d4, 0x2345eff8, 0x84e6f244, 0x23159b88, 0x84d934b1, 0x22e541af, 
   0x84cb8a1b, 0x22b4e274, 0x84bdf286, 0x22847de0, 0x84b06df2, 0x225413f8, 
   0x84a2fc62, 0x2223a4c5, 0x84959dd9, 0x21f3304f, 0x84885258, 0x21c2b69c, 
   0x847b19e1, 0x219237b5, 0x846df477, 0x2161b3a0, 0x8460e21a, 0x21312a65, 
   0x8453e2cf, 0x21009c0c, 0x8446f695, 0x20d0089c, 0x843a1d70, 0x209f701c, 
   0x842d5762, 0x206ed295, 0x8420a46c, 0x203e300d, 0x84140490, 0x200d888d, 
   0x840777d0, 0x1fdcdc1b, 0x83fafe2e, 0x1fac2abf, 0x83ee97ad, 0x1f7b7481, 
   0x83e2444d, 0x1f4ab968, 0x83d60412, 0x1f19f97b, 0x83c9d6fc, 0x1ee934c3, 
   0x83bdbd0e, 0x1eb86b46, 0x83b1b649, 0x1e879d0d, 0x83a5c2b0, 0x1e56ca1e, 
   0x8399e244, 0x1e25f282, 0x838e1507, 0x1df5163f, 0x83825afb, 0x1dc4355e, 
   0x8376b422, 0x1d934fe5, 0x836b207d, 0x1d6265dd, 0x835fa00f, 0x1d31774d, 
   0x835432d8, 0x1d00843d, 0x8348d8dc, 0x1ccf8cb3, 0x833d921b, 0x1c9e90b8, 
   0x83325e97, 0x1c6d9053, 0x83273e52, 0x1c3c8b8c, 0x831c314e, 0x1c0b826a, 
   0x8311378d, 0x1bda74f6, 0x83065110, 0x1ba96335, 0x82fb7dd8, 0x1b784d30, 
   0x82f0bde8, 0x1b4732ef, 0x82e61141, 0x1b161479, 0x82db77e5, 0x1ae4f1d6, 
   0x82d0f1d5, 0x1ab3cb0d, 0x82c67f14, 0x1a82a026, 0x82bc1fa2, 0x1a517128, 
   0x82b1d381, 0x1a203e1b, 0x82a79ab3, 0x19ef0707, 0x829d753a, 0x19bdcbf3, 
   0x82936317, 0x198c8ce7, 0x8289644b, 0x195b49ea, 0x827f78d8, 0x192a0304, 
   0x8275a0c0, 0x18f8b83c, 0x826bdc04, 0x18c7699b, 0x82622aa6, 0x18961728, 
   0x82588ca7, 0x1864c0ea, 0x824f0208, 0x183366e9, 0x82458acc, 0x1802092c, 
   0x823c26f3, 0x17d0a7bc, 0x8232d67f, 0x179f429f, 0x82299971, 0x176dd9de, 
   0x82206fcc, 0x173c6d80, 0x82175990, 0x170afd8d, 0x820e56be, 0x16d98a0c, 
   0x82056758, 0x16a81305, 0x81fc8b60, 0x1676987f, 0x81f3c2d7, 0x16451a83, 
   0x81eb0dbe, 0x16139918, 0x81e26c16, 0x15e21445, 0x81d9dde1, 0x15b08c12, 
   0x81d16321, 0x157f0086, 0x81c8fbd6, 0x154d71aa, 0x81c0a801, 0x151bdf86, 
   0x81b867a5, 0x14ea4a1f, 0x81b03ac2, 0x14b8b17f, 0x81a82159, 0x148715ae, 
   0x81a01b6d, 0x145576b1, 0x819828fd, 0x1423d492, 0x81904a0c, 0x13f22f58, 
   0x81887e9a, 0x13c0870a, 0x8180c6a9, 0x138edbb1, 0x8179223a, 0x135d2d53, 
   0x8171914e, 0x132b7bf9, 0x816a13e6, 0x12f9c7aa, 0x8162aa04, 0x12c8106f, 
   0x815b53a8, 0x1296564d, 0x815410d4, 0x1264994e, 0x814ce188, 0x1232d979, 
   0x8145c5c7, 0x120116d5, 0x813ebd90, 0x11cf516a, 0x8137c8e6, 0x119d8941, 
   0x8130e7c9, 0x116bbe60, 0x812a1a3a, 0x1139f0cf, 0x8123603a, 0x11082096, 
   0x811cb9ca, 0x10d64dbd, 0x811626ec, 0x10a4784b, 0x810fa7a0, 0x1072a048, 
   0x81093be8, 0x1040c5bb, 0x8102e3c4, 0x100ee8ad, 0x80fc9f35, 0xfdd0926, 
   0x80f66e3c, 0xfab272b, 0x80f050db, 0xf7942c7, 0x80ea4712, 0xf475bff, 
   0x80e450e2, 0xf1572dc, 0x80de6e4c, 0xee38766, 0x80d89f51, 0xeb199a4, 
   0x80d2e3f2, 0xe7fa99e, 0x80cd3c2f, 0xe4db75b, 0x80c7a80a, 0xe1bc2e4, 
   0x80c22784, 0xde9cc40, 0x80bcba9d, 0xdb7d376, 0x80b76156, 0xd85d88f, 
   0x80b21baf, 0xd53db92, 0x80ace9ab, 0xd21dc87, 0x80a7cb49, 0xcefdb76, 
   0x80a2c08b, 0xcbdd865, 0x809dc971, 0xc8bd35e, 0x8098e5fb, 0xc59cc68, 
   0x8094162c, 0xc27c389, 0x808f5a02, 0xbf5b8cb, 0x808ab180, 0xbc3ac35, 
   0x80861ca6, 0xb919dcf, 0x80819b74, 0xb5f8d9f, 0x807d2dec, 0xb2d7baf, 
   0x8078d40d, 0xafb6805, 0x80748dd9, 0xac952aa, 0x80705b50, 0xa973ba5, 
   0x806c3c74, 0xa6522fe, 0x80683143, 0xa3308bd, 0x806439c0, 0xa00ece8, 
   0x806055eb, 0x9cecf89, 0x805c85c4, 0x99cb0a7, 0x8058c94c, 0x96a9049, 
   0x80552084, 0x9386e78, 0x80518b6b, 0x9064b3a, 0x804e0a04, 0x8d42699, 
   0x804a9c4d, 0x8a2009a, 0x80474248, 0x86fd947, 0x8043fbf6, 0x83db0a7, 
   0x8040c956, 0x80b86c2, 0x803daa6a, 0x7d95b9e, 0x803a9f31, 0x7a72f45, 
   0x8037a7ac, 0x77501be, 0x8034c3dd, 0x742d311, 0x8031f3c2, 0x710a345, 
   0x802f375d, 0x6de7262, 0x802c8ead, 0x6ac406f, 0x8029f9b4, 0x67a0d76, 
   0x80277872, 0x647d97c, 0x80250ae7, 0x615a48b, 0x8022b114, 0x5e36ea9, 
   0x80206af8, 0x5b137df, 0x801e3895, 0x57f0035, 0x801c19ea, 0x54cc7b1, 
   0x801a0ef8, 0x51a8e5c, 0x801817bf, 0x4e8543e, 0x80163440, 0x4b6195d, 
   0x8014647b, 0x483ddc3, 0x8012a86f, 0x451a177, 0x8011001f, 0x41f6480, 
   0x800f6b88, 0x3ed26e6, 0x800deaad, 0x3bae8b2, 0x800c7d8c, 0x388a9ea, 
   0x800b2427, 0x3566a96, 0x8009de7e, 0x3242abf, 0x8008ac90, 0x2f1ea6c, 
   0x80078e5e, 0x2bfa9a4, 0x800683e8, 0x28d6870, 0x80058d2f, 0x25b26d7, 
   0x8004aa32, 0x228e4e2, 0x8003daf1, 0x1f6a297, 0x80031f6d, 0x1c45ffe, 
   0x800277a6, 0x1921d20, 0x8001e39b, 0x15fda03, 0x8001634e, 0x12d96b1, 
   0x8000f6bd, 0xfb5330, 0x80009dea, 0xc90f88, 0x800058d4, 0x96cbc1, 
   0x8000277a, 0x6487e3, 0x800009df, 0x3243f5, 0x80000000, 0x0, 0x800009df, 
   0xffcdbc0b, 0x8000277a, 0xff9b781d, 0x800058d4, 0xff69343f, 0x80009dea, 
   0xff36f078, 0x8000f6bd, 0xff04acd0, 0x8001634e, 0xfed2694f, 0x8001e39b, 
   0xfea025fd, 0x800277a6, 0xfe6de2e0, 0x80031f6d, 0xfe3ba002, 0x8003daf1, 
   0xfe095d69, 0x8004aa32, 0xfdd71b1e, 0x80058d2f, 0xfda4d929, 0x800683e8, 
   0xfd729790, 0x80078e5e, 0xfd40565c, 0x8008ac90, 0xfd0e1594, 0x8009de7e, 
   0xfcdbd541, 0x800b2427, 0xfca9956a, 0x800c7d8c, 0xfc775616, 0x800deaad, 
   0xfc45174e, 0x800f6b88, 0xfc12d91a, 0x8011001f, 0xfbe09b80, 0x8012a86f, 
   0xfbae5e89, 0x8014647b, 0xfb7c223d, 0x80163440, 0xfb49e6a3, 0x801817bf, 
   0xfb17abc2, 0x801a0ef8, 0xfae571a4, 0x801c19ea, 0xfab3384f, 0x801e3895, 
   0xfa80ffcb, 0x80206af8, 0xfa4ec821, 0x8022b114, 0xfa1c9157, 0x80250ae7, 
   0xf9ea5b75, 0x80277872, 0xf9b82684, 0x8029f9b4, 0xf985f28a, 0x802c8ead, 
   0xf953bf91, 0x802f375d, 0xf9218d9e, 0x8031f3c2, 0xf8ef5cbb, 0x8034c3dd, 
   0xf8bd2cef, 0x8037a7ac, 0xf88afe42, 0x803a9f31, 0xf858d0bb, 0x803daa6a, 
   0xf826a462, 0x8040c956, 0xf7f4793e, 0x8043fbf6, 0xf7c24f59, 0x80474248, 
   0xf79026b9, 0x804a9c4d, 0xf75dff66, 0x804e0a04, 0xf72bd967, 0x80518b6b, 
   0xf6f9b4c6, 0x80552084, 0xf6c79188, 0x8058c94c, 0xf6956fb7, 0x805c85c4, 
   0xf6634f59, 0x806055eb, 0xf6313077, 0x806439c0, 0xf5ff1318, 0x80683143, 
   0xf5ccf743, 0x806c3c74, 0xf59add02, 0x80705b50, 0xf568c45b, 0x80748dd9, 
   0xf536ad56, 0x8078d40d, 0xf50497fb, 0x807d2dec, 0xf4d28451, 0x80819b74, 
   0xf4a07261, 0x80861ca6, 0xf46e6231, 0x808ab180, 0xf43c53cb, 0x808f5a02, 
   0xf40a4735, 0x8094162c, 0xf3d83c77, 0x8098e5fb, 0xf3a63398, 0x809dc971, 
   0xf3742ca2, 0x80a2c08b, 0xf342279b, 0x80a7cb49, 0xf310248a, 0x80ace9ab, 
   0xf2de2379, 0x80b21baf, 0xf2ac246e, 0x80b76156, 0xf27a2771, 0x80bcba9d, 
   0xf2482c8a, 0x80c22784, 0xf21633c0, 0x80c7a80a, 0xf1e43d1c, 0x80cd3c2f, 
   0xf1b248a5, 0x80d2e3f2, 0xf1805662, 0x80d89f51, 0xf14e665c, 0x80de6e4c, 
   0xf11c789a, 0x80e450e2, 0xf0ea8d24, 0x80ea4712, 0xf0b8a401, 0x80f050db, 
   0xf086bd39, 0x80f66e3c, 0xf054d8d5, 0x80fc9f35, 0xf022f6da, 0x8102e3c4, 
   0xeff11753, 0x81093be8, 0xefbf3a45, 0x810fa7a0, 0xef8d5fb8, 0x811626ec, 
   0xef5b87b5, 0x811cb9ca, 0xef29b243, 0x8123603a, 0xeef7df6a, 0x812a1a3a, 
   0xeec60f31, 0x8130e7c9, 0xee9441a0, 0x8137c8e6, 0xee6276bf, 0x813ebd90, 
   0xee30ae96, 0x8145c5c7, 0xedfee92b, 0x814ce188, 0xedcd2687, 0x815410d4, 
   0xed9b66b2, 0x815b53a8, 0xed69a9b3, 0x8162aa04, 0xed37ef91, 0x816a13e6, 
   0xed063856, 0x8171914e, 0xecd48407, 0x8179223a, 0xeca2d2ad, 0x8180c6a9, 
   0xec71244f, 0x81887e9a, 0xec3f78f6, 0x81904a0c, 0xec0dd0a8, 0x819828fd, 
   0xebdc2b6e, 0x81a01b6d, 0xebaa894f, 0x81a82159, 0xeb78ea52, 0x81b03ac2, 
   0xeb474e81, 0x81b867a5, 0xeb15b5e1, 0x81c0a801, 0xeae4207a, 0x81c8fbd6, 
   0xeab28e56, 0x81d16321, 0xea80ff7a, 0x81d9dde1, 0xea4f73ee, 0x81e26c16, 
   0xea1debbb, 0x81eb0dbe, 0xe9ec66e8, 0x81f3c2d7, 0xe9bae57d, 0x81fc8b60, 
   0xe9896781, 0x82056758, 0xe957ecfb, 0x820e56be, 0xe92675f4, 0x82175990, 
   0xe8f50273, 0x82206fcc, 0xe8c39280, 0x82299971, 0xe8922622, 0x8232d67f, 
   0xe860bd61, 0x823c26f3, 0xe82f5844, 0x82458acc, 0xe7fdf6d4, 0x824f0208, 
   0xe7cc9917, 0x82588ca7, 0xe79b3f16, 0x82622aa6, 0xe769e8d8, 0x826bdc04, 
   0xe7389665, 0x8275a0c0, 0xe70747c4, 0x827f78d8, 0xe6d5fcfc, 0x8289644b, 
   0xe6a4b616, 0x82936317, 0xe6737319, 0x829d753a, 0xe642340d, 0x82a79ab3, 
   0xe610f8f9, 0x82b1d381, 0xe5dfc1e5, 0x82bc1fa2, 0xe5ae8ed8, 0x82c67f14, 
   0xe57d5fda, 0x82d0f1d5, 0xe54c34f3, 0x82db77e5, 0xe51b0e2a, 0x82e61141, 
   0xe4e9eb87, 0x82f0bde8, 0xe4b8cd11, 0x82fb7dd8, 0xe487b2d0, 0x83065110, 
   0xe4569ccb, 0x8311378d, 0xe4258b0a, 0x831c314e, 0xe3f47d96, 0x83273e52, 
   0xe3c37474, 0x83325e97, 0xe3926fad, 0x833d921b, 0xe3616f48, 0x8348d8dc, 
   0xe330734d, 0x835432d8, 0xe2ff7bc3, 0x835fa00f, 0xe2ce88b3, 0x836b207d, 
   0xe29d9a23, 0x8376b422, 0xe26cb01b, 0x83825afb, 0xe23bcaa2, 0x838e1507, 
   0xe20ae9c1, 0x8399e244, 0xe1da0d7e, 0x83a5c2b0, 0xe1a935e2, 0x83b1b649, 
   0xe17862f3, 0x83bdbd0e, 0xe14794ba, 0x83c9d6fc, 0xe116cb3d, 0x83d60412, 
   0xe0e60685, 0x83e2444d, 0xe0b54698, 0x83ee97ad, 0xe0848b7f, 0x83fafe2e, 
   0xe053d541, 0x840777d0, 0xe02323e5, 0x84140490, 0xdff27773, 0x8420a46c, 
   0xdfc1cff3, 0x842d5762, 0xdf912d6b, 0x843a1d70, 0xdf608fe4, 0x8446f695, 
   0xdf2ff764, 0x8453e2cf, 0xdeff63f4, 0x8460e21a, 0xdeced59b, 0x846df477, 
   0xde9e4c60, 0x847b19e1, 0xde6dc84b, 0x84885258, 0xde3d4964, 0x84959dd9, 
   0xde0ccfb1, 0x84a2fc62, 0xdddc5b3b, 0x84b06df2, 0xddabec08, 0x84bdf286, 
   0xdd7b8220, 0x84cb8a1b, 0xdd4b1d8c, 0x84d934b1, 0xdd1abe51, 0x84e6f244, 
   0xdcea6478, 0x84f4c2d4, 0xdcba1008, 0x8502a65c, 0xdc89c109, 0x85109cdd, 
   0xdc597781, 0x851ea652, 0xdc293379, 0x852cc2bb, 0xdbf8f4f8, 0x853af214, 
   0xdbc8bc06, 0x8549345c, 0xdb9888a8, 0x85578991, 0xdb685ae9, 0x8565f1b0, 
   0xdb3832cd, 0x85746cb8, 0xdb08105e, 0x8582faa5, 0xdad7f3a2, 0x85919b76, 
   0xdaa7dca1, 0x85a04f28, 0xda77cb63, 0x85af15b9, 0xda47bfee, 0x85bdef28, 
   0xda17ba4a, 0x85ccdb70, 0xd9e7ba7f, 0x85dbda91, 0xd9b7c094, 0x85eaec88, 
   0xd987cc90, 0x85fa1153, 0xd957de7a, 0x860948ef, 0xd927f65b, 0x86189359, 
   0xd8f81439, 0x8627f091, 0xd8c8381d, 0x86376092, 0xd898620c, 0x8646e35c, 
   0xd868920f, 0x865678eb, 0xd838c82d, 0x8666213c, 0xd809046e, 0x8675dc4f, 
   0xd7d946d8, 0x8685aa20, 0xd7a98f73, 0x86958aac, 0xd779de47, 0x86a57df2, 
   0xd74a335b, 0x86b583ee, 0xd71a8eb5, 0x86c59c9f, 0xd6eaf05f, 0x86d5c802, 
   0xd6bb585e, 0x86e60614, 0xd68bc6ba, 0x86f656d3, 0xd65c3b7b, 0x8706ba3d, 
   0xd62cb6a8, 0x8717304e, 0xd5fd3848, 0x8727b905, 0xd5cdc062, 0x8738545e, 
   0xd59e4eff, 0x87490258, 0xd56ee424, 0x8759c2ef, 0xd53f7fda, 0x876a9621, 
   0xd5102228, 0x877b7bec, 0xd4e0cb15, 0x878c744d, 0xd4b17aa8, 0x879d7f41, 
   0xd48230e9, 0x87ae9cc5, 0xd452eddf, 0x87bfccd7, 0xd423b191, 0x87d10f75, 
   0xd3f47c06, 0x87e2649b, 0xd3c54d47, 0x87f3cc48, 0xd396255a, 0x88054677, 
   0xd3670446, 0x8816d327, 0xd337ea12, 0x88287256, 0xd308d6c7, 0x883a23ff, 
   0xd2d9ca6a, 0x884be821, 0xd2aac504, 0x885dbeb8, 0xd27bc69c, 0x886fa7c2, 
   0xd24ccf39, 0x8881a33d, 0xd21ddee2, 0x8893b125, 0xd1eef59e, 0x88a5d177, 
   0xd1c01375, 0x88b80432, 0xd191386e, 0x88ca4951, 0xd1626490, 0x88dca0d3, 
   0xd13397e2, 0x88ef0ab4, 0xd104d26b, 0x890186f2, 0xd0d61434, 0x89141589, 
   0xd0a75d42, 0x8926b677, 0xd078ad9e, 0x893969b9, 0xd04a054e, 0x894c2f4c, 
   0xd01b6459, 0x895f072e, 0xcfeccac7, 0x8971f15a, 0xcfbe389f, 0x8984edcf, 
   0xcf8fade9, 0x8997fc8a, 0xcf612aaa, 0x89ab1d87, 0xcf32aeeb, 0x89be50c3, 
   0xcf043ab3, 0x89d1963c, 0xced5ce08, 0x89e4edef, 0xcea768f2, 0x89f857d8, 
   0xce790b79, 0x8a0bd3f5, 0xce4ab5a2, 0x8a1f6243, 0xce1c6777, 0x8a3302be, 
   0xcdee20fc, 0x8a46b564, 0xcdbfe23a, 0x8a5a7a31, 0xcd91ab39, 0x8a6e5123, 
   0xcd637bfe, 0x8a823a36, 0xcd355491, 0x8a963567, 0xcd0734f9, 0x8aaa42b4, 
   0xccd91d3d, 0x8abe6219, 0xccab0d65, 0x8ad29394, 0xcc7d0578, 0x8ae6d720, 
   0xcc4f057c, 0x8afb2cbb, 0xcc210d79, 0x8b0f9462, 0xcbf31d75, 0x8b240e11, 
   0xcbc53579, 0x8b3899c6, 0xcb97558a, 0x8b4d377c, 0xcb697db0, 0x8b61e733, 
   0xcb3badf3, 0x8b76a8e4, 0xcb0de658, 0x8b8b7c8f, 0xcae026e8, 0x8ba0622f, 
   0xcab26fa9, 0x8bb559c1, 0xca84c0a3, 0x8bca6343, 0xca5719db, 0x8bdf7eb0, 
   0xca297b5a, 0x8bf4ac05, 0xc9fbe527, 0x8c09eb40, 0xc9ce5748, 0x8c1f3c5d, 
   0xc9a0d1c5, 0x8c349f58, 0xc97354a4, 0x8c4a142f, 0xc945dfec, 0x8c5f9ade, 
   0xc91873a5, 0x8c753362, 0xc8eb0fd6, 0x8c8addb7, 0xc8bdb485, 0x8ca099da, 
   0xc89061ba, 0x8cb667c8, 0xc863177b, 0x8ccc477d, 0xc835d5d0, 0x8ce238f6, 
   0xc8089cbf, 0x8cf83c30, 0xc7db6c50, 0x8d0e5127, 0xc7ae4489, 0x8d2477d8, 
   0xc7812572, 0x8d3ab03f, 0xc7540f11, 0x8d50fa59, 0xc727016d, 0x8d675623, 
   0xc6f9fc8d, 0x8d7dc399, 0xc6cd0079, 0x8d9442b8, 0xc6a00d37, 0x8daad37b, 
   0xc67322ce, 0x8dc175e0, 0xc6464144, 0x8dd829e4, 0xc61968a2, 0x8deeef82, 
   0xc5ec98ee, 0x8e05c6b7, 0xc5bfd22e, 0x8e1caf80, 0xc593146a, 0x8e33a9da, 
   0xc5665fa9, 0x8e4ab5bf, 0xc539b3f1, 0x8e61d32e, 0xc50d1149, 0x8e790222, 
   0xc4e077b8, 0x8e904298, 0xc4b3e746, 0x8ea7948c, 0xc4875ff9, 0x8ebef7fb, 
   0xc45ae1d7, 0x8ed66ce1, 0xc42e6ce8, 0x8eedf33b, 0xc4020133, 0x8f058b04, 
   0xc3d59ebe, 0x8f1d343a, 0xc3a94590, 0x8f34eed8, 0xc37cf5b0, 0x8f4cbadb, 
   0xc350af26, 0x8f649840, 0xc32471f7, 0x8f7c8701, 0xc2f83e2a, 0x8f94871d, 
   0xc2cc13c7, 0x8fac988f, 0xc29ff2d4, 0x8fc4bb53, 0xc273db58, 0x8fdcef66, 
   0xc247cd5a, 0x8ff534c4, 0xc21bc8e1, 0x900d8b69, 0xc1efcdf3, 0x9025f352, 
   0xc1c3dc97, 0x903e6c7b, 0xc197f4d4, 0x9056f6df, 0xc16c16b0, 0x906f927c, 
   0xc1404233, 0x90883f4d, 0xc1147764, 0x90a0fd4e, 0xc0e8b648, 0x90b9cc7d, 
   0xc0bcfee7, 0x90d2acd4, 0xc0915148, 0x90eb9e50, 0xc065ad70, 0x9104a0ee, 
   0xc03a1368, 0x911db4a9, 0xc00e8336, 0x9136d97d, 0xbfe2fcdf, 0x91500f67, 
   0xbfb7806c, 0x91695663, 0xbf8c0de3, 0x9182ae6d, 0xbf60a54a, 0x919c1781, 
   0xbf3546a8, 0x91b5919a, 0xbf09f205, 0x91cf1cb6, 0xbedea765, 0x91e8b8d0, 
   0xbeb366d1, 0x920265e4, 0xbe88304f, 0x921c23ef, 0xbe5d03e6, 0x9235f2ec, 
   0xbe31e19b, 0x924fd2d7, 0xbe06c977, 0x9269c3ac, 0xbddbbb7f, 0x9283c568, 
   0xbdb0b7bb, 0x929dd806, 0xbd85be30, 0x92b7fb82, 0xbd5acee5, 0x92d22fd9, 
   0xbd2fe9e2, 0x92ec7505, 0xbd050f2c, 0x9306cb04, 0xbcda3ecb, 0x932131d1, 
   0xbcaf78c4, 0x933ba968, 0xbc84bd1f, 0x935631c5, 0xbc5a0be2, 0x9370cae4, 
   0xbc2f6513, 0x938b74c1, 0xbc04c8ba, 0x93a62f57, 0xbbda36dd, 0x93c0faa3, 
   0xbbafaf82, 0x93dbd6a0, 0xbb8532b0, 0x93f6c34a, 0xbb5ac06d, 0x9411c09e, 
   0xbb3058c0, 0x942cce96, 0xbb05fbb0, 0x9447ed2f, 0xbadba943, 0x94631c65, 
   0xbab16180, 0x947e5c33, 0xba87246d, 0x9499ac95, 0xba5cf210, 0x94b50d87, 
   0xba32ca71, 0x94d07f05, 0xba08ad95, 0x94ec010b, 0xb9de9b83, 0x95079394, 
   0xb9b49442, 0x9523369c, 0xb98a97d8, 0x953eea1e, 0xb960a64c, 0x955aae17, 
   0xb936bfa4, 0x95768283, 0xb90ce3e6, 0x9592675c, 0xb8e31319, 0x95ae5c9f, 
   0xb8b94d44, 0x95ca6247, 0xb88f926d, 0x95e67850, 0xb865e299, 0x96029eb6, 
   0xb83c3dd1, 0x961ed574, 0xb812a41a, 0x963b1c86, 0xb7e9157a, 0x965773e7, 
   0xb7bf91f8, 0x9673db94, 0xb796199b, 0x96905388, 0xb76cac69, 0x96acdbbe, 
   0xb7434a67, 0x96c97432, 0xb719f39e, 0x96e61ce0, 0xb6f0a812, 0x9702d5c3, 
   0xb6c767ca, 0x971f9ed7, 0xb69e32cd, 0x973c7817, 0xb6750921, 0x9759617f, 
   0xb64beacd, 0x97765b0a, 0xb622d7d6, 0x979364b5, 0xb5f9d043, 0x97b07e7a, 
   0xb5d0d41a, 0x97cda855, 0xb5a7e362, 0x97eae242, 0xb57efe22, 0x98082c3b, 
   0xb556245e, 0x9825863d, 0xb52d561e, 0x9842f043, 0xb5049368, 0x98606a49, 
   0xb4dbdc42, 0x987df449, 0xb4b330b3, 0x989b8e40, 0xb48a90c0, 0x98b93828, 
   0xb461fc70, 0x98d6f1fe, 0xb43973ca, 0x98f4bbbc, 0xb410f6d3, 0x9912955f, 
   0xb3e88592, 0x99307ee0, 0xb3c0200c, 0x994e783d, 0xb397c649, 0x996c816f, 
   0xb36f784f, 0x998a9a74, 0xb3473623, 0x99a8c345, 0xb31effcc, 0x99c6fbde, 
   0xb2f6d550, 0x99e5443b, 0xb2ceb6b5, 0x9a039c57, 0xb2a6a402, 0x9a22042d, 
   0xb27e9d3c, 0x9a407bb9, 0xb256a26a, 0x9a5f02f5, 0xb22eb392, 0x9a7d99de, 
   0xb206d0ba, 0x9a9c406e, 0xb1def9e9, 0x9abaf6a1, 0xb1b72f23, 0x9ad9bc71, 
   0xb18f7071, 0x9af891db, 0xb167bdd7, 0x9b1776da, 0xb140175b, 0x9b366b68, 
   0xb1187d05, 0x9b556f81, 0xb0f0eeda, 0x9b748320, 0xb0c96ce0, 0x9b93a641, 
   0xb0a1f71d, 0x9bb2d8de, 0xb07a8d97, 0x9bd21af3, 0xb0533055, 0x9bf16c7a, 
   0xb02bdf5c, 0x9c10cd70, 0xb0049ab3, 0x9c303dcf, 0xafdd625f, 0x9c4fbd93, 
   0xafb63667, 0x9c6f4cb6, 0xaf8f16d1, 0x9c8eeb34, 0xaf6803a2, 0x9cae9907, 
   0xaf40fce1, 0x9cce562c, 0xaf1a0293, 0x9cee229c, 0xaef314c0, 0x9d0dfe54, 
   0xaecc336c, 0x9d2de94d, 0xaea55e9e, 0x9d4de385, 0xae7e965b, 0x9d6decf4, 
   0xae57daab, 0x9d8e0597, 0xae312b92, 0x9dae2d68, 0xae0a8916, 0x9dce6463, 
   0xade3f33e, 0x9deeaa82, 0xadbd6a10, 0x9e0effc1, 0xad96ed92, 0x9e2f641b, 
   0xad707dc8, 0x9e4fd78a, 0xad4a1aba, 0x9e705a09, 0xad23c46e, 0x9e90eb94, 
   0xacfd7ae8, 0x9eb18c26, 0xacd73e30, 0x9ed23bb9, 0xacb10e4b, 0x9ef2fa49, 
   0xac8aeb3e, 0x9f13c7d0, 0xac64d510, 0x9f34a449, 0xac3ecbc7, 0x9f558fb0, 
   0xac18cf69, 0x9f7689ff, 0xabf2dffb, 0x9f979331, 0xabccfd83, 0x9fb8ab41, 
   0xaba72807, 0x9fd9d22a, 0xab815f8d, 0x9ffb07e7, 0xab5ba41a, 0xa01c4c73, 
   0xab35f5b5, 0xa03d9fc8, 0xab105464, 0xa05f01e1, 0xaaeac02c, 0xa08072ba, 
   0xaac53912, 0xa0a1f24d, 0xaa9fbf1e, 0xa0c38095, 0xaa7a5253, 0xa0e51d8c, 
   0xaa54f2ba, 0xa106c92f, 0xaa2fa056, 0xa1288376, 0xaa0a5b2e, 0xa14a4c5e, 
   0xa9e52347, 0xa16c23e1, 0xa9bff8a8, 0xa18e09fa, 0xa99adb56, 0xa1affea3, 
   0xa975cb57, 0xa1d201d7, 0xa950c8b0, 0xa1f41392, 0xa92bd367, 0xa21633cd, 
   0xa906eb82, 0xa2386284, 0xa8e21106, 0xa25a9fb1, 0xa8bd43fa, 0xa27ceb4f, 
   0xa8988463, 0xa29f4559, 0xa873d246, 0xa2c1adc9, 0xa84f2daa, 0xa2e4249b, 
   0xa82a9693, 0xa306a9c8, 0xa8060d08, 0xa3293d4b, 0xa7e1910f, 0xa34bdf20, 
   0xa7bd22ac, 0xa36e8f41, 0xa798c1e5, 0xa3914da8, 0xa7746ec0, 0xa3b41a50, 
   0xa7502943, 0xa3d6f534, 0xa72bf174, 0xa3f9de4e, 0xa707c757, 0xa41cd599, 
   0xa6e3aaf2, 0xa43fdb10, 0xa6bf9c4b, 0xa462eeac, 0xa69b9b68, 0xa486106a, 
   0xa677a84e, 0xa4a94043, 0xa653c303, 0xa4cc7e32, 0xa62feb8b, 0xa4efca31, 
   0xa60c21ee, 0xa513243b, 0xa5e8662f, 0xa5368c4b, 0xa5c4b855, 0xa55a025b, 
   0xa5a11866, 0xa57d8666, 0xa57d8666, 0xa5a11866, 0xa55a025b, 0xa5c4b855, 
   0xa5368c4b, 0xa5e8662f, 0xa513243b, 0xa60c21ee, 0xa4efca31, 0xa62feb8b, 
   0xa4cc7e32, 0xa653c303, 0xa4a94043, 0xa677a84e, 0xa486106a, 0xa69b9b68, 
   0xa462eeac, 0xa6bf9c4b, 0xa43fdb10, 0xa6e3aaf2, 0xa41cd599, 0xa707c757, 
   0xa3f9de4e, 0xa72bf174, 0xa3d6f534, 0xa7502943, 0xa3b41a50, 0xa7746ec0, 
   0xa3914da8, 0xa798c1e5, 0xa36e8f41, 0xa7bd22ac, 0xa34bdf20, 0xa7e1910f, 
   0xa3293d4b, 0xa8060d08, 0xa306a9c8, 0xa82a9693, 0xa2e4249b, 0xa84f2daa, 
   0xa2c1adc9, 0xa873d246, 0xa29f4559, 0xa8988463, 0xa27ceb4f, 0xa8bd43fa, 
   0xa25a9fb1, 0xa8e21106, 0xa2386284, 0xa906eb82, 0xa21633cd, 0xa92bd367, 
   0xa1f41392, 0xa950c8b0, 0xa1d201d7, 0xa975cb57, 0xa1affea3, 0xa99adb56, 
   0xa18e09fa, 0xa9bff8a8, 0xa16c23e1, 0xa9e52347, 0xa14a4c5e, 0xaa0a5b2e, 
   0xa1288376, 0xaa2fa056, 0xa106c92f, 0xaa54f2ba, 0xa0e51d8c, 0xaa7a5253, 
   0xa0c38095, 0xaa9fbf1e, 0xa0a1f24d, 0xaac53912, 0xa08072ba, 0xaaeac02c, 
   0xa05f01e1, 0xab105464, 0xa03d9fc8, 0xab35f5b5, 0xa01c4c73, 0xab5ba41a, 
   0x9ffb07e7, 0xab815f8d, 0x9fd9d22a, 0xaba72807, 0x9fb8ab41, 0xabccfd83, 
   0x9f979331, 0xabf2dffb, 0x9f7689ff, 0xac18cf69, 0x9f558fb0, 0xac3ecbc7, 
   0x9f34a449, 0xac64d510, 0x9f13c7d0, 0xac8aeb3e, 0x9ef2fa49, 0xacb10e4b, 
   0x9ed23bb9, 0xacd73e30, 0x9eb18c26, 0xacfd7ae8, 0x9e90eb94, 0xad23c46e, 
   0x9e705a09, 0xad4a1aba, 0x9e4fd78a, 0xad707dc8, 0x9e2f641b, 0xad96ed92, 
   0x9e0effc1, 0xadbd6a10, 0x9deeaa82, 0xade3f33e, 0x9dce6463, 0xae0a8916, 
   0x9dae2d68, 0xae312b92, 0x9d8e0597, 0xae57daab, 0x9d6decf4, 0xae7e965b, 
   0x9d4de385, 0xaea55e9e, 0x9d2de94d, 0xaecc336c, 0x9d0dfe54, 0xaef314c0, 
   0x9cee229c, 0xaf1a0293, 0x9cce562c, 0xaf40fce1, 0x9cae9907, 0xaf6803a2, 
   0x9c8eeb34, 0xaf8f16d1, 0x9c6f4cb6, 0xafb63667, 0x9c4fbd93, 0xafdd625f, 
   0x9c303dcf, 0xb0049ab3, 0x9c10cd70, 0xb02bdf5c, 0x9bf16c7a, 0xb0533055, 
   0x9bd21af3, 0xb07a8d97, 0x9bb2d8de, 0xb0a1f71d, 0x9b93a641, 0xb0c96ce0, 
   0x9b748320, 0xb0f0eeda, 0x9b556f81, 0xb1187d05, 0x9b366b68, 0xb140175b, 
   0x9b1776da, 0xb167bdd7, 0x9af891db, 0xb18f7071, 0x9ad9bc71, 0xb1b72f23, 
   0x9abaf6a1, 0xb1def9e9, 0x9a9c406e, 0xb206d0ba, 0x9a7d99de, 0xb22eb392, 
   0x9a5f02f5, 0xb256a26a, 0x9a407bb9, 0xb27e9d3c, 0x9a22042d, 0xb2a6a402, 
   0x9a039c57, 0xb2ceb6b5, 0x99e5443b, 0xb2f6d550, 0x99c6fbde, 0xb31effcc, 
   0x99a8c345, 0xb3473623, 0x998a9a74, 0xb36f784f, 0x996c816f, 0xb397c649, 
   0x994e783d, 0xb3c0200c, 0x99307ee0, 0xb3e88592, 0x9912955f, 0xb410f6d3, 
   0x98f4bbbc, 0xb43973ca, 0x98d6f1fe, 0xb461fc70, 0x98b93828, 0xb48a90c0, 
   0x989b8e40, 0xb4b330b3, 0x987df449, 0xb4dbdc42, 0x98606a49, 0xb5049368, 
   0x9842f043, 0xb52d561e, 0x9825863d, 0xb556245e, 0x98082c3b, 0xb57efe22, 
   0x97eae242, 0xb5a7e362, 0x97cda855, 0xb5d0d41a, 0x97b07e7a, 0xb5f9d043, 
   0x979364b5, 0xb622d7d6, 0x97765b0a, 0xb64beacd, 0x9759617f, 0xb6750921, 
   0x973c7817, 0xb69e32cd, 0x971f9ed7, 0xb6c767ca, 0x9702d5c3, 0xb6f0a812, 
   0x96e61ce0, 0xb719f39e, 0x96c97432, 0xb7434a67, 0x96acdbbe, 0xb76cac69, 
   0x96905388, 0xb796199b, 0x9673db94, 0xb7bf91f8, 0x965773e7, 0xb7e9157a, 
   0x963b1c86, 0xb812a41a, 0x961ed574, 0xb83c3dd1, 0x96029eb6, 0xb865e299, 
   0x95e67850, 0xb88f926d, 0x95ca6247, 0xb8b94d44, 0x95ae5c9f, 0xb8e31319, 
   0x9592675c, 0xb90ce3e6, 0x95768283, 0xb936bfa4, 0x955aae17, 0xb960a64c, 
   0x953eea1e, 0xb98a97d8, 0x9523369c, 0xb9b49442, 0x95079394, 0xb9de9b83, 
   0x94ec010b, 0xba08ad95, 0x94d07f05, 0xba32ca71, 0x94b50d87, 0xba5cf210, 
   0x9499ac95, 0xba87246d, 0x947e5c33, 0xbab16180, 0x94631c65, 0xbadba943, 
   0x9447ed2f, 0xbb05fbb0, 0x942cce96, 0xbb3058c0, 0x9411c09e, 0xbb5ac06d, 
   0x93f6c34a, 0xbb8532b0, 0x93dbd6a0, 0xbbafaf82, 0x93c0faa3, 0xbbda36dd, 
   0x93a62f57, 0xbc04c8ba, 0x938b74c1, 0xbc2f6513, 0x9370cae4, 0xbc5a0be2, 
   0x935631c5, 0xbc84bd1f, 0x933ba968, 0xbcaf78c4, 0x932131d1, 0xbcda3ecb, 
   0x9306cb04, 0xbd050f2c, 0x92ec7505, 0xbd2fe9e2, 0x92d22fd9, 0xbd5acee5, 
   0x92b7fb82, 0xbd85be30, 0x929dd806, 0xbdb0b7bb, 0x9283c568, 0xbddbbb7f, 
   0x9269c3ac, 0xbe06c977, 0x924fd2d7, 0xbe31e19b, 0x9235f2ec, 0xbe5d03e6, 
   0x921c23ef, 0xbe88304f, 0x920265e4, 0xbeb366d1, 0x91e8b8d0, 0xbedea765, 
   0x91cf1cb6, 0xbf09f205, 0x91b5919a, 0xbf3546a8, 0x919c1781, 0xbf60a54a, 
   0x9182ae6d, 0xbf8c0de3, 0x91695663, 0xbfb7806c, 0x91500f67, 0xbfe2fcdf, 
   0x9136d97d, 0xc00e8336, 0x911db4a9, 0xc03a1368, 0x9104a0ee, 0xc065ad70, 
   0x90eb9e50, 0xc0915148, 0x90d2acd4, 0xc0bcfee7, 0x90b9cc7d, 0xc0e8b648, 
   0x90a0fd4e, 0xc1147764, 0x90883f4d, 0xc1404233, 0x906f927c, 0xc16c16b0, 
   0x9056f6df, 0xc197f4d4, 0x903e6c7b, 0xc1c3dc97, 0x9025f352, 0xc1efcdf3, 
   0x900d8b69, 0xc21bc8e1, 0x8ff534c4, 0xc247cd5a, 0x8fdcef66, 0xc273db58, 
   0x8fc4bb53, 0xc29ff2d4, 0x8fac988f, 0xc2cc13c7, 0x8f94871d, 0xc2f83e2a, 
   0x8f7c8701, 0xc32471f7, 0x8f649840, 0xc350af26, 0x8f4cbadb, 0xc37cf5b0, 
   0x8f34eed8, 0xc3a94590, 0x8f1d343a, 0xc3d59ebe, 0x8f058b04, 0xc4020133, 
   0x8eedf33b, 0xc42e6ce8, 0x8ed66ce1, 0xc45ae1d7, 0x8ebef7fb, 0xc4875ff9, 
   0x8ea7948c, 0xc4b3e746, 0x8e904298, 0xc4e077b8, 0x8e790222, 0xc50d1149, 
   0x8e61d32e, 0xc539b3f1, 0x8e4ab5bf, 0xc5665fa9, 0x8e33a9da, 0xc593146a, 
   0x8e1caf80, 0xc5bfd22e, 0x8e05c6b7, 0xc5ec98ee, 0x8deeef82, 0xc61968a2, 
   0x8dd829e4, 0xc6464144, 0x8dc175e0, 0xc67322ce, 0x8daad37b, 0xc6a00d37, 
   0x8d9442b8, 0xc6cd0079, 0x8d7dc399, 0xc6f9fc8d, 0x8d675623, 0xc727016d, 
   0x8d50fa59, 0xc7540f11, 0x8d3ab03f, 0xc7812572, 0x8d2477d8, 0xc7ae4489, 
   0x8d0e5127, 0xc7db6c50, 0x8cf83c30, 0xc8089cbf, 0x8ce238f6, 0xc835d5d0, 
   0x8ccc477d, 0xc863177b, 0x8cb667c8, 0xc89061ba, 0x8ca099da, 0xc8bdb485, 
   0x8c8addb7, 0xc8eb0fd6, 0x8c753362, 0xc91873a5, 0x8c5f9ade, 0xc945dfec, 
   0x8c4a142f, 0xc97354a4, 0x8c349f58, 0xc9a0d1c5, 0x8c1f3c5d, 0xc9ce5748, 
   0x8c09eb40, 0xc9fbe527, 0x8bf4ac05, 0xca297b5a, 0x8bdf7eb0, 0xca5719db, 
   0x8bca6343, 0xca84c0a3, 0x8bb559c1, 0xcab26fa9, 0x8ba0622f, 0xcae026e8, 
   0x8b8b7c8f, 0xcb0de658, 0x8b76a8e4, 0xcb3badf3, 0x8b61e733, 0xcb697db0, 
   0x8b4d377c, 0xcb97558a, 0x8b3899c6, 0xcbc53579, 0x8b240e11, 0xcbf31d75, 
   0x8b0f9462, 0xcc210d79, 0x8afb2cbb, 0xcc4f057c, 0x8ae6d720, 0xcc7d0578, 
   0x8ad29394, 0xccab0d65, 0x8abe6219, 0xccd91d3d, 0x8aaa42b4, 0xcd0734f9, 
   0x8a963567, 0xcd355491, 0x8a823a36, 0xcd637bfe, 0x8a6e5123, 0xcd91ab39, 
   0x8a5a7a31, 0xcdbfe23a, 0x8a46b564, 0xcdee20fc, 0x8a3302be, 0xce1c6777, 
   0x8a1f6243, 0xce4ab5a2, 0x8a0bd3f5, 0xce790b79, 0x89f857d8, 0xcea768f2, 
   0x89e4edef, 0xced5ce08, 0x89d1963c, 0xcf043ab3, 0x89be50c3, 0xcf32aeeb, 
   0x89ab1d87, 0xcf612aaa, 0x8997fc8a, 0xcf8fade9, 0x8984edcf, 0xcfbe389f, 
   0x8971f15a, 0xcfeccac7, 0x895f072e, 0xd01b6459, 0x894c2f4c, 0xd04a054e, 
   0x893969b9, 0xd078ad9e, 0x8926b677, 0xd0a75d42, 0x89141589, 0xd0d61434, 
   0x890186f2, 0xd104d26b, 0x88ef0ab4, 0xd13397e2, 0x88dca0d3, 0xd1626490, 
   0x88ca4951, 0xd191386e, 0x88b80432, 0xd1c01375, 0x88a5d177, 0xd1eef59e, 
   0x8893b125, 0xd21ddee2, 0x8881a33d, 0xd24ccf39, 0x886fa7c2, 0xd27bc69c, 
   0x885dbeb8, 0xd2aac504, 0x884be821, 0xd2d9ca6a, 0x883a23ff, 0xd308d6c7, 
   0x88287256, 0xd337ea12, 0x8816d327, 0xd3670446, 0x88054677, 0xd396255a, 
   0x87f3cc48, 0xd3c54d47, 0x87e2649b, 0xd3f47c06, 0x87d10f75, 0xd423b191, 
   0x87bfccd7, 0xd452eddf, 0x87ae9cc5, 0xd48230e9, 0x879d7f41, 0xd4b17aa8, 
   0x878c744d, 0xd4e0cb15, 0x877b7bec, 0xd5102228, 0x876a9621, 0xd53f7fda, 
   0x8759c2ef, 0xd56ee424, 0x87490258, 0xd59e4eff, 0x8738545e, 0xd5cdc062, 
   0x8727b905, 0xd5fd3848, 0x8717304e, 0xd62cb6a8, 0x8706ba3d, 0xd65c3b7b, 
   0x86f656d3, 0xd68bc6ba, 0x86e60614, 0xd6bb585e, 0x86d5c802, 0xd6eaf05f, 
   0x86c59c9f, 0xd71a8eb5, 0x86b583ee, 0xd74a335b, 0x86a57df2, 0xd779de47, 
   0x86958aac, 0xd7a98f73, 0x8685aa20, 0xd7d946d8, 0x8675dc4f, 0xd809046e, 
   0x8666213c, 0xd838c82d, 0x865678eb, 0xd868920f, 0x8646e35c, 0xd898620c, 
   0x86376092, 0xd8c8381d, 0x8627f091, 0xd8f81439, 0x86189359, 0xd927f65b, 
   0x860948ef, 0xd957de7a, 0x85fa1153, 0xd987cc90, 0x85eaec88, 0xd9b7c094, 
   0x85dbda91, 0xd9e7ba7f, 0x85ccdb70, 0xda17ba4a, 0x85bdef28, 0xda47bfee, 
   0x85af15b9, 0xda77cb63, 0x85a04f28, 0xdaa7dca1, 0x85919b76, 0xdad7f3a2, 
   0x8582faa5, 0xdb08105e, 0x85746cb8, 0xdb3832cd, 0x8565f1b0, 0xdb685ae9, 
   0x85578991, 0xdb9888a8, 0x8549345c, 0xdbc8bc06, 0x853af214, 0xdbf8f4f8, 
   0x852cc2bb, 0xdc293379, 0x851ea652, 0xdc597781, 0x85109cdd, 0xdc89c109, 
   0x8502a65c, 0xdcba1008, 0x84f4c2d4, 0xdcea6478, 0x84e6f244, 0xdd1abe51, 
   0x84d934b1, 0xdd4b1d8c, 0x84cb8a1b, 0xdd7b8220, 0x84bdf286, 0xddabec08, 
   0x84b06df2, 0xdddc5b3b, 0x84a2fc62, 0xde0ccfb1, 0x84959dd9, 0xde3d4964, 
   0x84885258, 0xde6dc84b, 0x847b19e1, 0xde9e4c60, 0x846df477, 0xdeced59b, 
   0x8460e21a, 0xdeff63f4, 0x8453e2cf, 0xdf2ff764, 0x8446f695, 0xdf608fe4, 
   0x843a1d70, 0xdf912d6b, 0x842d5762, 0xdfc1cff3, 0x8420a46c, 0xdff27773, 
   0x84140490, 0xe02323e5, 0x840777d0, 0xe053d541, 0x83fafe2e, 0xe0848b7f, 
   0x83ee97ad, 0xe0b54698, 0x83e2444d, 0xe0e60685, 0x83d60412, 0xe116cb3d, 
   0x83c9d6fc, 0xe14794ba, 0x83bdbd0e, 0xe17862f3, 0x83b1b649, 0xe1a935e2, 
   0x83a5c2b0, 0xe1da0d7e, 0x8399e244, 0xe20ae9c1, 0x838e1507, 0xe23bcaa2, 
   0x83825afb, 0xe26cb01b, 0x8376b422, 0xe29d9a23, 0x836b207d, 0xe2ce88b3, 
   0x835fa00f, 0xe2ff7bc3, 0x835432d8, 0xe330734d, 0x8348d8dc, 0xe3616f48, 
   0x833d921b, 0xe3926fad, 0x83325e97, 0xe3c37474, 0x83273e52, 0xe3f47d96, 
   0x831c314e, 0xe4258b0a, 0x8311378d, 0xe4569ccb, 0x83065110, 0xe487b2d0, 
   0x82fb7dd8, 0xe4b8cd11, 0x82f0bde8, 0xe4e9eb87, 0x82e61141, 0xe51b0e2a, 
   0x82db77e5, 0xe54c34f3, 0x82d0f1d5, 0xe57d5fda, 0x82c67f14, 0xe5ae8ed8, 
   0x82bc1fa2, 0xe5dfc1e5, 0x82b1d381, 0xe610f8f9, 0x82a79ab3, 0xe642340d, 
   0x829d753a, 0xe6737319, 0x82936317, 0xe6a4b616, 0x8289644b, 0xe6d5fcfc, 
   0x827f78d8, 0xe70747c4, 0x8275a0c0, 0xe7389665, 0x826bdc04, 0xe769e8d8, 
   0x82622aa6, 0xe79b3f16, 0x82588ca7, 0xe7cc9917, 0x824f0208, 0xe7fdf6d4, 
   0x82458acc, 0xe82f5844, 0x823c26f3, 0xe860bd61, 0x8232d67f, 0xe8922622, 
   0x82299971, 0xe8c39280, 0x82206fcc, 0xe8f50273, 0x82175990, 0xe92675f4, 
   0x820e56be, 0xe957ecfb, 0x82056758, 0xe9896781, 0x81fc8b60, 0xe9bae57d, 
   0x81f3c2d7, 0xe9ec66e8, 0x81eb0dbe, 0xea1debbb, 0x81e26c16, 0xea4f73ee, 
   0x81d9dde1, 0xea80ff7a, 0x81d16321, 0xeab28e56, 0x81c8fbd6, 0xeae4207a, 
   0x81c0a801, 0xeb15b5e1, 0x81b867a5, 0xeb474e81, 0x81b03ac2, 0xeb78ea52, 
   0x81a82159, 0xebaa894f, 0x81a01b6d, 0xebdc2b6e, 0x819828fd, 0xec0dd0a8, 
   0x81904a0c, 0xec3f78f6, 0x81887e9a, 0xec71244f, 0x8180c6a9, 0xeca2d2ad, 
   0x8179223a, 0xecd48407, 0x8171914e, 0xed063856, 0x816a13e6, 0xed37ef91, 
   0x8162aa04, 0xed69a9b3, 0x815b53a8, 0xed9b66b2, 0x815410d4, 0xedcd2687, 
   0x814ce188, 0xedfee92b, 0x8145c5c7, 0xee30ae96, 0x813ebd90, 0xee6276bf, 
   0x8137c8e6, 0xee9441a0, 0x8130e7c9, 0xeec60f31, 0x812a1a3a, 0xeef7df6a, 
   0x8123603a, 0xef29b243, 0x811cb9ca, 0xef5b87b5, 0x811626ec, 0xef8d5fb8, 
   0x810fa7a0, 0xefbf3a45, 0x81093be8, 0xeff11753, 0x8102e3c4, 0xf022f6da, 
   0x80fc9f35, 0xf054d8d5, 0x80f66e3c, 0xf086bd39, 0x80f050db, 0xf0b8a401, 
   0x80ea4712, 0xf0ea8d24, 0x80e450e2, 0xf11c789a, 0x80de6e4c, 0xf14e665c, 
   0x80d89f51, 0xf1805662, 0x80d2e3f2, 0xf1b248a5, 0x80cd3c2f, 0xf1e43d1c, 
   0x80c7a80a, 0xf21633c0, 0x80c22784, 0xf2482c8a, 0x80bcba9d, 0xf27a2771, 
   0x80b76156, 0xf2ac246e, 0x80b21baf, 0xf2de2379, 0x80ace9ab, 0xf310248a, 
   0x80a7cb49, 0xf342279b, 0x80a2c08b, 0xf3742ca2, 0x809dc971, 0xf3a63398, 
   0x8098e5fb, 0xf3d83c77, 0x8094162c, 0xf40a4735, 0x808f5a02, 0xf43c53cb, 
   0x808ab180, 0xf46e6231, 0x80861ca6, 0xf4a07261, 0x80819b74, 0xf4d28451, 
   0x807d2dec, 0xf50497fb, 0x8078d40d, 0xf536ad56, 0x80748dd9, 0xf568c45b, 
   0x80705b50, 0xf59add02, 0x806c3c74, 0xf5ccf743, 0x80683143, 0xf5ff1318, 
   0x806439c0, 0xf6313077, 0x806055eb, 0xf6634f59, 0x805c85c4, 0xf6956fb7, 
   0x8058c94c, 0xf6c79188, 0x80552084, 0xf6f9b4c6, 0x80518b6b, 0xf72bd967, 
   0x804e0a04, 0xf75dff66, 0x804a9c4d, 0xf79026b9, 0x80474248, 0xf7c24f59, 
   0x8043fbf6, 0xf7f4793e, 0x8040c956, 0xf826a462, 0x803daa6a, 0xf858d0bb, 
   0x803a9f31, 0xf88afe42, 0x8037a7ac, 0xf8bd2cef, 0x8034c3dd, 0xf8ef5cbb, 
   0x8031f3c2, 0xf9218d9e, 0x802f375d, 0xf953bf91, 0x802c8ead, 0xf985f28a, 
   0x8029f9b4, 0xf9b82684, 0x80277872, 0xf9ea5b75, 0x80250ae7, 0xfa1c9157, 
   0x8022b114, 0xfa4ec821, 0x80206af8, 0xfa80ffcb, 0x801e3895, 0xfab3384f, 
   0x801c19ea, 0xfae571a4, 0x801a0ef8, 0xfb17abc2, 0x801817bf, 0xfb49e6a3, 
   0x80163440, 0xfb7c223d, 0x8014647b, 0xfbae5e89, 0x8012a86f, 0xfbe09b80, 
   0x8011001f, 0xfc12d91a, 0x800f6b88, 0xfc45174e, 0x800deaad, 0xfc775616, 
   0x800c7d8c, 0xfca9956a, 0x800b2427, 0xfcdbd541, 0x8009de7e, 0xfd0e1594, 
   0x8008ac90, 0xfd40565c, 0x80078e5e, 0xfd729790, 0x800683e8, 0xfda4d929, 
   0x80058d2f, 0xfdd71b1e, 0x8004aa32, 0xfe095d69, 0x8003daf1, 0xfe3ba002, 
   0x80031f6d, 0xfe6de2e0, 0x800277a6, 0xfea025fd, 0x8001e39b, 0xfed2694f, 
   0x8001634e, 0xff04acd0, 0x8000f6bd, 0xff36f078, 0x80009dea, 0xff69343f, 
   0x800058d4, 0xff9b781d, 0x8000277a, 0xffcdbc0b, 0x800009df
};


/*    
* @brief  Q15 Twiddle factors Table    
*/

/**   
* \par    
* Example code for Q15 Twiddle factors Generation::    
* \par    
* <pre>for(i = 0; i< 3N/4; i++)    
* {    
*	twiddleCoefQ15[2*i]= cos(i * 2*PI/(float)N);    
*	twiddleCoefQ15[2*i+1]= sin(i * 2*PI/(float)N);    
* } </pre>    
* \par    
* where N = 4096	and PI = 3.14159265358979    
* \par    
* Cos and Sin values are interleaved fashion    
* \par    
* Convert Floating point to Q15(Fixed point 1.15):    
*	round(twiddleCoefQ15(i) * pow(2, 15))    
*    
*/

const q15_t ALIGN4 twiddleCoefQ15[6144] = {

  0x7fff, 0x0, 0x7fff, 0x32, 0x7fff, 0x65, 0x7fff, 0x97,
  0x7fff, 0xc9, 0x7fff, 0xfb, 0x7fff, 0x12e, 0x7ffe, 0x160,
  0x7ffe, 0x192, 0x7ffd, 0x1c4, 0x7ffc, 0x1f7, 0x7ffb, 0x229,
  0x7ffa, 0x25b, 0x7ff9, 0x28d, 0x7ff8, 0x2c0, 0x7ff7, 0x2f2,
  0x7ff6, 0x324, 0x7ff5, 0x356, 0x7ff4, 0x389, 0x7ff2, 0x3bb,
  0x7ff1, 0x3ed, 0x7fef, 0x41f, 0x7fed, 0x452, 0x7fec, 0x484,
  0x7fea, 0x4b6, 0x7fe8, 0x4e8, 0x7fe6, 0x51b, 0x7fe4, 0x54d,
  0x7fe2, 0x57f, 0x7fe0, 0x5b1, 0x7fdd, 0x5e3, 0x7fdb, 0x616,
  0x7fd9, 0x648, 0x7fd6, 0x67a, 0x7fd3, 0x6ac, 0x7fd1, 0x6de,
  0x7fce, 0x711, 0x7fcb, 0x743, 0x7fc8, 0x775, 0x7fc5, 0x7a7,
  0x7fc2, 0x7d9, 0x7fbf, 0x80c, 0x7fbc, 0x83e, 0x7fb9, 0x870,
  0x7fb5, 0x8a2, 0x7fb2, 0x8d4, 0x7fae, 0x906, 0x7fab, 0x938,
  0x7fa7, 0x96b, 0x7fa3, 0x99d, 0x7fa0, 0x9cf, 0x7f9c, 0xa01,
  0x7f98, 0xa33, 0x7f94, 0xa65, 0x7f90, 0xa97, 0x7f8b, 0xac9,
  0x7f87, 0xafb, 0x7f83, 0xb2d, 0x7f7e, 0xb60, 0x7f7a, 0xb92,
  0x7f75, 0xbc4, 0x7f71, 0xbf6, 0x7f6c, 0xc28, 0x7f67, 0xc5a,
  0x7f62, 0xc8c, 0x7f5d, 0xcbe, 0x7f58, 0xcf0, 0x7f53, 0xd22,
  0x7f4e, 0xd54, 0x7f49, 0xd86, 0x7f43, 0xdb8, 0x7f3e, 0xdea,
  0x7f38, 0xe1c, 0x7f33, 0xe4e, 0x7f2d, 0xe80, 0x7f27, 0xeb2,
  0x7f22, 0xee4, 0x7f1c, 0xf15, 0x7f16, 0xf47, 0x7f10, 0xf79,
  0x7f0a, 0xfab, 0x7f03, 0xfdd, 0x7efd, 0x100f, 0x7ef7, 0x1041,
  0x7ef0, 0x1073, 0x7eea, 0x10a4, 0x7ee3, 0x10d6, 0x7edd, 0x1108,
  0x7ed6, 0x113a, 0x7ecf, 0x116c, 0x7ec8, 0x119e, 0x7ec1, 0x11cf,
  0x7eba, 0x1201, 0x7eb3, 0x1233, 0x7eac, 0x1265, 0x7ea5, 0x1296,
  0x7e9d, 0x12c8, 0x7e96, 0x12fa, 0x7e8e, 0x132b, 0x7e87, 0x135d,
  0x7e7f, 0x138f, 0x7e78, 0x13c1, 0x7e70, 0x13f2, 0x7e68, 0x1424,
  0x7e60, 0x1455, 0x7e58, 0x1487, 0x7e50, 0x14b9, 0x7e48, 0x14ea,
  0x7e3f, 0x151c, 0x7e37, 0x154d, 0x7e2f, 0x157f, 0x7e26, 0x15b1,
  0x7e1e, 0x15e2, 0x7e15, 0x1614, 0x7e0c, 0x1645, 0x7e03, 0x1677,
  0x7dfb, 0x16a8, 0x7df2, 0x16da, 0x7de9, 0x170b, 0x7de0, 0x173c,
  0x7dd6, 0x176e, 0x7dcd, 0x179f, 0x7dc4, 0x17d1, 0x7dba, 0x1802,
  0x7db1, 0x1833, 0x7da7, 0x1865, 0x7d9e, 0x1896, 0x7d94, 0x18c7,
  0x7d8a, 0x18f9, 0x7d81, 0x192a, 0x7d77, 0x195b, 0x7d6d, 0x198d,
  0x7d63, 0x19be, 0x7d58, 0x19ef, 0x7d4e, 0x1a20, 0x7d44, 0x1a51,
  0x7d3a, 0x1a83, 0x7d2f, 0x1ab4, 0x7d25, 0x1ae5, 0x7d1a, 0x1b16,
  0x7d0f, 0x1b47, 0x7d05, 0x1b78, 0x7cfa, 0x1ba9, 0x7cef, 0x1bda,
  0x7ce4, 0x1c0c, 0x7cd9, 0x1c3d, 0x7cce, 0x1c6e, 0x7cc2, 0x1c9f,
  0x7cb7, 0x1cd0, 0x7cac, 0x1d01, 0x7ca0, 0x1d31, 0x7c95, 0x1d62,
  0x7c89, 0x1d93, 0x7c7e, 0x1dc4, 0x7c72, 0x1df5, 0x7c66, 0x1e26,
  0x7c5a, 0x1e57, 0x7c4e, 0x1e88, 0x7c42, 0x1eb8, 0x7c36, 0x1ee9,
  0x7c2a, 0x1f1a, 0x7c1e, 0x1f4b, 0x7c11, 0x1f7b, 0x7c05, 0x1fac,
  0x7bf9, 0x1fdd, 0x7bec, 0x200e, 0x7bdf, 0x203e, 0x7bd3, 0x206f,
  0x7bc6, 0x209f, 0x7bb9, 0x20d0, 0x7bac, 0x2101, 0x7b9f, 0x2131,
  0x7b92, 0x2162, 0x7b85, 0x2192, 0x7b78, 0x21c3, 0x7b6a, 0x21f3,
  0x7b5d, 0x2224, 0x7b50, 0x2254, 0x7b42, 0x2284, 0x7b34, 0x22b5,
  0x7b27, 0x22e5, 0x7b19, 0x2316, 0x7b0b, 0x2346, 0x7afd, 0x2376,
  0x7aef, 0x23a7, 0x7ae1, 0x23d7, 0x7ad3, 0x2407, 0x7ac5, 0x2437,
  0x7ab7, 0x2467, 0x7aa8, 0x2498, 0x7a9a, 0x24c8, 0x7a8c, 0x24f8,
  0x7a7d, 0x2528, 0x7a6e, 0x2558, 0x7a60, 0x2588, 0x7a51, 0x25b8,
  0x7a42, 0x25e8, 0x7a33, 0x2618, 0x7a24, 0x2648, 0x7a15, 0x2678,
  0x7a06, 0x26a8, 0x79f7, 0x26d8, 0x79e7, 0x2708, 0x79d8, 0x2738,
  0x79c9, 0x2768, 0x79b9, 0x2797, 0x79aa, 0x27c7, 0x799a, 0x27f7,
  0x798a, 0x2827, 0x797a, 0x2856, 0x796a, 0x2886, 0x795b, 0x28b6,
  0x794a, 0x28e5, 0x793a, 0x2915, 0x792a, 0x2945, 0x791a, 0x2974,
  0x790a, 0x29a4, 0x78f9, 0x29d3, 0x78e9, 0x2a03, 0x78d8, 0x2a32,
  0x78c8, 0x2a62, 0x78b7, 0x2a91, 0x78a6, 0x2ac1, 0x7895, 0x2af0,
  0x7885, 0x2b1f, 0x7874, 0x2b4f, 0x7863, 0x2b7e, 0x7851, 0x2bad,
  0x7840, 0x2bdc, 0x782f, 0x2c0c, 0x781e, 0x2c3b, 0x780c, 0x2c6a,
  0x77fb, 0x2c99, 0x77e9, 0x2cc8, 0x77d8, 0x2cf7, 0x77c6, 0x2d26,
  0x77b4, 0x2d55, 0x77a2, 0x2d84, 0x7790, 0x2db3, 0x777e, 0x2de2,
  0x776c, 0x2e11, 0x775a, 0x2e40, 0x7748, 0x2e6f, 0x7736, 0x2e9e,
  0x7723, 0x2ecc, 0x7711, 0x2efb, 0x76fe, 0x2f2a, 0x76ec, 0x2f59,
  0x76d9, 0x2f87, 0x76c7, 0x2fb6, 0x76b4, 0x2fe5, 0x76a1, 0x3013,
  0x768e, 0x3042, 0x767b, 0x3070, 0x7668, 0x309f, 0x7655, 0x30cd,
  0x7642, 0x30fc, 0x762e, 0x312a, 0x761b, 0x3159, 0x7608, 0x3187,
  0x75f4, 0x31b5, 0x75e1, 0x31e4, 0x75cd, 0x3212, 0x75b9, 0x3240,
  0x75a6, 0x326e, 0x7592, 0x329d, 0x757e, 0x32cb, 0x756a, 0x32f9,
  0x7556, 0x3327, 0x7542, 0x3355, 0x752d, 0x3383, 0x7519, 0x33b1,
  0x7505, 0x33df, 0x74f0, 0x340d, 0x74dc, 0x343b, 0x74c7, 0x3469,
  0x74b3, 0x3497, 0x749e, 0x34c4, 0x7489, 0x34f2, 0x7475, 0x3520,
  0x7460, 0x354e, 0x744b, 0x357b, 0x7436, 0x35a9, 0x7421, 0x35d7,
  0x740b, 0x3604, 0x73f6, 0x3632, 0x73e1, 0x365f, 0x73cb, 0x368d,
  0x73b6, 0x36ba, 0x73a0, 0x36e8, 0x738b, 0x3715, 0x7375, 0x3742,
  0x735f, 0x3770, 0x734a, 0x379d, 0x7334, 0x37ca, 0x731e, 0x37f7,
  0x7308, 0x3825, 0x72f2, 0x3852, 0x72dc, 0x387f, 0x72c5, 0x38ac,
  0x72af, 0x38d9, 0x7299, 0x3906, 0x7282, 0x3933, 0x726c, 0x3960,
  0x7255, 0x398d, 0x723f, 0x39ba, 0x7228, 0x39e7, 0x7211, 0x3a13,
  0x71fa, 0x3a40, 0x71e3, 0x3a6d, 0x71cc, 0x3a9a, 0x71b5, 0x3ac6,
  0x719e, 0x3af3, 0x7187, 0x3b20, 0x7170, 0x3b4c, 0x7158, 0x3b79,
  0x7141, 0x3ba5, 0x712a, 0x3bd2, 0x7112, 0x3bfe, 0x70fa, 0x3c2a,
  0x70e3, 0x3c57, 0x70cb, 0x3c83, 0x70b3, 0x3caf, 0x709b, 0x3cdc,
  0x7083, 0x3d08, 0x706b, 0x3d34, 0x7053, 0x3d60, 0x703b, 0x3d8c,
  0x7023, 0x3db8, 0x700b, 0x3de4, 0x6ff2, 0x3e10, 0x6fda, 0x3e3c,
  0x6fc2, 0x3e68, 0x6fa9, 0x3e94, 0x6f90, 0x3ec0, 0x6f78, 0x3eec,
  0x6f5f, 0x3f17, 0x6f46, 0x3f43, 0x6f2d, 0x3f6f, 0x6f14, 0x3f9a,
  0x6efb, 0x3fc6, 0x6ee2, 0x3ff1, 0x6ec9, 0x401d, 0x6eb0, 0x4048,
  0x6e97, 0x4074, 0x6e7d, 0x409f, 0x6e64, 0x40cb, 0x6e4a, 0x40f6,
  0x6e31, 0x4121, 0x6e17, 0x414d, 0x6dfe, 0x4178, 0x6de4, 0x41a3,
  0x6dca, 0x41ce, 0x6db0, 0x41f9, 0x6d96, 0x4224, 0x6d7c, 0x424f,
  0x6d62, 0x427a, 0x6d48, 0x42a5, 0x6d2e, 0x42d0, 0x6d14, 0x42fb,
  0x6cf9, 0x4326, 0x6cdf, 0x4351, 0x6cc4, 0x437b, 0x6caa, 0x43a6,
  0x6c8f, 0x43d1, 0x6c75, 0x43fb, 0x6c5a, 0x4426, 0x6c3f, 0x4450,
  0x6c24, 0x447b, 0x6c09, 0x44a5, 0x6bee, 0x44d0, 0x6bd3, 0x44fa,
  0x6bb8, 0x4524, 0x6b9d, 0x454f, 0x6b82, 0x4579, 0x6b66, 0x45a3,
  0x6b4b, 0x45cd, 0x6b30, 0x45f7, 0x6b14, 0x4621, 0x6af8, 0x464b,
  0x6add, 0x4675, 0x6ac1, 0x469f, 0x6aa5, 0x46c9, 0x6a89, 0x46f3,
  0x6a6e, 0x471d, 0x6a52, 0x4747, 0x6a36, 0x4770, 0x6a1a, 0x479a,
  0x69fd, 0x47c4, 0x69e1, 0x47ed, 0x69c5, 0x4817, 0x69a9, 0x4840,
  0x698c, 0x486a, 0x6970, 0x4893, 0x6953, 0x48bd, 0x6937, 0x48e6,
  0x691a, 0x490f, 0x68fd, 0x4939, 0x68e0, 0x4962, 0x68c4, 0x498b,
  0x68a7, 0x49b4, 0x688a, 0x49dd, 0x686d, 0x4a06, 0x6850, 0x4a2f,
  0x6832, 0x4a58, 0x6815, 0x4a81, 0x67f8, 0x4aaa, 0x67da, 0x4ad3,
  0x67bd, 0x4afb, 0x67a0, 0x4b24, 0x6782, 0x4b4d, 0x6764, 0x4b75,
  0x6747, 0x4b9e, 0x6729, 0x4bc7, 0x670b, 0x4bef, 0x66ed, 0x4c17,
  0x66d0, 0x4c40, 0x66b2, 0x4c68, 0x6693, 0x4c91, 0x6675, 0x4cb9,
  0x6657, 0x4ce1, 0x6639, 0x4d09, 0x661b, 0x4d31, 0x65fc, 0x4d59,
  0x65de, 0x4d81, 0x65c0, 0x4da9, 0x65a1, 0x4dd1, 0x6582, 0x4df9,
  0x6564, 0x4e21, 0x6545, 0x4e49, 0x6526, 0x4e71, 0x6507, 0x4e98,
  0x64e9, 0x4ec0, 0x64ca, 0x4ee8, 0x64ab, 0x4f0f, 0x648b, 0x4f37,
  0x646c, 0x4f5e, 0x644d, 0x4f85, 0x642e, 0x4fad, 0x640f, 0x4fd4,
  0x63ef, 0x4ffb, 0x63d0, 0x5023, 0x63b0, 0x504a, 0x6391, 0x5071,
  0x6371, 0x5098, 0x6351, 0x50bf, 0x6332, 0x50e6, 0x6312, 0x510d,
  0x62f2, 0x5134, 0x62d2, 0x515b, 0x62b2, 0x5181, 0x6292, 0x51a8,
  0x6272, 0x51cf, 0x6252, 0x51f5, 0x6232, 0x521c, 0x6211, 0x5243,
  0x61f1, 0x5269, 0x61d1, 0x5290, 0x61b0, 0x52b6, 0x6190, 0x52dc,
  0x616f, 0x5303, 0x614e, 0x5329, 0x612e, 0x534f, 0x610d, 0x5375,
  0x60ec, 0x539b, 0x60cb, 0x53c1, 0x60aa, 0x53e7, 0x6089, 0x540d,
  0x6068, 0x5433, 0x6047, 0x5459, 0x6026, 0x547f, 0x6005, 0x54a4,
  0x5fe4, 0x54ca, 0x5fc2, 0x54f0, 0x5fa1, 0x5515, 0x5f80, 0x553b,
  0x5f5e, 0x5560, 0x5f3c, 0x5586, 0x5f1b, 0x55ab, 0x5ef9, 0x55d0,
  0x5ed7, 0x55f6, 0x5eb6, 0x561b, 0x5e94, 0x5640, 0x5e72, 0x5665,
  0x5e50, 0x568a, 0x5e2e, 0x56af, 0x5e0c, 0x56d4, 0x5dea, 0x56f9,
  0x5dc8, 0x571e, 0x5da5, 0x5743, 0x5d83, 0x5767, 0x5d61, 0x578c,
  0x5d3e, 0x57b1, 0x5d1c, 0x57d5, 0x5cf9, 0x57fa, 0x5cd7, 0x581e,
  0x5cb4, 0x5843, 0x5c91, 0x5867, 0x5c6f, 0x588c, 0x5c4c, 0x58b0,
  0x5c29, 0x58d4, 0x5c06, 0x58f8, 0x5be3, 0x591c, 0x5bc0, 0x5940,
  0x5b9d, 0x5964, 0x5b7a, 0x5988, 0x5b57, 0x59ac, 0x5b34, 0x59d0,
  0x5b10, 0x59f4, 0x5aed, 0x5a18, 0x5ac9, 0x5a3b, 0x5aa6, 0x5a5f,
  0x5a82, 0x5a82, 0x5a5f, 0x5aa6, 0x5a3b, 0x5ac9, 0x5a18, 0x5aed,
  0x59f4, 0x5b10, 0x59d0, 0x5b34, 0x59ac, 0x5b57, 0x5988, 0x5b7a,
  0x5964, 0x5b9d, 0x5940, 0x5bc0, 0x591c, 0x5be3, 0x58f8, 0x5c06,
  0x58d4, 0x5c29, 0x58b0, 0x5c4c, 0x588c, 0x5c6f, 0x5867, 0x5c91,
  0x5843, 0x5cb4, 0x581e, 0x5cd7, 0x57fa, 0x5cf9, 0x57d5, 0x5d1c,
  0x57b1, 0x5d3e, 0x578c, 0x5d61, 0x5767, 0x5d83, 0x5743, 0x5da5,
  0x571e, 0x5dc8, 0x56f9, 0x5dea, 0x56d4, 0x5e0c, 0x56af, 0x5e2e,
  0x568a, 0x5e50, 0x5665, 0x5e72, 0x5640, 0x5e94, 0x561b, 0x5eb6,
  0x55f6, 0x5ed7, 0x55d0, 0x5ef9, 0x55ab, 0x5f1b, 0x5586, 0x5f3c,
  0x5560, 0x5f5e, 0x553b, 0x5f80, 0x5515, 0x5fa1, 0x54f0, 0x5fc2,
  0x54ca, 0x5fe4, 0x54a4, 0x6005, 0x547f, 0x6026, 0x5459, 0x6047,
  0x5433, 0x6068, 0x540d, 0x6089, 0x53e7, 0x60aa, 0x53c1, 0x60cb,
  0x539b, 0x60ec, 0x5375, 0x610d, 0x534f, 0x612e, 0x5329, 0x614e,
  0x5303, 0x616f, 0x52dc, 0x6190, 0x52b6, 0x61b0, 0x5290, 0x61d1,
  0x5269, 0x61f1, 0x5243, 0x6211, 0x521c, 0x6232, 0x51f5, 0x6252,
  0x51cf, 0x6272, 0x51a8, 0x6292, 0x5181, 0x62b2, 0x515b, 0x62d2,
  0x5134, 0x62f2, 0x510d, 0x6312, 0x50e6, 0x6332, 0x50bf, 0x6351,
  0x5098, 0x6371, 0x5071, 0x6391, 0x504a, 0x63b0, 0x5023, 0x63d0,
  0x4ffb, 0x63ef, 0x4fd4, 0x640f, 0x4fad, 0x642e, 0x4f85, 0x644d,
  0x4f5e, 0x646c, 0x4f37, 0x648b, 0x4f0f, 0x64ab, 0x4ee8, 0x64ca,
  0x4ec0, 0x64e9, 0x4e98, 0x6507, 0x4e71, 0x6526, 0x4e49, 0x6545,
  0x4e21, 0x6564, 0x4df9, 0x6582, 0x4dd1, 0x65a1, 0x4da9, 0x65c0,
  0x4d81, 0x65de, 0x4d59, 0x65fc, 0x4d31, 0x661b, 0x4d09, 0x6639,
  0x4ce1, 0x6657, 0x4cb9, 0x6675, 0x4c91, 0x6693, 0x4c68, 0x66b2,
  0x4c40, 0x66d0, 0x4c17, 0x66ed, 0x4bef, 0x670b, 0x4bc7, 0x6729,
  0x4b9e, 0x6747, 0x4b75, 0x6764, 0x4b4d, 0x6782, 0x4b24, 0x67a0,
  0x4afb, 0x67bd, 0x4ad3, 0x67da, 0x4aaa, 0x67f8, 0x4a81, 0x6815,
  0x4a58, 0x6832, 0x4a2f, 0x6850, 0x4a06, 0x686d, 0x49dd, 0x688a,
  0x49b4, 0x68a7, 0x498b, 0x68c4, 0x4962, 0x68e0, 0x4939, 0x68fd,
  0x490f, 0x691a, 0x48e6, 0x6937, 0x48bd, 0x6953, 0x4893, 0x6970,
  0x486a, 0x698c, 0x4840, 0x69a9, 0x4817, 0x69c5, 0x47ed, 0x69e1,
  0x47c4, 0x69fd, 0x479a, 0x6a1a, 0x4770, 0x6a36, 0x4747, 0x6a52,
  0x471d, 0x6a6e, 0x46f3, 0x6a89, 0x46c9, 0x6aa5, 0x469f, 0x6ac1,
  0x4675, 0x6add, 0x464b, 0x6af8, 0x4621, 0x6b14, 0x45f7, 0x6b30,
  0x45cd, 0x6b4b, 0x45a3, 0x6b66, 0x4579, 0x6b82, 0x454f, 0x6b9d,
  0x4524, 0x6bb8, 0x44fa, 0x6bd3, 0x44d0, 0x6bee, 0x44a5, 0x6c09,
  0x447b, 0x6c24, 0x4450, 0x6c3f, 0x4426, 0x6c5a, 0x43fb, 0x6c75,
  0x43d1, 0x6c8f, 0x43a6, 0x6caa, 0x437b, 0x6cc4, 0x4351, 0x6cdf,
  0x4326, 0x6cf9, 0x42fb, 0x6d14, 0x42d0, 0x6d2e, 0x42a5, 0x6d48,
  0x427a, 0x6d62, 0x424f, 0x6d7c, 0x4224, 0x6d96, 0x41f9, 0x6db0,
  0x41ce, 0x6dca, 0x41a3, 0x6de4, 0x4178, 0x6dfe, 0x414d, 0x6e17,
  0x4121, 0x6e31, 0x40f6, 0x6e4a, 0x40cb, 0x6e64, 0x409f, 0x6e7d,
  0x4074, 0x6e97, 0x4048, 0x6eb0, 0x401d, 0x6ec9, 0x3ff1, 0x6ee2,
  0x3fc6, 0x6efb, 0x3f9a, 0x6f14, 0x3f6f, 0x6f2d, 0x3f43, 0x6f46,
  0x3f17, 0x6f5f, 0x3eec, 0x6f78, 0x3ec0, 0x6f90, 0x3e94, 0x6fa9,
  0x3e68, 0x6fc2, 0x3e3c, 0x6fda, 0x3e10, 0x6ff2, 0x3de4, 0x700b,
  0x3db8, 0x7023, 0x3d8c, 0x703b, 0x3d60, 0x7053, 0x3d34, 0x706b,
  0x3d08, 0x7083, 0x3cdc, 0x709b, 0x3caf, 0x70b3, 0x3c83, 0x70cb,
  0x3c57, 0x70e3, 0x3c2a, 0x70fa, 0x3bfe, 0x7112, 0x3bd2, 0x712a,
  0x3ba5, 0x7141, 0x3b79, 0x7158, 0x3b4c, 0x7170, 0x3b20, 0x7187,
  0x3af3, 0x719e, 0x3ac6, 0x71b5, 0x3a9a, 0x71cc, 0x3a6d, 0x71e3,
  0x3a40, 0x71fa, 0x3a13, 0x7211, 0x39e7, 0x7228, 0x39ba, 0x723f,
  0x398d, 0x7255, 0x3960, 0x726c, 0x3933, 0x7282, 0x3906, 0x7299,
  0x38d9, 0x72af, 0x38ac, 0x72c5, 0x387f, 0x72dc, 0x3852, 0x72f2,
  0x3825, 0x7308, 0x37f7, 0x731e, 0x37ca, 0x7334, 0x379d, 0x734a,
  0x3770, 0x735f, 0x3742, 0x7375, 0x3715, 0x738b, 0x36e8, 0x73a0,
  0x36ba, 0x73b6, 0x368d, 0x73cb, 0x365f, 0x73e1, 0x3632, 0x73f6,
  0x3604, 0x740b, 0x35d7, 0x7421, 0x35a9, 0x7436, 0x357b, 0x744b,
  0x354e, 0x7460, 0x3520, 0x7475, 0x34f2, 0x7489, 0x34c4, 0x749e,
  0x3497, 0x74b3, 0x3469, 0x74c7, 0x343b, 0x74dc, 0x340d, 0x74f0,
  0x33df, 0x7505, 0x33b1, 0x7519, 0x3383, 0x752d, 0x3355, 0x7542,
  0x3327, 0x7556, 0x32f9, 0x756a, 0x32cb, 0x757e, 0x329d, 0x7592,
  0x326e, 0x75a6, 0x3240, 0x75b9, 0x3212, 0x75cd, 0x31e4, 0x75e1,
  0x31b5, 0x75f4, 0x3187, 0x7608, 0x3159, 0x761b, 0x312a, 0x762e,
  0x30fc, 0x7642, 0x30cd, 0x7655, 0x309f, 0x7668, 0x3070, 0x767b,
  0x3042, 0x768e, 0x3013, 0x76a1, 0x2fe5, 0x76b4, 0x2fb6, 0x76c7,
  0x2f87, 0x76d9, 0x2f59, 0x76ec, 0x2f2a, 0x76fe, 0x2efb, 0x7711,
  0x2ecc, 0x7723, 0x2e9e, 0x7736, 0x2e6f, 0x7748, 0x2e40, 0x775a,
  0x2e11, 0x776c, 0x2de2, 0x777e, 0x2db3, 0x7790, 0x2d84, 0x77a2,
  0x2d55, 0x77b4, 0x2d26, 0x77c6, 0x2cf7, 0x77d8, 0x2cc8, 0x77e9,
  0x2c99, 0x77fb, 0x2c6a, 0x780c, 0x2c3b, 0x781e, 0x2c0c, 0x782f,
  0x2bdc, 0x7840, 0x2bad, 0x7851, 0x2b7e, 0x7863, 0x2b4f, 0x7874,
  0x2b1f, 0x7885, 0x2af0, 0x7895, 0x2ac1, 0x78a6, 0x2a91, 0x78b7,
  0x2a62, 0x78c8, 0x2a32, 0x78d8, 0x2a03, 0x78e9, 0x29d3, 0x78f9,
  0x29a4, 0x790a, 0x2974, 0x791a, 0x2945, 0x792a, 0x2915, 0x793a,
  0x28e5, 0x794a, 0x28b6, 0x795b, 0x2886, 0x796a, 0x2856, 0x797a,
  0x2827, 0x798a, 0x27f7, 0x799a, 0x27c7, 0x79aa, 0x2797, 0x79b9,
  0x2768, 0x79c9, 0x2738, 0x79d8, 0x2708, 0x79e7, 0x26d8, 0x79f7,
  0x26a8, 0x7a06, 0x2678, 0x7a15, 0x2648, 0x7a24, 0x2618, 0x7a33,
  0x25e8, 0x7a42, 0x25b8, 0x7a51, 0x2588, 0x7a60, 0x2558, 0x7a6e,
  0x2528, 0x7a7d, 0x24f8, 0x7a8c, 0x24c8, 0x7a9a, 0x2498, 0x7aa8,
  0x2467, 0x7ab7, 0x2437, 0x7ac5, 0x2407, 0x7ad3, 0x23d7, 0x7ae1,
  0x23a7, 0x7aef, 0x2376, 0x7afd, 0x2346, 0x7b0b, 0x2316, 0x7b19,
  0x22e5, 0x7b27, 0x22b5, 0x7b34, 0x2284, 0x7b42, 0x2254, 0x7b50,
  0x2224, 0x7b5d, 0x21f3, 0x7b6a, 0x21c3, 0x7b78, 0x2192, 0x7b85,
  0x2162, 0x7b92, 0x2131, 0x7b9f, 0x2101, 0x7bac, 0x20d0, 0x7bb9,
  0x209f, 0x7bc6, 0x206f, 0x7bd3, 0x203e, 0x7bdf, 0x200e, 0x7bec,
  0x1fdd, 0x7bf9, 0x1fac, 0x7c05, 0x1f7b, 0x7c11, 0x1f4b, 0x7c1e,
  0x1f1a, 0x7c2a, 0x1ee9, 0x7c36, 0x1eb8, 0x7c42, 0x1e88, 0x7c4e,
  0x1e57, 0x7c5a, 0x1e26, 0x7c66, 0x1df5, 0x7c72, 0x1dc4, 0x7c7e,
  0x1d93, 0x7c89, 0x1d62, 0x7c95, 0x1d31, 0x7ca0, 0x1d01, 0x7cac,
  0x1cd0, 0x7cb7, 0x1c9f, 0x7cc2, 0x1c6e, 0x7cce, 0x1c3d, 0x7cd9,
  0x1c0c, 0x7ce4, 0x1bda, 0x7cef, 0x1ba9, 0x7cfa, 0x1b78, 0x7d05,
  0x1b47, 0x7d0f, 0x1b16, 0x7d1a, 0x1ae5, 0x7d25, 0x1ab4, 0x7d2f,
  0x1a83, 0x7d3a, 0x1a51, 0x7d44, 0x1a20, 0x7d4e, 0x19ef, 0x7d58,
  0x19be, 0x7d63, 0x198d, 0x7d6d, 0x195b, 0x7d77, 0x192a, 0x7d81,
  0x18f9, 0x7d8a, 0x18c7, 0x7d94, 0x1896, 0x7d9e, 0x1865, 0x7da7,
  0x1833, 0x7db1, 0x1802, 0x7dba, 0x17d1, 0x7dc4, 0x179f, 0x7dcd,
  0x176e, 0x7dd6, 0x173c, 0x7de0, 0x170b, 0x7de9, 0x16da, 0x7df2,
  0x16a8, 0x7dfb, 0x1677, 0x7e03, 0x1645, 0x7e0c, 0x1614, 0x7e15,
  0x15e2, 0x7e1e, 0x15b1, 0x7e26, 0x157f, 0x7e2f, 0x154d, 0x7e37,
  0x151c, 0x7e3f, 0x14ea, 0x7e48, 0x14b9, 0x7e50, 0x1487, 0x7e58,
  0x1455, 0x7e60, 0x1424, 0x7e68, 0x13f2, 0x7e70, 0x13c1, 0x7e78,
  0x138f, 0x7e7f, 0x135d, 0x7e87, 0x132b, 0x7e8e, 0x12fa, 0x7e96,
  0x12c8, 0x7e9d, 0x1296, 0x7ea5, 0x1265, 0x7eac, 0x1233, 0x7eb3,
  0x1201, 0x7eba, 0x11cf, 0x7ec1, 0x119e, 0x7ec8, 0x116c, 0x7ecf,
  0x113a, 0x7ed6, 0x1108, 0x7edd, 0x10d6, 0x7ee3, 0x10a4, 0x7eea,
  0x1073, 0x7ef0, 0x1041, 0x7ef7, 0x100f, 0x7efd, 0xfdd, 0x7f03,
  0xfab, 0x7f0a, 0xf79, 0x7f10, 0xf47, 0x7f16, 0xf15, 0x7f1c,
  0xee4, 0x7f22, 0xeb2, 0x7f27, 0xe80, 0x7f2d, 0xe4e, 0x7f33,
  0xe1c, 0x7f38, 0xdea, 0x7f3e, 0xdb8, 0x7f43, 0xd86, 0x7f49,
  0xd54, 0x7f4e, 0xd22, 0x7f53, 0xcf0, 0x7f58, 0xcbe, 0x7f5d,
  0xc8c, 0x7f62, 0xc5a, 0x7f67, 0xc28, 0x7f6c, 0xbf6, 0x7f71,
  0xbc4, 0x7f75, 0xb92, 0x7f7a, 0xb60, 0x7f7e, 0xb2d, 0x7f83,
  0xafb, 0x7f87, 0xac9, 0x7f8b, 0xa97, 0x7f90, 0xa65, 0x7f94,
  0xa33, 0x7f98, 0xa01, 0x7f9c, 0x9cf, 0x7fa0, 0x99d, 0x7fa3,
  0x96b, 0x7fa7, 0x938, 0x7fab, 0x906, 0x7fae, 0x8d4, 0x7fb2,
  0x8a2, 0x7fb5, 0x870, 0x7fb9, 0x83e, 0x7fbc, 0x80c, 0x7fbf,
  0x7d9, 0x7fc2, 0x7a7, 0x7fc5, 0x775, 0x7fc8, 0x743, 0x7fcb,
  0x711, 0x7fce, 0x6de, 0x7fd1, 0x6ac, 0x7fd3, 0x67a, 0x7fd6,
  0x648, 0x7fd9, 0x616, 0x7fdb, 0x5e3, 0x7fdd, 0x5b1, 0x7fe0,
  0x57f, 0x7fe2, 0x54d, 0x7fe4, 0x51b, 0x7fe6, 0x4e8, 0x7fe8,
  0x4b6, 0x7fea, 0x484, 0x7fec, 0x452, 0x7fed, 0x41f, 0x7fef,
  0x3ed, 0x7ff1, 0x3bb, 0x7ff2, 0x389, 0x7ff4, 0x356, 0x7ff5,
  0x324, 0x7ff6, 0x2f2, 0x7ff7, 0x2c0, 0x7ff8, 0x28d, 0x7ff9,
  0x25b, 0x7ffa, 0x229, 0x7ffb, 0x1f7, 0x7ffc, 0x1c4, 0x7ffd,
  0x192, 0x7ffe, 0x160, 0x7ffe, 0x12e, 0x7fff, 0xfb, 0x7fff,
  0xc9, 0x7fff, 0x97, 0x7fff, 0x65, 0x7fff, 0x32, 0x7fff,
  0x0, 0x7fff, 0xffce, 0x7fff, 0xff9b, 0x7fff, 0xff69, 0x7fff,
  0xff37, 0x7fff, 0xff05, 0x7fff, 0xfed2, 0x7fff, 0xfea0, 0x7ffe,
  0xfe6e, 0x7ffe, 0xfe3c, 0x7ffd, 0xfe09, 0x7ffc, 0xfdd7, 0x7ffb,
  0xfda5, 0x7ffa, 0xfd73, 0x7ff9, 0xfd40, 0x7ff8, 0xfd0e, 0x7ff7,
  0xfcdc, 0x7ff6, 0xfcaa, 0x7ff5, 0xfc77, 0x7ff4, 0xfc45, 0x7ff2,
  0xfc13, 0x7ff1, 0xfbe1, 0x7fef, 0xfbae, 0x7fed, 0xfb7c, 0x7fec,
  0xfb4a, 0x7fea, 0xfb18, 0x7fe8, 0xfae5, 0x7fe6, 0xfab3, 0x7fe4,
  0xfa81, 0x7fe2, 0xfa4f, 0x7fe0, 0xfa1d, 0x7fdd, 0xf9ea, 0x7fdb,
  0xf9b8, 0x7fd9, 0xf986, 0x7fd6, 0xf954, 0x7fd3, 0xf922, 0x7fd1,
  0xf8ef, 0x7fce, 0xf8bd, 0x7fcb, 0xf88b, 0x7fc8, 0xf859, 0x7fc5,
  0xf827, 0x7fc2, 0xf7f4, 0x7fbf, 0xf7c2, 0x7fbc, 0xf790, 0x7fb9,
  0xf75e, 0x7fb5, 0xf72c, 0x7fb2, 0xf6fa, 0x7fae, 0xf6c8, 0x7fab,
  0xf695, 0x7fa7, 0xf663, 0x7fa3, 0xf631, 0x7fa0, 0xf5ff, 0x7f9c,
  0xf5cd, 0x7f98, 0xf59b, 0x7f94, 0xf569, 0x7f90, 0xf537, 0x7f8b,
  0xf505, 0x7f87, 0xf4d3, 0x7f83, 0xf4a0, 0x7f7e, 0xf46e, 0x7f7a,
  0xf43c, 0x7f75, 0xf40a, 0x7f71, 0xf3d8, 0x7f6c, 0xf3a6, 0x7f67,
  0xf374, 0x7f62, 0xf342, 0x7f5d, 0xf310, 0x7f58, 0xf2de, 0x7f53,
  0xf2ac, 0x7f4e, 0xf27a, 0x7f49, 0xf248, 0x7f43, 0xf216, 0x7f3e,
  0xf1e4, 0x7f38, 0xf1b2, 0x7f33, 0xf180, 0x7f2d, 0xf14e, 0x7f27,
  0xf11c, 0x7f22, 0xf0eb, 0x7f1c, 0xf0b9, 0x7f16, 0xf087, 0x7f10,
  0xf055, 0x7f0a, 0xf023, 0x7f03, 0xeff1, 0x7efd, 0xefbf, 0x7ef7,
  0xef8d, 0x7ef0, 0xef5c, 0x7eea, 0xef2a, 0x7ee3, 0xeef8, 0x7edd,
  0xeec6, 0x7ed6, 0xee94, 0x7ecf, 0xee62, 0x7ec8, 0xee31, 0x7ec1,
  0xedff, 0x7eba, 0xedcd, 0x7eb3, 0xed9b, 0x7eac, 0xed6a, 0x7ea5,
  0xed38, 0x7e9d, 0xed06, 0x7e96, 0xecd5, 0x7e8e, 0xeca3, 0x7e87,
  0xec71, 0x7e7f, 0xec3f, 0x7e78, 0xec0e, 0x7e70, 0xebdc, 0x7e68,
  0xebab, 0x7e60, 0xeb79, 0x7e58, 0xeb47, 0x7e50, 0xeb16, 0x7e48,
  0xeae4, 0x7e3f, 0xeab3, 0x7e37, 0xea81, 0x7e2f, 0xea4f, 0x7e26,
  0xea1e, 0x7e1e, 0xe9ec, 0x7e15, 0xe9bb, 0x7e0c, 0xe989, 0x7e03,
  0xe958, 0x7dfb, 0xe926, 0x7df2, 0xe8f5, 0x7de9, 0xe8c4, 0x7de0,
  0xe892, 0x7dd6, 0xe861, 0x7dcd, 0xe82f, 0x7dc4, 0xe7fe, 0x7dba,
  0xe7cd, 0x7db1, 0xe79b, 0x7da7, 0xe76a, 0x7d9e, 0xe739, 0x7d94,
  0xe707, 0x7d8a, 0xe6d6, 0x7d81, 0xe6a5, 0x7d77, 0xe673, 0x7d6d,
  0xe642, 0x7d63, 0xe611, 0x7d58, 0xe5e0, 0x7d4e, 0xe5af, 0x7d44,
  0xe57d, 0x7d3a, 0xe54c, 0x7d2f, 0xe51b, 0x7d25, 0xe4ea, 0x7d1a,
  0xe4b9, 0x7d0f, 0xe488, 0x7d05, 0xe457, 0x7cfa, 0xe426, 0x7cef,
  0xe3f4, 0x7ce4, 0xe3c3, 0x7cd9, 0xe392, 0x7cce, 0xe361, 0x7cc2,
  0xe330, 0x7cb7, 0xe2ff, 0x7cac, 0xe2cf, 0x7ca0, 0xe29e, 0x7c95,
  0xe26d, 0x7c89, 0xe23c, 0x7c7e, 0xe20b, 0x7c72, 0xe1da, 0x7c66,
  0xe1a9, 0x7c5a, 0xe178, 0x7c4e, 0xe148, 0x7c42, 0xe117, 0x7c36,
  0xe0e6, 0x7c2a, 0xe0b5, 0x7c1e, 0xe085, 0x7c11, 0xe054, 0x7c05,
  0xe023, 0x7bf9, 0xdff2, 0x7bec, 0xdfc2, 0x7bdf, 0xdf91, 0x7bd3,
  0xdf61, 0x7bc6, 0xdf30, 0x7bb9, 0xdeff, 0x7bac, 0xdecf, 0x7b9f,
  0xde9e, 0x7b92, 0xde6e, 0x7b85, 0xde3d, 0x7b78, 0xde0d, 0x7b6a,
  0xdddc, 0x7b5d, 0xddac, 0x7b50, 0xdd7c, 0x7b42, 0xdd4b, 0x7b34,
  0xdd1b, 0x7b27, 0xdcea, 0x7b19, 0xdcba, 0x7b0b, 0xdc8a, 0x7afd,
  0xdc59, 0x7aef, 0xdc29, 0x7ae1, 0xdbf9, 0x7ad3, 0xdbc9, 0x7ac5,
  0xdb99, 0x7ab7, 0xdb68, 0x7aa8, 0xdb38, 0x7a9a, 0xdb08, 0x7a8c,
  0xdad8, 0x7a7d, 0xdaa8, 0x7a6e, 0xda78, 0x7a60, 0xda48, 0x7a51,
  0xda18, 0x7a42, 0xd9e8, 0x7a33, 0xd9b8, 0x7a24, 0xd988, 0x7a15,
  0xd958, 0x7a06, 0xd928, 0x79f7, 0xd8f8, 0x79e7, 0xd8c8, 0x79d8,
  0xd898, 0x79c9, 0xd869, 0x79b9, 0xd839, 0x79aa, 0xd809, 0x799a,
  0xd7d9, 0x798a, 0xd7aa, 0x797a, 0xd77a, 0x796a, 0xd74a, 0x795b,
  0xd71b, 0x794a, 0xd6eb, 0x793a, 0xd6bb, 0x792a, 0xd68c, 0x791a,
  0xd65c, 0x790a, 0xd62d, 0x78f9, 0xd5fd, 0x78e9, 0xd5ce, 0x78d8,
  0xd59e, 0x78c8, 0xd56f, 0x78b7, 0xd53f, 0x78a6, 0xd510, 0x7895,
  0xd4e1, 0x7885, 0xd4b1, 0x7874, 0xd482, 0x7863, 0xd453, 0x7851,
  0xd424, 0x7840, 0xd3f4, 0x782f, 0xd3c5, 0x781e, 0xd396, 0x780c,
  0xd367, 0x77fb, 0xd338, 0x77e9, 0xd309, 0x77d8, 0xd2da, 0x77c6,
  0xd2ab, 0x77b4, 0xd27c, 0x77a2, 0xd24d, 0x7790, 0xd21e, 0x777e,
  0xd1ef, 0x776c, 0xd1c0, 0x775a, 0xd191, 0x7748, 0xd162, 0x7736,
  0xd134, 0x7723, 0xd105, 0x7711, 0xd0d6, 0x76fe, 0xd0a7, 0x76ec,
  0xd079, 0x76d9, 0xd04a, 0x76c7, 0xd01b, 0x76b4, 0xcfed, 0x76a1,
  0xcfbe, 0x768e, 0xcf90, 0x767b, 0xcf61, 0x7668, 0xcf33, 0x7655,
  0xcf04, 0x7642, 0xced6, 0x762e, 0xcea7, 0x761b, 0xce79, 0x7608,
  0xce4b, 0x75f4, 0xce1c, 0x75e1, 0xcdee, 0x75cd, 0xcdc0, 0x75b9,
  0xcd92, 0x75a6, 0xcd63, 0x7592, 0xcd35, 0x757e, 0xcd07, 0x756a,
  0xccd9, 0x7556, 0xccab, 0x7542, 0xcc7d, 0x752d, 0xcc4f, 0x7519,
  0xcc21, 0x7505, 0xcbf3, 0x74f0, 0xcbc5, 0x74dc, 0xcb97, 0x74c7,
  0xcb69, 0x74b3, 0xcb3c, 0x749e, 0xcb0e, 0x7489, 0xcae0, 0x7475,
  0xcab2, 0x7460, 0xca85, 0x744b, 0xca57, 0x7436, 0xca29, 0x7421,
  0xc9fc, 0x740b, 0xc9ce, 0x73f6, 0xc9a1, 0x73e1, 0xc973, 0x73cb,
  0xc946, 0x73b6, 0xc918, 0x73a0, 0xc8eb, 0x738b, 0xc8be, 0x7375,
  0xc890, 0x735f, 0xc863, 0x734a, 0xc836, 0x7334, 0xc809, 0x731e,
  0xc7db, 0x7308, 0xc7ae, 0x72f2, 0xc781, 0x72dc, 0xc754, 0x72c5,
  0xc727, 0x72af, 0xc6fa, 0x7299, 0xc6cd, 0x7282, 0xc6a0, 0x726c,
  0xc673, 0x7255, 0xc646, 0x723f, 0xc619, 0x7228, 0xc5ed, 0x7211,
  0xc5c0, 0x71fa, 0xc593, 0x71e3, 0xc566, 0x71cc, 0xc53a, 0x71b5,
  0xc50d, 0x719e, 0xc4e0, 0x7187, 0xc4b4, 0x7170, 0xc487, 0x7158,
  0xc45b, 0x7141, 0xc42e, 0x712a, 0xc402, 0x7112, 0xc3d6, 0x70fa,
  0xc3a9, 0x70e3, 0xc37d, 0x70cb, 0xc351, 0x70b3, 0xc324, 0x709b,
  0xc2f8, 0x7083, 0xc2cc, 0x706b, 0xc2a0, 0x7053, 0xc274, 0x703b,
  0xc248, 0x7023, 0xc21c, 0x700b, 0xc1f0, 0x6ff2, 0xc1c4, 0x6fda,
  0xc198, 0x6fc2, 0xc16c, 0x6fa9, 0xc140, 0x6f90, 0xc114, 0x6f78,
  0xc0e9, 0x6f5f, 0xc0bd, 0x6f46, 0xc091, 0x6f2d, 0xc066, 0x6f14,
  0xc03a, 0x6efb, 0xc00f, 0x6ee2, 0xbfe3, 0x6ec9, 0xbfb8, 0x6eb0,
  0xbf8c, 0x6e97, 0xbf61, 0x6e7d, 0xbf35, 0x6e64, 0xbf0a, 0x6e4a,
  0xbedf, 0x6e31, 0xbeb3, 0x6e17, 0xbe88, 0x6dfe, 0xbe5d, 0x6de4,
  0xbe32, 0x6dca, 0xbe07, 0x6db0, 0xbddc, 0x6d96, 0xbdb1, 0x6d7c,
  0xbd86, 0x6d62, 0xbd5b, 0x6d48, 0xbd30, 0x6d2e, 0xbd05, 0x6d14,
  0xbcda, 0x6cf9, 0xbcaf, 0x6cdf, 0xbc85, 0x6cc4, 0xbc5a, 0x6caa,
  0xbc2f, 0x6c8f, 0xbc05, 0x6c75, 0xbbda, 0x6c5a, 0xbbb0, 0x6c3f,
  0xbb85, 0x6c24, 0xbb5b, 0x6c09, 0xbb30, 0x6bee, 0xbb06, 0x6bd3,
  0xbadc, 0x6bb8, 0xbab1, 0x6b9d, 0xba87, 0x6b82, 0xba5d, 0x6b66,
  0xba33, 0x6b4b, 0xba09, 0x6b30, 0xb9df, 0x6b14, 0xb9b5, 0x6af8,
  0xb98b, 0x6add, 0xb961, 0x6ac1, 0xb937, 0x6aa5, 0xb90d, 0x6a89,
  0xb8e3, 0x6a6e, 0xb8b9, 0x6a52, 0xb890, 0x6a36, 0xb866, 0x6a1a,
  0xb83c, 0x69fd, 0xb813, 0x69e1, 0xb7e9, 0x69c5, 0xb7c0, 0x69a9,
  0xb796, 0x698c, 0xb76d, 0x6970, 0xb743, 0x6953, 0xb71a, 0x6937,
  0xb6f1, 0x691a, 0xb6c7, 0x68fd, 0xb69e, 0x68e0, 0xb675, 0x68c4,
  0xb64c, 0x68a7, 0xb623, 0x688a, 0xb5fa, 0x686d, 0xb5d1, 0x6850,
  0xb5a8, 0x6832, 0xb57f, 0x6815, 0xb556, 0x67f8, 0xb52d, 0x67da,
  0xb505, 0x67bd, 0xb4dc, 0x67a0, 0xb4b3, 0x6782, 0xb48b, 0x6764,
  0xb462, 0x6747, 0xb439, 0x6729, 0xb411, 0x670b, 0xb3e9, 0x66ed,
  0xb3c0, 0x66d0, 0xb398, 0x66b2, 0xb36f, 0x6693, 0xb347, 0x6675,
  0xb31f, 0x6657, 0xb2f7, 0x6639, 0xb2cf, 0x661b, 0xb2a7, 0x65fc,
  0xb27f, 0x65de, 0xb257, 0x65c0, 0xb22f, 0x65a1, 0xb207, 0x6582,
  0xb1df, 0x6564, 0xb1b7, 0x6545, 0xb18f, 0x6526, 0xb168, 0x6507,
  0xb140, 0x64e9, 0xb118, 0x64ca, 0xb0f1, 0x64ab, 0xb0c9, 0x648b,
  0xb0a2, 0x646c, 0xb07b, 0x644d, 0xb053, 0x642e, 0xb02c, 0x640f,
  0xb005, 0x63ef, 0xafdd, 0x63d0, 0xafb6, 0x63b0, 0xaf8f, 0x6391,
  0xaf68, 0x6371, 0xaf41, 0x6351, 0xaf1a, 0x6332, 0xaef3, 0x6312,
  0xaecc, 0x62f2, 0xaea5, 0x62d2, 0xae7f, 0x62b2, 0xae58, 0x6292,
  0xae31, 0x6272, 0xae0b, 0x6252, 0xade4, 0x6232, 0xadbd, 0x6211,
  0xad97, 0x61f1, 0xad70, 0x61d1, 0xad4a, 0x61b0, 0xad24, 0x6190,
  0xacfd, 0x616f, 0xacd7, 0x614e, 0xacb1, 0x612e, 0xac8b, 0x610d,
  0xac65, 0x60ec, 0xac3f, 0x60cb, 0xac19, 0x60aa, 0xabf3, 0x6089,
  0xabcd, 0x6068, 0xaba7, 0x6047, 0xab81, 0x6026, 0xab5c, 0x6005,
  0xab36, 0x5fe4, 0xab10, 0x5fc2, 0xaaeb, 0x5fa1, 0xaac5, 0x5f80,
  0xaaa0, 0x5f5e, 0xaa7a, 0x5f3c, 0xaa55, 0x5f1b, 0xaa30, 0x5ef9,
  0xaa0a, 0x5ed7, 0xa9e5, 0x5eb6, 0xa9c0, 0x5e94, 0xa99b, 0x5e72,
  0xa976, 0x5e50, 0xa951, 0x5e2e, 0xa92c, 0x5e0c, 0xa907, 0x5dea,
  0xa8e2, 0x5dc8, 0xa8bd, 0x5da5, 0xa899, 0x5d83, 0xa874, 0x5d61,
  0xa84f, 0x5d3e, 0xa82b, 0x5d1c, 0xa806, 0x5cf9, 0xa7e2, 0x5cd7,
  0xa7bd, 0x5cb4, 0xa799, 0x5c91, 0xa774, 0x5c6f, 0xa750, 0x5c4c,
  0xa72c, 0x5c29, 0xa708, 0x5c06, 0xa6e4, 0x5be3, 0xa6c0, 0x5bc0,
  0xa69c, 0x5b9d, 0xa678, 0x5b7a, 0xa654, 0x5b57, 0xa630, 0x5b34,
  0xa60c, 0x5b10, 0xa5e8, 0x5aed, 0xa5c5, 0x5ac9, 0xa5a1, 0x5aa6,
  0xa57e, 0x5a82, 0xa55a, 0x5a5f, 0xa537, 0x5a3b, 0xa513, 0x5a18,
  0xa4f0, 0x59f4, 0xa4cc, 0x59d0, 0xa4a9, 0x59ac, 0xa486, 0x5988,
  0xa463, 0x5964, 0xa440, 0x5940, 0xa41d, 0x591c, 0xa3fa, 0x58f8,
  0xa3d7, 0x58d4, 0xa3b4, 0x58b0, 0xa391, 0x588c, 0xa36f, 0x5867,
  0xa34c, 0x5843, 0xa329, 0x581e, 0xa307, 0x57fa, 0xa2e4, 0x57d5,
  0xa2c2, 0x57b1, 0xa29f, 0x578c, 0xa27d, 0x5767, 0xa25b, 0x5743,
  0xa238, 0x571e, 0xa216, 0x56f9, 0xa1f4, 0x56d4, 0xa1d2, 0x56af,
  0xa1b0, 0x568a, 0xa18e, 0x5665, 0xa16c, 0x5640, 0xa14a, 0x561b,
  0xa129, 0x55f6, 0xa107, 0x55d0, 0xa0e5, 0x55ab, 0xa0c4, 0x5586,
  0xa0a2, 0x5560, 0xa080, 0x553b, 0xa05f, 0x5515, 0xa03e, 0x54f0,
  0xa01c, 0x54ca, 0x9ffb, 0x54a4, 0x9fda, 0x547f, 0x9fb9, 0x5459,
  0x9f98, 0x5433, 0x9f77, 0x540d, 0x9f56, 0x53e7, 0x9f35, 0x53c1,
  0x9f14, 0x539b, 0x9ef3, 0x5375, 0x9ed2, 0x534f, 0x9eb2, 0x5329,
  0x9e91, 0x5303, 0x9e70, 0x52dc, 0x9e50, 0x52b6, 0x9e2f, 0x5290,
  0x9e0f, 0x5269, 0x9def, 0x5243, 0x9dce, 0x521c, 0x9dae, 0x51f5,
  0x9d8e, 0x51cf, 0x9d6e, 0x51a8, 0x9d4e, 0x5181, 0x9d2e, 0x515b,
  0x9d0e, 0x5134, 0x9cee, 0x510d, 0x9cce, 0x50e6, 0x9caf, 0x50bf,
  0x9c8f, 0x5098, 0x9c6f, 0x5071, 0x9c50, 0x504a, 0x9c30, 0x5023,
  0x9c11, 0x4ffb, 0x9bf1, 0x4fd4, 0x9bd2, 0x4fad, 0x9bb3, 0x4f85,
  0x9b94, 0x4f5e, 0x9b75, 0x4f37, 0x9b55, 0x4f0f, 0x9b36, 0x4ee8,
  0x9b17, 0x4ec0, 0x9af9, 0x4e98, 0x9ada, 0x4e71, 0x9abb, 0x4e49,
  0x9a9c, 0x4e21, 0x9a7e, 0x4df9, 0x9a5f, 0x4dd1, 0x9a40, 0x4da9,
  0x9a22, 0x4d81, 0x9a04, 0x4d59, 0x99e5, 0x4d31, 0x99c7, 0x4d09,
  0x99a9, 0x4ce1, 0x998b, 0x4cb9, 0x996d, 0x4c91, 0x994e, 0x4c68,
  0x9930, 0x4c40, 0x9913, 0x4c17, 0x98f5, 0x4bef, 0x98d7, 0x4bc7,
  0x98b9, 0x4b9e, 0x989c, 0x4b75, 0x987e, 0x4b4d, 0x9860, 0x4b24,
  0x9843, 0x4afb, 0x9826, 0x4ad3, 0x9808, 0x4aaa, 0x97eb, 0x4a81,
  0x97ce, 0x4a58, 0x97b0, 0x4a2f, 0x9793, 0x4a06, 0x9776, 0x49dd,
  0x9759, 0x49b4, 0x973c, 0x498b, 0x9720, 0x4962, 0x9703, 0x4939,
  0x96e6, 0x490f, 0x96c9, 0x48e6, 0x96ad, 0x48bd, 0x9690, 0x4893,
  0x9674, 0x486a, 0x9657, 0x4840, 0x963b, 0x4817, 0x961f, 0x47ed,
  0x9603, 0x47c4, 0x95e6, 0x479a, 0x95ca, 0x4770, 0x95ae, 0x4747,
  0x9592, 0x471d, 0x9577, 0x46f3, 0x955b, 0x46c9, 0x953f, 0x469f,
  0x9523, 0x4675, 0x9508, 0x464b, 0x94ec, 0x4621, 0x94d0, 0x45f7,
  0x94b5, 0x45cd, 0x949a, 0x45a3, 0x947e, 0x4579, 0x9463, 0x454f,
  0x9448, 0x4524, 0x942d, 0x44fa, 0x9412, 0x44d0, 0x93f7, 0x44a5,
  0x93dc, 0x447b, 0x93c1, 0x4450, 0x93a6, 0x4426, 0x938b, 0x43fb,
  0x9371, 0x43d1, 0x9356, 0x43a6, 0x933c, 0x437b, 0x9321, 0x4351,
  0x9307, 0x4326, 0x92ec, 0x42fb, 0x92d2, 0x42d0, 0x92b8, 0x42a5,
  0x929e, 0x427a, 0x9284, 0x424f, 0x926a, 0x4224, 0x9250, 0x41f9,
  0x9236, 0x41ce, 0x921c, 0x41a3, 0x9202, 0x4178, 0x91e9, 0x414d,
  0x91cf, 0x4121, 0x91b6, 0x40f6, 0x919c, 0x40cb, 0x9183, 0x409f,
  0x9169, 0x4074, 0x9150, 0x4048, 0x9137, 0x401d, 0x911e, 0x3ff1,
  0x9105, 0x3fc6, 0x90ec, 0x3f9a, 0x90d3, 0x3f6f, 0x90ba, 0x3f43,
  0x90a1, 0x3f17, 0x9088, 0x3eec, 0x9070, 0x3ec0, 0x9057, 0x3e94,
  0x903e, 0x3e68, 0x9026, 0x3e3c, 0x900e, 0x3e10, 0x8ff5, 0x3de4,
  0x8fdd, 0x3db8, 0x8fc5, 0x3d8c, 0x8fad, 0x3d60, 0x8f95, 0x3d34,
  0x8f7d, 0x3d08, 0x8f65, 0x3cdc, 0x8f4d, 0x3caf, 0x8f35, 0x3c83,
  0x8f1d, 0x3c57, 0x8f06, 0x3c2a, 0x8eee, 0x3bfe, 0x8ed6, 0x3bd2,
  0x8ebf, 0x3ba5, 0x8ea8, 0x3b79, 0x8e90, 0x3b4c, 0x8e79, 0x3b20,
  0x8e62, 0x3af3, 0x8e4b, 0x3ac6, 0x8e34, 0x3a9a, 0x8e1d, 0x3a6d,
  0x8e06, 0x3a40, 0x8def, 0x3a13, 0x8dd8, 0x39e7, 0x8dc1, 0x39ba,
  0x8dab, 0x398d, 0x8d94, 0x3960, 0x8d7e, 0x3933, 0x8d67, 0x3906,
  0x8d51, 0x38d9, 0x8d3b, 0x38ac, 0x8d24, 0x387f, 0x8d0e, 0x3852,
  0x8cf8, 0x3825, 0x8ce2, 0x37f7, 0x8ccc, 0x37ca, 0x8cb6, 0x379d,
  0x8ca1, 0x3770, 0x8c8b, 0x3742, 0x8c75, 0x3715, 0x8c60, 0x36e8,
  0x8c4a, 0x36ba, 0x8c35, 0x368d, 0x8c1f, 0x365f, 0x8c0a, 0x3632,
  0x8bf5, 0x3604, 0x8bdf, 0x35d7, 0x8bca, 0x35a9, 0x8bb5, 0x357b,
  0x8ba0, 0x354e, 0x8b8b, 0x3520, 0x8b77, 0x34f2, 0x8b62, 0x34c4,
  0x8b4d, 0x3497, 0x8b39, 0x3469, 0x8b24, 0x343b, 0x8b10, 0x340d,
  0x8afb, 0x33df, 0x8ae7, 0x33b1, 0x8ad3, 0x3383, 0x8abe, 0x3355,
  0x8aaa, 0x3327, 0x8a96, 0x32f9, 0x8a82, 0x32cb, 0x8a6e, 0x329d,
  0x8a5a, 0x326e, 0x8a47, 0x3240, 0x8a33, 0x3212, 0x8a1f, 0x31e4,
  0x8a0c, 0x31b5, 0x89f8, 0x3187, 0x89e5, 0x3159, 0x89d2, 0x312a,
  0x89be, 0x30fc, 0x89ab, 0x30cd, 0x8998, 0x309f, 0x8985, 0x3070,
  0x8972, 0x3042, 0x895f, 0x3013, 0x894c, 0x2fe5, 0x8939, 0x2fb6,
  0x8927, 0x2f87, 0x8914, 0x2f59, 0x8902, 0x2f2a, 0x88ef, 0x2efb,
  0x88dd, 0x2ecc, 0x88ca, 0x2e9e, 0x88b8, 0x2e6f, 0x88a6, 0x2e40,
  0x8894, 0x2e11, 0x8882, 0x2de2, 0x8870, 0x2db3, 0x885e, 0x2d84,
  0x884c, 0x2d55, 0x883a, 0x2d26, 0x8828, 0x2cf7, 0x8817, 0x2cc8,
  0x8805, 0x2c99, 0x87f4, 0x2c6a, 0x87e2, 0x2c3b, 0x87d1, 0x2c0c,
  0x87c0, 0x2bdc, 0x87af, 0x2bad, 0x879d, 0x2b7e, 0x878c, 0x2b4f,
  0x877b, 0x2b1f, 0x876b, 0x2af0, 0x875a, 0x2ac1, 0x8749, 0x2a91,
  0x8738, 0x2a62, 0x8728, 0x2a32, 0x8717, 0x2a03, 0x8707, 0x29d3,
  0x86f6, 0x29a4, 0x86e6, 0x2974, 0x86d6, 0x2945, 0x86c6, 0x2915,
  0x86b6, 0x28e5, 0x86a5, 0x28b6, 0x8696, 0x2886, 0x8686, 0x2856,
  0x8676, 0x2827, 0x8666, 0x27f7, 0x8656, 0x27c7, 0x8647, 0x2797,
  0x8637, 0x2768, 0x8628, 0x2738, 0x8619, 0x2708, 0x8609, 0x26d8,
  0x85fa, 0x26a8, 0x85eb, 0x2678, 0x85dc, 0x2648, 0x85cd, 0x2618,
  0x85be, 0x25e8, 0x85af, 0x25b8, 0x85a0, 0x2588, 0x8592, 0x2558,
  0x8583, 0x2528, 0x8574, 0x24f8, 0x8566, 0x24c8, 0x8558, 0x2498,
  0x8549, 0x2467, 0x853b, 0x2437, 0x852d, 0x2407, 0x851f, 0x23d7,
  0x8511, 0x23a7, 0x8503, 0x2376, 0x84f5, 0x2346, 0x84e7, 0x2316,
  0x84d9, 0x22e5, 0x84cc, 0x22b5, 0x84be, 0x2284, 0x84b0, 0x2254,
  0x84a3, 0x2224, 0x8496, 0x21f3, 0x8488, 0x21c3, 0x847b, 0x2192,
  0x846e, 0x2162, 0x8461, 0x2131, 0x8454, 0x2101, 0x8447, 0x20d0,
  0x843a, 0x209f, 0x842d, 0x206f, 0x8421, 0x203e, 0x8414, 0x200e,
  0x8407, 0x1fdd, 0x83fb, 0x1fac, 0x83ef, 0x1f7b, 0x83e2, 0x1f4b,
  0x83d6, 0x1f1a, 0x83ca, 0x1ee9, 0x83be, 0x1eb8, 0x83b2, 0x1e88,
  0x83a6, 0x1e57, 0x839a, 0x1e26, 0x838e, 0x1df5, 0x8382, 0x1dc4,
  0x8377, 0x1d93, 0x836b, 0x1d62, 0x8360, 0x1d31, 0x8354, 0x1d01,
  0x8349, 0x1cd0, 0x833e, 0x1c9f, 0x8332, 0x1c6e, 0x8327, 0x1c3d,
  0x831c, 0x1c0c, 0x8311, 0x1bda, 0x8306, 0x1ba9, 0x82fb, 0x1b78,
  0x82f1, 0x1b47, 0x82e6, 0x1b16, 0x82db, 0x1ae5, 0x82d1, 0x1ab4,
  0x82c6, 0x1a83, 0x82bc, 0x1a51, 0x82b2, 0x1a20, 0x82a8, 0x19ef,
  0x829d, 0x19be, 0x8293, 0x198d, 0x8289, 0x195b, 0x827f, 0x192a,
  0x8276, 0x18f9, 0x826c, 0x18c7, 0x8262, 0x1896, 0x8259, 0x1865,
  0x824f, 0x1833, 0x8246, 0x1802, 0x823c, 0x17d1, 0x8233, 0x179f,
  0x822a, 0x176e, 0x8220, 0x173c, 0x8217, 0x170b, 0x820e, 0x16da,
  0x8205, 0x16a8, 0x81fd, 0x1677, 0x81f4, 0x1645, 0x81eb, 0x1614,
  0x81e2, 0x15e2, 0x81da, 0x15b1, 0x81d1, 0x157f, 0x81c9, 0x154d,
  0x81c1, 0x151c, 0x81b8, 0x14ea, 0x81b0, 0x14b9, 0x81a8, 0x1487,
  0x81a0, 0x1455, 0x8198, 0x1424, 0x8190, 0x13f2, 0x8188, 0x13c1,
  0x8181, 0x138f, 0x8179, 0x135d, 0x8172, 0x132b, 0x816a, 0x12fa,
  0x8163, 0x12c8, 0x815b, 0x1296, 0x8154, 0x1265, 0x814d, 0x1233,
  0x8146, 0x1201, 0x813f, 0x11cf, 0x8138, 0x119e, 0x8131, 0x116c,
  0x812a, 0x113a, 0x8123, 0x1108, 0x811d, 0x10d6, 0x8116, 0x10a4,
  0x8110, 0x1073, 0x8109, 0x1041, 0x8103, 0x100f, 0x80fd, 0xfdd,
  0x80f6, 0xfab, 0x80f0, 0xf79, 0x80ea, 0xf47, 0x80e4, 0xf15,
  0x80de, 0xee4, 0x80d9, 0xeb2, 0x80d3, 0xe80, 0x80cd, 0xe4e,
  0x80c8, 0xe1c, 0x80c2, 0xdea, 0x80bd, 0xdb8, 0x80b7, 0xd86,
  0x80b2, 0xd54, 0x80ad, 0xd22, 0x80a8, 0xcf0, 0x80a3, 0xcbe,
  0x809e, 0xc8c, 0x8099, 0xc5a, 0x8094, 0xc28, 0x808f, 0xbf6,
  0x808b, 0xbc4, 0x8086, 0xb92, 0x8082, 0xb60, 0x807d, 0xb2d,
  0x8079, 0xafb, 0x8075, 0xac9, 0x8070, 0xa97, 0x806c, 0xa65,
  0x8068, 0xa33, 0x8064, 0xa01, 0x8060, 0x9cf, 0x805d, 0x99d,
  0x8059, 0x96b, 0x8055, 0x938, 0x8052, 0x906, 0x804e, 0x8d4,
  0x804b, 0x8a2, 0x8047, 0x870, 0x8044, 0x83e, 0x8041, 0x80c,
  0x803e, 0x7d9, 0x803b, 0x7a7, 0x8038, 0x775, 0x8035, 0x743,
  0x8032, 0x711, 0x802f, 0x6de, 0x802d, 0x6ac, 0x802a, 0x67a,
  0x8027, 0x648, 0x8025, 0x616, 0x8023, 0x5e3, 0x8020, 0x5b1,
  0x801e, 0x57f, 0x801c, 0x54d, 0x801a, 0x51b, 0x8018, 0x4e8,
  0x8016, 0x4b6, 0x8014, 0x484, 0x8013, 0x452, 0x8011, 0x41f,
  0x800f, 0x3ed, 0x800e, 0x3bb, 0x800c, 0x389, 0x800b, 0x356,
  0x800a, 0x324, 0x8009, 0x2f2, 0x8008, 0x2c0, 0x8007, 0x28d,
  0x8006, 0x25b, 0x8005, 0x229, 0x8004, 0x1f7, 0x8003, 0x1c4,
  0x8002, 0x192, 0x8002, 0x160, 0x8001, 0x12e, 0x8001, 0xfb,
  0x8001, 0xc9, 0x8000, 0x97, 0x8000, 0x65, 0x8000, 0x32,
  0x8000, 0x0, 0x8000, 0xffce, 0x8000, 0xff9b, 0x8000, 0xff69,
  0x8001, 0xff37, 0x8001, 0xff05, 0x8001, 0xfed2, 0x8002, 0xfea0,
  0x8002, 0xfe6e, 0x8003, 0xfe3c, 0x8004, 0xfe09, 0x8005, 0xfdd7,
  0x8006, 0xfda5, 0x8007, 0xfd73, 0x8008, 0xfd40, 0x8009, 0xfd0e,
  0x800a, 0xfcdc, 0x800b, 0xfcaa, 0x800c, 0xfc77, 0x800e, 0xfc45,
  0x800f, 0xfc13, 0x8011, 0xfbe1, 0x8013, 0xfbae, 0x8014, 0xfb7c,
  0x8016, 0xfb4a, 0x8018, 0xfb18, 0x801a, 0xfae5, 0x801c, 0xfab3,
  0x801e, 0xfa81, 0x8020, 0xfa4f, 0x8023, 0xfa1d, 0x8025, 0xf9ea,
  0x8027, 0xf9b8, 0x802a, 0xf986, 0x802d, 0xf954, 0x802f, 0xf922,
  0x8032, 0xf8ef, 0x8035, 0xf8bd, 0x8038, 0xf88b, 0x803b, 0xf859,
  0x803e, 0xf827, 0x8041, 0xf7f4, 0x8044, 0xf7c2, 0x8047, 0xf790,
  0x804b, 0xf75e, 0x804e, 0xf72c, 0x8052, 0xf6fa, 0x8055, 0xf6c8,
  0x8059, 0xf695, 0x805d, 0xf663, 0x8060, 0xf631, 0x8064, 0xf5ff,
  0x8068, 0xf5cd, 0x806c, 0xf59b, 0x8070, 0xf569, 0x8075, 0xf537,
  0x8079, 0xf505, 0x807d, 0xf4d3, 0x8082, 0xf4a0, 0x8086, 0xf46e,
  0x808b, 0xf43c, 0x808f, 0xf40a, 0x8094, 0xf3d8, 0x8099, 0xf3a6,
  0x809e, 0xf374, 0x80a3, 0xf342, 0x80a8, 0xf310, 0x80ad, 0xf2de,
  0x80b2, 0xf2ac, 0x80b7, 0xf27a, 0x80bd, 0xf248, 0x80c2, 0xf216,
  0x80c8, 0xf1e4, 0x80cd, 0xf1b2, 0x80d3, 0xf180, 0x80d9, 0xf14e,
  0x80de, 0xf11c, 0x80e4, 0xf0eb, 0x80ea, 0xf0b9, 0x80f0, 0xf087,
  0x80f6, 0xf055, 0x80fd, 0xf023, 0x8103, 0xeff1, 0x8109, 0xefbf,
  0x8110, 0xef8d, 0x8116, 0xef5c, 0x811d, 0xef2a, 0x8123, 0xeef8,
  0x812a, 0xeec6, 0x8131, 0xee94, 0x8138, 0xee62, 0x813f, 0xee31,
  0x8146, 0xedff, 0x814d, 0xedcd, 0x8154, 0xed9b, 0x815b, 0xed6a,
  0x8163, 0xed38, 0x816a, 0xed06, 0x8172, 0xecd5, 0x8179, 0xeca3,
  0x8181, 0xec71, 0x8188, 0xec3f, 0x8190, 0xec0e, 0x8198, 0xebdc,
  0x81a0, 0xebab, 0x81a8, 0xeb79, 0x81b0, 0xeb47, 0x81b8, 0xeb16,
  0x81c1, 0xeae4, 0x81c9, 0xeab3, 0x81d1, 0xea81, 0x81da, 0xea4f,
  0x81e2, 0xea1e, 0x81eb, 0xe9ec, 0x81f4, 0xe9bb, 0x81fd, 0xe989,
  0x8205, 0xe958, 0x820e, 0xe926, 0x8217, 0xe8f5, 0x8220, 0xe8c4,
  0x822a, 0xe892, 0x8233, 0xe861, 0x823c, 0xe82f, 0x8246, 0xe7fe,
  0x824f, 0xe7cd, 0x8259, 0xe79b, 0x8262, 0xe76a, 0x826c, 0xe739,
  0x8276, 0xe707, 0x827f, 0xe6d6, 0x8289, 0xe6a5, 0x8293, 0xe673,
  0x829d, 0xe642, 0x82a8, 0xe611, 0x82b2, 0xe5e0, 0x82bc, 0xe5af,
  0x82c6, 0xe57d, 0x82d1, 0xe54c, 0x82db, 0xe51b, 0x82e6, 0xe4ea,
  0x82f1, 0xe4b9, 0x82fb, 0xe488, 0x8306, 0xe457, 0x8311, 0xe426,
  0x831c, 0xe3f4, 0x8327, 0xe3c3, 0x8332, 0xe392, 0x833e, 0xe361,
  0x8349, 0xe330, 0x8354, 0xe2ff, 0x8360, 0xe2cf, 0x836b, 0xe29e,
  0x8377, 0xe26d, 0x8382, 0xe23c, 0x838e, 0xe20b, 0x839a, 0xe1da,
  0x83a6, 0xe1a9, 0x83b2, 0xe178, 0x83be, 0xe148, 0x83ca, 0xe117,
  0x83d6, 0xe0e6, 0x83e2, 0xe0b5, 0x83ef, 0xe085, 0x83fb, 0xe054,
  0x8407, 0xe023, 0x8414, 0xdff2, 0x8421, 0xdfc2, 0x842d, 0xdf91,
  0x843a, 0xdf61, 0x8447, 0xdf30, 0x8454, 0xdeff, 0x8461, 0xdecf,
  0x846e, 0xde9e, 0x847b, 0xde6e, 0x8488, 0xde3d, 0x8496, 0xde0d,
  0x84a3, 0xdddc, 0x84b0, 0xddac, 0x84be, 0xdd7c, 0x84cc, 0xdd4b,
  0x84d9, 0xdd1b, 0x84e7, 0xdcea, 0x84f5, 0xdcba, 0x8503, 0xdc8a,
  0x8511, 0xdc59, 0x851f, 0xdc29, 0x852d, 0xdbf9, 0x853b, 0xdbc9,
  0x8549, 0xdb99, 0x8558, 0xdb68, 0x8566, 0xdb38, 0x8574, 0xdb08,
  0x8583, 0xdad8, 0x8592, 0xdaa8, 0x85a0, 0xda78, 0x85af, 0xda48,
  0x85be, 0xda18, 0x85cd, 0xd9e8, 0x85dc, 0xd9b8, 0x85eb, 0xd988,
  0x85fa, 0xd958, 0x8609, 0xd928, 0x8619, 0xd8f8, 0x8628, 0xd8c8,
  0x8637, 0xd898, 0x8647, 0xd869, 0x8656, 0xd839, 0x8666, 0xd809,
  0x8676, 0xd7d9, 0x8686, 0xd7aa, 0x8696, 0xd77a, 0x86a5, 0xd74a,
  0x86b6, 0xd71b, 0x86c6, 0xd6eb, 0x86d6, 0xd6bb, 0x86e6, 0xd68c,
  0x86f6, 0xd65c, 0x8707, 0xd62d, 0x8717, 0xd5fd, 0x8728, 0xd5ce,
  0x8738, 0xd59e, 0x8749, 0xd56f, 0x875a, 0xd53f, 0x876b, 0xd510,
  0x877b, 0xd4e1, 0x878c, 0xd4b1, 0x879d, 0xd482, 0x87af, 0xd453,
  0x87c0, 0xd424, 0x87d1, 0xd3f4, 0x87e2, 0xd3c5, 0x87f4, 0xd396,
  0x8805, 0xd367, 0x8817, 0xd338, 0x8828, 0xd309, 0x883a, 0xd2da,
  0x884c, 0xd2ab, 0x885e, 0xd27c, 0x8870, 0xd24d, 0x8882, 0xd21e,
  0x8894, 0xd1ef, 0x88a6, 0xd1c0, 0x88b8, 0xd191, 0x88ca, 0xd162,
  0x88dd, 0xd134, 0x88ef, 0xd105, 0x8902, 0xd0d6, 0x8914, 0xd0a7,
  0x8927, 0xd079, 0x8939, 0xd04a, 0x894c, 0xd01b, 0x895f, 0xcfed,
  0x8972, 0xcfbe, 0x8985, 0xcf90, 0x8998, 0xcf61, 0x89ab, 0xcf33,
  0x89be, 0xcf04, 0x89d2, 0xced6, 0x89e5, 0xcea7, 0x89f8, 0xce79,
  0x8a0c, 0xce4b, 0x8a1f, 0xce1c, 0x8a33, 0xcdee, 0x8a47, 0xcdc0,
  0x8a5a, 0xcd92, 0x8a6e, 0xcd63, 0x8a82, 0xcd35, 0x8a96, 0xcd07,
  0x8aaa, 0xccd9, 0x8abe, 0xccab, 0x8ad3, 0xcc7d, 0x8ae7, 0xcc4f,
  0x8afb, 0xcc21, 0x8b10, 0xcbf3, 0x8b24, 0xcbc5, 0x8b39, 0xcb97,
  0x8b4d, 0xcb69, 0x8b62, 0xcb3c, 0x8b77, 0xcb0e, 0x8b8b, 0xcae0,
  0x8ba0, 0xcab2, 0x8bb5, 0xca85, 0x8bca, 0xca57, 0x8bdf, 0xca29,
  0x8bf5, 0xc9fc, 0x8c0a, 0xc9ce, 0x8c1f, 0xc9a1, 0x8c35, 0xc973,
  0x8c4a, 0xc946, 0x8c60, 0xc918, 0x8c75, 0xc8eb, 0x8c8b, 0xc8be,
  0x8ca1, 0xc890, 0x8cb6, 0xc863, 0x8ccc, 0xc836, 0x8ce2, 0xc809,
  0x8cf8, 0xc7db, 0x8d0e, 0xc7ae, 0x8d24, 0xc781, 0x8d3b, 0xc754,
  0x8d51, 0xc727, 0x8d67, 0xc6fa, 0x8d7e, 0xc6cd, 0x8d94, 0xc6a0,
  0x8dab, 0xc673, 0x8dc1, 0xc646, 0x8dd8, 0xc619, 0x8def, 0xc5ed,
  0x8e06, 0xc5c0, 0x8e1d, 0xc593, 0x8e34, 0xc566, 0x8e4b, 0xc53a,
  0x8e62, 0xc50d, 0x8e79, 0xc4e0, 0x8e90, 0xc4b4, 0x8ea8, 0xc487,
  0x8ebf, 0xc45b, 0x8ed6, 0xc42e, 0x8eee, 0xc402, 0x8f06, 0xc3d6,
  0x8f1d, 0xc3a9, 0x8f35, 0xc37d, 0x8f4d, 0xc351, 0x8f65, 0xc324,
  0x8f7d, 0xc2f8, 0x8f95, 0xc2cc, 0x8fad, 0xc2a0, 0x8fc5, 0xc274,
  0x8fdd, 0xc248, 0x8ff5, 0xc21c, 0x900e, 0xc1f0, 0x9026, 0xc1c4,
  0x903e, 0xc198, 0x9057, 0xc16c, 0x9070, 0xc140, 0x9088, 0xc114,
  0x90a1, 0xc0e9, 0x90ba, 0xc0bd, 0x90d3, 0xc091, 0x90ec, 0xc066,
  0x9105, 0xc03a, 0x911e, 0xc00f, 0x9137, 0xbfe3, 0x9150, 0xbfb8,
  0x9169, 0xbf8c, 0x9183, 0xbf61, 0x919c, 0xbf35, 0x91b6, 0xbf0a,
  0x91cf, 0xbedf, 0x91e9, 0xbeb3, 0x9202, 0xbe88, 0x921c, 0xbe5d,
  0x9236, 0xbe32, 0x9250, 0xbe07, 0x926a, 0xbddc, 0x9284, 0xbdb1,
  0x929e, 0xbd86, 0x92b8, 0xbd5b, 0x92d2, 0xbd30, 0x92ec, 0xbd05,
  0x9307, 0xbcda, 0x9321, 0xbcaf, 0x933c, 0xbc85, 0x9356, 0xbc5a,
  0x9371, 0xbc2f, 0x938b, 0xbc05, 0x93a6, 0xbbda, 0x93c1, 0xbbb0,
  0x93dc, 0xbb85, 0x93f7, 0xbb5b, 0x9412, 0xbb30, 0x942d, 0xbb06,
  0x9448, 0xbadc, 0x9463, 0xbab1, 0x947e, 0xba87, 0x949a, 0xba5d,
  0x94b5, 0xba33, 0x94d0, 0xba09, 0x94ec, 0xb9df, 0x9508, 0xb9b5,
  0x9523, 0xb98b, 0x953f, 0xb961, 0x955b, 0xb937, 0x9577, 0xb90d,
  0x9592, 0xb8e3, 0x95ae, 0xb8b9, 0x95ca, 0xb890, 0x95e6, 0xb866,
  0x9603, 0xb83c, 0x961f, 0xb813, 0x963b, 0xb7e9, 0x9657, 0xb7c0,
  0x9674, 0xb796, 0x9690, 0xb76d, 0x96ad, 0xb743, 0x96c9, 0xb71a,
  0x96e6, 0xb6f1, 0x9703, 0xb6c7, 0x9720, 0xb69e, 0x973c, 0xb675,
  0x9759, 0xb64c, 0x9776, 0xb623, 0x9793, 0xb5fa, 0x97b0, 0xb5d1,
  0x97ce, 0xb5a8, 0x97eb, 0xb57f, 0x9808, 0xb556, 0x9826, 0xb52d,
  0x9843, 0xb505, 0x9860, 0xb4dc, 0x987e, 0xb4b3, 0x989c, 0xb48b,
  0x98b9, 0xb462, 0x98d7, 0xb439, 0x98f5, 0xb411, 0x9913, 0xb3e9,
  0x9930, 0xb3c0, 0x994e, 0xb398, 0x996d, 0xb36f, 0x998b, 0xb347,
  0x99a9, 0xb31f, 0x99c7, 0xb2f7, 0x99e5, 0xb2cf, 0x9a04, 0xb2a7,
  0x9a22, 0xb27f, 0x9a40, 0xb257, 0x9a5f, 0xb22f, 0x9a7e, 0xb207,
  0x9a9c, 0xb1df, 0x9abb, 0xb1b7, 0x9ada, 0xb18f, 0x9af9, 0xb168,
  0x9b17, 0xb140, 0x9b36, 0xb118, 0x9b55, 0xb0f1, 0x9b75, 0xb0c9,
  0x9b94, 0xb0a2, 0x9bb3, 0xb07b, 0x9bd2, 0xb053, 0x9bf1, 0xb02c,
  0x9c11, 0xb005, 0x9c30, 0xafdd, 0x9c50, 0xafb6, 0x9c6f, 0xaf8f,
  0x9c8f, 0xaf68, 0x9caf, 0xaf41, 0x9cce, 0xaf1a, 0x9cee, 0xaef3,
  0x9d0e, 0xaecc, 0x9d2e, 0xaea5, 0x9d4e, 0xae7f, 0x9d6e, 0xae58,
  0x9d8e, 0xae31, 0x9dae, 0xae0b, 0x9dce, 0xade4, 0x9def, 0xadbd,
  0x9e0f, 0xad97, 0x9e2f, 0xad70, 0x9e50, 0xad4a, 0x9e70, 0xad24,
  0x9e91, 0xacfd, 0x9eb2, 0xacd7, 0x9ed2, 0xacb1, 0x9ef3, 0xac8b,
  0x9f14, 0xac65, 0x9f35, 0xac3f, 0x9f56, 0xac19, 0x9f77, 0xabf3,
  0x9f98, 0xabcd, 0x9fb9, 0xaba7, 0x9fda, 0xab81, 0x9ffb, 0xab5c,
  0xa01c, 0xab36, 0xa03e, 0xab10, 0xa05f, 0xaaeb, 0xa080, 0xaac5,
  0xa0a2, 0xaaa0, 0xa0c4, 0xaa7a, 0xa0e5, 0xaa55, 0xa107, 0xaa30,
  0xa129, 0xaa0a, 0xa14a, 0xa9e5, 0xa16c, 0xa9c0, 0xa18e, 0xa99b,
  0xa1b0, 0xa976, 0xa1d2, 0xa951, 0xa1f4, 0xa92c, 0xa216, 0xa907,
  0xa238, 0xa8e2, 0xa25b, 0xa8bd, 0xa27d, 0xa899, 0xa29f, 0xa874,
  0xa2c2, 0xa84f, 0xa2e4, 0xa82b, 0xa307, 0xa806, 0xa329, 0xa7e2,
  0xa34c, 0xa7bd, 0xa36f, 0xa799, 0xa391, 0xa774, 0xa3b4, 0xa750,
  0xa3d7, 0xa72c, 0xa3fa, 0xa708, 0xa41d, 0xa6e4, 0xa440, 0xa6c0,
  0xa463, 0xa69c, 0xa486, 0xa678, 0xa4a9, 0xa654, 0xa4cc, 0xa630,
  0xa4f0, 0xa60c, 0xa513, 0xa5e8, 0xa537, 0xa5c5, 0xa55a, 0xa5a1,
  0xa57e, 0xa57e, 0xa5a1, 0xa55a, 0xa5c5, 0xa537, 0xa5e8, 0xa513,
  0xa60c, 0xa4f0, 0xa630, 0xa4cc, 0xa654, 0xa4a9, 0xa678, 0xa486,
  0xa69c, 0xa463, 0xa6c0, 0xa440, 0xa6e4, 0xa41d, 0xa708, 0xa3fa,
  0xa72c, 0xa3d7, 0xa750, 0xa3b4, 0xa774, 0xa391, 0xa799, 0xa36f,
  0xa7bd, 0xa34c, 0xa7e2, 0xa329, 0xa806, 0xa307, 0xa82b, 0xa2e4,
  0xa84f, 0xa2c2, 0xa874, 0xa29f, 0xa899, 0xa27d, 0xa8bd, 0xa25b,
  0xa8e2, 0xa238, 0xa907, 0xa216, 0xa92c, 0xa1f4, 0xa951, 0xa1d2,
  0xa976, 0xa1b0, 0xa99b, 0xa18e, 0xa9c0, 0xa16c, 0xa9e5, 0xa14a,
  0xaa0a, 0xa129, 0xaa30, 0xa107, 0xaa55, 0xa0e5, 0xaa7a, 0xa0c4,
  0xaaa0, 0xa0a2, 0xaac5, 0xa080, 0xaaeb, 0xa05f, 0xab10, 0xa03e,
  0xab36, 0xa01c, 0xab5c, 0x9ffb, 0xab81, 0x9fda, 0xaba7, 0x9fb9,
  0xabcd, 0x9f98, 0xabf3, 0x9f77, 0xac19, 0x9f56, 0xac3f, 0x9f35,
  0xac65, 0x9f14, 0xac8b, 0x9ef3, 0xacb1, 0x9ed2, 0xacd7, 0x9eb2,
  0xacfd, 0x9e91, 0xad24, 0x9e70, 0xad4a, 0x9e50, 0xad70, 0x9e2f,
  0xad97, 0x9e0f, 0xadbd, 0x9def, 0xade4, 0x9dce, 0xae0b, 0x9dae,
  0xae31, 0x9d8e, 0xae58, 0x9d6e, 0xae7f, 0x9d4e, 0xaea5, 0x9d2e,
  0xaecc, 0x9d0e, 0xaef3, 0x9cee, 0xaf1a, 0x9cce, 0xaf41, 0x9caf,
  0xaf68, 0x9c8f, 0xaf8f, 0x9c6f, 0xafb6, 0x9c50, 0xafdd, 0x9c30,
  0xb005, 0x9c11, 0xb02c, 0x9bf1, 0xb053, 0x9bd2, 0xb07b, 0x9bb3,
  0xb0a2, 0x9b94, 0xb0c9, 0x9b75, 0xb0f1, 0x9b55, 0xb118, 0x9b36,
  0xb140, 0x9b17, 0xb168, 0x9af9, 0xb18f, 0x9ada, 0xb1b7, 0x9abb,
  0xb1df, 0x9a9c, 0xb207, 0x9a7e, 0xb22f, 0x9a5f, 0xb257, 0x9a40,
  0xb27f, 0x9a22, 0xb2a7, 0x9a04, 0xb2cf, 0x99e5, 0xb2f7, 0x99c7,
  0xb31f, 0x99a9, 0xb347, 0x998b, 0xb36f, 0x996d, 0xb398, 0x994e,
  0xb3c0, 0x9930, 0xb3e9, 0x9913, 0xb411, 0x98f5, 0xb439, 0x98d7,
  0xb462, 0x98b9, 0xb48b, 0x989c, 0xb4b3, 0x987e, 0xb4dc, 0x9860,
  0xb505, 0x9843, 0xb52d, 0x9826, 0xb556, 0x9808, 0xb57f, 0x97eb,
  0xb5a8, 0x97ce, 0xb5d1, 0x97b0, 0xb5fa, 0x9793, 0xb623, 0x9776,
  0xb64c, 0x9759, 0xb675, 0x973c, 0xb69e, 0x9720, 0xb6c7, 0x9703,
  0xb6f1, 0x96e6, 0xb71a, 0x96c9, 0xb743, 0x96ad, 0xb76d, 0x9690,
  0xb796, 0x9674, 0xb7c0, 0x9657, 0xb7e9, 0x963b, 0xb813, 0x961f,
  0xb83c, 0x9603, 0xb866, 0x95e6, 0xb890, 0x95ca, 0xb8b9, 0x95ae,
  0xb8e3, 0x9592, 0xb90d, 0x9577, 0xb937, 0x955b, 0xb961, 0x953f,
  0xb98b, 0x9523, 0xb9b5, 0x9508, 0xb9df, 0x94ec, 0xba09, 0x94d0,
  0xba33, 0x94b5, 0xba5d, 0x949a, 0xba87, 0x947e, 0xbab1, 0x9463,
  0xbadc, 0x9448, 0xbb06, 0x942d, 0xbb30, 0x9412, 0xbb5b, 0x93f7,
  0xbb85, 0x93dc, 0xbbb0, 0x93c1, 0xbbda, 0x93a6, 0xbc05, 0x938b,
  0xbc2f, 0x9371, 0xbc5a, 0x9356, 0xbc85, 0x933c, 0xbcaf, 0x9321,
  0xbcda, 0x9307, 0xbd05, 0x92ec, 0xbd30, 0x92d2, 0xbd5b, 0x92b8,
  0xbd86, 0x929e, 0xbdb1, 0x9284, 0xbddc, 0x926a, 0xbe07, 0x9250,
  0xbe32, 0x9236, 0xbe5d, 0x921c, 0xbe88, 0x9202, 0xbeb3, 0x91e9,
  0xbedf, 0x91cf, 0xbf0a, 0x91b6, 0xbf35, 0x919c, 0xbf61, 0x9183,
  0xbf8c, 0x9169, 0xbfb8, 0x9150, 0xbfe3, 0x9137, 0xc00f, 0x911e,
  0xc03a, 0x9105, 0xc066, 0x90ec, 0xc091, 0x90d3, 0xc0bd, 0x90ba,
  0xc0e9, 0x90a1, 0xc114, 0x9088, 0xc140, 0x9070, 0xc16c, 0x9057,
  0xc198, 0x903e, 0xc1c4, 0x9026, 0xc1f0, 0x900e, 0xc21c, 0x8ff5,
  0xc248, 0x8fdd, 0xc274, 0x8fc5, 0xc2a0, 0x8fad, 0xc2cc, 0x8f95,
  0xc2f8, 0x8f7d, 0xc324, 0x8f65, 0xc351, 0x8f4d, 0xc37d, 0x8f35,
  0xc3a9, 0x8f1d, 0xc3d6, 0x8f06, 0xc402, 0x8eee, 0xc42e, 0x8ed6,
  0xc45b, 0x8ebf, 0xc487, 0x8ea8, 0xc4b4, 0x8e90, 0xc4e0, 0x8e79,
  0xc50d, 0x8e62, 0xc53a, 0x8e4b, 0xc566, 0x8e34, 0xc593, 0x8e1d,
  0xc5c0, 0x8e06, 0xc5ed, 0x8def, 0xc619, 0x8dd8, 0xc646, 0x8dc1,
  0xc673, 0x8dab, 0xc6a0, 0x8d94, 0xc6cd, 0x8d7e, 0xc6fa, 0x8d67,
  0xc727, 0x8d51, 0xc754, 0x8d3b, 0xc781, 0x8d24, 0xc7ae, 0x8d0e,
  0xc7db, 0x8cf8, 0xc809, 0x8ce2, 0xc836, 0x8ccc, 0xc863, 0x8cb6,
  0xc890, 0x8ca1, 0xc8be, 0x8c8b, 0xc8eb, 0x8c75, 0xc918, 0x8c60,
  0xc946, 0x8c4a, 0xc973, 0x8c35, 0xc9a1, 0x8c1f, 0xc9ce, 0x8c0a,
  0xc9fc, 0x8bf5, 0xca29, 0x8bdf, 0xca57, 0x8bca, 0xca85, 0x8bb5,
  0xcab2, 0x8ba0, 0xcae0, 0x8b8b, 0xcb0e, 0x8b77, 0xcb3c, 0x8b62,
  0xcb69, 0x8b4d, 0xcb97, 0x8b39, 0xcbc5, 0x8b24, 0xcbf3, 0x8b10,
  0xcc21, 0x8afb, 0xcc4f, 0x8ae7, 0xcc7d, 0x8ad3, 0xccab, 0x8abe,
  0xccd9, 0x8aaa, 0xcd07, 0x8a96, 0xcd35, 0x8a82, 0xcd63, 0x8a6e,
  0xcd92, 0x8a5a, 0xcdc0, 0x8a47, 0xcdee, 0x8a33, 0xce1c, 0x8a1f,
  0xce4b, 0x8a0c, 0xce79, 0x89f8, 0xcea7, 0x89e5, 0xced6, 0x89d2,
  0xcf04, 0x89be, 0xcf33, 0x89ab, 0xcf61, 0x8998, 0xcf90, 0x8985,
  0xcfbe, 0x8972, 0xcfed, 0x895f, 0xd01b, 0x894c, 0xd04a, 0x8939,
  0xd079, 0x8927, 0xd0a7, 0x8914, 0xd0d6, 0x8902, 0xd105, 0x88ef,
  0xd134, 0x88dd, 0xd162, 0x88ca, 0xd191, 0x88b8, 0xd1c0, 0x88a6,
  0xd1ef, 0x8894, 0xd21e, 0x8882, 0xd24d, 0x8870, 0xd27c, 0x885e,
  0xd2ab, 0x884c, 0xd2da, 0x883a, 0xd309, 0x8828, 0xd338, 0x8817,
  0xd367, 0x8805, 0xd396, 0x87f4, 0xd3c5, 0x87e2, 0xd3f4, 0x87d1,
  0xd424, 0x87c0, 0xd453, 0x87af, 0xd482, 0x879d, 0xd4b1, 0x878c,
  0xd4e1, 0x877b, 0xd510, 0x876b, 0xd53f, 0x875a, 0xd56f, 0x8749,
  0xd59e, 0x8738, 0xd5ce, 0x8728, 0xd5fd, 0x8717, 0xd62d, 0x8707,
  0xd65c, 0x86f6, 0xd68c, 0x86e6, 0xd6bb, 0x86d6, 0xd6eb, 0x86c6,
  0xd71b, 0x86b6, 0xd74a, 0x86a5, 0xd77a, 0x8696, 0xd7aa, 0x8686,
  0xd7d9, 0x8676, 0xd809, 0x8666, 0xd839, 0x8656, 0xd869, 0x8647,
  0xd898, 0x8637, 0xd8c8, 0x8628, 0xd8f8, 0x8619, 0xd928, 0x8609,
  0xd958, 0x85fa, 0xd988, 0x85eb, 0xd9b8, 0x85dc, 0xd9e8, 0x85cd,
  0xda18, 0x85be, 0xda48, 0x85af, 0xda78, 0x85a0, 0xdaa8, 0x8592,
  0xdad8, 0x8583, 0xdb08, 0x8574, 0xdb38, 0x8566, 0xdb68, 0x8558,
  0xdb99, 0x8549, 0xdbc9, 0x853b, 0xdbf9, 0x852d, 0xdc29, 0x851f,
  0xdc59, 0x8511, 0xdc8a, 0x8503, 0xdcba, 0x84f5, 0xdcea, 0x84e7,
  0xdd1b, 0x84d9, 0xdd4b, 0x84cc, 0xdd7c, 0x84be, 0xddac, 0x84b0,
  0xdddc, 0x84a3, 0xde0d, 0x8496, 0xde3d, 0x8488, 0xde6e, 0x847b,
  0xde9e, 0x846e, 0xdecf, 0x8461, 0xdeff, 0x8454, 0xdf30, 0x8447,
  0xdf61, 0x843a, 0xdf91, 0x842d, 0xdfc2, 0x8421, 0xdff2, 0x8414,
  0xe023, 0x8407, 0xe054, 0x83fb, 0xe085, 0x83ef, 0xe0b5, 0x83e2,
  0xe0e6, 0x83d6, 0xe117, 0x83ca, 0xe148, 0x83be, 0xe178, 0x83b2,
  0xe1a9, 0x83a6, 0xe1da, 0x839a, 0xe20b, 0x838e, 0xe23c, 0x8382,
  0xe26d, 0x8377, 0xe29e, 0x836b, 0xe2cf, 0x8360, 0xe2ff, 0x8354,
  0xe330, 0x8349, 0xe361, 0x833e, 0xe392, 0x8332, 0xe3c3, 0x8327,
  0xe3f4, 0x831c, 0xe426, 0x8311, 0xe457, 0x8306, 0xe488, 0x82fb,
  0xe4b9, 0x82f1, 0xe4ea, 0x82e6, 0xe51b, 0x82db, 0xe54c, 0x82d1,
  0xe57d, 0x82c6, 0xe5af, 0x82bc, 0xe5e0, 0x82b2, 0xe611, 0x82a8,
  0xe642, 0x829d, 0xe673, 0x8293, 0xe6a5, 0x8289, 0xe6d6, 0x827f,
  0xe707, 0x8276, 0xe739, 0x826c, 0xe76a, 0x8262, 0xe79b, 0x8259,
  0xe7cd, 0x824f, 0xe7fe, 0x8246, 0xe82f, 0x823c, 0xe861, 0x8233,
  0xe892, 0x822a, 0xe8c4, 0x8220, 0xe8f5, 0x8217, 0xe926, 0x820e,
  0xe958, 0x8205, 0xe989, 0x81fd, 0xe9bb, 0x81f4, 0xe9ec, 0x81eb,
  0xea1e, 0x81e2, 0xea4f, 0x81da, 0xea81, 0x81d1, 0xeab3, 0x81c9,
  0xeae4, 0x81c1, 0xeb16, 0x81b8, 0xeb47, 0x81b0, 0xeb79, 0x81a8,
  0xebab, 0x81a0, 0xebdc, 0x8198, 0xec0e, 0x8190, 0xec3f, 0x8188,
  0xec71, 0x8181, 0xeca3, 0x8179, 0xecd5, 0x8172, 0xed06, 0x816a,
  0xed38, 0x8163, 0xed6a, 0x815b, 0xed9b, 0x8154, 0xedcd, 0x814d,
  0xedff, 0x8146, 0xee31, 0x813f, 0xee62, 0x8138, 0xee94, 0x8131,
  0xeec6, 0x812a, 0xeef8, 0x8123, 0xef2a, 0x811d, 0xef5c, 0x8116,
  0xef8d, 0x8110, 0xefbf, 0x8109, 0xeff1, 0x8103, 0xf023, 0x80fd,
  0xf055, 0x80f6, 0xf087, 0x80f0, 0xf0b9, 0x80ea, 0xf0eb, 0x80e4,
  0xf11c, 0x80de, 0xf14e, 0x80d9, 0xf180, 0x80d3, 0xf1b2, 0x80cd,
 0xf1e4, 0x80c8, 0xf216, 0x80c2, 0xf248, 0x80bd, 0xf27a, 0x80b7,
 0xf2ac, 0x80b2, 0xf2de, 0x80ad, 0xf310, 0x80a8, 0xf342, 0x80a3,
 0xf374, 0x809e, 0xf3a6, 0x8099, 0xf3d8, 0x8094, 0xf40a, 0x808f,
 0xf43c, 0x808b, 0xf46e, 0x8086, 0xf4a0, 0x8082, 0xf4d3, 0x807d,
 0xf505, 0x8079, 0xf537, 0x8075, 0xf569, 0x8070, 0xf59b, 0x806c,
 0xf5cd, 0x8068, 0xf5ff, 0x8064, 0xf631, 0x8060, 0xf663, 0x805d,
 0xf695, 0x8059, 0xf6c8, 0x8055, 0xf6fa, 0x8052, 0xf72c, 0x804e,
 0xf75e, 0x804b, 0xf790, 0x8047, 0xf7c2, 0x8044, 0xf7f4, 0x8041,
 0xf827, 0x803e, 0xf859, 0x803b, 0xf88b, 0x8038, 0xf8bd, 0x8035,
 0xf8ef, 0x8032, 0xf922, 0x802f, 0xf954, 0x802d, 0xf986, 0x802a,
 0xf9b8, 0x8027, 0xf9ea, 0x8025, 0xfa1d, 0x8023, 0xfa4f, 0x8020,
 0xfa81, 0x801e, 0xfab3, 0x801c, 0xfae5, 0x801a, 0xfb18, 0x8018,
 0xfb4a, 0x8016, 0xfb7c, 0x8014, 0xfbae, 0x8013, 0xfbe1, 0x8011,
 0xfc13, 0x800f, 0xfc45, 0x800e, 0xfc77, 0x800c, 0xfcaa, 0x800b,
 0xfcdc, 0x800a, 0xfd0e, 0x8009, 0xfd40, 0x8008, 0xfd73, 0x8007,
 0xfda5, 0x8006, 0xfdd7, 0x8005, 0xfe09, 0x8004, 0xfe3c, 0x8003,
 0xfe6e, 0x8002, 0xfea0, 0x8002, 0xfed2, 0x8001, 0xff05, 0x8001,
 0xff37, 0x8001, 0xff69, 0x8000, 0xff9b, 0x8000, 0xffce, 0x8000,
};

/**    
* @} end of CFFT_CIFFT group    
*/

/*    
* @brief  Q15 table for reciprocal    
*/
const q15_t ALIGN4 armRecipTableQ15[64] = {
 0x7F03, 0x7D13, 0x7B31, 0x795E, 0x7798, 0x75E0,
 0x7434, 0x7294, 0x70FF, 0x6F76, 0x6DF6, 0x6C82,
 0x6B16, 0x69B5, 0x685C, 0x670C, 0x65C4, 0x6484,
 0x634C, 0x621C, 0x60F3, 0x5FD0, 0x5EB5, 0x5DA0,
 0x5C91, 0x5B88, 0x5A85, 0x5988, 0x5890, 0x579E,
 0x56B0, 0x55C8, 0x54E4, 0x5405, 0x532B, 0x5255,
 0x5183, 0x50B6, 0x4FEC, 0x4F26, 0x4E64, 0x4DA6,
 0x4CEC, 0x4C34, 0x4B81, 0x4AD0, 0x4A23, 0x4978,
 0x48D1, 0x482D, 0x478C, 0x46ED, 0x4651, 0x45B8,
 0x4521, 0x448D, 0x43FC, 0x436C, 0x42DF, 0x4255,
 0x41CC, 0x4146, 0x40C2, 0x4040
};

/*    
* @brief  Q31 table for reciprocal    
*/
const q31_t armRecipTableQ31[64] = {
  0x7F03F03F, 0x7D137420, 0x7B31E739, 0x795E9F94, 0x7798FD29, 0x75E06928,
  0x7434554D, 0x72943B4B, 0x70FF9C40, 0x6F760031, 0x6DF6F593, 0x6C8210E3,
  0x6B16EC3A, 0x69B526F6, 0x685C655F, 0x670C505D, 0x65C4952D, 0x6484E519,
  0x634CF53E, 0x621C7E4F, 0x60F33C61, 0x5FD0EEB3, 0x5EB55785, 0x5DA03BEB,
  0x5C9163A1, 0x5B8898E6, 0x5A85A85A, 0x598860DF, 0x58909373, 0x579E1318,
  0x56B0B4B8, 0x55C84F0B, 0x54E4BA80, 0x5405D124, 0x532B6E8F, 0x52556FD0,
  0x5183B35A, 0x50B618F3, 0x4FEC81A2, 0x4F26CFA2, 0x4E64E64E, 0x4DA6AA1D,
  0x4CEC008B, 0x4C34D010, 0x4B810016, 0x4AD078EF, 0x4A2323C4, 0x4978EA96,
  0x48D1B827, 0x482D77FE, 0x478C1657, 0x46ED801D, 0x4651A2E5, 0x45B86CE2,
  0x4521CCE1, 0x448DB244, 0x43FC0CFA, 0x436CCD78, 0x42DFE4B4, 0x42554426,
  0x41CCDDB6, 0x4146A3C6, 0x40C28923, 0x40408102
};

const uint16_t armBitRevIndexTable16[ARMBITREVINDEXTABLE__16_TABLE_LENGTH] = 
{
   //8x2, size 20
   8,64, 24,72, 16,64, 40,80, 32,64, 56,88, 48,72, 88,104, 72,96, 104,112
};

const uint16_t armBitRevIndexTable32[ARMBITREVINDEXTABLE__32_TABLE_LENGTH] = 
{
   //8x4, size 48
   8,64, 16,128, 24,192, 32,64, 40,72, 48,136, 56,200, 64,128, 72,80, 88,208,
   80,144, 96,192, 104,208, 112,152, 120,216, 136,192, 144,160, 168,208,
   152,224, 176,208, 184,232, 216,240, 200,224, 232,240
};

const uint16_t armBitRevIndexTable64[ARMBITREVINDEXTABLE__64_TABLE_LENGTH] = 
{   
   //radix 8, size 56
   8,64, 16,128, 24,192, 32,256, 40,320, 48,384, 56,448, 80,136, 88,200, 
   96,264, 104,328, 112,392, 120,456, 152,208, 160,272, 168,336, 176,400, 
   184,464, 224,280, 232,344, 240,408, 248,472, 296,352, 304,416, 312,480, 
   368,424, 376,488, 440,496
};

const uint16_t armBitRevIndexTable128[ARMBITREVINDEXTABLE_128_TABLE_LENGTH] = 
{
   //8x2, size 208
   8,512, 16,64, 24,576, 32,128, 40,640, 48,192, 56,704, 64,256, 72,768, 
   80,320, 88,832, 96,384, 104,896, 112,448, 120,960, 128,512, 136,520, 
   144,768, 152,584, 160,520, 168,648, 176,200, 184,712, 192,264, 200,776, 
   208,328, 216,840, 224,392, 232,904, 240,456, 248,968, 264,528, 272,320, 
   280,592, 288,768, 296,656, 304,328, 312,720, 328,784, 344,848, 352,400, 
   360,912, 368,464, 376,976, 384,576, 392,536, 400,832, 408,600, 416,584, 
   424,664, 432,840, 440,728, 448,592, 456,792, 464,848, 472,856, 480,600, 
   488,920, 496,856, 504,984, 520,544, 528,576, 536,608, 552,672, 560,608, 
   568,736, 576,768, 584,800, 592,832, 600,864, 608,800, 616,928, 624,864, 
   632,992, 648,672, 656,896, 664,928, 688,904, 696,744, 704,896, 712,808, 
   720,912, 728,872, 736,928, 744,936, 752,920, 760,1000, 776,800, 784,832, 
   792,864, 808,904, 816,864, 824,920, 840,864, 856,880, 872,944, 888,1008, 
   904,928, 912,960, 920,992, 944,968, 952,1000, 968,992, 984,1008
};

const uint16_t armBitRevIndexTable256[ARMBITREVINDEXTABLE_256_TABLE_LENGTH] = 
{
   //8x4, size 440
   8,512, 16,1024, 24,1536, 32,64, 40,576, 48,1088, 56,1600, 64,128, 72,640, 
   80,1152, 88,1664, 96,192, 104,704, 112,1216, 120,1728, 128,256, 136,768, 
   144,1280, 152,1792, 160,320, 168,832, 176,1344, 184,1856, 192,384, 
   200,896, 208,1408, 216,1920, 224,448, 232,960, 240,1472, 248,1984, 
   256,512, 264,520, 272,1032, 280,1544, 288,640, 296,584, 304,1096, 312,1608, 
   320,768, 328,648, 336,1160, 344,1672, 352,896, 360,712, 368,1224, 376,1736, 
   384,520, 392,776, 400,1288, 408,1800, 416,648, 424,840, 432,1352, 440,1864, 
   448,776, 456,904, 464,1416, 472,1928, 480,904, 488,968, 496,1480, 504,1992, 
   520,528, 512,1024, 528,1040, 536,1552, 544,1152, 552,592, 560,1104, 
   568,1616, 576,1280, 584,656, 592,1168, 600,1680, 608,1408, 616,720, 
   624,1232, 632,1744, 640,1032, 648,784, 656,1296, 664,1808, 672,1160, 
   680,848, 688,1360, 696,1872, 704,1288, 712,912, 720,1424, 728,1936, 
   736,1416, 744,976, 752,1488, 760,2000, 768,1536, 776,1552, 784,1048, 
   792,1560, 800,1664, 808,1680, 816,1112, 824,1624, 832,1792, 840,1808, 
   848,1176, 856,1688, 864,1920, 872,1936, 880,1240, 888,1752, 896,1544, 
   904,1560, 912,1304, 920,1816, 928,1672, 936,1688, 944,1368, 952,1880, 
   960,1800, 968,1816, 976,1432, 984,1944, 992,1928, 1000,1944, 1008,1496, 
   1016,2008, 1032,1152, 1040,1056, 1048,1568, 1064,1408, 1072,1120, 
   1080,1632, 1088,1536, 1096,1160, 1104,1184, 1112,1696, 1120,1552, 
   1128,1416, 1136,1248, 1144,1760, 1160,1664, 1168,1312, 1176,1824, 
   1184,1544, 1192,1920, 1200,1376, 1208,1888, 1216,1568, 1224,1672, 
   1232,1440, 1240,1952, 1248,1560, 1256,1928, 1264,1504, 1272,2016, 
   1288,1312, 1296,1408, 1304,1576, 1320,1424, 1328,1416, 1336,1640, 
   1344,1792, 1352,1824, 1360,1920, 1368,1704, 1376,1800, 1384,1432, 
   1392,1928, 1400,1768, 1416,1680, 1432,1832, 1440,1576, 1448,1936, 
   1456,1832, 1464,1896, 1472,1808, 1480,1688, 1488,1936, 1496,1960, 
   1504,1816, 1512,1944, 1520,1944, 1528,2024, 1560,1584, 1592,1648, 
   1600,1792, 1608,1920, 1616,1800, 1624,1712, 1632,1808, 1640,1936, 
   1648,1816, 1656,1776, 1672,1696, 1688,1840, 1704,1952, 1712,1928, 
   1720,1904, 1728,1824, 1736,1952, 1744,1832, 1752,1968, 1760,1840, 
   1768,1960, 1776,1944, 1784,2032, 1864,1872, 1848,1944, 1872,1888, 
   1880,1904, 1888,1984, 1896,2000, 1912,2032, 1904,2016, 1976,2032,
   1960,1968, 2008,2032, 1992,2016, 2024,2032
};

const uint16_t armBitRevIndexTable512[ARMBITREVINDEXTABLE_512_TABLE_LENGTH] = 
{
   //radix 8, size 448
   8,512, 16,1024, 24,1536, 32,2048, 40,2560, 48,3072, 56,3584, 72,576, 
   80,1088, 88,1600, 96,2112, 104,2624, 112,3136, 120,3648, 136,640, 144,1152, 
   152,1664, 160,2176, 168,2688, 176,3200, 184,3712, 200,704, 208,1216, 
   216,1728, 224,2240, 232,2752, 240,3264, 248,3776, 264,768, 272,1280, 
   280,1792, 288,2304, 296,2816, 304,3328, 312,3840, 328,832, 336,1344, 
   344,1856, 352,2368, 360,2880, 368,3392, 376,3904, 392,896, 400,1408, 
   408,1920, 416,2432, 424,2944, 432,3456, 440,3968, 456,960, 464,1472, 
   472,1984, 480,2496, 488,3008, 496,3520, 504,4032, 528,1032, 536,1544, 
   544,2056, 552,2568, 560,3080, 568,3592, 592,1096, 600,1608, 608,2120, 
   616,2632, 624,3144, 632,3656, 656,1160, 664,1672, 672,2184, 680,2696, 
   688,3208, 696,3720, 720,1224, 728,1736, 736,2248, 744,2760, 752,3272, 
   760,3784, 784,1288, 792,1800, 800,2312, 808,2824, 816,3336, 824,3848, 
   848,1352, 856,1864, 864,2376, 872,2888, 880,3400, 888,3912, 912,1416, 
   920,1928, 928,2440, 936,2952, 944,3464, 952,3976, 976,1480, 984,1992, 
   992,2504, 1000,3016, 1008,3528, 1016,4040, 1048,1552, 1056,2064, 1064,2576, 
   1072,3088, 1080,3600, 1112,1616, 1120,2128, 1128,2640, 1136,3152, 
   1144,3664, 1176,1680, 1184,2192, 1192,2704, 1200,3216, 1208,3728, 
   1240,1744, 1248,2256, 1256,2768, 1264,3280, 1272,3792, 1304,1808, 
   1312,2320, 1320,2832, 1328,3344, 1336,3856, 1368,1872, 1376,2384, 
   1384,2896, 1392,3408, 1400,3920, 1432,1936, 1440,2448, 1448,2960, 
   1456,3472, 1464,3984, 1496,2000, 1504,2512, 1512,3024, 1520,3536, 
   1528,4048, 1568,2072, 1576,2584, 1584,3096, 1592,3608, 1632,2136, 
   1640,2648, 1648,3160, 1656,3672, 1696,2200, 1704,2712, 1712,3224, 
   1720,3736, 1760,2264, 1768,2776, 1776,3288, 1784,3800, 1824,2328, 
   1832,2840, 1840,3352, 1848,3864, 1888,2392, 1896,2904, 1904,3416, 
   1912,3928, 1952,2456, 1960,2968, 1968,3480, 1976,3992, 2016,2520, 
   2024,3032, 2032,3544, 2040,4056, 2088,2592, 2096,3104, 2104,3616, 
   2152,2656, 2160,3168, 2168,3680, 2216,2720, 2224,3232, 2232,3744, 
   2280,2784, 2288,3296, 2296,3808, 2344,2848, 2352,3360, 2360,3872, 
   2408,2912, 2416,3424, 2424,3936, 2472,2976, 2480,3488, 2488,4000, 
   2536,3040, 2544,3552, 2552,4064, 2608,3112, 2616,3624, 2672,3176, 
   2680,3688, 2736,3240, 2744,3752, 2800,3304, 2808,3816, 2864,3368, 
   2872,3880, 2928,3432, 2936,3944, 2992,3496, 3000,4008, 3056,3560, 
   3064,4072, 3128,3632, 3192,3696, 3256,3760, 3320,3824, 3384,3888, 
   3448,3952, 3512,4016, 3576,4080
};

const uint16_t armBitRevIndexTable1024[ARMBITREVINDEXTABLE1024_TABLE_LENGTH] = 
{
   //8x2, size 1800
   8,4096, 16,512, 24,4608, 32,1024, 40,5120, 48,1536, 56,5632, 64,2048, 
   72,6144, 80,2560, 88,6656, 96,3072, 104,7168, 112,3584, 120,7680, 128,2048, 
   136,4160, 144,576, 152,4672, 160,1088, 168,5184, 176,1600, 184,5696, 
   192,2112, 200,6208, 208,2624, 216,6720, 224,3136, 232,7232, 240,3648, 
   248,7744, 256,2048, 264,4224, 272,640, 280,4736, 288,1152, 296,5248, 
   304,1664, 312,5760, 320,2176, 328,6272, 336,2688, 344,6784, 352,3200, 
   360,7296, 368,3712, 376,7808, 384,2112, 392,4288, 400,704, 408,4800, 
   416,1216, 424,5312, 432,1728, 440,5824, 448,2240, 456,6336, 464,2752, 
   472,6848, 480,3264, 488,7360, 496,3776, 504,7872, 512,2048, 520,4352, 
   528,768, 536,4864, 544,1280, 552,5376, 560,1792, 568,5888, 576,2304, 
   584,6400, 592,2816, 600,6912, 608,3328, 616,7424, 624,3840, 632,7936, 
   640,2176, 648,4416, 656,832, 664,4928, 672,1344, 680,5440, 688,1856, 
   696,5952, 704,2368, 712,6464, 720,2880, 728,6976, 736,3392, 744,7488, 
   752,3904, 760,8000, 768,2112, 776,4480, 784,896, 792,4992, 800,1408, 
   808,5504, 816,1920, 824,6016, 832,2432, 840,6528, 848,2944, 856,7040, 
   864,3456, 872,7552, 880,3968, 888,8064, 896,2240, 904,4544, 912,960, 
   920,5056, 928,1472, 936,5568, 944,1984, 952,6080, 960,2496, 968,6592, 
   976,3008, 984,7104, 992,3520, 1000,7616, 1008,4032, 1016,8128, 1024,4096, 
   1032,4104, 1040,4352, 1048,4616, 1056,4104, 1064,5128, 1072,1544, 
   1080,5640, 1088,2056, 1096,6152, 1104,2568, 1112,6664, 1120,3080, 
   1128,7176, 1136,3592, 1144,7688, 1152,6144, 1160,4168, 1168,6400, 
   1176,4680, 1184,6152, 1192,5192, 1200,1608, 1208,5704, 1216,2120, 
   1224,6216, 1232,2632, 1240,6728, 1248,3144, 1256,7240, 1264,3656, 
   1272,7752, 1280,4160, 1288,4232, 1296,4416, 1304,4744, 1312,4168, 
   1320,5256, 1328,1672, 1336,5768, 1344,2184, 1352,6280, 1360,2696, 
   1368,6792, 1376,3208, 1384,7304, 1392,3720, 1400,7816, 1408,6208, 
   1416,4296, 1424,6464, 1432,4808, 1440,6216, 1448,5320, 1456,1736, 
   1464,5832, 1472,2248, 1480,6344, 1488,2760, 1496,6856, 1504,3272, 
   1512,7368, 1520,3784, 1528,7880, 1536,4224, 1544,4360, 1552,4480, 
   1560,4872, 1568,4232, 1576,5384, 1584,1800, 1592,5896, 1600,2312, 
   1608,6408, 1616,2824, 1624,6920, 1632,3336, 1640,7432, 1648,3848, 
   1656,7944, 1664,6272, 1672,4424, 1680,6528, 1688,4936, 1696,6280, 
   1704,5448, 1712,1864, 1720,5960, 1728,2376, 1736,6472, 1744,2888, 
   1752,6984, 1760,3400, 1768,7496, 1776,3912, 1784,8008, 1792,4288, 
   1800,4488, 1808,4544, 1816,5000, 1824,4296, 1832,5512, 1840,1928, 
   1848,6024, 1856,2440, 1864,6536, 1872,2952, 1880,7048, 1888,3464, 
   1896,7560, 1904,3976, 1912,8072, 1920,6336, 1928,4552, 1936,6592, 
   1944,5064, 1952,6344, 1960,5576, 1968,1992, 1976,6088, 1984,2504, 
   1992,6600, 2000,3016, 2008,7112, 2016,3528, 2024,7624, 2032,4040, 
   2040,8136, 2056,4112, 2064,2112, 2072,4624, 2080,4352, 2088,5136, 
   2096,4480, 2104,5648, 2120,6160, 2128,2576, 2136,6672, 2144,3088, 
   2152,7184, 2160,3600, 2168,7696, 2176,2560, 2184,4176, 2192,2816, 
   2200,4688, 2208,2568, 2216,5200, 2224,2824, 2232,5712, 2240,2576, 
   2248,6224, 2256,2640, 2264,6736, 2272,3152, 2280,7248, 2288,3664, 
   2296,7760, 2312,4240, 2320,2432, 2328,4752, 2336,6400, 2344,5264, 
   2352,6528, 2360,5776, 2368,2816, 2376,6288, 2384,2704, 2392,6800, 
   2400,3216, 2408,7312, 2416,3728, 2424,7824, 2432,2624, 2440,4304, 
   2448,2880, 2456,4816, 2464,2632, 2472,5328, 2480,2888, 2488,5840, 
   2496,2640, 2504,6352, 2512,2768, 2520,6864, 2528,3280, 2536,7376, 
   2544,3792, 2552,7888, 2568,4368, 2584,4880, 2592,4416, 2600,5392, 
   2608,4544, 2616,5904, 2632,6416, 2640,2832, 2648,6928, 2656,3344, 
   2664,7440, 2672,3856, 2680,7952, 2696,4432, 2704,2944, 2712,4944, 
   2720,4432, 2728,5456, 2736,2952, 2744,5968, 2752,2944, 2760,6480, 
   2768,2896, 2776,6992, 2784,3408, 2792,7504, 2800,3920, 2808,8016, 
   2824,4496, 2840,5008, 2848,6464, 2856,5520, 2864,6592, 2872,6032, 
   2888,6544, 2896,2960, 2904,7056, 2912,3472, 2920,7568, 2928,3984, 
   2936,8080, 2952,4560, 2960,3008, 2968,5072, 2976,6480, 2984,5584, 
   2992,3016, 3000,6096, 3016,6608, 3032,7120, 3040,3536, 3048,7632, 
   3056,4048, 3064,8144, 3072,4608, 3080,4120, 3088,4864, 3096,4632, 
   3104,4616, 3112,5144, 3120,4872, 3128,5656, 3136,4624, 3144,6168, 
   3152,4880, 3160,6680, 3168,4632, 3176,7192, 3184,3608, 3192,7704, 
   3200,6656, 3208,4184, 3216,6912, 3224,4696, 3232,6664, 3240,5208, 
   3248,6920, 3256,5720, 3264,6672, 3272,6232, 3280,6928, 3288,6744, 
   3296,6680, 3304,7256, 3312,3672, 3320,7768, 3328,4672, 3336,4248, 
   3344,4928, 3352,4760, 3360,4680, 3368,5272, 3376,4936, 3384,5784, 
   3392,4688, 3400,6296, 3408,4944, 3416,6808, 3424,4696, 3432,7320, 
   3440,3736, 3448,7832, 3456,6720, 3464,4312, 3472,6976, 3480,4824, 
   3488,6728, 3496,5336, 3504,6984, 3512,5848, 3520,6736, 3528,6360, 
   3536,6992, 3544,6872, 3552,6744, 3560,7384, 3568,3800, 3576,7896, 
   3584,4736, 3592,4376, 3600,4992, 3608,4888, 3616,4744, 3624,5400, 
   3632,5000, 3640,5912, 3648,4752, 3656,6424, 3664,5008, 3672,6936, 
   3680,4760, 3688,7448, 3696,3864, 3704,7960, 3712,6784, 3720,4440, 
   3728,7040, 3736,4952, 3744,6792, 3752,5464, 3760,7048, 3768,5976, 
   3776,6800, 3784,6488, 3792,7056, 3800,7000, 3808,6808, 3816,7512, 
   3824,3928, 3832,8024, 3840,4800, 3848,4504, 3856,5056, 3864,5016, 
   3872,4808, 3880,5528, 3888,5064, 3896,6040, 3904,4816, 3912,6552, 
   3920,5072, 3928,7064, 3936,4824, 3944,7576, 3952,3992, 3960,8088, 
   3968,6848, 3976,4568, 3984,7104, 3992,5080, 4000,6856, 4008,5592, 
   4016,7112, 4024,6104, 4032,6864, 4040,6616, 4048,7120, 4056,7128, 
   4064,6872, 4072,7640, 4080,7128, 4088,8152, 4104,4128, 4112,4160, 
   4120,4640, 4136,5152, 4144,4232, 4152,5664, 4160,4352, 4168,6176, 
   4176,4416, 4184,6688, 4192,4616, 4200,7200, 4208,4744, 4216,7712, 
   4224,4608, 4232,4616, 4240,4672, 4248,4704, 4256,4640, 4264,5216, 
   4272,4704, 4280,5728, 4288,4864, 4296,6240, 4304,4928, 4312,6752, 
   4320,4632, 4328,7264, 4336,4760, 4344,7776, 4360,4640, 4368,4416, 
   4376,4768, 4384,6152, 4392,5280, 4400,6280, 4408,5792, 4424,6304, 
   4440,6816, 4448,6664, 4456,7328, 4464,6792, 4472,7840, 4480,4624, 
   4488,4632, 4496,4688, 4504,4832, 4512,6168, 4520,5344, 4528,6296, 
   4536,5856, 4544,4880, 4552,6368, 4560,4944, 4568,6880, 4576,6680, 
   4584,7392, 4592,6808, 4600,7904, 4608,6144, 4616,6152, 4624,6208, 
   4632,4896, 4640,6176, 4648,5408, 4656,6240, 4664,5920, 4672,6400, 
   4680,6432, 4688,6464, 4696,6944, 4704,6432, 4712,7456, 4720,4808, 
   4728,7968, 4736,6656, 4744,6664, 4752,6720, 4760,4960, 4768,6688, 
   4776,5472, 4784,6752, 4792,5984, 4800,6912, 4808,6496, 4816,6976, 
   4824,7008, 4832,6944, 4840,7520, 4848,7008, 4856,8032, 4864,6160, 
   4872,6168, 4880,6224, 4888,5024, 4896,6216, 4904,5536, 4912,6344, 
   4920,6048, 4928,6416, 4936,6560, 4944,6480, 4952,7072, 4960,6728, 
   4968,7584, 4976,6856, 4984,8096, 4992,6672, 5000,6680, 5008,6736, 
   5016,5088, 5024,6232, 5032,5600, 5040,6360, 5048,6112, 5056,6928, 
   5064,6624, 5072,6992, 5080,7136, 5088,6744, 5096,7648, 5104,6872, 
   5112,8160, 5128,5152, 5136,5376, 5144,5408, 5168,5384, 5176,5672, 
   5184,5376, 5192,6184, 5200,5392, 5208,6696, 5216,5408, 5224,7208, 
   5232,5400, 5240,7720, 5248,7168, 5256,7200, 5264,7424, 5272,7456, 
   5280,7176, 5288,7208, 5296,7432, 5304,5736, 5312,7184, 5320,6248, 
   5328,7440, 5336,6760, 5344,7192, 5352,7272, 5360,7448, 5368,7784, 
   5384,5408, 5392,5440, 5400,5472, 5408,6184, 5416,7208, 5424,5448, 
   5432,5800, 5448,6312, 5464,6824, 5472,6696, 5480,7336, 5488,6824, 
   5496,7848, 5504,7232, 5512,7264, 5520,7488, 5528,7520, 5536,7240, 
   5544,7272, 5552,7496, 5560,5864, 5568,7248, 5576,6376, 5584,7504, 
   5592,6888, 5600,7256, 5608,7400, 5616,7512, 5624,7912, 5632,7168, 
   5640,7176, 5648,7232, 5656,7240, 5664,7200, 5672,7208, 5680,7264, 
   5688,5928, 5696,7424, 5704,6440, 5712,7488, 5720,6952, 5728,7456, 
   5736,7464, 5744,7520, 5752,7976, 5760,7296, 5768,7328, 5776,7552, 
   5784,7584, 5792,7304, 5800,7336, 5808,7560, 5816,5992, 5824,7312, 
   5832,6504, 5840,7568, 5848,7016, 5856,7320, 5864,7528, 5872,7576, 
   5880,8040, 5888,7184, 5896,7192, 5904,7248, 5912,7256, 5920,6248, 
   5928,7272, 5936,6376, 5944,6056, 5952,7440, 5960,6568, 5968,7504, 
   5976,7080, 5984,6760, 5992,7592, 6000,6888, 6008,8104, 6016,7360, 
   6024,7392, 6032,7616, 6040,7648, 6048,7368, 6056,7400, 6064,7624, 
   6072,6120, 6080,7376, 6088,6632, 6096,7632, 6104,7144, 6112,7384, 
   6120,7656, 6128,7640, 6136,8168, 6168,6240, 6192,6216, 6200,7264, 
   6232,6704, 6248,7216, 6256,6680, 6264,7728, 6272,6656, 6280,6664, 
   6288,6912, 6296,6496, 6304,6688, 6312,6696, 6320,6944, 6328,7520, 
   6336,6672, 6344,6680, 6352,6928, 6360,6768, 6368,6704, 6376,7280, 
   6384,6744, 6392,7792, 6408,6432, 6424,6752, 6440,7432, 6448,6536, 
   6456,7560, 6472,6944, 6488,6832, 6496,6920, 6504,7344, 6512,7048, 
   6520,7856, 6528,6720, 6536,6728, 6544,6976, 6552,7008, 6560,6752, 
   6568,7448, 6576,7008, 6584,7576, 6592,6736, 6600,6744, 6608,6992, 
   6616,6896, 6624,6936, 6632,7408, 6640,7064, 6648,7920, 6712,7280, 
   6744,6960, 6760,7472, 6768,6936, 6776,7984, 6800,6848, 6808,6856, 
   6832,6880, 6840,6888, 6848,7040, 6856,7048, 6864,7104, 6872,7024, 
   6880,7072, 6888,7536, 6896,7136, 6904,8048, 6952,7496, 6968,7624, 
   6984,7008, 7000,7088, 7016,7600, 7024,7112, 7032,8112, 7056,7104, 
   7064,7112, 7080,7512, 7088,7136, 7096,7640, 7128,7152, 7144,7664, 
   7160,8176, 7176,7200, 7192,7216, 7224,7272, 7240,7264, 7256,7280, 
   7288,7736, 7296,7680, 7304,7712, 7312,7936, 7320,7968, 7328,7688, 
   7336,7720, 7344,7944, 7352,7976, 7360,7696, 7368,7728, 7376,7952, 
   7384,7984, 7392,7704, 7400,7736, 7408,7960, 7416,7800, 7432,7456, 
   7448,7472, 7480,7592, 7496,7520, 7512,7536, 7528,7976, 7544,7864, 
   7552,7744, 7560,7776, 7568,8000, 7576,8032, 7584,7752, 7592,7784, 
   7600,8008, 7608,8040, 7616,7760, 7624,7792, 7632,8016, 7640,8048, 
   7648,7768, 7656,7800, 7664,8024, 7672,7928, 7688,7712, 7704,7728, 
   7752,7776, 7768,7792, 7800,7992, 7816,7840, 7824,8064, 7832,8096, 
   7856,8072, 7864,8104, 7872,8064, 7880,8072, 7888,8080, 7896,8112, 
   7904,8096, 7912,8104, 7920,8088, 7928,8056, 7944,7968, 7960,7984, 
   8008,8032, 8024,8048, 8056,8120, 8072,8096, 8080,8128, 8088,8160, 
   8112,8136, 8120,8168, 8136,8160, 8152,8176
};

const uint16_t armBitRevIndexTable2048[ARMBITREVINDEXTABLE2048_TABLE_LENGTH] = 
{
   //8x2, size 3808
   8,4096, 16,8192, 24,12288, 32,512, 40,4608, 48,8704, 56,12800, 64,1024, 
   72,5120, 80,9216, 88,13312, 96,1536, 104,5632, 112,9728, 120,13824, 
   128,2048, 136,6144, 144,10240, 152,14336, 160,2560, 168,6656, 176,10752, 
   184,14848, 192,3072, 200,7168, 208,11264, 216,15360, 224,3584, 232,7680, 
   240,11776, 248,15872, 256,1024, 264,4160, 272,8256, 280,12352, 288,576, 
   296,4672, 304,8768, 312,12864, 320,1088, 328,5184, 336,9280, 344,13376, 
   352,1600, 360,5696, 368,9792, 376,13888, 384,2112, 392,6208, 400,10304, 
   408,14400, 416,2624, 424,6720, 432,10816, 440,14912, 448,3136, 456,7232, 
   464,11328, 472,15424, 480,3648, 488,7744, 496,11840, 504,15936, 512,2048, 
   520,4224, 528,8320, 536,12416, 544,640, 552,4736, 560,8832, 568,12928, 
   576,1152, 584,5248, 592,9344, 600,13440, 608,1664, 616,5760, 624,9856, 
   632,13952, 640,2176, 648,6272, 656,10368, 664,14464, 672,2688, 680,6784, 
   688,10880, 696,14976, 704,3200, 712,7296, 720,11392, 728,15488, 736,3712, 
   744,7808, 752,11904, 760,16000, 768,3072, 776,4288, 784,8384, 792,12480, 
   800,3200, 808,4800, 816,8896, 824,12992, 832,1216, 840,5312, 848,9408, 
   856,13504, 864,1728, 872,5824, 880,9920, 888,14016, 896,2240, 904,6336, 
   912,10432, 920,14528, 928,2752, 936,6848, 944,10944, 952,15040, 960,3264, 
   968,7360, 976,11456, 984,15552, 992,3776, 1000,7872, 1008,11968, 1016,16064, 
   1032,4352, 1040,8448, 1048,12544, 1056,3072, 1064,4864, 1072,8960, 
   1080,13056, 1088,1280, 1096,5376, 1104,9472, 1112,13568, 1120,1792, 
   1128,5888, 1136,9984, 1144,14080, 1152,2304, 1160,6400, 1168,10496, 
   1176,14592, 1184,2816, 1192,6912, 1200,11008, 1208,15104, 1216,3328, 
   1224,7424, 1232,11520, 1240,15616, 1248,3840, 1256,7936, 1264,12032, 
   1272,16128, 1288,4416, 1296,8512, 1304,12608, 1312,3328, 1320,4928, 
   1328,9024, 1336,13120, 1352,5440, 1360,9536, 1368,13632, 1376,1856, 
   1384,5952, 1392,10048, 1400,14144, 1408,2368, 1416,6464, 1424,10560, 
   1432,14656, 1440,2880, 1448,6976, 1456,11072, 1464,15168, 1472,3392, 
   1480,7488, 1488,11584, 1496,15680, 1504,3904, 1512,8000, 1520,12096, 
   1528,16192, 1536,2112, 1544,4480, 1552,8576, 1560,12672, 1568,2240, 
   1576,4992, 1584,9088, 1592,13184, 1600,2368, 1608,5504, 1616,9600, 
   1624,13696, 1632,1920, 1640,6016, 1648,10112, 1656,14208, 1664,2432, 
   1672,6528, 1680,10624, 1688,14720, 1696,2944, 1704,7040, 1712,11136, 
   1720,15232, 1728,3456, 1736,7552, 1744,11648, 1752,15744, 1760,3968, 
   1768,8064, 1776,12160, 1784,16256, 1792,3136, 1800,4544, 1808,8640, 
   1816,12736, 1824,3264, 1832,5056, 1840,9152, 1848,13248, 1856,3392, 
   1864,5568, 1872,9664, 1880,13760, 1888,1984, 1896,6080, 1904,10176, 
   1912,14272, 1920,2496, 1928,6592, 1936,10688, 1944,14784, 1952,3008, 
   1960,7104, 1968,11200, 1976,15296, 1984,3520, 1992,7616, 2000,11712, 
   2008,15808, 2016,4032, 2024,8128, 2032,12224, 2040,16320, 2048,4096, 
   2056,4104, 2064,8200, 2072,12296, 2080,4224, 2088,4616, 2096,8712, 
   2104,12808, 2112,4352, 2120,5128, 2128,9224, 2136,13320, 2144,4480, 
   2152,5640, 2160,9736, 2168,13832, 2176,4104, 2184,6152, 2192,10248, 
   2200,14344, 2208,2568, 2216,6664, 2224,10760, 2232,14856, 2240,3080, 
   2248,7176, 2256,11272, 2264,15368, 2272,3592, 2280,7688, 2288,11784, 
   2296,15880, 2304,5120, 2312,4168, 2320,8264, 2328,12360, 2336,5248, 
   2344,4680, 2352,8776, 2360,12872, 2368,5376, 2376,5192, 2384,9288, 
   2392,13384, 2400,5504, 2408,5704, 2416,9800, 2424,13896, 2432,5128, 
   2440,6216, 2448,10312, 2456,14408, 2464,2632, 2472,6728, 2480,10824, 
   2488,14920, 2496,3144, 2504,7240, 2512,11336, 2520,15432, 2528,3656, 
   2536,7752, 2544,11848, 2552,15944, 2560,6144, 2568,4232, 2576,8328, 
   2584,12424, 2592,6272, 2600,4744, 2608,8840, 2616,12936, 2624,6400, 
   2632,5256, 2640,9352, 2648,13448, 2656,6528, 2664,5768, 2672,9864, 
   2680,13960, 2688,6152, 2696,6280, 2704,10376, 2712,14472, 2720,6280, 
   2728,6792, 2736,10888, 2744,14984, 2752,3208, 2760,7304, 2768,11400, 
   2776,15496, 2784,3720, 2792,7816, 2800,11912, 2808,16008, 2816,7168, 
   2824,4296, 2832,8392, 2840,12488, 2848,7296, 2856,4808, 2864,8904, 
   2872,13000, 2880,7424, 2888,5320, 2896,9416, 2904,13512, 2912,7552, 
   2920,5832, 2928,9928, 2936,14024, 2944,7176, 2952,6344, 2960,10440, 
   2968,14536, 2976,7304, 2984,6856, 2992,10952, 3000,15048, 3008,3272, 
   3016,7368, 3024,11464, 3032,15560, 3040,3784, 3048,7880, 3056,11976, 
   3064,16072, 3072,4160, 3080,4360, 3088,8456, 3096,12552, 3104,4288, 
   3112,4872, 3120,8968, 3128,13064, 3136,4416, 3144,5384, 3152,9480, 
   3160,13576, 3168,4544, 3176,5896, 3184,9992, 3192,14088, 3200,4168, 
   3208,6408, 3216,10504, 3224,14600, 3232,4296, 3240,6920, 3248,11016, 
   3256,15112, 3264,3336, 3272,7432, 3280,11528, 3288,15624, 3296,3848, 
   3304,7944, 3312,12040, 3320,16136, 3328,5184, 3336,4424, 3344,8520, 
   3352,12616, 3360,5312, 3368,4936, 3376,9032, 3384,13128, 3392,5440, 
   3400,5448, 3408,9544, 3416,13640, 3424,5568, 3432,5960, 3440,10056, 
   3448,14152, 3456,5192, 3464,6472, 3472,10568, 3480,14664, 3488,5320, 
   3496,6984, 3504,11080, 3512,15176, 3520,5448, 3528,7496, 3536,11592, 
   3544,15688, 3552,3912, 3560,8008, 3568,12104, 3576,16200, 3584,6208, 
   3592,4488, 3600,8584, 3608,12680, 3616,6336, 3624,5000, 3632,9096, 
   3640,13192, 3648,6464, 3656,5512, 3664,9608, 3672,13704, 3680,6592, 
   3688,6024, 3696,10120, 3704,14216, 3712,6216, 3720,6536, 3728,10632, 
   3736,14728, 3744,6344, 3752,7048, 3760,11144, 3768,15240, 3776,6472, 
   3784,7560, 3792,11656, 3800,15752, 3808,3976, 3816,8072, 3824,12168, 
   3832,16264, 3840,7232, 3848,4552, 3856,8648, 3864,12744, 3872,7360, 
   3880,5064, 3888,9160, 3896,13256, 3904,7488, 3912,5576, 3920,9672, 
   3928,13768, 3936,7616, 3944,6088, 3952,10184, 3960,14280, 3968,7240, 
   3976,6600, 3984,10696, 3992,14792, 4000,7368, 4008,7112, 4016,11208, 
   4024,15304, 4032,7496, 4040,7624, 4048,11720, 4056,15816, 4064,7624, 
   4072,8136, 4080,12232, 4088,16328, 4096,8192, 4104,4112, 4112,8208, 
   4120,12304, 4128,8320, 4136,4624, 4144,8720, 4152,12816, 4160,8448, 
   4168,5136, 4176,9232, 4184,13328, 4192,8576, 4200,5648, 4208,9744, 
   4216,13840, 4224,8200, 4232,6160, 4240,10256, 4248,14352, 4256,8328, 
   4264,6672, 4272,10768, 4280,14864, 4288,8456, 4296,7184, 4304,11280, 
   4312,15376, 4320,8584, 4328,7696, 4336,11792, 4344,15888, 4352,9216, 
   4360,9232, 4368,8272, 4376,12368, 4384,9344, 4392,4688, 4400,8784, 
   4408,12880, 4416,9472, 4424,5200, 4432,9296, 4440,13392, 4448,9600, 
   4456,5712, 4464,9808, 4472,13904, 4480,9224, 4488,6224, 4496,10320, 
   4504,14416, 4512,9352, 4520,6736, 4528,10832, 4536,14928, 4544,9480, 
   4552,7248, 4560,11344, 4568,15440, 4576,9608, 4584,7760, 4592,11856, 
   4600,15952, 4608,10240, 4616,10256, 4624,8336, 4632,12432, 4640,10368, 
   4648,4752, 4656,8848, 4664,12944, 4672,10496, 4680,5264, 4688,9360, 
   4696,13456, 4704,10624, 4712,5776, 4720,9872, 4728,13968, 4736,10248, 
   4744,6288, 4752,10384, 4760,14480, 4768,10376, 4776,6800, 4784,10896, 
   4792,14992, 4800,10504, 4808,7312, 4816,11408, 4824,15504, 4832,10632, 
   4840,7824, 4848,11920, 4856,16016, 4864,11264, 4872,11280, 4880,8400, 
   4888,12496, 4896,11392, 4904,11408, 4912,8912, 4920,13008, 4928,11520, 
   4936,5328, 4944,9424, 4952,13520, 4960,11648, 4968,5840, 4976,9936, 
   4984,14032, 4992,11272, 5000,6352, 5008,10448, 5016,14544, 5024,11400, 
   5032,6864, 5040,10960, 5048,15056, 5056,11528, 5064,7376, 5072,11472, 
   5080,15568, 5088,11656, 5096,7888, 5104,11984, 5112,16080, 5120,8256, 
   5128,8272, 5136,8464, 5144,12560, 5152,8384, 5160,8400, 5168,8976, 
   5176,13072, 5184,8512, 5192,5392, 5200,9488, 5208,13584, 5216,8640, 
   5224,5904, 5232,10000, 5240,14096, 5248,8264, 5256,6416, 5264,10512, 
   5272,14608, 5280,8392, 5288,6928, 5296,11024, 5304,15120, 5312,8520, 
   5320,7440, 5328,11536, 5336,15632, 5344,8648, 5352,7952, 5360,12048, 
   5368,16144, 5376,9280, 5384,9296, 5392,8528, 5400,12624, 5408,9408, 
   5416,9424, 5424,9040, 5432,13136, 5440,9536, 5448,5456, 5456,9552, 
   5464,13648, 5472,9664, 5480,5968, 5488,10064, 5496,14160, 5504,9288, 
   5512,6480, 5520,10576, 5528,14672, 5536,9416, 5544,6992, 5552,11088, 
   5560,15184, 5568,9544, 5576,7504, 5584,11600, 5592,15696, 5600,9672, 
   5608,8016, 5616,12112, 5624,16208, 5632,10304, 5640,10320, 5648,8592, 
   5656,12688, 5664,10432, 5672,10448, 5680,9104, 5688,13200, 5696,10560, 
   5704,10576, 5712,9616, 5720,13712, 5728,10688, 5736,6032, 5744,10128, 
   5752,14224, 5760,10312, 5768,6544, 5776,10640, 5784,14736, 5792,10440, 
   5800,7056, 5808,11152, 5816,15248, 5824,10568, 5832,7568, 5840,11664, 
   5848,15760, 5856,10696, 5864,8080, 5872,12176, 5880,16272, 5888,11328, 
   5896,11344, 5904,8656, 5912,12752, 5920,11456, 5928,11472, 5936,9168, 
   5944,13264, 5952,11584, 5960,11600, 5968,9680, 5976,13776, 5984,11712, 
   5992,6096, 6000,10192, 6008,14288, 6016,11336, 6024,6608, 6032,10704, 
   6040,14800, 6048,11464, 6056,7120, 6064,11216, 6072,15312, 6080,11592, 
   6088,7632, 6096,11728, 6104,15824, 6112,11720, 6120,8144, 6128,12240, 
   6136,16336, 6144,12288, 6152,12304, 6160,8216, 6168,12312, 6176,12416, 
   6184,12432, 6192,8728, 6200,12824, 6208,12544, 6216,12560, 6224,9240, 
   6232,13336, 6240,12672, 6248,12688, 6256,9752, 6264,13848, 6272,12296, 
   6280,12312, 6288,10264, 6296,14360, 6304,12424, 6312,6680, 6320,10776, 
   6328,14872, 6336,12552, 6344,7192, 6352,11288, 6360,15384, 6368,12680, 
   6376,7704, 6384,11800, 6392,15896, 6400,13312, 6408,13328, 6416,8280, 
   6424,12376, 6432,13440, 6440,13456, 6448,8792, 6456,12888, 6464,13568, 
   6472,13584, 6480,9304, 6488,13400, 6496,13696, 6504,13712, 6512,9816, 
   6520,13912, 6528,13320, 6536,13336, 6544,10328, 6552,14424, 6560,13448, 
   6568,6744, 6576,10840, 6584,14936, 6592,13576, 6600,7256, 6608,11352, 
   6616,15448, 6624,13704, 6632,7768, 6640,11864, 6648,15960, 6656,14336, 
   6664,14352, 6672,8344, 6680,12440, 6688,14464, 6696,14480, 6704,8856, 
   6712,12952, 6720,14592, 6728,14608, 6736,9368, 6744,13464, 6752,14720, 
   6760,14736, 6768,9880, 6776,13976, 6784,14344, 6792,14360, 6800,10392, 
   6808,14488, 6816,14472, 6824,14488, 6832,10904, 6840,15000, 6848,14600, 
   6856,7320, 6864,11416, 6872,15512, 6880,14728, 6888,7832, 6896,11928, 
   6904,16024, 6912,15360, 6920,15376, 6928,8408, 6936,12504, 6944,15488, 
   6952,15504, 6960,8920, 6968,13016, 6976,15616, 6984,15632, 6992,9432, 
   7000,13528, 7008,15744, 7016,15760, 7024,9944, 7032,14040, 7040,15368, 
   7048,15384, 7056,10456, 7064,14552, 7072,15496, 7080,15512, 7088,10968, 
   7096,15064, 7104,15624, 7112,7384, 7120,11480, 7128,15576, 7136,15752, 
   7144,7896, 7152,11992, 7160,16088, 7168,12352, 7176,12368, 7184,8472, 
   7192,12568, 7200,12480, 7208,12496, 7216,8984, 7224,13080, 7232,12608, 
   7240,12624, 7248,9496, 7256,13592, 7264,12736, 7272,12752, 7280,10008, 
   7288,14104, 7296,12360, 7304,12376, 7312,10520, 7320,14616, 7328,12488, 
   7336,12504, 7344,11032, 7352,15128, 7360,12616, 7368,7448, 7376,11544, 
   7384,15640, 7392,12744, 7400,7960, 7408,12056, 7416,16152, 7424,13376, 
   7432,13392, 7440,8536, 7448,12632, 7456,13504, 7464,13520, 7472,9048, 
   7480,13144, 7488,13632, 7496,13648, 7504,9560, 7512,13656, 7520,13760, 
   7528,13776, 7536,10072, 7544,14168, 7552,13384, 7560,13400, 7568,10584, 
   7576,14680, 7584,13512, 7592,13528, 7600,11096, 7608,15192, 7616,13640, 
   7624,13656, 7632,11608, 7640,15704, 7648,13768, 7656,8024, 7664,12120, 
   7672,16216, 7680,14400, 7688,14416, 7696,8600, 7704,12696, 7712,14528, 
   7720,14544, 7728,9112, 7736,13208, 7744,14656, 7752,14672, 7760,9624, 
   7768,13720, 7776,14784, 7784,14800, 7792,10136, 7800,14232, 7808,14408, 
   7816,14424, 7824,10648, 7832,14744, 7840,14536, 7848,14552, 7856,11160, 
   7864,15256, 7872,14664, 7880,14680, 7888,11672, 7896,15768, 7904,14792, 
   7912,8088, 7920,12184, 7928,16280, 7936,15424, 7944,15440, 7952,8664, 
   7960,12760, 7968,15552, 7976,15568, 7984,9176, 7992,13272, 8000,15680, 
   8008,15696, 8016,9688, 8024,13784, 8032,15808, 8040,15824, 8048,10200, 
   8056,14296, 8064,15432, 8072,15448, 8080,10712, 8088,14808, 8096,15560, 
   8104,15576, 8112,11224, 8120,15320, 8128,15688, 8136,15704, 8144,11736, 
   8152,15832, 8160,15816, 8168,15832, 8176,12248, 8184,16344, 8200,8320, 
   8208,8224, 8216,12320, 8232,10368, 8240,8736, 8248,12832, 8256,8448, 
   8264,8384, 8272,9248, 8280,13344, 8288,9232, 8296,10432, 8304,9760, 
   8312,13856, 8328,12416, 8336,10272, 8344,14368, 8352,12296, 8360,14464, 
   8368,10784, 8376,14880, 8384,8456, 8392,12480, 8400,11296, 8408,15392, 
   8416,12552, 8424,14528, 8432,11808, 8440,15904, 8448,9216, 8456,8576, 
   8464,9232, 8472,12384, 8480,9248, 8488,10624, 8496,8800, 8504,12896, 
   8512,9472, 8520,8640, 8528,9312, 8536,13408, 8544,9296, 8552,10688, 
   8560,9824, 8568,13920, 8576,9224, 8584,12672, 8592,10336, 8600,14432, 
   8608,13320, 8616,14720, 8624,10848, 8632,14944, 8640,9480, 8648,12736, 
   8656,11360, 8664,15456, 8672,13576, 8680,14784, 8688,11872, 8696,15968, 
   8704,12288, 8712,12416, 8720,12296, 8728,12448, 8736,12304, 8744,10376, 
   8752,8864, 8760,12960, 8768,12352, 8776,12480, 8784,9376, 8792,13472, 
   8800,12368, 8808,10440, 8816,9888, 8824,13984, 8832,12320, 8840,12424, 
   8848,10400, 8856,14496, 8864,12312, 8872,14472, 8880,10912, 8888,15008, 
   8896,12384, 8904,12488, 8912,11424, 8920,15520, 8928,12568, 8936,14536, 
   8944,11936, 8952,16032, 8960,12544, 8968,12672, 8976,12552, 8984,12512, 
   8992,12560, 9000,10632, 9008,12568, 9016,13024, 9024,12608, 9032,12736, 
   9040,9440, 9048,13536, 9056,12624, 9064,10696, 9072,9952, 9080,14048, 
   9088,9240, 9096,12680, 9104,10464, 9112,14560, 9120,13336, 9128,14728, 
   9136,10976, 9144,15072, 9152,9496, 9160,12744, 9168,11488, 9176,15584, 
   9184,13592, 9192,14792, 9200,12000, 9208,16096, 9224,9344, 9232,9248, 
   9240,12576, 9256,11392, 9264,12560, 9272,13088, 9280,9472, 9288,9408, 
   9296,9504, 9304,13600, 9312,9488, 9320,11456, 9328,10016, 9336,14112, 
   9352,13440, 9360,10528, 9368,14624, 9376,12360, 9384,15488, 9392,11040, 
   9400,15136, 9408,9480, 9416,13504, 9424,11552, 9432,15648, 9440,12616, 
   9448,15552, 9456,12064, 9464,16160, 9480,9600, 9488,9504, 9496,12640, 
   9512,11648, 9520,12624, 9528,13152, 9544,9664, 9552,9568, 9560,13664, 
   9576,11712, 9584,10080, 9592,14176, 9608,13696, 9616,10592, 9624,14688, 
   9632,13384, 9640,15744, 9648,11104, 9656,15200, 9672,13760, 9680,11616, 
   9688,15712, 9696,13640, 9704,15808, 9712,12128, 9720,16224, 9728,13312, 
   9736,13440, 9744,13320, 9752,12704, 9760,13328, 9768,11400, 9776,13336, 
   9784,13216, 9792,13376, 9800,13504, 9808,13384, 9816,13728, 9824,13392, 
   9832,11464, 9840,10144, 9848,14240, 9856,13344, 9864,13448, 9872,10656, 
   9880,14752, 9888,12376, 9896,15496, 9904,11168, 9912,15264, 9920,13408, 
   9928,13512, 9936,11680, 9944,15776, 9952,12632, 9960,15560, 9968,12192, 
   9976,16288, 9984,13568, 9992,13696, 10000,13576, 10008,12768, 10016,13584, 
   10024,11656, 10032,13592, 10040,13280, 10048,13632, 10056,13760, 
   10064,13640, 10072,13792, 10080,13648, 10088,11720, 10096,10208, 
   10104,14304, 10112,13600, 10120,13704, 10128,10720, 10136,14816, 
   10144,13400, 10152,15752, 10160,11232, 10168,15328, 10176,13664, 
   10184,13768, 10192,11744, 10200,15840, 10208,13656, 10216,15816, 
   10224,12256, 10232,16352, 10248,10272, 10256,10368, 10264,12328, 
   10280,10384, 10288,10376, 10296,12840, 10304,11264, 10312,11296, 
   10320,11392, 10328,13352, 10336,11272, 10344,10448, 10352,11400, 
   10360,13864, 10376,12432, 10392,14376, 10400,12328, 10408,14480, 
   10416,10792, 10424,14888, 10432,11280, 10440,12496, 10448,11304, 
   10456,15400, 10464,11288, 10472,14544, 10480,11816, 10488,15912, 
   10496,11264, 10504,11272, 10512,11280, 10520,12392, 10528,11296, 
   10536,10640, 10544,12496, 10552,12904, 10560,11328, 10568,11360, 
   10576,11456, 10584,13416, 10592,11336, 10600,10704, 10608,11464, 
   10616,13928, 10624,11392, 10632,12688, 10640,11304, 10648,14440, 
   10656,13352, 10664,14736, 10672,10856, 10680,14952, 10688,11344, 
   10696,12752, 10704,11368, 10712,15464, 10720,11352, 10728,14800, 
   10736,11880, 10744,15976, 10752,14336, 10760,14368, 10768,14464, 
   10776,12456, 10784,14344, 10792,14376, 10800,14472, 10808,12968, 
   10816,15360, 10824,15392, 10832,15488, 10840,13480, 10848,15368, 
   10856,15400, 10864,15496, 10872,13992, 10880,14352, 10888,12440, 
   10896,14480, 10904,14504, 10912,14360, 10920,14488, 10928,14488, 
   10936,15016, 10944,15376, 10952,12504, 10960,11432, 10968,15528, 
   10976,15384, 10984,14552, 10992,11944, 11000,16040, 11008,14400, 
   11016,14432, 11024,14528, 11032,12520, 11040,14408, 11048,14440, 
   11056,14536, 11064,13032, 11072,15424, 11080,15456, 11088,15552, 
   11096,13544, 11104,15432, 11112,15464, 11120,15560, 11128,14056, 
   11136,14416, 11144,12696, 11152,14544, 11160,14568, 11168,14424, 
   11176,14744, 11184,14552, 11192,15080, 11200,15440, 11208,12760, 
   11216,11496, 11224,15592, 11232,15448, 11240,14808, 11248,12008, 
   11256,16104, 11272,11296, 11280,11392, 11288,12584, 11304,11408, 
   11312,12688, 11320,13096, 11328,11520, 11336,11552, 11344,11648, 
   11352,13608, 11360,11528, 11368,11472, 11376,11656, 11384,14120, 
   11400,13456, 11416,14632, 11424,12392, 11432,15504, 11440,14440, 
   11448,15144, 11456,11536, 11464,13520, 11472,11560, 11480,15656, 
   11488,11544, 11496,15568, 11504,12072, 11512,16168, 11528,11552, 
   11536,11648, 11544,12648, 11560,11664, 11568,12752, 11576,13160, 
   11592,11616, 11600,11712, 11608,13672, 11624,11728, 11632,11720, 
   11640,14184, 11656,13712, 11672,14696, 11680,13416, 11688,15760, 
   11696,15464, 11704,15208, 11720,13776, 11736,15720, 11744,13672, 
   11752,15824, 11760,12136, 11768,16232, 11776,14592, 11784,14624, 
   11792,14720, 11800,12712, 11808,14600, 11816,14632, 11824,14728, 
   11832,13224, 11840,15616, 11848,15648, 11856,15744, 11864,13736, 
   11872,15624, 11880,15656, 11888,15752, 11896,14248, 11904,14608, 
   11912,13464, 11920,14736, 11928,14760, 11936,14616, 11944,15512, 
   11952,14744, 11960,15272, 11968,15632, 11976,13528, 11984,15760, 
   11992,15784, 12000,15640, 12008,15576, 12016,12200, 12024,16296, 
   12032,14656, 12040,14688, 12048,14784, 12056,12776, 12064,14664, 
   12072,14696, 12080,14792, 12088,13288, 12096,15680, 12104,15712, 
   12112,15808, 12120,13800, 12128,15688, 12136,15720, 12144,15816, 
   12152,14312, 12160,14672, 12168,13720, 12176,14800, 12184,14824, 
   12192,14680, 12200,15768, 12208,14808, 12216,15336, 12224,15696, 
   12232,13784, 12240,15824, 12248,15848, 12256,15704, 12264,15832, 
   12272,15832, 12280,16360, 12312,12336, 12344,12848, 12352,12544, 
   12360,12552, 12368,12560, 12376,13360, 12384,12576, 12392,12584, 
   12400,13336, 12408,13872, 12424,12448, 12440,14384, 12456,14496, 
   12464,14472, 12472,14896, 12480,12672, 12488,12512, 12496,12688, 
   12504,15408, 12512,12680, 12520,14560, 12528,14728, 12536,15920, 
   12544,13312, 12552,13320, 12560,13328, 12568,13336, 12576,13344, 
   12584,13352, 12592,13360, 12600,12912, 12608,13568, 12616,13576, 
   12624,13584, 12632,13424, 12640,13600, 12648,13608, 12656,13400, 
   12664,13936, 12672,13440, 12680,12704, 12688,13456, 12696,14448, 
   12704,13448, 12712,14752, 12720,15496, 12728,14960, 12736,13696, 
   12744,12768, 12752,13712, 12760,15472, 12768,13704, 12776,14816, 
   12784,15752, 12792,15984, 12800,14336, 12808,14464, 12816,14344, 
   12824,14472, 12832,14352, 12840,14480, 12848,14360, 12856,12976, 
   12864,14400, 12872,14528, 12880,14408, 12888,13488, 12896,14416, 
   12904,14544, 12912,14424, 12920,14000, 12928,14368, 12936,14496, 
   12944,14376, 12952,14512, 12960,14384, 12968,14504, 12976,14488, 
   12984,15024, 12992,14432, 13000,14560, 13008,14440, 13016,15536, 
   13024,14448, 13032,14568, 13040,14744, 13048,16048, 13056,14592, 
   13064,14720, 13072,14600, 13080,14728, 13088,14608, 13096,14736, 
   13104,14616, 13112,14744, 13120,14656, 13128,14784, 13136,14664, 
   13144,13552, 13152,14672, 13160,14800, 13168,14680, 13176,14064, 
   13184,14624, 13192,14752, 13200,14632, 13208,14576, 13216,13464, 
   13224,14760, 13232,15512, 13240,15088, 13248,14688, 13256,14816, 
   13264,14696, 13272,15600, 13280,13720, 13288,14824, 13296,15768, 
   13304,16112, 13336,13360, 13368,14616, 13376,13568, 13384,13576, 
   13392,13584, 13400,13616, 13408,13600, 13416,13608, 13424,13592, 
   13432,14128, 13448,13472, 13464,14640, 13480,15520, 13488,14536, 
   13496,15152, 13504,13696, 13512,13536, 13520,13712, 13528,15664, 
   13536,13704, 13544,15584, 13552,14792, 13560,16176, 13592,13616, 
   13624,14680, 13656,13680, 13688,14192, 13704,13728, 13720,14704, 
   13736,15776, 13744,15560, 13752,15216, 13768,13792, 13784,15728, 
   13800,15840, 13808,15816, 13816,16240, 13824,15360, 13832,15488, 
   13840,15368, 13848,15496, 13856,15376, 13864,15504, 13872,15384, 
   13880,15512, 13888,15424, 13896,15552, 13904,15432, 13912,15560, 
   13920,15440, 13928,15568, 13936,15448, 13944,14256, 13952,15392, 
   13960,15520, 13968,15400, 13976,14768, 13984,15408, 13992,15528, 
   14000,14552, 14008,15280, 14016,15456, 14024,15584, 14032,15464, 
   14040,15792, 14048,15472, 14056,15592, 14064,14808, 14072,16304, 
   14080,15616, 14088,15744, 14096,15624, 14104,15752, 14112,15632, 
   14120,15760, 14128,15640, 14136,15768, 14144,15680, 14152,15808, 
   14160,15688, 14168,15816, 14176,15696, 14184,15824, 14192,15704, 
   14200,14320, 14208,15648, 14216,15776, 14224,15656, 14232,14832, 
   14240,15664, 14248,15784, 14256,15576, 14264,15344, 14272,15712, 
   14280,15840, 14288,15720, 14296,15856, 14304,15728, 14312,15848, 
   14320,15832, 14328,16368, 14392,14488, 14400,14592, 14408,14600, 
   14416,14608, 14424,14616, 14432,14624, 14440,14632, 14448,14640, 
   14456,15512, 14504,14512, 14520,14904, 14528,14720, 14536,14728, 
   14544,14736, 14552,15416, 14560,14752, 14568,14576, 14584,15928,
   14576,14760, 14592,15360, 14600,15368, 14608,15376, 14616,15384, 
   14624,15392, 14632,15400, 14640,15408, 14648,15416, 14656,15616, 
   14664,15624, 14672,15632, 14680,15640, 14688,15648, 14696,15656, 
   14704,15664, 14712,15576, 14720,15488, 14728,15496, 14736,15504, 
   14744,15512, 14752,15520, 14760,14768, 14776,14968, 14768,15528, 
   14784,15744, 14792,15752, 14800,15760, 14808,15480, 14816,15776, 
   14824,14832, 14840,15992, 14832,15784, 14856,14864, 14864,14880, 
   14872,14896, 14880,14976, 14888,14992, 14896,15008, 14904,15024, 
   14912,15104, 14920,15120, 14928,15136, 14936,15152, 14944,15232, 
   14952,15248, 14960,15264, 14968,15280, 14984,15008, 15000,15024, 
   15016,15024, 15040,15112, 15048,15128, 15056,15144, 15064,15544, 
   15072,15240, 15080,15256, 15088,15272, 15096,16056, 15104,15872, 
   15112,15888, 15120,15904, 15128,15920, 15136,16000, 15144,16016, 
   15152,16032, 15160,16048, 15168,16128, 15176,16144, 15184,16160, 
   15192,16176, 15200,16256, 15208,16272, 15216,16288, 15224,16304, 
   15232,15880, 15240,15896, 15248,15912, 15256,15928, 15264,16008, 
   15272,16024, 15280,16040, 15288,16056, 15296,16136, 15304,16152, 
   15312,16168, 15320,15608, 15328,16264, 15336,16280, 15344,16296, 
   15352,16120, 15416,15512, 15424,15616, 15432,15624, 15440,15632, 
   15448,15640, 15456,15648, 15464,15656, 15472,15664, 15480,15768, 
   15528,15536, 15544,16048, 15552,15744, 15560,15752, 15568,15760, 
   15576,15672, 15584,15776, 15592,15600, 15600,15784, 15608,16184, 
   15672,15768, 15736,15832, 15784,15792, 15800,16304, 15848,15856, 
   15880,16000, 15864,16248, 15888,16000, 15896,16008, 15904,16000, 
   15912,16016, 15920,16008, 15928,16024, 15936,16128, 15944,16160, 
   15952,16256, 15960,16288, 15968,16136, 15976,16168, 15984,16264, 
   15992,16296, 16008,16032, 16024,16040, 16064,16144, 16040,16048, 
   16072,16176, 16080,16272, 16088,16304, 16096,16152, 16104,16184, 
   16112,16280, 16136,16256, 16120,16312, 16144,16256, 16152,16264, 
   16160,16256, 16168,16272, 16176,16264, 16184,16280, 16200,16208, 
   16208,16224, 16216,16240, 16224,16320, 16232,16336, 16240,16352, 
   16248,16368, 16264,16288, 16280,16296, 16296,16304, 16344,16368,
   16328,16352, 16360,16368
};

const uint16_t armBitRevIndexTable4096[ARMBITREVINDEXTABLE4096_TABLE_LENGTH] = 
{
   //radix 8, size 4032
   8,4096, 16,8192, 24,12288, 32,16384, 40,20480, 48,24576, 56,28672, 64,512, 
   72,4608, 80,8704, 88,12800, 96,16896, 104,20992, 112,25088, 120,29184, 
   128,1024, 136,5120, 144,9216, 152,13312, 160,17408, 168,21504, 176,25600, 
   184,29696, 192,1536, 200,5632, 208,9728, 216,13824, 224,17920, 232,22016, 
   240,26112, 248,30208, 256,2048, 264,6144, 272,10240, 280,14336, 288,18432, 
   296,22528, 304,26624, 312,30720, 320,2560, 328,6656, 336,10752, 344,14848, 
   352,18944, 360,23040, 368,27136, 376,31232, 384,3072, 392,7168, 400,11264, 
   408,15360, 416,19456, 424,23552, 432,27648, 440,31744, 448,3584, 456,7680, 
   464,11776, 472,15872, 480,19968, 488,24064, 496,28160, 504,32256, 520,4160, 
   528,8256, 536,12352, 544,16448, 552,20544, 560,24640, 568,28736, 584,4672, 
   592,8768, 600,12864, 608,16960, 616,21056, 624,25152, 632,29248, 640,1088, 
   648,5184, 656,9280, 664,13376, 672,17472, 680,21568, 688,25664, 696,29760, 
   704,1600, 712,5696, 720,9792, 728,13888, 736,17984, 744,22080, 752,26176, 
   760,30272, 768,2112, 776,6208, 784,10304, 792,14400, 800,18496, 808,22592, 
   816,26688, 824,30784, 832,2624, 840,6720, 848,10816, 856,14912, 864,19008, 
   872,23104, 880,27200, 888,31296, 896,3136, 904,7232, 912,11328, 920,15424, 
   928,19520, 936,23616, 944,27712, 952,31808, 960,3648, 968,7744, 976,11840, 
   984,15936, 992,20032, 1000,24128, 1008,28224, 1016,32320, 1032,4224, 
   1040,8320, 1048,12416, 1056,16512, 1064,20608, 1072,24704, 1080,28800, 
   1096,4736, 1104,8832, 1112,12928, 1120,17024, 1128,21120, 1136,25216, 
   1144,29312, 1160,5248, 1168,9344, 1176,13440, 1184,17536, 1192,21632, 
   1200,25728, 1208,29824, 1216,1664, 1224,5760, 1232,9856, 1240,13952, 
   1248,18048, 1256,22144, 1264,26240, 1272,30336, 1280,2176, 1288,6272, 
   1296,10368, 1304,14464, 1312,18560, 1320,22656, 1328,26752, 1336,30848, 
   1344,2688, 1352,6784, 1360,10880, 1368,14976, 1376,19072, 1384,23168, 
   1392,27264, 1400,31360, 1408,3200, 1416,7296, 1424,11392, 1432,15488, 
   1440,19584, 1448,23680, 1456,27776, 1464,31872, 1472,3712, 1480,7808, 
   1488,11904, 1496,16000, 1504,20096, 1512,24192, 1520,28288, 1528,32384, 
   1544,4288, 1552,8384, 1560,12480, 1568,16576, 1576,20672, 1584,24768, 
   1592,28864, 1608,4800, 1616,8896, 1624,12992, 1632,17088, 1640,21184, 
   1648,25280, 1656,29376, 1672,5312, 1680,9408, 1688,13504, 1696,17600, 
   1704,21696, 1712,25792, 1720,29888, 1736,5824, 1744,9920, 1752,14016, 
   1760,18112, 1768,22208, 1776,26304, 1784,30400, 1792,2240, 1800,6336, 
   1808,10432, 1816,14528, 1824,18624, 1832,22720, 1840,26816, 1848,30912, 
   1856,2752, 1864,6848, 1872,10944, 1880,15040, 1888,19136, 1896,23232, 
   1904,27328, 1912,31424, 1920,3264, 1928,7360, 1936,11456, 1944,15552, 
   1952,19648, 1960,23744, 1968,27840, 1976,31936, 1984,3776, 1992,7872, 
   2000,11968, 2008,16064, 2016,20160, 2024,24256, 2032,28352, 2040,32448, 
   2056,4352, 2064,8448, 2072,12544, 2080,16640, 2088,20736, 2096,24832, 
   2104,28928, 2120,4864, 2128,8960, 2136,13056, 2144,17152, 2152,21248, 
   2160,25344, 2168,29440, 2184,5376, 2192,9472, 2200,13568, 2208,17664, 
   2216,21760, 2224,25856, 2232,29952, 2248,5888, 2256,9984, 2264,14080, 
   2272,18176, 2280,22272, 2288,26368, 2296,30464, 2312,6400, 2320,10496, 
   2328,14592, 2336,18688, 2344,22784, 2352,26880, 2360,30976, 2368,2816, 
   2376,6912, 2384,11008, 2392,15104, 2400,19200, 2408,23296, 2416,27392, 
   2424,31488, 2432,3328, 2440,7424, 2448,11520, 2456,15616, 2464,19712, 
   2472,23808, 2480,27904, 2488,32000, 2496,3840, 2504,7936, 2512,12032, 
   2520,16128, 2528,20224, 2536,24320, 2544,28416, 2552,32512, 2568,4416, 
   2576,8512, 2584,12608, 2592,16704, 2600,20800, 2608,24896, 2616,28992, 
   2632,4928, 2640,9024, 2648,13120, 2656,17216, 2664,21312, 2672,25408, 
   2680,29504, 2696,5440, 2704,9536, 2712,13632, 2720,17728, 2728,21824, 
   2736,25920, 2744,30016, 2760,5952, 2768,10048, 2776,14144, 2784,18240, 
   2792,22336, 2800,26432, 2808,30528, 2824,6464, 2832,10560, 2840,14656, 
   2848,18752, 2856,22848, 2864,26944, 2872,31040, 2888,6976, 2896,11072, 
   2904,15168, 2912,19264, 2920,23360, 2928,27456, 2936,31552, 2944,3392, 
   2952,7488, 2960,11584, 2968,15680, 2976,19776, 2984,23872, 2992,27968, 
   3000,32064, 3008,3904, 3016,8000, 3024,12096, 3032,16192, 3040,20288, 
   3048,24384, 3056,28480, 3064,32576, 3080,4480, 3088,8576, 3096,12672, 
   3104,16768, 3112,20864, 3120,24960, 3128,29056, 3144,4992, 3152,9088, 
   3160,13184, 3168,17280, 3176,21376, 3184,25472, 3192,29568, 3208,5504, 
   3216,9600, 3224,13696, 3232,17792, 3240,21888, 3248,25984, 3256,30080, 
   3272,6016, 3280,10112, 3288,14208, 3296,18304, 3304,22400, 3312,26496, 
   3320,30592, 3336,6528, 3344,10624, 3352,14720, 3360,18816, 3368,22912, 
   3376,27008, 3384,31104, 3400,7040, 3408,11136, 3416,15232, 3424,19328, 
   3432,23424, 3440,27520, 3448,31616, 3464,7552, 3472,11648, 3480,15744, 
   3488,19840, 3496,23936, 3504,28032, 3512,32128, 3520,3968, 3528,8064, 
   3536,12160, 3544,16256, 3552,20352, 3560,24448, 3568,28544, 3576,32640, 
   3592,4544, 3600,8640, 3608,12736, 3616,16832, 3624,20928, 3632,25024, 
   3640,29120, 3656,5056, 3664,9152, 3672,13248, 3680,17344, 3688,21440, 
   3696,25536, 3704,29632, 3720,5568, 3728,9664, 3736,13760, 3744,17856, 
   3752,21952, 3760,26048, 3768,30144, 3784,6080, 3792,10176, 3800,14272, 
   3808,18368, 3816,22464, 3824,26560, 3832,30656, 3848,6592, 3856,10688, 
   3864,14784, 3872,18880, 3880,22976, 3888,27072, 3896,31168, 3912,7104, 
   3920,11200, 3928,15296, 3936,19392, 3944,23488, 3952,27584, 3960,31680, 
   3976,7616, 3984,11712, 3992,15808, 4000,19904, 4008,24000, 4016,28096, 
   4024,32192, 4040,8128, 4048,12224, 4056,16320, 4064,20416, 4072,24512, 
   4080,28608, 4088,32704, 4112,8200, 4120,12296, 4128,16392, 4136,20488, 
   4144,24584, 4152,28680, 4168,4616, 4176,8712, 4184,12808, 4192,16904, 
   4200,21000, 4208,25096, 4216,29192, 4232,5128, 4240,9224, 4248,13320, 
   4256,17416, 4264,21512, 4272,25608, 4280,29704, 4296,5640, 4304,9736, 
   4312,13832, 4320,17928, 4328,22024, 4336,26120, 4344,30216, 4360,6152, 
   4368,10248, 4376,14344, 4384,18440, 4392,22536, 4400,26632, 4408,30728, 
   4424,6664, 4432,10760, 4440,14856, 4448,18952, 4456,23048, 4464,27144, 
   4472,31240, 4488,7176, 4496,11272, 4504,15368, 4512,19464, 4520,23560, 
   4528,27656, 4536,31752, 4552,7688, 4560,11784, 4568,15880, 4576,19976, 
   4584,24072, 4592,28168, 4600,32264, 4624,8264, 4632,12360, 4640,16456, 
   4648,20552, 4656,24648, 4664,28744, 4688,8776, 4696,12872, 4704,16968, 
   4712,21064, 4720,25160, 4728,29256, 4744,5192, 4752,9288, 4760,13384, 
   4768,17480, 4776,21576, 4784,25672, 4792,29768, 4808,5704, 4816,9800, 
   4824,13896, 4832,17992, 4840,22088, 4848,26184, 4856,30280, 4872,6216, 
   4880,10312, 4888,14408, 4896,18504, 4904,22600, 4912,26696, 4920,30792, 
   4936,6728, 4944,10824, 4952,14920, 4960,19016, 4968,23112, 4976,27208, 
   4984,31304, 5000,7240, 5008,11336, 5016,15432, 5024,19528, 5032,23624, 
   5040,27720, 5048,31816, 5064,7752, 5072,11848, 5080,15944, 5088,20040, 
   5096,24136, 5104,28232, 5112,32328, 5136,8328, 5144,12424, 5152,16520, 
   5160,20616, 5168,24712, 5176,28808, 5200,8840, 5208,12936, 5216,17032, 
   5224,21128, 5232,25224, 5240,29320, 5264,9352, 5272,13448, 5280,17544, 
   5288,21640, 5296,25736, 5304,29832, 5320,5768, 5328,9864, 5336,13960, 
   5344,18056, 5352,22152, 5360,26248, 5368,30344, 5384,6280, 5392,10376, 
   5400,14472, 5408,18568, 5416,22664, 5424,26760, 5432,30856, 5448,6792, 
   5456,10888, 5464,14984, 5472,19080, 5480,23176, 5488,27272, 5496,31368, 
   5512,7304, 5520,11400, 5528,15496, 5536,19592, 5544,23688, 5552,27784, 
   5560,31880, 5576,7816, 5584,11912, 5592,16008, 5600,20104, 5608,24200, 
   5616,28296, 5624,32392, 5648,8392, 5656,12488, 5664,16584, 5672,20680, 
   5680,24776, 5688,28872, 5712,8904, 5720,13000, 5728,17096, 5736,21192, 
   5744,25288, 5752,29384, 5776,9416, 5784,13512, 5792,17608, 5800,21704, 
   5808,25800, 5816,29896, 5840,9928, 5848,14024, 5856,18120, 5864,22216, 
   5872,26312, 5880,30408, 5896,6344, 5904,10440, 5912,14536, 5920,18632, 
   5928,22728, 5936,26824, 5944,30920, 5960,6856, 5968,10952, 5976,15048, 
   5984,19144, 5992,23240, 6000,27336, 6008,31432, 6024,7368, 6032,11464, 
   6040,15560, 6048,19656, 6056,23752, 6064,27848, 6072,31944, 6088,7880, 
   6096,11976, 6104,16072, 6112,20168, 6120,24264, 6128,28360, 6136,32456, 
   6160,8456, 6168,12552, 6176,16648, 6184,20744, 6192,24840, 6200,28936, 
   6224,8968, 6232,13064, 6240,17160, 6248,21256, 6256,25352, 6264,29448, 
   6288,9480, 6296,13576, 6304,17672, 6312,21768, 6320,25864, 6328,29960, 
   6352,9992, 6360,14088, 6368,18184, 6376,22280, 6384,26376, 6392,30472, 
   6416,10504, 6424,14600, 6432,18696, 6440,22792, 6448,26888, 6456,30984, 
   6472,6920, 6480,11016, 6488,15112, 6496,19208, 6504,23304, 6512,27400, 
   6520,31496, 6536,7432, 6544,11528, 6552,15624, 6560,19720, 6568,23816, 
   6576,27912, 6584,32008, 6600,7944, 6608,12040, 6616,16136, 6624,20232, 
   6632,24328, 6640,28424, 6648,32520, 6672,8520, 6680,12616, 6688,16712, 
   6696,20808, 6704,24904, 6712,29000, 6736,9032, 6744,13128, 6752,17224, 
   6760,21320, 6768,25416, 6776,29512, 6800,9544, 6808,13640, 6816,17736, 
   6824,21832, 6832,25928, 6840,30024, 6864,10056, 6872,14152, 6880,18248, 
   6888,22344, 6896,26440, 6904,30536, 6928,10568, 6936,14664, 6944,18760, 
   6952,22856, 6960,26952, 6968,31048, 6992,11080, 7000,15176, 7008,19272, 
   7016,23368, 7024,27464, 7032,31560, 7048,7496, 7056,11592, 7064,15688, 
   7072,19784, 7080,23880, 7088,27976, 7096,32072, 7112,8008, 7120,12104, 
   7128,16200, 7136,20296, 7144,24392, 7152,28488, 7160,32584, 7184,8584, 
   7192,12680, 7200,16776, 7208,20872, 7216,24968, 7224,29064, 7248,9096, 
   7256,13192, 7264,17288, 7272,21384, 7280,25480, 7288,29576, 7312,9608, 
   7320,13704, 7328,17800, 7336,21896, 7344,25992, 7352,30088, 7376,10120, 
   7384,14216, 7392,18312, 7400,22408, 7408,26504, 7416,30600, 7440,10632, 
   7448,14728, 7456,18824, 7464,22920, 7472,27016, 7480,31112, 7504,11144, 
   7512,15240, 7520,19336, 7528,23432, 7536,27528, 7544,31624, 7568,11656, 
   7576,15752, 7584,19848, 7592,23944, 7600,28040, 7608,32136, 7624,8072, 
   7632,12168, 7640,16264, 7648,20360, 7656,24456, 7664,28552, 7672,32648, 
   7696,8648, 7704,12744, 7712,16840, 7720,20936, 7728,25032, 7736,29128, 
   7760,9160, 7768,13256, 7776,17352, 7784,21448, 7792,25544, 7800,29640, 
   7824,9672, 7832,13768, 7840,17864, 7848,21960, 7856,26056, 7864,30152, 
   7888,10184, 7896,14280, 7904,18376, 7912,22472, 7920,26568, 7928,30664, 
   7952,10696, 7960,14792, 7968,18888, 7976,22984, 7984,27080, 7992,31176, 
   8016,11208, 8024,15304, 8032,19400, 8040,23496, 8048,27592, 8056,31688, 
   8080,11720, 8088,15816, 8096,19912, 8104,24008, 8112,28104, 8120,32200, 
   8144,12232, 8152,16328, 8160,20424, 8168,24520, 8176,28616, 8184,32712, 
   8216,12304, 8224,16400, 8232,20496, 8240,24592, 8248,28688, 8272,8720, 
   8280,12816, 8288,16912, 8296,21008, 8304,25104, 8312,29200, 8336,9232, 
   8344,13328, 8352,17424, 8360,21520, 8368,25616, 8376,29712, 8400,9744, 
   8408,13840, 8416,17936, 8424,22032, 8432,26128, 8440,30224, 8464,10256, 
   8472,14352, 8480,18448, 8488,22544, 8496,26640, 8504,30736, 8528,10768, 
   8536,14864, 8544,18960, 8552,23056, 8560,27152, 8568,31248, 8592,11280, 
   8600,15376, 8608,19472, 8616,23568, 8624,27664, 8632,31760, 8656,11792, 
   8664,15888, 8672,19984, 8680,24080, 8688,28176, 8696,32272, 8728,12368, 
   8736,16464, 8744,20560, 8752,24656, 8760,28752, 8792,12880, 8800,16976, 
   8808,21072, 8816,25168, 8824,29264, 8848,9296, 8856,13392, 8864,17488, 
   8872,21584, 8880,25680, 8888,29776, 8912,9808, 8920,13904, 8928,18000, 
   8936,22096, 8944,26192, 8952,30288, 8976,10320, 8984,14416, 8992,18512, 
   9000,22608, 9008,26704, 9016,30800, 9040,10832, 9048,14928, 9056,19024, 
   9064,23120, 9072,27216, 9080,31312, 9104,11344, 9112,15440, 9120,19536, 
   9128,23632, 9136,27728, 9144,31824, 9168,11856, 9176,15952, 9184,20048, 
   9192,24144, 9200,28240, 9208,32336, 9240,12432, 9248,16528, 9256,20624, 
   9264,24720, 9272,28816, 9304,12944, 9312,17040, 9320,21136, 9328,25232, 
   9336,29328, 9368,13456, 9376,17552, 9384,21648, 9392,25744, 9400,29840, 
   9424,9872, 9432,13968, 9440,18064, 9448,22160, 9456,26256, 9464,30352, 
   9488,10384, 9496,14480, 9504,18576, 9512,22672, 9520,26768, 9528,30864, 
   9552,10896, 9560,14992, 9568,19088, 9576,23184, 9584,27280, 9592,31376, 
   9616,11408, 9624,15504, 9632,19600, 9640,23696, 9648,27792, 9656,31888, 
   9680,11920, 9688,16016, 9696,20112, 9704,24208, 9712,28304, 9720,32400, 
   9752,12496, 9760,16592, 9768,20688, 9776,24784, 9784,28880, 9816,13008, 
   9824,17104, 9832,21200, 9840,25296, 9848,29392, 9880,13520, 9888,17616, 
   9896,21712, 9904,25808, 9912,29904, 9944,14032, 9952,18128, 9960,22224, 
   9968,26320, 9976,30416, 10000,10448, 10008,14544, 10016,18640, 10024,22736, 
   10032,26832, 10040,30928, 10064,10960, 10072,15056, 10080,19152, 
   10088,23248, 10096,27344, 10104,31440, 10128,11472, 10136,15568, 
   10144,19664, 10152,23760, 10160,27856, 10168,31952, 10192,11984, 
   10200,16080, 10208,20176, 10216,24272, 10224,28368, 10232,32464, 
   10264,12560, 10272,16656, 10280,20752, 10288,24848, 10296,28944, 
   10328,13072, 10336,17168, 10344,21264, 10352,25360, 10360,29456, 
   10392,13584, 10400,17680, 10408,21776, 10416,25872, 10424,29968, 
   10456,14096, 10464,18192, 10472,22288, 10480,26384, 10488,30480, 
   10520,14608, 10528,18704, 10536,22800, 10544,26896, 10552,30992, 
   10576,11024, 10584,15120, 10592,19216, 10600,23312, 10608,27408, 
   10616,31504, 10640,11536, 10648,15632, 10656,19728, 10664,23824, 
   10672,27920, 10680,32016, 10704,12048, 10712,16144, 10720,20240, 
   10728,24336, 10736,28432, 10744,32528, 10776,12624, 10784,16720, 
   10792,20816, 10800,24912, 10808,29008, 10840,13136, 10848,17232, 
   10856,21328, 10864,25424, 10872,29520, 10904,13648, 10912,17744, 
   10920,21840, 10928,25936, 10936,30032, 10968,14160, 10976,18256, 
   10984,22352, 10992,26448, 11000,30544, 11032,14672, 11040,18768, 
   11048,22864, 11056,26960, 11064,31056, 11096,15184, 11104,19280, 
   11112,23376, 11120,27472, 11128,31568, 11152,11600, 11160,15696, 
   11168,19792, 11176,23888, 11184,27984, 11192,32080, 11216,12112, 
   11224,16208, 11232,20304, 11240,24400, 11248,28496, 11256,32592, 
   11288,12688, 11296,16784, 11304,20880, 11312,24976, 11320,29072, 
   11352,13200, 11360,17296, 11368,21392, 11376,25488, 11384,29584, 
   11416,13712, 11424,17808, 11432,21904, 11440,26000, 11448,30096, 
   11480,14224, 11488,18320, 11496,22416, 11504,26512, 11512,30608, 
   11544,14736, 11552,18832, 11560,22928, 11568,27024, 11576,31120, 
   11608,15248, 11616,19344, 11624,23440, 11632,27536, 11640,31632, 
   11672,15760, 11680,19856, 11688,23952, 11696,28048, 11704,32144, 
   11728,12176, 11736,16272, 11744,20368, 11752,24464, 11760,28560, 
   11768,32656, 11800,12752, 11808,16848, 11816,20944, 11824,25040, 
   11832,29136, 11864,13264, 11872,17360, 11880,21456, 11888,25552, 
   11896,29648, 11928,13776, 11936,17872, 11944,21968, 11952,26064, 
   11960,30160, 11992,14288, 12000,18384, 12008,22480, 12016,26576, 
   12024,30672, 12056,14800, 12064,18896, 12072,22992, 12080,27088, 
   12088,31184, 12120,15312, 12128,19408, 12136,23504, 12144,27600, 
   12152,31696, 12184,15824, 12192,19920, 12200,24016, 12208,28112, 
   12216,32208, 12248,16336, 12256,20432, 12264,24528, 12272,28624, 
   12280,32720, 12320,16408, 12328,20504, 12336,24600, 12344,28696, 
   12376,12824, 12384,16920, 12392,21016, 12400,25112, 12408,29208, 
   12440,13336, 12448,17432, 12456,21528, 12464,25624, 12472,29720, 
   12504,13848, 12512,17944, 12520,22040, 12528,26136, 12536,30232, 
   12568,14360, 12576,18456, 12584,22552, 12592,26648, 12600,30744, 
   12632,14872, 12640,18968, 12648,23064, 12656,27160, 12664,31256, 
   12696,15384, 12704,19480, 12712,23576, 12720,27672, 12728,31768, 
   12760,15896, 12768,19992, 12776,24088, 12784,28184, 12792,32280, 
   12832,16472, 12840,20568, 12848,24664, 12856,28760, 12896,16984, 
   12904,21080, 12912,25176, 12920,29272, 12952,13400, 12960,17496, 
   12968,21592, 12976,25688, 12984,29784, 13016,13912, 13024,18008, 
   13032,22104, 13040,26200, 13048,30296, 13080,14424, 13088,18520, 
   13096,22616, 13104,26712, 13112,30808, 13144,14936, 13152,19032, 
   13160,23128, 13168,27224, 13176,31320, 13208,15448, 13216,19544, 
   13224,23640, 13232,27736, 13240,31832, 13272,15960, 13280,20056, 
   13288,24152, 13296,28248, 13304,32344, 13344,16536, 13352,20632, 
   13360,24728, 13368,28824, 13408,17048, 13416,21144, 13424,25240, 
   13432,29336, 13472,17560, 13480,21656, 13488,25752, 13496,29848, 
   13528,13976, 13536,18072, 13544,22168, 13552,26264, 13560,30360, 
   13592,14488, 13600,18584, 13608,22680, 13616,26776, 13624,30872, 
   13656,15000, 13664,19096, 13672,23192, 13680,27288, 13688,31384, 
   13720,15512, 13728,19608, 13736,23704, 13744,27800, 13752,31896, 
   13784,16024, 13792,20120, 13800,24216, 13808,28312, 13816,32408, 
   13856,16600, 13864,20696, 13872,24792, 13880,28888, 13920,17112, 
   13928,21208, 13936,25304, 13944,29400, 13984,17624, 13992,21720, 
   14000,25816, 14008,29912, 14048,18136, 14056,22232, 14064,26328, 
   14072,30424, 14104,14552, 14112,18648, 14120,22744, 14128,26840, 
   14136,30936, 14168,15064, 14176,19160, 14184,23256, 14192,27352, 
   14200,31448, 14232,15576, 14240,19672, 14248,23768, 14256,27864, 
   14264,31960, 14296,16088, 14304,20184, 14312,24280, 14320,28376, 
   14328,32472, 14368,16664, 14376,20760, 14384,24856, 14392,28952, 
   14432,17176, 14440,21272, 14448,25368, 14456,29464, 14496,17688, 
   14504,21784, 14512,25880, 14520,29976, 14560,18200, 14568,22296, 
   14576,26392, 14584,30488, 14624,18712, 14632,22808, 14640,26904, 
   14648,31000, 14680,15128, 14688,19224, 14696,23320, 14704,27416, 
   14712,31512, 14744,15640, 14752,19736, 14760,23832, 14768,27928, 
   14776,32024, 14808,16152, 14816,20248, 14824,24344, 14832,28440, 
   14840,32536, 14880,16728, 14888,20824, 14896,24920, 14904,29016, 
   14944,17240, 14952,21336, 14960,25432, 14968,29528, 15008,17752, 
   15016,21848, 15024,25944, 15032,30040, 15072,18264, 15080,22360, 
   15088,26456, 15096,30552, 15136,18776, 15144,22872, 15152,26968, 
   15160,31064, 15200,19288, 15208,23384, 15216,27480, 15224,31576, 
   15256,15704, 15264,19800, 15272,23896, 15280,27992, 15288,32088, 
   15320,16216, 15328,20312, 15336,24408, 15344,28504, 15352,32600, 
   15392,16792, 15400,20888, 15408,24984, 15416,29080, 15456,17304, 
   15464,21400, 15472,25496, 15480,29592, 15520,17816, 15528,21912, 
   15536,26008, 15544,30104, 15584,18328, 15592,22424, 15600,26520, 
   15608,30616, 15648,18840, 15656,22936, 15664,27032, 15672,31128, 
   15712,19352, 15720,23448, 15728,27544, 15736,31640, 15776,19864, 
   15784,23960, 15792,28056, 15800,32152, 15832,16280, 15840,20376, 
   15848,24472, 15856,28568, 15864,32664, 15904,16856, 15912,20952, 
   15920,25048, 15928,29144, 15968,17368, 15976,21464, 15984,25560, 
   15992,29656, 16032,17880, 16040,21976, 16048,26072, 16056,30168, 
   16096,18392, 16104,22488, 16112,26584, 16120,30680, 16160,18904, 
   16168,23000, 16176,27096, 16184,31192, 16224,19416, 16232,23512, 
   16240,27608, 16248,31704, 16288,19928, 16296,24024, 16304,28120, 
   16312,32216, 16352,20440, 16360,24536, 16368,28632, 16376,32728, 
   16424,20512, 16432,24608, 16440,28704, 16480,16928, 16488,21024, 
   16496,25120, 16504,29216, 16544,17440, 16552,21536, 16560,25632, 
   16568,29728, 16608,17952, 16616,22048, 16624,26144, 16632,30240, 
   16672,18464, 16680,22560, 16688,26656, 16696,30752, 16736,18976, 
   16744,23072, 16752,27168, 16760,31264, 16800,19488, 16808,23584, 
   16816,27680, 16824,31776, 16864,20000, 16872,24096, 16880,28192, 
   16888,32288, 16936,20576, 16944,24672, 16952,28768, 17000,21088, 
   17008,25184, 17016,29280, 17056,17504, 17064,21600, 17072,25696, 
   17080,29792, 17120,18016, 17128,22112, 17136,26208, 17144,30304, 
   17184,18528, 17192,22624, 17200,26720, 17208,30816, 17248,19040, 
   17256,23136, 17264,27232, 17272,31328, 17312,19552, 17320,23648, 
   17328,27744, 17336,31840, 17376,20064, 17384,24160, 17392,28256, 
   17400,32352, 17448,20640, 17456,24736, 17464,28832, 17512,21152, 
   17520,25248, 17528,29344, 17576,21664, 17584,25760, 17592,29856, 
   17632,18080, 17640,22176, 17648,26272, 17656,30368, 17696,18592, 
   17704,22688, 17712,26784, 17720,30880, 17760,19104, 17768,23200, 
   17776,27296, 17784,31392, 17824,19616, 17832,23712, 17840,27808, 
   17848,31904, 17888,20128, 17896,24224, 17904,28320, 17912,32416, 
   17960,20704, 17968,24800, 17976,28896, 18024,21216, 18032,25312, 
   18040,29408, 18088,21728, 18096,25824, 18104,29920, 18152,22240, 
   18160,26336, 18168,30432, 18208,18656, 18216,22752, 18224,26848, 
   18232,30944, 18272,19168, 18280,23264, 18288,27360, 18296,31456, 
   18336,19680, 18344,23776, 18352,27872, 18360,31968, 18400,20192, 
   18408,24288, 18416,28384, 18424,32480, 18472,20768, 18480,24864, 
   18488,28960, 18536,21280, 18544,25376, 18552,29472, 18600,21792, 
   18608,25888, 18616,29984, 18664,22304, 18672,26400, 18680,30496, 
   18728,22816, 18736,26912, 18744,31008, 18784,19232, 18792,23328, 
   18800,27424, 18808,31520, 18848,19744, 18856,23840, 18864,27936, 
   18872,32032, 18912,20256, 18920,24352, 18928,28448, 18936,32544, 
   18984,20832, 18992,24928, 19000,29024, 19048,21344, 19056,25440, 
   19064,29536, 19112,21856, 19120,25952, 19128,30048, 19176,22368, 
   19184,26464, 19192,30560, 19240,22880, 19248,26976, 19256,31072, 
   19304,23392, 19312,27488, 19320,31584, 19360,19808, 19368,23904, 
   19376,28000, 19384,32096, 19424,20320, 19432,24416, 19440,28512, 
   19448,32608, 19496,20896, 19504,24992, 19512,29088, 19560,21408, 
   19568,25504, 19576,29600, 19624,21920, 19632,26016, 19640,30112, 
   19688,22432, 19696,26528, 19704,30624, 19752,22944, 19760,27040, 
   19768,31136, 19816,23456, 19824,27552, 19832,31648, 19880,23968, 
   19888,28064, 19896,32160, 19936,20384, 19944,24480, 19952,28576, 
   19960,32672, 20008,20960, 20016,25056, 20024,29152, 20072,21472, 
   20080,25568, 20088,29664, 20136,21984, 20144,26080, 20152,30176, 
   20200,22496, 20208,26592, 20216,30688, 20264,23008, 20272,27104, 
   20280,31200, 20328,23520, 20336,27616, 20344,31712, 20392,24032, 
   20400,28128, 20408,32224, 20456,24544, 20464,28640, 20472,32736, 
   20528,24616, 20536,28712, 20584,21032, 20592,25128, 20600,29224, 
   20648,21544, 20656,25640, 20664,29736, 20712,22056, 20720,26152, 
   20728,30248, 20776,22568, 20784,26664, 20792,30760, 20840,23080, 
   20848,27176, 20856,31272, 20904,23592, 20912,27688, 20920,31784, 
   20968,24104, 20976,28200, 20984,32296, 21040,24680, 21048,28776, 
   21104,25192, 21112,29288, 21160,21608, 21168,25704, 21176,29800, 
   21224,22120, 21232,26216, 21240,30312, 21288,22632, 21296,26728, 
   21304,30824, 21352,23144, 21360,27240, 21368,31336, 21416,23656, 
   21424,27752, 21432,31848, 21480,24168, 21488,28264, 21496,32360, 
   21552,24744, 21560,28840, 21616,25256, 21624,29352, 21680,25768, 
   21688,29864, 21736,22184, 21744,26280, 21752,30376, 21800,22696, 
   21808,26792, 21816,30888, 21864,23208, 21872,27304, 21880,31400, 
   21928,23720, 21936,27816, 21944,31912, 21992,24232, 22000,28328, 
   22008,32424, 22064,24808, 22072,28904, 22128,25320, 22136,29416, 
   22192,25832, 22200,29928, 22256,26344, 22264,30440, 22312,22760, 
   22320,26856, 22328,30952, 22376,23272, 22384,27368, 22392,31464, 
   22440,23784, 22448,27880, 22456,31976, 22504,24296, 22512,28392, 
   22520,32488, 22576,24872, 22584,28968, 22640,25384, 22648,29480, 
   22704,25896, 22712,29992, 22768,26408, 22776,30504, 22832,26920, 
   22840,31016, 22888,23336, 22896,27432, 22904,31528, 22952,23848, 
   22960,27944, 22968,32040, 23016,24360, 23024,28456, 23032,32552, 
   23088,24936, 23096,29032, 23152,25448, 23160,29544, 23216,25960, 
   23224,30056, 23280,26472, 23288,30568, 23344,26984, 23352,31080, 
   23408,27496, 23416,31592, 23464,23912, 23472,28008, 23480,32104, 
   23528,24424, 23536,28520, 23544,32616, 23600,25000, 23608,29096, 
   23664,25512, 23672,29608, 23728,26024, 23736,30120, 23792,26536, 
   23800,30632, 23856,27048, 23864,31144, 23920,27560, 23928,31656, 
   23984,28072, 23992,32168, 24040,24488, 24048,28584, 24056,32680, 
   24112,25064, 24120,29160, 24176,25576, 24184,29672, 24240,26088, 
   24248,30184, 24304,26600, 24312,30696, 24368,27112, 24376,31208, 
   24432,27624, 24440,31720, 24496,28136, 24504,32232, 24560,28648, 
   24568,32744, 24632,28720, 24688,25136, 24696,29232, 24752,25648, 
   24760,29744, 24816,26160, 24824,30256, 24880,26672, 24888,30768, 
   24944,27184, 24952,31280, 25008,27696, 25016,31792, 25072,28208, 
   25080,32304, 25144,28784, 25208,29296, 25264,25712, 25272,29808, 
   25328,26224, 25336,30320, 25392,26736, 25400,30832, 25456,27248, 
   25464,31344, 25520,27760, 25528,31856, 25584,28272, 25592,32368, 
   25656,28848, 25720,29360, 25784,29872, 25840,26288, 25848,30384, 
   25904,26800, 25912,30896, 25968,27312, 25976,31408, 26032,27824, 
   26040,31920, 26096,28336, 26104,32432, 26168,28912, 26232,29424, 
   26296,29936, 26360,30448, 26416,26864, 26424,30960, 26480,27376, 
   26488,31472, 26544,27888, 26552,31984, 26608,28400, 26616,32496, 
   26680,28976, 26744,29488, 26808,30000, 26872,30512, 26936,31024, 
   26992,27440, 27000,31536, 27056,27952, 27064,32048, 27120,28464, 
   27128,32560, 27192,29040, 27256,29552, 27320,30064, 27384,30576, 
   27448,31088, 27512,31600, 27568,28016, 27576,32112, 27632,28528, 
   27640,32624, 27704,29104, 27768,29616, 27832,30128, 27896,30640, 
   27960,31152, 28024,31664, 28088,32176, 28144,28592, 28152,32688, 
   28216,29168, 28280,29680, 28344,30192, 28408,30704, 28472,31216, 
   28536,31728, 28600,32240, 28664,32752, 28792,29240, 28856,29752, 
   28920,30264, 28984,30776, 29048,31288, 29112,31800, 29176,32312, 
   29368,29816, 29432,30328, 29496,30840, 29560,31352, 29624,31864, 
   29688,32376, 29944,30392, 30008,30904, 30072,31416, 30136,31928, 
   30200,32440, 30520,30968, 30584,31480, 30648,31992, 30712,32504, 
   31096,31544, 31160,32056, 31224,32568, 31672,32120, 31736,32632, 
   32248,32696
};


/**    
* \par    
* Example code for Floating-point RFFT Twiddle factors Generation:    
* \par    
* <pre>TW = exp(2*pi*i*[0:L/2-1]/L - pi/2*i).' </pre>    
* \par    
* Real and Imag values are in interleaved fashion    
*/
const float32_t twiddleCoef_rfft_32[32] = {
0.0f			,	1.0f			,
0.195090322f	,	0.98078528f 	,
0.382683432f	,	0.923879533f	,
0.555570233f	,	0.831469612f	,
0.707106781f	,	0.707106781f	,
0.831469612f	,	0.555570233f	,
0.923879533f	,	0.382683432f    ,	
0.98078528f		,	0.195090322f	,
1.0f			,	0.0f			,
0.98078528f		,	-0.195090322f	,
0.923879533f	,	-0.382683432f	,
0.831469612f	,	-0.555570233f	,
0.707106781f	,	-0.707106781f	,
0.555570233f	,	-0.831469612f	,
0.382683432f	,	-0.923879533f	,
0.195090322f	,	-0.98078528f	
};

const float32_t twiddleCoef_rfft_64[64] = {
0.0f,	1.0f,
0.098017140329561f,	0.995184726672197f,
0.195090322016128f,	0.98078528040323f,
0.290284677254462f,	0.956940335732209f,
0.38268343236509f,	0.923879532511287f,
0.471396736825998f,	0.881921264348355f,
0.555570233019602f,	0.831469612302545f,
0.634393284163645f,	0.773010453362737f,
0.707106781186547f,	0.707106781186548f,
0.773010453362737f,	0.634393284163645f,
0.831469612302545f,	0.555570233019602f,
0.881921264348355f,	0.471396736825998f,
0.923879532511287f,	0.38268343236509f,
0.956940335732209f,	0.290284677254462f,
0.98078528040323f,	0.195090322016128f,
0.995184726672197f,	0.098017140329561f,
1.0f,	0.0f,
0.995184726672197f,	-0.098017140329561f,
0.98078528040323f,	-0.195090322016128f,
0.956940335732209f,	-0.290284677254462f,
0.923879532511287f,	-0.38268343236509f,
0.881921264348355f,	-0.471396736825998f,
0.831469612302545f,	-0.555570233019602f,
0.773010453362737f,	-0.634393284163645f,
0.707106781186548f,	-0.707106781186547f,
0.634393284163645f,	-0.773010453362737f,
0.555570233019602f,	-0.831469612302545f,
0.471396736825998f,	-0.881921264348355f,
0.38268343236509f,	-0.923879532511287f,
0.290284677254462f,	-0.956940335732209f,
0.195090322016129f,	-0.98078528040323f,
0.098017140329561f,	-0.995184726672197f
};

const float32_t twiddleCoef_rfft_128[128] = {
    0.000000000f,  1.000000000f,
    0.049067674f,  0.998795456f,
    0.098017140f,  0.995184727f,
    0.146730474f,  0.989176510f,
    0.195090322f,  0.980785280f,
    0.242980180f,  0.970031253f,
    0.290284677f,  0.956940336f,
    0.336889853f,  0.941544065f,
    0.382683432f,  0.923879533f,
    0.427555093f,  0.903989293f,
    0.471396737f,  0.881921264f,
    0.514102744f,  0.857728610f,
    0.555570233f,  0.831469612f,
    0.595699304f,  0.803207531f,
    0.634393284f,  0.773010453f,
    0.671558955f,  0.740951125f,
    0.707106781f,  0.707106781f,
    0.740951125f,  0.671558955f,
    0.773010453f,  0.634393284f,
    0.803207531f,  0.595699304f,
    0.831469612f,  0.555570233f,
    0.857728610f,  0.514102744f,
    0.881921264f,  0.471396737f,
    0.903989293f,  0.427555093f,
    0.923879533f,  0.382683432f,
    0.941544065f,  0.336889853f,
    0.956940336f,  0.290284677f,
    0.970031253f,  0.242980180f,
    0.980785280f,  0.195090322f,
    0.989176510f,  0.146730474f,
    0.995184727f,  0.098017140f,
    0.998795456f,  0.049067674f,
    1.000000000f,  0.000000000f,
    0.998795456f, -0.049067674f,
    0.995184727f, -0.098017140f,
    0.989176510f, -0.146730474f,
    0.980785280f, -0.195090322f,
    0.970031253f, -0.242980180f,
    0.956940336f, -0.290284677f,
    0.941544065f, -0.336889853f,
    0.923879533f, -0.382683432f,
    0.903989293f, -0.427555093f,
    0.881921264f, -0.471396737f,
    0.857728610f, -0.514102744f,
    0.831469612f, -0.555570233f,
    0.803207531f, -0.595699304f,
    0.773010453f, -0.634393284f,
    0.740951125f, -0.671558955f,
    0.707106781f, -0.707106781f,
    0.671558955f, -0.740951125f,
    0.634393284f, -0.773010453f,
    0.595699304f, -0.803207531f,
    0.555570233f, -0.831469612f,
    0.514102744f, -0.857728610f,
    0.471396737f, -0.881921264f,
    0.427555093f, -0.903989293f,
    0.382683432f, -0.923879533f,
    0.336889853f, -0.941544065f,
    0.290284677f, -0.956940336f,
    0.242980180f, -0.970031253f,
    0.195090322f, -0.980785280f,
    0.146730474f, -0.989176510f,
    0.098017140f, -0.995184727f,
    0.049067674f, -0.998795456f
};

const float32_t twiddleCoef_rfft_256[256] = {
    0.000000000f,  1.000000000f,
    0.024541229f,  0.999698819f,
    0.049067674f,  0.998795456f,
    0.073564564f,  0.997290457f,
    0.098017140f,  0.995184727f,
    0.122410675f,  0.992479535f,
    0.146730474f,  0.989176510f,
    0.170961889f,  0.985277642f,
    0.195090322f,  0.980785280f,
    0.219101240f,  0.975702130f,
    0.242980180f,  0.970031253f,
    0.266712757f,  0.963776066f,
    0.290284677f,  0.956940336f,
    0.313681740f,  0.949528181f,
    0.336889853f,  0.941544065f,
    0.359895037f,  0.932992799f,
    0.382683432f,  0.923879533f,
    0.405241314f,  0.914209756f,
    0.427555093f,  0.903989293f,
    0.449611330f,  0.893224301f,
    0.471396737f,  0.881921264f,
    0.492898192f,  0.870086991f,
    0.514102744f,  0.857728610f,
    0.534997620f,  0.844853565f,
    0.555570233f,  0.831469612f,
    0.575808191f,  0.817584813f,
    0.595699304f,  0.803207531f,
    0.615231591f,  0.788346428f,
    0.634393284f,  0.773010453f,
    0.653172843f,  0.757208847f,
    0.671558955f,  0.740951125f,
    0.689540545f,  0.724247083f,
    0.707106781f,  0.707106781f,
    0.724247083f,  0.689540545f,
    0.740951125f,  0.671558955f,
    0.757208847f,  0.653172843f,
    0.773010453f,  0.634393284f,
    0.788346428f,  0.615231591f,
    0.803207531f,  0.595699304f,
    0.817584813f,  0.575808191f,
    0.831469612f,  0.555570233f,
    0.844853565f,  0.534997620f,
    0.857728610f,  0.514102744f,
    0.870086991f,  0.492898192f,
    0.881921264f,  0.471396737f,
    0.893224301f,  0.449611330f,
    0.903989293f,  0.427555093f,
    0.914209756f,  0.405241314f,
    0.923879533f,  0.382683432f,
    0.932992799f,  0.359895037f,
    0.941544065f,  0.336889853f,
    0.949528181f,  0.313681740f,
    0.956940336f,  0.290284677f,
    0.963776066f,  0.266712757f,
    0.970031253f,  0.242980180f,
    0.975702130f,  0.219101240f,
    0.980785280f,  0.195090322f,
    0.985277642f,  0.170961889f,
    0.989176510f,  0.146730474f,
    0.992479535f,  0.122410675f,
    0.995184727f,  0.098017140f,
    0.997290457f,  0.073564564f,
    0.998795456f,  0.049067674f,
    0.999698819f,  0.024541229f,
    1.000000000f,  0.000000000f,
    0.999698819f, -0.024541229f,
    0.998795456f, -0.049067674f,
    0.997290457f, -0.073564564f,
    0.995184727f, -0.098017140f,
    0.992479535f, -0.122410675f,
    0.989176510f, -0.146730474f,
    0.985277642f, -0.170961889f,
    0.980785280f, -0.195090322f,
    0.975702130f, -0.219101240f,
    0.970031253f, -0.242980180f,
    0.963776066f, -0.266712757f,
    0.956940336f, -0.290284677f,
    0.949528181f, -0.313681740f,
    0.941544065f, -0.336889853f,
    0.932992799f, -0.359895037f,
    0.923879533f, -0.382683432f,
    0.914209756f, -0.405241314f,
    0.903989293f, -0.427555093f,
    0.893224301f, -0.449611330f,
    0.881921264f, -0.471396737f,
    0.870086991f, -0.492898192f,
    0.857728610f, -0.514102744f,
    0.844853565f, -0.534997620f,
    0.831469612f, -0.555570233f,
    0.817584813f, -0.575808191f,
    0.803207531f, -0.595699304f,
    0.788346428f, -0.615231591f,
    0.773010453f, -0.634393284f,
    0.757208847f, -0.653172843f,
    0.740951125f, -0.671558955f,
    0.724247083f, -0.689540545f,
    0.707106781f, -0.707106781f,
    0.689540545f, -0.724247083f,
    0.671558955f, -0.740951125f,
    0.653172843f, -0.757208847f,
    0.634393284f, -0.773010453f,
    0.615231591f, -0.788346428f,
    0.595699304f, -0.803207531f,
    0.575808191f, -0.817584813f,
    0.555570233f, -0.831469612f,
    0.534997620f, -0.844853565f,
    0.514102744f, -0.857728610f,
    0.492898192f, -0.870086991f,
    0.471396737f, -0.881921264f,
    0.449611330f, -0.893224301f,
    0.427555093f, -0.903989293f,
    0.405241314f, -0.914209756f,
    0.382683432f, -0.923879533f,
    0.359895037f, -0.932992799f,
    0.336889853f, -0.941544065f,
    0.313681740f, -0.949528181f,
    0.290284677f, -0.956940336f,
    0.266712757f, -0.963776066f,
    0.242980180f, -0.970031253f,
    0.219101240f, -0.975702130f,
    0.195090322f, -0.980785280f,
    0.170961889f, -0.985277642f,
    0.146730474f, -0.989176510f,
    0.122410675f, -0.992479535f,
    0.098017140f, -0.995184727f,
    0.073564564f, -0.997290457f,
    0.049067674f, -0.998795456f,
    0.024541229f, -0.999698819f
};

const float32_t twiddleCoef_rfft_512[512] = {
    0.000000000f,  1.000000000f,
    0.012271538f,  0.999924702f,
    0.024541229f,  0.999698819f,
    0.036807223f,  0.999322385f,
    0.049067674f,  0.998795456f,
    0.061320736f,  0.998118113f,
    0.073564564f,  0.997290457f,
    0.085797312f,  0.996312612f,
    0.098017140f,  0.995184727f,
    0.110222207f,  0.993906970f,
    0.122410675f,  0.992479535f,
    0.134580709f,  0.990902635f,
    0.146730474f,  0.989176510f,
    0.158858143f,  0.987301418f,
    0.170961889f,  0.985277642f,
    0.183039888f,  0.983105487f,
    0.195090322f,  0.980785280f,
    0.207111376f,  0.978317371f,
    0.219101240f,  0.975702130f,
    0.231058108f,  0.972939952f,
    0.242980180f,  0.970031253f,
    0.254865660f,  0.966976471f,
    0.266712757f,  0.963776066f,
    0.278519689f,  0.960430519f,
    0.290284677f,  0.956940336f,
    0.302005949f,  0.953306040f,
    0.313681740f,  0.949528181f,
    0.325310292f,  0.945607325f,
    0.336889853f,  0.941544065f,
    0.348418680f,  0.937339012f,
    0.359895037f,  0.932992799f,
    0.371317194f,  0.928506080f,
    0.382683432f,  0.923879533f,
    0.393992040f,  0.919113852f,
    0.405241314f,  0.914209756f,
    0.416429560f,  0.909167983f,
    0.427555093f,  0.903989293f,
    0.438616239f,  0.898674466f,
    0.449611330f,  0.893224301f,
    0.460538711f,  0.887639620f,
    0.471396737f,  0.881921264f,
    0.482183772f,  0.876070094f,
    0.492898192f,  0.870086991f,
    0.503538384f,  0.863972856f,
    0.514102744f,  0.857728610f,
    0.524589683f,  0.851355193f,
    0.534997620f,  0.844853565f,
    0.545324988f,  0.838224706f,
    0.555570233f,  0.831469612f,
    0.565731811f,  0.824589303f,
    0.575808191f,  0.817584813f,
    0.585797857f,  0.810457198f,
    0.595699304f,  0.803207531f,
    0.605511041f,  0.795836905f,
    0.615231591f,  0.788346428f,
    0.624859488f,  0.780737229f,
    0.634393284f,  0.773010453f,
    0.643831543f,  0.765167266f,
    0.653172843f,  0.757208847f,
    0.662415778f,  0.749136395f,
    0.671558955f,  0.740951125f,
    0.680600998f,  0.732654272f,
    0.689540545f,  0.724247083f,
    0.698376249f,  0.715730825f,
    0.707106781f,  0.707106781f,
    0.715730825f,  0.698376249f,
    0.724247083f,  0.689540545f,
    0.732654272f,  0.680600998f,
    0.740951125f,  0.671558955f,
    0.749136395f,  0.662415778f,
    0.757208847f,  0.653172843f,
    0.765167266f,  0.643831543f,
    0.773010453f,  0.634393284f,
    0.780737229f,  0.624859488f,
    0.788346428f,  0.615231591f,
    0.795836905f,  0.605511041f,
    0.803207531f,  0.595699304f,
    0.810457198f,  0.585797857f,
    0.817584813f,  0.575808191f,
    0.824589303f,  0.565731811f,
    0.831469612f,  0.555570233f,
    0.838224706f,  0.545324988f,
    0.844853565f,  0.534997620f,
    0.851355193f,  0.524589683f,
    0.857728610f,  0.514102744f,
    0.863972856f,  0.503538384f,
    0.870086991f,  0.492898192f,
    0.876070094f,  0.482183772f,
    0.881921264f,  0.471396737f,
    0.887639620f,  0.460538711f,
    0.893224301f,  0.449611330f,
    0.898674466f,  0.438616239f,
    0.903989293f,  0.427555093f,
    0.909167983f,  0.416429560f,
    0.914209756f,  0.405241314f,
    0.919113852f,  0.393992040f,
    0.923879533f,  0.382683432f,
    0.928506080f,  0.371317194f,
    0.932992799f,  0.359895037f,
    0.937339012f,  0.348418680f,
    0.941544065f,  0.336889853f,
    0.945607325f,  0.325310292f,
    0.949528181f,  0.313681740f,
    0.953306040f,  0.302005949f,
    0.956940336f,  0.290284677f,
    0.960430519f,  0.278519689f,
    0.963776066f,  0.266712757f,
    0.966976471f,  0.254865660f,
    0.970031253f,  0.242980180f,
    0.972939952f,  0.231058108f,
    0.975702130f,  0.219101240f,
    0.978317371f,  0.207111376f,
    0.980785280f,  0.195090322f,
    0.983105487f,  0.183039888f,
    0.985277642f,  0.170961889f,
    0.987301418f,  0.158858143f,
    0.989176510f,  0.146730474f,
    0.990902635f,  0.134580709f,
    0.992479535f,  0.122410675f,
    0.993906970f,  0.110222207f,
    0.995184727f,  0.098017140f,
    0.996312612f,  0.085797312f,
    0.997290457f,  0.073564564f,
    0.998118113f,  0.061320736f,
    0.998795456f,  0.049067674f,
    0.999322385f,  0.036807223f,
    0.999698819f,  0.024541229f,
    0.999924702f,  0.012271538f,
    1.000000000f,  0.000000000f,
    0.999924702f, -0.012271538f,
    0.999698819f, -0.024541229f,
    0.999322385f, -0.036807223f,
    0.998795456f, -0.049067674f,
    0.998118113f, -0.061320736f,
    0.997290457f, -0.073564564f,
    0.996312612f, -0.085797312f,
    0.995184727f, -0.098017140f,
    0.993906970f, -0.110222207f,
    0.992479535f, -0.122410675f,
    0.990902635f, -0.134580709f,
    0.989176510f, -0.146730474f,
    0.987301418f, -0.158858143f,
    0.985277642f, -0.170961889f,
    0.983105487f, -0.183039888f,
    0.980785280f, -0.195090322f,
    0.978317371f, -0.207111376f,
    0.975702130f, -0.219101240f,
    0.972939952f, -0.231058108f,
    0.970031253f, -0.242980180f,
    0.966976471f, -0.254865660f,
    0.963776066f, -0.266712757f,
    0.960430519f, -0.278519689f,
    0.956940336f, -0.290284677f,
    0.953306040f, -0.302005949f,
    0.949528181f, -0.313681740f,
    0.945607325f, -0.325310292f,
    0.941544065f, -0.336889853f,
    0.937339012f, -0.348418680f,
    0.932992799f, -0.359895037f,
    0.928506080f, -0.371317194f,
    0.923879533f, -0.382683432f,
    0.919113852f, -0.393992040f,
    0.914209756f, -0.405241314f,
    0.909167983f, -0.416429560f,
    0.903989293f, -0.427555093f,
    0.898674466f, -0.438616239f,
    0.893224301f, -0.449611330f,
    0.887639620f, -0.460538711f,
    0.881921264f, -0.471396737f,
    0.876070094f, -0.482183772f,
    0.870086991f, -0.492898192f,
    0.863972856f, -0.503538384f,
    0.857728610f, -0.514102744f,
    0.851355193f, -0.524589683f,
    0.844853565f, -0.534997620f,
    0.838224706f, -0.545324988f,
    0.831469612f, -0.555570233f,
    0.824589303f, -0.565731811f,
    0.817584813f, -0.575808191f,
    0.810457198f, -0.585797857f,
    0.803207531f, -0.595699304f,
    0.795836905f, -0.605511041f,
    0.788346428f, -0.615231591f,
    0.780737229f, -0.624859488f,
    0.773010453f, -0.634393284f,
    0.765167266f, -0.643831543f,
    0.757208847f, -0.653172843f,
    0.749136395f, -0.662415778f,
    0.740951125f, -0.671558955f,
    0.732654272f, -0.680600998f,
    0.724247083f, -0.689540545f,
    0.715730825f, -0.698376249f,
    0.707106781f, -0.707106781f,
    0.698376249f, -0.715730825f,
    0.689540545f, -0.724247083f,
    0.680600998f, -0.732654272f,
    0.671558955f, -0.740951125f,
    0.662415778f, -0.749136395f,
    0.653172843f, -0.757208847f,
    0.643831543f, -0.765167266f,
    0.634393284f, -0.773010453f,
    0.624859488f, -0.780737229f,
    0.615231591f, -0.788346428f,
    0.605511041f, -0.795836905f,
    0.595699304f, -0.803207531f,
    0.585797857f, -0.810457198f,
    0.575808191f, -0.817584813f,
    0.565731811f, -0.824589303f,
    0.555570233f, -0.831469612f,
    0.545324988f, -0.838224706f,
    0.534997620f, -0.844853565f,
    0.524589683f, -0.851355193f,
    0.514102744f, -0.857728610f,
    0.503538384f, -0.863972856f,
    0.492898192f, -0.870086991f,
    0.482183772f, -0.876070094f,
    0.471396737f, -0.881921264f,
    0.460538711f, -0.887639620f,
    0.449611330f, -0.893224301f,
    0.438616239f, -0.898674466f,
    0.427555093f, -0.903989293f,
    0.416429560f, -0.909167983f,
    0.405241314f, -0.914209756f,
    0.393992040f, -0.919113852f,
    0.382683432f, -0.923879533f,
    0.371317194f, -0.928506080f,
    0.359895037f, -0.932992799f,
    0.348418680f, -0.937339012f,
    0.336889853f, -0.941544065f,
    0.325310292f, -0.945607325f,
    0.313681740f, -0.949528181f,
    0.302005949f, -0.953306040f,
    0.290284677f, -0.956940336f,
    0.278519689f, -0.960430519f,
    0.266712757f, -0.963776066f,
    0.254865660f, -0.966976471f,
    0.242980180f, -0.970031253f,
    0.231058108f, -0.972939952f,
    0.219101240f, -0.975702130f,
    0.207111376f, -0.978317371f,
    0.195090322f, -0.980785280f,
    0.183039888f, -0.983105487f,
    0.170961889f, -0.985277642f,
    0.158858143f, -0.987301418f,
    0.146730474f, -0.989176510f,
    0.134580709f, -0.990902635f,
    0.122410675f, -0.992479535f,
    0.110222207f, -0.993906970f,
    0.098017140f, -0.995184727f,
    0.085797312f, -0.996312612f,
    0.073564564f, -0.997290457f,
    0.061320736f, -0.998118113f,
    0.049067674f, -0.998795456f,
    0.036807223f, -0.999322385f,
    0.024541229f, -0.999698819f,
    0.012271538f, -0.999924702f
};

const float32_t twiddleCoef_rfft_1024[1024] = {
    0.000000000f,  1.000000000f,
    0.006135885f,  0.999981175f,
    0.012271538f,  0.999924702f,
    0.018406730f,  0.999830582f,
    0.024541229f,  0.999698819f,
    0.030674803f,  0.999529418f,
    0.036807223f,  0.999322385f,
    0.042938257f,  0.999077728f,
    0.049067674f,  0.998795456f,
    0.055195244f,  0.998475581f,
    0.061320736f,  0.998118113f,
    0.067443920f,  0.997723067f,
    0.073564564f,  0.997290457f,
    0.079682438f,  0.996820299f,
    0.085797312f,  0.996312612f,
    0.091908956f,  0.995767414f,
    0.098017140f,  0.995184727f,
    0.104121634f,  0.994564571f,
    0.110222207f,  0.993906970f,
    0.116318631f,  0.993211949f,
    0.122410675f,  0.992479535f,
    0.128498111f,  0.991709754f,
    0.134580709f,  0.990902635f,
    0.140658239f,  0.990058210f,
    0.146730474f,  0.989176510f,
    0.152797185f,  0.988257568f,
    0.158858143f,  0.987301418f,
    0.164913120f,  0.986308097f,
    0.170961889f,  0.985277642f,
    0.177004220f,  0.984210092f,
    0.183039888f,  0.983105487f,
    0.189068664f,  0.981963869f,
    0.195090322f,  0.980785280f,
    0.201104635f,  0.979569766f,
    0.207111376f,  0.978317371f,
    0.213110320f,  0.977028143f,
    0.219101240f,  0.975702130f,
    0.225083911f,  0.974339383f,
    0.231058108f,  0.972939952f,
    0.237023606f,  0.971503891f,
    0.242980180f,  0.970031253f,
    0.248927606f,  0.968522094f,
    0.254865660f,  0.966976471f,
    0.260794118f,  0.965394442f,
    0.266712757f,  0.963776066f,
    0.272621355f,  0.962121404f,
    0.278519689f,  0.960430519f,
    0.284407537f,  0.958703475f,
    0.290284677f,  0.956940336f,
    0.296150888f,  0.955141168f,
    0.302005949f,  0.953306040f,
    0.307849640f,  0.951435021f,
    0.313681740f,  0.949528181f,
    0.319502031f,  0.947585591f,
    0.325310292f,  0.945607325f,
    0.331106306f,  0.943593458f,
    0.336889853f,  0.941544065f,
    0.342660717f,  0.939459224f,
    0.348418680f,  0.937339012f,
    0.354163525f,  0.935183510f,
    0.359895037f,  0.932992799f,
    0.365612998f,  0.930766961f,
    0.371317194f,  0.928506080f,
    0.377007410f,  0.926210242f,
    0.382683432f,  0.923879533f,
    0.388345047f,  0.921514039f,
    0.393992040f,  0.919113852f,
    0.399624200f,  0.916679060f,
    0.405241314f,  0.914209756f,
    0.410843171f,  0.911706032f,
    0.416429560f,  0.909167983f,
    0.422000271f,  0.906595705f,
    0.427555093f,  0.903989293f,
    0.433093819f,  0.901348847f,
    0.438616239f,  0.898674466f,
    0.444122145f,  0.895966250f,
    0.449611330f,  0.893224301f,
    0.455083587f,  0.890448723f,
    0.460538711f,  0.887639620f,
    0.465976496f,  0.884797098f,
    0.471396737f,  0.881921264f,
    0.476799230f,  0.879012226f,
    0.482183772f,  0.876070094f,
    0.487550160f,  0.873094978f,
    0.492898192f,  0.870086991f,
    0.498227667f,  0.867046246f,
    0.503538384f,  0.863972856f,
    0.508830143f,  0.860866939f,
    0.514102744f,  0.857728610f,
    0.519355990f,  0.854557988f,
    0.524589683f,  0.851355193f,
    0.529803625f,  0.848120345f,
    0.534997620f,  0.844853565f,
    0.540171473f,  0.841554977f,
    0.545324988f,  0.838224706f,
    0.550457973f,  0.834862875f,
    0.555570233f,  0.831469612f,
    0.560661576f,  0.828045045f,
    0.565731811f,  0.824589303f,
    0.570780746f,  0.821102515f,
    0.575808191f,  0.817584813f,
    0.580813958f,  0.814036330f,
    0.585797857f,  0.810457198f,
    0.590759702f,  0.806847554f,
    0.595699304f,  0.803207531f,
    0.600616479f,  0.799537269f,
    0.605511041f,  0.795836905f,
    0.610382806f,  0.792106577f,
    0.615231591f,  0.788346428f,
    0.620057212f,  0.784556597f,
    0.624859488f,  0.780737229f,
    0.629638239f,  0.776888466f,
    0.634393284f,  0.773010453f,
    0.639124445f,  0.769103338f,
    0.643831543f,  0.765167266f,
    0.648514401f,  0.761202385f,
    0.653172843f,  0.757208847f,
    0.657806693f,  0.753186799f,
    0.662415778f,  0.749136395f,
    0.666999922f,  0.745057785f,
    0.671558955f,  0.740951125f,
    0.676092704f,  0.736816569f,
    0.680600998f,  0.732654272f,
    0.685083668f,  0.728464390f,
    0.689540545f,  0.724247083f,
    0.693971461f,  0.720002508f,
    0.698376249f,  0.715730825f,
    0.702754744f,  0.711432196f,
    0.707106781f,  0.707106781f,
    0.711432196f,  0.702754744f,
    0.715730825f,  0.698376249f,
    0.720002508f,  0.693971461f,
    0.724247083f,  0.689540545f,
    0.728464390f,  0.685083668f,
    0.732654272f,  0.680600998f,
    0.736816569f,  0.676092704f,
    0.740951125f,  0.671558955f,
    0.745057785f,  0.666999922f,
    0.749136395f,  0.662415778f,
    0.753186799f,  0.657806693f,
    0.757208847f,  0.653172843f,
    0.761202385f,  0.648514401f,
    0.765167266f,  0.643831543f,
    0.769103338f,  0.639124445f,
    0.773010453f,  0.634393284f,
    0.776888466f,  0.629638239f,
    0.780737229f,  0.624859488f,
    0.784556597f,  0.620057212f,
    0.788346428f,  0.615231591f,
    0.792106577f,  0.610382806f,
    0.795836905f,  0.605511041f,
    0.799537269f,  0.600616479f,
    0.803207531f,  0.595699304f,
    0.806847554f,  0.590759702f,
    0.810457198f,  0.585797857f,
    0.814036330f,  0.580813958f,
    0.817584813f,  0.575808191f,
    0.821102515f,  0.570780746f,
    0.824589303f,  0.565731811f,
    0.828045045f,  0.560661576f,
    0.831469612f,  0.555570233f,
    0.834862875f,  0.550457973f,
    0.838224706f,  0.545324988f,
    0.841554977f,  0.540171473f,
    0.844853565f,  0.534997620f,
    0.848120345f,  0.529803625f,
    0.851355193f,  0.524589683f,
    0.854557988f,  0.519355990f,
    0.857728610f,  0.514102744f,
    0.860866939f,  0.508830143f,
    0.863972856f,  0.503538384f,
    0.867046246f,  0.498227667f,
    0.870086991f,  0.492898192f,
    0.873094978f,  0.487550160f,
    0.876070094f,  0.482183772f,
    0.879012226f,  0.476799230f,
    0.881921264f,  0.471396737f,
    0.884797098f,  0.465976496f,
    0.887639620f,  0.460538711f,
    0.890448723f,  0.455083587f,
    0.893224301f,  0.449611330f,
    0.895966250f,  0.444122145f,
    0.898674466f,  0.438616239f,
    0.901348847f,  0.433093819f,
    0.903989293f,  0.427555093f,
    0.906595705f,  0.422000271f,
    0.909167983f,  0.416429560f,
    0.911706032f,  0.410843171f,
    0.914209756f,  0.405241314f,
    0.916679060f,  0.399624200f,
    0.919113852f,  0.393992040f,
    0.921514039f,  0.388345047f,
    0.923879533f,  0.382683432f,
    0.926210242f,  0.377007410f,
    0.928506080f,  0.371317194f,
    0.930766961f,  0.365612998f,
    0.932992799f,  0.359895037f,
    0.935183510f,  0.354163525f,
    0.937339012f,  0.348418680f,
    0.939459224f,  0.342660717f,
    0.941544065f,  0.336889853f,
    0.943593458f,  0.331106306f,
    0.945607325f,  0.325310292f,
    0.947585591f,  0.319502031f,
    0.949528181f,  0.313681740f,
    0.951435021f,  0.307849640f,
    0.953306040f,  0.302005949f,
    0.955141168f,  0.296150888f,
    0.956940336f,  0.290284677f,
    0.958703475f,  0.284407537f,
    0.960430519f,  0.278519689f,
    0.962121404f,  0.272621355f,
    0.963776066f,  0.266712757f,
    0.965394442f,  0.260794118f,
    0.966976471f,  0.254865660f,
    0.968522094f,  0.248927606f,
    0.970031253f,  0.242980180f,
    0.971503891f,  0.237023606f,
    0.972939952f,  0.231058108f,
    0.974339383f,  0.225083911f,
    0.975702130f,  0.219101240f,
    0.977028143f,  0.213110320f,
    0.978317371f,  0.207111376f,
    0.979569766f,  0.201104635f,
    0.980785280f,  0.195090322f,
    0.981963869f,  0.189068664f,
    0.983105487f,  0.183039888f,
    0.984210092f,  0.177004220f,
    0.985277642f,  0.170961889f,
    0.986308097f,  0.164913120f,
    0.987301418f,  0.158858143f,
    0.988257568f,  0.152797185f,
    0.989176510f,  0.146730474f,
    0.990058210f,  0.140658239f,
    0.990902635f,  0.134580709f,
    0.991709754f,  0.128498111f,
    0.992479535f,  0.122410675f,
    0.993211949f,  0.116318631f,
    0.993906970f,  0.110222207f,
    0.994564571f,  0.104121634f,
    0.995184727f,  0.098017140f,
    0.995767414f,  0.091908956f,
    0.996312612f,  0.085797312f,
    0.996820299f,  0.079682438f,
    0.997290457f,  0.073564564f,
    0.997723067f,  0.067443920f,
    0.998118113f,  0.061320736f,
    0.998475581f,  0.055195244f,
    0.998795456f,  0.049067674f,
    0.999077728f,  0.042938257f,
    0.999322385f,  0.036807223f,
    0.999529418f,  0.030674803f,
    0.999698819f,  0.024541229f,
    0.999830582f,  0.018406730f,
    0.999924702f,  0.012271538f,
    0.999981175f,  0.006135885f,
    1.000000000f,  0.000000000f,
    0.999981175f, -0.006135885f,
    0.999924702f, -0.012271538f,
    0.999830582f, -0.018406730f,
    0.999698819f, -0.024541229f,
    0.999529418f, -0.030674803f,
    0.999322385f, -0.036807223f,
    0.999077728f, -0.042938257f,
    0.998795456f, -0.049067674f,
    0.998475581f, -0.055195244f,
    0.998118113f, -0.061320736f,
    0.997723067f, -0.067443920f,
    0.997290457f, -0.073564564f,
    0.996820299f, -0.079682438f,
    0.996312612f, -0.085797312f,
    0.995767414f, -0.091908956f,
    0.995184727f, -0.098017140f,
    0.994564571f, -0.104121634f,
    0.993906970f, -0.110222207f,
    0.993211949f, -0.116318631f,
    0.992479535f, -0.122410675f,
    0.991709754f, -0.128498111f,
    0.990902635f, -0.134580709f,
    0.990058210f, -0.140658239f,
    0.989176510f, -0.146730474f,
    0.988257568f, -0.152797185f,
    0.987301418f, -0.158858143f,
    0.986308097f, -0.164913120f,
    0.985277642f, -0.170961889f,
    0.984210092f, -0.177004220f,
    0.983105487f, -0.183039888f,
    0.981963869f, -0.189068664f,
    0.980785280f, -0.195090322f,
    0.979569766f, -0.201104635f,
    0.978317371f, -0.207111376f,
    0.977028143f, -0.213110320f,
    0.975702130f, -0.219101240f,
    0.974339383f, -0.225083911f,
    0.972939952f, -0.231058108f,
    0.971503891f, -0.237023606f,
    0.970031253f, -0.242980180f,
    0.968522094f, -0.248927606f,
    0.966976471f, -0.254865660f,
    0.965394442f, -0.260794118f,
    0.963776066f, -0.266712757f,
    0.962121404f, -0.272621355f,
    0.960430519f, -0.278519689f,
    0.958703475f, -0.284407537f,
    0.956940336f, -0.290284677f,
    0.955141168f, -0.296150888f,
    0.953306040f, -0.302005949f,
    0.951435021f, -0.307849640f,
    0.949528181f, -0.313681740f,
    0.947585591f, -0.319502031f,
    0.945607325f, -0.325310292f,
    0.943593458f, -0.331106306f,
    0.941544065f, -0.336889853f,
    0.939459224f, -0.342660717f,
    0.937339012f, -0.348418680f,
    0.935183510f, -0.354163525f,
    0.932992799f, -0.359895037f,
    0.930766961f, -0.365612998f,
    0.928506080f, -0.371317194f,
    0.926210242f, -0.377007410f,
    0.923879533f, -0.382683432f,
    0.921514039f, -0.388345047f,
    0.919113852f, -0.393992040f,
    0.916679060f, -0.399624200f,
    0.914209756f, -0.405241314f,
    0.911706032f, -0.410843171f,
    0.909167983f, -0.416429560f,
    0.906595705f, -0.422000271f,
    0.903989293f, -0.427555093f,
    0.901348847f, -0.433093819f,
    0.898674466f, -0.438616239f,
    0.895966250f, -0.444122145f,
    0.893224301f, -0.449611330f,
    0.890448723f, -0.455083587f,
    0.887639620f, -0.460538711f,
    0.884797098f, -0.465976496f,
    0.881921264f, -0.471396737f,
    0.879012226f, -0.476799230f,
    0.876070094f, -0.482183772f,
    0.873094978f, -0.487550160f,
    0.870086991f, -0.492898192f,
    0.867046246f, -0.498227667f,
    0.863972856f, -0.503538384f,
    0.860866939f, -0.508830143f,
    0.857728610f, -0.514102744f,
    0.854557988f, -0.519355990f,
    0.851355193f, -0.524589683f,
    0.848120345f, -0.529803625f,
    0.844853565f, -0.534997620f,
    0.841554977f, -0.540171473f,
    0.838224706f, -0.545324988f,
    0.834862875f, -0.550457973f,
    0.831469612f, -0.555570233f,
    0.828045045f, -0.560661576f,
    0.824589303f, -0.565731811f,
    0.821102515f, -0.570780746f,
    0.817584813f, -0.575808191f,
    0.814036330f, -0.580813958f,
    0.810457198f, -0.585797857f,
    0.806847554f, -0.590759702f,
    0.803207531f, -0.595699304f,
    0.799537269f, -0.600616479f,
    0.795836905f, -0.605511041f,
    0.792106577f, -0.610382806f,
    0.788346428f, -0.615231591f,
    0.784556597f, -0.620057212f,
    0.780737229f, -0.624859488f,
    0.776888466f, -0.629638239f,
    0.773010453f, -0.634393284f,
    0.769103338f, -0.639124445f,
    0.765167266f, -0.643831543f,
    0.761202385f, -0.648514401f,
    0.757208847f, -0.653172843f,
    0.753186799f, -0.657806693f,
    0.749136395f, -0.662415778f,
    0.745057785f, -0.666999922f,
    0.740951125f, -0.671558955f,
    0.736816569f, -0.676092704f,
    0.732654272f, -0.680600998f,
    0.728464390f, -0.685083668f,
    0.724247083f, -0.689540545f,
    0.720002508f, -0.693971461f,
    0.715730825f, -0.698376249f,
    0.711432196f, -0.702754744f,
    0.707106781f, -0.707106781f,
    0.702754744f, -0.711432196f,
    0.698376249f, -0.715730825f,
    0.693971461f, -0.720002508f,
    0.689540545f, -0.724247083f,
    0.685083668f, -0.728464390f,
    0.680600998f, -0.732654272f,
    0.676092704f, -0.736816569f,
    0.671558955f, -0.740951125f,
    0.666999922f, -0.745057785f,
    0.662415778f, -0.749136395f,
    0.657806693f, -0.753186799f,
    0.653172843f, -0.757208847f,
    0.648514401f, -0.761202385f,
    0.643831543f, -0.765167266f,
    0.639124445f, -0.769103338f,
    0.634393284f, -0.773010453f,
    0.629638239f, -0.776888466f,
    0.624859488f, -0.780737229f,
    0.620057212f, -0.784556597f,
    0.615231591f, -0.788346428f,
    0.610382806f, -0.792106577f,
    0.605511041f, -0.795836905f,
    0.600616479f, -0.799537269f,
    0.595699304f, -0.803207531f,
    0.590759702f, -0.806847554f,
    0.585797857f, -0.810457198f,
    0.580813958f, -0.814036330f,
    0.575808191f, -0.817584813f,
    0.570780746f, -0.821102515f,
    0.565731811f, -0.824589303f,
    0.560661576f, -0.828045045f,
    0.555570233f, -0.831469612f,
    0.550457973f, -0.834862875f,
    0.545324988f, -0.838224706f,
    0.540171473f, -0.841554977f,
    0.534997620f, -0.844853565f,
    0.529803625f, -0.848120345f,
    0.524589683f, -0.851355193f,
    0.519355990f, -0.854557988f,
    0.514102744f, -0.857728610f,
    0.508830143f, -0.860866939f,
    0.503538384f, -0.863972856f,
    0.498227667f, -0.867046246f,
    0.492898192f, -0.870086991f,
    0.487550160f, -0.873094978f,
    0.482183772f, -0.876070094f,
    0.476799230f, -0.879012226f,
    0.471396737f, -0.881921264f,
    0.465976496f, -0.884797098f,
    0.460538711f, -0.887639620f,
    0.455083587f, -0.890448723f,
    0.449611330f, -0.893224301f,
    0.444122145f, -0.895966250f,
    0.438616239f, -0.898674466f,
    0.433093819f, -0.901348847f,
    0.427555093f, -0.903989293f,
    0.422000271f, -0.906595705f,
    0.416429560f, -0.909167983f,
    0.410843171f, -0.911706032f,
    0.405241314f, -0.914209756f,
    0.399624200f, -0.916679060f,
    0.393992040f, -0.919113852f,
    0.388345047f, -0.921514039f,
    0.382683432f, -0.923879533f,
    0.377007410f, -0.926210242f,
    0.371317194f, -0.928506080f,
    0.365612998f, -0.930766961f,
    0.359895037f, -0.932992799f,
    0.354163525f, -0.935183510f,
    0.348418680f, -0.937339012f,
    0.342660717f, -0.939459224f,
    0.336889853f, -0.941544065f,
    0.331106306f, -0.943593458f,
    0.325310292f, -0.945607325f,
    0.319502031f, -0.947585591f,
    0.313681740f, -0.949528181f,
    0.307849640f, -0.951435021f,
    0.302005949f, -0.953306040f,
    0.296150888f, -0.955141168f,
    0.290284677f, -0.956940336f,
    0.284407537f, -0.958703475f,
    0.278519689f, -0.960430519f,
    0.272621355f, -0.962121404f,
    0.266712757f, -0.963776066f,
    0.260794118f, -0.965394442f,
    0.254865660f, -0.966976471f,
    0.248927606f, -0.968522094f,
    0.242980180f, -0.970031253f,
    0.237023606f, -0.971503891f,
    0.231058108f, -0.972939952f,
    0.225083911f, -0.974339383f,
    0.219101240f, -0.975702130f,
    0.213110320f, -0.977028143f,
    0.207111376f, -0.978317371f,
    0.201104635f, -0.979569766f,
    0.195090322f, -0.980785280f,
    0.189068664f, -0.981963869f,
    0.183039888f, -0.983105487f,
    0.177004220f, -0.984210092f,
    0.170961889f, -0.985277642f,
    0.164913120f, -0.986308097f,
    0.158858143f, -0.987301418f,
    0.152797185f, -0.988257568f,
    0.146730474f, -0.989176510f,
    0.140658239f, -0.990058210f,
    0.134580709f, -0.990902635f,
    0.128498111f, -0.991709754f,
    0.122410675f, -0.992479535f,
    0.116318631f, -0.993211949f,
    0.110222207f, -0.993906970f,
    0.104121634f, -0.994564571f,
    0.098017140f, -0.995184727f,
    0.091908956f, -0.995767414f,
    0.085797312f, -0.996312612f,
    0.079682438f, -0.996820299f,
    0.073564564f, -0.997290457f,
    0.067443920f, -0.997723067f,
    0.061320736f, -0.998118113f,
    0.055195244f, -0.998475581f,
    0.049067674f, -0.998795456f,
    0.042938257f, -0.999077728f,
    0.036807223f, -0.999322385f,
    0.030674803f, -0.999529418f,
    0.024541229f, -0.999698819f,
    0.018406730f, -0.999830582f,
    0.012271538f, -0.999924702f,
    0.006135885f, -0.999981175f
};

const float32_t twiddleCoef_rfft_2048[2048] = {
    0.000000000f,  1.000000000f,
    0.003067957f,  0.999995294f,
    0.006135885f,  0.999981175f,
    0.009203755f,  0.999957645f,
    0.012271538f,  0.999924702f,
    0.015339206f,  0.999882347f,
    0.018406730f,  0.999830582f,
    0.021474080f,  0.999769405f,
    0.024541229f,  0.999698819f,
    0.027608146f,  0.999618822f,
    0.030674803f,  0.999529418f,
    0.033741172f,  0.999430605f,
    0.036807223f,  0.999322385f,
    0.039872928f,  0.999204759f,
    0.042938257f,  0.999077728f,
    0.046003182f,  0.998941293f,
    0.049067674f,  0.998795456f,
    0.052131705f,  0.998640218f,
    0.055195244f,  0.998475581f,
    0.058258265f,  0.998301545f,
    0.061320736f,  0.998118113f,
    0.064382631f,  0.997925286f,
    0.067443920f,  0.997723067f,
    0.070504573f,  0.997511456f,
    0.073564564f,  0.997290457f,
    0.076623861f,  0.997060070f,
    0.079682438f,  0.996820299f,
    0.082740265f,  0.996571146f,
    0.085797312f,  0.996312612f,
    0.088853553f,  0.996044701f,
    0.091908956f,  0.995767414f,
    0.094963495f,  0.995480755f,
    0.098017140f,  0.995184727f,
    0.101069863f,  0.994879331f,
    0.104121634f,  0.994564571f,
    0.107172425f,  0.994240449f,
    0.110222207f,  0.993906970f,
    0.113270952f,  0.993564136f,
    0.116318631f,  0.993211949f,
    0.119365215f,  0.992850414f,
    0.122410675f,  0.992479535f,
    0.125454983f,  0.992099313f,
    0.128498111f,  0.991709754f,
    0.131540029f,  0.991310860f,
    0.134580709f,  0.990902635f,
    0.137620122f,  0.990485084f,
    0.140658239f,  0.990058210f,
    0.143695033f,  0.989622017f,
    0.146730474f,  0.989176510f,
    0.149764535f,  0.988721692f,
    0.152797185f,  0.988257568f,
    0.155828398f,  0.987784142f,
    0.158858143f,  0.987301418f,
    0.161886394f,  0.986809402f,
    0.164913120f,  0.986308097f,
    0.167938295f,  0.985797509f,
    0.170961889f,  0.985277642f,
    0.173983873f,  0.984748502f,
    0.177004220f,  0.984210092f,
    0.180022901f,  0.983662419f,
    0.183039888f,  0.983105487f,
    0.186055152f,  0.982539302f,
    0.189068664f,  0.981963869f,
    0.192080397f,  0.981379193f,
    0.195090322f,  0.980785280f,
    0.198098411f,  0.980182136f,
    0.201104635f,  0.979569766f,
    0.204108966f,  0.978948175f,
    0.207111376f,  0.978317371f,
    0.210111837f,  0.977677358f,
    0.213110320f,  0.977028143f,
    0.216106797f,  0.976369731f,
    0.219101240f,  0.975702130f,
    0.222093621f,  0.975025345f,
    0.225083911f,  0.974339383f,
    0.228072083f,  0.973644250f,
    0.231058108f,  0.972939952f,
    0.234041959f,  0.972226497f,
    0.237023606f,  0.971503891f,
    0.240003022f,  0.970772141f,
    0.242980180f,  0.970031253f,
    0.245955050f,  0.969281235f,
    0.248927606f,  0.968522094f,
    0.251897818f,  0.967753837f,
    0.254865660f,  0.966976471f,
    0.257831102f,  0.966190003f,
    0.260794118f,  0.965394442f,
    0.263754679f,  0.964589793f,
    0.266712757f,  0.963776066f,
    0.269668326f,  0.962953267f,
    0.272621355f,  0.962121404f,
    0.275571819f,  0.961280486f,
    0.278519689f,  0.960430519f,
    0.281464938f,  0.959571513f,
    0.284407537f,  0.958703475f,
    0.287347460f,  0.957826413f,
    0.290284677f,  0.956940336f,
    0.293219163f,  0.956045251f,
    0.296150888f,  0.955141168f,
    0.299079826f,  0.954228095f,
    0.302005949f,  0.953306040f,
    0.304929230f,  0.952375013f,
    0.307849640f,  0.951435021f,
    0.310767153f,  0.950486074f,
    0.313681740f,  0.949528181f,
    0.316593376f,  0.948561350f,
    0.319502031f,  0.947585591f,
    0.322407679f,  0.946600913f,
    0.325310292f,  0.945607325f,
    0.328209844f,  0.944604837f,
    0.331106306f,  0.943593458f,
    0.333999651f,  0.942573198f,
    0.336889853f,  0.941544065f,
    0.339776884f,  0.940506071f,
    0.342660717f,  0.939459224f,
    0.345541325f,  0.938403534f,
    0.348418680f,  0.937339012f,
    0.351292756f,  0.936265667f,
    0.354163525f,  0.935183510f,
    0.357030961f,  0.934092550f,
    0.359895037f,  0.932992799f,
    0.362755724f,  0.931884266f,
    0.365612998f,  0.930766961f,
    0.368466830f,  0.929640896f,
    0.371317194f,  0.928506080f,
    0.374164063f,  0.927362526f,
    0.377007410f,  0.926210242f,
    0.379847209f,  0.925049241f,
    0.382683432f,  0.923879533f,
    0.385516054f,  0.922701128f,
    0.388345047f,  0.921514039f,
    0.391170384f,  0.920318277f,
    0.393992040f,  0.919113852f,
    0.396809987f,  0.917900776f,
    0.399624200f,  0.916679060f,
    0.402434651f,  0.915448716f,
    0.405241314f,  0.914209756f,
    0.408044163f,  0.912962190f,
    0.410843171f,  0.911706032f,
    0.413638312f,  0.910441292f,
    0.416429560f,  0.909167983f,
    0.419216888f,  0.907886116f,
    0.422000271f,  0.906595705f,
    0.424779681f,  0.905296759f,
    0.427555093f,  0.903989293f,
    0.430326481f,  0.902673318f,
    0.433093819f,  0.901348847f,
    0.435857080f,  0.900015892f,
    0.438616239f,  0.898674466f,
    0.441371269f,  0.897324581f,
    0.444122145f,  0.895966250f,
    0.446868840f,  0.894599486f,
    0.449611330f,  0.893224301f,
    0.452349587f,  0.891840709f,
    0.455083587f,  0.890448723f,
    0.457813304f,  0.889048356f,
    0.460538711f,  0.887639620f,
    0.463259784f,  0.886222530f,
    0.465976496f,  0.884797098f,
    0.468688822f,  0.883363339f,
    0.471396737f,  0.881921264f,
    0.474100215f,  0.880470889f,
    0.476799230f,  0.879012226f,
    0.479493758f,  0.877545290f,
    0.482183772f,  0.876070094f,
    0.484869248f,  0.874586652f,
    0.487550160f,  0.873094978f,
    0.490226483f,  0.871595087f,
    0.492898192f,  0.870086991f,
    0.495565262f,  0.868570706f,
    0.498227667f,  0.867046246f,
    0.500885383f,  0.865513624f,
    0.503538384f,  0.863972856f,
    0.506186645f,  0.862423956f,
    0.508830143f,  0.860866939f,
    0.511468850f,  0.859301818f,
    0.514102744f,  0.857728610f,
    0.516731799f,  0.856147328f,
    0.519355990f,  0.854557988f,
    0.521975293f,  0.852960605f,
    0.524589683f,  0.851355193f,
    0.527199135f,  0.849741768f,
    0.529803625f,  0.848120345f,
    0.532403128f,  0.846490939f,
    0.534997620f,  0.844853565f,
    0.537587076f,  0.843208240f,
    0.540171473f,  0.841554977f,
    0.542750785f,  0.839893794f,
    0.545324988f,  0.838224706f,
    0.547894059f,  0.836547727f,
    0.550457973f,  0.834862875f,
    0.553016706f,  0.833170165f,
    0.555570233f,  0.831469612f,
    0.558118531f,  0.829761234f,
    0.560661576f,  0.828045045f,
    0.563199344f,  0.826321063f,
    0.565731811f,  0.824589303f,
    0.568258953f,  0.822849781f,
    0.570780746f,  0.821102515f,
    0.573297167f,  0.819347520f,
    0.575808191f,  0.817584813f,
    0.578313796f,  0.815814411f,
    0.580813958f,  0.814036330f,
    0.583308653f,  0.812250587f,
    0.585797857f,  0.810457198f,
    0.588281548f,  0.808656182f,
    0.590759702f,  0.806847554f,
    0.593232295f,  0.805031331f,
    0.595699304f,  0.803207531f,
    0.598160707f,  0.801376172f,
    0.600616479f,  0.799537269f,
    0.603066599f,  0.797690841f,
    0.605511041f,  0.795836905f,
    0.607949785f,  0.793975478f,
    0.610382806f,  0.792106577f,
    0.612810082f,  0.790230221f,
    0.615231591f,  0.788346428f,
    0.617647308f,  0.786455214f,
    0.620057212f,  0.784556597f,
    0.622461279f,  0.782650596f,
    0.624859488f,  0.780737229f,
    0.627251815f,  0.778816512f,
    0.629638239f,  0.776888466f,
    0.632018736f,  0.774953107f,
    0.634393284f,  0.773010453f,
    0.636761861f,  0.771060524f,
    0.639124445f,  0.769103338f,
    0.641481013f,  0.767138912f,
    0.643831543f,  0.765167266f,
    0.646176013f,  0.763188417f,
    0.648514401f,  0.761202385f,
    0.650846685f,  0.759209189f,
    0.653172843f,  0.757208847f,
    0.655492853f,  0.755201377f,
    0.657806693f,  0.753186799f,
    0.660114342f,  0.751165132f,
    0.662415778f,  0.749136395f,
    0.664710978f,  0.747100606f,
    0.666999922f,  0.745057785f,
    0.669282588f,  0.743007952f,
    0.671558955f,  0.740951125f,
    0.673829000f,  0.738887324f,
    0.676092704f,  0.736816569f,
    0.678350043f,  0.734738878f,
    0.680600998f,  0.732654272f,
    0.682845546f,  0.730562769f,
    0.685083668f,  0.728464390f,
    0.687315341f,  0.726359155f,
    0.689540545f,  0.724247083f,
    0.691759258f,  0.722128194f,
    0.693971461f,  0.720002508f,
    0.696177131f,  0.717870045f,
    0.698376249f,  0.715730825f,
    0.700568794f,  0.713584869f,
    0.702754744f,  0.711432196f,
    0.704934080f,  0.709272826f,
    0.707106781f,  0.707106781f,
    0.709272826f,  0.704934080f,
    0.711432196f,  0.702754744f,
    0.713584869f,  0.700568794f,
    0.715730825f,  0.698376249f,
    0.717870045f,  0.696177131f,
    0.720002508f,  0.693971461f,
    0.722128194f,  0.691759258f,
    0.724247083f,  0.689540545f,
    0.726359155f,  0.687315341f,
    0.728464390f,  0.685083668f,
    0.730562769f,  0.682845546f,
    0.732654272f,  0.680600998f,
    0.734738878f,  0.678350043f,
    0.736816569f,  0.676092704f,
    0.738887324f,  0.673829000f,
    0.740951125f,  0.671558955f,
    0.743007952f,  0.669282588f,
    0.745057785f,  0.666999922f,
    0.747100606f,  0.664710978f,
    0.749136395f,  0.662415778f,
    0.751165132f,  0.660114342f,
    0.753186799f,  0.657806693f,
    0.755201377f,  0.655492853f,
    0.757208847f,  0.653172843f,
    0.759209189f,  0.650846685f,
    0.761202385f,  0.648514401f,
    0.763188417f,  0.646176013f,
    0.765167266f,  0.643831543f,
    0.767138912f,  0.641481013f,
    0.769103338f,  0.639124445f,
    0.771060524f,  0.636761861f,
    0.773010453f,  0.634393284f,
    0.774953107f,  0.632018736f,
    0.776888466f,  0.629638239f,
    0.778816512f,  0.627251815f,
    0.780737229f,  0.624859488f,
    0.782650596f,  0.622461279f,
    0.784556597f,  0.620057212f,
    0.786455214f,  0.617647308f,
    0.788346428f,  0.615231591f,
    0.790230221f,  0.612810082f,
    0.792106577f,  0.610382806f,
    0.793975478f,  0.607949785f,
    0.795836905f,  0.605511041f,
    0.797690841f,  0.603066599f,
    0.799537269f,  0.600616479f,
    0.801376172f,  0.598160707f,
    0.803207531f,  0.595699304f,
    0.805031331f,  0.593232295f,
    0.806847554f,  0.590759702f,
    0.808656182f,  0.588281548f,
    0.810457198f,  0.585797857f,
    0.812250587f,  0.583308653f,
    0.814036330f,  0.580813958f,
    0.815814411f,  0.578313796f,
    0.817584813f,  0.575808191f,
    0.819347520f,  0.573297167f,
    0.821102515f,  0.570780746f,
    0.822849781f,  0.568258953f,
    0.824589303f,  0.565731811f,
    0.826321063f,  0.563199344f,
    0.828045045f,  0.560661576f,
    0.829761234f,  0.558118531f,
    0.831469612f,  0.555570233f,
    0.833170165f,  0.553016706f,
    0.834862875f,  0.550457973f,
    0.836547727f,  0.547894059f,
    0.838224706f,  0.545324988f,
    0.839893794f,  0.542750785f,
    0.841554977f,  0.540171473f,
    0.843208240f,  0.537587076f,
    0.844853565f,  0.534997620f,
    0.846490939f,  0.532403128f,
    0.848120345f,  0.529803625f,
    0.849741768f,  0.527199135f,
    0.851355193f,  0.524589683f,
    0.852960605f,  0.521975293f,
    0.854557988f,  0.519355990f,
    0.856147328f,  0.516731799f,
    0.857728610f,  0.514102744f,
    0.859301818f,  0.511468850f,
    0.860866939f,  0.508830143f,
    0.862423956f,  0.506186645f,
    0.863972856f,  0.503538384f,
    0.865513624f,  0.500885383f,
    0.867046246f,  0.498227667f,
    0.868570706f,  0.495565262f,
    0.870086991f,  0.492898192f,
    0.871595087f,  0.490226483f,
    0.873094978f,  0.487550160f,
    0.874586652f,  0.484869248f,
    0.876070094f,  0.482183772f,
    0.877545290f,  0.479493758f,
    0.879012226f,  0.476799230f,
    0.880470889f,  0.474100215f,
    0.881921264f,  0.471396737f,
    0.883363339f,  0.468688822f,
    0.884797098f,  0.465976496f,
    0.886222530f,  0.463259784f,
    0.887639620f,  0.460538711f,
    0.889048356f,  0.457813304f,
    0.890448723f,  0.455083587f,
    0.891840709f,  0.452349587f,
    0.893224301f,  0.449611330f,
    0.894599486f,  0.446868840f,
    0.895966250f,  0.444122145f,
    0.897324581f,  0.441371269f,
    0.898674466f,  0.438616239f,
    0.900015892f,  0.435857080f,
    0.901348847f,  0.433093819f,
    0.902673318f,  0.430326481f,
    0.903989293f,  0.427555093f,
    0.905296759f,  0.424779681f,
    0.906595705f,  0.422000271f,
    0.907886116f,  0.419216888f,
    0.909167983f,  0.416429560f,
    0.910441292f,  0.413638312f,
    0.911706032f,  0.410843171f,
    0.912962190f,  0.408044163f,
    0.914209756f,  0.405241314f,
    0.915448716f,  0.402434651f,
    0.916679060f,  0.399624200f,
    0.917900776f,  0.396809987f,
    0.919113852f,  0.393992040f,
    0.920318277f,  0.391170384f,
    0.921514039f,  0.388345047f,
    0.922701128f,  0.385516054f,
    0.923879533f,  0.382683432f,
    0.925049241f,  0.379847209f,
    0.926210242f,  0.377007410f,
    0.927362526f,  0.374164063f,
    0.928506080f,  0.371317194f,
    0.929640896f,  0.368466830f,
    0.930766961f,  0.365612998f,
    0.931884266f,  0.362755724f,
    0.932992799f,  0.359895037f,
    0.934092550f,  0.357030961f,
    0.935183510f,  0.354163525f,
    0.936265667f,  0.351292756f,
    0.937339012f,  0.348418680f,
    0.938403534f,  0.345541325f,
    0.939459224f,  0.342660717f,
    0.940506071f,  0.339776884f,
    0.941544065f,  0.336889853f,
    0.942573198f,  0.333999651f,
    0.943593458f,  0.331106306f,
    0.944604837f,  0.328209844f,
    0.945607325f,  0.325310292f,
    0.946600913f,  0.322407679f,
    0.947585591f,  0.319502031f,
    0.948561350f,  0.316593376f,
    0.949528181f,  0.313681740f,
    0.950486074f,  0.310767153f,
    0.951435021f,  0.307849640f,
    0.952375013f,  0.304929230f,
    0.953306040f,  0.302005949f,
    0.954228095f,  0.299079826f,
    0.955141168f,  0.296150888f,
    0.956045251f,  0.293219163f,
    0.956940336f,  0.290284677f,
    0.957826413f,  0.287347460f,
    0.958703475f,  0.284407537f,
    0.959571513f,  0.281464938f,
    0.960430519f,  0.278519689f,
    0.961280486f,  0.275571819f,
    0.962121404f,  0.272621355f,
    0.962953267f,  0.269668326f,
    0.963776066f,  0.266712757f,
    0.964589793f,  0.263754679f,
    0.965394442f,  0.260794118f,
    0.966190003f,  0.257831102f,
    0.966976471f,  0.254865660f,
    0.967753837f,  0.251897818f,
    0.968522094f,  0.248927606f,
    0.969281235f,  0.245955050f,
    0.970031253f,  0.242980180f,
    0.970772141f,  0.240003022f,
    0.971503891f,  0.237023606f,
    0.972226497f,  0.234041959f,
    0.972939952f,  0.231058108f,
    0.973644250f,  0.228072083f,
    0.974339383f,  0.225083911f,
    0.975025345f,  0.222093621f,
    0.975702130f,  0.219101240f,
    0.976369731f,  0.216106797f,
    0.977028143f,  0.213110320f,
    0.977677358f,  0.210111837f,
    0.978317371f,  0.207111376f,
    0.978948175f,  0.204108966f,
    0.979569766f,  0.201104635f,
    0.980182136f,  0.198098411f,
    0.980785280f,  0.195090322f,
    0.981379193f,  0.192080397f,
    0.981963869f,  0.189068664f,
    0.982539302f,  0.186055152f,
    0.983105487f,  0.183039888f,
    0.983662419f,  0.180022901f,
    0.984210092f,  0.177004220f,
    0.984748502f,  0.173983873f,
    0.985277642f,  0.170961889f,
    0.985797509f,  0.167938295f,
    0.986308097f,  0.164913120f,
    0.986809402f,  0.161886394f,
    0.987301418f,  0.158858143f,
    0.987784142f,  0.155828398f,
    0.988257568f,  0.152797185f,
    0.988721692f,  0.149764535f,
    0.989176510f,  0.146730474f,
    0.989622017f,  0.143695033f,
    0.990058210f,  0.140658239f,
    0.990485084f,  0.137620122f,
    0.990902635f,  0.134580709f,
    0.991310860f,  0.131540029f,
    0.991709754f,  0.128498111f,
    0.992099313f,  0.125454983f,
    0.992479535f,  0.122410675f,
    0.992850414f,  0.119365215f,
    0.993211949f,  0.116318631f,
    0.993564136f,  0.113270952f,
    0.993906970f,  0.110222207f,
    0.994240449f,  0.107172425f,
    0.994564571f,  0.104121634f,
    0.994879331f,  0.101069863f,
    0.995184727f,  0.098017140f,
    0.995480755f,  0.094963495f,
    0.995767414f,  0.091908956f,
    0.996044701f,  0.088853553f,
    0.996312612f,  0.085797312f,
    0.996571146f,  0.082740265f,
    0.996820299f,  0.079682438f,
    0.997060070f,  0.076623861f,
    0.997290457f,  0.073564564f,
    0.997511456f,  0.070504573f,
    0.997723067f,  0.067443920f,
    0.997925286f,  0.064382631f,
    0.998118113f,  0.061320736f,
    0.998301545f,  0.058258265f,
    0.998475581f,  0.055195244f,
    0.998640218f,  0.052131705f,
    0.998795456f,  0.049067674f,
    0.998941293f,  0.046003182f,
    0.999077728f,  0.042938257f,
    0.999204759f,  0.039872928f,
    0.999322385f,  0.036807223f,
    0.999430605f,  0.033741172f,
    0.999529418f,  0.030674803f,
    0.999618822f,  0.027608146f,
    0.999698819f,  0.024541229f,
    0.999769405f,  0.021474080f,
    0.999830582f,  0.018406730f,
    0.999882347f,  0.015339206f,
    0.999924702f,  0.012271538f,
    0.999957645f,  0.009203755f,
    0.999981175f,  0.006135885f,
    0.999995294f,  0.003067957f,
    1.000000000f,  0.000000000f,
    0.999995294f, -0.003067957f,
    0.999981175f, -0.006135885f,
    0.999957645f, -0.009203755f,
    0.999924702f, -0.012271538f,
    0.999882347f, -0.015339206f,
    0.999830582f, -0.018406730f,
    0.999769405f, -0.021474080f,
    0.999698819f, -0.024541229f,
    0.999618822f, -0.027608146f,
    0.999529418f, -0.030674803f,
    0.999430605f, -0.033741172f,
    0.999322385f, -0.036807223f,
    0.999204759f, -0.039872928f,
    0.999077728f, -0.042938257f,
    0.998941293f, -0.046003182f,
    0.998795456f, -0.049067674f,
    0.998640218f, -0.052131705f,
    0.998475581f, -0.055195244f,
    0.998301545f, -0.058258265f,
    0.998118113f, -0.061320736f,
    0.997925286f, -0.064382631f,
    0.997723067f, -0.067443920f,
    0.997511456f, -0.070504573f,
    0.997290457f, -0.073564564f,
    0.997060070f, -0.076623861f,
    0.996820299f, -0.079682438f,
    0.996571146f, -0.082740265f,
    0.996312612f, -0.085797312f,
    0.996044701f, -0.088853553f,
    0.995767414f, -0.091908956f,
    0.995480755f, -0.094963495f,
    0.995184727f, -0.098017140f,
    0.994879331f, -0.101069863f,
    0.994564571f, -0.104121634f,
    0.994240449f, -0.107172425f,
    0.993906970f, -0.110222207f,
    0.993564136f, -0.113270952f,
    0.993211949f, -0.116318631f,
    0.992850414f, -0.119365215f,
    0.992479535f, -0.122410675f,
    0.992099313f, -0.125454983f,
    0.991709754f, -0.128498111f,
    0.991310860f, -0.131540029f,
    0.990902635f, -0.134580709f,
    0.990485084f, -0.137620122f,
    0.990058210f, -0.140658239f,
    0.989622017f, -0.143695033f,
    0.989176510f, -0.146730474f,
    0.988721692f, -0.149764535f,
    0.988257568f, -0.152797185f,
    0.987784142f, -0.155828398f,
    0.987301418f, -0.158858143f,
    0.986809402f, -0.161886394f,
    0.986308097f, -0.164913120f,
    0.985797509f, -0.167938295f,
    0.985277642f, -0.170961889f,
    0.984748502f, -0.173983873f,
    0.984210092f, -0.177004220f,
    0.983662419f, -0.180022901f,
    0.983105487f, -0.183039888f,
    0.982539302f, -0.186055152f,
    0.981963869f, -0.189068664f,
    0.981379193f, -0.192080397f,
    0.980785280f, -0.195090322f,
    0.980182136f, -0.198098411f,
    0.979569766f, -0.201104635f,
    0.978948175f, -0.204108966f,
    0.978317371f, -0.207111376f,
    0.977677358f, -0.210111837f,
    0.977028143f, -0.213110320f,
    0.976369731f, -0.216106797f,
    0.975702130f, -0.219101240f,
    0.975025345f, -0.222093621f,
    0.974339383f, -0.225083911f,
    0.973644250f, -0.228072083f,
    0.972939952f, -0.231058108f,
    0.972226497f, -0.234041959f,
    0.971503891f, -0.237023606f,
    0.970772141f, -0.240003022f,
    0.970031253f, -0.242980180f,
    0.969281235f, -0.245955050f,
    0.968522094f, -0.248927606f,
    0.967753837f, -0.251897818f,
    0.966976471f, -0.254865660f,
    0.966190003f, -0.257831102f,
    0.965394442f, -0.260794118f,
    0.964589793f, -0.263754679f,
    0.963776066f, -0.266712757f,
    0.962953267f, -0.269668326f,
    0.962121404f, -0.272621355f,
    0.961280486f, -0.275571819f,
    0.960430519f, -0.278519689f,
    0.959571513f, -0.281464938f,
    0.958703475f, -0.284407537f,
    0.957826413f, -0.287347460f,
    0.956940336f, -0.290284677f,
    0.956045251f, -0.293219163f,
    0.955141168f, -0.296150888f,
    0.954228095f, -0.299079826f,
    0.953306040f, -0.302005949f,
    0.952375013f, -0.304929230f,
    0.951435021f, -0.307849640f,
    0.950486074f, -0.310767153f,
    0.949528181f, -0.313681740f,
    0.948561350f, -0.316593376f,
    0.947585591f, -0.319502031f,
    0.946600913f, -0.322407679f,
    0.945607325f, -0.325310292f,
    0.944604837f, -0.328209844f,
    0.943593458f, -0.331106306f,
    0.942573198f, -0.333999651f,
    0.941544065f, -0.336889853f,
    0.940506071f, -0.339776884f,
    0.939459224f, -0.342660717f,
    0.938403534f, -0.345541325f,
    0.937339012f, -0.348418680f,
    0.936265667f, -0.351292756f,
    0.935183510f, -0.354163525f,
    0.934092550f, -0.357030961f,
    0.932992799f, -0.359895037f,
    0.931884266f, -0.362755724f,
    0.930766961f, -0.365612998f,
    0.929640896f, -0.368466830f,
    0.928506080f, -0.371317194f,
    0.927362526f, -0.374164063f,
    0.926210242f, -0.377007410f,
    0.925049241f, -0.379847209f,
    0.923879533f, -0.382683432f,
    0.922701128f, -0.385516054f,
    0.921514039f, -0.388345047f,
    0.920318277f, -0.391170384f,
    0.919113852f, -0.393992040f,
    0.917900776f, -0.396809987f,
    0.916679060f, -0.399624200f,
    0.915448716f, -0.402434651f,
    0.914209756f, -0.405241314f,
    0.912962190f, -0.408044163f,
    0.911706032f, -0.410843171f,
    0.910441292f, -0.413638312f,
    0.909167983f, -0.416429560f,
    0.907886116f, -0.419216888f,
    0.906595705f, -0.422000271f,
    0.905296759f, -0.424779681f,
    0.903989293f, -0.427555093f,
    0.902673318f, -0.430326481f,
    0.901348847f, -0.433093819f,
    0.900015892f, -0.435857080f,
    0.898674466f, -0.438616239f,
    0.897324581f, -0.441371269f,
    0.895966250f, -0.444122145f,
    0.894599486f, -0.446868840f,
    0.893224301f, -0.449611330f,
    0.891840709f, -0.452349587f,
    0.890448723f, -0.455083587f,
    0.889048356f, -0.457813304f,
    0.887639620f, -0.460538711f,
    0.886222530f, -0.463259784f,
    0.884797098f, -0.465976496f,
    0.883363339f, -0.468688822f,
    0.881921264f, -0.471396737f,
    0.880470889f, -0.474100215f,
    0.879012226f, -0.476799230f,
    0.877545290f, -0.479493758f,
    0.876070094f, -0.482183772f,
    0.874586652f, -0.484869248f,
    0.873094978f, -0.487550160f,
    0.871595087f, -0.490226483f,
    0.870086991f, -0.492898192f,
    0.868570706f, -0.495565262f,
    0.867046246f, -0.498227667f,
    0.865513624f, -0.500885383f,
    0.863972856f, -0.503538384f,
    0.862423956f, -0.506186645f,
    0.860866939f, -0.508830143f,
    0.859301818f, -0.511468850f,
    0.857728610f, -0.514102744f,
    0.856147328f, -0.516731799f,
    0.854557988f, -0.519355990f,
    0.852960605f, -0.521975293f,
    0.851355193f, -0.524589683f,
    0.849741768f, -0.527199135f,
    0.848120345f, -0.529803625f,
    0.846490939f, -0.532403128f,
    0.844853565f, -0.534997620f,
    0.843208240f, -0.537587076f,
    0.841554977f, -0.540171473f,
    0.839893794f, -0.542750785f,
    0.838224706f, -0.545324988f,
    0.836547727f, -0.547894059f,
    0.834862875f, -0.550457973f,
    0.833170165f, -0.553016706f,
    0.831469612f, -0.555570233f,
    0.829761234f, -0.558118531f,
    0.828045045f, -0.560661576f,
    0.826321063f, -0.563199344f,
    0.824589303f, -0.565731811f,
    0.822849781f, -0.568258953f,
    0.821102515f, -0.570780746f,
    0.819347520f, -0.573297167f,
    0.817584813f, -0.575808191f,
    0.815814411f, -0.578313796f,
    0.814036330f, -0.580813958f,
    0.812250587f, -0.583308653f,
    0.810457198f, -0.585797857f,
    0.808656182f, -0.588281548f,
    0.806847554f, -0.590759702f,
    0.805031331f, -0.593232295f,
    0.803207531f, -0.595699304f,
    0.801376172f, -0.598160707f,
    0.799537269f, -0.600616479f,
    0.797690841f, -0.603066599f,
    0.795836905f, -0.605511041f,
    0.793975478f, -0.607949785f,
    0.792106577f, -0.610382806f,
    0.790230221f, -0.612810082f,
    0.788346428f, -0.615231591f,
    0.786455214f, -0.617647308f,
    0.784556597f, -0.620057212f,
    0.782650596f, -0.622461279f,
    0.780737229f, -0.624859488f,
    0.778816512f, -0.627251815f,
    0.776888466f, -0.629638239f,
    0.774953107f, -0.632018736f,
    0.773010453f, -0.634393284f,
    0.771060524f, -0.636761861f,
    0.769103338f, -0.639124445f,
    0.767138912f, -0.641481013f,
    0.765167266f, -0.643831543f,
    0.763188417f, -0.646176013f,
    0.761202385f, -0.648514401f,
    0.759209189f, -0.650846685f,
    0.757208847f, -0.653172843f,
    0.755201377f, -0.655492853f,
    0.753186799f, -0.657806693f,
    0.751165132f, -0.660114342f,
    0.749136395f, -0.662415778f,
    0.747100606f, -0.664710978f,
    0.745057785f, -0.666999922f,
    0.743007952f, -0.669282588f,
    0.740951125f, -0.671558955f,
    0.738887324f, -0.673829000f,
    0.736816569f, -0.676092704f,
    0.734738878f, -0.678350043f,
    0.732654272f, -0.680600998f,
    0.730562769f, -0.682845546f,
    0.728464390f, -0.685083668f,
    0.726359155f, -0.687315341f,
    0.724247083f, -0.689540545f,
    0.722128194f, -0.691759258f,
    0.720002508f, -0.693971461f,
    0.717870045f, -0.696177131f,
    0.715730825f, -0.698376249f,
    0.713584869f, -0.700568794f,
    0.711432196f, -0.702754744f,
    0.709272826f, -0.704934080f,
    0.707106781f, -0.707106781f,
    0.704934080f, -0.709272826f,
    0.702754744f, -0.711432196f,
    0.700568794f, -0.713584869f,
    0.698376249f, -0.715730825f,
    0.696177131f, -0.717870045f,
    0.693971461f, -0.720002508f,
    0.691759258f, -0.722128194f,
    0.689540545f, -0.724247083f,
    0.687315341f, -0.726359155f,
    0.685083668f, -0.728464390f,
    0.682845546f, -0.730562769f,
    0.680600998f, -0.732654272f,
    0.678350043f, -0.734738878f,
    0.676092704f, -0.736816569f,
    0.673829000f, -0.738887324f,
    0.671558955f, -0.740951125f,
    0.669282588f, -0.743007952f,
    0.666999922f, -0.745057785f,
    0.664710978f, -0.747100606f,
    0.662415778f, -0.749136395f,
    0.660114342f, -0.751165132f,
    0.657806693f, -0.753186799f,
    0.655492853f, -0.755201377f,
    0.653172843f, -0.757208847f,
    0.650846685f, -0.759209189f,
    0.648514401f, -0.761202385f,
    0.646176013f, -0.763188417f,
    0.643831543f, -0.765167266f,
    0.641481013f, -0.767138912f,
    0.639124445f, -0.769103338f,
    0.636761861f, -0.771060524f,
    0.634393284f, -0.773010453f,
    0.632018736f, -0.774953107f,
    0.629638239f, -0.776888466f,
    0.627251815f, -0.778816512f,
    0.624859488f, -0.780737229f,
    0.622461279f, -0.782650596f,
    0.620057212f, -0.784556597f,
    0.617647308f, -0.786455214f,
    0.615231591f, -0.788346428f,
    0.612810082f, -0.790230221f,
    0.610382806f, -0.792106577f,
    0.607949785f, -0.793975478f,
    0.605511041f, -0.795836905f,
    0.603066599f, -0.797690841f,
    0.600616479f, -0.799537269f,
    0.598160707f, -0.801376172f,
    0.595699304f, -0.803207531f,
    0.593232295f, -0.805031331f,
    0.590759702f, -0.806847554f,
    0.588281548f, -0.808656182f,
    0.585797857f, -0.810457198f,
    0.583308653f, -0.812250587f,
    0.580813958f, -0.814036330f,
    0.578313796f, -0.815814411f,
    0.575808191f, -0.817584813f,
    0.573297167f, -0.819347520f,
    0.570780746f, -0.821102515f,
    0.568258953f, -0.822849781f,
    0.565731811f, -0.824589303f,
    0.563199344f, -0.826321063f,
    0.560661576f, -0.828045045f,
    0.558118531f, -0.829761234f,
    0.555570233f, -0.831469612f,
    0.553016706f, -0.833170165f,
    0.550457973f, -0.834862875f,
    0.547894059f, -0.836547727f,
    0.545324988f, -0.838224706f,
    0.542750785f, -0.839893794f,
    0.540171473f, -0.841554977f,
    0.537587076f, -0.843208240f,
    0.534997620f, -0.844853565f,
    0.532403128f, -0.846490939f,
    0.529803625f, -0.848120345f,
    0.527199135f, -0.849741768f,
    0.524589683f, -0.851355193f,
    0.521975293f, -0.852960605f,
    0.519355990f, -0.854557988f,
    0.516731799f, -0.856147328f,
    0.514102744f, -0.857728610f,
    0.511468850f, -0.859301818f,
    0.508830143f, -0.860866939f,
    0.506186645f, -0.862423956f,
    0.503538384f, -0.863972856f,
    0.500885383f, -0.865513624f,
    0.498227667f, -0.867046246f,
    0.495565262f, -0.868570706f,
    0.492898192f, -0.870086991f,
    0.490226483f, -0.871595087f,
    0.487550160f, -0.873094978f,
    0.484869248f, -0.874586652f,
    0.482183772f, -0.876070094f,
    0.479493758f, -0.877545290f,
    0.476799230f, -0.879012226f,
    0.474100215f, -0.880470889f,
    0.471396737f, -0.881921264f,
    0.468688822f, -0.883363339f,
    0.465976496f, -0.884797098f,
    0.463259784f, -0.886222530f,
    0.460538711f, -0.887639620f,
    0.457813304f, -0.889048356f,
    0.455083587f, -0.890448723f,
    0.452349587f, -0.891840709f,
    0.449611330f, -0.893224301f,
    0.446868840f, -0.894599486f,
    0.444122145f, -0.895966250f,
    0.441371269f, -0.897324581f,
    0.438616239f, -0.898674466f,
    0.435857080f, -0.900015892f,
    0.433093819f, -0.901348847f,
    0.430326481f, -0.902673318f,
    0.427555093f, -0.903989293f,
    0.424779681f, -0.905296759f,
    0.422000271f, -0.906595705f,
    0.419216888f, -0.907886116f,
    0.416429560f, -0.909167983f,
    0.413638312f, -0.910441292f,
    0.410843171f, -0.911706032f,
    0.408044163f, -0.912962190f,
    0.405241314f, -0.914209756f,
    0.402434651f, -0.915448716f,
    0.399624200f, -0.916679060f,
    0.396809987f, -0.917900776f,
    0.393992040f, -0.919113852f,
    0.391170384f, -0.920318277f,
    0.388345047f, -0.921514039f,
    0.385516054f, -0.922701128f,
    0.382683432f, -0.923879533f,
    0.379847209f, -0.925049241f,
    0.377007410f, -0.926210242f,
    0.374164063f, -0.927362526f,
    0.371317194f, -0.928506080f,
    0.368466830f, -0.929640896f,
    0.365612998f, -0.930766961f,
    0.362755724f, -0.931884266f,
    0.359895037f, -0.932992799f,
    0.357030961f, -0.934092550f,
    0.354163525f, -0.935183510f,
    0.351292756f, -0.936265667f,
    0.348418680f, -0.937339012f,
    0.345541325f, -0.938403534f,
    0.342660717f, -0.939459224f,
    0.339776884f, -0.940506071f,
    0.336889853f, -0.941544065f,
    0.333999651f, -0.942573198f,
    0.331106306f, -0.943593458f,
    0.328209844f, -0.944604837f,
    0.325310292f, -0.945607325f,
    0.322407679f, -0.946600913f,
    0.319502031f, -0.947585591f,
    0.316593376f, -0.948561350f,
    0.313681740f, -0.949528181f,
    0.310767153f, -0.950486074f,
    0.307849640f, -0.951435021f,
    0.304929230f, -0.952375013f,
    0.302005949f, -0.953306040f,
    0.299079826f, -0.954228095f,
    0.296150888f, -0.955141168f,
    0.293219163f, -0.956045251f,
    0.290284677f, -0.956940336f,
    0.287347460f, -0.957826413f,
    0.284407537f, -0.958703475f,
    0.281464938f, -0.959571513f,
    0.278519689f, -0.960430519f,
    0.275571819f, -0.961280486f,
    0.272621355f, -0.962121404f,
    0.269668326f, -0.962953267f,
    0.266712757f, -0.963776066f,
    0.263754679f, -0.964589793f,
    0.260794118f, -0.965394442f,
    0.257831102f, -0.966190003f,
    0.254865660f, -0.966976471f,
    0.251897818f, -0.967753837f,
    0.248927606f, -0.968522094f,
    0.245955050f, -0.969281235f,
    0.242980180f, -0.970031253f,
    0.240003022f, -0.970772141f,
    0.237023606f, -0.971503891f,
    0.234041959f, -0.972226497f,
    0.231058108f, -0.972939952f,
    0.228072083f, -0.973644250f,
    0.225083911f, -0.974339383f,
    0.222093621f, -0.975025345f,
    0.219101240f, -0.975702130f,
    0.216106797f, -0.976369731f,
    0.213110320f, -0.977028143f,
    0.210111837f, -0.977677358f,
    0.207111376f, -0.978317371f,
    0.204108966f, -0.978948175f,
    0.201104635f, -0.979569766f,
    0.198098411f, -0.980182136f,
    0.195090322f, -0.980785280f,
    0.192080397f, -0.981379193f,
    0.189068664f, -0.981963869f,
    0.186055152f, -0.982539302f,
    0.183039888f, -0.983105487f,
    0.180022901f, -0.983662419f,
    0.177004220f, -0.984210092f,
    0.173983873f, -0.984748502f,
    0.170961889f, -0.985277642f,
    0.167938295f, -0.985797509f,
    0.164913120f, -0.986308097f,
    0.161886394f, -0.986809402f,
    0.158858143f, -0.987301418f,
    0.155828398f, -0.987784142f,
    0.152797185f, -0.988257568f,
    0.149764535f, -0.988721692f,
    0.146730474f, -0.989176510f,
    0.143695033f, -0.989622017f,
    0.140658239f, -0.990058210f,
    0.137620122f, -0.990485084f,
    0.134580709f, -0.990902635f,
    0.131540029f, -0.991310860f,
    0.128498111f, -0.991709754f,
    0.125454983f, -0.992099313f,
    0.122410675f, -0.992479535f,
    0.119365215f, -0.992850414f,
    0.116318631f, -0.993211949f,
    0.113270952f, -0.993564136f,
    0.110222207f, -0.993906970f,
    0.107172425f, -0.994240449f,
    0.104121634f, -0.994564571f,
    0.101069863f, -0.994879331f,
    0.098017140f, -0.995184727f,
    0.094963495f, -0.995480755f,
    0.091908956f, -0.995767414f,
    0.088853553f, -0.996044701f,
    0.085797312f, -0.996312612f,
    0.082740265f, -0.996571146f,
    0.079682438f, -0.996820299f,
    0.076623861f, -0.997060070f,
    0.073564564f, -0.997290457f,
    0.070504573f, -0.997511456f,
    0.067443920f, -0.997723067f,
    0.064382631f, -0.997925286f,
    0.061320736f, -0.998118113f,
    0.058258265f, -0.998301545f,
    0.055195244f, -0.998475581f,
    0.052131705f, -0.998640218f,
    0.049067674f, -0.998795456f,
    0.046003182f, -0.998941293f,
    0.042938257f, -0.999077728f,
    0.039872928f, -0.999204759f,
    0.036807223f, -0.999322385f,
    0.033741172f, -0.999430605f,
    0.030674803f, -0.999529418f,
    0.027608146f, -0.999618822f,
    0.024541229f, -0.999698819f,
    0.021474080f, -0.999769405f,
    0.018406730f, -0.999830582f,
    0.015339206f, -0.999882347f,
    0.012271538f, -0.999924702f,
    0.009203755f, -0.999957645f,
    0.006135885f, -0.999981175f,
    0.003067957f, -0.999995294f
};

const float32_t twiddleCoef_rfft_4096[4096] = {
    0.000000000f,  1.000000000f,
    0.001533980f,  0.999998823f,
    0.003067957f,  0.999995294f,
    0.004601926f,  0.999989411f,
    0.006135885f,  0.999981175f,
    0.007669829f,  0.999970586f,
    0.009203755f,  0.999957645f,
    0.010737659f,  0.999942350f,
    0.012271538f,  0.999924702f,
    0.013805389f,  0.999904701f,
    0.015339206f,  0.999882347f,
    0.016872988f,  0.999857641f,
    0.018406730f,  0.999830582f,
    0.019940429f,  0.999801170f,
    0.021474080f,  0.999769405f,
    0.023007681f,  0.999735288f,
    0.024541229f,  0.999698819f,
    0.026074718f,  0.999659997f,
    0.027608146f,  0.999618822f,
    0.029141509f,  0.999575296f,
    0.030674803f,  0.999529418f,
    0.032208025f,  0.999481187f,
    0.033741172f,  0.999430605f,
    0.035274239f,  0.999377670f,
    0.036807223f,  0.999322385f,
    0.038340120f,  0.999264747f,
    0.039872928f,  0.999204759f,
    0.041405641f,  0.999142419f,
    0.042938257f,  0.999077728f,
    0.044470772f,  0.999010686f,
    0.046003182f,  0.998941293f,
    0.047535484f,  0.998869550f,
    0.049067674f,  0.998795456f,
    0.050599749f,  0.998719012f,
    0.052131705f,  0.998640218f,
    0.053663538f,  0.998559074f,
    0.055195244f,  0.998475581f,
    0.056726821f,  0.998389737f,
    0.058258265f,  0.998301545f,
    0.059789571f,  0.998211003f,
    0.061320736f,  0.998118113f,
    0.062851758f,  0.998022874f,
    0.064382631f,  0.997925286f,
    0.065913353f,  0.997825350f,
    0.067443920f,  0.997723067f,
    0.068974328f,  0.997618435f,
    0.070504573f,  0.997511456f,
    0.072034653f,  0.997402130f,
    0.073564564f,  0.997290457f,
    0.075094301f,  0.997176437f,
    0.076623861f,  0.997060070f,
    0.078153242f,  0.996941358f,
    0.079682438f,  0.996820299f,
    0.081211447f,  0.996696895f,
    0.082740265f,  0.996571146f,
    0.084268888f,  0.996443051f,
    0.085797312f,  0.996312612f,
    0.087325535f,  0.996179829f,
    0.088853553f,  0.996044701f,
    0.090381361f,  0.995907229f,
    0.091908956f,  0.995767414f,
    0.093436336f,  0.995625256f,
    0.094963495f,  0.995480755f,
    0.096490431f,  0.995333912f,
    0.098017140f,  0.995184727f,
    0.099543619f,  0.995033199f,
    0.101069863f,  0.994879331f,
    0.102595869f,  0.994723121f,
    0.104121634f,  0.994564571f,
    0.105647154f,  0.994403680f,
    0.107172425f,  0.994240449f,
    0.108697444f,  0.994074879f,
    0.110222207f,  0.993906970f,
    0.111746711f,  0.993736722f,
    0.113270952f,  0.993564136f,
    0.114794927f,  0.993389211f,
    0.116318631f,  0.993211949f,
    0.117842062f,  0.993032350f,
    0.119365215f,  0.992850414f,
    0.120888087f,  0.992666142f,
    0.122410675f,  0.992479535f,
    0.123932975f,  0.992290591f,
    0.125454983f,  0.992099313f,
    0.126976696f,  0.991905700f,
    0.128498111f,  0.991709754f,
    0.130019223f,  0.991511473f,
    0.131540029f,  0.991310860f,
    0.133060525f,  0.991107914f,
    0.134580709f,  0.990902635f,
    0.136100575f,  0.990695025f,
    0.137620122f,  0.990485084f,
    0.139139344f,  0.990272812f,
    0.140658239f,  0.990058210f,
    0.142176804f,  0.989841278f,
    0.143695033f,  0.989622017f,
    0.145212925f,  0.989400428f,
    0.146730474f,  0.989176510f,
    0.148247679f,  0.988950265f,
    0.149764535f,  0.988721692f,
    0.151281038f,  0.988490793f,
    0.152797185f,  0.988257568f,
    0.154312973f,  0.988022017f,
    0.155828398f,  0.987784142f,
    0.157343456f,  0.987543942f,
    0.158858143f,  0.987301418f,
    0.160372457f,  0.987056571f,
    0.161886394f,  0.986809402f,
    0.163399949f,  0.986559910f,
    0.164913120f,  0.986308097f,
    0.166425904f,  0.986053963f,
    0.167938295f,  0.985797509f,
    0.169450291f,  0.985538735f,
    0.170961889f,  0.985277642f,
    0.172473084f,  0.985014231f,
    0.173983873f,  0.984748502f,
    0.175494253f,  0.984480455f,
    0.177004220f,  0.984210092f,
    0.178513771f,  0.983937413f,
    0.180022901f,  0.983662419f,
    0.181531608f,  0.983385110f,
    0.183039888f,  0.983105487f,
    0.184547737f,  0.982823551f,
    0.186055152f,  0.982539302f,
    0.187562129f,  0.982252741f,
    0.189068664f,  0.981963869f,
    0.190574755f,  0.981672686f,
    0.192080397f,  0.981379193f,
    0.193585587f,  0.981083391f,
    0.195090322f,  0.980785280f,
    0.196594598f,  0.980484862f,
    0.198098411f,  0.980182136f,
    0.199601758f,  0.979877104f,
    0.201104635f,  0.979569766f,
    0.202607039f,  0.979260123f,
    0.204108966f,  0.978948175f,
    0.205610413f,  0.978633924f,
    0.207111376f,  0.978317371f,
    0.208611852f,  0.977998515f,
    0.210111837f,  0.977677358f,
    0.211611327f,  0.977353900f,
    0.213110320f,  0.977028143f,
    0.214608811f,  0.976700086f,
    0.216106797f,  0.976369731f,
    0.217604275f,  0.976037079f,
    0.219101240f,  0.975702130f,
    0.220597690f,  0.975364885f,
    0.222093621f,  0.975025345f,
    0.223589029f,  0.974683511f,
    0.225083911f,  0.974339383f,
    0.226578264f,  0.973992962f,
    0.228072083f,  0.973644250f,
    0.229565366f,  0.973293246f,
    0.231058108f,  0.972939952f,
    0.232550307f,  0.972584369f,
    0.234041959f,  0.972226497f,
    0.235533059f,  0.971866337f,
    0.237023606f,  0.971503891f,
    0.238513595f,  0.971139158f,
    0.240003022f,  0.970772141f,
    0.241491885f,  0.970402839f,
    0.242980180f,  0.970031253f,
    0.244467903f,  0.969657385f,
    0.245955050f,  0.969281235f,
    0.247441619f,  0.968902805f,
    0.248927606f,  0.968522094f,
    0.250413007f,  0.968139105f,
    0.251897818f,  0.967753837f,
    0.253382037f,  0.967366292f,
    0.254865660f,  0.966976471f,
    0.256348682f,  0.966584374f,
    0.257831102f,  0.966190003f,
    0.259312915f,  0.965793359f,
    0.260794118f,  0.965394442f,
    0.262274707f,  0.964993253f,
    0.263754679f,  0.964589793f,
    0.265234030f,  0.964184064f,
    0.266712757f,  0.963776066f,
    0.268190857f,  0.963365800f,
    0.269668326f,  0.962953267f,
    0.271145160f,  0.962538468f,
    0.272621355f,  0.962121404f,
    0.274096910f,  0.961702077f,
    0.275571819f,  0.961280486f,
    0.277046080f,  0.960856633f,
    0.278519689f,  0.960430519f,
    0.279992643f,  0.960002146f,
    0.281464938f,  0.959571513f,
    0.282936570f,  0.959138622f,
    0.284407537f,  0.958703475f,
    0.285877835f,  0.958266071f,
    0.287347460f,  0.957826413f,
    0.288816408f,  0.957384501f,
    0.290284677f,  0.956940336f,
    0.291752263f,  0.956493919f,
    0.293219163f,  0.956045251f,
    0.294685372f,  0.955594334f,
    0.296150888f,  0.955141168f,
    0.297615707f,  0.954685755f,
    0.299079826f,  0.954228095f,
    0.300543241f,  0.953768190f,
    0.302005949f,  0.953306040f,
    0.303467947f,  0.952841648f,
    0.304929230f,  0.952375013f,
    0.306389795f,  0.951906137f,
    0.307849640f,  0.951435021f,
    0.309308760f,  0.950961666f,
    0.310767153f,  0.950486074f,
    0.312224814f,  0.950008245f,
    0.313681740f,  0.949528181f,
    0.315137929f,  0.949045882f,
    0.316593376f,  0.948561350f,
    0.318048077f,  0.948074586f,
    0.319502031f,  0.947585591f,
    0.320955232f,  0.947094366f,
    0.322407679f,  0.946600913f,
    0.323859367f,  0.946105232f,
    0.325310292f,  0.945607325f,
    0.326760452f,  0.945107193f,
    0.328209844f,  0.944604837f,
    0.329658463f,  0.944100258f,
    0.331106306f,  0.943593458f,
    0.332553370f,  0.943084437f,
    0.333999651f,  0.942573198f,
    0.335445147f,  0.942059740f,
    0.336889853f,  0.941544065f,
    0.338333767f,  0.941026175f,
    0.339776884f,  0.940506071f,
    0.341219202f,  0.939983753f,
    0.342660717f,  0.939459224f,
    0.344101426f,  0.938932484f,
    0.345541325f,  0.938403534f,
    0.346980411f,  0.937872376f,
    0.348418680f,  0.937339012f,
    0.349856130f,  0.936803442f,
    0.351292756f,  0.936265667f,
    0.352728556f,  0.935725689f,
    0.354163525f,  0.935183510f,
    0.355597662f,  0.934639130f,
    0.357030961f,  0.934092550f,
    0.358463421f,  0.933543773f,
    0.359895037f,  0.932992799f,
    0.361325806f,  0.932439629f,
    0.362755724f,  0.931884266f,
    0.364184790f,  0.931326709f,
    0.365612998f,  0.930766961f,
    0.367040346f,  0.930205023f,
    0.368466830f,  0.929640896f,
    0.369892447f,  0.929074581f,
    0.371317194f,  0.928506080f,
    0.372741067f,  0.927935395f,
    0.374164063f,  0.927362526f,
    0.375586178f,  0.926787474f,
    0.377007410f,  0.926210242f,
    0.378427755f,  0.925630831f,
    0.379847209f,  0.925049241f,
    0.381265769f,  0.924465474f,
    0.382683432f,  0.923879533f,
    0.384100195f,  0.923291417f,
    0.385516054f,  0.922701128f,
    0.386931006f,  0.922108669f,
    0.388345047f,  0.921514039f,
    0.389758174f,  0.920917242f,
    0.391170384f,  0.920318277f,
    0.392581674f,  0.919717146f,
    0.393992040f,  0.919113852f,
    0.395401479f,  0.918508394f,
    0.396809987f,  0.917900776f,
    0.398217562f,  0.917290997f,
    0.399624200f,  0.916679060f,
    0.401029897f,  0.916064966f,
    0.402434651f,  0.915448716f,
    0.403838458f,  0.914830312f,
    0.405241314f,  0.914209756f,
    0.406643217f,  0.913587048f,
    0.408044163f,  0.912962190f,
    0.409444149f,  0.912335185f,
    0.410843171f,  0.911706032f,
    0.412241227f,  0.911074734f,
    0.413638312f,  0.910441292f,
    0.415034424f,  0.909805708f,
    0.416429560f,  0.909167983f,
    0.417823716f,  0.908528119f,
    0.419216888f,  0.907886116f,
    0.420609074f,  0.907241978f,
    0.422000271f,  0.906595705f,
    0.423390474f,  0.905947298f,
    0.424779681f,  0.905296759f,
    0.426167889f,  0.904644091f,
    0.427555093f,  0.903989293f,
    0.428941292f,  0.903332368f,
    0.430326481f,  0.902673318f,
    0.431710658f,  0.902012144f,
    0.433093819f,  0.901348847f,
    0.434475961f,  0.900683429f,
    0.435857080f,  0.900015892f,
    0.437237174f,  0.899346237f,
    0.438616239f,  0.898674466f,
    0.439994271f,  0.898000580f,
    0.441371269f,  0.897324581f,
    0.442747228f,  0.896646470f,
    0.444122145f,  0.895966250f,
    0.445496017f,  0.895283921f,
    0.446868840f,  0.894599486f,
    0.448240612f,  0.893912945f,
    0.449611330f,  0.893224301f,
    0.450980989f,  0.892533555f,
    0.452349587f,  0.891840709f,
    0.453717121f,  0.891145765f,
    0.455083587f,  0.890448723f,
    0.456448982f,  0.889749586f,
    0.457813304f,  0.889048356f,
    0.459176548f,  0.888345033f,
    0.460538711f,  0.887639620f,
    0.461899791f,  0.886932119f,
    0.463259784f,  0.886222530f,
    0.464618686f,  0.885510856f,
    0.465976496f,  0.884797098f,
    0.467333209f,  0.884081259f,
    0.468688822f,  0.883363339f,
    0.470043332f,  0.882643340f,
    0.471396737f,  0.881921264f,
    0.472749032f,  0.881197113f,
    0.474100215f,  0.880470889f,
    0.475450282f,  0.879742593f,
    0.476799230f,  0.879012226f,
    0.478147056f,  0.878279792f,
    0.479493758f,  0.877545290f,
    0.480839331f,  0.876808724f,
    0.482183772f,  0.876070094f,
    0.483527079f,  0.875329403f,
    0.484869248f,  0.874586652f,
    0.486210276f,  0.873841843f,
    0.487550160f,  0.873094978f,
    0.488888897f,  0.872346059f,
    0.490226483f,  0.871595087f,
    0.491562916f,  0.870842063f,
    0.492898192f,  0.870086991f,
    0.494232309f,  0.869329871f,
    0.495565262f,  0.868570706f,
    0.496897049f,  0.867809497f,
    0.498227667f,  0.867046246f,
    0.499557113f,  0.866280954f,
    0.500885383f,  0.865513624f,
    0.502212474f,  0.864744258f,
    0.503538384f,  0.863972856f,
    0.504863109f,  0.863199422f,
    0.506186645f,  0.862423956f,
    0.507508991f,  0.861646461f,
    0.508830143f,  0.860866939f,
    0.510150097f,  0.860085390f,
    0.511468850f,  0.859301818f,
    0.512786401f,  0.858516224f,
    0.514102744f,  0.857728610f,
    0.515417878f,  0.856938977f,
    0.516731799f,  0.856147328f,
    0.518044504f,  0.855353665f,
    0.519355990f,  0.854557988f,
    0.520666254f,  0.853760301f,
    0.521975293f,  0.852960605f,
    0.523283103f,  0.852158902f,
    0.524589683f,  0.851355193f,
    0.525895027f,  0.850549481f,
    0.527199135f,  0.849741768f,
    0.528502002f,  0.848932055f,
    0.529803625f,  0.848120345f,
    0.531104001f,  0.847306639f,
    0.532403128f,  0.846490939f,
    0.533701002f,  0.845673247f,
    0.534997620f,  0.844853565f,
    0.536292979f,  0.844031895f,
    0.537587076f,  0.843208240f,
    0.538879909f,  0.842382600f,
    0.540171473f,  0.841554977f,
    0.541461766f,  0.840725375f,
    0.542750785f,  0.839893794f,
    0.544038527f,  0.839060237f,
    0.545324988f,  0.838224706f,
    0.546610167f,  0.837387202f,
    0.547894059f,  0.836547727f,
    0.549176662f,  0.835706284f,
    0.550457973f,  0.834862875f,
    0.551737988f,  0.834017501f,
    0.553016706f,  0.833170165f,
    0.554294121f,  0.832320868f,
    0.555570233f,  0.831469612f,
    0.556845037f,  0.830616400f,
    0.558118531f,  0.829761234f,
    0.559390712f,  0.828904115f,
    0.560661576f,  0.828045045f,
    0.561931121f,  0.827184027f,
    0.563199344f,  0.826321063f,
    0.564466242f,  0.825456154f,
    0.565731811f,  0.824589303f,
    0.566996049f,  0.823720511f,
    0.568258953f,  0.822849781f,
    0.569520519f,  0.821977115f,
    0.570780746f,  0.821102515f,
    0.572039629f,  0.820225983f,
    0.573297167f,  0.819347520f,
    0.574553355f,  0.818467130f,
    0.575808191f,  0.817584813f,
    0.577061673f,  0.816700573f,
    0.578313796f,  0.815814411f,
    0.579564559f,  0.814926329f,
    0.580813958f,  0.814036330f,
    0.582061990f,  0.813144415f,
    0.583308653f,  0.812250587f,
    0.584553943f,  0.811354847f,
    0.585797857f,  0.810457198f,
    0.587040394f,  0.809557642f,
    0.588281548f,  0.808656182f,
    0.589521319f,  0.807752818f,
    0.590759702f,  0.806847554f,
    0.591996695f,  0.805940391f,
    0.593232295f,  0.805031331f,
    0.594466499f,  0.804120377f,
    0.595699304f,  0.803207531f,
    0.596930708f,  0.802292796f,
    0.598160707f,  0.801376172f,
    0.599389298f,  0.800457662f,
    0.600616479f,  0.799537269f,
    0.601842247f,  0.798614995f,
    0.603066599f,  0.797690841f,
    0.604289531f,  0.796764810f,
    0.605511041f,  0.795836905f,
    0.606731127f,  0.794907126f,
    0.607949785f,  0.793975478f,
    0.609167012f,  0.793041960f,
    0.610382806f,  0.792106577f,
    0.611597164f,  0.791169330f,
    0.612810082f,  0.790230221f,
    0.614021559f,  0.789289253f,
    0.615231591f,  0.788346428f,
    0.616440175f,  0.787401747f,
    0.617647308f,  0.786455214f,
    0.618852988f,  0.785506830f,
    0.620057212f,  0.784556597f,
    0.621259977f,  0.783604519f,
    0.622461279f,  0.782650596f,
    0.623661118f,  0.781694832f,
    0.624859488f,  0.780737229f,
    0.626056388f,  0.779777788f,
    0.627251815f,  0.778816512f,
    0.628445767f,  0.777853404f,
    0.629638239f,  0.776888466f,
    0.630829230f,  0.775921699f,
    0.632018736f,  0.774953107f,
    0.633206755f,  0.773982691f,
    0.634393284f,  0.773010453f,
    0.635578320f,  0.772036397f,
    0.636761861f,  0.771060524f,
    0.637943904f,  0.770082837f,
    0.639124445f,  0.769103338f,
    0.640303482f,  0.768122029f,
    0.641481013f,  0.767138912f,
    0.642657034f,  0.766153990f,
    0.643831543f,  0.765167266f,
    0.645004537f,  0.764178741f,
    0.646176013f,  0.763188417f,
    0.647345969f,  0.762196298f,
    0.648514401f,  0.761202385f,
    0.649681307f,  0.760206682f,
    0.650846685f,  0.759209189f,
    0.652010531f,  0.758209910f,
    0.653172843f,  0.757208847f,
    0.654333618f,  0.756206001f,
    0.655492853f,  0.755201377f,
    0.656650546f,  0.754194975f,
    0.657806693f,  0.753186799f,
    0.658961293f,  0.752176850f,
    0.660114342f,  0.751165132f,
    0.661265838f,  0.750151646f,
    0.662415778f,  0.749136395f,
    0.663564159f,  0.748119380f,
    0.664710978f,  0.747100606f,
    0.665856234f,  0.746080074f,
    0.666999922f,  0.745057785f,
    0.668142041f,  0.744033744f,
    0.669282588f,  0.743007952f,
    0.670421560f,  0.741980412f,
    0.671558955f,  0.740951125f,
    0.672694769f,  0.739920095f,
    0.673829000f,  0.738887324f,
    0.674961646f,  0.737852815f,
    0.676092704f,  0.736816569f,
    0.677222170f,  0.735778589f,
    0.678350043f,  0.734738878f,
    0.679476320f,  0.733697438f,
    0.680600998f,  0.732654272f,
    0.681724074f,  0.731609381f,
    0.682845546f,  0.730562769f,
    0.683965412f,  0.729514438f,
    0.685083668f,  0.728464390f,
    0.686200312f,  0.727412629f,
    0.687315341f,  0.726359155f,
    0.688428753f,  0.725303972f,
    0.689540545f,  0.724247083f,
    0.690650714f,  0.723188489f,
    0.691759258f,  0.722128194f,
    0.692866175f,  0.721066199f,
    0.693971461f,  0.720002508f,
    0.695075114f,  0.718937122f,
    0.696177131f,  0.717870045f,
    0.697277511f,  0.716801279f,
    0.698376249f,  0.715730825f,
    0.699473345f,  0.714658688f,
    0.700568794f,  0.713584869f,
    0.701662595f,  0.712509371f,
    0.702754744f,  0.711432196f,
    0.703845241f,  0.710353347f,
    0.704934080f,  0.709272826f,
    0.706021261f,  0.708190637f,
    0.707106781f,  0.707106781f,
    0.708190637f,  0.706021261f,
    0.709272826f,  0.704934080f,
    0.710353347f,  0.703845241f,
    0.711432196f,  0.702754744f,
    0.712509371f,  0.701662595f,
    0.713584869f,  0.700568794f,
    0.714658688f,  0.699473345f,
    0.715730825f,  0.698376249f,
    0.716801279f,  0.697277511f,
    0.717870045f,  0.696177131f,
    0.718937122f,  0.695075114f,
    0.720002508f,  0.693971461f,
    0.721066199f,  0.692866175f,
    0.722128194f,  0.691759258f,
    0.723188489f,  0.690650714f,
    0.724247083f,  0.689540545f,
    0.725303972f,  0.688428753f,
    0.726359155f,  0.687315341f,
    0.727412629f,  0.686200312f,
    0.728464390f,  0.685083668f,
    0.729514438f,  0.683965412f,
    0.730562769f,  0.682845546f,
    0.731609381f,  0.681724074f,
    0.732654272f,  0.680600998f,
    0.733697438f,  0.679476320f,
    0.734738878f,  0.678350043f,
    0.735778589f,  0.677222170f,
    0.736816569f,  0.676092704f,
    0.737852815f,  0.674961646f,
    0.738887324f,  0.673829000f,
    0.739920095f,  0.672694769f,
    0.740951125f,  0.671558955f,
    0.741980412f,  0.670421560f,
    0.743007952f,  0.669282588f,
    0.744033744f,  0.668142041f,
    0.745057785f,  0.666999922f,
    0.746080074f,  0.665856234f,
    0.747100606f,  0.664710978f,
    0.748119380f,  0.663564159f,
    0.749136395f,  0.662415778f,
    0.750151646f,  0.661265838f,
    0.751165132f,  0.660114342f,
    0.752176850f,  0.658961293f,
    0.753186799f,  0.657806693f,
    0.754194975f,  0.656650546f,
    0.755201377f,  0.655492853f,
    0.756206001f,  0.654333618f,
    0.757208847f,  0.653172843f,
    0.758209910f,  0.652010531f,
    0.759209189f,  0.650846685f,
    0.760206682f,  0.649681307f,
    0.761202385f,  0.648514401f,
    0.762196298f,  0.647345969f,
    0.763188417f,  0.646176013f,
    0.764178741f,  0.645004537f,
    0.765167266f,  0.643831543f,
    0.766153990f,  0.642657034f,
    0.767138912f,  0.641481013f,
    0.768122029f,  0.640303482f,
    0.769103338f,  0.639124445f,
    0.770082837f,  0.637943904f,
    0.771060524f,  0.636761861f,
    0.772036397f,  0.635578320f,
    0.773010453f,  0.634393284f,
    0.773982691f,  0.633206755f,
    0.774953107f,  0.632018736f,
    0.775921699f,  0.630829230f,
    0.776888466f,  0.629638239f,
    0.777853404f,  0.628445767f,
    0.778816512f,  0.627251815f,
    0.779777788f,  0.626056388f,
    0.780737229f,  0.624859488f,
    0.781694832f,  0.623661118f,
    0.782650596f,  0.622461279f,
    0.783604519f,  0.621259977f,
    0.784556597f,  0.620057212f,
    0.785506830f,  0.618852988f,
    0.786455214f,  0.617647308f,
    0.787401747f,  0.616440175f,
    0.788346428f,  0.615231591f,
    0.789289253f,  0.614021559f,
    0.790230221f,  0.612810082f,
    0.791169330f,  0.611597164f,
    0.792106577f,  0.610382806f,
    0.793041960f,  0.609167012f,
    0.793975478f,  0.607949785f,
    0.794907126f,  0.606731127f,
    0.795836905f,  0.605511041f,
    0.796764810f,  0.604289531f,
    0.797690841f,  0.603066599f,
    0.798614995f,  0.601842247f,
    0.799537269f,  0.600616479f,
    0.800457662f,  0.599389298f,
    0.801376172f,  0.598160707f,
    0.802292796f,  0.596930708f,
    0.803207531f,  0.595699304f,
    0.804120377f,  0.594466499f,
    0.805031331f,  0.593232295f,
    0.805940391f,  0.591996695f,
    0.806847554f,  0.590759702f,
    0.807752818f,  0.589521319f,
    0.808656182f,  0.588281548f,
    0.809557642f,  0.587040394f,
    0.810457198f,  0.585797857f,
    0.811354847f,  0.584553943f,
    0.812250587f,  0.583308653f,
    0.813144415f,  0.582061990f,
    0.814036330f,  0.580813958f,
    0.814926329f,  0.579564559f,
    0.815814411f,  0.578313796f,
    0.816700573f,  0.577061673f,
    0.817584813f,  0.575808191f,
    0.818467130f,  0.574553355f,
    0.819347520f,  0.573297167f,
    0.820225983f,  0.572039629f,
    0.821102515f,  0.570780746f,
    0.821977115f,  0.569520519f,
    0.822849781f,  0.568258953f,
    0.823720511f,  0.566996049f,
    0.824589303f,  0.565731811f,
    0.825456154f,  0.564466242f,
    0.826321063f,  0.563199344f,
    0.827184027f,  0.561931121f,
    0.828045045f,  0.560661576f,
    0.828904115f,  0.559390712f,
    0.829761234f,  0.558118531f,
    0.830616400f,  0.556845037f,
    0.831469612f,  0.555570233f,
    0.832320868f,  0.554294121f,
    0.833170165f,  0.553016706f,
    0.834017501f,  0.551737988f,
    0.834862875f,  0.550457973f,
    0.835706284f,  0.549176662f,
    0.836547727f,  0.547894059f,
    0.837387202f,  0.546610167f,
    0.838224706f,  0.545324988f,
    0.839060237f,  0.544038527f,
    0.839893794f,  0.542750785f,
    0.840725375f,  0.541461766f,
    0.841554977f,  0.540171473f,
    0.842382600f,  0.538879909f,
    0.843208240f,  0.537587076f,
    0.844031895f,  0.536292979f,
    0.844853565f,  0.534997620f,
    0.845673247f,  0.533701002f,
    0.846490939f,  0.532403128f,
    0.847306639f,  0.531104001f,
    0.848120345f,  0.529803625f,
    0.848932055f,  0.528502002f,
    0.849741768f,  0.527199135f,
    0.850549481f,  0.525895027f,
    0.851355193f,  0.524589683f,
    0.852158902f,  0.523283103f,
    0.852960605f,  0.521975293f,
    0.853760301f,  0.520666254f,
    0.854557988f,  0.519355990f,
    0.855353665f,  0.518044504f,
    0.856147328f,  0.516731799f,
    0.856938977f,  0.515417878f,
    0.857728610f,  0.514102744f,
    0.858516224f,  0.512786401f,
    0.859301818f,  0.511468850f,
    0.860085390f,  0.510150097f,
    0.860866939f,  0.508830143f,
    0.861646461f,  0.507508991f,
    0.862423956f,  0.506186645f,
    0.863199422f,  0.504863109f,
    0.863972856f,  0.503538384f,
    0.864744258f,  0.502212474f,
    0.865513624f,  0.500885383f,
    0.866280954f,  0.499557113f,
    0.867046246f,  0.498227667f,
    0.867809497f,  0.496897049f,
    0.868570706f,  0.495565262f,
    0.869329871f,  0.494232309f,
    0.870086991f,  0.492898192f,
    0.870842063f,  0.491562916f,
    0.871595087f,  0.490226483f,
    0.872346059f,  0.488888897f,
    0.873094978f,  0.487550160f,
    0.873841843f,  0.486210276f,
    0.874586652f,  0.484869248f,
    0.875329403f,  0.483527079f,
    0.876070094f,  0.482183772f,
    0.876808724f,  0.480839331f,
    0.877545290f,  0.479493758f,
    0.878279792f,  0.478147056f,
    0.879012226f,  0.476799230f,
    0.879742593f,  0.475450282f,
    0.880470889f,  0.474100215f,
    0.881197113f,  0.472749032f,
    0.881921264f,  0.471396737f,
    0.882643340f,  0.470043332f,
    0.883363339f,  0.468688822f,
    0.884081259f,  0.467333209f,
    0.884797098f,  0.465976496f,
    0.885510856f,  0.464618686f,
    0.886222530f,  0.463259784f,
    0.886932119f,  0.461899791f,
    0.887639620f,  0.460538711f,
    0.888345033f,  0.459176548f,
    0.889048356f,  0.457813304f,
    0.889749586f,  0.456448982f,
    0.890448723f,  0.455083587f,
    0.891145765f,  0.453717121f,
    0.891840709f,  0.452349587f,
    0.892533555f,  0.450980989f,
    0.893224301f,  0.449611330f,
    0.893912945f,  0.448240612f,
    0.894599486f,  0.446868840f,
    0.895283921f,  0.445496017f,
    0.895966250f,  0.444122145f,
    0.896646470f,  0.442747228f,
    0.897324581f,  0.441371269f,
    0.898000580f,  0.439994271f,
    0.898674466f,  0.438616239f,
    0.899346237f,  0.437237174f,
    0.900015892f,  0.435857080f,
    0.900683429f,  0.434475961f,
    0.901348847f,  0.433093819f,
    0.902012144f,  0.431710658f,
    0.902673318f,  0.430326481f,
    0.903332368f,  0.428941292f,
    0.903989293f,  0.427555093f,
    0.904644091f,  0.426167889f,
    0.905296759f,  0.424779681f,
    0.905947298f,  0.423390474f,
    0.906595705f,  0.422000271f,
    0.907241978f,  0.420609074f,
    0.907886116f,  0.419216888f,
    0.908528119f,  0.417823716f,
    0.909167983f,  0.416429560f,
    0.909805708f,  0.415034424f,
    0.910441292f,  0.413638312f,
    0.911074734f,  0.412241227f,
    0.911706032f,  0.410843171f,
    0.912335185f,  0.409444149f,
    0.912962190f,  0.408044163f,
    0.913587048f,  0.406643217f,
    0.914209756f,  0.405241314f,
    0.914830312f,  0.403838458f,
    0.915448716f,  0.402434651f,
    0.916064966f,  0.401029897f,
    0.916679060f,  0.399624200f,
    0.917290997f,  0.398217562f,
    0.917900776f,  0.396809987f,
    0.918508394f,  0.395401479f,
    0.919113852f,  0.393992040f,
    0.919717146f,  0.392581674f,
    0.920318277f,  0.391170384f,
    0.920917242f,  0.389758174f,
    0.921514039f,  0.388345047f,
    0.922108669f,  0.386931006f,
    0.922701128f,  0.385516054f,
    0.923291417f,  0.384100195f,
    0.923879533f,  0.382683432f,
    0.924465474f,  0.381265769f,
    0.925049241f,  0.379847209f,
    0.925630831f,  0.378427755f,
    0.926210242f,  0.377007410f,
    0.926787474f,  0.375586178f,
    0.927362526f,  0.374164063f,
    0.927935395f,  0.372741067f,
    0.928506080f,  0.371317194f,
    0.929074581f,  0.369892447f,
    0.929640896f,  0.368466830f,
    0.930205023f,  0.367040346f,
    0.930766961f,  0.365612998f,
    0.931326709f,  0.364184790f,
    0.931884266f,  0.362755724f,
    0.932439629f,  0.361325806f,
    0.932992799f,  0.359895037f,
    0.933543773f,  0.358463421f,
    0.934092550f,  0.357030961f,
    0.934639130f,  0.355597662f,
    0.935183510f,  0.354163525f,
    0.935725689f,  0.352728556f,
    0.936265667f,  0.351292756f,
    0.936803442f,  0.349856130f,
    0.937339012f,  0.348418680f,
    0.937872376f,  0.346980411f,
    0.938403534f,  0.345541325f,
    0.938932484f,  0.344101426f,
    0.939459224f,  0.342660717f,
    0.939983753f,  0.341219202f,
    0.940506071f,  0.339776884f,
    0.941026175f,  0.338333767f,
    0.941544065f,  0.336889853f,
    0.942059740f,  0.335445147f,
    0.942573198f,  0.333999651f,
    0.943084437f,  0.332553370f,
    0.943593458f,  0.331106306f,
    0.944100258f,  0.329658463f,
    0.944604837f,  0.328209844f,
    0.945107193f,  0.326760452f,
    0.945607325f,  0.325310292f,
    0.946105232f,  0.323859367f,
    0.946600913f,  0.322407679f,
    0.947094366f,  0.320955232f,
    0.947585591f,  0.319502031f,
    0.948074586f,  0.318048077f,
    0.948561350f,  0.316593376f,
    0.949045882f,  0.315137929f,
    0.949528181f,  0.313681740f,
    0.950008245f,  0.312224814f,
    0.950486074f,  0.310767153f,
    0.950961666f,  0.309308760f,
    0.951435021f,  0.307849640f,
    0.951906137f,  0.306389795f,
    0.952375013f,  0.304929230f,
    0.952841648f,  0.303467947f,
    0.953306040f,  0.302005949f,
    0.953768190f,  0.300543241f,
    0.954228095f,  0.299079826f,
    0.954685755f,  0.297615707f,
    0.955141168f,  0.296150888f,
    0.955594334f,  0.294685372f,
    0.956045251f,  0.293219163f,
    0.956493919f,  0.291752263f,
    0.956940336f,  0.290284677f,
    0.957384501f,  0.288816408f,
    0.957826413f,  0.287347460f,
    0.958266071f,  0.285877835f,
    0.958703475f,  0.284407537f,
    0.959138622f,  0.282936570f,
    0.959571513f,  0.281464938f,
    0.960002146f,  0.279992643f,
    0.960430519f,  0.278519689f,
    0.960856633f,  0.277046080f,
    0.961280486f,  0.275571819f,
    0.961702077f,  0.274096910f,
    0.962121404f,  0.272621355f,
    0.962538468f,  0.271145160f,
    0.962953267f,  0.269668326f,
    0.963365800f,  0.268190857f,
    0.963776066f,  0.266712757f,
    0.964184064f,  0.265234030f,
    0.964589793f,  0.263754679f,
    0.964993253f,  0.262274707f,
    0.965394442f,  0.260794118f,
    0.965793359f,  0.259312915f,
    0.966190003f,  0.257831102f,
    0.966584374f,  0.256348682f,
    0.966976471f,  0.254865660f,
    0.967366292f,  0.253382037f,
    0.967753837f,  0.251897818f,
    0.968139105f,  0.250413007f,
    0.968522094f,  0.248927606f,
    0.968902805f,  0.247441619f,
    0.969281235f,  0.245955050f,
    0.969657385f,  0.244467903f,
    0.970031253f,  0.242980180f,
    0.970402839f,  0.241491885f,
    0.970772141f,  0.240003022f,
    0.971139158f,  0.238513595f,
    0.971503891f,  0.237023606f,
    0.971866337f,  0.235533059f,
    0.972226497f,  0.234041959f,
    0.972584369f,  0.232550307f,
    0.972939952f,  0.231058108f,
    0.973293246f,  0.229565366f,
    0.973644250f,  0.228072083f,
    0.973992962f,  0.226578264f,
    0.974339383f,  0.225083911f,
    0.974683511f,  0.223589029f,
    0.975025345f,  0.222093621f,
    0.975364885f,  0.220597690f,
    0.975702130f,  0.219101240f,
    0.976037079f,  0.217604275f,
    0.976369731f,  0.216106797f,
    0.976700086f,  0.214608811f,
    0.977028143f,  0.213110320f,
    0.977353900f,  0.211611327f,
    0.977677358f,  0.210111837f,
    0.977998515f,  0.208611852f,
    0.978317371f,  0.207111376f,
    0.978633924f,  0.205610413f,
    0.978948175f,  0.204108966f,
    0.979260123f,  0.202607039f,
    0.979569766f,  0.201104635f,
    0.979877104f,  0.199601758f,
    0.980182136f,  0.198098411f,
    0.980484862f,  0.196594598f,
    0.980785280f,  0.195090322f,
    0.981083391f,  0.193585587f,
    0.981379193f,  0.192080397f,
    0.981672686f,  0.190574755f,
    0.981963869f,  0.189068664f,
    0.982252741f,  0.187562129f,
    0.982539302f,  0.186055152f,
    0.982823551f,  0.184547737f,
    0.983105487f,  0.183039888f,
    0.983385110f,  0.181531608f,
    0.983662419f,  0.180022901f,
    0.983937413f,  0.178513771f,
    0.984210092f,  0.177004220f,
    0.984480455f,  0.175494253f,
    0.984748502f,  0.173983873f,
    0.985014231f,  0.172473084f,
    0.985277642f,  0.170961889f,
    0.985538735f,  0.169450291f,
    0.985797509f,  0.167938295f,
    0.986053963f,  0.166425904f,
    0.986308097f,  0.164913120f,
    0.986559910f,  0.163399949f,
    0.986809402f,  0.161886394f,
    0.987056571f,  0.160372457f,
    0.987301418f,  0.158858143f,
    0.987543942f,  0.157343456f,
    0.987784142f,  0.155828398f,
    0.988022017f,  0.154312973f,
    0.988257568f,  0.152797185f,
    0.988490793f,  0.151281038f,
    0.988721692f,  0.149764535f,
    0.988950265f,  0.148247679f,
    0.989176510f,  0.146730474f,
    0.989400428f,  0.145212925f,
    0.989622017f,  0.143695033f,
    0.989841278f,  0.142176804f,
    0.990058210f,  0.140658239f,
    0.990272812f,  0.139139344f,
    0.990485084f,  0.137620122f,
    0.990695025f,  0.136100575f,
    0.990902635f,  0.134580709f,
    0.991107914f,  0.133060525f,
    0.991310860f,  0.131540029f,
    0.991511473f,  0.130019223f,
    0.991709754f,  0.128498111f,
    0.991905700f,  0.126976696f,
    0.992099313f,  0.125454983f,
    0.992290591f,  0.123932975f,
    0.992479535f,  0.122410675f,
    0.992666142f,  0.120888087f,
    0.992850414f,  0.119365215f,
    0.993032350f,  0.117842062f,
    0.993211949f,  0.116318631f,
    0.993389211f,  0.114794927f,
    0.993564136f,  0.113270952f,
    0.993736722f,  0.111746711f,
    0.993906970f,  0.110222207f,
    0.994074879f,  0.108697444f,
    0.994240449f,  0.107172425f,
    0.994403680f,  0.105647154f,
    0.994564571f,  0.104121634f,
    0.994723121f,  0.102595869f,
    0.994879331f,  0.101069863f,
    0.995033199f,  0.099543619f,
    0.995184727f,  0.098017140f,
    0.995333912f,  0.096490431f,
    0.995480755f,  0.094963495f,
    0.995625256f,  0.093436336f,
    0.995767414f,  0.091908956f,
    0.995907229f,  0.090381361f,
    0.996044701f,  0.088853553f,
    0.996179829f,  0.087325535f,
    0.996312612f,  0.085797312f,
    0.996443051f,  0.084268888f,
    0.996571146f,  0.082740265f,
    0.996696895f,  0.081211447f,
    0.996820299f,  0.079682438f,
    0.996941358f,  0.078153242f,
    0.997060070f,  0.076623861f,
    0.997176437f,  0.075094301f,
    0.997290457f,  0.073564564f,
    0.997402130f,  0.072034653f,
    0.997511456f,  0.070504573f,
    0.997618435f,  0.068974328f,
    0.997723067f,  0.067443920f,
    0.997825350f,  0.065913353f,
    0.997925286f,  0.064382631f,
    0.998022874f,  0.062851758f,
    0.998118113f,  0.061320736f,
    0.998211003f,  0.059789571f,
    0.998301545f,  0.058258265f,
    0.998389737f,  0.056726821f,
    0.998475581f,  0.055195244f,
    0.998559074f,  0.053663538f,
    0.998640218f,  0.052131705f,
    0.998719012f,  0.050599749f,
    0.998795456f,  0.049067674f,
    0.998869550f,  0.047535484f,
    0.998941293f,  0.046003182f,
    0.999010686f,  0.044470772f,
    0.999077728f,  0.042938257f,
    0.999142419f,  0.041405641f,
    0.999204759f,  0.039872928f,
    0.999264747f,  0.038340120f,
    0.999322385f,  0.036807223f,
    0.999377670f,  0.035274239f,
    0.999430605f,  0.033741172f,
    0.999481187f,  0.032208025f,
    0.999529418f,  0.030674803f,
    0.999575296f,  0.029141509f,
    0.999618822f,  0.027608146f,
    0.999659997f,  0.026074718f,
    0.999698819f,  0.024541229f,
    0.999735288f,  0.023007681f,
    0.999769405f,  0.021474080f,
    0.999801170f,  0.019940429f,
    0.999830582f,  0.018406730f,
    0.999857641f,  0.016872988f,
    0.999882347f,  0.015339206f,
    0.999904701f,  0.013805389f,
    0.999924702f,  0.012271538f,
    0.999942350f,  0.010737659f,
    0.999957645f,  0.009203755f,
    0.999970586f,  0.007669829f,
    0.999981175f,  0.006135885f,
    0.999989411f,  0.004601926f,
    0.999995294f,  0.003067957f,
    0.999998823f,  0.001533980f,
    1.000000000f,  0.000000000f,
    0.999998823f, -0.001533980f,
    0.999995294f, -0.003067957f,
    0.999989411f, -0.004601926f,
    0.999981175f, -0.006135885f,
    0.999970586f, -0.007669829f,
    0.999957645f, -0.009203755f,
    0.999942350f, -0.010737659f,
    0.999924702f, -0.012271538f,
    0.999904701f, -0.013805389f,
    0.999882347f, -0.015339206f,
    0.999857641f, -0.016872988f,
    0.999830582f, -0.018406730f,
    0.999801170f, -0.019940429f,
    0.999769405f, -0.021474080f,
    0.999735288f, -0.023007681f,
    0.999698819f, -0.024541229f,
    0.999659997f, -0.026074718f,
    0.999618822f, -0.027608146f,
    0.999575296f, -0.029141509f,
    0.999529418f, -0.030674803f,
    0.999481187f, -0.032208025f,
    0.999430605f, -0.033741172f,
    0.999377670f, -0.035274239f,
    0.999322385f, -0.036807223f,
    0.999264747f, -0.038340120f,
    0.999204759f, -0.039872928f,
    0.999142419f, -0.041405641f,
    0.999077728f, -0.042938257f,
    0.999010686f, -0.044470772f,
    0.998941293f, -0.046003182f,
    0.998869550f, -0.047535484f,
    0.998795456f, -0.049067674f,
    0.998719012f, -0.050599749f,
    0.998640218f, -0.052131705f,
    0.998559074f, -0.053663538f,
    0.998475581f, -0.055195244f,
    0.998389737f, -0.056726821f,
    0.998301545f, -0.058258265f,
    0.998211003f, -0.059789571f,
    0.998118113f, -0.061320736f,
    0.998022874f, -0.062851758f,
    0.997925286f, -0.064382631f,
    0.997825350f, -0.065913353f,
    0.997723067f, -0.067443920f,
    0.997618435f, -0.068974328f,
    0.997511456f, -0.070504573f,
    0.997402130f, -0.072034653f,
    0.997290457f, -0.073564564f,
    0.997176437f, -0.075094301f,
    0.997060070f, -0.076623861f,
    0.996941358f, -0.078153242f,
    0.996820299f, -0.079682438f,
    0.996696895f, -0.081211447f,
    0.996571146f, -0.082740265f,
    0.996443051f, -0.084268888f,
    0.996312612f, -0.085797312f,
    0.996179829f, -0.087325535f,
    0.996044701f, -0.088853553f,
    0.995907229f, -0.090381361f,
    0.995767414f, -0.091908956f,
    0.995625256f, -0.093436336f,
    0.995480755f, -0.094963495f,
    0.995333912f, -0.096490431f,
    0.995184727f, -0.098017140f,
    0.995033199f, -0.099543619f,
    0.994879331f, -0.101069863f,
    0.994723121f, -0.102595869f,
    0.994564571f, -0.104121634f,
    0.994403680f, -0.105647154f,
    0.994240449f, -0.107172425f,
    0.994074879f, -0.108697444f,
    0.993906970f, -0.110222207f,
    0.993736722f, -0.111746711f,
    0.993564136f, -0.113270952f,
    0.993389211f, -0.114794927f,
    0.993211949f, -0.116318631f,
    0.993032350f, -0.117842062f,
    0.992850414f, -0.119365215f,
    0.992666142f, -0.120888087f,
    0.992479535f, -0.122410675f,
    0.992290591f, -0.123932975f,
    0.992099313f, -0.125454983f,
    0.991905700f, -0.126976696f,
    0.991709754f, -0.128498111f,
    0.991511473f, -0.130019223f,
    0.991310860f, -0.131540029f,
    0.991107914f, -0.133060525f,
    0.990902635f, -0.134580709f,
    0.990695025f, -0.136100575f,
    0.990485084f, -0.137620122f,
    0.990272812f, -0.139139344f,
    0.990058210f, -0.140658239f,
    0.989841278f, -0.142176804f,
    0.989622017f, -0.143695033f,
    0.989400428f, -0.145212925f,
    0.989176510f, -0.146730474f,
    0.988950265f, -0.148247679f,
    0.988721692f, -0.149764535f,
    0.988490793f, -0.151281038f,
    0.988257568f, -0.152797185f,
    0.988022017f, -0.154312973f,
    0.987784142f, -0.155828398f,
    0.987543942f, -0.157343456f,
    0.987301418f, -0.158858143f,
    0.987056571f, -0.160372457f,
    0.986809402f, -0.161886394f,
    0.986559910f, -0.163399949f,
    0.986308097f, -0.164913120f,
    0.986053963f, -0.166425904f,
    0.985797509f, -0.167938295f,
    0.985538735f, -0.169450291f,
    0.985277642f, -0.170961889f,
    0.985014231f, -0.172473084f,
    0.984748502f, -0.173983873f,
    0.984480455f, -0.175494253f,
    0.984210092f, -0.177004220f,
    0.983937413f, -0.178513771f,
    0.983662419f, -0.180022901f,
    0.983385110f, -0.181531608f,
    0.983105487f, -0.183039888f,
    0.982823551f, -0.184547737f,
    0.982539302f, -0.186055152f,
    0.982252741f, -0.187562129f,
    0.981963869f, -0.189068664f,
    0.981672686f, -0.190574755f,
    0.981379193f, -0.192080397f,
    0.981083391f, -0.193585587f,
    0.980785280f, -0.195090322f,
    0.980484862f, -0.196594598f,
    0.980182136f, -0.198098411f,
    0.979877104f, -0.199601758f,
    0.979569766f, -0.201104635f,
    0.979260123f, -0.202607039f,
    0.978948175f, -0.204108966f,
    0.978633924f, -0.205610413f,
    0.978317371f, -0.207111376f,
    0.977998515f, -0.208611852f,
    0.977677358f, -0.210111837f,
    0.977353900f, -0.211611327f,
    0.977028143f, -0.213110320f,
    0.976700086f, -0.214608811f,
    0.976369731f, -0.216106797f,
    0.976037079f, -0.217604275f,
    0.975702130f, -0.219101240f,
    0.975364885f, -0.220597690f,
    0.975025345f, -0.222093621f,
    0.974683511f, -0.223589029f,
    0.974339383f, -0.225083911f,
    0.973992962f, -0.226578264f,
    0.973644250f, -0.228072083f,
    0.973293246f, -0.229565366f,
    0.972939952f, -0.231058108f,
    0.972584369f, -0.232550307f,
    0.972226497f, -0.234041959f,
    0.971866337f, -0.235533059f,
    0.971503891f, -0.237023606f,
    0.971139158f, -0.238513595f,
    0.970772141f, -0.240003022f,
    0.970402839f, -0.241491885f,
    0.970031253f, -0.242980180f,
    0.969657385f, -0.244467903f,
    0.969281235f, -0.245955050f,
    0.968902805f, -0.247441619f,
    0.968522094f, -0.248927606f,
    0.968139105f, -0.250413007f,
    0.967753837f, -0.251897818f,
    0.967366292f, -0.253382037f,
    0.966976471f, -0.254865660f,
    0.966584374f, -0.256348682f,
    0.966190003f, -0.257831102f,
    0.965793359f, -0.259312915f,
    0.965394442f, -0.260794118f,
    0.964993253f, -0.262274707f,
    0.964589793f, -0.263754679f,
    0.964184064f, -0.265234030f,
    0.963776066f, -0.266712757f,
    0.963365800f, -0.268190857f,
    0.962953267f, -0.269668326f,
    0.962538468f, -0.271145160f,
    0.962121404f, -0.272621355f,
    0.961702077f, -0.274096910f,
    0.961280486f, -0.275571819f,
    0.960856633f, -0.277046080f,
    0.960430519f, -0.278519689f,
    0.960002146f, -0.279992643f,
    0.959571513f, -0.281464938f,
    0.959138622f, -0.282936570f,
    0.958703475f, -0.284407537f,
    0.958266071f, -0.285877835f,
    0.957826413f, -0.287347460f,
    0.957384501f, -0.288816408f,
    0.956940336f, -0.290284677f,
    0.956493919f, -0.291752263f,
    0.956045251f, -0.293219163f,
    0.955594334f, -0.294685372f,
    0.955141168f, -0.296150888f,
    0.954685755f, -0.297615707f,
    0.954228095f, -0.299079826f,
    0.953768190f, -0.300543241f,
    0.953306040f, -0.302005949f,
    0.952841648f, -0.303467947f,
    0.952375013f, -0.304929230f,
    0.951906137f, -0.306389795f,
    0.951435021f, -0.307849640f,
    0.950961666f, -0.309308760f,
    0.950486074f, -0.310767153f,
    0.950008245f, -0.312224814f,
    0.949528181f, -0.313681740f,
    0.949045882f, -0.315137929f,
    0.948561350f, -0.316593376f,
    0.948074586f, -0.318048077f,
    0.947585591f, -0.319502031f,
    0.947094366f, -0.320955232f,
    0.946600913f, -0.322407679f,
    0.946105232f, -0.323859367f,
    0.945607325f, -0.325310292f,
    0.945107193f, -0.326760452f,
    0.944604837f, -0.328209844f,
    0.944100258f, -0.329658463f,
    0.943593458f, -0.331106306f,
    0.943084437f, -0.332553370f,
    0.942573198f, -0.333999651f,
    0.942059740f, -0.335445147f,
    0.941544065f, -0.336889853f,
    0.941026175f, -0.338333767f,
    0.940506071f, -0.339776884f,
    0.939983753f, -0.341219202f,
    0.939459224f, -0.342660717f,
    0.938932484f, -0.344101426f,
    0.938403534f, -0.345541325f,
    0.937872376f, -0.346980411f,
    0.937339012f, -0.348418680f,
    0.936803442f, -0.349856130f,
    0.936265667f, -0.351292756f,
    0.935725689f, -0.352728556f,
    0.935183510f, -0.354163525f,
    0.934639130f, -0.355597662f,
    0.934092550f, -0.357030961f,
    0.933543773f, -0.358463421f,
    0.932992799f, -0.359895037f,
    0.932439629f, -0.361325806f,
    0.931884266f, -0.362755724f,
    0.931326709f, -0.364184790f,
    0.930766961f, -0.365612998f,
    0.930205023f, -0.367040346f,
    0.929640896f, -0.368466830f,
    0.929074581f, -0.369892447f,
    0.928506080f, -0.371317194f,
    0.927935395f, -0.372741067f,
    0.927362526f, -0.374164063f,
    0.926787474f, -0.375586178f,
    0.926210242f, -0.377007410f,
    0.925630831f, -0.378427755f,
    0.925049241f, -0.379847209f,
    0.924465474f, -0.381265769f,
    0.923879533f, -0.382683432f,
    0.923291417f, -0.384100195f,
    0.922701128f, -0.385516054f,
    0.922108669f, -0.386931006f,
    0.921514039f, -0.388345047f,
    0.920917242f, -0.389758174f,
    0.920318277f, -0.391170384f,
    0.919717146f, -0.392581674f,
    0.919113852f, -0.393992040f,
    0.918508394f, -0.395401479f,
    0.917900776f, -0.396809987f,
    0.917290997f, -0.398217562f,
    0.916679060f, -0.399624200f,
    0.916064966f, -0.401029897f,
    0.915448716f, -0.402434651f,
    0.914830312f, -0.403838458f,
    0.914209756f, -0.405241314f,
    0.913587048f, -0.406643217f,
    0.912962190f, -0.408044163f,
    0.912335185f, -0.409444149f,
    0.911706032f, -0.410843171f,
    0.911074734f, -0.412241227f,
    0.910441292f, -0.413638312f,
    0.909805708f, -0.415034424f,
    0.909167983f, -0.416429560f,
    0.908528119f, -0.417823716f,
    0.907886116f, -0.419216888f,
    0.907241978f, -0.420609074f,
    0.906595705f, -0.422000271f,
    0.905947298f, -0.423390474f,
    0.905296759f, -0.424779681f,
    0.904644091f, -0.426167889f,
    0.903989293f, -0.427555093f,
    0.903332368f, -0.428941292f,
    0.902673318f, -0.430326481f,
    0.902012144f, -0.431710658f,
    0.901348847f, -0.433093819f,
    0.900683429f, -0.434475961f,
    0.900015892f, -0.435857080f,
    0.899346237f, -0.437237174f,
    0.898674466f, -0.438616239f,
    0.898000580f, -0.439994271f,
    0.897324581f, -0.441371269f,
    0.896646470f, -0.442747228f,
    0.895966250f, -0.444122145f,
    0.895283921f, -0.445496017f,
    0.894599486f, -0.446868840f,
    0.893912945f, -0.448240612f,
    0.893224301f, -0.449611330f,
    0.892533555f, -0.450980989f,
    0.891840709f, -0.452349587f,
    0.891145765f, -0.453717121f,
    0.890448723f, -0.455083587f,
    0.889749586f, -0.456448982f,
    0.889048356f, -0.457813304f,
    0.888345033f, -0.459176548f,
    0.887639620f, -0.460538711f,
    0.886932119f, -0.461899791f,
    0.886222530f, -0.463259784f,
    0.885510856f, -0.464618686f,
    0.884797098f, -0.465976496f,
    0.884081259f, -0.467333209f,
    0.883363339f, -0.468688822f,
    0.882643340f, -0.470043332f,
    0.881921264f, -0.471396737f,
    0.881197113f, -0.472749032f,
    0.880470889f, -0.474100215f,
    0.879742593f, -0.475450282f,
    0.879012226f, -0.476799230f,
    0.878279792f, -0.478147056f,
    0.877545290f, -0.479493758f,
    0.876808724f, -0.480839331f,
    0.876070094f, -0.482183772f,
    0.875329403f, -0.483527079f,
    0.874586652f, -0.484869248f,
    0.873841843f, -0.486210276f,
    0.873094978f, -0.487550160f,
    0.872346059f, -0.488888897f,
    0.871595087f, -0.490226483f,
    0.870842063f, -0.491562916f,
    0.870086991f, -0.492898192f,
    0.869329871f, -0.494232309f,
    0.868570706f, -0.495565262f,
    0.867809497f, -0.496897049f,
    0.867046246f, -0.498227667f,
    0.866280954f, -0.499557113f,
    0.865513624f, -0.500885383f,
    0.864744258f, -0.502212474f,
    0.863972856f, -0.503538384f,
    0.863199422f, -0.504863109f,
    0.862423956f, -0.506186645f,
    0.861646461f, -0.507508991f,
    0.860866939f, -0.508830143f,
    0.860085390f, -0.510150097f,
    0.859301818f, -0.511468850f,
    0.858516224f, -0.512786401f,
    0.857728610f, -0.514102744f,
    0.856938977f, -0.515417878f,
    0.856147328f, -0.516731799f,
    0.855353665f, -0.518044504f,
    0.854557988f, -0.519355990f,
    0.853760301f, -0.520666254f,
    0.852960605f, -0.521975293f,
    0.852158902f, -0.523283103f,
    0.851355193f, -0.524589683f,
    0.850549481f, -0.525895027f,
    0.849741768f, -0.527199135f,
    0.848932055f, -0.528502002f,
    0.848120345f, -0.529803625f,
    0.847306639f, -0.531104001f,
    0.846490939f, -0.532403128f,
    0.845673247f, -0.533701002f,
    0.844853565f, -0.534997620f,
    0.844031895f, -0.536292979f,
    0.843208240f, -0.537587076f,
    0.842382600f, -0.538879909f,
    0.841554977f, -0.540171473f,
    0.840725375f, -0.541461766f,
    0.839893794f, -0.542750785f,
    0.839060237f, -0.544038527f,
    0.838224706f, -0.545324988f,
    0.837387202f, -0.546610167f,
    0.836547727f, -0.547894059f,
    0.835706284f, -0.549176662f,
    0.834862875f, -0.550457973f,
    0.834017501f, -0.551737988f,
    0.833170165f, -0.553016706f,
    0.832320868f, -0.554294121f,
    0.831469612f, -0.555570233f,
    0.830616400f, -0.556845037f,
    0.829761234f, -0.558118531f,
    0.828904115f, -0.559390712f,
    0.828045045f, -0.560661576f,
    0.827184027f, -0.561931121f,
    0.826321063f, -0.563199344f,
    0.825456154f, -0.564466242f,
    0.824589303f, -0.565731811f,
    0.823720511f, -0.566996049f,
    0.822849781f, -0.568258953f,
    0.821977115f, -0.569520519f,
    0.821102515f, -0.570780746f,
    0.820225983f, -0.572039629f,
    0.819347520f, -0.573297167f,
    0.818467130f, -0.574553355f,
    0.817584813f, -0.575808191f,
    0.816700573f, -0.577061673f,
    0.815814411f, -0.578313796f,
    0.814926329f, -0.579564559f,
    0.814036330f, -0.580813958f,
    0.813144415f, -0.582061990f,
    0.812250587f, -0.583308653f,
    0.811354847f, -0.584553943f,
    0.810457198f, -0.585797857f,
    0.809557642f, -0.587040394f,
    0.808656182f, -0.588281548f,
    0.807752818f, -0.589521319f,
    0.806847554f, -0.590759702f,
    0.805940391f, -0.591996695f,
    0.805031331f, -0.593232295f,
    0.804120377f, -0.594466499f,
    0.803207531f, -0.595699304f,
    0.802292796f, -0.596930708f,
    0.801376172f, -0.598160707f,
    0.800457662f, -0.599389298f,
    0.799537269f, -0.600616479f,
    0.798614995f, -0.601842247f,
    0.797690841f, -0.603066599f,
    0.796764810f, -0.604289531f,
    0.795836905f, -0.605511041f,
    0.794907126f, -0.606731127f,
    0.793975478f, -0.607949785f,
    0.793041960f, -0.609167012f,
    0.792106577f, -0.610382806f,
    0.791169330f, -0.611597164f,
    0.790230221f, -0.612810082f,
    0.789289253f, -0.614021559f,
    0.788346428f, -0.615231591f,
    0.787401747f, -0.616440175f,
    0.786455214f, -0.617647308f,
    0.785506830f, -0.618852988f,
    0.784556597f, -0.620057212f,
    0.783604519f, -0.621259977f,
    0.782650596f, -0.622461279f,
    0.781694832f, -0.623661118f,
    0.780737229f, -0.624859488f,
    0.779777788f, -0.626056388f,
    0.778816512f, -0.627251815f,
    0.777853404f, -0.628445767f,
    0.776888466f, -0.629638239f,
    0.775921699f, -0.630829230f,
    0.774953107f, -0.632018736f,
    0.773982691f, -0.633206755f,
    0.773010453f, -0.634393284f,
    0.772036397f, -0.635578320f,
    0.771060524f, -0.636761861f,
    0.770082837f, -0.637943904f,
    0.769103338f, -0.639124445f,
    0.768122029f, -0.640303482f,
    0.767138912f, -0.641481013f,
    0.766153990f, -0.642657034f,
    0.765167266f, -0.643831543f,
    0.764178741f, -0.645004537f,
    0.763188417f, -0.646176013f,
    0.762196298f, -0.647345969f,
    0.761202385f, -0.648514401f,
    0.760206682f, -0.649681307f,
    0.759209189f, -0.650846685f,
    0.758209910f, -0.652010531f,
    0.757208847f, -0.653172843f,
    0.756206001f, -0.654333618f,
    0.755201377f, -0.655492853f,
    0.754194975f, -0.656650546f,
    0.753186799f, -0.657806693f,
    0.752176850f, -0.658961293f,
    0.751165132f, -0.660114342f,
    0.750151646f, -0.661265838f,
    0.749136395f, -0.662415778f,
    0.748119380f, -0.663564159f,
    0.747100606f, -0.664710978f,
    0.746080074f, -0.665856234f,
    0.745057785f, -0.666999922f,
    0.744033744f, -0.668142041f,
    0.743007952f, -0.669282588f,
    0.741980412f, -0.670421560f,
    0.740951125f, -0.671558955f,
    0.739920095f, -0.672694769f,
    0.738887324f, -0.673829000f,
    0.737852815f, -0.674961646f,
    0.736816569f, -0.676092704f,
    0.735778589f, -0.677222170f,
    0.734738878f, -0.678350043f,
    0.733697438f, -0.679476320f,
    0.732654272f, -0.680600998f,
    0.731609381f, -0.681724074f,
    0.730562769f, -0.682845546f,
    0.729514438f, -0.683965412f,
    0.728464390f, -0.685083668f,
    0.727412629f, -0.686200312f,
    0.726359155f, -0.687315341f,
    0.725303972f, -0.688428753f,
    0.724247083f, -0.689540545f,
    0.723188489f, -0.690650714f,
    0.722128194f, -0.691759258f,
    0.721066199f, -0.692866175f,
    0.720002508f, -0.693971461f,
    0.718937122f, -0.695075114f,
    0.717870045f, -0.696177131f,
    0.716801279f, -0.697277511f,
    0.715730825f, -0.698376249f,
    0.714658688f, -0.699473345f,
    0.713584869f, -0.700568794f,
    0.712509371f, -0.701662595f,
    0.711432196f, -0.702754744f,
    0.710353347f, -0.703845241f,
    0.709272826f, -0.704934080f,
    0.708190637f, -0.706021261f,
    0.707106781f, -0.707106781f,
    0.706021261f, -0.708190637f,
    0.704934080f, -0.709272826f,
    0.703845241f, -0.710353347f,
    0.702754744f, -0.711432196f,
    0.701662595f, -0.712509371f,
    0.700568794f, -0.713584869f,
    0.699473345f, -0.714658688f,
    0.698376249f, -0.715730825f,
    0.697277511f, -0.716801279f,
    0.696177131f, -0.717870045f,
    0.695075114f, -0.718937122f,
    0.693971461f, -0.720002508f,
    0.692866175f, -0.721066199f,
    0.691759258f, -0.722128194f,
    0.690650714f, -0.723188489f,
    0.689540545f, -0.724247083f,
    0.688428753f, -0.725303972f,
    0.687315341f, -0.726359155f,
    0.686200312f, -0.727412629f,
    0.685083668f, -0.728464390f,
    0.683965412f, -0.729514438f,
    0.682845546f, -0.730562769f,
    0.681724074f, -0.731609381f,
    0.680600998f, -0.732654272f,
    0.679476320f, -0.733697438f,
    0.678350043f, -0.734738878f,
    0.677222170f, -0.735778589f,
    0.676092704f, -0.736816569f,
    0.674961646f, -0.737852815f,
    0.673829000f, -0.738887324f,
    0.672694769f, -0.739920095f,
    0.671558955f, -0.740951125f,
    0.670421560f, -0.741980412f,
    0.669282588f, -0.743007952f,
    0.668142041f, -0.744033744f,
    0.666999922f, -0.745057785f,
    0.665856234f, -0.746080074f,
    0.664710978f, -0.747100606f,
    0.663564159f, -0.748119380f,
    0.662415778f, -0.749136395f,
    0.661265838f, -0.750151646f,
    0.660114342f, -0.751165132f,
    0.658961293f, -0.752176850f,
    0.657806693f, -0.753186799f,
    0.656650546f, -0.754194975f,
    0.655492853f, -0.755201377f,
    0.654333618f, -0.756206001f,
    0.653172843f, -0.757208847f,
    0.652010531f, -0.758209910f,
    0.650846685f, -0.759209189f,
    0.649681307f, -0.760206682f,
    0.648514401f, -0.761202385f,
    0.647345969f, -0.762196298f,
    0.646176013f, -0.763188417f,
    0.645004537f, -0.764178741f,
    0.643831543f, -0.765167266f,
    0.642657034f, -0.766153990f,
    0.641481013f, -0.767138912f,
    0.640303482f, -0.768122029f,
    0.639124445f, -0.769103338f,
    0.637943904f, -0.770082837f,
    0.636761861f, -0.771060524f,
    0.635578320f, -0.772036397f,
    0.634393284f, -0.773010453f,
    0.633206755f, -0.773982691f,
    0.632018736f, -0.774953107f,
    0.630829230f, -0.775921699f,
    0.629638239f, -0.776888466f,
    0.628445767f, -0.777853404f,
    0.627251815f, -0.778816512f,
    0.626056388f, -0.779777788f,
    0.624859488f, -0.780737229f,
    0.623661118f, -0.781694832f,
    0.622461279f, -0.782650596f,
    0.621259977f, -0.783604519f,
    0.620057212f, -0.784556597f,
    0.618852988f, -0.785506830f,
    0.617647308f, -0.786455214f,
    0.616440175f, -0.787401747f,
    0.615231591f, -0.788346428f,
    0.614021559f, -0.789289253f,
    0.612810082f, -0.790230221f,
    0.611597164f, -0.791169330f,
    0.610382806f, -0.792106577f,
    0.609167012f, -0.793041960f,
    0.607949785f, -0.793975478f,
    0.606731127f, -0.794907126f,
    0.605511041f, -0.795836905f,
    0.604289531f, -0.796764810f,
    0.603066599f, -0.797690841f,
    0.601842247f, -0.798614995f,
    0.600616479f, -0.799537269f,
    0.599389298f, -0.800457662f,
    0.598160707f, -0.801376172f,
    0.596930708f, -0.802292796f,
    0.595699304f, -0.803207531f,
    0.594466499f, -0.804120377f,
    0.593232295f, -0.805031331f,
    0.591996695f, -0.805940391f,
    0.590759702f, -0.806847554f,
    0.589521319f, -0.807752818f,
    0.588281548f, -0.808656182f,
    0.587040394f, -0.809557642f,
    0.585797857f, -0.810457198f,
    0.584553943f, -0.811354847f,
    0.583308653f, -0.812250587f,
    0.582061990f, -0.813144415f,
    0.580813958f, -0.814036330f,
    0.579564559f, -0.814926329f,
    0.578313796f, -0.815814411f,
    0.577061673f, -0.816700573f,
    0.575808191f, -0.817584813f,
    0.574553355f, -0.818467130f,
    0.573297167f, -0.819347520f,
    0.572039629f, -0.820225983f,
    0.570780746f, -0.821102515f,
    0.569520519f, -0.821977115f,
    0.568258953f, -0.822849781f,
    0.566996049f, -0.823720511f,
    0.565731811f, -0.824589303f,
    0.564466242f, -0.825456154f,
    0.563199344f, -0.826321063f,
    0.561931121f, -0.827184027f,
    0.560661576f, -0.828045045f,
    0.559390712f, -0.828904115f,
    0.558118531f, -0.829761234f,
    0.556845037f, -0.830616400f,
    0.555570233f, -0.831469612f,
    0.554294121f, -0.832320868f,
    0.553016706f, -0.833170165f,
    0.551737988f, -0.834017501f,
    0.550457973f, -0.834862875f,
    0.549176662f, -0.835706284f,
    0.547894059f, -0.836547727f,
    0.546610167f, -0.837387202f,
    0.545324988f, -0.838224706f,
    0.544038527f, -0.839060237f,
    0.542750785f, -0.839893794f,
    0.541461766f, -0.840725375f,
    0.540171473f, -0.841554977f,
    0.538879909f, -0.842382600f,
    0.537587076f, -0.843208240f,
    0.536292979f, -0.844031895f,
    0.534997620f, -0.844853565f,
    0.533701002f, -0.845673247f,
    0.532403128f, -0.846490939f,
    0.531104001f, -0.847306639f,
    0.529803625f, -0.848120345f,
    0.528502002f, -0.848932055f,
    0.527199135f, -0.849741768f,
    0.525895027f, -0.850549481f,
    0.524589683f, -0.851355193f,
    0.523283103f, -0.852158902f,
    0.521975293f, -0.852960605f,
    0.520666254f, -0.853760301f,
    0.519355990f, -0.854557988f,
    0.518044504f, -0.855353665f,
    0.516731799f, -0.856147328f,
    0.515417878f, -0.856938977f,
    0.514102744f, -0.857728610f,
    0.512786401f, -0.858516224f,
    0.511468850f, -0.859301818f,
    0.510150097f, -0.860085390f,
    0.508830143f, -0.860866939f,
    0.507508991f, -0.861646461f,
    0.506186645f, -0.862423956f,
    0.504863109f, -0.863199422f,
    0.503538384f, -0.863972856f,
    0.502212474f, -0.864744258f,
    0.500885383f, -0.865513624f,
    0.499557113f, -0.866280954f,
    0.498227667f, -0.867046246f,
    0.496897049f, -0.867809497f,
    0.495565262f, -0.868570706f,
    0.494232309f, -0.869329871f,
    0.492898192f, -0.870086991f,
    0.491562916f, -0.870842063f,
    0.490226483f, -0.871595087f,
    0.488888897f, -0.872346059f,
    0.487550160f, -0.873094978f,
    0.486210276f, -0.873841843f,
    0.484869248f, -0.874586652f,
    0.483527079f, -0.875329403f,
    0.482183772f, -0.876070094f,
    0.480839331f, -0.876808724f,
    0.479493758f, -0.877545290f,
    0.478147056f, -0.878279792f,
    0.476799230f, -0.879012226f,
    0.475450282f, -0.879742593f,
    0.474100215f, -0.880470889f,
    0.472749032f, -0.881197113f,
    0.471396737f, -0.881921264f,
    0.470043332f, -0.882643340f,
    0.468688822f, -0.883363339f,
    0.467333209f, -0.884081259f,
    0.465976496f, -0.884797098f,
    0.464618686f, -0.885510856f,
    0.463259784f, -0.886222530f,
    0.461899791f, -0.886932119f,
    0.460538711f, -0.887639620f,
    0.459176548f, -0.888345033f,
    0.457813304f, -0.889048356f,
    0.456448982f, -0.889749586f,
    0.455083587f, -0.890448723f,
    0.453717121f, -0.891145765f,
    0.452349587f, -0.891840709f,
    0.450980989f, -0.892533555f,
    0.449611330f, -0.893224301f,
    0.448240612f, -0.893912945f,
    0.446868840f, -0.894599486f,
    0.445496017f, -0.895283921f,
    0.444122145f, -0.895966250f,
    0.442747228f, -0.896646470f,
    0.441371269f, -0.897324581f,
    0.439994271f, -0.898000580f,
    0.438616239f, -0.898674466f,
    0.437237174f, -0.899346237f,
    0.435857080f, -0.900015892f,
    0.434475961f, -0.900683429f,
    0.433093819f, -0.901348847f,
    0.431710658f, -0.902012144f,
    0.430326481f, -0.902673318f,
    0.428941292f, -0.903332368f,
    0.427555093f, -0.903989293f,
    0.426167889f, -0.904644091f,
    0.424779681f, -0.905296759f,
    0.423390474f, -0.905947298f,
    0.422000271f, -0.906595705f,
    0.420609074f, -0.907241978f,
    0.419216888f, -0.907886116f,
    0.417823716f, -0.908528119f,
    0.416429560f, -0.909167983f,
    0.415034424f, -0.909805708f,
    0.413638312f, -0.910441292f,
    0.412241227f, -0.911074734f,
    0.410843171f, -0.911706032f,
    0.409444149f, -0.912335185f,
    0.408044163f, -0.912962190f,
    0.406643217f, -0.913587048f,
    0.405241314f, -0.914209756f,
    0.403838458f, -0.914830312f,
    0.402434651f, -0.915448716f,
    0.401029897f, -0.916064966f,
    0.399624200f, -0.916679060f,
    0.398217562f, -0.917290997f,
    0.396809987f, -0.917900776f,
    0.395401479f, -0.918508394f,
    0.393992040f, -0.919113852f,
    0.392581674f, -0.919717146f,
    0.391170384f, -0.920318277f,
    0.389758174f, -0.920917242f,
    0.388345047f, -0.921514039f,
    0.386931006f, -0.922108669f,
    0.385516054f, -0.922701128f,
    0.384100195f, -0.923291417f,
    0.382683432f, -0.923879533f,
    0.381265769f, -0.924465474f,
    0.379847209f, -0.925049241f,
    0.378427755f, -0.925630831f,
    0.377007410f, -0.926210242f,
    0.375586178f, -0.926787474f,
    0.374164063f, -0.927362526f,
    0.372741067f, -0.927935395f,
    0.371317194f, -0.928506080f,
    0.369892447f, -0.929074581f,
    0.368466830f, -0.929640896f,
    0.367040346f, -0.930205023f,
    0.365612998f, -0.930766961f,
    0.364184790f, -0.931326709f,
    0.362755724f, -0.931884266f,
    0.361325806f, -0.932439629f,
    0.359895037f, -0.932992799f,
    0.358463421f, -0.933543773f,
    0.357030961f, -0.934092550f,
    0.355597662f, -0.934639130f,
    0.354163525f, -0.935183510f,
    0.352728556f, -0.935725689f,
    0.351292756f, -0.936265667f,
    0.349856130f, -0.936803442f,
    0.348418680f, -0.937339012f,
    0.346980411f, -0.937872376f,
    0.345541325f, -0.938403534f,
    0.344101426f, -0.938932484f,
    0.342660717f, -0.939459224f,
    0.341219202f, -0.939983753f,
    0.339776884f, -0.940506071f,
    0.338333767f, -0.941026175f,
    0.336889853f, -0.941544065f,
    0.335445147f, -0.942059740f,
    0.333999651f, -0.942573198f,
    0.332553370f, -0.943084437f,
    0.331106306f, -0.943593458f,
    0.329658463f, -0.944100258f,
    0.328209844f, -0.944604837f,
    0.326760452f, -0.945107193f,
    0.325310292f, -0.945607325f,
    0.323859367f, -0.946105232f,
    0.322407679f, -0.946600913f,
    0.320955232f, -0.947094366f,
    0.319502031f, -0.947585591f,
    0.318048077f, -0.948074586f,
    0.316593376f, -0.948561350f,
    0.315137929f, -0.949045882f,
    0.313681740f, -0.949528181f,
    0.312224814f, -0.950008245f,
    0.310767153f, -0.950486074f,
    0.309308760f, -0.950961666f,
    0.307849640f, -0.951435021f,
    0.306389795f, -0.951906137f,
    0.304929230f, -0.952375013f,
    0.303467947f, -0.952841648f,
    0.302005949f, -0.953306040f,
    0.300543241f, -0.953768190f,
    0.299079826f, -0.954228095f,
    0.297615707f, -0.954685755f,
    0.296150888f, -0.955141168f,
    0.294685372f, -0.955594334f,
    0.293219163f, -0.956045251f,
    0.291752263f, -0.956493919f,
    0.290284677f, -0.956940336f,
    0.288816408f, -0.957384501f,
    0.287347460f, -0.957826413f,
    0.285877835f, -0.958266071f,
    0.284407537f, -0.958703475f,
    0.282936570f, -0.959138622f,
    0.281464938f, -0.959571513f,
    0.279992643f, -0.960002146f,
    0.278519689f, -0.960430519f,
    0.277046080f, -0.960856633f,
    0.275571819f, -0.961280486f,
    0.274096910f, -0.961702077f,
    0.272621355f, -0.962121404f,
    0.271145160f, -0.962538468f,
    0.269668326f, -0.962953267f,
    0.268190857f, -0.963365800f,
    0.266712757f, -0.963776066f,
    0.265234030f, -0.964184064f,
    0.263754679f, -0.964589793f,
    0.262274707f, -0.964993253f,
    0.260794118f, -0.965394442f,
    0.259312915f, -0.965793359f,
    0.257831102f, -0.966190003f,
    0.256348682f, -0.966584374f,
    0.254865660f, -0.966976471f,
    0.253382037f, -0.967366292f,
    0.251897818f, -0.967753837f,
    0.250413007f, -0.968139105f,
    0.248927606f, -0.968522094f,
    0.247441619f, -0.968902805f,
    0.245955050f, -0.969281235f,
    0.244467903f, -0.969657385f,
    0.242980180f, -0.970031253f,
    0.241491885f, -0.970402839f,
    0.240003022f, -0.970772141f,
    0.238513595f, -0.971139158f,
    0.237023606f, -0.971503891f,
    0.235533059f, -0.971866337f,
    0.234041959f, -0.972226497f,
    0.232550307f, -0.972584369f,
    0.231058108f, -0.972939952f,
    0.229565366f, -0.973293246f,
    0.228072083f, -0.973644250f,
    0.226578264f, -0.973992962f,
    0.225083911f, -0.974339383f,
    0.223589029f, -0.974683511f,
    0.222093621f, -0.975025345f,
    0.220597690f, -0.975364885f,
    0.219101240f, -0.975702130f,
    0.217604275f, -0.976037079f,
    0.216106797f, -0.976369731f,
    0.214608811f, -0.976700086f,
    0.213110320f, -0.977028143f,
    0.211611327f, -0.977353900f,
    0.210111837f, -0.977677358f,
    0.208611852f, -0.977998515f,
    0.207111376f, -0.978317371f,
    0.205610413f, -0.978633924f,
    0.204108966f, -0.978948175f,
    0.202607039f, -0.979260123f,
    0.201104635f, -0.979569766f,
    0.199601758f, -0.979877104f,
    0.198098411f, -0.980182136f,
    0.196594598f, -0.980484862f,
    0.195090322f, -0.980785280f,
    0.193585587f, -0.981083391f,
    0.192080397f, -0.981379193f,
    0.190574755f, -0.981672686f,
    0.189068664f, -0.981963869f,
    0.187562129f, -0.982252741f,
    0.186055152f, -0.982539302f,
    0.184547737f, -0.982823551f,
    0.183039888f, -0.983105487f,
    0.181531608f, -0.983385110f,
    0.180022901f, -0.983662419f,
    0.178513771f, -0.983937413f,
    0.177004220f, -0.984210092f,
    0.175494253f, -0.984480455f,
    0.173983873f, -0.984748502f,
    0.172473084f, -0.985014231f,
    0.170961889f, -0.985277642f,
    0.169450291f, -0.985538735f,
    0.167938295f, -0.985797509f,
    0.166425904f, -0.986053963f,
    0.164913120f, -0.986308097f,
    0.163399949f, -0.986559910f,
    0.161886394f, -0.986809402f,
    0.160372457f, -0.987056571f,
    0.158858143f, -0.987301418f,
    0.157343456f, -0.987543942f,
    0.155828398f, -0.987784142f,
    0.154312973f, -0.988022017f,
    0.152797185f, -0.988257568f,
    0.151281038f, -0.988490793f,
    0.149764535f, -0.988721692f,
    0.148247679f, -0.988950265f,
    0.146730474f, -0.989176510f,
    0.145212925f, -0.989400428f,
    0.143695033f, -0.989622017f,
    0.142176804f, -0.989841278f,
    0.140658239f, -0.990058210f,
    0.139139344f, -0.990272812f,
    0.137620122f, -0.990485084f,
    0.136100575f, -0.990695025f,
    0.134580709f, -0.990902635f,
    0.133060525f, -0.991107914f,
    0.131540029f, -0.991310860f,
    0.130019223f, -0.991511473f,
    0.128498111f, -0.991709754f,
    0.126976696f, -0.991905700f,
    0.125454983f, -0.992099313f,
    0.123932975f, -0.992290591f,
    0.122410675f, -0.992479535f,
    0.120888087f, -0.992666142f,
    0.119365215f, -0.992850414f,
    0.117842062f, -0.993032350f,
    0.116318631f, -0.993211949f,
    0.114794927f, -0.993389211f,
    0.113270952f, -0.993564136f,
    0.111746711f, -0.993736722f,
    0.110222207f, -0.993906970f,
    0.108697444f, -0.994074879f,
    0.107172425f, -0.994240449f,
    0.105647154f, -0.994403680f,
    0.104121634f, -0.994564571f,
    0.102595869f, -0.994723121f,
    0.101069863f, -0.994879331f,
    0.099543619f, -0.995033199f,
    0.098017140f, -0.995184727f,
    0.096490431f, -0.995333912f,
    0.094963495f, -0.995480755f,
    0.093436336f, -0.995625256f,
    0.091908956f, -0.995767414f,
    0.090381361f, -0.995907229f,
    0.088853553f, -0.996044701f,
    0.087325535f, -0.996179829f,
    0.085797312f, -0.996312612f,
    0.084268888f, -0.996443051f,
    0.082740265f, -0.996571146f,
    0.081211447f, -0.996696895f,
    0.079682438f, -0.996820299f,
    0.078153242f, -0.996941358f,
    0.076623861f, -0.997060070f,
    0.075094301f, -0.997176437f,
    0.073564564f, -0.997290457f,
    0.072034653f, -0.997402130f,
    0.070504573f, -0.997511456f,
    0.068974328f, -0.997618435f,
    0.067443920f, -0.997723067f,
    0.065913353f, -0.997825350f,
    0.064382631f, -0.997925286f,
    0.062851758f, -0.998022874f,
    0.061320736f, -0.998118113f,
    0.059789571f, -0.998211003f,
    0.058258265f, -0.998301545f,
    0.056726821f, -0.998389737f,
    0.055195244f, -0.998475581f,
    0.053663538f, -0.998559074f,
    0.052131705f, -0.998640218f,
    0.050599749f, -0.998719012f,
    0.049067674f, -0.998795456f,
    0.047535484f, -0.998869550f,
    0.046003182f, -0.998941293f,
    0.044470772f, -0.999010686f,
    0.042938257f, -0.999077728f,
    0.041405641f, -0.999142419f,
    0.039872928f, -0.999204759f,
    0.038340120f, -0.999264747f,
    0.036807223f, -0.999322385f,
    0.035274239f, -0.999377670f,
    0.033741172f, -0.999430605f,
    0.032208025f, -0.999481187f,
    0.030674803f, -0.999529418f,
    0.029141509f, -0.999575296f,
    0.027608146f, -0.999618822f,
    0.026074718f, -0.999659997f,
    0.024541229f, -0.999698819f,
    0.023007681f, -0.999735288f,
    0.021474080f, -0.999769405f,
    0.019940429f, -0.999801170f,
    0.018406730f, -0.999830582f,
    0.016872988f, -0.999857641f,
    0.015339206f, -0.999882347f,
    0.013805389f, -0.999904701f,
    0.012271538f, -0.999924702f,
    0.010737659f, -0.999942350f,
    0.009203755f, -0.999957645f,
    0.007669829f, -0.999970586f,
    0.006135885f, -0.999981175f,
    0.004601926f, -0.999989411f,
    0.003067957f, -0.999995294f,
    0.001533980f, -0.999998823f
};


/**   
 * \par    
 * Example code for the generation of the floating-point sine table:
 * <pre>
 * tableSize = 512;    
 * for(n = 0; n < (tableSize + 1); n++)    
 * {    
 *	sinTable[n]=sin(2*pi*n/tableSize);    
 * }</pre>    
 * \par    
 * where pi value is  3.14159265358979    
 */

const float32_t sinTable_f32[FAST_MATH_TABLE_SIZE + 1] = {
   0.00000000f, 0.01227154f, 0.02454123f, 0.03680722f, 0.04906767f, 0.06132074f,
   0.07356456f, 0.08579731f, 0.09801714f, 0.11022221f, 0.12241068f, 0.13458071f,
   0.14673047f, 0.15885814f, 0.17096189f, 0.18303989f, 0.19509032f, 0.20711138f,
   0.21910124f, 0.23105811f, 0.24298018f, 0.25486566f, 0.26671276f, 0.27851969f,
   0.29028468f, 0.30200595f, 0.31368174f, 0.32531029f, 0.33688985f, 0.34841868f,
   0.35989504f, 0.37131719f, 0.38268343f, 0.39399204f, 0.40524131f, 0.41642956f,
   0.42755509f, 0.43861624f, 0.44961133f, 0.46053871f, 0.47139674f, 0.48218377f,
   0.49289819f, 0.50353838f, 0.51410274f, 0.52458968f, 0.53499762f, 0.54532499f,
   0.55557023f, 0.56573181f, 0.57580819f, 0.58579786f, 0.59569930f, 0.60551104f,
   0.61523159f, 0.62485949f, 0.63439328f, 0.64383154f, 0.65317284f, 0.66241578f,
   0.67155895f, 0.68060100f, 0.68954054f, 0.69837625f, 0.70710678f, 0.71573083f,
   0.72424708f, 0.73265427f, 0.74095113f, 0.74913639f, 0.75720885f, 0.76516727f,
   0.77301045f, 0.78073723f, 0.78834643f, 0.79583690f, 0.80320753f, 0.81045720f,
   0.81758481f, 0.82458930f, 0.83146961f, 0.83822471f, 0.84485357f, 0.85135519f,
   0.85772861f, 0.86397286f, 0.87008699f, 0.87607009f, 0.88192126f, 0.88763962f,
   0.89322430f, 0.89867447f, 0.90398929f, 0.90916798f, 0.91420976f, 0.91911385f,
   0.92387953f, 0.92850608f, 0.93299280f, 0.93733901f, 0.94154407f, 0.94560733f,
   0.94952818f, 0.95330604f, 0.95694034f, 0.96043052f, 0.96377607f, 0.96697647f,
   0.97003125f, 0.97293995f, 0.97570213f, 0.97831737f, 0.98078528f, 0.98310549f,
   0.98527764f, 0.98730142f, 0.98917651f, 0.99090264f, 0.99247953f, 0.99390697f,
   0.99518473f, 0.99631261f, 0.99729046f, 0.99811811f, 0.99879546f, 0.99932238f,
   0.99969882f, 0.99992470f, 1.00000000f, 0.99992470f, 0.99969882f, 0.99932238f,
   0.99879546f, 0.99811811f, 0.99729046f, 0.99631261f, 0.99518473f, 0.99390697f,
   0.99247953f, 0.99090264f, 0.98917651f, 0.98730142f, 0.98527764f, 0.98310549f,
   0.98078528f, 0.97831737f, 0.97570213f, 0.97293995f, 0.97003125f, 0.96697647f,
   0.96377607f, 0.96043052f, 0.95694034f, 0.95330604f, 0.94952818f, 0.94560733f,
   0.94154407f, 0.93733901f, 0.93299280f, 0.92850608f, 0.92387953f, 0.91911385f,
   0.91420976f, 0.90916798f, 0.90398929f, 0.89867447f, 0.89322430f, 0.88763962f,
   0.88192126f, 0.87607009f, 0.87008699f, 0.86397286f, 0.85772861f, 0.85135519f,
   0.84485357f, 0.83822471f, 0.83146961f, 0.82458930f, 0.81758481f, 0.81045720f,
   0.80320753f, 0.79583690f, 0.78834643f, 0.78073723f, 0.77301045f, 0.76516727f,
   0.75720885f, 0.74913639f, 0.74095113f, 0.73265427f, 0.72424708f, 0.71573083f,
   0.70710678f, 0.69837625f, 0.68954054f, 0.68060100f, 0.67155895f, 0.66241578f,
   0.65317284f, 0.64383154f, 0.63439328f, 0.62485949f, 0.61523159f, 0.60551104f,
   0.59569930f, 0.58579786f, 0.57580819f, 0.56573181f, 0.55557023f, 0.54532499f,
   0.53499762f, 0.52458968f, 0.51410274f, 0.50353838f, 0.49289819f, 0.48218377f,
   0.47139674f, 0.46053871f, 0.44961133f, 0.43861624f, 0.42755509f, 0.41642956f,
   0.40524131f, 0.39399204f, 0.38268343f, 0.37131719f, 0.35989504f, 0.34841868f,
   0.33688985f, 0.32531029f, 0.31368174f, 0.30200595f, 0.29028468f, 0.27851969f,
   0.26671276f, 0.25486566f, 0.24298018f, 0.23105811f, 0.21910124f, 0.20711138f,
   0.19509032f, 0.18303989f, 0.17096189f, 0.15885814f, 0.14673047f, 0.13458071f,
   0.12241068f, 0.11022221f, 0.09801714f, 0.08579731f, 0.07356456f, 0.06132074f,
   0.04906767f, 0.03680722f, 0.02454123f, 0.01227154f, 0.00000000f, -0.01227154f,
   -0.02454123f, -0.03680722f, -0.04906767f, -0.06132074f, -0.07356456f,
   -0.08579731f, -0.09801714f, -0.11022221f, -0.12241068f, -0.13458071f,
   -0.14673047f, -0.15885814f, -0.17096189f, -0.18303989f, -0.19509032f, 
   -0.20711138f, -0.21910124f, -0.23105811f, -0.24298018f, -0.25486566f, 
   -0.26671276f, -0.27851969f, -0.29028468f, -0.30200595f, -0.31368174f, 
   -0.32531029f, -0.33688985f, -0.34841868f, -0.35989504f, -0.37131719f, 
   -0.38268343f, -0.39399204f, -0.40524131f, -0.41642956f, -0.42755509f, 
   -0.43861624f, -0.44961133f, -0.46053871f, -0.47139674f, -0.48218377f, 
   -0.49289819f, -0.50353838f, -0.51410274f, -0.52458968f, -0.53499762f, 
   -0.54532499f, -0.55557023f, -0.56573181f, -0.57580819f, -0.58579786f, 
   -0.59569930f, -0.60551104f, -0.61523159f, -0.62485949f, -0.63439328f, 
   -0.64383154f, -0.65317284f, -0.66241578f, -0.67155895f, -0.68060100f, 
   -0.68954054f, -0.69837625f, -0.70710678f, -0.71573083f, -0.72424708f, 
   -0.73265427f, -0.74095113f, -0.74913639f, -0.75720885f, -0.76516727f, 
   -0.77301045f, -0.78073723f, -0.78834643f, -0.79583690f, -0.80320753f, 
   -0.81045720f, -0.81758481f, -0.82458930f, -0.83146961f, -0.83822471f, 
   -0.84485357f, -0.85135519f, -0.85772861f, -0.86397286f, -0.87008699f, 
   -0.87607009f, -0.88192126f, -0.88763962f, -0.89322430f, -0.89867447f, 
   -0.90398929f, -0.90916798f, -0.91420976f, -0.91911385f, -0.92387953f, 
   -0.92850608f, -0.93299280f, -0.93733901f, -0.94154407f, -0.94560733f, 
   -0.94952818f, -0.95330604f, -0.95694034f, -0.96043052f, -0.96377607f, 
   -0.96697647f, -0.97003125f, -0.97293995f, -0.97570213f, -0.97831737f, 
   -0.98078528f, -0.98310549f, -0.98527764f, -0.98730142f, -0.98917651f, 
   -0.99090264f, -0.99247953f, -0.99390697f, -0.99518473f, -0.99631261f, 
   -0.99729046f, -0.99811811f, -0.99879546f, -0.99932238f, -0.99969882f, 
   -0.99992470f, -1.00000000f, -0.99992470f, -0.99969882f, -0.99932238f, 
   -0.99879546f, -0.99811811f, -0.99729046f, -0.99631261f, -0.99518473f, 
   -0.99390697f, -0.99247953f, -0.99090264f, -0.98917651f, -0.98730142f, 
   -0.98527764f, -0.98310549f, -0.98078528f, -0.97831737f, -0.97570213f, 
   -0.97293995f, -0.97003125f, -0.96697647f, -0.96377607f, -0.96043052f, 
   -0.95694034f, -0.95330604f, -0.94952818f, -0.94560733f, -0.94154407f, 
   -0.93733901f, -0.93299280f, -0.92850608f, -0.92387953f, -0.91911385f, 
   -0.91420976f, -0.90916798f, -0.90398929f, -0.89867447f, -0.89322430f, 
   -0.88763962f, -0.88192126f, -0.87607009f, -0.87008699f, -0.86397286f, 
   -0.85772861f, -0.85135519f, -0.84485357f, -0.83822471f, -0.83146961f, 
   -0.82458930f, -0.81758481f, -0.81045720f, -0.80320753f, -0.79583690f, 
   -0.78834643f, -0.78073723f, -0.77301045f, -0.76516727f, -0.75720885f, 
   -0.74913639f, -0.74095113f, -0.73265427f, -0.72424708f, -0.71573083f, 
   -0.70710678f, -0.69837625f, -0.68954054f, -0.68060100f, -0.67155895f, 
   -0.66241578f, -0.65317284f, -0.64383154f, -0.63439328f, -0.62485949f, 
   -0.61523159f, -0.60551104f, -0.59569930f, -0.58579786f, -0.57580819f, 
   -0.56573181f, -0.55557023f, -0.54532499f, -0.53499762f, -0.52458968f, 
   -0.51410274f, -0.50353838f, -0.49289819f, -0.48218377f, -0.47139674f, 
   -0.46053871f, -0.44961133f, -0.43861624f, -0.42755509f, -0.41642956f, 
   -0.40524131f, -0.39399204f, -0.38268343f, -0.37131719f, -0.35989504f, 
   -0.34841868f, -0.33688985f, -0.32531029f, -0.31368174f, -0.30200595f, 
   -0.29028468f, -0.27851969f, -0.26671276f, -0.25486566f, -0.24298018f, 
   -0.23105811f, -0.21910124f, -0.20711138f, -0.19509032f, -0.18303989f, 
   -0.17096189f, -0.15885814f, -0.14673047f, -0.13458071f, -0.12241068f, 
   -0.11022221f, -0.09801714f, -0.08579731f, -0.07356456f, -0.06132074f, 
   -0.04906767f, -0.03680722f, -0.02454123f, -0.01227154f, -0.00000000f
};

/**   
 * \par    
 * Table values are in Q31 (1.31 fixed-point format) and generation is done in 
 * three steps.  First,  generate sin values in floating point:    
 * <pre>
 * tableSize = 512;      
 * for(n = 0; n < (tableSize + 1); n++)    
 * {    
 *	sinTable[n]= sin(2*pi*n/tableSize);    
 * } </pre>    
 * where pi value is  3.14159265358979    
 * \par    
 * Second, convert floating-point to Q31 (Fixed point):
 *	(sinTable[i] * pow(2, 31))    
 * \par    
 * Finally, round to the nearest integer value:
 * 	sinTable[i] += (sinTable[i] > 0 ? 0.5 :-0.5);    
 */
const q31_t sinTable_q31[FAST_MATH_TABLE_SIZE + 1] = {
   0x00000000, 0x01921D20, 0x03242ABF, 0x04B6195D, 0x0647D97C, 0x07D95B9E, 
   0x096A9049, 0x0AFB6805, 0x0C8BD35E, 0x0E1BC2E4, 0x0FAB272B, 0x1139F0CF, 
   0x12C8106F, 0x145576B1, 0x15E21445, 0x176DD9DE, 0x18F8B83C, 0x1A82A026, 
   0x1C0B826A, 0x1D934FE5, 0x1F19F97B, 0x209F701C, 0x2223A4C5, 0x23A6887F, 
   0x25280C5E, 0x26A82186, 0x2826B928, 0x29A3C485, 0x2B1F34EB, 0x2C98FBBA, 
   0x2E110A62, 0x2F875262, 0x30FBC54D, 0x326E54C7, 0x33DEF287, 0x354D9057, 
   0x36BA2014, 0x382493B0, 0x398CDD32, 0x3AF2EEB7, 0x3C56BA70, 0x3DB832A6, 
   0x3F1749B8, 0x4073F21D, 0x41CE1E65, 0x4325C135, 0x447ACD50, 0x45CD358F, 
   0x471CECE7, 0x4869E665, 0x49B41533, 0x4AFB6C98, 0x4C3FDFF4, 0x4D8162C4, 
   0x4EBFE8A5, 0x4FFB654D, 0x5133CC94, 0x5269126E, 0x539B2AF0, 0x54CA0A4B, 
   0x55F5A4D2, 0x571DEEFA, 0x5842DD54, 0x59646498, 0x5A82799A, 0x5B9D1154, 
   0x5CB420E0, 0x5DC79D7C, 0x5ED77C8A, 0x5FE3B38D, 0x60EC3830, 0x61F1003F, 
   0x62F201AC, 0x63EF3290, 0x64E88926, 0x65DDFBD3, 0x66CF8120, 0x67BD0FBD, 
   0x68A69E81, 0x698C246C, 0x6A6D98A4, 0x6B4AF279, 0x6C242960, 0x6CF934FC, 
   0x6DCA0D14, 0x6E96A99D, 0x6F5F02B2, 0x7023109A, 0x70E2CBC6, 0x719E2CD2, 
   0x72552C85, 0x7307C3D0, 0x73B5EBD1, 0x745F9DD1, 0x7504D345, 0x75A585CF, 
   0x7641AF3D, 0x76D94989, 0x776C4EDB, 0x77FAB989, 0x78848414, 0x7909A92D, 
   0x798A23B1, 0x7A05EEAD, 0x7A7D055B, 0x7AEF6323, 0x7B5D039E, 0x7BC5E290, 
   0x7C29FBEE, 0x7C894BDE, 0x7CE3CEB2, 0x7D3980EC, 0x7D8A5F40, 0x7DD6668F, 
   0x7E1D93EA, 0x7E5FE493, 0x7E9D55FC, 0x7ED5E5C6, 0x7F0991C4, 0x7F3857F6, 
   0x7F62368F, 0x7F872BF3, 0x7FA736B4, 0x7FC25596, 0x7FD8878E, 0x7FE9CBC0, 
   0x7FF62182, 0x7FFD885A, 0x7FFFFFFF, 0x7FFD885A, 0x7FF62182, 0x7FE9CBC0, 
   0x7FD8878E, 0x7FC25596, 0x7FA736B4, 0x7F872BF3, 0x7F62368F, 0x7F3857F6, 
   0x7F0991C4, 0x7ED5E5C6, 0x7E9D55FC, 0x7E5FE493, 0x7E1D93EA, 0x7DD6668F, 
   0x7D8A5F40, 0x7D3980EC, 0x7CE3CEB2, 0x7C894BDE, 0x7C29FBEE, 0x7BC5E290, 
   0x7B5D039E, 0x7AEF6323, 0x7A7D055B, 0x7A05EEAD, 0x798A23B1, 0x7909A92D, 
   0x78848414, 0x77FAB989, 0x776C4EDB, 0x76D94989, 0x7641AF3D, 0x75A585CF, 
   0x7504D345, 0x745F9DD1, 0x73B5EBD1, 0x7307C3D0, 0x72552C85, 0x719E2CD2, 
   0x70E2CBC6, 0x7023109A, 0x6F5F02B2, 0x6E96A99D, 0x6DCA0D14, 0x6CF934FC, 
   0x6C242960, 0x6B4AF279, 0x6A6D98A4, 0x698C246C, 0x68A69E81, 0x67BD0FBD, 
   0x66CF8120, 0x65DDFBD3, 0x64E88926, 0x63EF3290, 0x62F201AC, 0x61F1003F, 
   0x60EC3830, 0x5FE3B38D, 0x5ED77C8A, 0x5DC79D7C, 0x5CB420E0, 0x5B9D1154, 
   0x5A82799A, 0x59646498, 0x5842DD54, 0x571DEEFA, 0x55F5A4D2, 0x54CA0A4B, 
   0x539B2AF0, 0x5269126E, 0x5133CC94, 0x4FFB654D, 0x4EBFE8A5, 0x4D8162C4, 
   0x4C3FDFF4, 0x4AFB6C98, 0x49B41533, 0x4869E665, 0x471CECE7, 0x45CD358F, 
   0x447ACD50, 0x4325C135, 0x41CE1E65, 0x4073F21D, 0x3F1749B8, 0x3DB832A6, 
   0x3C56BA70, 0x3AF2EEB7, 0x398CDD32, 0x382493B0, 0x36BA2014, 0x354D9057, 
   0x33DEF287, 0x326E54C7, 0x30FBC54D, 0x2F875262, 0x2E110A62, 0x2C98FBBA, 
   0x2B1F34EB, 0x29A3C485, 0x2826B928, 0x26A82186, 0x25280C5E, 0x23A6887F, 
   0x2223A4C5, 0x209F701C, 0x1F19F97B, 0x1D934FE5, 0x1C0B826A, 0x1A82A026, 
   0x18F8B83C, 0x176DD9DE, 0x15E21445, 0x145576B1, 0x12C8106F, 0x1139F0CF, 
   0x0FAB272B, 0x0E1BC2E4, 0x0C8BD35E, 0x0AFB6805, 0x096A9049, 0x07D95B9E, 
   0x0647D97C, 0x04B6195D, 0x03242ABF, 0x01921D20, 0x00000000, 0xFE6DE2E0, 
   0xFCDBD541, 0xFB49E6A3, 0xF9B82684, 0xF826A462, 0xF6956FB7, 0xF50497FB, 
   0xF3742CA2, 0xF1E43D1C, 0xF054D8D5, 0xEEC60F31, 0xED37EF91, 0xEBAA894F, 
   0xEA1DEBBB, 0xE8922622, 0xE70747C4, 0xE57D5FDA, 0xE3F47D96, 0xE26CB01B, 
   0xE0E60685, 0xDF608FE4, 0xDDDC5B3B, 0xDC597781, 0xDAD7F3A2, 0xD957DE7A, 
   0xD7D946D8, 0xD65C3B7B, 0xD4E0CB15, 0xD3670446, 0xD1EEF59E, 0xD078AD9E, 
   0xCF043AB3, 0xCD91AB39, 0xCC210D79, 0xCAB26FA9, 0xC945DFEC, 0xC7DB6C50, 
   0xC67322CE, 0xC50D1149, 0xC3A94590, 0xC247CD5A, 0xC0E8B648, 0xBF8C0DE3, 
   0xBE31E19B, 0xBCDA3ECB, 0xBB8532B0, 0xBA32CA71, 0xB8E31319, 0xB796199B, 
   0xB64BEACD, 0xB5049368, 0xB3C0200C, 0xB27E9D3C, 0xB140175B, 0xB0049AB3, 
   0xAECC336C, 0xAD96ED92, 0xAC64D510, 0xAB35F5B5, 0xAA0A5B2E, 0xA8E21106, 
   0xA7BD22AC, 0xA69B9B68, 0xA57D8666, 0xA462EEAC, 0xA34BDF20, 0xA2386284, 
   0xA1288376, 0xA01C4C73, 0x9F13C7D0, 0x9E0EFFC1, 0x9D0DFE54, 0x9C10CD70, 
   0x9B1776DA, 0x9A22042D, 0x99307EE0, 0x9842F043, 0x9759617F, 0x9673DB94, 
   0x9592675C, 0x94B50D87, 0x93DBD6A0, 0x9306CB04, 0x9235F2EC, 0x91695663, 
   0x90A0FD4E, 0x8FDCEF66, 0x8F1D343A, 0x8E61D32E, 0x8DAAD37B, 0x8CF83C30, 
   0x8C4A142F, 0x8BA0622F, 0x8AFB2CBB, 0x8A5A7A31, 0x89BE50C3, 0x8926B677, 
   0x8893B125, 0x88054677, 0x877B7BEC, 0x86F656D3, 0x8675DC4F, 0x85FA1153, 
   0x8582FAA5, 0x85109CDD, 0x84A2FC62, 0x843A1D70, 0x83D60412, 0x8376B422, 
   0x831C314E, 0x82C67F14, 0x8275A0C0, 0x82299971, 0x81E26C16, 0x81A01B6D, 
   0x8162AA04, 0x812A1A3A, 0x80F66E3C, 0x80C7A80A, 0x809DC971, 0x8078D40D, 
   0x8058C94C, 0x803DAA6A, 0x80277872, 0x80163440, 0x8009DE7E, 0x800277A6, 
   0x80000000, 0x800277A6, 0x8009DE7E, 0x80163440, 0x80277872, 0x803DAA6A, 
   0x8058C94C, 0x8078D40D, 0x809DC971, 0x80C7A80A, 0x80F66E3C, 0x812A1A3A, 
   0x8162AA04, 0x81A01B6D, 0x81E26C16, 0x82299971, 0x8275A0C0, 0x82C67F14, 
   0x831C314E, 0x8376B422, 0x83D60412, 0x843A1D70, 0x84A2FC62, 0x85109CDD, 
   0x8582FAA5, 0x85FA1153, 0x8675DC4F, 0x86F656D3, 0x877B7BEC, 0x88054677, 
   0x8893B125, 0x8926B677, 0x89BE50C3, 0x8A5A7A31, 0x8AFB2CBB, 0x8BA0622F, 
   0x8C4A142F, 0x8CF83C30, 0x8DAAD37B, 0x8E61D32E, 0x8F1D343A, 0x8FDCEF66, 
   0x90A0FD4E, 0x91695663, 0x9235F2EC, 0x9306CB04, 0x93DBD6A0, 0x94B50D87, 
   0x9592675C, 0x9673DB94, 0x9759617F, 0x9842F043, 0x99307EE0, 0x9A22042D, 
   0x9B1776DA, 0x9C10CD70, 0x9D0DFE54, 0x9E0EFFC1, 0x9F13C7D0, 0xA01C4C73, 
   0xA1288376, 0xA2386284, 0xA34BDF20, 0xA462EEAC, 0xA57D8666, 0xA69B9B68, 
   0xA7BD22AC, 0xA8E21106, 0xAA0A5B2E, 0xAB35F5B5, 0xAC64D510, 0xAD96ED92, 
   0xAECC336C, 0xB0049AB3, 0xB140175B, 0xB27E9D3C, 0xB3C0200C, 0xB5049368, 
   0xB64BEACD, 0xB796199B, 0xB8E31319, 0xBA32CA71, 0xBB8532B0, 0xBCDA3ECB, 
   0xBE31E19B, 0xBF8C0DE3, 0xC0E8B648, 0xC247CD5A, 0xC3A94590, 0xC50D1149, 
   0xC67322CE, 0xC7DB6C50, 0xC945DFEC, 0xCAB26FA9, 0xCC210D79, 0xCD91AB39, 
   0xCF043AB3, 0xD078AD9E, 0xD1EEF59E, 0xD3670446, 0xD4E0CB15, 0xD65C3B7B, 
   0xD7D946D8, 0xD957DE7A, 0xDAD7F3A2, 0xDC597781, 0xDDDC5B3B, 0xDF608FE4, 
   0xE0E60685, 0xE26CB01B, 0xE3F47D96, 0xE57D5FDA, 0xE70747C4, 0xE8922622, 
   0xEA1DEBBB, 0xEBAA894F, 0xED37EF91, 0xEEC60F31, 0xF054D8D5, 0xF1E43D1C, 
   0xF3742CA2, 0xF50497FB, 0xF6956FB7, 0xF826A462, 0xF9B82684, 0xFB49E6A3, 
   0xFCDBD541, 0xFE6DE2E0, 0x00000000 
};

/**   
 * \par    
 * Table values are in Q15 (1.15 fixed-point format) and generation is done in 
 * three steps.  First,  generate sin values in floating point:    
 * <pre>
 * tableSize = 512;      
 * for(n = 0; n < (tableSize + 1); n++)    
 * {    
 *	sinTable[n]= sin(2*pi*n/tableSize);    
 * } </pre>    
 * where pi value is  3.14159265358979    
 * \par    
 * Second, convert floating-point to Q15 (Fixed point):
 *	(sinTable[i] * pow(2, 15))    
 * \par    
 * Finally, round to the nearest integer value:
 * 	sinTable[i] += (sinTable[i] > 0 ? 0.5 :-0.5);    
 */
const q15_t sinTable_q15[FAST_MATH_TABLE_SIZE + 1] = {
   0x0000, 0x0192, 0x0324, 0x04B6, 0x0648, 0x07D9, 0x096B, 0x0AFB, 0x0C8C, 0x0E1C, 0x0FAB, 0x113A, 0x12C8,
   0x1455, 0x15E2, 0x176E, 0x18F9, 0x1A83, 0x1C0C, 0x1D93, 0x1F1A, 0x209F, 0x2224, 0x23A7, 0x2528, 0x26A8,
   0x2827, 0x29A4, 0x2B1F, 0x2C99, 0x2E11, 0x2F87, 0x30FC, 0x326E, 0x33DF, 0x354E, 0x36BA, 0x3825, 0x398D,
   0x3AF3, 0x3C57, 0x3DB8, 0x3F17, 0x4074, 0x41CE, 0x4326, 0x447B, 0x45CD, 0x471D, 0x486A, 0x49B4, 0x4AFB,
   0x4C40, 0x4D81, 0x4EC0, 0x4FFB, 0x5134, 0x5269, 0x539B, 0x54CA, 0x55F6, 0x571E, 0x5843, 0x5964, 0x5A82,
   0x5B9D, 0x5CB4, 0x5DC8, 0x5ED7, 0x5FE4, 0x60EC, 0x61F1, 0x62F2, 0x63EF, 0x64E9, 0x65DE, 0x66D0, 0x67BD,
   0x68A7, 0x698C, 0x6A6E, 0x6B4B, 0x6C24, 0x6CF9, 0x6DCA, 0x6E97, 0x6F5F, 0x7023, 0x70E3, 0x719E, 0x7255,
   0x7308, 0x73B6, 0x7460, 0x7505, 0x75A6, 0x7642, 0x76D9, 0x776C, 0x77FB, 0x7885, 0x790A, 0x798A, 0x7A06,
   0x7A7D, 0x7AEF, 0x7B5D, 0x7BC6, 0x7C2A, 0x7C89, 0x7CE4, 0x7D3A, 0x7D8A, 0x7DD6, 0x7E1E, 0x7E60, 0x7E9D,
   0x7ED6, 0x7F0A, 0x7F38, 0x7F62, 0x7F87, 0x7FA7, 0x7FC2, 0x7FD9, 0x7FEA, 0x7FF6, 0x7FFE, 0x7FFF, 0x7FFE,
   0x7FF6, 0x7FEA, 0x7FD9, 0x7FC2, 0x7FA7, 0x7F87, 0x7F62, 0x7F38, 0x7F0A, 0x7ED6, 0x7E9D, 0x7E60, 0x7E1E,
   0x7DD6, 0x7D8A, 0x7D3A, 0x7CE4, 0x7C89, 0x7C2A, 0x7BC6, 0x7B5D, 0x7AEF, 0x7A7D, 0x7A06, 0x798A, 0x790A,
   0x7885, 0x77FB, 0x776C, 0x76D9, 0x7642, 0x75A6, 0x7505, 0x7460, 0x73B6, 0x7308, 0x7255, 0x719E, 0x70E3,
   0x7023, 0x6F5F, 0x6E97, 0x6DCA, 0x6CF9, 0x6C24, 0x6B4B, 0x6A6E, 0x698C, 0x68A7, 0x67BD, 0x66D0, 0x65DE,
   0x64E9, 0x63EF, 0x62F2, 0x61F1, 0x60EC, 0x5FE4, 0x5ED7, 0x5DC8, 0x5CB4, 0x5B9D, 0x5A82, 0x5964, 0x5843,
   0x571E, 0x55F6, 0x54CA, 0x539B, 0x5269, 0x5134, 0x4FFB, 0x4EC0, 0x4D81, 0x4C40, 0x4AFB, 0x49B4, 0x486A,
   0x471D, 0x45CD, 0x447B, 0x4326, 0x41CE, 0x4074, 0x3F17, 0x3DB8, 0x3C57, 0x3AF3, 0x398D, 0x3825, 0x36BA,
   0x354E, 0x33DF, 0x326E, 0x30FC, 0x2F87, 0x2E11, 0x2C99, 0x2B1F, 0x29A4, 0x2827, 0x26A8, 0x2528, 0x23A7,
   0x2224, 0x209F, 0x1F1A, 0x1D93, 0x1C0C, 0x1A83, 0x18F9, 0x176E, 0x15E2, 0x1455, 0x12C8, 0x113A, 0x0FAB,
   0x0E1C, 0x0C8C, 0x0AFB, 0x096B, 0x07D9, 0x0648, 0x04B6, 0x0324, 0x0192, 0x0000, 0xFE6E, 0xFCDC, 0xFB4A,
   0xF9B8, 0xF827, 0xF695, 0xF505, 0xF374, 0xF1E4, 0xF055, 0xEEC6, 0xED38, 0xEBAB, 0xEA1E, 0xE892, 0xE707,
   0xE57D, 0xE3F4, 0xE26D, 0xE0E6, 0xDF61, 0xDDDC, 0xDC59, 0xDAD8, 0xD958, 0xD7D9, 0xD65C, 0xD4E1, 0xD367,
   0xD1EF, 0xD079, 0xCF04, 0xCD92, 0xCC21, 0xCAB2, 0xC946, 0xC7DB, 0xC673, 0xC50D, 0xC3A9, 0xC248, 0xC0E9,
   0xBF8C, 0xBE32, 0xBCDA, 0xBB85, 0xBA33, 0xB8E3, 0xB796, 0xB64C, 0xB505, 0xB3C0, 0xB27F, 0xB140, 0xB005,
   0xAECC, 0xAD97, 0xAC65, 0xAB36, 0xAA0A, 0xA8E2, 0xA7BD, 0xA69C, 0xA57E, 0xA463, 0xA34C, 0xA238, 0xA129,
   0xA01C, 0x9F14, 0x9E0F, 0x9D0E, 0x9C11, 0x9B17, 0x9A22, 0x9930, 0x9843, 0x9759, 0x9674, 0x9592, 0x94B5,
   0x93DC, 0x9307, 0x9236, 0x9169, 0x90A1, 0x8FDD, 0x8F1D, 0x8E62, 0x8DAB, 0x8CF8, 0x8C4A, 0x8BA0, 0x8AFB,
   0x8A5A, 0x89BE, 0x8927, 0x8894, 0x8805, 0x877B, 0x86F6, 0x8676, 0x85FA, 0x8583, 0x8511, 0x84A3, 0x843A,
   0x83D6, 0x8377, 0x831C, 0x82C6, 0x8276, 0x822A, 0x81E2, 0x81A0, 0x8163, 0x812A, 0x80F6, 0x80C8, 0x809E,
   0x8079, 0x8059, 0x803E, 0x8027, 0x8016, 0x800A, 0x8002, 0x8000, 0x8002, 0x800A, 0x8016, 0x8027, 0x803E,
   0x8059, 0x8079, 0x809E, 0x80C8, 0x80F6, 0x812A, 0x8163, 0x81A0, 0x81E2, 0x822A, 0x8276, 0x82C6, 0x831C,
   0x8377, 0x83D6, 0x843A, 0x84A3, 0x8511, 0x8583, 0x85FA, 0x8676, 0x86F6, 0x877B, 0x8805, 0x8894, 0x8927,
   0x89BE, 0x8A5A, 0x8AFB, 0x8BA0, 0x8C4A, 0x8CF8, 0x8DAB, 0x8E62, 0x8F1D, 0x8FDD, 0x90A1, 0x9169, 0x9236,
   0x9307, 0x93DC, 0x94B5, 0x9592, 0x9674, 0x9759, 0x9843, 0x9930, 0x9A22, 0x9B17, 0x9C11, 0x9D0E, 0x9E0F,
   0x9F14, 0xA01C, 0xA129, 0xA238, 0xA34C, 0xA463, 0xA57E, 0xA69C, 0xA7BD, 0xA8E2, 0xAA0A, 0xAB36, 0xAC65,
   0xAD97, 0xAECC, 0xB005, 0xB140, 0xB27F, 0xB3C0, 0xB505, 0xB64C, 0xB796, 0xB8E3, 0xBA33, 0xBB85, 0xBCDA,
   0xBE32, 0xBF8C, 0xC0E9, 0xC248, 0xC3A9, 0xC50D, 0xC673, 0xC7DB, 0xC946, 0xCAB2, 0xCC21, 0xCD92, 0xCF04,
   0xD079, 0xD1EF, 0xD367, 0xD4E1, 0xD65C, 0xD7D9, 0xD958, 0xDAD8, 0xDC59, 0xDDDC, 0xDF61, 0xE0E6, 0xE26D,
   0xE3F4, 0xE57D, 0xE707, 0xE892, 0xEA1E, 0xEBAB, 0xED38, 0xEEC6, 0xF055, 0xF1E4, 0xF374, 0xF505, 0xF695,
   0xF827, 0xF9B8, 0xFB4A, 0xFCDC, 0xFE6E, 0x0000
};
