#include "mbed.h"
#include "CHEM_BOX_INTERFACE.h"
#include "MODSERIAL.h"
#include <stdio.h>
#include <stdarg.h>

//Mbed Objects

DigitalOut MBED_LED1(LED1);

DigitalOut AIO_ADC1_CS(p30);     

DigitalOut AIO_ADC2_CS(p29);  

PwmOut BUZZER_CONTROL(p26);

DigitalOut MCU_SR_CLEAR(p25);    

DigitalOut AIO_DAC1_CS(p24);    

DigitalOut AIO_DAC2_CS(p23);     

DigitalOut RED_LED(p20);

DigitalOut GREEN_LED(p14);
 

DigitalOut MFC_POWER_CONTROL(p22); 

PwmOut FAN_CONTROL(p21);       

SPI SPI1(p5,p6,p7);

DigitalOut MCU_SR_LOAD(p8); 

SPI SPI0(p11,p12,p13); 


BusOut TEMP_SENSE_ADDRESS(p15,p16,p17,p18);

DigitalOut TEMP_SENSE_CS(p19); 


MODSERIAL RS232_0(p9, p10, 1024, 1024); 

MODSERIAL RS232_1(p28, p27, 1024, 1024); 

// Make TX buffer 1024bytes and RX buffer use 512bytes.
MODSERIAL PC(USBTX, USBRX, 1024, 1024); // tx, rx

//Local Variables


static uint8_t TerminalEcho = 1;

static uint8_t HeaterBits = 0;

static uint16_t SolenoidBits = 0;

static uint8_t DigitalOutputBits = 0;

Timeout BuzzTimeout;
     
static uint16_t Thermocouple_FAULT = 0;     

static uint16_t Thermocouple_SCV = 0;     

static uint16_t Thermocouple_SCG = 0;     

static uint16_t Thermocouple_OC= 0;     

static float Temperature[12];

static float InternalTemperature[12];

static uint16_t ReadRawADC(uint8_t Channel);


//Local Functions

void InitTerminal();

void WriteRAW_DAC_Value(uint8_t Channel,uint16_t Data);



void InitChemBox()
{

    AIO_ADC1_CS = 1;     
    
    AIO_ADC2_CS = 1;   
    
    BUZZER_CONTROL.period_ms(1.0);
    
    MCU_SR_CLEAR = 1;   
    
    AIO_ADC1_CS = 1;    
    
    AIO_ADC2_CS = 1;    
    
    MFC_POWER_CONTROL = 0;
    
    FAN_CONTROL.period_us(1000);    
    
    FAN_CONTROL.write(0);
        
    SPI1.format(8,0);
    
    SPI1.frequency(4000000);
    
    MCU_SR_LOAD = 0;
    
    SPI0.format(8,0);
    
    SPI0.frequency(4000000);
    
    TEMP_SENSE_ADDRESS = 0;
    
    TEMP_SENSE_CS = 1;

    HeaterBits = 0;

    SolenoidBits = 0;

    DigitalOutputBits = 0;

    Thermocouple_FAULT = 0;     

    Thermocouple_SCV = 0;     
    
    Thermocouple_SCG = 0;     
    
    Thermocouple_OC= 0;     

    InitTerminal();
    
    RS232_0.baud(19200);
  
    RS232_1.baud(9600);
    

}


void SetFanSpeed(uint8_t S)
{
  if(S>100)
    S = 100;
    
    FAN_CONTROL = (float)(S)/100.0;
   
}

void EnableFan()
{
    SetFanSpeed(100);
}

void DisableFan()
{
    SetFanSpeed(0);
}    

void BuzzOff()
{
    BUZZER_CONTROL = 0;
}

void Buzz(float Time)
{
   BUZZER_CONTROL = 0.5;
   BuzzTimeout.attach(&BuzzOff, Time);
  
}


void EnableHeater(uint8_t RelayIndex)
{
    HeaterBits |= (1<<RelayIndex);
}

void DisableHeater(uint8_t RelayIndex)
{
    HeaterBits &= ~(1<<RelayIndex);
}

void EnableSolenoidValve(uint8_t SolenoidIndex)
{
    SolenoidBits |= (1<<SolenoidIndex);
}

void DisableSolenoidValue(uint8_t SolenoidIndex)
{
    SolenoidBits &= ~(1<<SolenoidIndex);
}

void DisableAllHeatersAndSolenoids()
{
    
    SolenoidBits = 0;
    HeaterBits = 0;

    MCU_SR_CLEAR = 1;
    
    MCU_SR_CLEAR = 0;

    MCU_SR_CLEAR = 1;
    
    MCU_SR_LOAD = 1;
   
    MCU_SR_LOAD = 0;
}

void EnableMiscDigitalOutput(uint8_t DigitalOutIndex)
{
    DigitalOutputBits |= (1<<DigitalOutIndex);
}

void DisableMiscDigitalOutput(uint8_t DigitalOutIndex)
{
    DigitalOutputBits &= ~(1<<DigitalOutIndex);
}

void FlushDigitalIO()
{
    SPI1.format(8,0);
    
    SPI1.write((SolenoidBits >> 8) & 0xFF);
    SPI1.write(SolenoidBits & 0xFF);
    SPI1.write(HeaterBits & 0xFF);
    SPI1.write(DigitalOutputBits & 0xFF);
    
    MCU_SR_LOAD = 1;
    MCU_SR_LOAD = 0;
}

//Make sure to call  ReadThermocouple before you call this so internal variables are updated
uint16_t ReadThermocouple_OC() 
{
    return Thermocouple_OC;
}
//Make sure to call  ReadThermocouple before you call this so internal variables are updated
uint16_t ReadThermocouple_SCG()
{
    return Thermocouple_SCG;
}
//Make sure to call  ReadThermocouple before you call this so internal variables are updated
uint16_t ReadThermocouple_SCV()
{
    return  Thermocouple_SCV;
}
//Make sure to call  ReadThermocouple before you call this so internal variables are updated
uint16_t ReadThermocouple_FAULT()
{
    return  Thermocouple_FAULT;
}


float ReadInternalTemperature(uint8_t ThermocoupleIndex)
{
    ReadThermocouple(ThermocoupleIndex); //this will yank out the Data
    return InternalTemperature[ThermocoupleIndex];
}


   
float ReadThermocouple(uint8_t ThermocoupleIndex)
{
    uint8_t i=0;
    uint32_t    ThermocoupleData = 0;
    uint8_t TempData[4];
    
    int16_t InternalTemp = 0;
    int16_t ThermocoupleTemp = 0;
    
    //reset SPi format
    SPI1.format(8,0);
    
    TEMP_SENSE_ADDRESS = ThermocoupleIndex & 0x1f;
    
    TEMP_SENSE_CS = 0;
    
    for(i=0;i<4;i++)
        TempData[i] = SPI1.write(0);
    
    TEMP_SENSE_CS = 1;


    ThermocoupleData =   (uint32_t)(TempData[3])         | 
                        (((uint32_t)(TempData[2]))<<8)   |
                        (((uint32_t)(TempData[1]))<<16)  |
                        (((uint32_t)(TempData[0]))<<24);
                        
    
    if(ThermocoupleData & 0x01)
         Thermocouple_OC |= (1<<ThermocoupleIndex);
    else
         Thermocouple_OC &= ~(1<<ThermocoupleIndex);       

    if(ThermocoupleData & 0x02)
        Thermocouple_SCG |= (1<<ThermocoupleIndex);
    else
        Thermocouple_SCG &= ~(1<<ThermocoupleIndex);

    if(ThermocoupleData & 0x04)
        Thermocouple_SCV |= (1<<ThermocoupleIndex);
    else
        Thermocouple_SCV &= ~(1<<ThermocoupleIndex);

    if(ThermocoupleData & (1<<16))
        Thermocouple_FAULT |= (1<<ThermocoupleIndex);
    else
        Thermocouple_FAULT &= ~(1<<ThermocoupleIndex);

    if(ThermocoupleData & (1<<15))
        InternalTemp = (int16_t) ( ( (ThermocoupleData>>4) & 0xFFF) | 0xF000);  //Sign extend in this case.... we need to map a 12 bit signed number to 16-bits
    else
        InternalTemp = (int16_t)( ( (ThermocoupleData>>4) & 0xFFF));


     if(ThermocoupleData & (0x80000000))
        ThermocoupleTemp = (int16_t)(((ThermocoupleData>>18) & 0x2FFF) | 0xC000);  //Sign extend in this case.... we need to map a 14 bit signed number to 16-bits
    else
        ThermocoupleTemp = (int16_t)(((ThermocoupleData>>18) & 0x2FFF));

     Temperature[ThermocoupleIndex] = (float)ThermocoupleTemp/4.0;

     InternalTemperature[ThermocoupleIndex] = (float)InternalTemp/16.0;;

    return  Temperature[ThermocoupleIndex];
}

float ReadMFC_AnalogInput(uint8_t Channel)
{
    if(Channel > 7)
        Channel = 7;

    return ((float)(ReadRawADC(Channel)) /4095.0) * 5.0;

}

void EnableMFC_Power()
{
    MFC_POWER_CONTROL = 1;
}

void DisableMFC_Power()
{
    MFC_POWER_CONTROL = 0;
}


float ReadMISC_AnalogInput(uint8_t Channel)
{

    if(Channel > 3)
        Channel = 3;

    return ((float)(ReadRawADC(Channel + 9)) /4095.0) * 5.0;

}



float Read4to20(uint8_t Channel)
{

    if(Channel > 1)
        Channel = 1;

    return (((float)(ReadRawADC(Channel + 7)) /4095.0) * 5.0) / 82;
    
}


float ReadBP_AnalogInput()
{

    return ((float)(ReadRawADC(15)) /4095.0) * 10.0;

}




static uint16_t ReadRawADC_Single(uint8_t Channel)
{
    uint8_t ControlByte[3];
    uint8_t ADC_Data[3];
    uint16_t V;
    
    SPI0.format(8,0);   //The ADC requires mode 0,0
    
    /*See Microchip manual DS21298E-page 21*/
    
    ControlByte[0] = (((Channel&0x07)>>2) & 0x01) | (3<<1);
    ControlByte[1] = Channel<<6;
    ControlByte[2] = 0;
    
    
    if(Channel<8)
        AIO_ADC1_CS = 0;
    else
        AIO_ADC2_CS = 0;
            
        //unroll the loop    
        ADC_Data[0] = SPI0.write(ControlByte[0]);
        ADC_Data[1] = SPI0.write(ControlByte[1]);
        ADC_Data[2] = SPI0.write(ControlByte[2]);
   
        AIO_ADC1_CS = 1;
        AIO_ADC2_CS = 1;
        
        
        V = ((uint16_t)(ADC_Data[1])<<8 | (uint16_t)(ADC_Data[2])) & 0xFFF;
   
        return (V);
        
}

#define FILTER_SIZE 64

static uint16_t ReadRawADC(uint8_t Channel)
{
    uint32_t Value = 0;
    
    for(int i=0;i<FILTER_SIZE;i++)
    {
        Value+=ReadRawADC_Single(Channel);
    }
    
    Value = Value / FILTER_SIZE;
    
    return Value;
    
}

void WriteMFC_AnalogOut(uint8_t Channel,float Value)
{

    if(Channel>7)
        Channel = 7;
        
    if(Value >5.0)
        Value = 5.0;
     
    if(Value<0.0)
        Value = 0.0;
  
    WriteRAW_DAC_Value(Channel,(uint16_t)((Value/5.0) * 4095));
 
}

void WriteMISC_AnalogOut(uint8_t Channel,float Value)
{
    if(Channel>3)
        Channel = 3;
        
    if(Value >5.0)
        Value = 5.0;
     
    if(Value<0.0)
        Value = 0.0;

        WriteRAW_DAC_Value(8+Channel,(uint16_t)((Value/5.0)*4095));
}

void WriteBP_AnalogOut(float Value)
{
        
    if(Value >10.0)
        Value = 10.0;
     
    if(Value<0.0)
        Value = 0.0;

        WriteRAW_DAC_Value(12,(uint16_t)((Value/10.0)*4095));
}


void WriteRAW_DAC_Value(uint8_t Channel,uint16_t Data)
{

    uint16_t DataOut;
    
    if(Channel<8)
        AIO_DAC1_CS = 0;
    else
    {
        AIO_DAC2_CS = 0;
        Channel -= 8;
     }   

    SPI0.format(8,1);   //The DAC requires mode 0,1
   
    DataOut = ((uint16_t)(Channel) & 0x7)<<12  | (Data & 0xFFF);
    
    SPI0.write((DataOut>>8)&0xFF);
    SPI0.write(DataOut&0xFF);

    AIO_DAC1_CS = 1;
    AIO_DAC2_CS = 1;

}




#define MAX_TERMINAL_LINE_CHARS 128
#define MAX_TERMINAL_CMD_CHARS  64

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

typedef void (*TerminalCallback)(char *);


typedef struct 
{
    const char *CommandString;
    TerminalCallback Callback;
    const char *HelpString;
        
} TerminalCallbackRecord;

//Callback function prototypes
void TerminalCmd_Help(char *arg);
void TerminalCmd_Stub(char *arg);
void TerminalCmd_EnableHeater(char *arg);
void TerminalCmd_DisableHeater(char *arg);
void TerminalCmd_EnableSolenoidValve(char *arg);
void TerminalCmd_DisableSolenoidValue(char *arg);
void TerminalCmd_DisableAllHeatersAndSolenoids(char *arg);
void TerminalCmd_EnableMiscDigitalOutput(char *arg);
void TerminalCmd_DisableMiscDigitalOutput(char *arg);
void TerminalCmd_FlushDigitalIO(char *arg);
void TerminalCmd_FanOn(char *arg);
void TerminalCmd_FanOff(char *arg);
void TerminalCmd_Buzz(char *arg);
void TerminalCmd_T(char *arg);
void TerminalCmd_MFCI(char *arg);
void TerminalCmd_MFCO(char *arg);
void TerminalCmd_4TO20(char *arg);
void TerminalCmd_AIN(char *arg);
void TerminalCmd_MFCON(char *arg);
void TerminalCmd_MFCOFF(char *arg);
void TerminalCmd_AOUT(char *arg);
void TerminalCmd_Reset(char *arg);
void TerminalCmd_ECHO_OFF(char *arg);
void TerminalCmd_BPOUT(char *arg);
void TerminalCmd_BPIN(char *arg);
void TerminalCmd_P0(char *arg);
void TerminalCmd_P1(char *arg);
void TerminalCmd_P0B(char *arg);
void TerminalCmd_P1B(char *arg);

//Populate this array with the callback functions and their terminal command string
TerminalCallbackRecord MyTerminalCallbackRecords[] ={   {"reset",TerminalCmd_Reset,"Resets the CHEM box"},

                                                        {"help",TerminalCmd_Help,"Lists available commands"},
                                                        
                                                        {"EH",TerminalCmd_EnableHeater,"Enables a heater channel.  Argument should be between 0 and 7.   Outputs will update when a FDIO command is issued"},
                                                        
                                                        {"DH",TerminalCmd_DisableHeater,"Disables a heater channel.  Argument should be between 0 and 7. Outputs will update when a FDIO command is issued"},
                                                        
                                                        {"ESV",TerminalCmd_EnableSolenoidValve,"Enables a solenoid channel. Argument should be between 0 and 11.  Outputs will update when a FDIO command is issued"},
                                                        
                                                        {"DSV",TerminalCmd_DisableSolenoidValue,"Disables a solenoid channel. Argument should be between 0 and 11.  Outputs will update when a FFDIO command is issued"},
                                                        
                                                        {"DAHAS",TerminalCmd_DisableAllHeatersAndSolenoids,"Disables all heaters and solenoids.  Command is immediately executed."},
                                                        
                                                        {"EMDO",TerminalCmd_EnableMiscDigitalOutput,"Enables a misc. digital output.  Argument should be between 0 and 3.  Output will update when a FDIO command is issued"},
                                                        
                                                        {"DMDO",TerminalCmd_DisableMiscDigitalOutput,"Enables a misc. digital output.  Argument should be between 0 and 3.  Output will update when a FDIO command is issued"},
                                                        
                                                        {"FDIO",TerminalCmd_FlushDigitalIO,"Updates the all of the digital IO channels"},
                                                        
                                                        {"FON",TerminalCmd_FanOn,"Turns on the fans"},
                                                        
                                                        {"FOFF",TerminalCmd_FanOff,"Turns off the fans"},
                                                        
                                                        {"BUZZ",TerminalCmd_Buzz,"Buzz for a little bit.  Argument should be a floating point number representing the number of seconds to buzz"},
                                                        
                                                        {"T",TerminalCmd_T,"Read thermocouple channel"},
                                                        
                                                        {"MFCI",TerminalCmd_MFCI,"Reads in voltage from MFC channel"},
                                                        
                                                        {"MFCO",TerminalCmd_MFCO,"Sets voltage at MFC output channel.  First argument should be the channel.  Second argument should be the voltage.  I.E.  MFCO 1.45"},
                                                        
                                                        {"AOUT",TerminalCmd_AOUT,"Sets voltage at misc. output channel. First argument should be the channel.  Second argument should be the voltage.  I.E.  AOUT 3.211"},
                                                        
                                                        {"4TO20",TerminalCmd_4TO20,"Reads a 4 to 20 mA channel"},
                                                        
                                                        {"AIN",TerminalCmd_AIN,"Reads a general purpose analog in channel"},
                                                        
                                                        {"MFCON",TerminalCmd_MFCON,"Turns on the MFC power"},
                                                        
                                                        {"MFCOFF",TerminalCmd_MFCOFF,"Turns off the MFC power"},
                                                        
                                                        {"BPOUT",TerminalCmd_BPOUT,"Sets a control voltage between 0 and 10 for the backpressure regulator.  I.E. BPOUT 1.21"},
                                                    
                                                        {"BPIN",TerminalCmd_BPIN,"Reads the Back pressure regulator feedback.   I.E. BPIN "},
                                                    
                                                        {"P0",TerminalCmd_P0,"Sends a command over RS232 port 0.  A carriage return will be added automatically."},
                                                  
                                                        {"P1",TerminalCmd_P1,"Sends a command over RS232 port 1.   A carriage return and line feed will be added automatically."},
                                                    
                                                        {"P0B",TerminalCmd_P0B,"Change the baud rate of Port 0 (default 19200)"},
                                                  
                                                        {"P1B",TerminalCmd_P1B,"Change the baud rate of Port 1 (default 9600)"},
                                                   
                                                        {"ECHO_OFF",TerminalCmd_ECHO_OFF,"Disables echoing of characters"}
                                                    };


extern "C" void mbed_reset();

void TerminalCmd_Reset(char *arg)
{
    mbed_reset();
}



void TerminalCmd_Stub(char *arg)
{
    PC.printf("stub \r\n");
}

void TerminalCmd_ECHO_OFF(char *arg)
{
    TerminalEcho = 0;
}

uint8_t RS232_Buff[2][256];

uint32_t RS232_Idx[2];


void ResetRS232_Response(uint32_t Channel)
{
      if(Channel == 0)
      {
        RS232_Idx[0] = 0;
      }  
      else
      {
          RS232_Idx[1] = 0;
       }
}

void TerminalCmd_P0(char *arg)
{
    ResetRS232_Response(0);
    RS232_0.printf("%s\r",arg);
}

void TerminalCmd_P1(char *arg)
{
    ResetRS232_Response(1);
    RS232_1.printf("%s\r\n",arg);
}


void ProcessRS232()
{
    uint8_t Val;
     
    while(RS232_0.readable())
    {
        Val = RS232_0.getc();
        
        if(Val == 0x03)
        {
            
            if(RS232_Idx[0]>0)
            {
                RS232_Buff[0][RS232_Idx[0]++] = 0;
                PC.printf("P:0:%s\r\n",&RS232_Buff[0][0]);
                ResetRS232_Response(0);
                break;
            }
        }
        else
        { 
            if((Val>0x020) && (Val<0x7F) && (RS232_Idx[0]<(sizeof(RS232_Buff)-1)))
            {
                RS232_Buff[0][RS232_Idx[0]++] = Val;
            }
        }
    }
        
        
    while(RS232_1.readable())
    {
        Val = RS232_1.getc();
        
        if(Val == 0x03)
        {
            if(RS232_Idx[1]>0)
            {
                RS232_Buff[1][RS232_Idx[1]++] = 0;
                PC.printf("P:1:%s\r\n",&RS232_Buff[1][0]);
                ResetRS232_Response(1);
                break;
            }
        }
        else
        { 
            if((Val>0x020) && (Val <0x7F) && (RS232_Idx[1]<(sizeof(RS232_Buff)-1)))
            {
                RS232_Buff[1][RS232_Idx[1]++] = Val;
            }
        }
    }
}


void TerminalCmd_P0B(char *arg)
{
    int Baud = 0;
    if(sscanf(arg,"%d",&Baud) == 1)
    {
         RS232_0.baud(Baud);
         PC.printf("Port  0 baud changed to %d",Baud);
    }
 }

void TerminalCmd_P1B(char *arg)
{
    int Baud = 0;
    if(sscanf(arg,"%d",&Baud) == 1)
    {
         RS232_1.baud(Baud);
         PC.printf("Port  1 baud changed to %d",Baud);
    }
 }

void TerminalCmd_MFCI(char *arg)
{
    int Channel = -1;
    float Data;
    
    if(sscanf(arg,"%d",&Channel) == 1)
    {
        if(Channel>=0 && Channel <=6)
        {
            Data =  ReadMFC_AnalogInput(Channel);
            PC.printf("MFCI:%d:%.3f",Channel,Data);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 6",Channel);
        }
    }
    else
    {
        for(Channel = 0; Channel<=6; Channel++)
        {
            Data =  ReadMFC_AnalogInput(Channel);
            PC.printf("MFCI:%d:%.3f\r\n",Channel,Data);
        }    
    }
}

void TerminalCmd_MFCON(char *arg)
{
    EnableMFC_Power();
}


void TerminalCmd_MFCOFF(char *arg)
{
    DisableMFC_Power();
}


void TerminalCmd_MFCO(char *arg)
{
    int Channel = -1;
    float Data = 0.0;
    
    if(sscanf(arg,"%d %f",&Channel,&Data) == 2)
    {
        if(Channel>=0 && Channel <=7)
        {
            WriteMFC_AnalogOut(Channel,Data);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 1",Channel);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Channel should be an integer between 0 and 7 and value should be a float between 0.0 and 5.0. i.e. MFCO 2 4.45",arg);
    }
}

void TerminalCmd_AOUT(char *arg)
{
    int Channel = -1;
    float Data = 0.0;
    
    if(sscanf(arg,"%d %f",&Channel,&Data) == 2)
    {
        if(Channel>=0 && Channel <=3)
        {
            WriteMISC_AnalogOut(Channel,Data);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 3",Channel);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Channel should be an integer between 0 and 7 and value should be a float between 0.0 and 5.0.   i.e. AOUT 1 1.25",arg);
    }
}


void TerminalCmd_BPOUT(char *arg)
{
    float Data = 0.0;
    
    if(sscanf(arg,"%f",&Data) == 1)
    {
          WriteBP_AnalogOut(Data);
    }
    else
    {
        PC.printf("Bad argument... %s.  value should be a float between 0.0 and 10.0.   i.e. BPOUT 1.25",arg);
    }
}

void TerminalCmd_BPIN(char *arg)
{
    float Data;
    Data =   ReadBP_AnalogInput();
    PC.printf("BPIN:%.3f",Data);
}

void TerminalCmd_4TO20(char *arg)
{
    int Channel = -1;
    float Data;
    
    if(sscanf(arg,"%d",&Channel) == 1)
    {
        if(Channel>=0 && Channel <=1)
        {
            Data =  Read4to20(Channel);
            PC.printf("4TO20:%d:%.3f",Channel,Data);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 1",Channel);
        }
    }
    else
    {
      
        for(Channel = 0;Channel<=1;Channel++)
        {
            Data =  Read4to20(Channel);
            PC.printf("4TO20:%d:%.5f\r\n",Channel,Data);
        }
      
    }


}

void TerminalCmd_AIN(char *arg)
{
    int Channel = -1;
    float Data;
    
    if(sscanf(arg,"%d",&Channel) == 1)
    {
        if(Channel>=0 && Channel <=3)
        {
            Data =  ReadMISC_AnalogInput(Channel);
            PC.printf("AIN:%d:%.3f",Channel,Data);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 3",Channel);
        }
    }
    else
    {
        for(Channel = 0;Channel<=3;Channel++)
        {
            Data =  ReadMISC_AnalogInput(Channel);
            PC.printf("AIN:%d:%.3f\r\n",Channel,Data);
        }
    }
}



void TerminalCmd_EnableHeater(char *arg)
{
    int Channel = -1;
    
    if(sscanf(arg,"%d",&Channel) == 1)
    {
        if(Channel>=0 && Channel <=7)
        {
            EnableHeater(Channel);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 7",Channel);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Should be integer between 0 and 7",arg);
    }
}

void TerminalCmd_DisableHeater(char *arg)
{
int Channel = -1;
    
    if(sscanf(arg,"%d",&Channel) == 1)
    {
        if(Channel>=0 && Channel <=7)
        {
            DisableHeater(Channel);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 7",Channel);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Should be integer between 0 and 7",arg);
    }

}

void TerminalCmd_EnableSolenoidValve(char *arg)
{

int Channel = -1;
    
    if(sscanf(arg,"%d",&Channel) == 1)
    {
        if(Channel>=0 && Channel <=11)
        {
            EnableSolenoidValve(Channel);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 11",Channel);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Should be integer between 0 and 11",arg);
    }
}

void TerminalCmd_DisableSolenoidValue(char *arg)
{
int Channel = -1;

   if(sscanf(arg,"%d",&Channel) == 1)
    {
        if( Channel >= 0 && Channel <= 11)
        {
            DisableSolenoidValue(Channel);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 11",Channel);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Should be integer between 0 and 11",arg);
    }

}
void TerminalCmd_DisableAllHeatersAndSolenoids(char *arg)
{
    
    DisableAllHeatersAndSolenoids();

}

void TerminalCmd_EnableMiscDigitalOutput(char *arg)
{
int Channel = -1;

  if(sscanf(arg,"%d",&Channel) == 1)
    {
        if(Channel>=0 && Channel <=3)
        {
            EnableMiscDigitalOutput(Channel);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 3",Channel);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Should be integer between 0 and 3",arg);
    }

}

void TerminalCmd_DisableMiscDigitalOutput(char *arg)
{
int Channel = -1;

  if(sscanf(arg,"%d",&Channel) == 1)
    {
        if(Channel>=0 && Channel <=3)
        {
           DisableMiscDigitalOutput(Channel);
        }
        else
        {
            PC.printf("%d is an invalid channel.   Channel should be integer between 0 and 3",Channel);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Should be integer between 0 and 3",arg);
    }

}

void TerminalCmd_FlushDigitalIO(char *arg)
{
    FlushDigitalIO();
}

void TerminalCmd_FanOn(char *arg)
{
     SetFanSpeed(100);   //PWMing the FANs doesn't work with the ME40100V1 models!   WE will just on or off
     MBED_LED1 = 1;
}

void TerminalCmd_FanOff(char *arg)
{
     SetFanSpeed(0);   //PWMing the FANs doesn't work with the ME40100V1 models!   WE will just on or off
      MBED_LED1 = 0;
}

void TerminalCmd_Fan(char *arg)
{
int Speed = -1;

 if(sscanf(arg,"%d",&Speed) == 1)
    {
        if(Speed>=0 && Speed<=100)
        {
           SetFanSpeed(Speed);
        }
        else
        {
            PC.printf("%d is an invalid speed.   Speed should be between 0 and 100",Speed);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Should be integer between 0 and 100",arg);
    }

}


void TerminalCmd_T(char *arg)
{
    float Temp = 0;
    int Channel = -1;
    int Status = 0;
    
     if(sscanf(arg,"%d",&Channel) == 1)
        {
                Temp = ReadThermocouple(Channel);
                
                if(ReadThermocouple_FAULT() & (1<<Channel))
                {
                    Status = 1;
                }
                else
                {
                    Status = 0;
                }
                
                 PC.printf("TEMP:%d:%.2f:%d\r\n",Channel,Temp,Status);
        }
        else
        {
            for(Channel = 0; Channel<12;Channel++)
            {
                 Temp = ReadThermocouple(Channel);
                 
                 if(ReadThermocouple_FAULT() & (1<<Channel))
                 {
                    Status = 1;
                 }
                 else
                 {
                    Status = 0;
                 }
                 
                  PC.printf("TEMP:%d:%.2f:%d\r\n",Channel,Temp,Status);
            }
        }
}

void TerminalCmd_Buzz(char *arg)
{
 
 float T = -1.0;

 if(sscanf(arg,"%f",&T) == 1)
    {
        if(T>=0.0 && T<=5.0)
        {
           Buzz(T);
        }
        else
        {
            PC.printf("%f is an invalid time period for buzz.   Time should be between 0.0 and 5.0 seconds",T);
        }
    }
    else
    {
        PC.printf("Bad argument... %s.  Should be float between 0.0 and 5.0",arg);
    }

}




//*****************************************************************
//Plumbing.....
//*****************************************************************

#define NUM_TERMINAL_COMMANDS  (sizeof(MyTerminalCallbackRecords)/sizeof(TerminalCallbackRecord))

char TerminalLineBuf[MAX_TERMINAL_LINE_CHARS];
uint8_t TerminalPos;
char TerminalCmdBuf[MAX_TERMINAL_CMD_CHARS+1];
char TerminalArgs[MAX_TERMINAL_LINE_CHARS-MAX_TERMINAL_CMD_CHARS];
uint8_t NextCharIn;
uint8_t CmdFound;
 
void TerminalBootMsg()
{

PC.printf("\r\n\r\n"); 
PC.printf("***********************************\r\n"); 
PC.printf("CHEM Control Box                   \r\n");
PC.printf("API Version %s \r\n",API_VERSION);
PC.printf("Copyright (C) <2013>  Eli Hughes\r\n");
PC.printf("Wavenumber LLC\r\n"); 
PC.printf("***********************************\r\n\r\n>"); 

}

void InitTerminal()
{

    PC.baud(115200);
    TerminalPos = 0;
    CmdFound = 0;
    TerminalBootMsg();
}

void TerminalCmd_Help(char *arg)
{
    uint8_t i;

    PC.printf("\r\n\r\nCommand List:\r\n");
    PC.printf("----------------------\r\n");

    for(i=0;i<NUM_TERMINAL_COMMANDS;i++)
    {
         PC.printf("%s  ---->  %s\r\n\r\n",MyTerminalCallbackRecords[i].CommandString,MyTerminalCallbackRecords[i].HelpString);    
    }

    PC.printf("\r\n\r\n");
}

void TerminalCmd_Reboot(char *arg)
{
      TerminalBootMsg();
}

void ProcessTerminal()
{
     uint8_t i,j;
     uint8_t ArgsFound;
        
    if(PC.readable())
    {
       NextCharIn = PC.getc();
       
        switch(NextCharIn)
        {
            case '\r':
             
             TerminalLineBuf[TerminalPos++] = 0x0;
             
             if(TerminalEcho)
             {
                 PC.putc(NextCharIn);
             }
             
             if(TerminalPos > 1)
             {
                 //find the command
                 i=0;
                 while(TerminalLineBuf[i]>0x20 &&  TerminalLineBuf[i]<0x7f)
                 {
                      TerminalCmdBuf[i] = TerminalLineBuf[i];
                      i++;
    
                    if(i==MAX_TERMINAL_CMD_CHARS)
                        {
                         break;
                        }
                 }
                    
                TerminalCmdBuf[i] = 0;
                TerminalCmdBuf[i+1] = 0;
                
                
                ArgsFound = TRUE;
                memset(TerminalArgs,0x00,sizeof(TerminalArgs));
                //scan for num terminator or next non whitespace
                while(TerminalLineBuf[i]<=0x20 && (i<MAX_TERMINAL_LINE_CHARS))
                {
                    if(TerminalLineBuf[i] == 0x00)
                    {
                    
                        //if we find a NULL terminator before a non whitespace character they flag for no arguments
                        ArgsFound = FALSE;
                        break;
                    }   
                    i++; 
                }
                
                if(ArgsFound == TRUE)
                {
                    strcpy(TerminalArgs,&TerminalLineBuf[i]);
                    
                    //trim trailing whitespace
                    i = sizeof(TerminalArgs)-1;
                    
                    while((TerminalArgs[i]<0x21) && (i>0))
                    {
                        TerminalArgs[i]= 0x00;
                        i--;
                    }       
                }
                
                CmdFound = FALSE;
                for(j=0;j<NUM_TERMINAL_COMMANDS;j++)
                {           
                    if(strcmp(TerminalCmdBuf,MyTerminalCallbackRecords[j].CommandString) == 0)
                    {
                        PC.printf("\r\n");
                        if(MyTerminalCallbackRecords[j].Callback != NULL)
                            MyTerminalCallbackRecords[j].Callback(TerminalArgs);
                    
                        CmdFound = TRUE;
                        break;
                    }             
                }        
                if(CmdFound == FALSE)
                {
                  PC.printf("\r\n%s command not recognized.\r\n\r\n",TerminalCmdBuf);
                  TerminalCmd_Help("no arg");
                  
                }
              }    
             PC.printf("\r\n");
             TerminalPos = 0;
            
            break;
            
            case '\b':
                if(TerminalPos > 0)
                {
                    TerminalPos--;  
                    if(TerminalEcho)
                    {  
                        PC.putc(NextCharIn);
                    }
                }
            break;
            
            default:
                
                if(TerminalPos == 0 && NextCharIn == 0x020)
                {
                     //Do nothing if space bar is pressed at beginning of line    
                }
                   else if(NextCharIn >= 0x20 && NextCharIn<0x7F)
                {
                    
                    if(TerminalPos < MAX_TERMINAL_LINE_CHARS-1)
                    {
                         TerminalLineBuf[TerminalPos++] = NextCharIn;
                         if(TerminalEcho)
                         {
                            PC.putc(NextCharIn);
                         }
                    }
                }
            
            break;
        
        }
    }
 
}




