/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: User-defined applications such as GPS, Temp, Accelerometer, LED indications etc.
            Event based actions such as LED blink on Tx, LED toggle on downlink etc

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Uttam Bhat
*/

#ifndef __APPLICATION_H__
#define __APPLICATION_H__

#include "board.h"
#include "Common.h"
#include "SerialDisplay.h"

/*!
 * LED ON/OFF definition
 */
#define LED_ON          0
#define LED_OFF         1

/*!
 * LED types
 */
enum eLedType
{
    Red = 0,// Red LED
    Yellow = 1, // Yellow LED
    Green = 2, // Green LED
    Usr = 3, // USR LED
};

/*!
 * user-defined Application names
 */
enum eAppType
{
    AppPressr = 0, // Pressure measurement application
    AppTemp, // Temperature measurement application
    AppHumid, // Humidity measurement application
    AppBat, // Battery measurement application
    AppRamp, // Generate Ramp bytes
    AppAccl, // Accelerometer application    
    AppPushButton, // Push button application example
};

/*!
 * Application
 */
class Application 
{
public:
    /*!  
     * \brief Initializes TX buffer address to private pointer
     *
     * \param [IN] memptr pointer to TX buffer
     */
    Application( uint8_t * memptr );

    /*!
     * \brief Application Destructor
     */
    ~Application( );
    
     /*!     
     * \brief Call application type and append to application buffer
     *
     * \param [IN] AppType application type: GPS, Battery, Temp etc.
     */
    void ApplicationCall( eAppType AppType );
     
     /*!     
     * \brief Append data to the TX buffer
     *
     * \param [IN] pData pointer to the data to be appended to TX buffer
     * \param [IN] Len lenght of the data to be appended to TX buffer
     */
    void ApplicationAppendData( uint8_t *pData, uint8_t len );
    
     /*!     
     * \brief Point the pointer corresponding to the TX buffer to specified index
     *
     * \param [IN] ptrPos index to where to point to in the TX buffer
     */
    void ApplicationPtrPos( uint8_t ptrPos );

private:
    /*!
     * Index to the application buffer
     */
    uint8_t BuffPtr;
    /*!
     * Application buffer address pointer
     */
    uint8_t *BuffAddr;   
};

class TimerLed
{
public:
    TimerEvent_t LedTimer;      
    TimerLed( eLedType );
    ~TimerLed( );
};

extern TimerLed RedLedTimer;

extern TimerLed YellowLedTimer;

extern TimerLed GreenLedTimer;

extern Application LoRaApp;

extern bool AppLed;

extern bool VerticalStatus;

#ifdef USE_CAYENNE_LPP
extern uint8_t maxLPPsize[4];
#endif

/*!  
* \brief Blink LED
*
* \param [IN] led Red, yellow or green LED
* \param [IN] time time in usec for which the LED must be ON
 */
void BlinkLED( eLedType led, uint32_t time );

/*!  
* \brief Turn LED ON if OFF; OFF if ON
*
* \param [IN] led Red, yellow or green LED
*/
void ToggleLED( eLedType led );

/*!  
* \brief Control the State of the LED
*
* \param [IN] led Red, yellow or green LED
* \param [IN] state ON or OFF
*/
void CtrlLED( eLedType led, uint8_t state );

void CheckOrientation( void );

#endif // __APPLICATION_H__