/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: Define events during Join, Tx & Rx
            Prepare TX packet by appending with appropriate application data

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Uttam Bhat
*/

#include "LoRaEventProc.h"

/*!
 * Defines the application data transmission duty cycle
 */
uint32_t TxDutyCycleTime = APP_TX_DUTYCYCLE;

bool AppLed = 0;

uint8_t AppCase = 0;

int32_t AppTxDutyCycle = APP_TX_DUTYCYCLE;

int32_t AppTxDutyCycleRnd = APP_TX_DUTYCYCLE_RND;

/*!
 * \brief   Prepares the payload of the frame based on application port
 */
void PrepareLoRaFrame( uint8_t port )
{

    switch( port )
    {    
    case 5: 
        {
            uint8_t tmp;
            uint8_t tmpLength;
            uint8_t ptrIndex = 0;

            // Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );

            if( IsDeviceReady == true )
            {
                if( DutyCycleIndx == 0 )
                {
                    tmp = 0xAA;     
                }
                else if( DutyCycleIndx == 1 )
                {
                    tmp = 0xBB;
                }
                else if( DutyCycleIndx > 1 )
                {
                    tmp = 0xCC;
                }
            }
            else
            {
                tmp = 0xFF;
            }
            tmpLength = 1;

            LoRaApp.ApplicationAppendData( &tmp, tmpLength ); // Populate lower nibble of 0th Byte with LED state

            /*!
            *  Generate Ramp data bytes
            *  Appends incremental values of 1 Byte each to TX buffer until Full
            */
            LoRaApp.ApplicationCall( AppRamp );

            break;
        }

    case 8: 
        { 
            uint8_t ptrIndex = 0;
                        
            //Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );                         
                        
            /*!
            *  Read Pressure
            *  Appends 2 Bytes to TX buffer
            */
            LoRaApp.ApplicationCall( AppPressr );
            
            /*!
            *  Read Temperature
            *  Appends 1 Byte to TX buffer
            */
            LoRaApp.ApplicationCall( AppTemp );
            
            /*!
            *  Read Humidity
            *  Appends 1 Byte to TX buffer
            */
            LoRaApp.ApplicationCall( AppHumid );

            /*!
            *  Read Accelerometer
            *  Appends 2 Bytes to TX buffer
            *   Value       Orientation
            *   0x99 0x00   horizontal + faceup
            *   0x66 0x00   horizontal + facedown
            *   0x00 0x11   vertical
            */
            LoRaApp.ApplicationCall( AppAccl ); // Generate Accelerometer data bytes    


            break;
        }
       
        // IKAS sensor using Cayenne Payload Format
        case 9: 
        { 
#ifdef USE_CAYENNE_LPP        
            uint8_t ptrIndex = 0;
            
            uint16_t AppSize = 0;
            
            uint8_t tmp[2] = {0};
            
            //Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );
            
            AppCase = AppCase > 3 ? 0 : AppCase;
            
            while( 1 )
            {            
                switch( AppCase ) 
                {
                    case 0: 
                    {
                        tmp[0] = 0;
                        tmp[1] = 115; // Data Type PRESSURE: 115
                        LoRaApp.ApplicationAppendData( tmp, 2 );
                        
                        /*!
                        *  Read Pressure
                        *  Appends 2 Bytes to TX buffer
                        */
                        LoRaApp.ApplicationCall( AppPressr );
                        
                        AppSize += maxLPPsize[AppCase];

                        break;
                    }
                    
                    case 1: 
                    {
                        tmp[0] = 1;
                        tmp[1] = 103; // Data Type TEMP: 103
                        LoRaApp.ApplicationAppendData( tmp, 2 );
                    
                        /*!
                        *  Read Temperature
                        *  Appends 1 Byte to TX buffer
                        */
                        LoRaApp.ApplicationCall( AppTemp );
                        
                        AppSize += maxLPPsize[AppCase];

                        break;
                    }
                    
                    case 2: 
                    {
                        tmp[0] = 2;
                        tmp[1] = 104; // Data Type HUMIDITY: 103
                        LoRaApp.ApplicationAppendData( tmp, 2 );
                        
                        /*!
                        *  Read Humidity
                        *  Appends 1 Byte to TX buffer
                        */
                        LoRaApp.ApplicationCall( AppHumid );
                        
                        AppSize += maxLPPsize[AppCase];

                        break;
                    }
                    
                    case 3: 
                    {
                        tmp[0] = 3;
                        tmp[1] = 113; // Data Type Accl: 113
                        LoRaApp.ApplicationAppendData( tmp, 2 );
                        /*!
                        *  Read Accelerometer                        
                        */
                        LoRaApp.ApplicationCall( AppAccl ); // Generate Accelerometer data bytes
                        
                        AppSize += maxLPPsize[AppCase];

                        break;
                    }
                }
                
                AppCase++;
                
                if( AppSize + maxLPPsize[AppCase] > LORAWAN_APP_DATA_SIZE )
                {
                    break;
                }
            }
                        
            AppDataSize = AppSize;            
#endif
            break;
        }

    // Push-Button Demo     
    case 11: 
        { 
            uint8_t ptrIndex = 0;
            
            //Point the pointer to position index of Tx Buffer
            LoRaApp.ApplicationPtrPos( ptrIndex );
            
            LoRaApp.ApplicationCall( AppPushButton ); // Transmit uplink counter   
            
            break;
        }
    
    default:
        break;
    }
}


/*!
 * \brief   Sets Interrupt for next payload transmission
 */
void InitNextTxInterrupt( uint8_t port )
{   
    switch( port )
    {  
    /* GPS Application Demo
        Set Timer interrupt for next uplink
    */
    case 5:
        { 
            // Schedule next packet transmission
            TxDutyCycleTime = AppTxDutyCycle + randr( -AppTxDutyCycleRnd, AppTxDutyCycleRnd );
            TimerSetValue( &TxNextPacketTimer, TxDutyCycleTime );
            TimerStart( &TxNextPacketTimer );
            break;
        }

    /* Senet + M2X demo
    Set Timer interrupt for next uplink
    */
    case 6: 
        { 
            
        }
        
    /* Senet GPS Demo 
       Set Timer interrupt for next uplink
    */  
    case 7: 
        { 
            // Schedule next packet transmission
            TxDutyCycleTime = APP_TX_DUTYCYCLE + randr( -APP_TX_DUTYCYCLE_RND, APP_TX_DUTYCYCLE_RND );
            TimerSetValue( &TxNextPacketTimer, TxDutyCycleTime );
            TimerStart( &TxNextPacketTimer );
            break;
        }

    /* Push Button Demo 
       Send Packet Immedietly if PC0 = GND
    */  
    case 11: 
        {             
            volatile bool PushButtonStatus;

            PushButtonStatus = UsrButton;
            
            if(PushButtonStatus == 0)
            {
                // Send Pkt immedietly if PC = GND
                DeviceState = DEVICE_STATE_SEND;
                NextTx = true;
            }
            else
            {
                // Keep polling
                IsTxIntUpdate = true;
            }
            
            break;
        }   

    /* Orientation Demo 
       Send Packet Immedietly if Mote is Vertical
    */  
    case 12: 
        {             
            CheckOrientation( );
            
            if(VerticalStatus == true)
            {
                // Send Pkt immedietly if PC = GND
                DeviceState = DEVICE_STATE_SEND;
                NextTx = true;
            }
            else
            {
                // Keep polling
                IsTxIntUpdate = true;
            }
            break;
        }   

    /* Compliance Test 
       Set Timer interrupt for next uplink
    */
    case 224:
        {               
            // Schedule next packet transmission
            TimerSetValue( &TxNextPacketTimer, COMPLIANCE_TX_DUTYCYCLE );
            TimerStart( &TxNextPacketTimer );
            break;
        }

    default:
        {
            // Schedule next packet transmission
            TimerSetValue( &TxNextPacketTimer, TxDutyCycleTime );
            TimerStart( &TxNextPacketTimer );
            break;
        }
    }

}

void InitDutyCycleTimer( void )
{       
    if( DutyCycleIndx == 0 )
    {
        AppTxDutyCycle = APP_TX_DUTYCYCLE_1;
        AppTxDutyCycleRnd = APP_TX_DUTYCYCLE_RND_1;       
    }
    else if( DutyCycleIndx == 1 )
    {
        AppTxDutyCycle = APP_TX_DUTYCYCLE_2;
        AppTxDutyCycleRnd = APP_TX_DUTYCYCLE_RND_2;     
    }
    else if( DutyCycleIndx > 1 )
    {
        AppTxDutyCycle = APP_TX_DUTYCYCLE_3;
        AppTxDutyCycleRnd = APP_TX_DUTYCYCLE_RND_3;     
    }
    
    // 30 min timer
    TxDutyCycleTime =  1800 * USEC;
    TimerSetValue( &TxDutyCycleTimer, TxDutyCycleTime );
    TimerStart( &TxDutyCycleTimer );
}

void InitDelayTimer( void )
{   
    // 30 min timer
    TxDutyCycleTime =  1800 * USEC;
    TimerSetValue( &TxDelayTimer, TxDutyCycleTime );
    TimerStart( &TxDelayTimer );
}

/*!
 * \brief   What to do during JOIN process ? blink/toggle LED etc.
 */
void JoinEvent( void )
{
    // CtrlLED is defined in LoRaBoardAppIf.h
    // param 1: LED color (Red, Yellow or Green)
    // param 2: LED_ON or LED_OFF
    //CtrlLED( Red, LED_ON ); 
}


/*!
 * \brief   What to do during TX ? blink/toggle LED etc.
 */
void TxEvent( void )
{
    /*
    int blinkTime = 25000;

    Blink Red LED for 25msec
    BlinkLED( Red, blinkTime );
    */
}

void RxEvent()
{
    /*
    // Toggle yellow LED
    //ToggleLED( Yellow );  

    // If Rx Data is 0x01 turn on Green LED else if 0x0 Turn Green LED off
    if( LoRaMacDownlinkStatus.BufferSize == 1 )
    {       
        if( LoRaMacDownlinkStatus.Buffer[0] == 0x01 )
        {
            AppLed = 1;
        }
        else
        {
            if( LoRaMacDownlinkStatus.Buffer[0] == 0x00 )
            {
                AppLed = 0;
            }
        }
    }

    if( AppLed != 0 )
    {
        // Turn  USR_LED ON
        //UsrLED = 3.3;
    }
    else
    {
        // Turn  USR_LED OFF
        //UsrLED = 0;
    }
    */
}

