#ifndef SPI25LC080_H
#define SPI25LC080_H

/**
 * Interface to the 25LC080 chip (http://ww1.microchip.com/downloads/en/DeviceDoc/22151b.pdf)
 *
 * @code
 * #include "mbed.h"
 *
 * SPI25LC080 e2prom;
 *
 * int main(void) {
 *    uint8_t buff[20];
 *    uint16_t addr = ...
 *
 *    while(1) {
 *       // read from e2prom
 *       e2prom.read(addr, buff, 20);
 *
 *       // write to e2prom
 *       e2prom.write(addr, buff, 20);
 *    }
 * }
 * @endcode
 */
class SPI25LC080 {
public:

    /** Create an interface to the 25LC080 chip
     *
     *  @param mosi the SPI MOSI pin
     *  @param sclk the SPI SCLK pin
     *  @param miso the SPI MISO pin
     */
    SPI25LC080(PinName mosi = P0_14, PinName sclk = P0_12, PinName miso = P0_15, PinName ssel = P0_13);
    
    /** Reads from the E2PROM
     *
     *  @param address the address to read from
     *  @param pBuf where to store the read data
     *  @param length the number of bytes to read
     */
    void read(uint16_t address, uint8_t* pData, uint32_t length);

    /** Writes to the E2PROM
     *
     *  @param address the address to write to
     *  @param pBuf the data to write
     *  @param length the number of bytes to write
     */
    void write(uint16_t address, uint8_t* pData, uint32_t length);
    
    /** Reads the status register 
     */
    uint8_t status();

private:
    SPI        m_spi;
    DigitalOut m_ssel;
    
    /** Sets the write enable latch/bit 
     */
    void write_enable();    
};

#endif

