/*
 *  Copyright 2013 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/******************************************************************************
 * Includes
 *****************************************************************************/

#include "mbed.h"
#include "TestAcc.h"
#include "MMA7455.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/

#define ACC_XMIN  (1<<0)  
#define ACC_XMAX  (1<<1)  
#define ACC_YMIN  (1<<2)  
#define ACC_YMAX  (1<<3)  
#define ACC_ZMIN  (1<<4)  
#define ACC_ZMAX  (1<<5)  
  
#define ACC_MIN_LIMIT  (-50)
#define ACC_MAX_LIMIT  ( 50)

#define ACC_THRESHOLD  (2)

/******************************************************************************
 * Public Functions
 *****************************************************************************/

/*
   Prerequisites:
 
   - For this test to work jumpers JP8 and JP9 on the LPC4088 Experiment Base Board
     must both be in positions 1-2
*/

bool TestAcc::runTest() {
    MMA7455 sensor(P0_27, P0_28);

    // Initialize the accelerometer
    if (!sensor.setMode(MMA7455::ModeMeasurement)) {
        printf("Unable to set mode for MMA7455!\n");
        return false;
    }

    // Calibrate it. It does not matter if it is on a level surface
    // as this test is only interested in relative values.
    if (!sensor.calibrate()) {
        printf("Failed to calibrate MMA7455!\n");
        return false;
    }

    int val[3] = {    0,    0,    0};
    int max[3] = {-1000,-1000,-1000};
    int min[3] = { 1000, 1000, 1000};
    int i;
    Timer t;
    t.start();
    
    // Read the x, y and z values every 100ms for up to five seconds.
    // When the accelerometer is working those values will sightly vary
    // even if the board is still. When a ACC_THRESHOLD difference has
    // been detected in each of the three axis the test is considered
    // successful.
    while (t.read() < 5) {
        if (!sensor.read(val[0], val[1], val[2])) {
            printf("Failed to read accelerometer data!\n");
            return false;
        }
        printf("ACC: x,y,z = {%5d, %5d, %5d}\n", val[0], val[1], val[2]);
        for (i = 0; i < 3; i++) {
            if (val[i] < min[i]) {
                min[i] = val[i];
            }
            if (val[i] > max[i]) {
                max[i] = val[i];
            }
        }
        for (i = 0; i < 3; i++) {
            if ((max[i] - min[i]) < ACC_THRESHOLD) {
                break;
            }
            if (i == 2) {
                printf("All three axis work\n");
                return true;
            }
        }
        wait(0.1);
    }
    printf("Accelerometer data invalid\n");
    printf("Not enough variation X {%d..%d}, Y {%d..%d}, Z {%d..%d}!\n",
           min[0], max[0], min[1], max[1], min[2], max[2]);
    return false;
}

bool TestAcc::alternativeTest() {
    MMA7455 sensor(P0_27, P0_28);

    // Initialize the accelerometer
    if (!sensor.setMode(MMA7455::ModeMeasurement)) {
        printf("Unable to set mode for MMA7455!\n");
        return false;
    }

    // Calibrate it. The board must be on a flat surface 
    // during calibration!
    if (!sensor.calibrate()) {
        printf("Failed to calibrate MMA7455!\n");
        return false;
    }

    printf("Now tilt the board in all directions (max 10 seconds)...\n");
    int x=0, y=0, z=0;
    uint8_t done = 0;
    char msg[30] = {0};

    Timer t;
    t.start();
    
    // Read the x, y and z values every 100ms for up to ten seconds.
    // Lift and tilt the board along all axis and look at the printed
    // output. There are ACC_MIN_LIMIT and ACC_MAX_LIMIT limits and 
    // as the value of an axis passes that limit it is printed and 
    // if all six limits (min/max for x,y and z) are passed within the
    // time limit the test is considered successful.
    while ((t.read() < 10) && (done != 0x3f)) {
        if (!sensor.read(x, y, z)) {
            printf("Failed to read accelerometer data!\n");
            return false;
        }
        printf("ACC: x,y,z = {%5d, %5d, %5d} %s\n", x, y, z, msg);
        if ((x < ACC_MIN_LIMIT) && !(done & ACC_XMIN)) {
            done |= ACC_XMIN;
            printf("Tilted XMIN\n");
        } else if ((x > ACC_MAX_LIMIT) && !(done & ACC_XMAX)) {
            done |= ACC_XMAX;
            printf("Tilted XMAX\n");
        }
        if ((y < ACC_MIN_LIMIT) && !(done & ACC_YMIN)) {
            done |= ACC_YMIN;
            printf("Tilted YMIN\n");
        } else if ((y > ACC_MAX_LIMIT) && !(done & ACC_YMAX)) {
            done |= ACC_YMAX;
            printf("Tilted XMAX\n");
        }
        if ((z < ACC_MIN_LIMIT) && !(done & ACC_ZMIN)) {
            done |= ACC_ZMIN;
            printf("Tilted ZMIN\n");
        } else if ((z > ACC_MAX_LIMIT) && !(done & ACC_ZMAX)) {
            done |= ACC_ZMAX;
            printf("Tilted ZMAX\n");
        }
        wait(0.1);
    }
    printf("Done with ACC tests!\n");
    return (t.read() < 10);
}

