
#ifndef EWGUI_H
#define EWGUI_H

#include "stdint.h"
#include "GUI.h"

/**
 * This is the main class to get Segger's emwin library up-and-running
 * with the mbed online tools.
 *
 * This class handles the porting/integration layer in emwin and the virtual
 * methods must be implemented in a sub-class to handle such things as
 * memory allocation, framebuffer association, display size, and touch
 * events.
 *
 */
class EwGui {
public:


    /**
     * Returns the address to a memory block which may be used by emwin
     * to allocate objects. The memory block returned will be given to emwin
     * by calling GUI_ALLOC_AssignMemory.
     */
    virtual void* getMemoryBlockAddress() = 0;

    /**
     * Returns the size of the memory block returned by
     * getMemoryBlockAddress().
     */
    virtual uint32_t getMemoryBlockSize() = 0;

    /**
     * Returns the width of the display.
     */
    virtual uint32_t getDisplayWidth() = 0;

    /**
     * Returns the height of the display.
     */
    virtual uint32_t getDisplayHeight() = 0;

    /**
     * Returns the address of the framebuffer (video RAM). This address will
     * be given to emwin by a call to LCD_SetVRAMAddrEx.
     */
    virtual void* getFrameBufferAddress() = 0;

    /**
     * Get touch coordinates (if the display has a touch panel)
     *
     * @param x the x coordinate
     * @param y the y coordinate
     * $param z the z coordinate. If the value is > 0 the panel is touched.
     */
    virtual void getTouchValues(int32_t* x, int32_t* y, int32_t* z) = 0;

    /**
     * Must be called regularly for emwin to work properly
     */
    void exec();

    /**
     * Must be called regularly for touch events to be retrieved
     */
    void execTouch();

    /**
     * Returns the x coordinate of the latest touch event
     */
    int32_t getTouchX() {return _touchX;}

    /**
     * Returns the y coordinate of the latest touch event
     */
    int32_t getTouchY() {return _touchY;}

protected:
    EwGui();
    void init();

private:

    bool _initialized;
    bool _penIsDown;
    int32_t _touchX;
    int32_t _touchY;
    GUI_PID_STATE _touchState;

};

#endif
