#include "RAMFileSystem.h"
#include "mbed_debug.h"

#define RAMFS_DBG       0

#define SECTOR_SIZE   512

RAMFileSystem::RAMFileSystem(uint32_t addr, uint32_t size, const char* name) :
    FATFileSystem(name) {
    memStart = addr;
    memSize = size - (size % SECTOR_SIZE);
    status = 1; //1: disk not initialized
}

int RAMFileSystem::disk_initialize() {
    debug_if(RAMFS_DBG, "init RAM fs\n");
    status = 0; //OK
    return status;
}

int RAMFileSystem::disk_write(const uint8_t *buffer, uint64_t sector) {
    debug_if(RAMFS_DBG, "write to sector %llu\n", sector);
    if (sector >= disk_sectors()) {
        return 1;
    }
    
    memcpy((uint8_t*)(memStart + SECTOR_SIZE*((uint32_t)sector)), buffer, SECTOR_SIZE);
    return 0;
}

int RAMFileSystem::disk_read(uint8_t *buffer, uint64_t sector) {
    debug_if(RAMFS_DBG, "read from sector %llu\n", sector);
    if (sector >= disk_sectors()) {
        return 1;
    }
    
    memcpy(buffer, (uint8_t*)(memStart + SECTOR_SIZE*((uint32_t)sector)), SECTOR_SIZE);
    return 0;
}

int RAMFileSystem::disk_nwrite(const uint8_t *buffer, uint64_t sector, uint8_t count) {
    debug_if(RAMFS_DBG, "write to sector(s) %llu..%llu\n", sector, sector+count);
    if ((sector+count-1) >= disk_sectors()) {
        return 1;
    }
    
    memcpy((uint8_t*)(memStart + SECTOR_SIZE*((uint32_t)sector)), buffer, SECTOR_SIZE*count);
    return 0;
}

int RAMFileSystem::disk_nread(uint8_t *buffer, uint64_t sector, uint8_t count) {
    debug_if(RAMFS_DBG, "read from sector(s) %llu..%llu\n", sector, sector+count);
    if ((sector+count-1) >= disk_sectors()) {
        return 1;
    }
    
    memcpy(buffer, (uint8_t*)(memStart + SECTOR_SIZE*((uint32_t)sector)), SECTOR_SIZE*count);
    return 0;
}

int RAMFileSystem::disk_status() { 
    debug_if(RAMFS_DBG, "disk status %d\n", status);
    return status; 
}
int RAMFileSystem::disk_sync() { 
    return 0; 
}
uint64_t RAMFileSystem::disk_sectors() { 
    debug_if(RAMFS_DBG, "returning fs has %u sectors\n", memSize/SECTOR_SIZE);
    return memSize/SECTOR_SIZE; 
}

uint64_t RAMFileSystem::disk_size() {
    return memSize;
}

