/******************************************************************************
 * Includes
 *****************************************************************************/
#include "mbed.h"
#include "sdram.h"

/******************************************************************************
 * Local variables
 *****************************************************************************/

// led1 + led 2 -> active low
static DigitalOut led1(LED1);
static DigitalOut led2(LED2);
// led3 + led 4 -> active high
static DigitalOut led3(LED3);
static DigitalOut led4(LED4);

static DigitalIn p2_10(P2_10);
static InterruptIn wkp(P2_10);

static int cnt = 0;

/******************************************************************************
 * Local functions
 *****************************************************************************/

#define MIND_BUSY           0x00000001  /* MII is Busy                       */
#define DP83848C_DEF_ADR    0x0100      /* Default PHY device address        */
#define MII_WR_TOUT         0x00050000  /* MII Read timeout count            */
#define PHY_REG_BMCR        0x00        /* Basic Mode Control Register       */
#define PHY_BMCR_PWR_DOWN   (1 << 11)    /* 1 = General power down mode       */

static int phy_write(unsigned int PhyReg, unsigned short Data) {
    unsigned int timeOut;

    LPC_EMAC->MADR = DP83848C_DEF_ADR | PhyReg;
    LPC_EMAC->MWTD = Data;

    for(timeOut = 0; timeOut < MII_WR_TOUT; timeOut++) {     /* Wait until operation completed */
        if((LPC_EMAC->MIND & MIND_BUSY) == 0) {
            return 0;
        }
    }

    return -1;
}

static void powerDown() {
    // SRC[SLEEPDEEP] set to 1 = sleep
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;

    LPC_SC->PCON = 0x01;
    
    // wait for interrupt
    __WFI();

}

static void incCnt() {

    // A counter is increased by one when the LPC4088 is brought out of
    // power down. The 4 LEDs will reflect the state of the counter.
    // Please note that when a LED is on more power will be consumed.
    // Press the button enough times (16) and all LEDs will be off
    // again.

    cnt++;
        
    if (cnt & 0x01) {
        led4 = 1;
    }
    else {
        led4 = 0;    
    }
    
    if (cnt & 0x02) {
        led3 = 1;
    }
    else {
        led3 = 0;    
    }    
    
    if (cnt & 0x04) {
        led2 = 0;
    }
    else {
        led2 = 1;    
    }    
    
    if (cnt & 0x08) {
        led1 = 0;
    }
    else {
        led1 = 1;    
    }        
}

/******************************************************************************
 * Local functions
 *****************************************************************************/

int main() {
    // user push button is available on P2.10. Make sure pull-up is enabled
    // on this pin since it is put low when the button is pressed.
    p2_10.mode(PullUp);
    
    // enable interrupt on p2.10 and register interrupt handler. When the 
    // user-push button is pressed the application will be brought out
    // of power-down mode and incCnt function will be called. 
    wkp.rise(&incCnt);
    
    // turn off all LEDs by default
    led1 = 1;
    led2 = 1;
    led3 = 0;
    led4 = 0;
    

    // ------------------------------------------------------->
    // Initialize SDRAM and set CKE 0 to make sure SDRAM is
    // in power-down mode. CKE is also connected to a 1K5 pull
    // down resistor
    sdram_init();
    DigitalOut cke(P2_24);
    cke = 0; // set low not to draw any current
    // <-------------------------------------------------------
    
    
    // ------------------------------------------------------->
    // Initialize Ethernet by instantiating the Ethernet class
    // The Ethernet interface doesn't contain a power-down
    // method so we reimplement phy_write in this app and 
    // puts the PHY in power-down here
    Ethernet eth;    
    phy_write(PHY_REG_BMCR, PHY_BMCR_PWR_DOWN);
    // <-------------------------------------------------------

    
    // must be low to disable U7 (STMPS2171STR)
    DigitalOut p1_19(P1_19);
    p1_19 = 0;


    // p2.18 connected to 1K5 pull-down resistor
    DigitalOut p2_18(P2_18);
    p2_18 = 0;
    
    
    while(1) {  
        // when the above initialization is done, all LEDs are off, 
        // HDK is disaabled (jumper on JP3 removed) and LPC4088 in
        // power-down, the measured current consumption is ~5 mA.
        
        powerDown();
        
        // In this example we are not doing anything here when the LPC4088 is 
        // brought out of power-down. Instead the incCnt function is registerd
        // as an interrupt handler and will be called when the user push-button
        // is pressed.
        
    }    
}