#include "EwGuiImpl.h"

#include "LcdController.h"
#include "EaLcdBoard.h"
#include "sdram.h"
#include "AR1021.h"
#include "TSC2046.h"


#define PIXEL_SIZE (2)

EwGuiImpl::EwGuiImpl() {

    EaLcdBoard lcdBoard(P0_27, P0_28);
    EaLcdBoard::Result result;
    LcdController::Config lcdCfg;
    EaLcdBoard::TouchParams_t touchParams;

    do {

        result = lcdBoard.open(NULL, NULL);
        if (result != EaLcdBoard::Ok) {
            printf("Failed to open display: %d\n", result);
            break;
        }

        result = lcdBoard.getLcdConfig(&lcdCfg);
        if (result != EaLcdBoard::Ok) {
            printf("Failed to get LCD configuration: %d\n", result);
            break;
        }

        _width = lcdCfg.width;
        _height = lcdCfg.height;

        // allocate buffer, width x height x 2 (2 bytes = 16 bit color data)
        _fb = (uint32_t)malloc(_width*_height*PIXEL_SIZE);
        if (_fb == 0) {
            printf("Failed to allocate frame buffer\n");
            break;
        }

        result = lcdBoard.setFrameBuffer(_fb);
        if (result != EaLcdBoard::Ok) {
            printf("Failed to activate frameBuffer: %d\n", result);
            break;
        }

        _memSz = 1024*1024*5;
        _mem = (uint32_t)malloc(_memSz);
        if (_mem == 0) {
            printf("Failed to allocate memory block for emwin\n");
            break;
        }


        memset((void*)_fb, 0x0, _width*_height*PIXEL_SIZE);

        result = lcdBoard.getTouchParameters(&touchParams);
        if (result != EaLcdBoard::Ok) {
            printf("Failed to get touch panel parameters: %d\n", result);
            break;
        }

        // create touch panel
        switch(touchParams.panelId) {
        case EaLcdBoard::TouchPanel_AR1021:
            printf("creating AR1021 touch panel\n");
            _touch = new AR1021(P2_27, P2_26, P2_22, P2_21, P2_25);
            break;

        case EaLcdBoard::TouchPanel_TSC2046:
        case EaLcdBoard::TouchPanelUnknown:
        default:
            // we always default to TSC2046 even if we cannot
            // detect which panel is used

            printf("creating TSC2046 touch panel\n");

            _touch = new TSC2046(P2_27, P2_26, P2_22, P2_21);
            break;

        }

        if (!_touch->init(_width, _height)) {
            printf("TouchPanel.init failed\n");
            break;
        }

        init();

    } while(0);


}

void* EwGuiImpl::getMemoryBlockAddress() {
    return (void*)_mem;
}

uint32_t EwGuiImpl::getMemoryBlockSize() {
    return _memSz;
}

uint32_t EwGuiImpl::getDisplayWidth() {
    return _width;
}

uint32_t EwGuiImpl::getDisplayHeight() {
    return _height;
}

void* EwGuiImpl::getFrameBufferAddress() {
    return (void*)_fb;
}

void EwGuiImpl::getTouchValues(int32_t* x, int32_t* y, int32_t* z) {

    if (!x || !y || !z) return;

    TouchPanel::touchCoordinate_t coord;
    _touch->read(coord);

    *x = coord.x;
    *y = coord.y;
    *z = coord.z;
}

void EwGuiImpl::calibrate() {

    uint16_t x = 0;
    uint16_t y = 0;
    bool hasMorePoints = false;

    EwPainter painter;
    painter.saveContext();

    do {

        printf("Starting calibration\n");
        if (!_touch->calibrateStart()) {
            printf("Failed to start calibration\n");
            break;
        }

        do {
            if (!_touch->getNextCalibratePoint(&x, &y)) {
                printf("Failed to get next calibrate point\n");
                break;
            }

            printf("calib: x=%d, y=%d\n", x, y);
            drawCalibPoint(painter, x, y);

            if (!_touch->waitForCalibratePoint(&hasMorePoints, 0)) {
                printf("Failed waiting for calibration point\n");
                break;
            }

        } while(hasMorePoints);

        printf("Calibration done\n");


    } while (0);


    painter.clear();
    painter.restoreContext();
}

void EwGuiImpl::drawCalibPoint(EwPainter &painter, int32_t x, int32_t y) {

    painter.clear();
    painter.setColor(EW_WHITE);
    painter.drawStringHorizCenter("Touch circle to calibrate",
            getDisplayWidth()/2, getDisplayHeight()/2);

    painter.fillCircle(x, y, 10);
    painter.setColor(EW_BLACK);
    painter.fillCircle(x, y, 5);
}
