/*
 * Servo.cpp - libreria para el manejo del servo sg90
 * @Author Patricio Silva
 * @Date 13/10/2019
 */

#include "Servo.h"


void setup(_Servo *s, PwmOut *pin){
    s->currentPos = 90;
    s->finalPos = 90;
    s->speed = 0;
    s->moving = false;
    s->pwm90 = 127;
    s->pinPWM = pin;
    s->period_us = 20000;
    s->min_pulse_width_us = 1000;
    s->max_pulse_width_us = 2000;
    s->currentPulseWidth = (s->max_pulse_width_us-s->min_pulse_width_us)/2;
    s->pinPWM->period_us(s->period_us);
    s->pinPWM->pulsewidth_us(s->currentPulseWidth); //90° - entre 1ms y 2ms (0.001s y 0.002s)
}



void setPos(_Servo *s, uint8_t pos){
    /* Calculo la brecha entre pwm90 y el valor teorico de 127 para aplicarlo proporcionalmente
     * Geometricamente se comporta como dos rectas que se cortan en el punto correspondiente a los 90° que es el punto de calibración
     * pero que siguen pasando por (0, 0) y (256, 180)
     */
    volatile uint8_t raw; // Angulo ajustado por la calibración
    if(pos<90)
        raw = (pos*s->pwm90)/90;
    else
        raw = s->pwm90+(((pos-90)*(255-s->pwm90))/90);
    setRaw(s, raw);
    s->currentPos = pos;
    /* Con esto puedo  llamar a esta funcion para mover el servo a cualquier posición (sin aplicar velocidad) y
     * evito que una posterior llamada a inPos() intente mover el servo a una finalPos antigua
     */
    if(!s->moving)
        s->finalPos = s->currentPos;
}


void setPos(_Servo *s, uint8_t pos, uint16_t speed){
    s->startMov = us_ticker_read();
    s->finalPos = pos;
    s->speed = speed;
    s->initPos = s->currentPos;
    s->moving = true;
}

bool inPos(_Servo *s){
    if(s->currentPos == s->finalPos){
        s->moving = false;  //termine el movimiento
        return true;
    }
    uint32_t curTime = us_ticker_read() - s->startMov;
    uint8_t newPos;
    // newPos depende del hacia que lado debe moverse
    if(s->initPos < s->finalPos) //debo sumar grados 
        newPos = s->initPos + ((curTime * (6*s->speed))/1000000); // speed en rpm = 6°/s
    else
        newPos = s->initPos - ((curTime * (6*s->speed))/1000000); // speed en rpm = 6°/s
    
    if(newPos != s->currentPos) // solo ejecuto un cambio en pwm si la posición debe cambiar para este momento
        setPos(s, newPos); // Posicion en la que debo estar en el momento actual
    return false;
}

void setRaw(_Servo *s, uint8_t raw){
    s->currentPulseWidth = ((raw*(s->max_pulse_width_us - s->min_pulse_width_us))/255)+s->min_pulse_width_us;
    s->pinPWM->pulsewidth_us(s->currentPulseWidth);
}

void calibrate(_Servo *s, uint8_t p){
    s->pwm90 = p;
    setPos(s, s->currentPos);
}

void setPeriod_us(_Servo *s, uint16_t us){
    s->period_us = us;
    setPos(s, s->currentPos);
}

void setMinPulseWidth_us(_Servo *s, uint16_t us){
    s->min_pulse_width_us = us;
    setPos(s, s->currentPos);
}

void setMaxPulseWidth_us(_Servo *s, uint16_t us){
    s->max_pulse_width_us = us;
    setPos(s, s->currentPos);
}