/*
 * Sensor.cpp - libreria para el manejo del sensor ultrasonico hcsr04 y optico tcrt5000
 * @Author Patricio Silva
 * @Date 14/10/2019
 */

#include "Sensor.h"

// Optico
void setup(_Sensor *s, AnalogIn *pin){
    s->pinA = pin;
}

// De horquilla
void setup(_Sensor *s, DigitalIn *pin, uint8_t c, uint8_t p){
    s->pinEcho = pin;
    s->slots = c;
    s->perim = p;
    s->rpm = 0;
    s->ctime = 0;
    s->dtime = 0;
}

// Ultrasonico
void setup(_Sensor *s, DigitalOut *pint, DigitalIn *pine){
    s->pinTrigger = pint;
    s->pinEcho = pine;
    s->pinTrigger->write(0);
    s->ctime = 0;
    s->dtime = 0;
}

////////////  Sensor ultrasonico
bool init(_Sensor *s){
    if(s->pinEcho->read())
        return false;
    s->pinTrigger->write(1);
    s->stage = 1;
    s->ctime = us_ticker_read();
    return true;
}


void reset(_Sensor *s){
    s->stage = 0;
    s->ctime = 0;
}



uint16_t isReady(_Sensor *s){
    switch(s->stage){
        case 0:
            return 0;
        case 1:
            if(us_ticker_read() > (s->ctime + 0x0A)){
                s->pinTrigger->write(0);
                s->ctime = us_ticker_read();
                s->stage = 2;
            }
            break;
        case 2:
            if(s->pinEcho->read()){
                s->ctime = us_ticker_read();
                s->dtime = us_ticker_read();
                s->stage = 3;
            }
            break;
        case 3:
            if(!s->pinEcho->read() && us_ticker_read() > (s->dtime + 20)){ // tiene que haber sido HIGH al menos 20us
                // tiempo es desde el inicio de nivel alto hasta el ultimo ingreso
                // no el momento actual. Voy a estar midiendo menor distancia de la real
                s->stage = 0;
                return ((s->ctime-s->dtime)*10)/58;
            }
            s->ctime = us_ticker_read();  // marca de tiempo
            break;
        default:
            s->stage = 0;
    }
    return 0;
}



uint16_t ping(_Sensor *s, uint16_t max){
    s->dtime = us_ticker_read();  // momento de inicio
    s->ctime = us_ticker_read();
    // Disparo
    s->pinTrigger->write(1);
    wait_us(10);
    s->pinTrigger->write(0);
    
    // Espero que empiece el echo o pase mas tiempo del maximo
    while(!s->pinEcho->read() && s->ctime < (s->dtime + (max*6) + 20))  // que no pase mas del tiempo maximo
        s->ctime = us_ticker_read();  // guardo el momento en que llego al nivel HIGH
        
    // Paso mas del tiempo maximo, no hubo echo
    if(us_ticker_read() > (s->dtime + (max*6) + 20))
        return 0;
    
    // Espero que temine el hecho o pase el tiempo
    while(s->pinEcho->read() || us_ticker_read() < (s->ctime + 20)){ // HIGH por al menos 20us y que no pase mas del tiempo maximo
        // Paso mas del tiempo maximo y echo no paso nunca a 0
        if(us_ticker_read() > (s->ctime + (max*6)))
            return 0;
    }
    
    s->dtime = us_ticker_read() - s->ctime;
    return ((s->dtime*10)/58);
}

////////////  Sensor optico
uint16_t getValue(_Sensor *s){
    return s->pinA->read_u16();
}


////////////  Sensor optico de horquilla
uint32_t getMs(_Sensor *s){
    return s->ctime-s->dtime;
}

uint16_t getRpm(_Sensor *s){
    if(s->slot_count > 8){ // reinicio la cuenta cada 6 slots, conservando la velocidad teorica de 1 pulsos
        s->rpm = (((60000000/(((s->dtime - s->ctime) * s->slots)/s->slot_count)) * 2) + s->rpm)/3;
        s->ctime = us_ticker_read();
        s->slot_count = 0;
    }else if((us_ticker_read() - s->dtime) > 50000){ // Si hay mas de 50ms desde el ultimo registro de pulso fuerzo un reinicio a 0
        s->ctime = us_ticker_read();
        s->slot_count = 0;
        s->rpm = 0;
    }
    return s->rpm;
}


void distanceStart(_Sensor *s){
    s->dist_count = 0;
}

uint16_t getDistance(_Sensor *s){
    return (s->slots*s->perim)/s->dist_count;
}

void notify(_Sensor *s){
    if(us_ticker_read() > (s->dtime + 5)){ // al menos 5us para validar un pulso
        s->dtime = us_ticker_read();
        s->slot_count++;
        s->dist_count++;
    }
}


