/*
 * PSerial.h - libreria de manejo del protocolo de comnicación por puerto serie
 * @Author Patricio Silva
 * @Date 13/10/2019
 */
 #ifndef PSERIAL_H
 #define PSERIAL_H


#define RXBUFFERSIZE 256
#define TXBUFFERSIZE 256
#define RXCMDTIME 50000
#define TXCMDTIME 50000


#include "mbed.h"

////////////////////////////////////////////////////////////////////////
// Tipos de dato
////////////////////////////////////////////////////////////////////////

/*
 * Indices de lectura y escritura del buffer circular
 * checksum y marcas de tiempo para el envio por puerto serie
 * empty indica si la cola esta vacia o llena cuando indexR=indexW
 */
typedef struct{
  uint8_t indexR;
  uint8_t indexW;
  uint8_t buf[TXBUFFERSIZE];
  uint8_t chksum;
  bool empty;
  unsigned long cmdTime;
  uint8_t chksumHeader;
} _txbuffer;

/*
 * Indices de lectura y escritura del buffer circular
 * marcas de tiempo, indice de comando y tamaño del comando
 */
typedef struct{
  uint8_t indexR;
  uint8_t indexW;
  uint8_t buf[RXBUFFERSIZE];
  unsigned long cmdTime;
  uint8_t cmdIndex;
  uint8_t stage;
  bool cmd;
  uint8_t chksum;
  uint16_t payloadSize;
} _rxbuffer;


typedef struct{
    _rxbuffer RX;
    _txbuffer TX;
    Serial *port;
}_Pserial;


////////////////////////////////////////////////////////////////////////
// Prototipos
////////////////////////////////////////////////////////////////////////

/*
 * Inicializa el puerto serie
 */
void serialSetup(_Pserial *, Serial *);


/* Escribe en el puerto serie los datos del buffer
 * en cualquier escribo solo de a 1 byte por llamada
 */
void serialSubmit(_Pserial *);


/* Carga al buffer lo recibido por puerto serie
 */
void serialReceive(_Pserial *);


/*  Busca si hay un frame en el buffer que esté correcto y verifica el checksum
 *  Si hay un header mueve el indice RX.cmdIndex a la posicion donde comienza el comando
 *  y setea a true el flag RX.cmd. Mueve RX.indexR a esa posición para liberar el espacio
 *  ocupado por el header. No procesa un nuevo comando hasta que no se ha procesado el anterior
 *  notificandolo con clearCmd()
 */
void serialGetFrame(_Pserial *);


/*
 * Escribe en el buffer de salida un frame con un tamaño de comando+payload+checksum = s
 * retorna true cuando termina de escribir, falso si no hay suficiente espacio (y restablece los indices)
 */
bool serialEnqueueHeader(_Pserial *, uint16_t);


/* Escribe el byte en el buffer de salida, retorna falso si no hay lugar suficiente para hacerlo
 */
bool serialEnqueueData(_Pserial *, uint8_t);


/* Encola el checksum
 */
bool serialEnqueueChksum(_Pserial *);


/* Borra el comando actual para que se pueda recibir el siguiente
 */
void serialCmdClean(_Pserial *);

//devuelve lo que recibe
void serialGoBack(_Pserial *);


#endif  // PSERIAL_H