#include "pca9548.h"
#include "sfh7779.h"

#define SENSORS_PER_STRIP   8

typedef struct {
    bool                    present;
    sfh7779_measurements_t  sample;
} sfh7779_data_t;

typedef struct {
    sfh7779_data_t  sensor[SENSORS_PER_STRIP];
} att160826_data_t;

class ATT160826
{
public:
    /**
    * Constructor
    *
    * @param i2c I2C class servicing the strip
    * @param addr_3bit address of the strip (A0-A2 of I2C multiplexer)
    *          Valid values are 0-7
    * @param reset optional DigitalOut class that resets the strip
    *          Not needed if hardware reset is not used or if reset 
    *          is not managed by the class.
    */
    ATT160826(I2C * i2c, uint8_t addr_3bit, DigitalOut * reset = NULL) {
        _pca9548 = new PCA9548(i2c, addr_3bit, reset);
        for (int x = 0; x < SENSORS_PER_STRIP; x++) {
            _sfh7779[x] = new SFH7779(i2c);
        }
    }

    /**
    * Destructor
    */
    ~ATT160826() {
        if (_pca9548) {
            delete _pca9548;
        }
        for (int x = 0; x < SENSORS_PER_STRIP; x++) {
            if (_sfh7779[x]) {
                delete _sfh7779[x];
            }
        }
    }

    /**
    * Sample all sensors on the strip (proximity, visual ambient light, and infrared ambient light).
    * Sensors are all activated, sampled, and then placed in standby mode to conserve power consumption.
    *
    * @param att160826_data pointer to a buffer which will be filled with a sensor data
    *
    * @returns true if successfull
    */
    bool scan(att160826_data_t * att160826_data) {
        
        // Clear all previous measurements before starting the scan
        memset(att160826_data, 0, sizeof(att160826_data_t));
        
        if (_pca9548) {
            // Enable all sensors
            for (int x = 0; x < SENSORS_PER_STRIP; x++) {
                int channel_idx = x;
                bool ok = _pca9548->select_channel(channel_idx)
                          && _sfh7779[x]                            
                          && _sfh7779[x]->enable();
                att160826_data->sensor[x].present = ok;
            }
            
            // Collect data from all sensors (throw away 1st sample and keep 2nd)
            for (int y = 0; y < 1; y++) {
                for (int x = 0; x < SENSORS_PER_STRIP; x++) {
                    int channel_idx = x;
                    if (att160826_data->sensor[x].present &&
                        _pca9548->select_channel(channel_idx)) {
                        _sfh7779[x]->read(&(att160826_data->sensor[x].sample), 500);
                    }
                }
            }
            /*
            // Disable all sensors
            for (int x = 0; x < SENSORS_PER_STRIP; x++) {
                int channel_idx = x;
                bool ok = _pca9548->select_channel(channel_idx)
                          && _sfh7779[x]->disable();
            }
            */
            // Disconnect all sensors from the I2C bus
            _pca9548->reset();
        }
        
        return true;
    }

protected:
    PCA9548 *       _pca9548;
    SFH7779 *       _sfh7779[SENSORS_PER_STRIP];
};
