// NXP PCA9548: 8-channel I2C switch with reset

#define PCA9548_BASE_ADDR_7BIT  0x70

#define MAX_CHANNELS    8

#define ASSERT_RESET    0
#define NEGATE_RESET    1

class PCA9548
{
public:
    /**
    * Constructor
    *
    * @param i2c I2C class servicing the multiplexer
    * @param addr_3bit address of the multiplexer (A0-A2 pin strapping)
    *          Valid values are 0-7
    * @param reset optional DigitalOut class that resets the multiplexer (disables all I2C channels)
    *          Not needed if hardware reset is not used or if reset 
    *          is not managed by the class.
    */
    PCA9548(I2C * i2c, uint8_t addr_3bit, DigitalOut * reset = NULL) : _i2c(i2c), _reset(reset) {
        _addr_8bit = ((addr_3bit & 0x7) + PCA9548_BASE_ADDR_7BIT) << 1;
    }

    /**
    * Reset the multiplexer.  All devices connected to the downstream side of
    * the multiplexer are removed from the I2C bus.  If hardware reset is being
    * used, reset is done via asserting the reset pin.  Otherwise, reset is
    * accomplished by deselecting all channels through soft configuration.
    *
    * @returns true if successful
    */
    bool reset(void) {
        if (_reset) {
            _reset->write(ASSERT_RESET);
            return true;
        } else {
            const char channel = 0;
            return _i2c->write(_addr_8bit, &channel, 1) == 0;
        }
    }

    /**
    * Enable access to one of the eight devices on the downstream side of
    * the multiplexer.
    *
    * @param channel channel to activate.  Valid values are 0-7.
    *
    * @returns true if successful
    */
    bool select_channel(const uint8_t channel) {
        if (channel < MAX_CHANNELS) {
            char channel_mask = 1 << channel;
            if (_reset) {
                _reset->write(NEGATE_RESET);
            }
            return _i2c->write(_addr_8bit, (const char *)&channel_mask, 1) == 0;
        } else {
            return false;
        }
    }

protected:
    int             _addr_8bit;
    I2C *           _i2c;
    DigitalOut *    _reset;
};
