#include "mbed.h"
#include "rtos.h"
#include "EthernetInterface.h"
#include "ExperimentServer.h"
#include "QEI.h"
#include "MotorShield.h" 
#include "HardwareSetup.h"

#define NUM_INPUTS 3
#define NUM_OUTPUTS 5

//Measured values
float velocity = 0;
float current = 0;
float theta = 0;

//Set values
float current_d = 0;
float kp = 0;
float ki = 0;

//Controller values
float volt = 0;
float duty = 0;

Serial pc(USBTX, USBRX);    // USB Serial Terminal
ExperimentServer server;    // Object that lets us communicate with MATLAB
Timer t;                    // Timer to measure elapsed time of experiment
Ticker ControlLoop;       //Ticker object created

QEI encoderA(PE_9,PE_11, NC, 1200, QEI::X4_ENCODING);  // MOTOR A ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderB(PA_5, PB_3, NC, 1200, QEI::X4_ENCODING);  // MOTOR B ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderC(PC_6, PC_7, NC, 1200, QEI::X4_ENCODING);  // MOTOR C ENCODER (no index, 1200 counts/rev, Quadrature encoding)
QEI encoderD(PD_12, PD_13, NC, 1200, QEI::X4_ENCODING);// MOTOR D ENCODER (no index, 1200 counts/rev, Quadrature encoding)

MotorShield motorShield(24000); //initialize the motor shield with a period of 12000 clock ticks or ~10kHZ

void current_control()   //Current control function
{
    float error = 0;
    theta = encoderA.getPulses()*(6.2831/1200.0);
    velocity = encoderA.getVelocity()*(6.2831/1200.0);
    current = -(motorShield.readCurrentA()*(52.8/65536.0)-26.4); //read current for motor A in amps. Note: this is a slightly different current sensor so its a different conversion than last lab.            
    error = current_d - current;
    
    volt = 0; //CHANGE THIS TO SET CURRENT CONTROLLER VALUES
    
    duty  = volt/12.0;
    if (duty > 1){
        duty = 1;
    }
    if (duty <-1){
        duty = -1;   
    }
    if (duty >= 0){
        motorShield.motorAWrite(duty, 0); 
    }
    else if (duty < 0){
        motorShield.motorAWrite(abs(duty), 1);
    }
}


int main (void)
{
    // Link the terminal with our server and start it up
    server.attachTerminal(pc);
    server.init();
    
    // Continually get input from MATLAB and run experiments
    float input_params[NUM_INPUTS];
    pc.printf("%f",input_params[0]);
    
    while(1) {
        if (server.getParams(input_params,NUM_INPUTS)) {
            kp = input_params[0];
            ki = input_params[1];
            current_d = input_params[2];

            ControlLoop.attach(&current_control,0.0001); //Run current controller at 10khz
            
            // Setup experiment
            t.reset();
            t.start();
            encoderA.reset();
            encoderB.reset();
            encoderC.reset();
            encoderD.reset();

            motorShield.motorAWrite(0, 0); //turn motor A off
            
            //use the motor shield as follows:
            //motorShield.motorAWrite(DUTY CYCLE, DIRECTION), DIRECTION = 0 is forward, DIRECTION =1 is backwards.
             
            // Run experiment
            while( t.read() < 5 ) {
                // Perform control loop logic
                //current_d = 0; // Set commanded current from impedance controller here.
                
                // Send data to MATLAB
                float output_data[NUM_OUTPUTS];
                output_data[0] = t.read();
                output_data[1] = theta;
                output_data[2] = velocity;
                output_data[3] = current;
                output_data[4] = volt;

                server.sendData(output_data,NUM_OUTPUTS);              
                wait(.001); //run outer control loop at 1kHz
            }
            // Cleanup after experiment
            ControlLoop.detach();
            server.setExperimentComplete();
            motorShield.motorAWrite(0, 0); //turn motor A off
        } // end if
    } // end while
    
} // end main

