#ifndef SIMPLEGUI_WIDGET_H
#define SIMPLEGUI_WIDGET_H

class Widget;

#include "GraphicsContext.h"
#include "EventListener.h"
#include "Rectangle.h"
#include "Point.h"

/**
* A basic widget draws itself in a rectangular area
**/

class Widget : public EventListener
{

public:

    Widget(GraphicsContext* context);
    
    virtual void setParent(Widget* parent);
    virtual Widget *getParent();
    
    /********************************************************
    * Methods relating to containment
    * In the base class these methods do nothing.
    * They are implemented in container classes
    ********************************************************/
    virtual void attach(Widget *widget) {}
    virtual void detach(Widget *widget) {}
    
    /********************************************************
    * Common Widget methods
    ********************************************************/    
    virtual void setLocation(int x, int y);
    virtual void setSize(int width, int height);
    virtual void setWidth(int width);
    virtual void setHeight(int height);
    virtual int x();
    virtual int y();
    virtual int height();
    virtual int width();
    
    virtual void setOffset(int x, int y);
    virtual int offsetX();
    virtual int offsetY();
    
    virtual void setForeground(uint16_t color);
    virtual void setBackground(uint16_t color);
    
    virtual void setPadding(int pixels);
    virtual void setBorder(int width, uint16_t colour);    
   
    /**
    * Cause the widget to redraw itself if is is dirty or damaged
    **/
    virtual void draw();
    virtual void clear();
    
    void show();
    void hide();
    bool isHidden();
    
    /**
    * Implementation of EventListener
    **/
    virtual void handleEvent(Event e);
    virtual void setEventHandler(EventHandler *handler);
    virtual void unsetEventHandler(EventHandler *handler);
    
    bool intersects(Widget *widget);

    /**
    * Mark the widget as needing to be redrawn
    **/
    void dirty();
    bool isDirty();
    /**
    * Mark the widget and all its children as dirty
    **/
    virtual void dirtyAll();
    
    /**
    * Mark the widget as having changed dimensions or location
    **/
    void damage();
    bool isDamaged();

    void adjust();

    /**
    * Convenience method
    **/
    GraphicsDisplay *display();
    virtual void _reenumerateEvents();
    virtual void _reenumerateHandledEvents();
    virtual uint16_t _getHandledEvents();
    
protected:

    virtual void _draw();
    virtual void _clear();
    
    virtual void _dirty();
    virtual void _damage();
    virtual void _adjust();
    
    /**
    * Methods to help with event handling
    **/
    bool _isEventTarget(Event e);
    
    uint16_t _handledEvents;
    
    Widget* _parent;
    GraphicsContext* _context;
    uint16_t _fg, _bg;
    
    Point _offset;
    Rectangle _outer;
    Rectangle _inner;
    
    bool _hidden;
    int _padding;
    int _borderWidth;
    uint16_t _borderColour;

    bool _adjusting;

    bool _dirtied;
    bool _dirtying;

    bool _damaged;
    bool _damaging;
    
    LinkedList<EventHandler> _handlers;
};

#endif