/*
Bryce Williams
09/14/2014

General/Basic Motor Driver Class providing Motor Control using a COTS Motor Driver

    Class based off of Christopher Hasler's Motordriver library found at
    https://developer.mbed.org/cookbook/Motor 
    
    
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

#ifndef MOTORDRIVER_H
#define MOTORDRIVER_H

#include "mbed.h"

typedef enum {ERROR=0, DRIVING_CW=1, DRIVING_CCW=2, BRAKING=3, COASTING=4}Code_t;
typedef struct{
    Code_t code;     
    float  value;
}State_t;

class MotorDriver{
    public:
        /*
            Constructor of MotorDriver Objects
            @param in_1         Direction Input 1
            @param in_2         Direction Input 2
            @param pwm          PWM speed control input
            @param pwmFreq      PWM frequency, some motors may whine at lower freqs
            @param isBrakable   Boolean value indicating whether or not the 
                                motor driver is brakeable(see your datasheet)
        */
        MotorDriver(DigitalOut in1, DigitalOut in2, PwmOut pwm, float pwmFreq, bool isBrakeable = false);
        
        /*
            Sets speed of motor normalized between -1.0 to 1.0 
            @param speed    Value -1.0 to 1.0 (>0 CW at speed as percentage)
                                              (<0 CCW at speed as percentage)
                                              (=0 speed is zero)
            @return state of the motor
            NOTE: This method will NOT allow user to instantaneously swithch 
            from CW to CCW or vise versa. Doing so will cause the motor to 
            be put in to a BRAKE condition, while the motor_state.code will 
            be updated to ERROR. User should avoid trying to do this, call 
            first setSpeed(0) or brake().  
        */
        State_t setSpeed(float speed);
        
        /*
            Same as setSpeed(float speed), however does not impose the safety disallowing
            instantaneous reversal of motor direction. It is up to the user to ensure they
            do not blow up their motor. 
        */
        State_t forceSetSpeed(float speed);
        
        /*
            Put motor into braked config
            @param intensity How hard to brake (0.0 to 1.0)
            @return state of the motors
        */
        State_t brake(float intensity);
        
        /*
            Put motor into stop/coast config
        */
        State_t coast();
        
        
        /*
            Get state of the motor
            @return state of the motor
        */
        State_t getState();
        
    protected:                        // Protected so objects that inherit can access
        State_t    motor_state;
        DigitalOut _in1;
        DigitalOut _in2;
        PwmOut     _pwm;
        bool       isBrakeable;
        void determineState();        // Determine motor state based on in1, in2, and pwm
};

#endif