/*
 
Copyright (c) 2012-2014 RedBearLab
 
Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
and associated documentation files (the "Software"), to deal in the Software without restriction, 
including without limitation the rights to use, copy, modify, merge, publish, distribute, sublicense, 
and/or sell copies of the Software, and to permit persons to whom the Software is furnished to do so, 
subject to the following conditions:
The above copyright notice and this permission notice shall be included in all copies or substantial portions of the Software.
 
THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, 
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR 
PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE 
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 
*/
 
/*
 *    The application works with the BlueJelly.js
 *
 *    http://jellyware.jp/kurage/
 *    https://github.com/electricbaka/bluejelly
 *
 */
 
//======================================================================
//Grobal
//====================================================================== 
//------------------------------------------------------------
//Include Header Files
//------------------------------------------------------------ 
#include "mbed.h"
#include "ble/BLE.h"
 
 
//------------------------------------------------------------
//Definition
//------------------------------------------------------------ 
#define TXRX_BUF_LEN 20                     //max 20[byte]
#define DEVICE_LOCAL_NAME "BlueJelly"     
#define ADVERTISING_INTERVAL 160            //160 * 0.625[ms] = 100[ms] 
#define TICKER_TIME 1000                    //1000[us] = 1[ms]
#define DIGITAL_OUT_PIN P0_29
#define ANALOG_IN_PIN1 P0_5
 
//HeartRate
int timer_num = 0;
int threshold_v = 650;
int threshold_t = 300;  

 
//------------------------------------------------------------
//Object generation
//------------------------------------------------------------ 
BLE blenano;
DigitalOut      LED_SET(DIGITAL_OUT_PIN);
AnalogIn        ANALOG1(ANALOG_IN_PIN1);
 
 
//------------------------------------------------------------
//Service & Characteristic Setting
//------------------------------------------------------------ 
//Service UUID
static const uint8_t base_uuid[] = { 0x71, 0x3D, 0x00, 0x00, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;
 
//Characteristic UUID
static const uint8_t tx_uuid[]   = { 0x71, 0x3D, 0x00, 0x03, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;
static const uint8_t rx_uuid[]   = { 0x71, 0x3D, 0x00, 0x02, 0x50, 0x3E, 0x4C, 0x75, 0xBA, 0x94, 0x31, 0x48, 0xF1, 0x8D, 0x94, 0x1E } ;
 
//Characteristic Value
uint8_t txPayload[TXRX_BUF_LEN] = {0,};
uint8_t rxPayload[TXRX_BUF_LEN] = {0,};
 
//Characteristic Property Setting etc
GattCharacteristic  txCharacteristic (tx_uuid, txPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic  rxCharacteristic (rx_uuid, rxPayload, 1, TXRX_BUF_LEN, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY| GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ);
GattCharacteristic *myChars[] = {&txCharacteristic, &rxCharacteristic};
 
//Service Setting
GattService         myService(base_uuid, myChars, sizeof(myChars) / sizeof(GattCharacteristic *));
 
 
//======================================================================
//onDisconnection
//======================================================================
void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    blenano.startAdvertising();
}
 
 
//======================================================================
//onDataWritten
//======================================================================
void WrittenHandler(const GattWriteCallbackParams *Handler)
{   
    uint8_t buf[TXRX_BUF_LEN];
    uint16_t bytesRead;
    
    if (Handler->handle == txCharacteristic.getValueAttribute().getHandle()) 
    {
        blenano.readCharacteristicValue(txCharacteristic.getValueAttribute().getHandle(), buf, &bytesRead);
        memset(txPayload, 0, TXRX_BUF_LEN);
        memcpy(txPayload, buf, TXRX_BUF_LEN); 
       
        if(buf[0] == 1)
            LED_SET = 1;
        else
            LED_SET = 0;
    }
}
 
 
//======================================================================
//onTimeout
//======================================================================
void m_status_check_handle(void)
{   
    timer_num ++;
}
 
 
//======================================================================
//convert reverse UUID
//======================================================================
void reverseUUID(const uint8_t* src, uint8_t* dst)
{
    int i;
    
    for(i=0;i<16;i++)
        dst[i] = src[15 - i];
}
 
 
//======================================================================
//main
//======================================================================
int main(void)
{
    uint8_t base_uuid_rev[16];
 
    //Timer Setting [us]
    Ticker ticker;
    ticker.attach_us(m_status_check_handle, TICKER_TIME);
    
    //BLE init
    blenano.init();
    
    //EventListener
    blenano.onDisconnection(disconnectionCallback);
    blenano.onDataWritten(WrittenHandler);  
 
    //------------------------------------------------------------
    //setup advertising 
    //------------------------------------------------------------
    //Classic BT not support
    blenano.accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED);
    
    //Connectable to Central
    blenano.setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    
    //Local Name
    blenano.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME,
                                    (const uint8_t *)DEVICE_LOCAL_NAME, sizeof(DEVICE_LOCAL_NAME) - 1);
    
    //GAP AdvertisingData                                
    reverseUUID(base_uuid, base_uuid_rev);  
    blenano.accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
                                    (uint8_t *)base_uuid_rev, sizeof(base_uuid));
                                    
    //Advertising Interval 
    blenano.setAdvertisingInterval(ADVERTISING_INTERVAL);
 
    //Add Service
    blenano.addService(myService);
    
    //Start Advertising
    blenano.startAdvertising(); 
    
    //------------------------------------------------------------
    //Loop
    //------------------------------------------------------------
    int timer_ms;
    
    while(1)
    {
        blenano.waitForEvent(); 
        
        float s = ANALOG1;
        uint16_t value = s * 1024;
                        
        //---------------------------------------------------------------------------------------------       
        //Detect HeartRate Peak
        //--------------------------------------------------------------------------------------------- 
        timer_ms = TICKER_TIME/1000 * timer_num;
        if(value >= threshold_v && timer_ms > threshold_t)
        {    
            //clear timer_num
            timer_num = 0;                
            
            uint8_t buf[2];
            buf[0] = (timer_ms >> 8);
            buf[1] = timer_ms;
            
            //Send out
            blenano.updateCharacteristicValue(rxCharacteristic.getValueAttribute().getHandle(), buf, 2);    
        }       
    }
}