#include "GameEngine.h"

//Defining LCD
//VCC,SCE,RST,D/C,MOSI,SCLK,LED
N5110 lcd(p14,p8,p9,p10,p11,p13,p21);

//Defining Joystick y  x  
Joystick joystick(p20,p19);

//Creating bus to write to LEDs
BusOut led_output(LED4,LED3,LED2,LED1);

//Bus for RGB LED R , G ,  B
BusOut rgb_led( p24, p23, p22);

//Defining Buttons as Interrupts
InterruptIn button_A(p29);
InterruptIn button_B(p28);
InterruptIn button_C(p27);


//Sets flags for interrupts
volatile int g_button_A_flag = 0;
volatile int g_button_B_flag = 0;
volatile int g_button_C_flag = 0;

void button_A_isr();
void button_B_isr();
void button_B_isr();

// Button A event-triggered interrupt
void button_A_isr()
{
    //set flags in ISR
    g_button_A_flag = 1;
}

// Button B event-triggered interrupt
void button_B_isr()
{
    //set flags in ISR
    g_button_B_flag = 1;
}

// Button C event-triggered interrupt
void button_C_isr()
{
    //set flags in ISR
    g_button_C_flag = 1;
}

// Zombie time-triggered interrupt
void GameEngine::zombie_timer_isr()
{
    zombie_timer_flag = 1;   // set flag in ISR
}


GameEngine::GameEngine() {
    // set-up the zombie ticker so that the ISR it is called every 1 second
    ticker.attach(callback(this, &GameEngine::zombie_timer_isr) , 1);   
}

GameEngine::~GameEngine() {

}

void GameEngine::init() {
    
    //Creating random seed from reading adc
    srand(joystick.read_adc()*joystick.read_adc2()*64000);
    
    // PCB has external pull-down resistors so turn the internal ones off
    // (default for DigitalIn)
    button_A.mode(PullNone);
    button_B.mode(PullNone);
    button_C.mode(PullNone);
    
    //Set  to rising edge
    button_A.rise(&button_A_isr);
    button_B.rise(&button_B_isr);
    button_C.rise(&button_C_isr);
    
    //Initialise everything
    lcd.init();
    lcd.setContrast(0.5);
    menu.init();
    crosshair.init();
    joystick.init();
    HUD.init();
    house.init();
    
    //Setting intial values
    zombie_spawn_timer = 0;
    spawn_zombie_counter = 0;
    zombie_timer_flag = 0;
    spawn_timer_goal = 5; 
    menu_selection_flag = 0;
    tactical_nuke_flag = 0;
    
    //Clearing LEDs and Shift Register
    shift.write(0x00);
    rgb_led.write(0b111);
}

// Run Menu

void GameEngine::run_menu() {
    direction = joystick.get_direction();
    lcd.clear();
    menu.scroller(direction);
    menu.draw(lcd);
    select_option();
    lcd.refresh();
    wait_ms(2000/20);
}

// Play Game

void GameEngine::play_game() {
        
        direction = joystick.get_direction();
        lcd.clear();
        crosshair.travel(direction);
        HUD.draw_bullets(lcd);
        house.draw(lcd);
        draw_zombies();
        HUD.draw(lcd);
        crosshair.draw(lcd);
        check_zombie_house_collision();
        check_house_health();
        tactical_nuke_leds();
        HUD.draw_points(lcd);
        
        if(check_A_pressed()) {
            HUD.bullet_shot();
            rgb_led.write(0b110);
            wait(0.02);
            rgb_led.write(0b111);
            crosshair_aiming_at_zombie();
            }
        
        if(check_B_pressed()) {
            HUD.reload();
            rgb_led.write(0b101);
            wait(0.02);
            rgb_led.write(0b111);
            }
        
        if(check_C_pressed()) {
            if(tactical_nuke_flag == 1) {
                //printf("\nTactical Nuke Deployed!!");
                deploy_tactical_nuke();
                }
            }
        
        if (zombie_timer_flag) {
            spawn_timer++;
            zombie_timer_flag = 0;
            }
            
        
        //Checking to see if spawn timer is reached
        if(spawn_timer >= spawn_timer_goal) {
            spawn_zombies();
            spawn_timer = 0;
            }
            
        //if(crosshair_aiming_at_zombie()) {
            //printf("\nBoolean aiming function working correctly");
        //    }
        
        lcd.refresh();
        wait_ms(2000/30);
    }

//Function for spawning zombies
void GameEngine::spawn_zombies() {
        
        // Creating new Zombie object
        Zombie new_zombie;
        
        //Initialise New Zombie
        new_zombie.init();

        // Stores zombie object in vector 
        zombie_vector.push_back(new_zombie);
    
        spawn_zombie_counter ++;
        
        //Reducing time between spawns
        if(spawn_zombie_counter %10 == 0 && spawn_timer_goal > 2) {
            spawn_timer_goal--;
            }
}

void GameEngine::draw_zombies() {
    //Uses Zombie vector to draw each one 
    for (int i = 0; i < zombie_vector.size(); i++) {
        zombie_vector[i].draw(lcd);
    }
}


// Zombie House Collision 
void GameEngine::check_zombie_house_collision() {
    for (int i = 0; i < zombie_vector.size(); i++) {
        
        Vector2D zombie_pos = zombie_vector[i].get_position();
        
        //printf("%d\n", zombie_x_pos);
        
        if(zombie_pos.x <= 11) {
            HUD.set_house_health(HUD.get_house_health() - 2);
            zombie_vector.erase(zombie_vector.begin() + i);
            }
    }
}

//Check House health
void GameEngine::check_house_health() {
    if(HUD.get_house_health() == 0) {
        menu_selection_flag = 4;
        }
    }

//Crosshair aiming at Zombie
void GameEngine::crosshair_aiming_at_zombie() {
    for (int i = 0; i < zombie_vector.size(); i++) {
        
        Vector2D crosshair_pos = crosshair.get_position();
        Vector2D zombie_pos = zombie_vector[i].get_position();
        
        int crosshair_x_pos = crosshair_pos.x + 2;
        int crosshair_y_pos = crosshair_pos.y + 2;
        
        int zombie_x_pos = zombie_pos.x;
        int zombie_y_pos = zombie_pos.y;
        
        //printf("\n x pos = %d", crosshair_x_pos);
        //printf("\n y pos = %d", crosshair_y_pos);
        
        if(crosshair_x_pos >= (zombie_x_pos + 2)  && crosshair_x_pos <= (zombie_x_pos + 
        5) && crosshair_y_pos >= zombie_y_pos && crosshair_y_pos 
        <= (zombie_y_pos + 5)) {
            //printf("\n Crosshair is aiming at Zombie");
            zombie_is_shot(i);
            }
        }
}

//Erases zombie, creates zombie death image, adds kills and points
void GameEngine::zombie_is_shot(int i) {
                
            if (HUD.get_number_of_bullets() > 0){ 
                //printf("\nZombie has been shot!");
                zombie_vector.erase(zombie_vector.begin() + i);
                draw_zombie_death(lcd, i);
                HUD.set_number_of_kills(HUD.get_number_of_kills() + 1);
                HUD.set_number_of_points(HUD.get_number_of_points() + 1);
                //printf("%d\n",HUD.get_number_of_kills());
            }
    }

//Sprite for zombie death
const int zombie_death_sprite[11][10] = {
    {0,0,0,0,0,0,1,1,0,0},
    {1,0,0,0,0,1,1,1,0,0},
    {0,1,0,0,0,0,1,0,0,0},
    {0,0,1,0,0,0,0,0,1,1},
    {0,0,0,0,0,0,0,1,1,1},
    {0,0,1,0,0,0,0,1,1,1},
    {0,1,0,0,0,0,0,1,1,0},
    {0,0,0,0,1,0,0,0,0,0},
    {0,0,0,0,0,0,0,0,0,0},
    {0,0,0,1,1,0,0,0,0,1},
    {0,0,1,1,0,0,0,0,1,1},

};

//Draws zombie death sprite
void GameEngine::draw_zombie_death(N5110 &lcd, int i) {
    
    Vector2D zombie_pos = zombie_vector[i].get_position();
    
    int zombie_death_x_pos = zombie_pos.x;
    int zombie_death_y_pos = zombie_pos.y;
    
    lcd.drawSprite(zombie_death_x_pos, zombie_death_y_pos, 10, 11, (int*)zombie_death_sprite);
    wait(0.02);

}  

//Controls the 4 leds on the microcontroller board
void GameEngine::tactical_nuke_leds() {
    
    if(HUD.get_number_of_kills() >= 10) {
        led_output = 0b0001;
        }
    if(HUD.get_number_of_kills() >= 20) {
        led_output = 0b0011;
        }
    if(HUD.get_number_of_kills() >= 30) {
        led_output = 0b0111;
        }
    if(HUD.get_number_of_kills() >= 40) {
        led_output = 0b1111;
        tactical_nuke_flag = 1;
        HUD.draw_small_nuke(lcd);
        }
    }

//Runs tactical nuke visuals
void GameEngine::deploy_tactical_nuke() {
    
    //Resetting flag, number of kills & leds
    tactical_nuke_flag = 0;
    HUD.set_number_of_kills(0);
    led_output = 0b0000;
    
    //Visual Displays of the Nuke
    
    HUD.draw_tactical_nuke(lcd);
    lcd.refresh();
    shift.write(0x4F);
    rgb_led.write(0b100);
    wait(0.5);
    
    lcd.clear();
    lcd.refresh();
    rgb_led.write(0b111);
    wait(0.5);
    
    HUD.draw_tactical_nuke(lcd);
    lcd.refresh();
    shift.write(0x5B);
    rgb_led.write(0b100);
    wait(0.5);
    
    lcd.clear();
    lcd.refresh();
    rgb_led.write(0b111);
    wait(0.5);
    
    HUD.draw_tactical_nuke(lcd);
    shift.write(0x06);
    rgb_led.write(0b100);
    lcd.refresh();
    wait(0.5);
    
    lcd.clear();
    lcd.refresh();
    wait(0.5);
    
    shift.write(0x00);
    rgb_led.write(0b100);
    zombie_vector.clear();
    lcd.refresh();
    wait(0.5);
    rgb_led.write(0b111);
    
}

// Check if A pressed
bool GameEngine::check_A_pressed() {
    while(1) {
                //Function to test if interrupt has occurred
                if (g_button_A_flag) {
                        
                //Clearing the flag if interrupt has occurred
                g_button_A_flag = 0;
                
                return true;
            }
        break;
    }
    return false;
}

// Check if B pressed
bool GameEngine::check_B_pressed() {
    while(1) {
                //Function to test if interrupt has occurred
                if (g_button_B_flag) {
                        
                //Clearing the flag if interrupt has occurred
                g_button_B_flag = 0;
                
                return true;
                
                
            }
        break;
    }
    return false;
}

// Check if C pressed
bool GameEngine::check_C_pressed() {
    while(1) {
                //Function to test if interrupt has occurred
                if (g_button_C_flag) {
                        
                //Clearing the flag if interrupt has occurred
                g_button_C_flag = 0;
                return true;
            }
        break;
    }
    return false;
}

//Selects option from menu screen
void GameEngine::select_option() {
    if(menu.get_arrow_y_pos() == 15) {
        if(check_A_pressed()) {
        //printf("\nMenu Select Play Game");
        menu_selection_flag = 1;
            }
        }
    if(menu.get_arrow_y_pos() == 23) {
        if(check_A_pressed()) {
        //printf("\nMenu Select Tutorial");
        menu_selection_flag = 2;
            }
        }
    if(menu.get_arrow_y_pos() == 31) {
        if(check_A_pressed()) {
        //printf("\nMenu Select Settings");
        menu_selection_flag = 3;
            }
        }
}

//Menu selection flag
int GameEngine::get_menu_selection_flag(){
         return menu_selection_flag;
         }

//Runs tutorial from menu
void GameEngine::run_tutorial() {
    lcd.clear();
    menu.draw_tutorial(lcd);
    if(check_A_pressed()) {
        wait(0.5);
        if(menu.get_tutorial_page() <= 10) {
            menu.set_tutorial_page(menu.get_tutorial_page() + 1);
            }
        if(menu.get_tutorial_page() == 11) {
            menu_selection_flag = 0;
            menu.set_tutorial_page(1);
            }
        //lcd.clear();
        //lcd.refresh();
        }
    lcd.refresh();
    wait_ms(2000/20);
}

//Runs settings from menu
void GameEngine::run_settings() {
    lcd.clear();
    menu.draw_settings(lcd);
    if(check_A_pressed()) {
        wait(0.3);
        if(menu.get_tutorial_page() <= 2) {
            menu.set_settings_page(menu.get_settings_page() + 1);
            }
        if(menu.get_settings_page() == 3) {
            menu_selection_flag = 0;
            menu.set_settings_page(1);
            }
        lcd.clear();
        lcd.refresh();
        }
    lcd.refresh();
    wait_ms(2000/20);
}


//Runs Game over page once house health is 0
void GameEngine::run_game_over() {
    lcd.clear();
    lcd.printString("GAME OVER!",10,1);
    
    score = HUD.get_number_of_points();
    
    char buffer[3];  
    sprintf(buffer,"%.3d",score); 
    lcd.printString(buffer,52,3);
    
    lcd.printString("Score:",12,3);
    
    lcd.printString("Menu (A)", 16, 5);
    if(check_A_pressed()) {
        menu_selection_flag = 0;
        init();
        zombie_vector.clear();
        }
    lcd.refresh();
    wait_ms(2000/20);
    }

         