#ifndef ENGINE_H
#define ENGINE_H
#include "N5110.h"
#include "Gamepad.h"
#include "Character.h"
#include "Levels.h"
#include "FXOS8700CQ.h"
#include "Science.h"



/** Engine Class
@brief A class to run a game loop
@brief Interfaces with the N5110 class, Gamepad class, Levels class, FXOS8700CQ class, Levels class, Science class, and Character class.
@author Victoria Filor
@date May 17
*/



/** current coordinate struct */
typedef struct current_coordinate_struct {//https://www.quora.com/How-can-I-return-a-structure-from-a-function-in-c
    int current_x; /**< integer for the current_x coordinate */
    int current_y; /**< integer for the current_y coordinate */
} current_coordinate_struct;



/** new coordinate struct */
typedef struct new_coordinate_struct {//https://www.quora.com/How-can-I-return-a-structure-from-a-function-in-c
    int new_x; /**< integer for the new_x coordinate */
    int new_y; /**< integer for the new_y coordinate */
} new_coordinate_struct;


class Engine
{
public:


    void init();


    /** handles the input
    *
    *Handles the inputs and returns the current coordinates of the character
    *@param new_x - new_x from calculations 
    *@param new_y - new_y from calculations
    *@param Gamepad - to interact with the Gamepad class
    *@param N5110 - to interact with the Nokia N5110 class
    *@param Character - to interact with the Character class
    *@param Levels - to interact with the Levels class
    *@param Science - to interact with the Science class
    *
    *@return current_coordinate_struct - returns a struct containing the current coordinate
    */
    current_coordinate_struct handle_input(int new_x, int new_y, Gamepad &pad, N5110 &lcd,Character &character, Levels &levels, Science &science);//smooth sensor, current = new/fullycorrected



    /** update variables
    *
    * Updates the coordinates based on movements and collisions of the character
    *@param radius - radius of the character 
    *@param current_x - current central x coordinate of the character
    *@param current_y - current central y coordinate of the character
    *@param Device - to read from the accelerometer of the FRDM K64F
    *@param N5110 - to interact with the Nokia N5110 class
    *@param Levels - to interact with the Levels class
    *@param Science - to interact with the Science class
    *@param Gamepad - to interact with the Gamepad class
    *
    *@return new_coordinate_struct - returns a struct containing the new coordinate of where the character shall move to
    */
    new_coordinate_struct update(int radius, int current_x, int current_y, Science &science, FXOS8700CQ &device, N5110 &lcd, Levels &levels, Gamepad &pad);



    /** draw
    *
    *Draws the maze, exit and character as appropriate
    *@param current_x - current centre x coordinate of the character
    *@param current_y - current centre y coordinate of the character
    *@param radius - radius of the character 
    *@param Character - to draw the character 
    *@param N5110 - to interact with the Nokia N5110 class
    *@param Levels - to interact with the Levels class
    *
    *@return current_coordinate_struct - returns a struct containing the current coordinate
    */
    void draw(int new_x, int new_y, int radius, Character &character, Levels &levels, N5110 &lcd); //uses the fully corrected/ latest new_x/y to update
    
    
    
    /** checks if levels need updating
    *
    *Compares the position of the where the character will be in the next frame to the exit, to see if the next level needs to be set
    *@param current_x - next central x coordinate of the character
    *@param current_y - next central y coordinate of the character
    *@param Levels - to interact with the Levels class
    */
    void check_for_level_update( int new_x, int new_y, Levels &levels);

private:

    
    
    volatile int level_index; // index indicating the level

    
};

#endif