#ifndef MENU_H
#define MENU_H

#include <stdlib.h>    // used to generate random number values

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"
#include "MazeEngine.h"
#include "Animations.h"
#include "Maze.h"
#include "Ball.h"

/**
 @brief Menu Class runs the menu which the program uses as a basis.
 @author Max Houghton
 @date April 4 2017
 */

class Menu
{
    
public:
    
    /**
     *  @details - constructor
     */
    Menu();
    
    /**
     *  @details - destructor
     */
    ~Menu();
    
    /** Initialise Menu
     *
     * @details - Creates objects and default parameters for the game
     *
     */
    void init();
    
    /** Main Menu Function
     *
     *  @details - The main function which runs the game and allows player to select all different game parameters.
     *  @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param device - FXOS8700CQ Libary, used to read data from accelerometer
     *  @param animate - Animations Library to provide short animations.
     *
     */
    void main(N5110 &lcd,
              Gamepad &pad,
              FXOS8700CQ &device,
              Animations &animate);
    
    /** Intro Function
     *
     *  @details - The main function which runs the game and allows player to select all different game parameters.
     *  @details - Called from main file when the game is loading.
     *  @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param animate - Animations Library to provide short animations.
     *
     */
    void intro(N5110 &lcd, Gamepad &pad, Animations &animate);
    
private:
    
    /**
     *
     * @details - Game Loop function.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param device - FXOS8700CQ Libary, used to read data from accelerometer
     *  @param animate - Animations Library to provide short animations.
     *  @param difficulty - Integer value to specificy desired difficulty; maze Index is randomly chosen according to difficulty value.
     *  @param tone - Boolean value to select whether sound is used or not.
     *  @param FPS - Specifies the frame rate at which the game is to be played at.
     *
     */
    void playGame(N5110 &lcd,
                  Gamepad &pad,
                  FXOS8700CQ &device,
                  Animations &animate,
                  int mazeIndex,
                  bool tone,
                  float FPS);
    
    /**
     *
     * @details - Prints all chosen game parameters to serial port - used mostly for debugging.
     * @param mazeIndex - specifies which of the 10 mazes is to be drawn.
     * @param x - x coordiante for centre of ball when drawn initially.
     * @param y - y coordiante for centre of ball when drawn initially.
     * @param control - Boolean value to specify desired control method.
     * @param colour - Boolean value to select type of ball; filled or transparent.
     * @param FPS - Specifies the frame rate at which the game is to be played at.
     *
     */
    void printGameParameters(int mazeIndex,
                             int x,
                             int y,
                             int radius,
                             bool control,
                             bool colour,
                             float FPS);
    
    
    /**
     *
     * @details - Game Options Function to allow user to choose game style.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param animate - Animations Library to provide short animations.
     *  @param tone - Boolean value to select whether sound is used or not.
     */
    void options(N5110 &lcd, Gamepad &pad, Animations &animate, bool tone);
    
    
    /**
     *
     * @details - LCD Settings Function to allow user to alter appearance of game on LCD.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param animate - Animations Library to provide short animations.    
     *
     */
    void lcdSettings(N5110 &lcd, Gamepad &pad, Animations &animate);
    
    /**
     *
     * @details - Sound Settings to allow user to turn sound on or off. This changes the boolean value of the '_tone' variable.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param animate - Animations Library to animate slide bar to display brightness level settings.
     *
     */
    void soundSettings(N5110 &lcd, Gamepad &pad, Animations &animate);
    
    /**
     *
     * @details - Control Options Function which allows user select control technique. This chanes the boolean value of the '_control' variable.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param animate -Animations Library to provide animations for either joystick or gamepad tilt selection.
     *
     */
    void controlOptions(N5110 &lcd, Gamepad &pad, Animations &animate);
    
    /**
     *
     * @details - Difficulty Options Function to allow player to select difficulty level. This changes the value of integer variable '_difficulty' to 1, 2, 3 or 4. The maze index for the maze is then selected randomly according to this difficulty value.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *
     */
    void difficultyOptions(N5110 &lcd, Gamepad &pad);
    
    /**
     *
     * @details - Ball colour function to select the style of ball used in the game. This function changes the value of boolean variable '_colour'; true corresponds to a transparent fill and false to solid fill.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param animate - Animations Library to provide animations for either solid or transparent fill selection.
     *
     */
    void ballColourOptions(N5110 &lcd, Gamepad &pad, Animations &animate);
    
    /**
     *s
     * @details - LCD Inverse Colour function which allows user to switch colours on the LCD.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *
     */
    void lcdInverseColour(N5110 &lcd, Gamepad &pad);
    
    /**
     * @details - LCD Background Colour function which changes the PWM value of the LCD backlight LED.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     *  @param animate - Animations Library to provide animation for switch.
     */
    void lcdBackgroundColour(N5110 &lcd, Gamepad &pad, Animations &animate);
    
    /**
     * @details - Animate Joystick function creats a short animation of moving joystick.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param animate - Animations Library to provide animation for moving joystick bitmaps.
     */
    void animateJoystick(N5110 &lcd, Animations &animate);
    
    /**
     * @details - Animate Gamepad function creats a short animation of moving gamepad.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param animate - Animations Library to provide animation for moving gamepad bitmaps.
     */
    void animateGamepad(N5110 &lcd, Animations &animate);
    
    /**
     * @details - Animate Stickman function creates a short animation of jumping man to signify goal reached.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to flash LEDS on and off.
     *  @param animate - Animations Library to provide animation for moving stickman bitmap.
     */
    void animateStickman(N5110 &lcd, Gamepad &pad, Animations &animate);
    
    /**
     * @details - Game Speed function which changes the FPS used in the game loop.
     * @param lcd - N5110 Library, used for drawing strings and printing bitmaps
     *  @param pad - Gamepad Library, used to read inputs from peripheral devides on the handheld gamepad device.
     */
    void speedSettings(N5110 &lcd, Gamepad &pad);
    
    /**
     * @details - Returns a random number between two set values
     * @param difficulty - Specified value dictating the level of maze to be played
     */
    int randomMazeIndexGenerator(int difficulty);
    
    /**
     * @details - Creates _engine object to be used when running the game loop.
     * @param _engine - instance of MazeEngine class.
     * @return _mazeIndex
     */
    MazeEngine _engine;
    
    /**
     * @details - Integer to select the specific maze to be drawn by engine.
     * @param _mazeIndex - variable to be passed down to engine.
     */
    int _mazeIndex;
    
    /**
     * @details - Integer to select the desired difficulty by user.
     * @param _difficulty - variable to be passed down to engine. Value used in randomMazeIndexGenerator() function
     */
    int _difficulty;
    
    /**
     * @details - Integer to select the frame rate at which the game is updated at.
     * @param _FPS - value is used as delay between updates when running game loop
     */
    float _FPS;
    
    /**
     * @details - Boolean value to select the desired control method by user.
     * @param _control - value of true corresponds to joystick input, false corresponds to accelerometer values used as input.
     */
    bool _control;
    
    /**
     * @details - Boolean value to select the desired ball fill type.
     * @param _control - value of true corresponds to transparent fill, false corresponds to solid fill of ball when playing the game.
     */
    bool _colour;
    
    /**
     * @details - Boolean value to signify when the goal has been reached.
     * @param _goal - when this variable becomes true, the particular maze has been completed.
     */
    bool _goal;
    
    /**
     * @details - Boolean value to dictate if the buzzer on the gamepad is to be used in the menu when moving between options or selecting desired parameters.
     * @param _tone - A true value allows for sound to be used in the menu.
     */
    bool _tone;
    
};

#endif /* MENU_H */
