#ifndef BALL_H
#define BALL_H

#include "mbed.h"
#include "N5110.h"
#include "Gamepad.h"

/**
 @brief Ball Class contains necessary parameters to draw the ball of the correct position, size and colour.
 @author Max Houghton
 @date March 19 2017
 */

class Ball
{
    
public:
    
    /**
     *  @details - constructor
     */
    Ball();
    
    /**
     *  @details - destructor
     */
    ~Ball();
    
    /** Initialise Ball
     *
     *  @details - Initiases the ball with the appropriate different parameters chosen by user.
     *  @param - x - specifies the x coordinate for the starting position of the ball.
     *  @param - y - specifies the y coordinate for the starting position of the ball.
     *  @param - radius - specifies size of the ball to be drawn.
     *  @param - colour - specifies if the ball is either transparent or filled.
     */
    void init(float x, float y, int radius, bool colour);
    
    /** Update
     *
     *  @details - Function takes the 2D Vector 'position' and updates the current position of the ball by adding on the values of 'position'.
     *  @param - position - 2D Vector struct passed down by engine containing values for the centre of the ball for the next iteration.
     *
     */
    void update(Vector2D position);
    
    /** Draw
     *
     *  @details - Function draws the ball using the current set values for the centre of the ball and its radius.
     *  @param - lcd - N5110 Library used to draw the circle representing the ball.
     *
     */
    void draw(N5110 &lcd);
    
    /** Get Position
     *
     *  @details - Function returns a 2D Vector struct containing information about the ball's current position. This is used when checking for wall collisions in the Maze Engine class.
     *
     */
    Vector2D getPosition();
    
    /** Get Velocity
     *
     *  @details - Function returns a 2D Vector struct containing information about the ball's next position.
     *  @return position
     *
     */
    Vector2D getVelocity();
    
    /** Set Position
     *
     *  @details - This function dictates the position of the ball for the next update of the game. It is used in the wall collision check and goal check functions.
     *  @param - p - 2D Vector struct containing the location of the ball to be set to.
     *  @return velocity
     *
     */
    void setPosition(Vector2D p);
    
    /** Set Velocity
     *
     *  @details - This decides where the ball is to travel next.
     *  @param - v - 2D Vector struct which has the location of the ball for the next iteration.
     *
     */
    void setVelocity(Vector2D v);
    
private:
    
    /**
     *  @details - Struct containing x and y values for the following position of the ball
     */
    Vector2D velocity;
    
    /** Check For Interference
     *
     *  @details - When using the accelerometer data for position values, small but non-zero values can cause the ball undesirably. This function removes all small values of the data ensuring the ball only moves according to deliberate movements of the accelerometer.
     *  @return velocity
     *
     */
    Vector2D checkForInterference(Vector2D velocity);
    
    /**
     *  @param _radius - Integer to decide size of the ball to be drawn.
     */
    int _radius;
    
    /**
     *  @param _x - Float value specifying x coordinate of centre of ball.
     */
    float _x;
    
    /**
     *  @param _y - Float value specifying y coordinate of centre of ball.
     */
    float _y;
    
    /**
     *  @param _colour - Boolean value specifying fill style of the ball; true corresponds to transparent fill and false to solid fill.
     *  @return _colour
     */
    bool _colour;
    
};

#endif /* BALL_H */
