/** Maze Engine class
 @brief Class runs the game loop, drawing the mazes and checking for wall collisions and goals.
 @author Max Houghton
 @date March 3 2017
 */

#ifndef MAZEENGINE_H
#define MAZEENGINE_H

#include "mbed.h"
#include "math.h"
#include "N5110.h"
#include "Gamepad.h"
#include "FXOS8700CQ.h"
#include "Ball.h"
#include "Maze.h"

class MazeEngine
{
    
public:
    
    /**
     *  @details - constructor
     */
    MazeEngine();
    
    /**
     *  @details - destructor
     */
    ~MazeEngine();
    
    /** Initiase Game
     *
     *   @details - Initiases the game with the appropriate different parameters chosen by user
     *   @param - mazeIndex - selects the maze to be drawn
     *   @param - x - specifies the x coordinate for the starting position of the ball
     *   @param - y - specifies the y coordinate for the starting position of the ball
     *   @param - radius - specifies size of the ball to be drawn
     *   @param - control - selects the ball control method; joystick or accelerometer
     *   @param - colour - specifies if the ball is either transparent or filled
     */
    void init(int mazeIndex,    // sets the difficulty of the maze
              int x,            // x coordinate of ball centre
              int y,            // y coordinate of ball centre
              int radius,       // radius of the ball
              bool control,      // option for controlling ball
              bool colour);     // option for style of ball
    
    /** Read Input
     *
     *  @details - Reads the input from either the joystick or the accelerometer. If (boolean) control value is true, joystick is used, otherwise the accelerometer is used.
     *  @param - pad - Gamepad Library.
     *  @param - device - FXOS8700CQ Library.
     */
    void readInput(Gamepad &pad, FXOS8700CQ &device);
    
    /** Read Joystick Input
     *
     *  @param - pad - Gamepad Library.
     *  2D Vector position (struct) is filled with (x,y) values using get_mapped_coord()
     *  function from Gamepad library.
     *  Values are between -1 and 1.
     *  These are they used as the ball's position in the next update of the loop.
     */
    void readJoystickInput(Gamepad &pad);
    
    /** Read Accelerometer
     *
     *  @details - Struct 'Data' is filled with values using get_values() function from FXOS8700CQ library. These data values are then used to calculate the pitch and roll for the device. Pitch and roll are both floats between -1 and 1. Pitch value equates to motion in the x axis, and roll in the y axis.
     *  @param - device - the FXOS8700CQ Library.
     */
    void readAccelerometer(FXOS8700CQ &device);
    
    /** Update
     *
     *  @details - Updates the game using the position parameters found previously. Initially checks for goal, if so, quit function. If no goal, the position of the ball is updated and the space around the ball is checked for the next loop.
     *  @param - position - 2D Vector struct holding ball position.
     */
    void update(N5110 &lcd);
    
    /** Draw
     *
     *  @details - Draws both the maze and the ball. Maze Array is used when performing wall collision checks. Acquired prior to drawing the ball to avoid confusion between ball edges and walls.
     *  @param - lcd - N5110 Library.
     */
    void draw(N5110 &lcd);
    
    /** Check Wall Collision
     *
     *  @details - Checks the position of the ball using the getPosition() function from the ball class. The distance between the centre and the outermost turned on pixel is calculated. If the value calculated is different than the radius value, there must be a pixel. Therefore stop movement in that direction.
     *  @param - lcd - N5110 Library used to create lcd buffer for maze and ball.
     */
    void checkWallCollision(N5110 &lcd);
    
    /** Check Goal
     *
     *  @details - Position of the ball is acquired using getPosition() function from Ball class. For extreme maze (index 9), goal location is at centre of maze. For all other mazes, goal is achieved when x coordinate of centre of ball exceeds (+ radius) the maximum value of the LCD buffer, 83. (i.e it has gone off the screen)
     *  @param - lcd - N5110 Library.
     */
    bool checkGoal();
    
    /** Get Maze Array
     *
     *  @details - Array used for wall collision checking is aquired. Function scans through each pixel in the LCD and sets value in corresponding array to 1 if the pixel is turned on
     *  @param - lcd - N5110 Library.
     */
    void getMazeArray(N5110 &lcd);
    
    /** Goal
     *
     *  @param - _goal - Boolean variable stating whether the goal has been achieved or not.
     */
    bool _goal;
    
private:
    
    /**
     *  @param - _maze - Create instance of Maze class.
     */
    Maze _maze;
    
    /**
     *  @param - _ball_ - Create instance of Ball class.
     */
    Ball _ball;
    
    /**
     *  @param - position - 2D Vector struct storing the position of the ball.
     */
    Vector2D position;
    
    /**
     *  @param - _mazeIndex - Integer variable to store maze index. This is used to draw the appropriate maze.
     */
    int _mazeIndex;
    
    /**
     *  @param - _x - coordinate for centre of the ball.
     */
    int _x;
    
    /**
     *  @param - _y - coordinate for centre of the ball.
     */
    int _y;
    
    /**
     *  @param - _radius - value of the size of the ball
     */
    int _radius;
    
    /**
     *  @param - _control - Boolean variable to state the control method. If _control is true, the joystick is used for an input, otherwise the accelerometer is used.
     */
    bool _control;
    
    /**
     *  @param - _colour - Boolean variable to state the ball fill style. If _colour is true, the ball has a transparent fill, otherwise the ball has a solid fill.
     */
    bool _colour;
    
    /**
     *  @param - _mazArray[][] - 2D Float Array storing the locations of the turned on pixels. Used for wall collision checks.
     */
    float _mazeArray[84][48];
};

#endif
