/**
@file main.h
@brief Header file containing functions prototypes, defines and global variables.
@brief Revision 6.0.
@author Toby O'Connell
@date May 2015
*/

#include "mbed.h"
#include "N5110.h"
#include "SRF08.h"
#include <string>
#include "PinDetect.h"
#include "USBMIDI.h"
#include <fstream>
#include <sstream>
#include <vector>
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

#ifndef MAIN_H
#define MAIN_H

/**  
@namespace SRF08
@brief SRF08 ultrasonic range finder setup
*/
SRF08 SRF08(p28, p27, 0xE2);
/**  
@namespace MCP4151
@brief SPI setup for MCP4151 digital potentiometer
*/
SPI MCP4151(p5, NC, p7); // SPI pins for MCP4151 (MOSI, MISO, SCLK)
/**  
@namespace CS
@brief GPIO output for MCP4151 chip select
*/
DigitalOut CS(p8);
/**   
@namespace buttonA
@brief GPIO input for button A
*/
/**  
@namespace buttonB
@brief GPIO input for button B
*/
/**  
@namespace buttonC
@brief GPIO input for button C
*/
/**  
@namespace buttonD
@brief GPIO input for button D
*/
PinDetect buttonA(p15), buttonB(p16), buttonC(p17), buttonD(p18); //Use PinDetect for debounced interupts
/**  
@namespace lcd
@brief N5110 LCD screen setup
*/
N5110 lcd(p14,p9,p10,p11,p13,p21);
/**  
@namespace midi
@brief USBMIDI object for MIDI send/receive
*/
USBMIDI midi;
/**  
@namespace startRangingTicker
@brief Ticker object to start the SRF08 ranging at regular intervals
*/
Ticker startRangingTicker;
/**  
@namespace getRangeTimeout
@brief Timeout object to read the SRF08 range after a set interval
*/
Timeout getRangeTimeout;
/**  
@namespace local
@brief Setup the local file system
*/
LocalFileSystem local("local"); 

unsigned char imgbuffer[3*84*48]; /*!< Stores the pixel data from the buttons.bmp image */
FILE *img; /*!< Stores the the buttons.bmp image */
vector<float> smoothingValues; /*!< Stores previous distance fractions */
int oldValue; /*!< Parameter value stored when entering a parameter editing screen */
int measuredDistance; /*!< Distance read from the SRF08 in CM */
float distanceFraction; /*!< Fraction representing distance between minDistance and maxDistance */
int receivedMIDIValue; /*!< Control change value read in through USB MIDI port */
string valueLists[1][3] = {{"Sensor","MIDI","None"}}; /*!< Stores list type value contents */
int maxNumberValues[6] = {10000, 3, 6000, 8, 15, 127}; /*!< Defines limits for number type values */
int numberValues[11]; /*!< Stores number type values */
int increment; /*!< Set to '1' or '-1' depending the on corresponding button press.*/
int valLen; /*!< The length of the string containing the parameter value */
int maxValLen; /*!< The length of the string containing the value limit */
int state = 0; /*!< Finite state machine state */
int menuArrowPosition = 0; /*!< Selection arrow pointer for menu system */
int digitArrowPosition = 0; /*!< Selection arrow pointer for digit increment/decrement */
int aButtonFlag = 0; /*!< Flag when button A is pressed */
int bButtonFlag = 0; /*!< Flag when button B is pressed */
int cButtonFlag = 0; /*!< Flag when button C is pressed */
int dButtonFlag = 0; /*!< Flag when button D is pressed */
int rangingStartFlag = 0; /*!< Flag when startRangingTicker triggers */
int rangeGetFlag = 0; /*!< Flag when getRangeTimeout triggers */
int receivedMIDIFlag = 0; /*!< Flag when MIDI message is received for the correct control change and channel */
int measuredDistanceFlag = 0; /*!< Flag when a distance measurement is made */

/**
Reads numatic varaiables from the VALS.csv file
*/
void readNumberValuesFromFile();
/**
Writes numatic varaiables to the VALS.csv file
*/
void writeNumberValuesToFile();
/**
Raises startRangingFlag
*/
void startRangingFlag();
/**
Raises getRangeFlags
*/
void getRangeFlag();
/**
Starts the SRF08 ranging and attaches a ticker to read the range
*/
void rangingStart();
/**
Converts the measured distance to a float between 0 and 1
*/
void rangeGet();
/**
* Returns the relevant parameter to its previous value if it is greater than its limit
* 
* Ensures the minimum distance is less than the maximum distance
* 
* Ensures the minimum resistance is less than the maximum resistance
*/
void checkValidity();
/**
Sets the range of the SRF08 and resises the smoothing values vector
*/
void immediateUpdates();
/**
Converts a float between 0 and 1 to an integer between 0 and 255 before writing it to the MCP4151 digital potentiometer
@param value - float between 0 and 1
*/
void writeMCP4151(float value);
/**
* Converts a float between 0 and 1 to an integer between 0 and 127 
* 
* Sends control change MIDI message 
@param value - float between 0 and 1
*/
void writeMIDI(float value);
/**
* Catches MIDI input messages and stores them if they match the input control change and input channel
* 
* Raises receivedMIDIFlag 
@param value - float between 0 and 1
*/
void receivedMIDI(MIDIMessage);
/**
* Stores previous distance fractions for average smoothing
@param value - float between 0 and 1
@returns smoothed distance fraction
*/
float smooth(float value);
/**
Raises aButtonFlag
*/

void aPressed();
/**
Raises bButtonFlag
*/
void bPressed();
/**
Raises cButtonFlag
*/
void cPressed();
/**
Raises dButtonFlag
*/
void dPressed();
/**
* Saves the initial parameter to oldValue when entering a parameter editing page
* 
* Calls checkValidity(), writeNumberValuesToFile() and immediateUpdates() before leaving a parameter editing page
* 
* Changes the FSM state
* 
* Resets menu and digit pointer positions
* 
* calls showScreen()
*/
void screenSelect();
/**
* Revets relevant parameter to oldValue if leaving a parameter editing page
*
* Changes the FSM state
*
* Resets menu and digit pointer positions
*
* calls showScreen()
*/
void screenBack();
/**
moves the menu pointer position up
calls showScreen()
*/
void menuUp();
/**
moves the menu pointer position down
calls showScreen()
*/
void menuDown();
/**
moves the digit pointer position right
calls showScreen()
*/
void incrementDigitArrowPos();
/**
moves the digit pointer position left
calls showScreen()
*/
void decrementDigitArrowPos();
/**
Makes the increment variable equal 1
calls amendValue()
*/
void incrementValue();
/**
Makes the increment variable equal -1
calls amendValue()
*/
void decrementValue();
/**
Increments or decrements the relevant parameter based on its how it is used and the increment value.
*/
void amendValue();
/**
Increments or decrements the digit pointed to by the digit pointer 
@param value - Integer to be incremented / decremented
@returns incremented / decremented integer
*/
int amendNumberValue(int value);
/**
Displays numeric values and fills unused digit spaces with zeros
@param value - Integer to be displayed
@param maxValue - Maximum value the integer can be
*/
void displayNumber(int value, int maxValue);
/**
Displays values from a list
@param list[] - The array storing the list items
@param listIndex - The index of the list item to be displayed
*/
void displayList(string list[], int listIndex);
/**
Displays boolean type values
@param toggle - The boolean type value to be displayed
*/
void displayBool(bool toggle);
/**
Displays the incoming SRF08 distance, the MIDI input value, the MIDI output value and the resistance output value
*/
void displayMeasured();
/**
* Clears the display
*
* Prints the title
*
* Prints menu items if the state is a menu screen
*
* Calls displayNumber(), displayList() or displayBool() if the state is a parameter screen
*
* Calls displayMeasured() if the state is the measurements display screen
*
* Calls drawBMP()
*/
void showScreen(); 
/**
Draws the image buffer to the screen
*/
void drawBMP(); 

enum {INT, INDEX, BOOL};

/**
Defines the FSM state parameters
*/
struct State {
    char *title; /*!< Menu title*/
    char *list[4]; /*!< Menu options list*/
    int listLength; /*!< Menu options list length*/
    int previousState; /*!< Previous FSM state*/
    int nextState[4]; /*!< Next FSM states*/
    int screenType; /*!< Type of screen*/

    int associatedValueType; /*!< Usage type of the parameter associated with parameter editing page*/
    int associatedValueIndex; /*!< Index of parameter associated with certain parameter editing pages*/
    int associatedMaxValueIndex; /*!< Index of parameter limit associated with certain parameter editing pages*/
    int associatedValueList; /*!< List associated with certain parameter editing pages*/
};
typedef const struct State Menu;
 
enum {menuType, pageType, displayType};
enum {resistanceLimit, resistanceSourceIndexLimit, distanceLimit, distanceSmoothingLimit, MIDIChannelLimit, MIDIControllerLimit};
enum {resistanceSource};
enum {sensorSource, MIDISource, noSource};
enum {
    mainMenu,
    resistanceMenu,
    distanceMenu,
    MIDIMenu,
    displayPage,
    maxResistancePage,
    minResistancePage,
    resistanceSourcePage,
    maxDistancePage,
    minDistancePage,
    distanceSmoothingPage,
    MIDIInputMenu,
    MIDIOutputMenu,
    MIDIInputChannelPage,
    MIDIInputControllerPage,
    MIDIOutputChannelPage,
    MIDIOutputControllerPage,
    MIDIOutputOnPage
};
enum {
    maxResistance, 
    minResistance,
    resistanceSourceIndex,
    maxDistance,
    minDistance,
    distanceSmoothing,
    MIDIInputChannel,
    MIDIInputController,
    MIDIOutputChannel,
    MIDIOutputController,
    MIDIOutputOn
};
    

Menu menu[23] = {
    {"MAIN MENU",{"Resistance", "Distance", "MIDI", "Display"},4,mainMenu,{resistanceMenu,distanceMenu,MIDIMenu,displayPage},menuType,NULL,NULL,NULL,NULL}, 
 
    {"RESISTANCE",{"Maximum", "Minimum", "Source", ""},3,mainMenu,{maxResistancePage,minResistancePage,resistanceSourcePage,NULL},menuType,NULL,NULL,NULL,NULL},
    {"DISTANCE",{"Maximum", "Minimum", "Smoothing", ""},3,mainMenu,{maxDistancePage,minDistancePage,distanceSmoothingPage,NULL},menuType,NULL,NULL,NULL,NULL},
    {"MIDI",{"Input", "Output", "", ""},2,mainMenu,{MIDIInputMenu,MIDIOutputMenu,0,0},menuType,NULL,NULL,NULL,NULL},
    {"DISPLAY",{"", "", "", ""},NULL,mainMenu,{mainMenu,NULL,NULL,NULL},displayType,NULL,NULL,NULL,NULL},
 
    {"MAXIMUM (OHM)",{"", "", "", ""},NULL,resistanceMenu,{resistanceMenu,NULL,NULL,NULL},pageType,INT,maxResistance,resistanceLimit,NULL}, //Maximum Resistance
    {"MINIMUM (OHM)",{"", "", "", ""},NULL,resistanceMenu,{resistanceMenu,NULL,NULL,NULL},pageType,INT,minResistance,resistanceLimit,NULL}, //Minimum Resistance
    {"SOURCE",{"", "", "", ""},NULL,resistanceMenu,{resistanceMenu,NULL,NULL,NULL},pageType,INDEX,resistanceSourceIndex,resistanceSourceIndexLimit,NULL}, //Resistance input source
    
    {"MAXIMUM (mm)",{"", "", "", ""},NULL,distanceMenu,{distanceMenu,NULL,NULL,NULL},pageType,INT,maxDistance,distanceLimit,NULL},
    {"MINIMUM (mm)",{"", "", "", ""},NULL,distanceMenu,{distanceMenu,NULL,NULL,NULL},pageType,INT,minDistance,distanceLimit,NULL}, 
    {"SMOOTHING",{"", "", "", ""},NULL,distanceMenu,{distanceMenu,NULL,NULL,NULL},pageType,INT,distanceSmoothing,distanceSmoothingLimit,NULL}, 
 
    {"INPUT",{"Channel", "Controller", "", ""},2,MIDIMenu,{MIDIInputChannelPage,MIDIInputControllerPage,NULL,NULL},menuType,NULL,NULL,NULL,NULL}, //12
    {"OUTPUT",{"Channel", "Controller", "On/Off", ""},3,MIDIMenu,{MIDIOutputChannelPage,MIDIOutputControllerPage,MIDIOutputOnPage,NULL},menuType,NULL,NULL,NULL,NULL}, //12
    
    {"CHANNEL",{"", "", "", ""},NULL,MIDIInputMenu,{MIDIInputMenu,NULL,NULL,NULL},pageType,INT,MIDIInputChannel,MIDIChannelLimit,NULL}, 
    {"CONTROLLER",{"", "", "", ""},NULL,MIDIInputMenu,{MIDIInputMenu,NULL,NULL,NULL},pageType,INT,MIDIInputController,MIDIControllerLimit,NULL}, 
    
    {"CHANNEL",{"", "", "", ""},NULL,MIDIOutputMenu,{MIDIOutputMenu,NULL,NULL,NULL},pageType,INT,MIDIOutputChannel,MIDIChannelLimit,NULL},
    {"CONTROLLER",{"", "", "", ""},NULL,MIDIOutputMenu,{MIDIOutputMenu,NULL,NULL,NULL},pageType,INT,MIDIOutputController,MIDIControllerLimit,NULL}, 
    {"ON/OFF",{"", "", "", ""},NULL,MIDIOutputMenu,{MIDIOutputMenu,NULL,NULL,NULL},pageType,BOOL,MIDIOutputOn,NULL,NULL}, 
}; /*!< Finite state machine */

#endif