/**
* @file main.h
* @brief Distance Sensor Project
* @brief Header file containing functions, definitions and global variables.
* @brief Version: 1.0b
* @author Sam Russell
* @date March 2015
*/

/*********************************/
// *  Libaries & Definitions   * //
/********************************/

#include "mbed.h"

#include "SRF02.h"
// @brief Power Control Library that simplifies interacting with the SRF02 distance sensor.
// @author Craig.A.Evans


#include "N5110.h"
// @brief N5110 Display Library that simplifies and initalizes interactions with Nokia N5110 display.
// @author Craig.A.Evans

#include "PowerControl/PowerControl.h"
// @brief Power Control Library that allows for the mbed to be put into different power saving modes.
// @author Michael.Wei


#include "PowerControl/EthernetPowerControl.h"
// @brief Ethernet Power Control Library that allows for the mbed ethernet to be powered down.
// @author Michael.Wei

/**
* @namespace Local
* @brief Creates a local file system on the mbed flash memory.
*/
LocalFileSystem local("local");

/**
* @namespace Timer
* @brief Configures the ticker and defines the ticker as timer.
*/
Ticker timer;

/**
* @namespace PC
* @brief Defines the serial which is used to transmit/recieve data to/from the PC via USB.
*/
Serial pc(USBTX,USBRX);

/********************/
// *  Inputs    * //
/********************/

SRF02 SRF02(p28,p27);
/**
* @namespace SRF02
* @brief Data (SDA) and clock (SCL) for the sensor.
*/

AnalogIn Switch(p16);
/**
* @namespace Switch
* @brief On/Off (BOOLEAN) states, OFF: Does not log data to file ON: Dos Log Data to file.
*/

InterruptIn Button(p17);
/**
* @namespace Button
* @brief Three states, Toggles between Normal, Quiet and Power Saving modes.
*/

/********************/
// *  Outputs    * //
/********************/

AnalogOut logLED(p18);
/**
* @namespace logLED
* @brief Indicates if the data recorded is being logged to file or not.
*/

//          vcc,sce,rst,dc,mosi,clk,led
N5110 display(p7,p8,p9,p10,p11,p13,p26);
/**
* @namespace Display
* @brief Defines the Nokia N5110 as the output display and all the corresponding pins.
*/

PwmOut buzzer(p21); 
/**
* @namespace Buzzer
* @brief defines the buzzer pulse width modulation output.
*/

PwmOut backlight(p26);
/**
* @namespace Baclight
* @brief defines the baclight of the display pulse width modulation output.
*/

BusOut leds (LED1,LED2,LED3,LED4);
/**
* @namespace LEDs
* @brief Defines the set of on board LEDs.
*/

/***********************************/
// *  Flags/Variables/Buffers   * //
/***********************************/

int timerflag = 0; /*!<BOOLEAN: initialise timer flag to equal 0, when this flag expires run void timerExpired(). . */
int distance = 0; /*!<Initialise distance integer value to 0. */
int buttonFlag = 0; /*!<BOOLEAN: initialise buttonFlag to be 0, when this flag is toggled run void buttonPressed(). */
int setTimeFlag = 0; /*!<BOOLEAN: initialise setTimeFlag to equal 0, when this flag is toggled run void setTime().  */
int mode = 1; /*!<Initialise mode to equal 0 (normal mode), this value will define which mode the device is running.. */
char rxString[16]; /*!< Buffer to store the recieved string. */
char buffer[30]; /*!< Buffer to store received UNIX time string */
char buffer2[50]; /*!< Buffer to store the converted (float to string) distance value so it can be printed to the display. */


/********************/
// *  Funcitons    * //
/********************/

/**
* @brief Creates the void which takes the two types of data into the function.
* @author Craig.A.Evans
*/
void writeDataToFile(char* data, float Distancedata); 

/**
* @brief Function that generates and clears the welcome screen, using the N5110 library designed by Craig.A.Evans
*/
void welcomeScreen(); 

/**
* @brief Function which creates and updates the visual and audible indication of the distance.
* @param int buzz
* @param int mode
* @param float distance
*/
void barChart();

/**
* @brief DEBUG CODE: Function which when an error is found, all the on board LED's willl flash.
*/
void error(int code);

/**
* @brief Function that sets the time when the setTimeFlag is high.
*/
void setTime();

/**
* @brief ISR (Interupt Service Routine): Interupts when a time is set over the serial and prepares to set that time by setting the setTimeFlag high.
*/
void serialISR();

/**
* @Brief When the timer expires this funciton will set the timerFlag high indicating to the main function that a specified amount of time has passed (1 second).
*/
void timerExpired();

/**
* @brief Function that gets 10 distance values from the sensor and averages them to get an accurate distance reading ready to use, using the SRF02 library designed by Craig.A.Evans
* @param float d1
* @param float d2
* @param float d3
* @param float d4
* @param float d5
* @param float d6
* @param float d7
* @param float d8
* @param float d9
* @param float d10
* @return float distance
*/
float avgDist();

/**
* @brief ISR (Interupt Service Routine): Whenever the button is pressed this funciton is executed which toggles the mode integer value, allowing the user to cycle through modes.
*/
void buttonPressed();
