/**
@file main.cpp
@brief Radar/Distance Measurement program implementation
@author Karina Sodipo
@date   May 2015
*/

#include "mbed.h"
#include "N5110.h"
#include "SRF02.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

/**
@see - mbed CookBook (Power Management)
*/

/// function to power down the USB interface
#define USR_POWERDOWN    (0x104)
int semihost_powerdown() {
    uint32_t arg;
    return __semihost(USR_POWERDOWN, &arg);
}


/**  
@namespace lcd
@brief LCD screen display with VCC,SCE,RST,D/C,MOSI,SCLK,LED pins
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26); // 

/**  
@namespace sensor
@brief SRF02 ultrasonic distance sensor
*/
SRF02 sensor(p28,p27);

/**  
@namespace leds
@brief the on-board mbed LEDs in a BusOut class
*/
BusOut leds(LED4,LED3,LED2,LED1);

/**  
@namespace led
@brief LED proximity indicator
*/
PwmOut led(p24);

/**
@namespace screenInvert
@brief button to switch from black-on-white to white-on-black
*/
InterruptIn screenInvert(p16);

/**  
@namespace unitSW
@brief button to switch from cm to inches
*/
InterruptIn unitSW(p17);

/**  
@namespace radarDist
@brief slide switch toggling radar and distance displays
*/
InterruptIn radarDist(p18);

/**  
@namespace POT
@brief potentiometer to change backlight brightness on LCD screen
*/
AnalogIn POT(p20);

/**  
@namespace buzzer
@brief Piezo buzzer for audio alerts (distance below 50cm / 20inch threshold AND countdown to sensor update)
*/
PwmOut buzzer(p21);



/// function to display the checkerboard pattern
void checkerboard();

/// function for LED and Piezo alerts in CM
void proximityCm();

/// function for LED and Piezo alerts in INCHES
void proximityIn();

/// function to define states of screenInvert button
void initDisplay();

/// function to define states of unitsSW button
void initMeasureIn();

/// function for potentiometer to change screen brightness
void brightness();

/// function to display readings/radar in black-on-white mode
void normal();

/// function to display readings/radar in white-on-black mode
void invert();

/// function to measure distance in cm
void centimetre();

/// function to measure distance in inches
void inches();

/// function for radar display
void radar();



/// buffer to store data sent to the screen
char buffer[14];

/// Change units to measure distance in, modified in units() ISR
int unit = 0; /*!< 'unit' toggled in units() ISR */

/// ISR for toggling between CM and INCH readings
void units() {
    
    unit++;
    if (unit > 1) {  // check whether unit modes stays within bounds
        unit = 0;
    }
    
}



/// pixel display style, modified in pixels() ISR
int pixel = 0; /*!< 'pixel' set in pixels() ISR */

/// ISR for toggling between black-on-white and white-on-black display
void pixels() {
    
    pixel++;
    if (pixel > 1) {  // check whether display modes stays within bounds
        pixel = 0;
    }
}



int main() {
       
       /// Power down Ethernet interface to save ~ 175mW
       PHY_PowerDown();
       
       /// Power down magic USB interface chip to save ~ 150mW
       int result = semihost_powerdown();
       
       /// Disable clock on unused peripherals
       Peripheral_PowerDown(0x2000); /// CAN controller 1
       Peripheral_PowerDown(0x4000); /// CAN controller 2
       
       /// initialize the screen
       lcd.init();
       
       /// call pixels() on rising edge when button pressed
       screenInvert.mode(PullUp);
       screenInvert.rise(&pixels);
       
       /// call units() on rising edge when button pressed
       unitSW.mode(PullUp);
       unitSW.rise(&units);
       
       /*
       /// switch to radar display
       radarDist.rise(&radar);
       /// switch to distance reader in cm
       radarDist.fall(&centimetre);
       */
       
       /// Device welcome messages - print strings of characters at defined coordinates
       lcd.printString("Radar/Distance",0,1);
       lcd.printString("Measuring",15,3);
       lcd.printString("tool",30,5);
       wait(2); /// hold message for 2 secs
       lcd.refresh(); /// clear screen
       
       lcd.printString("A project",15,1);
       lcd.printString("by",35,3);
       lcd.printString("Karina Sodipo",3,5);
       wait(2);
       lcd.refresh();
       
       lcd.printString("University",10,1);
       lcd.printString("of Leeds",15,3);
       lcd.printString("ELEC2645",15,5);
       wait(2);
       lcd.refresh();
       
       /// display a checkerboard pattern to illustrate transition from start-up to main device functionality
       checkerboard();
       wait(2);
       lcd.clear(); /// clear the buffer
       lcd.refresh();
       
       while(1) { /// loop infinitely
            
            brightness();
            
            initMeasureIn();
            
            initDisplay();
            
    }

}



/**
@see - University of Leeds ELEC1620 Digital Electronics & Microcontrollers lecture slides
*/
/// function to display the checkerboard pattern
void checkerboard() {
    for (int i = 0; i < 84; i+=2) {
        for (int j = 0; j < 48; j+=2) {
            lcd.setPixel(i,j);
        }
    }
    lcd.refresh();
}



/// function to indicate object proximity on LED and buzzer in CM
/// Buzzer counts down to sensor update
/// LED and buzzer indicate when proximity is < 50 cm
void proximityCm() {
    
    int distance = sensor.getDistanceCm();
    
    /// beep at max volume (dutycycle = 50%) for 3 sec
    if (distance < 50) {
        led = 1;
        buzzer = 0.5;
        wait(0.5);
        led = 0;
        buzzer = 0;
        wait(0.25);
        led = 1;
        buzzer = 0.5;
        wait(0.5);
        led = 0;
        buzzer = 0;
        wait(0.25);
        led = 1;
        buzzer = 0.5;
        wait(0.5);
        led = 0;
        buzzer = 0;
        wait(0.25);
        led = 1;
        buzzer = 0.5;
        wait(0.5);
        led = 0;
        buzzer = 0;
        wait(0.25);
        lcd.refresh();
        }
    /// quieter buzzer countdown to next update
    else {
        buzzer = 1;
        wait(0.5);
        buzzer = 0;
        wait(0.5);
        buzzer = 1;
        wait(0.5);
        buzzer = 0;
        wait(0.5);
        buzzer = 1;
        wait(0.5);
        buzzer = 0;
        wait(0.5);
        lcd.refresh();
    }

}



/// function to indicate object proximity on LED and buzzer in INCHES
/// Buzzer counts down to sensor update
/// LED and buzzer indicate when proximity is < 20 inches (~ 50 cm)
void proximityIn() {
    
    int distance = sensor.getDistanceInch();
    
    /// beep at max volume (dutycycle = 50%) for 3 sec
    if (distance < 20) {
        led = 1;
        buzzer = 0.5;
        wait(0.5);
        led = 0;
        buzzer = 0;
        wait(0.25);
        led = 1;
        buzzer = 0.5;
        wait(0.5);
        led = 0;
        buzzer = 0;
        wait(0.25);
        led = 1;
        buzzer = 0.5;
        wait(0.5);
        led = 0;
        buzzer = 0;
        wait(0.25);
        led = 1;
        buzzer = 0.5;
        wait(0.5);
        led = 0;
        buzzer = 0;
        wait(0.25);
        lcd.refresh();
        }
    /// quieter buzzer countdown to next update
    else {
        buzzer = 1;
        wait(0.5);
        buzzer = 1;
        wait(0.5);
        buzzer = 1;
        wait(0.5);
        buzzer = 1;
        wait(0.5);
        buzzer = 1;
        wait(0.5);
        buzzer = 1;
        wait(0.5);
        /// clear screen and buffer
        lcd.clear();
    }

}



/// function to change units of readings: 0 - centimetres, 1 - inches
void initMeasureIn() {
    
    if (unit == 0) {
        /// switch LED4 on to indicate unit
        leds = 8;
        centimetre();
    }
    
    if (unit == 1) {
        /// switch LED2 on to indicate unit
        leds = 2;
        inches();
    }
    
}



/// function to change display modes: 0 - normal display, 1 - inverted display
void initDisplay() {
    
    if (pixel == 0) {
        normal();
    }
    
    if (pixel == 1) {
        invert();
    }
    
}



/// function to control screen backlight brightness
void brightness() {
    lcd.setBrightness(POT);
}



/// function to set pixel display as black-on-white
void normal() {
    lcd.normalMode();
}



/// function to set pixel display as white-on-black
void invert() {
    lcd.inverseMode();    
}



/// function to read distances in centimetres
void centimetre() {
        
        int distance = sensor.getDistanceCm();
        /// Print the formatted output from the sensor
        int length = sprintf(buffer,"D = %d cm",distance);
        
        /// Only allow data with length < or = 24 to be displayed
        if (length<= 14)
            lcd.printString(buffer,0,3);
            proximityCm();
            /// short delay before next measurement
            wait(0.5);
        
}



/// function to read distances in inches
void inches() {
    
    int distance = sensor.getDistanceInch();
    /// Print the formatted output from the sensor
    int length = sprintf(buffer,"D = %d inches",distance);
            
    if (length<= 14)
        lcd.printString(buffer,0,3);
        proximityIn();
        /// short delay before next measurement
        wait(0.5);

}



/// function for radar display - not called because distance reading feature takes precedence in while loop
/// Attempt at using a switch statement unsuccessful
void radar() {
    
    lcd.clear();
    
    int distance = sensor.getDistanceCm();
    
    /// Assume max detectable distance is 240 cm.
    /// Dividing by 48 pixels across screen gives 5cm for every pixel.
    /// Set each pixel along the centre of the screen (x = 42) to 1
    /// for distance increases by 5 cm.
    /// Set any distance above 230cm to final pixel (41,48).
    if (distance < 5) {
        lcd.setPixel(42,1);
    }
    else if (distance < 10) {
        lcd.setPixel(42,2);
    }
    else if (distance < 15) {
        lcd.setPixel(42,3);
    }
    else if (distance < 20) {
        lcd.setPixel(42,4);
    }
    else if (distance < 25) {
        lcd.setPixel(42,5);
    }
    else if (distance < 30) {
        lcd.setPixel(42,6);
    }
    else if (distance < 35) {
        lcd.setPixel(42,7);
    }
    else if (distance < 40) {
        lcd.setPixel(42,8);
    }
    else if (distance < 45) {
        lcd.setPixel(42,9);
    }
    else if (distance < 50) {
        lcd.setPixel(42,10);
    }
    else if (distance < 55) {
        lcd.setPixel(42,11);
    }
    else if (distance < 60) {
        lcd.setPixel(42,12);
    }
    else if (distance < 65) {
        lcd.setPixel(42,13);
    }
    else if (distance < 70) {
        lcd.setPixel(42,14);
    }
    else if (distance < 75) {
        lcd.setPixel(42,15);
    }
    else if (distance < 80) {
        lcd.setPixel(42,16);
    }
    else if (distance < 85) {
        lcd.setPixel(42,17);
    }
    else if (distance < 90) {
        lcd.setPixel(42,18);
    }
    else if (distance < 95) {
        lcd.setPixel(42,19);
    }
    else if (distance < 100) {
        lcd.setPixel(42,20);
    }
    else if (distance < 105) {
        lcd.setPixel(42,21);
    }
    else if (distance < 110) {
        lcd.setPixel(42,22);
    }
    else if (distance < 115) {
        lcd.setPixel(42,23);
    }
    else if (distance < 120) {
        lcd.setPixel(42,24);
    }
    else if (distance < 125) {
        lcd.setPixel(42,25);
    }
    else if (distance < 130) {
        lcd.setPixel(42,26);
    }
    else if (distance < 135) {
        lcd.setPixel(42,27);
    }
    else if (distance < 140) {
        lcd.setPixel(42,28);
    }
    else if (distance < 145) {
        lcd.setPixel(42,29);
    }
    else if (distance < 150) {
        lcd.setPixel(42,30);
    }
    else if (distance < 155) {
        lcd.setPixel(42,31);
    }
    else if (distance < 160) {
        lcd.setPixel(42,32);
    }
    else if (distance < 165) {
        lcd.setPixel(42,33);
    }
    else if (distance < 170) {
        lcd.setPixel(42,34);
    }
    else if (distance < 175) {
        lcd.setPixel(42,35);
    }
    else if (distance < 180) {
        lcd.setPixel(42,36);
    }
    else if (distance < 185) {
        lcd.setPixel(42,37);
    }
    else if (distance < 190) {
        lcd.setPixel(42,38);
    }
    else if (distance < 195) {
        lcd.setPixel(42,39);
    }
    else if (distance < 200) {
        lcd.setPixel(42,40);
    }
    else if (distance < 205) {
        lcd.setPixel(42,41);
    }
    else if (distance < 210) {
        lcd.setPixel(42,42);
    }
    else if (distance < 215) {
        lcd.setPixel(42,43);
    }
    else if (distance < 220) {
        lcd.setPixel(42,44);
    }
    else if (distance < 225) {
        lcd.setPixel(42,45);
    }
    else if (distance < 230) {
        lcd.setPixel(42,46);
    }
    else {
        lcd.setPixel(42,47);
    }
    wait(3);
    lcd.refresh();
}
