/**
@file main.h
@brief Header File containing function prototypes,defines and global variables.
@author Giorgos Savvides SID:200805533
@date May 2015
*/

#include "N5110.h"
#include "PowerControl.h"
#include "EthernetPowerControl.h"
#include "beep.h"

#define DIRECTION_TOLERANCE 0.05

/**
@namespace Joystick button
@brief Digital Input for button status
*/
DigitalIn button(p17);

/**
@namespace Reset Button
@brief Digital Input for Reset Button Status
*/
InterruptIn reset(p25);

/**
@namespace Reset Button
@brief Interrupt for Start Button Status
*/
InterruptIn start(p24);

/**
@namespace AnalogIn y-position
@brief AnalogIn input to read potentiometer value
*/
AnalogIn yPot(p18);

/**
@namespace AnalogIn x-position
@brief AnalogIn input to read potentiometer value
*/
AnalogIn xPot(p19);

/**
@namespace Buzzer
@brief Buzzer object of Beep.h class to control the buzzer component
*/
Beep buzzer(p21);


/**
@namespace LCD
@brief LCD object of N5110.h class, sets the LCD pin connections
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);


/**
@namespace pollJoystick
@brief timer that reads Joystick value every certain amount of time
*/
Ticker pollJoystick;

/**
@namespace Player Timer
@brief timer that reads Player's movements every certain amount of time
*/
Ticker timer;


/** List of Directions within the game enum class */
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    CENTRE,
    UNKNOWN
};


typedef struct JoyStick Joystick; /*!< A structure that includes joystick properties */
struct JoyStick {
    float x;    /*!< current x value*/
    float x0;   /*!< 'centred' x value */
    float y;    /*!< current y value */
    float y0;   /*!< 'centred' y value */
    int button; /*!< button state (assume pull-down used, so 1 = pressed, 0 = unpressed) */
    DirectionName direction;  /*!< current direction */
};


Joystick joystick; /*!< Create Joystick struct variable*/

int printFlag = 0;
int startButtonFlag=0;
int resetButtonFlag=0;


char livesBuffer[1]; /*!< character Buffer stores number of lives*/
char roundBuffer[1]; /*!< character Buffer stores number of rounds*/
char coinsBuffer[2]; /*!< character Buffer stores number of coins*/
char soundString[3]; /*!< character Buffer stores Sound State String*/
char brightnessBuffer[2]; /*!< character Buffer stores value of Brightness*/

int gamePlays=1; /*!< integer indicating that game is being played*/
int sounds=1; /*!< integer indicating that sound is ON or OFF*/
int coinAppear=1;  /*!< integer indicating that coin has appeared*/
int optionsPointer=1; /*!< integer pointing which submenu has been selectd in Options Menu*/

int p=-20; /*!< enemy 2 y-position*/
int q=-100; /*!< enemy 3 y-position*/
int c=0; /*!< coin y-position*/
int j=0; /*!< enemy 1 y-position*/


int enemy1x=46; /*!< enemy 1 initial x-position*/
int enemy2x=6; /*!< enemy 2 initial x-position*/
int enemy3x=26; /*!< enemy 3 initial x-position*/
int xPos=6; /*!<coin's initial x-position*/

int x=26; /*!<player's initial x-position*/
int v=30; /*!<player's initial y-position*/
int w=8; /*!<player's width */
int h=12; /*!<player's height */

int lives; /*!<number of lives */
int coins; /*!<number of coins */
int round; /*!<number of rounds */

float brightness=0.9; /*!<brightness value (0.1-0.9)*/
int brightnessDisplay=9; /*!<brightness value displayed to the user (1-9)*/

int table[84][48]; /*!<table array that will store cell conditions (either 0,1,2)*/

int a=4; /*!<value of the enemies acceleration, integer being added to enemies y-positions*/

/* FUNCTION DECLERATIONS */

/** Set the Game Lanes and Borders
*   @brief  Draws the lanes of the road, that are fixed.
*   @brief  This function is called when the player moves on the screen and pixels need to be reset 
*/
void gameLanes();


/** Initialise Table Array
*   @brief  It initialises the array used in the game by setting all the array cells value equal to 0 
*/
void initTable();


/** Calibrate Joystick Function
*   @brief  Read default positions of the joystick to calibrate later readings 
*/
void calibrateJoystick();


/** Update Joystick Function 
*   @brief  read current joystick values relative to calibrated values (in range -0.5 to 0.5, 0.0 is centred) 
*/
void updateJoystick();


/** Clear Rectangle Function
*   @brief  This function clears the rectangle
*   @param  x - x-coordinate of top left point
*   @param  v - y-coordinate of top left point
*   @param  w - width of the Rectangle
*   @param  h - height of the Rectangle
*   */
void clearRect(int x,int v,int w,int h); //clear position


/** MovePlayer Function
*   @brief  This function is being called every 0.1 sec, checks the direction of the joystick 
*   and moves the Player by clearing and drawing Rectangle 
*/
void movePlayer();


/** Loose Function
*   @brief  This function is being called when the player loose (i.e touches an enemy) 
*/
void loose();


/** Check Player Function
*   @brief  Function can be used to check if the Player touched an enemy
*   or if he has touched a coin. Depending on the situation calls the 
*   appropriate function and implements some certain tasks
*/
void checkPlayerPos();//Constantly checking player's position


/** Enemy 1 Move Function
*   @brief Enemy Function is responsible to move the Enemy downwards
*   The functions use an integer that defines the y position of the enemy
*   As this integer is increasing, rectangle is being drawn and cleared continuously
*   This integer variable restarts from a random value after reaching a specific point
*/
void enemy1MovesDown();


/** Enemy 2 Move Function
*   @brief Enemy Function is responsible to move the Enemy downwards
*   The functions use an integer that defines the y position of the enemy
*   As this integer is increasing, rectangle is being drawn and cleared continuously
*   This integer variable restarts from a random value after reaching a specific point
*/
void enemy2MovesDown();


/** Enemy 3 Move Function
*   @brief Enemy Function is responsible to move the Enemy downwards
*   The functions use an integer that defines the y position of the enemy
*   As this integer is increasing, rectangle is being drawn and cleared continuously
*   This integer variable restarts from a random value after reaching a specific point
*/
void enemy3MovesDown();


/** Coin Moves Function
*   @brief  Function that causes a coin to move downwards
*   It calls the drawCoin and clearCoin functions so the coin is continuously moving down when it appears
*/
void coinMoves();


/** Draw Coin Function
*   @brief  Function that draws a coin on the screen
*   This is a circle with a 'c' symbol at the centre of the circle
*   It calls the drawCircle function and setPixel function to draw the coin
*   It also sets the array cells value equal to 2, indicating there is a coin
*   @param  c - the y coordinate of the centre of the coin
*/
void drawCoin(int c);


/** Clear Coin Function
*   @brief  Function that clears the coin that was drawn previously
*   This is a circle with a 'c' symbol at the centre of the circle
*   It calls the clearCircle function and clearPixel function to clear the coin
*   It also clears the array cells (set their values to 0), indicating there is nothing in these cells
*   @param  c - the y coordinate of the centre of the coin
*/
void clearCoin(int c);


/** Set Circle Cells Function
*   @brief  Function is responsible to set the values of the Table Array cells within a Circle range equal to 2
*   This is done to identify where the coin is at a certain time
*   @param x0   -the x coordinate of the Circle's centre
*   @param y0   -the y coordinate of the Circle's centre
*   @param radius   -radius of the circle
*/
void setCircleCells(int x0,int y0,int radius); //Set selected Circle cells to 1


/** Clear Circle Cells Function
*   @brief  Function that clears the values of the Table Array cells within a Circle (set them equal to 0)
*   @param x0   -the x coordinate of the Circle's centre
*   @param y0   -the y coordinate of the Circle's centre
*   @param radius   -radius of the circle
*/
void clearCircleCells(int x0,int y0,int radius);//Set selected Circle cells to 0


/** Set Rectangle Cells Function
*   @brief  Function that cells the values of the Table Array cells within a Rectanlge Shape equal to 1
*   @brief  This is done for all enemies moving downwards, thus we can check where the enemies are located
*   @param x    -the x-direction of the Rectangle's top left corner
*   @param v    -the y-direction of the Rectangle's top right corner
*   @param w    -the width of the Rectangle
*   @param h    -the height of the Rectangle
*/
void setRectCells(int x,int v,int w,int h); //set Cells range to 1


/** Clear Cells Function
*   @brief  Clears the pixel and sets the array table cells value equal to 0 , from 0 to x and from 0 to y
*   @param x    -x coordinate of the array to clear
*   @param y    -y coordinate of the array to clear
*/
void clearCells(int x,int y); //set Cells range to 0


/** Start Button Pressed Function
*   @brief  Flips the Start Button Flag indicator showing that button was pressed
*/
void startButtonPressed();//Function that checks for an Interupt at pin 24

/** Start Button Pressed Function
*   @brief  Flips the Start Button Flag indicator showing that button was pressed
*/
void resetButtonPressed();//Function that checks for an Interupt at pin 25

/** Set Line's Cells Function
*   @brief  Sets the cells of a line equal to 2
*   @param  x0 - x-coordinate of first point
*   @param  y0 - y-coordinate of first point
*   @param  x1 - x-coordinate of last point
*   @param  y1 - y-coordinate of last point
*   @param  type - 0 white,1 black,2 dotted
*/
void setLineCells(int x0,int y0,int x1,int y1,int type);


/** Clear Line's Cells Function
*   @brief  Clears the cells of a line(set them equal to 0)
*   @param  x0 - x-coordinate of first point
*   @param  y0 - y-coordinate of first point
*   @param  x1 - x-coordinate of last point
*   @param  y1 - y-coordinate of last point
*   @param  type - 0 white,1 black,2 dotted
*/
void clearLineCells(int x0,int y0,int x1,int y1,int type);


/** Initialisation Game Function
*   @brief  Sets the initial game values
*/
void gameReset();