/**
@file main.cpp
@brief ELEC2645 Embedded Systems Project
@brief Ultrasonic Distance sensor/Car Parker
@brief Designed to help car owners park thier cars more effectively
it basically uses burst of sound waves to measure the distance of objects
from the device
@brief Revision 1.0
@author Adolphus George Emmanuel
@date May 2015
*/

#ifndef MAIN_H
#define MAIN_H

#include "mbed.h"
#include "N5110.h"
#include "SRF02.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

/**
@see URL: http://mbed.org/users/AjK/libraries/PinDetect/lkyxpw  for the PinDetect
*/
#include "PinDetect.h"


#define USR_POWERDOWN  (0x104)

/**
@namespace Nokia N5110 LCD, LEDs, Temperatuure Sensor, Distnce Sensor, Buzzer, Button
@brief GPIOs for LCD, LEDs, Temperature and Distance sensor, Printing of serial bus,Button debounce, LDR, buzzer, etc.
*/
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);
BusOut leds(LED1, LED2, LED3, LED4);  // Creat a bus out for on board LEDs
I2C tmp102(p28,p27);                  // Creat a new I2C instance for temperature sensor and initialise
SRF02 srf(p28, p27);                  // Creat a new I2C intsance for Distance sensor and initialise
Serial serial(USBTX,USBRX);           // for serial output through USB
PinDetect  pin (p15);                 // an interupt button
AnalogIn ain(p16);                    // analogue in for the LDR used to control the sleep mode
AnalogIn ain2(p20);
InterruptIn button(p15);
Timeout timer;
PwmOut buzzer(p21); // buzzer output
//////////////////////////////////////////////
PwmOut led(p26);  // pwm to control LCD backlight
DigitalOut sce(p8);  // for LCD SCe
SPI spi(p11,NC, p13); //create a new SPI intance and initialise
DigitalOut pwr(p7);   // for LCD power control
DigitalOut dc(p10);  // for LCD Dc input
PwmOut led2(p24); // Pwm for LED alert

Ticker timer2; // creat ticker function
Ticker timer3;
//////////////////////////////////////////////////////
Timeout callUpTime;

/**
Creates a flag to be used by the button
@returns an integer
*/
int buttonFlag= 0;
void buttonPressed()
{
    buttonFlag = 1;
}

/**
Defines the frequency of the pwm and the beat for sound and LED flashing for alerts
@param frequncy -  float for frequency
@param beat - to set the tempo
@see https://developer.mbed.org/media/uploads/phill/mbed_course_notes_-_pulse_width_modulation.pdf
*/
float frequency[]= {659,554};  //frequency array for both sound and LED (Taken from the mbed website)
float beat[]= {1,1}; //beat array for both sound and LED
//////////////////////////////////////////////////////
/**
Defines the x,y pixels for the lcd display
*/
int next[84][48];
int nx=84;
int ny =48;
int i=0;
int j=0;

/**
Gets the distance measurement from the sensor
*/
int distance = srf.getDistanceCm();
//////////////////////////////////////////////////////

void error(); // Error laert function for the range finer
void setBrightness(float brightness);  // Set Brightness function
void clearRAM();  // Clear RAM function
void setXYAddress(int x, int y);  // Set XY Address function
void resting();  // Function to get distance and display and to geive alerts and also place the car image according to the distance
void turnOfDisplay();  // To turn of display
void turnOnDisplay();  // To turn on display
void sendCommand(unsigned char command);
void buzzerAlert();  // Buzzer alert function
void initialise();  // Function to initialise LCd and display welcome messag
void initSPI(); // To initialse SPI
void flipped();
void flippedStop();
void setPixel();
void controlBrightness();
/////////////////////////////////////

void initTMP102();  // To initialise Temperature
void stopper(); // To get and dispaly temprature
float vol; // float voltage for the potentiometer
////////////////////////////////

void firstImage(); // To draw the vertical brick wall
////////////////////////
// Debounce for button pressed to call up the time and Brightness with battery life infromation
void p15Pressed(void);
void p15Released(void);
///////////////////////////////////////

void serialISR(); // ISR that is called when serial data is received
void setTime(); // function to set the UNIX time
int setTimeFlag = 0; // flag for ISR
char rxString[16]; // buffer to store received string
char buffer[30];  // creat a charracter
void timerInterrupt();
///////////////////////////////

void setter(int, int, int, int, int, int, int, int, int,int, int, int, int,int, int, int); // Intergers to store the position of the car images
void movingCar(int);
void nextGeneration(); // TO clear and set the pixels according to the next set of conditions
void drawLine();  //For horizontal line
////////////////////////////////////////////////////////////
float brightness; // float for brightness values
void battery();  // to indicate battery life
//#Temperature Declearation
void error();
float getTemperature(); // float to get temperature
void ledAlert(int);

/**
@brief Defines the temprature sensors Registers and Address
@Tag for Temperature sensor
@address for ADD0 connected
*/
#define TMP102_ADD 0x48
#define TMP102_R_ADD 0x91
#define TMP102_W_ADD 0x90

// Register address for temperature sensor
#define TEMP_REG 0x00
#define CONFIG_REG 0x01
#define THIGH_REG 0x02
#define TLOW_REG 0x03
////////////////////////////////////////////////////////////////
/**
@brief to power Down the Ethernet
*/
int semihost_powerdown()
{
    uint32_t arg;
    return __semihost(USR_POWERDOWN, &arg);
}
#endif

int main()
{
    // To intialise lcd and dispaly welcome messages.
    initialise();
    //////////////////////////////////
    // For getting time and date and Temperature
    initTMP102();
    serial.attach(&serialISR); // attach serial ISR
    //set_time(1431109980); // initialise time to what what is inputed
    ///////////////////////////////////////

    // to define button thats calls up the time and date it works with the button using debounce
    pin.mode(PullDown); // To pull down the button to zero
    pin.attach_asserted(&p15Pressed);
    pin.attach_deasserted(&p15Released);
    pin.setSampleFrequency();

    /////////////////////////////////////
    // To power down USB and Ethernet
    int result = semihost_powerdown();
    PHY_PowerDown();

    button.rise(&buttonPressed); // For button

    while(1) {
        // resting();

        nextGeneration();
        float voltage = 5.0 * ain.read(); // to declear and initialise the LDR
        if(voltage <=2.50) { // If there is light
            resting();
            setBrightness( brightness);


        } else if(voltage >= 2.55) {  // If the garrage is dark
            turnOfDisplay();
            led2 = 0;
            buzzer = 0;
            timer2.detach();

        }
        Sleep();
    }
}

/**
@brief This function is used to set the X-Y address in RAM for futire data write to the LCD
*/
void setXYAddress(int x, int y)
{

    if (x > 83);
    x  = 83;
    if (y > 5);
    y = 5;
    if(x < 0);
    x =0;
    if (y < 0);
    y=0;

}

/**
@brief This functions both the setPixel and drawLine function draws the horizontal line which acts as the floor or main road
which the car is moving on
*/
void setPixel()
{
    for (i = 0; i < WIDTH; i++) {
        lcd.setPixel(i, 24);
    }
    lcd.refresh();
}

// Function to draw horizontal line
void drawLine()
{
    for (i = 0; i < WIDTH; i++) {
        lcd.setPixel(i, 25);
    }
    lcd.refresh();
}

/**
@brief This turns of the LCD completely
*/
void turnOfDisplay()
{
    setBrightness(0.0);  // turn backlight off
    clearRAM();   // clear RAM to ensure specified current consumption

}

/**
@brief By turning the potentiometer, this function allows you to control the LCD brightneess
the potentiometer uses the 3.3V output from the mbed as its input and uses an analouge in to feedback to the mbed
but it must be noted that the signal or voltage the mbed recieves is between 0V to 1V it can not read in anything above this from the potenteiometer
*/
void setBrightness(float brightness)
{
    float vol = 3.3*ain2;
    float voltage = 5.0*ain;
    if ((vol >= 0.0) && (vol < 0.3)) {
        brightness = 0.0;
    }
    if ((vol >= 0.3)  && (vol  <0.6)) {
        brightness = 0.1;
    }
    if ((vol >= 0.6) && (vol < 0.9)) {
        brightness = 0.2;
    }
    if ((vol >= 0.9)  && (vol <1.2)) {
        brightness = 0.3;
    }
    if ((vol >= 1.2)  && (vol <1.5)) {
        brightness = 0.4;
    }

    if ((vol >= 1.5) && (vol < 1.8)) {
        brightness = 0.5;
    }
    if ((vol >= 1.8)  && (vol  <2.1)) {
        brightness = 0.6;
    }
    if ((vol >= 2.1) && (vol < 2.4)) {
        brightness = 0.7;
    }
    if ((vol >= 2.4)&& (vol < 2.7)) {
        brightness = 0.8;
    }
    if ((vol >= 2.7)&& (vol < 3)) {
        brightness = 0.9;
    }
    if ((vol >= 3.0)&& (vol <=3.3)) {
        brightness = 1.0;
    }
    if ((voltage >= 2.55)&& (vol == vol)) {
        brightness = 0.0;
    }
    led.write(brightness);

}

/**
@brief This function writes a 0 to the 504 bytes to clear the RAM in other to reduce the current waste
*/
void clearRAM()
{
    int i;
    sce.write(0);
    for(i = 0; i <504; i++) {
        spi.write(0x00);
    }
    sce.write(1);
}

/**
@brief This function "resting" contians codes which gets the distance from the sensor and displays it on the LCD
Also still in this function other functions like the getTempearature are called as they all are display at same time on the screen
And all the alerts both sound and virtual associated with the car parking aspect of the device are implement here

for PWM out code for buzzer sound,
@see https://developer.mbed.org/media/uploads/phill/mbed_course_notes_-_pulse_width_modulation.pdf
*/
void resting()
{
    drawLine();
    firstImage();
    led2 = 0;
    buzzer =0;
    stopper();
/////////////////////////////////////////
//Get Distance from sensor dispaly and depending on the conditions, get the appropriate alert and information
// This is where the values for the x and y cordinates to draw the car is also inputted and also the alert the triggers the buzzer and lights up the LED
    int distance = srf.getDistanceCm(); // get distance in cm and save in distance

    char buffer[14];
    int length = sprintf(buffer, "%2d cm",distance); // To define the length of the charracter which must not be more than 14 charracters

    if (length <= 14) {
        if (distance > 249) {   // Distance above 249cm which is the max range the sensor can detect
            lcd.printString("Out of Range", 4, 1);
        }
        if ((distance >= 232) && (distance <=249)) {
            lcd.printString(buffer, 25,0);
            setter(68, 78, 65, 81, 18, 21, 21, 23, 68, 70,76 ,78, 23 ,25, 23, 25); // These are the values that draws the car
            lcd.printString("Very safe", 15, 1);
        }
        if ((distance >= 215)  && (distance  <= 232)) {
            lcd.printString(buffer, 25,0);
            setter(63, 73, 60, 76, 18, 21, 21, 23, 63, 65, 71 ,73, 23 ,25, 23, 25);
            lcd.printString("Very Safe", 15, 1);
        }
        if((distance >= 198) && (distance <= 215)) {
            lcd.printString(buffer, 25,0);
            setter(58, 68, 55, 71, 18, 21, 21, 23, 58, 60, 66 ,68, 23 ,25, 23, 25);
            lcd.printString("Very Safe", 15, 1);
        }


        if ((distance >= 181)  && (distance <=198)) {
            lcd.printString(buffer, 25,0);
            setter(53, 63, 50, 66, 18, 21, 21, 23, 53, 55, 61 ,63,23 ,25, 23, 25);
            lcd.printString("Very Safe", 15, 1);
        }
        if((distance >= 164)  && (distance <=181)) {

            lcd.printString(buffer, 25,0);
            setter(48, 58, 45, 61, 18, 21, 21, 23, 48, 50, 56 ,58, 23 ,25, 23, 25);
            lcd.printString("Safe Distance", 4, 1);

        }

        if ((distance >= 150) && (distance <= 164)) {
            lcd.printString(buffer, 25,0);
            setter(43, 53, 40, 56, 18, 21, 21, 23, 43, 45, 51 ,53, 23 ,25, 23, 25);
            lcd.printString("Safe Distance", 4, 1);

        }
        if((distance >= 133)  && (distance  <=150)) {
            lcd.printString(buffer, 25,0);
            setter(38, 48, 35, 51, 18, 21, 21, 23, 38, 40, 46 ,48,23 ,25, 23, 25);
            lcd.printString("Safe Distance", 4, 1);

        }
        if((distance >= 116) && (distance <= 133)) {
            lcd.printString(buffer, 25,0);
            setter(33, 43, 30, 46, 18, 21, 21, 23, 33, 35, 41 ,43, 23 ,25, 23, 25);
            lcd.printString("Safe Distance", 4, 1);

        }
        if((distance >= 99)  && (distance <=116)) {
            lcd.printString(buffer, 25,0);
            setter(28, 38, 25, 41, 18, 21, 21, 23, 28, 30, 36 ,38, 23 ,25, 23, 25);
            lcd.printString("Safe Distance", 4, 1);

        }
        if((distance >= 82)  && (distance <=99)) {
            lcd.printString(buffer, 25,0);
            setter(23, 33, 20, 36, 18, 21, 21, 23, 23, 25, 31 ,33, 23 ,25, 23, 25);
            lcd.printString("Getting Close", 4, 1);
            for (int i=0; i<=1; i++) {
                buzzer.period(1/(frequency[i])); // set PWM period
                buzzer=0.9; // set duty cycle
                wait(0.5*beat[i]); // hold for beat period
                led2 = 1;
            }
        }
        if((distance >= 65) && (distance <= 82)) {
            lcd.printString(buffer, 25,0);
            setter(18, 28, 15, 31, 18, 21, 21, 23, 18, 20, 26 ,28,23 ,25, 23, 25);
            lcd.printString("Getting Close", 4, 1);
            for (int i=0; i<=1; i++) {
                buzzer.period(1/(frequency[i])); // set PWM period
                buzzer=0.9; // set duty cycle
                wait(0.5*beat[i]); // hold for beat period
                led2 = 1;
            }
        }
        if((distance >= 48)  && (distance<=65)) {
            lcd.printString(buffer, 25,0);
            setter(13, 23, 10, 26,18, 21, 21, 23, 13, 15, 21 ,23,23 ,25, 23, 25);
            lcd.printString("Too Close", 15, 1);
            for (int i=0; i<=1; i++) {
                buzzer.period(1/(frequency[i])); // set PWM period
                buzzer=0.9; // set duty cycle
                wait(0.1*beat[i]); // hold for beat period
                led2 = 1;
            }
        }
        if ((distance >= 31)  && (distance <=48)) {
            lcd.printString(buffer, 25,0);
            setter(8, 18, 5, 21, 18, 21, 21, 23, 8, 10, 16 ,18,23 ,25, 23, 25);
            lcd.printString("Too Close", 15, 1);
            for (int i=0; i<=1; i++) {
                buzzer.period(1/(frequency[i])); // set PWM period
                buzzer=0.9; // set duty cycle
                wait(0.1*beat[i]); // hold for beat period
                led2 = 1;
            }

        }
        if (distance <=31) {  // though the sensor can detect up to 13cm but it was set to 31cm so as to always give drivers safe distance from other objects
            lcd.printString(buffer, 25,0);
            setter(5, 15, 2, 18, 18, 21, 21, 23, 5, 7, 13 ,15, 23 ,25, 23, 25);
            lcd.printString("Limit Reached", 4, 1);
            for (int i=0; i<=1; i++) {
                buzzer.period(1/(frequency[i])); // set PWM period
                buzzer=0.09; // set duty cycle
                wait(0.01*beat[i]); // hold for beat period
                led2 = 1;

            }
        }
    }
    wait (0.5);  // A little delay though not really needed for the distance but was added to help to be more accurate. Ticker timer would be better off
    // but at the time of this project was not really working one of the future aims is to find a way to make it work

    lcd.refresh();

}
/**
@brief This function is used to draw the vertical brick wall which show the limit the car sensor can car and to alert the driver that he or she is getting close to an object
*/
void firstImage()
{
    for (j = 0; j < 26; j++) {
        lcd.setPixel(0, j);
    }
    for (j = 0; j < 26; j++) {
        lcd.setPixel(1, j);
    }
    lcd.refresh();


}

/**
@brief This is declears variables which are later used to draw the moving car depending on the values of the x and y cordinates inputted
*/
void setter(int xVert, int xHor, int xVert2, int xHor2, int yVert, int yHor, int yVert2, int yHor2, int xVert3, int xHor3, int xVert4, int xHor4, int yVert3, int yHor3, int yVert4, int yHor4)
{
    int i,j;

// Take the given values and draw the car
    for (i = xVert; i < xHor; i++) {
        for (j = yVert; j < yHor; j++) {

            lcd.setPixel(i, j);
        }
    }
    for (i = xVert2; i < xHor2; i++) {
        for (j = yVert2; j < yHor2; j++) {
            lcd.setPixel(i, j);

        }
    }
    for (i = xVert3; i < xHor3; i++) {
        for (j = yVert3; j < yHor3; j++) {

            lcd.setPixel(i, j);
        }
    }
    for (i = xVert4; i < xHor4; i++) {
        for (j = yVert4; j < yHor4; j++) {
            lcd.setPixel(i, j);

        }
        lcd.refresh();
    }



    lcd.refresh();
}



void nextGeneration()  // Function for generting the next set of pixels based on the conditions
{
    int nx =84;
    int ny =48;
    for (int i=0; i<nx; i++) {
        for (int j=0; j<ny; j++) {
            if (next[i][j]==1) {
                lcd.setPixel(i,j);
            } else if(next[i][j]==0) {
                lcd.clearPixel(i,j);
            }
        }
    }
    lcd.refresh();
}

/**
@brief Here the LCD initialisation function is called and the welcome messages and also authors details are
implmemet which runs only once whenever the device is powered on
*/
void initialise()
{
    lcd.init();  // initialise lcd
    initTMP102();
    wait(1.0);   // delay
    lcd.printString("Welcome",20,3); // welcome screen
    wait(2.0);  // dealy
    lcd.clear();
    lcd.printString("Adolphus G. E.",1,1); // welcome screen
    wait(0.5);  // dealy

    lcd.printString("SID: 200183574",1,2); // welcome screen
    wait(0.5);  // dealy
    lcd.printString("ELEC2645",1,4); // welcome screen
    wait(2.0);  // dealy
    lcd.clear(); // clear welcome screen
    setXYAddress(0,0);  // set lcd address
    lcd.refresh();  // refresh

}

/**
@brief Functions to get and display the time
*/
void tempTime()
{

    time_t seconds = time(NULL);
    strftime(buffer,32,"%I:%M %p",localtime(&seconds));
    int length = sprintf(buffer,"%s", buffer);
    if (length <= 14) {
        lcd.printString(buffer,0,0);    // To display time
        timerInterrupt(); // To dispaly Date
        controlBrightness();
        battery();

        if (setTimeFlag) { // if updated time has been sent
            setTimeFlag = 0; // clear flag
            setTime(); // update time
        }

    }

}
/**
@brief Functions to get and display the date
*/
void timerInterrupt()
{
    char buffer[30];  // buffer used to store time string
    time_t seconds = time(NULL); // get current time
// format time into a string (time and date)
    strftime(buffer, 30 , "%D", localtime(&seconds));
    int length = sprintf(buffer,"%s", buffer);
    if (length <= 14) {
        lcd.printString(buffer,0,1);  // dispaly Date
    }
    lcd.refresh();

}

void setTime()
{
// print time for debugging
    serial.printf("set_time - %s",rxString);
// atoi() converts a string to an integer
    int time = atoi(rxString);
// update the time
    set_time(time);
}

void serialISR()
{
// when a serial interrupt occurs, read rx string into buffer
    serial.gets(rxString,16);
// set flag
    setTimeFlag = 1;
}

/**
@brief Displays the temperature and also calls contains the alert messages for the temperature aspect of the device
*/
void stopper()
{
    float temperature = getTemperature();
    char buffer[14];
    int length = sprintf(buffer,"Temp: %2.f C", temperature);
    if (length <= 14) {
        lcd.printString(buffer,13,4);
        if (temperature < 10) {

            lcd.printString("Cold Weather", 8, 5);
        }
        if (temperature >=10 && temperature <=20) {

            lcd.printString("Fairly Warm", 8, 5);
        }
        if (temperature >20) {

            lcd.printString("Warm Weather", 8, 5);
        }
    }
    lcd.refresh();

}
/**
@brief hangs in infinite loop flashing error code when something gors wrong for instance is the distance sensor was not detected
*/
void error(int code)
{
    while(1) {
        leds = 0;
        wait(0.25);
        leds = code;
        wait(0.25);
    }
}

/**
@brief Function to initialise the temperature sensor
*/
void initTMP102()
{
    tmp102.frequency(400000); // set bus speed to 400 kHz
    int ack; // used to store acknowledgement bit
    char data[2]; // array for data
    char reg = CONFIG_REG; // register address
//////// Read current status of configuration register ///////

    ack = tmp102.write(TMP102_W_ADD,&reg,1); // send the slave write address and the configuration register address
    if (ack)
        error(1); // if we don't receive acknowledgement, flash error message
    ack = tmp102.read(TMP102_R_ADD,data,2); // read default 2 bytes from configuration register and store in buffer
    if (ack)
        error(2); // if we don't receive acknowledgement, flash error message
///////// Configure the register //////////
// set conversion rate to 1 Hz
    data[1] |= (1 << 6); // set bit 6
    data[1] &= ~(1 << 7); // clear bit 7

//////// Send the configured register to the slave ////////////

    ack = tmp102.write(TMP102_W_ADD,&reg,1); // send the slave write address and the configuration register address
    if (ack)
        error(3); // if we don't receive acknowledgement, flash error message
    ack = tmp102.write(TMP102_W_ADD,data,2); // send 2 data bytes to the configuration register
    if (ack)
        error(4); // if we don't receive acknowledgement, flash error message

}


/**
@brief Function to the temperature  from the Temearature sensor
*/
float getTemperature()

{
    int ack; // used to store acknowledgement bit
    char data[2]; // array for data
    char reg = TEMP_REG; // temperature register address
    ack = tmp102.write(TMP102_W_ADD,&reg,1); // send temperature register address
    if (ack)
        error(5); // if we don't receive acknowledgement, flash error message
    ack = tmp102.read(TMP102_R_ADD,data,2); // read 2 bytes from temperature register and store in array
    if (ack)
        error(6); // if we don't receive acknowledgement, flash error message
    int temperature = (data[0] << 4) | (data[1] >> 4);
    return temperature*0.0625;
}

/**
@brief To clear the screen and prepare to show the date and time when the button is pressed
*/
void p15Pressed(void)// To clear the screen before calling  up the function
{
    lcd.clear();
    setXYAddress(0,0);
    wait(0.5);
    lcd.refresh();
}

/**
@brief The function is part of the PinDetect library which dose the callback function
*/
void p15Released(void) // To show the content of the function (Date and time, Brightness and battey)
{
    buzzer =0;
    led2 =0;
    tempTime(); // This is the name of the function that contains the informations to be dispalyed

    wait(3.0);
    lcd.clear();
    lcd.refresh();
}

/**
@biref When the buton is pressed, this function enables the user to view the percentage of the brightness
*/
void controlBrightness()
{
    float vol = 3.3*ain2;
    if ((vol >= 0.0) && (vol < 0.3)) {
        lcd.printString("Brightness=0%",0,3);
    }
    if ((vol >= 0.3)  && (vol  <0.6)) {
        lcd.printString("Brightness=10%",0,3);
    }
    if ((vol >= 0.6) && (vol < 0.9)) {
        lcd.printString("Brightness=20%",0,3);
    }
    if ((vol >= 0.9)  && (vol <1.2)) {
        lcd.printString("Brightness=30%",0,3);
    }
    if ((vol >= 1.2)  && (vol <1.5)) {
        lcd.printString("Brightness=40%",0,3);
    }

    if ((vol >= 1.5) && (vol < 1.8)) {
        lcd.printString("Brightness=50%",0,3);
    }
    if ((vol >= 1.8)  && (vol  <2.1)) {
        lcd.printString("Brightness=60%",0,3);
    }
    if ((vol >= 2.1) && (vol < 2.4)) {
        lcd.printString("Brightness=70%",0,3);
    }
    if ((vol >= 2.4)&& (vol < 2.7)) {
        lcd.printString("Brightness=80%",0,3);
    }
    if ((vol >= 2.7)&& (vol < 3)) {
        lcd.printString("Brightness=90%",0,3);
    }
    if ((vol >= 3.0)&& (vol <=3.3)) {
        lcd.printString("Brightness=100%",0,3);
    }
}

/**
@brief Function to draw the battery symbol, though at the time of this project a way of getting the battery life was not included but will be added in the next revision
*/
void battery()
{
    lcd.printString("100%", 23, 5);

    for (j = 37; j < 47; j++) {
        lcd.setPixel(3, j);
    }
    for (j = 37; j <47; j++) {
        lcd.setPixel(20, j);
    }


    for (i = 3; i < 21; i++) {
        lcd.setPixel(i, 37);
    }
    for (i = 3; i <21; i++) {
        lcd.setPixel(i, 46);
    }


    for (j = 40; j < 44; j++) {
        lcd.setPixel(21, j);
    }
    for (j = 40; j <44; j++) {
        lcd.setPixel(22, j);
    }

    lcd.refresh();
}
