/**
*   @file BrickBreaker.h
*
*   @brief Header file containg function prototypes and variables. 
*   @brief The program runs a simple version of brickbreaker on a nokia N5110 display.
*   @brief The game is controlled by a 2-axis joystick and buttons.
*
*   @brief Credits to Craig A. Evans for his implementation of the 2-axis Joystick and the basis for the N5110 library.
*
*   @brief Credits to Andy Kirkham for his PinDetect library
*
*   @brief Credits to Toyomasa Watarai for his PowerControl Library
*
*   @author Calum L. Johnston
*   @date 11.05.2015
*/


#include "mbed.h"
#include "N5110.h"
#include "PinDetect.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

#define USR_POWERDOWN   (0x104)

// change this to alter tolerance of joystick direction
#define DIRECTION_TOLERANCE 0.05

//included for power control
int semihost_powerdown() {
    
    uint32_t arg;
    return __semihost(USR_POWERDOWN, &arg);
}




// connections for joystick

/** Joystick Button
*
*   Button for menu navigation and other controls. PinDetect was used to overcome the inherent debounce of the buttons
*/
PinDetect button(p17);

/** Joystick x-axis
*
*   Horizontal motion of the joystick, read as a potentiometer.
*/
AnalogIn xPot(p15);

/** Joystick y-axis
*
*   Veritcal motion of the joystick, read as a potentiometer.
*/
AnalogIn yPot(p16);

//connections for other controls

/** Play/Pause
*
*   Button to pause and resume the game. PinDetect was used to overcome the inherent debounce of the buttons
*/
PinDetect playPause(p12);

/** Volume
*
*   Potentiometer to control the volume of the buzzer
*/
AnalogIn volPot(p20);

/** Noise Pin  
*
*   Pin to create noise for the randomisation of the levels
*/
AnalogIn noise(p19);


/** Local File System
*
*   Create  Local filesystem to store the highscores
*/

LocalFileSystem local("local");

/** Direction Name
*
*   Enum type to store the names of the joystick directions
*/
enum DirectionName {
    UP,
    DOWN,
    LEFT,
    RIGHT,
    CENTRE,
    UPLEFT,
    UPRIGHT,
    DOWNLEFT,
    DOWNRIGHT,
    UNKNOWN
};

/** Joystick
*
*   Struct for the Joystick
*/
typedef struct JoyStick Joystick;
struct JoyStick {
    float x;    // current x value
    float x0;   // 'centred' x value
    float y;    // current y value
    float y0;   // 'centred' y value
    int button; // button state (assume pull-down used, so 1 = pressed, 0 = unpressed)
    DirectionName direction;  // current direction
};

Joystick joystick;

/** High Score 1
*
*   Variable to store the highscores from the memory while the program is running. 
*   This is compared with the player's score at the end of the game and overwritten if necessary.
*/
int highscore1; 

/** High Score 2
*
*   Variable to store the highscores from the memory while the program is running. 
*   This is compared with the player's score at the end of the game and overwritten if necessary.
*/
int highscore2; 

/** High Score 3
*
*   Variable to store the highscores from the memory while the program is running. 
*   This is compared with the player's score at the end of the game and overwritten if necessary.
*/
int highscore3; 


/** Score Flag
*
*   Set to 1 on initial conditions, set to 0 if a score overwrites a high score.
*   This is to stop the same score from overwriting multiple entries in the high scores.
*/
int scoreFlag = 1;

/** Ball x-Position
*
*   Stores the x-Position of the ball for passing to lcd.drawRect()
*/
int bx = 42;

/** Ball y-Position
*
*   Stores the y-position of the ball for passing to lcd.drawRect()
*/
int by = 30; 

/** Ball Width
*
*   Stores the width of the ball for passing to lcd.drawRect()
*/
int bw = 1; 

/** Ball Height
*
*   Stores the Height of the ball for passing to lcd.drawRect()
*/
int bh = 1; 

/** Ball Fill 
*
*   Stores the Fill State of the ball for passing to lcd.drawRect()
*/
int bf = 1; 

/** Ball Direction
*
*   Direction of the ball, quantised into 16 directions of motion
*/
int d = 0;

/** Paddle x-Position
*
*   Stores the x-Position of the paddle for passing to lcd.drawRect()
*/
int px = 38; 

/** Paddle y-Position
*
*   Stores the y-position of the paddle for passing to lcd.drawRect()
*/
int py = 40; 

/** Paddle Width
*
*   Stores the width of the paddle for passing to lcd.drawRect()
*/
int pw = 8; 

/** Paddle Height
*
*   Stores the Height of the paddle for passing to lcd.drawRect()
*/
int ph = 2; 


/** Ball Surround
*
*   Array to store the active pixels around the ball
*/
int surround[12] = {};

/** Touch Flag
*
*   Set when the ball touches any set pixel on the screen to flag it for setAngle()
*/
int touchFlag = 0;

/** Bricks x-Coordinates
*
*   Array to index the x-positions of the bricks
*/
int bricksx[8] = {3, 12, 21, 30, 39, 48, 57, 66};

/** Bricks y-Coordinates
*
*   Array to index the y-positions of the bricks
*/
int bricksy[4] = {3, 8, 13, 18};

/** State of Bricks
*
*   Array to index the state of the bricks: 1 is set, 0 is cleared
*/
int bricks[4][8]; 

/** Display Menu
*
*   Display the menu - for passing to the main() function
*   One of several variables to control what is displayed on the screen
*/
int displayMenu = 1;

/** Display Help
*
*   Display the Help - for passing to the main() function
*   One of several variables to control what is displayed on the screen
*/
int displayHelp = 0;

/** Display High Scores
*
*   Display the High Scores - for passing to the main() function
*   One of several variables to control what is displayed on the screen
*/
int displayHighScores = 0;

/** Display Game
*
*   Display the Game - for passing to the main() function
*   One of several variables to control what is displayed on the screen
*/
int displayGame = 0;

/** Display Game Over
*
*   Display the Game Over - for passing to the main() function
*   One of several variables to control what is displayed on the screen
*/
int displayGameOver = 0;

/** Menu Select
*
*   Used to control the position of the cursor on the screen
*/
int menuSelect = 1;

/** Help Screen
*
*   Used to select which of the help pages to load
*/
int helpScreen = 0;

/** Help Select
*
*   Used to control the position of the cursor on the help screen
*/
int helpSelect = 35;

/** Level
*
*   Used to set the level - for passing to initBricks()
*/
int level = 0;

/** Start/Stop
*
*   Variable to determine whether the game is paused - 1 is start, 0 is stop.
*/
int startStop = 1;

/** Lives
*
*   Variable to store the remaining number of player lives
*/
int lives = 3;

/** Score
*
*   Variable to store the current player score
*/
int score = 0;

/** Border Flag
*
*   Set when the borders have been drawn to stop multiple attempts
*/
int borderFlag = 0;

/** Brick Clear Flag
*
*   Set to 1 for the bricks to be cleared, set to 0 once cleared to stop blank space from being repeatedly cleared
*/
int clearFlag[4][8];

/** Brick Draw Flag
*
*   Set to 1 for the bricks to be drawn, set to 0 once drawn to stop unnecessary overwriting
*/
int brickDrawFlag[4][8];

/** Poll Joystick
*
*   Timer to regularly update the game joystick values
*/
Ticker pollJoystick;

/** Move Paddle
*
*   Timer to regularly move the paddle
*/
Ticker movePaddle;

/** Select
*
*   Timer to update the cursor throughout the game menus
*/
Ticker select;

/** Read High Scores
*
*   Reads the current values of the high scores from the internal memory
*/
void readHighScores();

/** Write High Scores
*
*   Writes the updated values of the high scores to the internal memory
*/
void writeHighScores();


/** Initialise the Bricks
*
*   Populate the array storing the state of the bricks for drawing
*
*   @param l - The game level to be initialised. Predefined when l<2 and random afterwards.
*/
void initBricks(int l);

/** Do the Bricks
*
*   Does getBrickTouch() on all bricks, draws any that are set to be drawn and clears any that are set to be cleared
*/
void doBricks();

/** Get Brick Touch
*
* Checks to see if any of the pixels surrounding a given brick are set - i.e. if something is touching one
*
* @param x - the horizontal index of the brick check
* @param y - the vertical index of the brick to check
*/
void getBrickTouch(int x, int y);

/** Clear Bricks
*
*   Loops through all bricks and clears any that are set to be cleared
*/
void clearBricks();

/** Calibrate Joystick
*
*   Initialises the built in button in the joystick in PullUp mode. Also reads the initial values of the 
*   joystick for reference.
*/
void calibrateJoystick();

/** Update Joystick
*
*   Checks the x and y values of the potentiometers and assigns the joystick a direction based on that.
*/
void updateJoystick();

/** Menu
*
*   Displays the menu screen and moves the cursor to the current position
*/
void menu();

/** Get Menu Select
*
*   Checks the joystick direction and assigns the cursor a new position based on that
*/
void getMenuSelect();

/** New Screen
*
*   Called on a click of the joystick from the menu, changes the information displayed on the screen.
*   Depends on the position of the cursor
*/
void newScreen();

/** Help
*
*   Displays the various pages of help available, moves the cursor to the current position
*/
void help();

/** High Scores
*
*   Pulls the high score data from the internal memory and displays them on the screen 
*/
void highScores();

/** Game 
*
*   Pulls together various other functions to run the game
*
*   @param g - If true, game runs. If false, game stops. Used to pause the game
*/
void game(int g);

/** Ball
*
*   @brief Move the ball
*
*   2-step movement for the ball - combines moveBall1() and moveBall2() with setAngle() and doBricks() to ensure game updates properly.
*
*   NB: wait() functions were included to slow down the motion of the ball to a reasonable speed. A timer would have been preferable but the game
*   ceased to function if this implementation was used.
*/
void ball();

/** Paddle
*
*   @brief Move the paddle left and right 
*
*   Checks the position of the joystick and updates the postion of the paddle  based on that.
*
*   NB: The entire row is cleared when in theory only the prior position should need to be. In practise this does not work and the row does not
*   properly cleared.
*/
void paddle();

/** Get Number of Bricks
*   
*   @return - Number of remaining bricks
*
*   Finds the number of bricks remaining on the screen
*/
int getNumBricks();

/** New Level
*   
*   @brief Updates the level and redraws the bricks 
*
*   Called when getNumBricks() returns 0
*/
void newLevel();

/** Pause
*
*   Reverses the value of StartStop to Pause/Unpause the game
*/
void pause();

/** Move Ball (1)
*    
*   First iteration of ball motion
*
*   @param d - The direction to move the ball in - quantised into 16 separate directions
*/
void moveBall1(int d);

/** Move Ball (2)
*    
*   Second iteration of ball motion
*
*   @param d - The direction to move the ball in - quantised into 16 separate directions
*/
void moveBall2(int d);

/** Initialise the Borders
*   
*   Draws the boundaries so that the ball cannot leave the screen through the sides or top.
*   Also draws some graphics for containing the scores and remaining lives.
*/
void borderInit();

/** Get Ball Touch Flag
*
*   Check to see if the ball is touching anything
*/
void getTouchFlag();

/** Set Ball Angle
*
*   @return d The direction that the ball is to move in
*
*   Set the new direction of the ball based on what the edges of the ball are touching and the current angle of the ball
*/
int setAngle();

/** Displays Score
*   
*   Displays the current score on the screen. The score is displayed vertical but the score is often multiple
*   digits so modulus was used to split up the digits.
*/
void dispScore();


/** Display Lives
*
*   Displays the current number of remaining lives on the screen
*/
void dispLives();

/** Lost Life
*
*   @return 1 - If a life is lost. I think this was for debugging CHECK THIS LATER
*/
int lifeLost();

/** Game Over
*   
*   Displays the dreaded Game Over screen. If the score is greater than any of the high scores they will be updated accordingly and the player informed.
*/
void gameOver();