/**
@file main.cpp
@brief Main code for Weather Logger
@author Alex Raper
@date May 2015
*/

#include "mbed.h"
#include "N5110.h"
#include "BMP180.h"
#include "PowerControl/PowerControl.h"
#include "PowerControl/EthernetPowerControl.h"

#define USR_POWERDOWN (0x104)

//!    VCC,SCE,RST,D/C,MOSI,SCLK,LED
/** Can also power (VCC) directly from VOUT (3.3 V) -
Can give better performance due to current limitation from GPIO pin
*/

N5110 lcd(p7,p8,p9,p10,p11,p13,p26); //!<pins for nokia screen


/**
@namespace timer
@brief create a ticker object to read the data from the sensor
*/
Ticker timer; 

/**
@namespace timeOut
@brief only loops once to show menu
*/
Timeout timeOut;

/**
@namespace leds
@brief leds used for error function
*/
BusOut leds(LED4,LED3,LED2,LED1);

/**
@namespace bmp180
@brief barometric pressure sensor
*/
BMP180 bmp180(p28,p27);

/**
@namespace serial
@brief communication over the serial port
*/
Serial serial(USBTX,USBRX);

/**
@namespace POT
@brief petentiometer used for menu
*/
AnalogIn POT(p20);


/**
@namespace graphButton
@brief button to select graph
*/
InterruptIn graphButton(p16);

/**
@namespace currentButton
@brief button to select current
*/
InterruptIn currentButton(p15);

/**
@namespace red
@brief led for displaying error
*/
PwmOut red(p24);

/**
@namespace buzzer
@brief buzzer for displaying error
*/
PwmOut buzzer(p21);

/**
@param timerFlag - flag for timer
@param timeOutFlag - flag for timeOut
@param graphFlag - flag for graph button
@param currentFlag - flag for current button
@param semihost_powerdown - used to powerdown USB interface
*/


int timerFlag= 0;
int timeOutFlag = 0;
int graphFlag = 0;
int currentFlag = 0;
int semihost_powerdown();
int main(); //!<main code
char rxString[16]; //!<buffer to store received string
char bufferTime[14];//!<array to store time
char bufferDate[14];//!<array to store date
float temperature; //!<stores temperature value as a float (has to be a float or graph doesn't work)
float pressure;//!<stores pressure value as a float
void displayTemperature();//!<function to display current temperature value
void displayPressure();//!<function to display current pressure value
void receiveData();//!<function to get the data from the BMP180
void displayTemperatureGraph();//!<function to display the graph of temperature
void displayPressureGraph();//!<function to display the graph of pressure
void error();//!<function to display visual error when values go above threshold
void time();//!<function to display time and date
void setTime(); //!<function to set the UNIX time
void interrupt(); //!<for RTC
void menu(); //!<function to display main menu screen

int semihost_powerdown()//!<function to power-down USB interface
{
    uint32_t arg;
    return __semihost(USR_POWERDOWN, &arg);
}

void screen() //!<welcome screen
{
    lcd.printString("Weather Logger",0,2);
    lcd.printString("3000",32,3);
    lcd.printString("Alex Raper",15,5);
    lcd.refresh(); //clear screen
}

void menu() //!<main menu screen
{
    
    lcd.clear();
    if (POT>=(3.0/4.0)) { //!<if the potentiometer is pointing to the left (temperature)
        lcd.printString("Temperature",2,2); //!<data type label
        lcd.printString(">",76,2); //!<arrow denoting a different screen can be shown by pointing potentiometer in right direction
        lcd.printString("Graph",0,5);//!<graph button label
        lcd.printString("Current",42,5);//!<current button label
        if (graphFlag) { //!<if the graph button is pressed
            graphFlag = 0; //!<reset graph flag to 0
            displayTemperatureGraph(); //!<display temperature graph from function


        }
        if (currentFlag ==1 ) { //!<if the current button is pressed
            currentFlag = 0; //!<reset current flag to 0

            displayTemperature(); //!<display current temperature value
        }
    } else if ((3.0/4.0)>POT && POT>=(1.0/2.0)) {
        lcd.printString("Pressure",18,2);//!<data type label
        lcd.printString("<",2,2);//!<arrow denoting a different screen can be shown by pointing potentiometer in left direction
        lcd.printString(">",76,2); //!<arrow denoting a different screen can be shown by pointing potentiometer in right direction
        lcd.printString("Graph",0,5);//!<graph button label
        lcd.printString("Current",42,5);//!<current button label
        if (graphFlag) { //!<if the graph button is pressed
            graphFlag = 0; //!<reset graph flag to 0
            displayPressureGraph();//!<display pressure graph from function


        }

        if (currentFlag) { //!<if the current button is pressed
            currentFlag = 0; //!<reset current flag to 0
            displayPressure();//!<display current pressure value
        }
    } else if ((1.0/2.0)>POT && POT>=(1.0/4.0)) {
        time();
        lcd.printString("Date",30,0);//!<date label
        lcd.printString(">",76,2); //!<arrow denoting a different screen can be shown by pointing potentiometer in right direction
        lcd.printString("<",2,2);//!<arrow denoting a different screen can be shown by pointing potentiometer in left direction
        lcd.printString(bufferDate,19,2);//!<display date
    } else {
        time();
        lcd.printString("Time",30,0);//!<time stamp
        lcd.printString("<",2,2);//!<arrow denoting a different screen can be shown by pointing potentiometer in left direction
        lcd.printString(bufferTime,19,2);//!<display time
    }
}




void time(){ //!<function to get time

time_t seconds = time(NULL); //!<get current time
        strftime(bufferTime, 14 , "%X", localtime(&seconds));//!<format time into a string
        strftime(bufferDate, 14 , "%D", localtime(&seconds));//!<format date into a string
}

void timerExpired() //!<Interrupt service routine
{
    timerFlag = 1 ; //!<set timer flag
}

void timeOutStopped() //!<Interrupt service routine
{

    timeOutFlag = 1; //!<set time out flag
}

void graphButtonPressed () //!<Interrupt service routine
{
    graphFlag = 1; //!<set graph flag
    //serial.printf("graph button pressed "); //!<used for debugging
}

void currentButtonPressed () //!<Interrupt service routine
{
    currentFlag = 1; //!<set current flag
    //serial.printf("current button pressed "); //!<used for debugging
}
void receiveData() //!<function to receive data from BMP
{

    Measurement measurement;  //!<measurement structure declared in BMP180 class

    //! read values (T in Celsius and P in mb) and print over serial port
    measurement = bmp180.readValues();
    //serial.printf("T = %.2f C P = %.2f mb\n",measurement.temperature,measurement.pressure); //!<used for debugging

    temperature = measurement.temperature ; //!<formatted temperature
    pressure = measurement.pressure ; //!<formatted pressure

    error();

}


void displayTemperature() //!<function to display formatted temperature
{
    while(1) {
        if (timerFlag ==1) { //!<if the timer flag is set (Timer every 1.0 second)
            timerFlag = 0; //!<reset flag to 0
            lcd.clear(); //!<clear screen
            receiveData(); //!<get data from function
            char buffer[14];  //!<each character is 6 pixels wide, screen is 84 pixels (84/6 = 14)
            //! so can display a string of a maximum 14 characters in length
            //!or create formatted strings - ensure they aren't more than 14 characters long
            int length = sprintf(buffer,"T = %.2f C",temperature); //!<print formatted data to buffer
            // it is important the format specifier ensures the length will fit in the buffer
            //serial.printf( " L %i",length); //!<used for debugging
            if (length <= 14)  //!<if string will fit on display
                lcd.printString(buffer,0,2);           //!<display on screen

            lcd.printString("Menu",60,5);//!<menu button label
            wait(0.1); //!<time inbetween each plot
            if (currentFlag == 1 ) {
                currentFlag = 0;
                break ;
            }
        }
    }
}

void displayPressure()   //!<function to display formatted pressure
{
    while(1) {
        if (timerFlag ==1) { //!<if the timer flag is set (Timer every 1.0 second)
            timerFlag = 0; //!<reset flag to 0
            lcd.clear(); //!<clear screen
            receiveData(); //!<get data from function
            char buffer1[14]; //!<same as temperature
            int length1 = sprintf(buffer1,"P = %.2fmb",pressure); //!<print formatted data to buffer1
            //serial.printf( " L1 %i",length1); //!<used for debugging
            if (length1 <= 14) //!<if string will fit on display
                lcd.printString(buffer1,0,2); //!<display on screen

            lcd.printString("Menu",60,5);//!<menu button label
            wait(0.1); //!<time inbetween each plot

            if (currentFlag == 1 ) {
                currentFlag = 0;
                break ;
            }
        }
    }
}

void displayTemperatureGraph()   //!<function to display temperature graph
{

    float temperatureArray[84]; //!<maximum capacity of array is 84
    int j=0; //!<initialise j to 0
    while(1) {
        lcd.clear(); //!<clear screen
        lcd.printString("Menu",0,5);//!<menu button label

        receiveData(); //!<get data from function
        temperatureArray[j] = (temperature/47); //!<stores current temperature value in array
        j++; //!<adds 1 to value of j to store next temperature value
        lcd.plotArray(temperatureArray); //!<plots the array as a line
        if (j>83) { //!<if the array is full
            memset (temperatureArray,0, sizeof(temperatureArray)); //!<reset to 0 so line graph starts again
            j = 0;
        }
        wait(0.1); //!<time inbetween each plot

        if (graphFlag == 1 ) {
            graphFlag = 0;
            break ;
        }
    }
}

void displayPressureGraph()   //!<function to display pressure graph
{

    float pressureArray[84]; //!<maximum capacity of array is 84
    int j=0; //!<initialise j to 0
    while(1) {
        lcd.clear(); //!<clear screen
        lcd.printString("Menu",0,5);//!<menu button label

        receiveData(); //!<get data from function
        pressureArray[j] = (pressure/1100); //!<stores current pressure value in array
        j++; //!<adds 1 to value of j to store next pressure value
        lcd.plotArray(pressureArray); //!<plots the array as a line
        if (j>83) { //!<if the array is full
            memset (pressureArray,0, sizeof(pressureArray));//!<reset to 0 so line graph starts again
            j = 0;
        }
        wait(0.1); //!<time inbetween each plot

        if (graphFlag == 1 ) {
            graphFlag = 0;
            break ;
        }
    }
}

void error() //!<error function
{

    if (temperature > 29 || pressure > 1000) { //!<if temperature or pressure goes above threshold values
        red = 1; //!<red LED turns on
        
        //!can't use buzzer because PWM signal is affecting LCD screen
        //buzzer.period(1/590);
        //buzzer=0.5; // set duty cycle
        

    } else {
        red=0; //!<red LED stays off
    }
}

int main() //!<main code
{
    PHY_PowerDown(); //!<ethernet power-down
    int result = semihost_powerdown(); //!<interface power-down
    lcd.init();//!<initialise display
    bmp180.init();//!<initiliase barometer
    timer.attach (&timerExpired , 1.0); //!<instead of using wait(1.0)
    timeOut.attach (&timeOutStopped , 2.0); //!<instead of using wait(2.0)
    graphButton.rise (&graphButtonPressed);
    currentButton.rise (&currentButtonPressed);
    red.period_us(25);
    //set_time(1430415950); //!<initialise time to now
    screen(); //!<show the welcome screen

    while(1) {
        time();//!<get time and date
        if (timeOutFlag ==1 ) {
            menu(); //!<display menu screen(s)
        }
        wait (0.1); //!<necessary wait so screen is clear and easy to read
    }
}
