/**
@file main.h
@brief Project: Ultrasonic Sensor
@author Adrian Wong
@date   April 2015
*/

#ifndef MAIN_H
#define MAIN_H

#define PI 3.14159265359

#include "mbed.h"

/**  
@namespace lcd
@brief Nokia LCD screen showing sensor measurements
@brief vcc, sce, rst, dc, mosi, clk, led backlight
*/ 
N5110 lcd(p7,p8,p9,p10,p11,p13,p26);
/**  
@namespace leds
@brief GPIO output for status onboard LEDs
*/ 
BusOut leds(LED4,LED3,LED2,LED1); 
/**  
@namespace sensor
@brief SRF02 sensor input
@brief sda, scl
*/ 
SRF02 sensor(p28,p27);
/**  
@namespace serial
@brief serial USB transmission line
*/ 
Serial serial(USBTX,USBRX);
/**  
@namespace timer1
@brief 1st ticker timer
*/ 
Ticker timer1; 
/**  
@namespace timer2
@brief 2nd ticker timer
*/ 
Ticker timer2; 
/**  
@namespace timer3
@brief 3rd ticker timer
*/ 
Ticker timer3; 
/**  
@namespace timer4
@brief 4th ticker timer
*/ 
Ticker timer4; 
/**  
@namespace timer5
@brief 5th ticker timer
*/ 
Ticker timer5; 
/**  
@namespace unit
@brief unit interrupt button
*/ 
InterruptIn unit(p16); 
/**  
@namespace display
@brief LCD display interrupt button
*/ 
InterruptIn display(p17); // input interrupt button
/**  
@namespace led1
@brief output showing on/off status
*/ 
DigitalOut led1(p24); 
/**  
@namespace led2
@brief warning LED
*/ 
DigitalOut led2(p23); // warning LED
/**  
@namespace buzzer
@brief warning buzzer
*/ 
PwmOut buzzer(p21); 
/**  
@namespace pot
@brief volume control for buzzer
*/ 
AnalogIn pot(p20);
/**  
@namespace local
@brief create local filesystem
*/ 
LocalFileSystem local("local"); 

/**
ISR that is called when serial data is received
@param buffer - input maximum stored information
@param setTimeFlag - integer to add
@returns setTimeFlag value and stored serial information
*/
void serialISR(); 
/**
function to set the UNIX time
@param time - atoi() converts a string to an integer
@returns updated time
*/
void setTime();
/**
buffer to store received string
@param setting buffer size
@returns stored string
*/
char rxString[16];
/**
function to action of unit button
@returns toggled unit button value
*/
void uPressed(); 
/**
function to action of display button
@returns toggled display button value
*/
void dPressed(); 
/**
function to action when timer1 triggers
@returns timer1Flag value
*/
void timer1Expired(); 
/**
function to action when timer2 triggers
@returns timer2Flag value
*/
void timer2Expired(); 
/**
function to action when timer3 triggers
@returns timer3Flag value
*/
void timer3Expired();
/**
function to action when timer4 triggers
@returns timer4Flag value
*/ 
void timer4Expired(); 
/**
function to action when timer5 triggers
@returns timer5Flag value
*/
void timer5Expired(); 
/**
warning function
@param vol = potentiometer reading divided by 5
@param buzzer = value of vol + 0.8
@param led2 = inverts led2 value
@returns buzzer and led2 value
*/
void warning(); 
/**
function to write data to file
@param buffer - set characters to be saved
@param distance - set distance value to be saved
@returns buffer and distance
*/
void writeDataToFile(char* buffer, float distance); 
/**
function to get distance
@param d1 to d10 - distance measured by sensor in cm
@param dt - average distance measured 
@returns dt
*/
float measureDistance(); // function to get distance

int timer1Flag = 0; /*!< timer1 flag set in ISR */
int timer2Flag = 0; /*!< timer2 flag set in ISR */
int timer3Flag = 0; /*!< timer3 flag set in ISR */
int timer4Flag = 0; /*!< timer4 flag set in ISR */
int timer5Flag = 0; /*!< timer5 flag set in ISR */
int uFlag = 0; /*!< unit button flag set in ISR */
int dFlag = 0; /*!< display button flag set in ISR */
int setTimeFlag = 0; /*!< setTime flag set in ISR */
int x = 84; /*!< size of set horizontal (x-axis) */
int y = 48; /*!< size of set vertically (y-axis) */

#endif
