#include "mbed.h"
#include "math.h"
#include "QEI.h"
#include "HIDScope.h"
#include "BiQuad.h"

HIDScope scope(3);

AnalogIn pot1(A2); // Controls potentiometer 1, which controls the reference position

InterruptIn motor1A(D13);   // Encoder 1a
InterruptIn motor1B(D12);   // Encoder 1b
QEI         Encoder1(D12,D13,NC,64);    // Note that we use X2 encoding, so there are 4200 counts/rotation, as opposed to 8400 from the documentation.

DigitalOut directionpin1(D7);
PwmOut pwmpin1(D6);
Serial pc(USBTX, USBRX); // tx, rx

Ticker Motor1;

double GetReferencePosition(){
    // Returns the wanted reference position
    const int CpR = 4200; //Counts Per Revolution; when the counts 4200, 1 revolution has been made. (Using X2 encoding)
    double ReferencePosition = CpR*pot1; // Returns a value between 0 and 4200, the wanted rotation. 
    return ReferencePosition;
}
    
double GetActualPosition(){
    double ActualPosition = Encoder1.getPulses(); // Gets the actual amount of counts from motor 1. Note the minus sign for convention (CCW = positive counts.)
    return ActualPosition;
}

double PID_controller(double error){
    static double error_integral = 0;
    static double error_prev = error; // initialization with this value only done once!
    static BiQuad LowPassFilter(0.0640, 0.1279, 0.0640, -1.1683, 0.4241);
    double Ts = 0.01;
    
    // Proportional part
        
        double Kp = 1;
        double u_k = Kp*error;

    // Integral part
        double Ki = 0.06;
        error_integral = error_integral + error * Ts;
        double u_i = Ki * error_integral;
    
    // Derivative part
        double Kd = 0.2;
        double error_derivative = (error - error_prev)/Ts;
        double filtered_error_derivative = LowPassFilter.step(error_derivative);
        double u_d = Kd * filtered_error_derivative;
        error_prev = error;
    // Sum all parts and return it
    return u_k + u_i + u_d;
}

void SetMotor1(double motorValue){
    // Given -1<=motorValue<=1, this sets the PWM and direction
    // bits for motor 1. Positive value makes motor rotating
    // clockwise. motorValues outside range are truncated to
    // within range
    if (motorValue >=0) directionpin1=1;
        else directionpin1=0;
    if (fabs(motorValue)>1) pwmpin1 = 1;
        else pwmpin1 = fabs(motorValue);
}

void ScopeData(double motorValue){
        scope.set(0,GetReferencePosition());  // Wanted amount of counts
        scope.set(1,-Encoder1.getPulses());  // Amount of counts of motor 1
        scope.set(2,motorValue);
        scope.send(); // send what's in scope memory to PC
}

void MeasureAndControl(void){
    // This function determines the desired velocity, measures the
    // actual velocity, and controls the motor with 
    // a simple Feedback controller. Call this from a Ticker.
    float ReferencePosition = GetReferencePosition();
    float ActualPosition = GetActualPosition();
    float motorValue = PID_controller(ReferencePosition - ActualPosition);
    SetMotor1(motorValue);
    ScopeData(motorValue);
}

int main(){
    pwmpin1.period_us(60);
   // Motor1.attach(&MeasureAndControl, 0.01); 
    while(1){
            float ReferencePosition = GetReferencePosition();
            float ActualPosition = GetActualPosition();
            float motorValue = PID_controller(ReferencePosition - ActualPosition);
            SetMotor1(motorValue);
            ScopeData(motorValue);
            wait(0.01f);
        }
    }