/**
@file SRF02.h

@brief Header file containing member functions and variables

*/

#ifndef SRF02_H
#define SRF02_H

// registers
#define CMD_REG         0x00
#define RANGE_H_REG     0x02
#define RANGE_L_REG     0x03

// commands
#define INCH_CMD    0x50
#define CM_CMD      0x51
#define US_CMD      0x52

#include "mbed.h"

/**
@brief Library for interfacing with SRF02 Ultrasonic Sensor in I2C
@see http://www.robot-electronics.co.uk/htm/srf02tech.htm

@brief Revision 1.1

@author Craig A. Evans
@date   June 2016
 *
 * Example:
 * @code

 #include "mbed.h"
 #include "SRF02.h"
 
 SRF02 sensor(p28,p27,0xE0);  // SDA, SCL, address

 int main() {

    while(1) {

        // read sensor distance in cm and print over serial port
        int distance = sensor.getDistanceCm();
        serial.printf("Distance = %d cm\n",distance);
        // short delay before next measurement
        wait(0.5);

    }
}
 * @endcode
 */

class SRF02
{
public:

    /** Create a SRF02 object connected to the specified I2C pins
    *
    * @param sdaPin - mbed SDA pin 
    * @param sclPin - mbed SCL pin
    * @param addr - write address of the SRF02 sensor 
    * 
    */
    SRF02(PinName sdaPin, PinName sclPin, char addr);
    /** Read distance in centimetres
    *
    * @returns distance in centimetres (int)
    * 
    */
    int getDistanceCm();
    /** Change I2C address of SRF02  sensor
    *
    * @param address - @see https://www.robot-electronics.co.uk/htm/srf02techI2C.htm for valid addresses
    *
    */
    void changeAddress(char addr);

private:
    /** Hangs in infinite loop flashing 'blue lights of death'
    *
    */ 
    void error();


private:  // private variables
    I2C* i2c;
    BusOut* leds;
    
    char w_addr_; // write address of sensor
    char r_addr_; // read address (= write address + 1)
};

#endif