/*

2645_FSM_UpDown_Counter

Sample code from ELEC2645

Demonstrates how to implement a simple FSM up/down counter

(c) Craig A. Evans, University of Leeds, Jan 2016

*/

#include "mbed.h"

// pre-processor directives

// defines directions as 0/1. Note UPPERCASE
#define UP 0
#define DOWN 1

// K64F on-board LEDs
DigitalOut r_led(LED_RED);
DigitalOut g_led(LED_GREEN);
DigitalOut b_led(LED_BLUE);
// K64F on-board switches
InterruptIn sw2(SW2);
InterruptIn sw3(SW3);
// Gamepad Button A
InterruptIn buttonA(PTB9);

// LEDs on Gamepad (1 to 4) - active-low 0 = on and 1 = off
BusOut output(PTA1,PTA2,PTC2,PTC3);

// array of states in the FSM, each element is the output of the counter
// set the output in binary to make it easier, 0 is LED on, 1 is LED off
int fsm[4] = {0b0111,0b1011,0b1101,0b1110};

// flag - must be volatile as changes within ISR
// g_ prefix makes it easier to distinguish it as global
volatile int g_buttonA_flag = 0;

// function prototypes
// set-up the on-board LEDs and switches
void init_K64F();
// Button A interrupt service routine
void buttonA_isr();

int main()
{
    // initialise on-board LED and switches
    init_K64F();
    
    // Button A is connected between the pin and 3.3 V, we therefore need to turn on the internal pull-down resister
    buttonA.mode(PullDown);
    
    // It will return 0 by default and a 1 when pressed i.e. cause a rising edge
    buttonA.rise(&buttonA_isr);
    
    // set inital state
    int state = 0;
    // set initial direction
    int direction = UP;

    while(1) {  // loop forever
    
        // check if flag i.e. interrupt has occured
        if (g_buttonA_flag) {
            g_buttonA_flag = 0;  // if it has, clear the flag

            // swap direction when button has been pressed
            // (could just use ! but want this to be explicit to aid understanding)
            if (direction == UP) {
                direction = DOWN;    
            }
            else {
                direction = UP;
            }
        }
    
        output = fsm[state];  // output current state

        // check which state we are in and see which the next state should be next depending on direction
        switch(state) {
            case 0:
                switch(direction) {
                    case UP:
                        state = 1;
                        break;
                    case DOWN:
                        state = 3;
                        break;
                }
                break;
            case 1:
                switch(direction) {
                    case UP:
                        state = 2;
                        break;
                    case DOWN:
                        state = 0;
                        break;
                }
                break;
            case 2:
                switch(direction) {
                    case UP:
                        state = 3;
                        break;
                    case DOWN:
                        state = 1;
                        break;
                }
                break;
            case 3:
                switch(direction) {
                    case UP:
                        state = 0;
                        break;
                    case DOWN:
                        state = 2;
                        break;
                }
                break;
            default:  // default case
                error("Invalid state!");  //invalid state - call error routine
                // or could jump to starting state i.e. state = 0
                break;
        }

    wait(0.5); // small delay


    }
}

void init_K64F()
{
    // on-board LEDs are active-low, so set pin high to turn them off.
    r_led = 1;
    g_led = 1;
    b_led = 1;

    // since the on-board switches have external pull-ups, we should disable the internal pull-down
    // resistors that are enabled by default using InterruptIn
    sw2.mode(PullNone);
    sw3.mode(PullNone);

}

// Button A event-triggered interrupt
void buttonA_isr()
{
    g_buttonA_flag = 1;   // set flag in ISR
}