// Encoder lib for user interfaces
// Extremely simple time driven state machin with inherent debouncing.
// No irq-jamming because of switch bouncing
// No counting on first edges when encoder leaves idle state (both switches open).
// Rock solid
/*
The MIT License (MIT)

Copyright (c) 2014 calima engineering

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
*/

#include "mbed.h"
#include "Encoder.h"
#include "EncoderConfig.h"

// Chose encoder pins, default, nucleo-XXX
#ifndef ENCODER_PIN_A
#define ENCODER_PIN_A D6
#endif

#ifndef ENCODER_PIN_B
#define ENCODER_PIN_B D7
#endif

#ifndef ENCODER_PIN_PB
#define ENCODER_PIN_PB D5
#endif

// Sample every ... microseconds, default
// 250µs is a good value for cheap 24 ppr encoders (ALPS EC12E24...) with strong bouncing.
// 1000µs is a good value for quality 24 ppr encoders (ALPS EC12E24...) with low bouncing.
// 250µs is a good value for fast 24+ ppr encoders (Sparkfun, Qubig Panel1 type...).
#ifndef ENCODER_SAMPLE_PERIOD_US
#define ENCODER_SAMPLE_PERIOD_US (250)
#endif

// Calc debounce counter reload
#define PB_DEBOUNCE_RELOAD (((uint32_t)ENCODER_PB_DEBOUNCE_MS * 1000uL) / ENCODER_SAMPLE_PERIOD_US)

// Counts steps := pulses
static volatile uint16_t Pulses_u16;

// State variables
static volatile bool Up, Down, Ready, PB_RawState_b, PB_State_b;

// Counter for debouncing
static volatile uint16_t PB_DebounceCounter_u16;

DigitalIn  PinA(ENCODER_PIN_A), PinB(ENCODER_PIN_B), PinPB(ENCODER_PIN_PB);

static  Ticker EncoderTick;
static void EncoderTickIRQ();

#if DEBUG_ENC
static DigitalOut DebugPin(D4);
#endif

// Call once in main.
void EncoderStart ()
{
    PinA.mode(PullUp);  // Encoder pin A
    PinB.mode(PullUp);  // Encoder pin B
    EncoderTick.attach_us (&EncoderTickIRQ, ENCODER_SAMPLE_PERIOD_US);
    PinPB.mode(PullUp); // Pushbutton pin
    PB_DebounceCounter_u16 = PB_DEBOUNCE_RELOAD;    // Start
}

// Switch off IRQ and change Pullups to PullDowns to save power
void EncoderStop ()
{
    PinA.mode(PullDown);
    PinB.mode(PullDown);
    PinPB.mode(PullDown);
    EncoderTick.detach();
}

// Get counting variable.
uint16_t EncoderGetPulses ()
{
    return Pulses_u16;
}

// Get pushbutton state.
bool EncoderGetPB ()
{
    return not PB_State_b;  // invert, pressed == true
}

// State machine. Runs in IRQ. Static.
static void EncoderTickIRQ()
{
#if DEBUG_ENC
    DebugPin = 0;
    DebugPin = 1;
    DebugPin = 0;
#endif
    // Encoder section
    // If-then-else structure is good because every pin is read max. once.
    if(PinB) {
        if(PinA) { // booth open?
            //no! Skips pulses at low speed! Ready = (!Up && !Down); (one extra cicle in idle mode)
            Ready = true;  // Reset all
            Up = false;
            Down = false;
        } else {
            Up = Ready;  // clockwise.
        }
    } else {
        if(PinA) { // counterclockwise?
            Down = Ready;
        } else {  // booth closed?
            if(Ready) {
                if (Up) {
                    Pulses_u16++; //moving forward
#if DEBUG_ENC
                    DebugPin = 1;
#endif
                }
                // Do not change to  ...else... construct!
                if (Down) {
                    Pulses_u16--; //moving reverse
#if DEBUG_ENC
                    DebugPin = 0;
#endif
                }
            }
            Ready = false;
        }
    }


// Pushbutton section
    if(PinPB == PB_RawState_b) {
        if(--PB_DebounceCounter_u16 == 0) {
            PB_State_b = PB_RawState_b;
            PB_DebounceCounter_u16 = PB_DEBOUNCE_RELOAD;    // Restart
        }
    } else {
        PB_RawState_b = PinPB;
        PB_DebounceCounter_u16 = PB_DEBOUNCE_RELOAD;    // Restart
    }
}

// End of file
