/*
 * Author: Edoardo De Marchi
 * Date: 05/10/13
 * Notes: Library for TEA5767 radio module
*/

#pragma once 
#include "mbed.h"


/** TEA5767 class 
 *
 */ 
class TEA5767
{

public:
       /** Creates an TEA5767 object connected to the specified I2C object
        *
        * @param sda  I2C data port
        * @param scl  I2C clock port
        * @param addr the I2C address of the TEA5767 device
        */ 
      TEA5767(PinName sda, PinName scl, int addr);
       
       /** Destroys a TEA5767 object
        *
        */
      ~TEA5767();
      
       /** Set the band FM
        *
        * @param valueBand set the band ('e' = EUROPE/US - 'j' = JAPANESE)
        */ 
      void SetBand(char valueBand);
      
       /** Set the radio frequency 
        *
        * @param freq frequency
        * @param side side injection mode ('h' = HIGH - 'l' = LOW)
        */ 
      void SetFrequency(double freq, char side = 'h');
           
       /** Auto search up frequency 
        *
        * @param freq frequency
        */       
      void SearchUp(float freq);

       /** Auto search down frequency
        *
        * @param freq frequency
        */ 
      void SearchDown(float freq);
      
       /** Read the current frequancy
        *
        * @returns the current frequency
        */ 
      float FreqCurrent();
      
       /** Checks if the address exist on an I2C bus 
        *
        * @returns 0 on success, or non-0 on failure
        */ 
      int CheckDevice();
      
       /** Read the signal level of the band
        *
        * @returns the signal level
        */ 
      int SignalLevel();


private:

       /** Initialization of the device
        *
        */ 
      void Init();
      
       /** calculation of the high or low side injection 
        *
        * @param freq the wanted tuning frequency
        * @param mode I2C high ('h') or low ('l') side injection
        * @returns decimal value of PLL word
        */ 
      unsigned int SideInjection(float freq, char mode);
      
       /** Set the 5 address of the device   
        *
        * @param N      14bits PLL word
        * @param data_1 set the bit 7 and 6 of 1st data byte (default value = 0x00)
        * @param data_3 set the 3th data byte (default value = 0x10)
        * @param data_4 set the 4th data byte (default value = 0x10)
        * @param data_5 set the 5th data byte (default value = 0x00)
        */ 
      void SetData(unsigned int N, char data_1 = 0x00, char data_3 = 0x10, char data_4 = 0x10, char data_5 = 0x00);
               
       /** Read the 5 address of the device   
        *
        * @param buf buffer in which the 5 addresses are saved
        */ 
      void GetData(char* buf);
      
      char band;
      char side;
      unsigned int frequency;
      I2C i2c;
      int addr; 

};