//------------------------------------------------------------------------------
//  イコライザ用フィルタのパラメータを設定し，その周波数特性を描画するためのクラス
//  
//  2016/05/09, Copyright (c) 2016 MIKAMI, Naoki
//------------------------------------------------------------------------------

#include "DesignerDrawer.hpp"

namespace Mikami
{
    // Constructor
    DesignerDrawer::DesignerDrawer(uint16_t x0, uint16_t y0,
                                   int bands, float f0, int fs, float db1)
        : lcd_(GuiBase::GetLcdPtr()), ts_(GuiBase::GetTsPtr()),
          X0_(x0), Y0_(y0), BANDS_(bands), Q_VAL_(1.0f/sqrtf(2.0f))
    {
        drawerObj_ = new FrqRespDrawer(x0, 50.0f, 20000.0f, 142,
                                       y0, -18, 18, db1, 6, fs);

        calculator_ = new GrEqParams(bands, fs);
        f0_ = new float[bands];
        for (int n=0; n<bands; n++)
            f0_[n] = f0*powf(2, n);

        ck_ = new BiquadGrEq::Coefs[bands];
        for (int n=0; n<bands; n++)
            ck_[n] = calculator_->Execute(n, f0_[n], 0, Q_VAL_);
        frqResp_ = new GrEqualizerFrqResp(bands);
        frqResp_->SetParams(ck_);

        // 周波数特性の描画
        DrawResponse();
    }

    DesignerDrawer::~DesignerDrawer()
    {
        delete[] frqResp_;
        delete[] ck_;
        delete[] f0_;
        delete calculator_;
        delete drawerObj_;
    }

    // 周波数特性の描画
    void DesignerDrawer::DrawResponse()
    {
        drawerObj_->DrawAxis();             // 目盛線の描画        
        FrqRespDrawer::AxisX_Char numX[] =  // 横軸の目盛値を描画する際に使う構造体の配列
            {{   50,  "50"}, {  100, "100"}, {  200, "200"}, {  500, "500"}, { 1000, "1k"},
             { 2000,  "2k"}, { 5000,  "5k"}, {10000, "10k"}, {20000, "20k"}};
        drawerObj_->DrawNumericX(numX, 9, 6, "Frequency [Hz]"); // 横軸の目盛
        drawerObj_->DrawNumericY(-24, -6, 6, "%3d");            // 縦軸の目盛値は 6 dB 間隔
        drawerObj_->DrawGraph(frqResp_);    // 周波数特性のカーブの描画
    }
    
    // 周波数特性の平坦化と描画
    void DesignerDrawer::DrawFlat()
    {
        for (int n=0; n<BANDS_; n++)
            ck_[n] = calculator_->Execute(n, f0_[n], 0, Q_VAL_);
        frqResp_->SetParams(ck_);
        drawerObj_->Erase();
        drawerObj_->DrawAxis();             // 目盛線の描画
        drawerObj_->DrawGraph(frqResp_);    // 周波数特性のグラフのカーブを描画する
    }

    // 特定のバンドのイコライザ用フィルタのパラメータの設定と周波数特性の再描画
    void DesignerDrawer::DesignAndRedraw(float gainDb, int n)
    {
        ck_[n] = calculator_->Execute(n, f0_[n],  gainDb, Q_VAL_);
        frqResp_->SetParam(ck_[n], n);
        drawerObj_->Erase();
        drawerObj_->DrawAxis();             // 目盛線の描画
        drawerObj_->DrawGraph(frqResp_);    // 周波数特性のグラフのカーブを描画する
    }
}
