/* KL25Z mbed library for ILI9341 touch devices
 * SPI Interface

 * Uses Peter Drescher ILI9341 Library
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */


#include "mbed.h"
#include "SPI_TFT_ILI9341.h"
#include "Arial12x12.h"
#include "touch.h"


#define     PIN_XP          PTB3
#define     PIN_XM          PTB1
#define     PIN_YP          PTB2
#define     PIN_YM          PTB0
#define     PIN_MOSI        PTD2
#define     PIN_MISO        PTD3
#define     PIN_SCLK        PTD1
#define     PIN_CS_TFT      PTA5
#define     PIN_DC_TFT      PTC8
#define     PIN_BL_TFT      PTC9
#define     PIN_CS_SD       PTA4


SPI_TFT_ILI9341 TFT(PIN_MOSI, PIN_MISO, PIN_SCLK, PIN_CS_TFT, PIN_BL_TFT, PIN_DC_TFT, "TFT");   //PIN_BL_TFT does not have to be connected
TouchScreen tft(PIN_XP, PIN_YP, PIN_XM, PIN_YM); //init TouchScreen port pins


// Paint application - Demonstate both TFT and Touch Screen
int ColorPaletteHigh = 15;  //Height of color palette
int color = White;  //Paint brush initial color
unsigned int colors[16] = {Red, Green, Blue, Cyan, Yellow, Purple, Pink, Black, Orange, Brown, BlueGreen, White, LightPink, DarkGrey, DarkGreen, Mustard};

int RXPLATE = 350;  //Set this to the resistance measured across XP and XM
int Q = 1024;  //10 bit TSC Resolution
float TS_MINX = 5500;   //Set this to screen x min
float TS_MAXX = 58100;  //Set this to screen x max
float TS_MINY = 4700;   //Set this to screen y min
float TS_MAXY = 57600;  //Set this to screen y max
int PRESSURE = 200;     //Set this to max pressure (z) that will register as a touch event
int v[3] ={0,0,0};      //array to roll touches for comparison and validation, initialized at zero briefly.

struct point
{
    int x;
    int y;
    int z;
};

typedef struct point Point;


Point point(int xx, int yy, int zz){
    Point p;
    p.x = xx;
    p.y = yy;
    p.z = zz;
    return p;
}


Point getPoint(){

    float z;
    int y2 = tft.readTouch(PTB3,PTB1,PTB2,PTB0);    //a(analog),i(analog),n(digital),m(digital)
    int x2 = tft.readTouch(PTB2,PTB0,PTB3,PTB1);    //Resources to help set these up
    int z2 = tft.readTouch(PTB0,PTB3,PTB2,PTB1);    //http://electronics.stackexchange.com/questions/113223/4-wire-resisitive-touch-screen-not-working
    int z1 = tft.readTouch(PTB3,PTB0,PTB2,PTB1);    //http://www.ti.com/lit/an/sbaa155a/sbaa155a.pdf
    
    if (z1!=0){
        z = (float)z2/(float)z1;
    }else{
        z = 0;
    }
    int x = x2;
    int y = y2;
    return point(x,y,z);
}

    
    
int main(){
    TFT.claim(stdout);          // send stdout to the TFT display
    TFT.set_orientation(0);
    TFT.set_font((unsigned char*) Arial12x12);  //set font to arial 12x12
    TFT.foreground(Cyan);      // set text chars to Cyan
    TFT.background(Purple);      // set text background to Purple

    for(int i = 0; i<16; i++){
        TFT.fillrect(i*15, 0, 15*(i+1), ColorPaletteHigh, colors[i]);   //Fill color palete with 16 colors defined in SPI_TFT_ILI9341.h
    }
    while(1){
        // a point object holds x y and z coordinates.
        Point p = getPoint();
        //map the ADC value read to into pixel co-ordinates
        p.x = map(p.x, TS_MINX, TS_MAXX, 0, 240);   //Use map functionm to size p.x to screen size
        p.y = map(p.y, TS_MINY, TS_MAXY, 0, 320);   //Use map functionm to size p.y to screen size
        p.z = 20+p.x * RXPLATE / Q * (p.z -1);  //Calc p.z based on above reference links
        
        v[2]=v[1];  //Rotate p.z touch values in array
        v[1]=v[0];
        v[0]=p.z;

        /*TFT.locate(60,90);
        printf("x = %d      \r\n", p.x);    //Uncomment these to see point data on screen and to obtain calibration values
        TFT.locate(60,110);
        printf("y = %d      \r\n", p.y);
        TFT.locate(60,130);
        printf("z = %d      \r\n", p.z);*/

        // we have some minimum pressure we consider 'valid'
        // pressure of 0 means no pressing!
        if (p.z > 40 && p.z < PRESSURE & p.z == (v[1]+v[2])/2) {    //Validate that p.z senses a touch event
            // Detect  paint brush color change
            if(int(p.y) < ColorPaletteHigh+2){
                color = colors[int(p.x)/15];    //Touch event occured in color palette, change color.
            }else{
                TFT.fillcircle(int(p.x),int(p.y),2,color);  //Touch event caused paint event with 2 pixel brush
            }
        }
    }
}