#ifndef _EC19_h_
#define _EC19_h_

#include "EC19_conf.h"

#include "mbed.h"
#include "CBuffer.h"
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

//Debug is disabled by default
#if defined(DEBUG) and (!defined(TARGET_LPC11U24))
#define DBG(x, ...) std::printf("[DBG]" x "\r\n", ##__VA_ARGS__);
#define WARN(x, ...) std::printf("[WARN]" x "\r\n", ##__VA_ARGS__);
#define ERR(x, ...) std::printf("[ERR]" x "\r\n", ##__VA_ARGS__);
#define INFO(x, ...) std::printf("[INFO]" x "\r\n", ##__VA_ARGS__);
#else
#define DBG(x, ...)
#define WARN(x, ...)
#define ERR(x, ...)
#define INFO(x, ...)
#endif

class EC19 {
public:

    /** Wi-Fi mode
     */
    enum WiFiMode {
        WM_INFRASTRUCTURE,
        WM_ADHOCK,
        WM_LIMITEDAP,
    };

    /** Wi-Fi security
     */
    enum Security {
        SEC_OPEN = 0,
        SEC_WPA_PSK = 1,
        SEC_WEP = 2,
        SEC_WPS_BUTTON = 3,
        SEC_WPS_PIN = 4,
    };

    /** TCP/IP protocol
     */
    enum Protocol {
        PROTO_UDP = 0,
        PROTO_TCP = 1,
        PROTO_HTTPGET,
        PROTO_HTTPPOST,
        PROTO_HTTPD,
    };

    /** Client/Server
     */
    enum Type {
        TYPE_CLIENT = 0,
        TYPE_SERVER = 1,
    };

    enum Mode {
        MODE_COMMAND,
        MODE_DATA,
    };

    enum Txs_Type {
        VERSION,
        STATUS,
        CLIENT,
        IP,
        MAC,
        SCAN_RESULTS,
        SCAN_SUB_LIST,
        POWERMODE,
        RSSI,
        AP,
        PROFILES,
        SENT,
        UDP_PRINT_EVENT,
        TCP_PRINT_EVENT,
        UDP_CONNECT,
        TCP_CONNECT,
        HOST_EVENT,
        SSL_CONNECT,
        SERVER_UP,
        SERVER_DOWN,
        UDP_RECEIVED,
        TCP_RECEIVED,
        APCLIENT,
        ERROR,
    };


    // ----- ec19.cpp -----
    /** Constructor
     * \param tx mbed pin to use for tx line of Serial interface
     * \param rx mbed pin to use for rx line of Serial interface
     * \param cts mbed pin to use for cts line of Serial interface
     * \param rts mbed pin to use for rts line of Serial interface
     * \param shutdown shutdown pin of the wifi module (default NULL)
     * \param baud baud rate of Serial interface (default 115200)
     */
    EC19 (PinName tx, PinName rx, PinName cts, PinName rts, PinName shutdown, int baud = 115200);

    /** Connect the wifi module to the ssid contained in the constructor.
     * @return 0 if connected, -1 otherwise
     */
    int join ();

    int dissociate ();

    void poll ();

    /** set Wi-Fi security parameter
     * @param sec Security
     * @param ssid SSID
     * @param pass pass phrase
     */
    int setSsid (Security sec, const char *ssid, const char *phrase);

    // ----- EC19_cmd.cpp -----
    int sendCommand (int cmd, const char *data, int length, int timeout = DEFAULT_WAIT_RESP_TIMEOUT);

//    int sendData(const char * data, int len, int timeout = CFG_TIMEOUT, const char * cmd = NULL);

    static EC19 * getInstance() {
        return _inst;
    };

protected:

    static EC19 * _inst;

    RawSerial _ec;

    int _baud;
    DigitalIn *_cts;
    DigitalOut *_rts;
    int _flow;
#if defined(TARGET_LPC1768) || defined(TARGET_LPC2368) || defined(TARGET_LPC4088) || defined(TARGET_LPC176X) || defined(TARGET_LPC408X)
    LPC_UART1_TypeDef *_uart;
#elif defined(TARGET_LPC11U24) || defined(TARGET_LPC11UXX)
    LPC_USART_Type *_uart;
#elif defined(TARGET_LPC11XX)
    LPC_UART_TypeDef *_uart;
#elif defined(TARGET_LPC81X)
    LPC_UART_TypeDef *_uart;
#endif
    DigitalOut _shutdown;


    struct STATE {
        WiFiMode wm;
        Security sec;
        char ssid[35];
        char pass[66];
        char ip[16];
        char netmask[16];
        char gateway[16];
        char nameserver[16];
        char mac[18];
        char resolv[16];
        char name[32];
        int rssi;
        bool dhcp;
        time_t time;

        bool initialized;
        bool associated;
        volatile Mode mode;
        int cid, datalen;

        volatile bool ok, failure, ready, dwait;
        int n;
        char *buf;
//        CircBuffer<char> *buf;
    } _state;

    // ----- EC19_msg.cpp -----
    void recvData (char c);
    int parseMessage ();
    void msgIp (const char *buf);
    void msgMac (const char *buf);
    void msgDhcp (const char *buf);
    void msgStatus (const char *buf);
    void msgClient (const char *buf);
    void msgAp (const char *buf);
    void msgRssi (const char *buf);
    void msgTcpConnect (const char *buf);
    void msgUdpConnect (const char *buf);
    void msgHostEvent (const char *buf);
    void msgTcpReceived (const char *buf);
    void msgUdpReceived (const char *buf);

    // ----- EC19_cmd.cpp -----
    void clearFlags ();
    int cmdBaud (int baud);
    int cmdAssociate (Security sec, const char *ssid, const char *phrase, int key = 0);
    int cmdDissociate ();
    int cmdMac (const char *mac = NULL);
    int cmdWps (Security sec, const char *pin);
    int cmdPowerMode (int mode, int interval);
    int cmdRssi ();
    int cmdSoftAp (Security sec, const char *ssid, const char *phrase, int ch = 0);
    int cmdTcpClient (const char *remote, int port);
    int cmdTcpServer (int port);
    int cmdUdpClient (const char *remote, int port);
    int cmdUdpServer (int port);
    int cmdClose (int cid);
    int cmdLookup (const char *name);
    int cmdCreateProfile (int pid, Security sec, const char *ssid, const char *phrase,
      char *ip = NULL, char *netmask = NULL, char *gateway = NULL, char *dns = NULL, int ap = 0, int ch = 0);
    int cmdEnableProfile (int pid);
    int cmdAutoProfile (int retry, int num);

    // ----- EC19_hal.cpp -----
    void setReset (bool flg);
    int getUart ();
    void putUart (char c);
    void setRts (bool flg);
    void initUart (PinName cts, PinName rts, int baud);
    void isrUart ();

};

#endif

