#include "mbed.h"
#include "wdt.h"

// Sample program using the Watchdog
// ---------------------------------
//    * Three threads co-operate to flash two LEDs
//    * A simple way to inject a fault, by pressing a button
//    * The watchdog is configured with a 32ms timeout

#define ON 1
#define OFF 0
DigitalOut led_red(LED_RED, ON);
DigitalIn button(PTD0, PullUp);
DigitalOut led1(PTC12, OFF);
AnalogIn ain(A0) ; // Analog input

Serial pc(USBTX, USBRX); // tx, rx, useful for debugging

// This ticker is used to feed the watch dog
Ticker tick;

// Threads
Thread threadT ; // timer thread
Thread threadLED1 ; // thread LED2
Thread threadADC ;

# define ON1 0x01
# define ON2 0x02
# define OFF1 0x04
# define OFF2 0x08
EventFlags signals;  // event flags for signalling

float volts; // adc voltage
volatile int a; // signal check

// ------------Fault Injection Button-------------
//  Wait while the button is down
//     Use this to simulate a STUCK fault
// -----------------------------------------------
void waitButton()
{
    while (!button) ;
}

// ---Thread for controlling LED 2----------------
//   Turn LED2 on/off in response to signals
// -----------------------------------------------
void led1_thread()    // method to run in thread
{
    int evt ;
    while (true) {
        evt = signals.wait_any(ON2 | OFF2); // wait for any signal
        if (evt & ON2) led1 = ON ;
        if (evt & OFF2) led1 = OFF ;
        waitButton() ; // POSSIBLE FAULT HERE

    }
}

// ---Thread for timing --------------------------
//   Send signals to the other threads
// -----------------------------------------------
void timer_thread()    // method to run in thread
{
    while (true) {
        ThisThread::sleep_for(1000) ;
        signals.set(ON2) ;
        a = 1;
        ThisThread::sleep_for(1000) ;
        signals.set(OFF2) ;
        a = 0;
    }
}

void adc_thread ()
{
    while(true) {
        volts = ain.read()*3.3f;
        pc.printf("Volt%3.3f%\n", volts);
        if(a == 1 && (volts < 1 || volts > 2)) { // due to odd resister values, v =! 1.5v
            led_red = 0;
        }
        if (a == 0 && volts > 0) {
            led_red = 0;
        }
        ThisThread::sleep_for(1000) ;
    }
}

// -----------MAIN-------------------------------
//    Configure watchdog. Start threads.
//    Show start up with RED for 1sec
//    Remember the watchdog is running
//       - 1024ms to set it once
//       - then must feed it every 32ms
// ----------------------------------------------

int main(void)
{
    wdt_32ms() ; // initialise watchdog - 32ms timeout
    wdt_kick_all() ;
    tick.attach_us(callback(&wdt_kick_all), 20000); // ticks every 20ms

    // start threads
    threadT.start(timer_thread) ; // start the timer thread
    threadLED1.start(led1_thread) ; // start the LED2 control thread
    threadADC.start(adc_thread) ; // start the adc control thread

    // main thread does nothing more
    while (true) {
        ThisThread::sleep_for(10000) ;
    }
}