/**
<Copyright Header>
Copyright (c) 2013 Jordan "Earlz" Earls  <http://Earlz.net>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.
2. Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.
3. The name of the author may not be used to endorse or promote products
  derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
</Copyright Header>
**/

#ifndef LIGHTVM_H
#define LIGHTVM_H

#ifdef __cplusplus
extern "C" {
#endif

//standard includes required

#include <stdint.h>
#include <stdlib.h>

//project specific includes

#include "config.h"

//"convenience" typedefs
#if (__INT_FAST16_MAX__ > 0xFFFFFFFF) //if it'd take it to 64 bit, manually use next best thing: 32bit
typedef uint32_t reg16_t;
#else
typedef uint_least16_t reg16_t;
#endif

#ifdef USE_OPCACHE
typedef 
#if OPTIMAL_OPCACHE_SIZE==1
uint8_t
#elif OPTIMAL_OPCACHE_SIZE==2
uint16_t
#elif OPTIMAL_OPCACHE_SIZE==4
uint32_t
#else
#error "OPTIMAL_OPCACHE_SIZE must be 1, 2, or 4"
#endif
uint_opcache_t;
#endif




//constants
#define REGISTER_IP 15
#define REGISTER_SP 14
#define REGISTER_TR 13
#define REGISTER_CR 12

//enumerations

typedef enum 
{
    None=0,
    InvalidOpcode=1
    
}lightvm_error_t;

//structures

typedef struct
{
    uint_fast16_t begin, end; //beginning and ending addresses of memory
    void *memory;
    void *next;
}extended_memory_node;

typedef struct
{
    uint8_t *memory; //primary memory
    uint_fast16_t memorysize; //primary memory size
    
    reg16_t *ip;
    reg16_t *sp;
    reg16_t *tr;
    reg16_t *cr;
    reg16_t immdreg; //temporary invisible register
    
    #ifdef INCLUDE_ERROR_MESSAGES
    const char *error;
    #endif
    lightvm_error_t errorcode;
    extended_memory_node* extended;
}lightvm_state;


typedef struct
{
    unsigned char byteops:1;
    unsigned char arg1_indirect:1;
    unsigned char arg2_indirect:1;
    unsigned char arg_immediate:1;
    unsigned char op:4;

}primary_opcode_t;


//prototypes

int lightvm_init(lightvm_state* state);
reg16_t *lightvm_register(lightvm_state* state, unsigned int reg);
uint8_t lightvm_next_op_byte(lightvm_state* s);
uint16_t lightvm_next_op_word(lightvm_state* s);

uint16_t *parse_16bit_arg1(lightvm_state* s, uint16_t opcode);
uint16_t *parse_16bit_arg2(lightvm_state* s, uint16_t opcode);
int lightvm_step_8bit(lightvm_state* s, uint8_t* arg1, uint8_t* arg2);
int lightvm_step_16bit(lightvm_state* s, uint16_t op, uint16_t* arg1, uint16_t* arg2);
int branch_short(lightvm_state* s, unsigned int primary, uint8_t arg);

void lightvm_push(lightvm_state* s, uint16_t value);
uint16_t lightvm_pop(lightvm_state* s);
uint16_t lightvm_peek(lightvm_state *s);

//opcodes

void op16_single_arg(lightvm_state* s, uint16_t op, uint16_t* arg2);

#ifdef __cplusplus
}
#endif

#endif
